/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  THIS IS AN UNPUBLISHED WORK WHICH CONTAINS CONFIDENTIAL INFORMATION
 *  FROM CCL, ITRI.  NO PART OF THIS WORK MAY BE USED IN ANY WAY WITHOUT
 *  THE PRIOR WRITTEN PERMISSION.  ANY UNAUTHORIZED USE COULD SUBJECT THE
 *  PERPETRATOR TO CRIMINAL AND CIVIL LIABILITY.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************/

// ********************************************
// Name   : gip.c
// Author : Liu, Ren Hao (N300, CCL/ITRI)
// Date   : 2003/06/26
// Note   : GARP Information Propagation 
// ********************************************

#include "common.h"
#include "garp_common.h"

#ifdef __GVRP__
#include "gvrp_common.h"
extern Gvr *my_gvr;
#endif

#ifdef __GMRP__
#include "gmrp_common.h"
extern Gmr *my_gmr;
#endif

extern Tuint32 gvrp_debug_level,gmrp_debug_level;

/******************************************************************************
* GIP : GARP INFORMATION PROPAGATION : CREATION, DESTRUCTION
*******************************************************************************/
/*
* GIP maintains a set of propagation counts for up to max attributes.
* It currently maintains no additional information, so the GIP instance
* is represented directly by a pointer to the array of propagation counts.
*
* Creates a new instance of GIP, allocating space for propagation counts
* for up to max_attributes.
*
* Returns True if the creation succeeded together with a pointer to the
* GIP information. This pointer is passed to gid_create_port() for ports
* using this instance of GIP and is saved along with GID information.
*/
Tbool gip_create_gip(Tuint32 max_attributes, Tuint32 **gip)
{ 
 Tuint32 *my_gip;

 if (!sysmalloc(sizeof(Tuint32)*max_attributes,(void **)&my_gip)){
     *gip=NULL;
     return(False);
 }
 
 *gip = my_gip; 
 return(True);

}

/*Destroys the instance of GIP, releasing previously allocated space.*/
void gip_destroy_gip(void *gip)
{ 
  sysfree(gip);
}


/******************************************************************************
* GIP : GARP INFORMATION PROPAGATION : CONNECT, DISCONNECT PORTS
*******************************************************************************/

static void gip_connect_into_ring(Gid *my_port)
{
 Gid *first_connected, *last_connected;

 my_port->is_connected = True;
 my_port->next_in_connected_ring = my_port;
 first_connected = my_port;

 do {
   first_connected = first_connected->next_in_port_ring;
 } while (!first_connected->is_connected);

 my_port->next_in_connected_ring = first_connected;
 last_connected = first_connected;
 
 while(last_connected->next_in_connected_ring != first_connected){
       last_connected = last_connected->next_in_connected_ring;
 }

 last_connected->next_in_connected_ring = my_port;
}


// remove my_port from propogation ring
static void gip_disconnect_from_ring(Gid *my_port)
{
  Gid *first_connected, *last_connected;

  first_connected = my_port->next_in_connected_ring;
  my_port->next_in_connected_ring = my_port;
  my_port->is_connected = False;
  last_connected = first_connected;

  while(last_connected->next_in_connected_ring != my_port){
	last_connected = last_connected->next_in_connected_ring;
  }

  last_connected->next_in_connected_ring = first_connected;
}


/*
* If a GID instance for this application and port number is found, is
* enabled, and is not already connected, then connect that port into the
* GIP propagation ring.
*
* Propagate every attribute that has been registered (i.e., the Registrar
* appears not to be Empty) on any other connected port, and that has in
* consequence a nonzero propagation count, to this port, generating a join
* request.
*
* Propagate every attribute that has been registered on this port and not
* on any others (having a propagation count of zero prior to connecting this
* port) to all the connected ports, updating propagation counts.
*
* Action any timers required. Mark the port as connected.
*
* Finds the port, checks that it is not already connected, and connects
* it into the GIP propagation ring that uses GIP field(s) in GID control
* blocks to link the source port to the ports to which the information is
* to be propagated.
*
* Propagates joins from and to the other already connected ports as
* necessary.
*/
void gip_connect_port(Garp *application, Tuint32 port_no)
{

 Gid 	 *my_port;
 Tuint32 gid_index;
 
 if (gid_find_port(application->gid, port_no, &my_port)) {
      
      if ((!my_port->is_enabled) || (my_port->is_connected)){
          return;
      }
     
      /*connects it into the GIP propagation ring */
      gip_connect_into_ring(my_port);

      for(gid_index = 1; gid_index <= application->last_gid_used;gid_index++) {//changed by rhliu 0->1
	    if (gip_propagates_to(my_port, gid_index)){
	        gid_join_request(my_port, gid_index);
	    }
  	    if (gid_registered_here(my_port, gid_index)){
	        gip_propagate_join(my_port, gid_index);
	    }
      }//for
      
      gip_do_actions(my_port);
      my_port->is_connected = True;
 } //if
    
}//gid_connect_port


/*
* Reverses the operations performed by gip_connect_port().
* Checks to ensure that the port is connected, and then disconnects it from
* the "GIP propagation ring". Propagates leaves to the other ports that
* remain in the ring and causes leaves to my_port as necessary.
*/
void gip_disconnect_port(Garp *application, Tuint32 port_no)
{ 

 Gid 	 *my_port;
 Tuint32 gid_index;
 
 if (gid_find_port(application->gid, port_no, &my_port)) {
 
       if ((!my_port->is_enabled) || (!my_port->is_connected)){
           return;
       }

       for (gid_index = 1; gid_index <= application->last_gid_used;gid_index++) {

	     if (gip_propagates_to(my_port, gid_index)){
	         gid_leave_request(my_port, gid_index);
	     }
	     if (gid_registered_here(my_port, gid_index)){
	         gip_propagate_leave(my_port, gid_index);
	     }
       }//for
      
       gip_do_actions(my_port);
       gip_disconnect_from_ring(my_port);
       my_port->is_connected = False;

 }//if
     
} //gip_disconnect_port


/*************************************************************************************
* When port enter blocking mode, it's gid can't connect to next_in_connected_ring
* so, when stp state became blocking, stp will call gvrp_stp_connect_port/gvrp_stp_disconnect_port
* to notify the gvrp protocol that the port state was changed.
***************************************************************************************/

//called by stp when port enter forwarding mode
void gvrp_stp_connect_port(Tuint32 port_no)
{
#ifdef __GVRP__
 Gid  *my_port;
 Garp *application=&my_gvr->garp;

 if(gvrp_debug_level & GVRP_DEBUG_VLAN){
    printk("GVRP: PORT %lu Enter Forwarding Mode \n",port_no+1);
 }

 if(gid_find_port(application->gid, port_no, &my_port)){
    gvr_added_port(my_gvr,port_no);
 }
#endif
}


//called by stp when port enter forwarding mode
void gmrp_stp_connect_port(Tuint32 port_no)
{
#ifdef __GMRP__
 Gid  *my_port;
 Garp *application=&my_gmr->garp;
 
 if(gmrp_debug_level & GMRP_DEBUG_GEN){
    printk("GMRP: PORT %lu Enter Forwarding Mode \n",port_no+1);
 }
 
 if(gid_find_port(application->gid, port_no, &my_port)){
    gmr_added_port(my_gmr,port_no);
 }
 
#endif

}

//called by stp when port enter blocking mode
void gvrp_stp_disconnect_port(Tuint32 port_no)
{
#ifdef __GVRP__
 Gid  *my_port;
 Garp *application=&my_gvr->garp;
 
 if(gvrp_debug_level & GVRP_DEBUG_VLAN){
    printk("GVRP: PORT %lu Leave Forwarding Mode \n",port_no+1);
 }
 
 if(gid_find_port( application->gid, port_no, &my_port)) {
    gid_leaveall(my_port);
    gip_disconnect_port( application,port_no);
 }
 
#endif   
}

//called by stp when port enter blocking mode
void gmrp_stp_disconnect_port(Tuint32 port_no)
{
#ifdef __GMRP__
 
 Gid  *my_port;
 Garp *application=&my_gmr->garp;
 
 if(gmrp_debug_level & GMRP_DEBUG_GEN){
    printk("GMRP: PORT %lu Leave Forwarding Mode \n",port_no+1);
 }
 
 if(gid_find_port( application->gid, port_no, &my_port)) {
    gid_leaveall(my_port);
    gip_disconnect_port( application,port_no);
 }
 
#endif   
}

/******************************************************************************
* GIP : GARP INFORMATION PROPAGATION : PROPAGATE SINGLE ATTRIBUTES
*******************************************************************************/
/*
* Propagates a join indication, causing join requests to other ports
* if required.
*
* The join needs to be propagated if either (a) this is the first port in
* the connected group to register membership, or (b) there is one other port
* in the group registering membership, but no further port that would cause
* a join request to that port.
*
* Propagates a join indication for a single attribute (identified
* by a combination of its attribute class and index) from my_port to other
* ports, causing join requests to those other ports if required.
*
* GIP maintains a joined membership count for the connected ports for each
* attribute (in a given context) so that leaves are not caused when joins
* from other ports would maintain membership.
*
* Because this count is maintained by dead-reckoning it is important
* that this function only be called when there is a change indication for
* the source port and index.
*/

void gip_propagate_join(Gid *my_port, Tuint32 gid_index)
{ 
 Tuint32 joining_members;
 Gid 	 *to_port;

 if (my_port->is_connected) {
 
    // GIP maintains a joined membership count for the connected ports for each attribute
    joining_members = (my_port->application->gip[gid_index] += 1);
    
    //there is one other port in the group registering membership
    if (joining_members <= 2) {
      
       to_port = my_port;
       
       while ((to_port = to_port->next_in_connected_ring) != my_port) {
 	//this is the first port in the connected group to register membership    
        if ((joining_members == 1)|| (gid_registered_here(to_port, gid_index))) {
            gid_join_request(to_port, gid_index);
	    to_port->application->join_propagated_fn(my_port->application, my_port, gid_index);
	}//if
       }//while
    }//if 
 }//if 

}


/*
* Propagates a leave indication for a single attribute, causing leave
* requests to those other ports if required.
*
* See the comments for gip_propagate_join() before reading further.
* This function decrements the dead-reckoning membership count.
*
* The first step is to check that this port is connected to any others; if
* not, the leave indication should not be propagated, nor should the joined
* membership be decremented. Otherwise, the leave will need to be propagated
* if this is either (a) the last port in the connected group to register
* membership, or (b) there is one other port in the group registering
* membership, in which case the leave request needs to be sent to that
* port alone.
*/

void gip_propagate_leave(Gid *my_port, Tuint32 gid_index)
{
 Tuint32 remaining_members;
 Gid 	 *to_port;

 if (my_port->is_connected) {

  remaining_members = (my_port->application->gip[gid_index] -= 1);
  
  //avoid something wrong (remaining_members can't less than zero)
  if(remaining_members<0){
     my_port->application->gip[gid_index]=remaining_members=0;
  }
  
  if(remaining_members <= 1) {
 
     to_port = my_port;

     while ((to_port = to_port->next_in_connected_ring) != my_port) {
   
          if((remaining_members == 0) || (gid_registered_here(to_port, gid_index))) {
              gid_leave_request(to_port, gid_index);
              to_port->application->leave_propagated_fn(my_port->application,my_port,gid_index);
	  }//if
     }//while
  }//if
 } //if

}

/*
* True if any other port is propagating the attribute associated with index
* to my_port.
*/
Tbool gip_propagates_to(Gid *my_port, Tuint32 gid_index)
{
 if ( (my_port->is_connected)&& ((my_port->application->gip[gid_index] == 2)
      ||((my_port->application->gip[gid_index] == 1) && (!gid_registered_here(my_port, gid_index))))){
    return(True);
 }
 else {
    return(False);
 }
}

/******************************************************************************
* GIP : GARP INFORMATION PROPAGATION : ACTION TIMERS
*******************************************************************************/
/*
* Calls GID to carry out GID scratchpad actions accumulated during this
* invocation of GARP for all the ports in the GIP propagation list,
* including the source port.
*/
void gip_do_actions(Gid *my_port)
{ 
 Gid *this_port = my_port;
 
 do {
   gid_do_actions(this_port);
 } while ((this_port = this_port->next_in_connected_ring) != my_port);

}


//get all info of gid which connected on gip ring
void gip_ring_info(Garp *application, Tuint8 gip_ring[])
{
 Gid 		*first_connected;
 Tuint32	first_port=0,i=0;
 
 if((first_connected=application->gid) !=NULL) {
    
    first_port=first_connected->port_no;

    //find first connected Gid    
    do {
        first_connected = first_connected->next_in_port_ring;
    }   while (!first_connected->is_connected && first_connected->port_no!= first_port);
        
     //if connected_ring has gid
    if(first_connected->is_connected) {
       first_port=first_connected->port_no;

     do {
         gip_ring[i++]=(first_connected->port_no)+1;//port num starts in "1" 
         first_connected=first_connected->next_in_connected_ring;
     }   while(first_connected->port_no !=first_port && i< MAX_LOGIC_PORT);     

    }//if

    gip_ring[i]=END_OF_GIP;//end of mark
         
 }
 else { // no gid in gip ring
    gip_ring[0]=END_OF_GIP;//end of mark
 }
 
}

