/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  THIS IS AN UNPUBLISHED WORK WHICH CONTAINS CONFIDENTIAL INFORMATION
 *  FROM CCL, ITRI.  NO PART OF THIS WORK MAY BE USED IN ANY WAY WITHOUT
 *  THE PRIOR WRITTEN PERMISSION.  ANY UNAUTHORIZED USE COULD SUBJECT THE
 *  PERPETRATOR TO CRIMINAL AND CIVIL LIABILITY.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************/

// ********************************************
// Name   : gidtt.c
// Author : Liu, Ren Hao (N300, CCL/ITRI)
// Date   : 2003/06/26
// Note   : GID PROTOCOL TRANSITION TABLES
// ********************************************

#include "common.h"
#include "garp_common.h"

#ifdef __GVRP__
#include "gvrp_common.h"
#endif

#ifdef __GMRP__
#include "gmrp_common.h"
#endif

extern Tuint32 gvrp_debug_level;

/******************************************************************************
* GIDTT : GID PROTOCOL TRANSITION TABLES : IMPLEMENTATION OVERVIEW
*******************************************************************************/

/* This implementation of GID uses transition tables directly. This makes
* the implementation appear very bulky, but the net code size impact may be
* less than the page of code required for an algorithmic-based implementation,
* depending on the processor. A processing-based implementation is planned as
* both alternatives may be interesting.
*
* The Applicant and the Registrar use separate transition tables. Both use
* a general transition table to handle most events, and separate smaller
* tables to determine behavior when a transmit opportunity occurs (both
* Applicant and Registrar), and when the leave timer expires (Registrar only).
*
* The stored states support management control directly - which leads to a
* total of 14 applicant states and 18 registrar states (the registrar states
* also incorporate leave timer support):
*
* The Applicant may be in one of the following management states:
*
* 1. Normal
* 2. No protocol
*
* The protocol machine is held quiet and never sends a message, even
* to prompt another GID participant on the same LAN to respond. In
* this state the Applicant does messages on the media so it can
* be toggled between Normal and No protocol with minimum
* disruption.
*
* The Registrar may be in one of the following management states:
*
* 1. Normal Registration.
* 2. Registration Fixed.
*
* The Registrar always reports "In" to the application and to GIP
* whatever has occurred on the LAN.
*
* 3. Registration Forbidden.
*
* The Registrar always reports "Empty" to the application and to GIP.
*
* A set of small tables is used to report aspects of the management state of
* both applicant and registrar.
*
* The main applicant transition table (applicant_tt) is indexed by current
* applicant state and GID event, and returns
*
* 1. The new applicant state.
* 2. A start join timer instruction, when required.
*
* The main registrar transition table (registrar_tt) is indexed by current
* registrar state and GID event, and returns
*
* 1. The new registrar state.
* 2. A join indication or a leave indication, when required.
* 3. A start leave timer instruction, when required.
*
* The only user interface to both these tables is through the public
* function gidtt_event(), which accepts and returns Gid_events (to report
* join or leave indications), and which writes timer start requests to the
* GID scratchpad directly.
*
* The Applicant transmit transition table (applicant_txtt) returns the new
* applicant state, the message to be transmitted, and whether the join timer
* should be restarted to transmit a further message. A modifier that
* determines whether a Join (selected for transmission by the Applicant table)
* should be transmitted as a JoinIn or as a JoinEmpty is taken from a
* Registrar state reporting table. The Registrar state is never modified by
* transmission.
*/

/******************************************************************************
* GIDTT : GID PROTOCOL TRANSITION TABLE : TABLE ENTRY DEFINITIONS
******************************************************************************
*/
/******************************************************************************
* GIDTT : GID PROTOCOL: MAIN APPLICANT TRANSITION TABLE
*******************************************************************************/
/* General applicant transition table. See description above. */

Applicant_tt_entry
applicant_tt[Number_of_gid_rcv_events + Number_of_gid_req_events +
Number_of_gid_amgt_events + Number_of_gid_rmgt_events][Number_of_applicant_states]=
{ 

 { /* Gid_null */
 /*Va */{Va, Nt},/*Aa */{Aa, Nt},/*Qa */{Qa, Nt},/*La */{La, Nt},
 /*Vp */{Vp, Nt},/*Ap */{Ap, Nt},/*Vp */{Vp, Nt},
 /*Vo */{Vo, Nt},/*Ao */{Ao, Nt},/*Qo */{Qo, Nt},/*Lo */{Lo, Nt},
 /*Von*/{Von,Nt},/*Aon*/{Aon,Nt},/*Qon*/{Qon,Nt}
 },

 { /* Gid_rcv_leaveempty : changed by rhliu Ao: Nt->Jt*/
 /*Va */{Vp, Nt},/*Aa */{Vp, Nt},/*Qa */{Vp, Jt},/*La */{Vo, Nt},
 /*Vp */{Vp, Nt},/*Ap */{Vp, Nt},/*Qp */{Vp, Jt},
 /*Vo */{Lo, Nt},/*Ao */{Lo, Jt},/*Qo */{Lo, Jt},/*Lo */{Vo, Nt},
 /*Von*/{Von,Nt},/*Aon*/{Von,Nt},/*Qon*/{Von,Nt}
 },
 
 { /* Gid_rcv_leavein :change by rhliu : Ao:Nt->Jt  Vo:Nt->Jt*/
 /*Va */{Va, Nt},/*Aa */{Va, Nt},/*Qa */{Vp, Jt},/*La */{La, Jt},
 /*Vp */{Vp, Nt},/*Ap */{Vp, Nt},/*Qp */{Vp, Jt},
 /*Vo */{Lo, Nt},/*Ao */{Lo, Jt},/*Qo */{Lo, Jt},/*Lo */{Vo, Nt},
 /*Von*/{Von,Nt},/*Aon*/{Von,Nt},/*Qon*/{Von,Nt}
 },
 
  { /* Gid_rcv_leaveall : added by rhliu: Ao:Nt->Jt */
 /*Va */{Vp, Nt},/*Aa */{Vp, Nt},/*Qa */{Vp, Jt},/*La */{Vo, Nt},
 /*Vp */{Vp, Nt},/*Ap */{Vp, Nt},/*Qp */{Vp, Jt},
 /*Vo */{Lo, Nt},/*Ao */{Lo, Jt},/*Qo */{Lo, Jt},/*Lo */{Vo, Nt},
 /*Von*/{Von,Nt},/*Aon*/{Von,Nt},/*Qon*/{Von,Nt}
 },
 
{ /* Gid_rcv_leaveall_range : added by rhliu */
 /*Va */{Vp, Nt},/*Aa */{Vp, Nt},/*Qa */{Vp, Jt},/*La */{Vo, Nt},
 /*Vp */{Vp, Nt},/*Ap */{Vp, Nt},/*Qp */{Vp, Jt},
 /*Vo */{Lo, Nt},/*Ao */{Lo, Jt},/*Qo */{Lo, Jt},/*Lo */{Vo, Nt},
 /*Von*/{Von,Nt},/*Aon*/{Von,Nt},/*Qon*/{Von,Nt}
 },

 { /* Gid_rcv_empty */
 /*Va */{Va, Nt},/*Aa */{Va, Nt},/*Qa */{Va, Jt},/*La */{La, Nt},
 /*Vp */{Vp, Nt},/*Ap */{Vp, Nt},/*Qp */{Vp, Jt},
 /*Vo */{Vo, Nt},/*Ao */{Vo, Nt},/*Qo */{Vo, Nt},/*Lo */{Vo, Nt},
 /*Von*/{Von,Nt},/*Aon*/{Von,Nt},/*Qon*/{Von,Nt}
 },
 
 { /* Gid_rcv_joinempty */
 /*Va */{Va, Nt},/*Aa */{Va, Nt},/*Qa */{Va, Jt},/*La */{Vo, Nt},
 /*Vp */{Vp, Nt},/*Ap */{Vp, Nt},/*Qp */{Vp, Jt},
 /*Vo */{Vo, Nt},/*Ao */{Vo, Nt},/*Qo */{Vo, Nt},/*Lo */{Vo, Nt},
 /*Von*/{Von,Nt},/*Aon*/{Von,Nt},/*Qon*/{Von,Jt}
 },

 { /* Gid_rcv_joinin */
 /*Va */{Aa, Nt},/*Aa */{Qa, Nt},/*Qa */{Qa, Nt},/*La */{La, Nt},
 /*Vp */{Ap, Nt},/*Ap */{Qp, Nt},/*Qp */{Qp, Nt},
 /*Vo */{Ao, Nt},/*Ao */{Qo, Nt},/*Qo */{Qo, Nt},/*Lo */{Ao, Nt},
 /*Von*/{Aon,Nt},/*Aon*/{Qon,Nt},/*Qon*/{Qon,Nt}
 },

 /* Handles repeated joins, i.e., joins for  states that are already in. 
  * Does not provide feedback for joins that are forbidden by management controls; 
  * the expectation is that this table will not be directly used by new management requests.*/

 { /*Gid_join, join request : AO,VP,Lo: Nt->Jt changed by rhliu */
 
 /*Va */{Va, Nt},/*Aa */{Aa, Nt},/*Qa */{Qa, Nt},/*La */{Va, Nt},
 /*Vp */{Vp, Nt},/*Ap */{Ap, Nt},/*Qp */{Qp, Nt},
 /*Vo */{Vp, Jt},/*Ao */{Ap, Jt},/*Qo */{Qp, Nt},/*Lo */{Vp, Jt},
 /*Von*/{Von,Nt},/*Aon*/{Aon,Nt},/*Qon*/{Qon,Nt}
 },


 { /* Gid_leave, leave request. See comments for join requests above. */
 /*Va */{La, Nt},/*Aa */{La, Nt},/*Qa */{La, Nt},/*La */{La, Nt},
 /*Vp */{Vo, Nt},/*Ap */{Ao, Nt},/*Qp */{Qo, Nt},
 /*Vo */{Vo, Nt},/*Ao */{Ao, Nt},/*Qo */{Qo, Nt},/*Lo */{Lo, Nt},
 /*Von*/{Von,Nt},/*Aon*/{Aon,Nt},/*Qon*/{Qon,Nt}
 },
 
 { /* Gid_normal_operation */
 /*Va */{Vp, Nt},/*Aa */{Vp, Nt},/*Qa */{Vp, Jt},/*La */{La, Nt},
 /*Vp */{Vp, Nt},/*Ap */{Vp, Nt},/*Qp */{Vp, Jt},
 /*Vo */{Va, Nt},/*Ao */{Va, Nt},/*Qo */{Va, Jt},/*Lo */{Lo, Nt},
 /*Von*/{Va, Nt},/*Aon*/{Va, Nt},/*Qon*/{Va, Jt}
 },
 
 { /* Gid_no_protocol */
 /*Va */{Von,Nt},/*Aa */{Aon,Nt},/*Qa */{Qon,Nt},/*La */{Von,Nt},
 /*Vp */{Von,Nt},/*Ap */{Aon,Nt},/*Qp */{Qon,Nt},
 /*Vo */{Von,Nt},/*Ao */{Aon,Nt},/*Qo */{Qon,Nt},/*Lo */{Von,Nt},
 /*Von*/{Von,Nt},/*Aon*/{Aon,Nt},/*Qon*/{Qon,Nt}
 },
 
 { /* Gid_normal_registration, same as Gid_null for the Applicant */
 /*Va */{Va, Nt},/*Aa */{Aa, Nt},/*Qa */{Qa, Nt},/*La */{La, Nt},
 /*Vp */{Vp, Nt},/*Ap */{Ap, Nt},/*Vp */{Vp, Nt},
 /*Vo */{Vo, Nt},/*Ao */{Ao, Nt},/*Qo */{Qo, Nt},/*Lo */{Lo, Nt},
 /*Von*/{Von,Nt},/*Aon*/{Aon,Nt},/*Qon*/{Qon,Nt}
 },
 
 { /* Gid_fix_registration, same as Gid_null for the Applicant */
 /*Va */{Va, Nt},/*Aa */{Aa, Nt},/*Qa */{Qa, Nt},/*La */{La, Nt},
 /*Vp */{Vp, Nt},/*Ap */{Ap, Nt},/*Vp */{Vp, Nt},
 /*Vo */{Vo, Nt},/*Ao */{Ao, Nt},/*Qo */{Qo, Nt},/*Lo */{Lo, Nt},
 /*Von*/{Von,Nt},/*Aon*/{Aon,Nt},/*Qon*/{Qon,Nt}
 },
 
 { /* Gid_forbid_registration, same as Gid_null for the Applicant */
 /*Va */{Va, Nt},/*Aa */{Aa, Nt},/*Qa */{Qa, Nt},/*La */{La, Nt},
 /*Vp */{Vp, Nt},/*Ap */{Ap, Nt},/*Vp */{Vp, Nt},
 /*Vo */{Vo, Nt},/*Ao */{Ao, Nt},/*Qo */{Qo, Nt},/*Lo */{Lo, Nt},
 /*Von*/{Von,Nt},/*Aon*/{Aon,Nt},/*Qon*/{Qon,Nt}
 }

};
/******************************************************************************
* GIDTT : GID PROTOCOL: MAIN REGISTRAR TRANSITION TABLE
*******************************************************************************/
Registrar_tt_entry
registrar_tt[Number_of_gid_rcv_events + Number_of_gid_req_events +
Number_of_gid_amgt_events + Number_of_gid_rmgt_events][Number_of_registrar_states] =
{
 
 { /* Gid_null */
 /*In */{Inn,Ni,Nt},
 /*Lv */{Lv, Ni,Nt},
 /*L3 */{L3, Ni,Nt},/*L2 */{L2, Ni,Nt},/*L1 */{L1, Ni,Nt},
 /*Mt */{Mt, Ni,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Lvr,Ni,Nt},
 /*L3r*/{L3r,Ni,Nt},/*L2r*/{L2r,Ni,Nt},/*L1r*/{L1r,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Inf,Ni,Nt},
 /*Lvf*/{Lvf,Ni,Nt},
 /*L3f*/{L3f,Ni,Nt},/*L2f*/{L2f,Ni,Nt},/*L1f*/{L1f,Ni,Nt},
 /*Mtf*/{Mtf,Ni,Nt}
 },

 { /* Gid_rcv_leaveempty */
 //Inr: Lvr->Inr/Lt->Nt changed by rhliu

 /*Inn*/{Lv, Ni,Lt},
 /*Lv */{Lv, Ni,Nt},
 /*L3 */{L3, Ni,Nt},/*L2 */{L2, Ni,Nt},/*L1 */{L1,Ni,Nt},
 /*Mt */{Mt, Ni,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Lvr,Ni,Nt},
 /*L3r*/{L3r,Ni,Nt},/*L2r*/{L2r,Ni,Nt},/*L1r*/{L1r,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Lvf,Ni,Lt},
 /*Lvf*/{Lvf,Ni,Nt},
 /*L3f*/{L3f,Ni,Nt},/*L2f*/{L2f,Ni,Nt},/*L1f*/{L1f,Ni,Nt},
 /*Mtf*/{Mtf,Ni,Nt}
 },
 
  { /* Gid_rcv_leavein : added by rhliu  */
 //Inr: Lvr->Inr/Lt->Nt changed by rhliu
 
 /*Inn*/{Lv, Ni,Lt},
 /*Lv */{Lv, Ni,Nt},
 /*L3 */{L3, Ni,Nt},/*L2 */{L2, Ni,Nt},/*L1 */{L1, Ni,Nt},
 /*Mt */{Mt, Ni,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Lvr,Ni,Nt},
 /*L3r*/{L3r,Ni,Nt},/*L2r*/{L2r,Ni,Nt},/*L1r*/{L1r,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Lvf,Ni,Lt},
 /*Lvf*/{Lvf,Ni,Nt},
 /*L3f*/{L3f,Ni,Nt},/*L2f*/{L2f,Ni,Nt},/*L1f*/{L1f,Ni,Nt},
 /*Mtf*/{Mtf,Ni,Nt}
 },
 
 { /* Gid_rcv_leaveall */
 //Inr: Lvr->Inr/Lt->Nt changed by rhliu
 
 /*Inn*/{Lv, Ni,Lt},
 /*Lv */{Lv, Ni,Nt},
 /*L3 */{L3, Ni,Nt},/*L2 */{L2, Ni,Nt},/*L1 */{L1, Ni,Nt},
 /*Mt */{Mt, Ni,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Lvr,Ni,Nt},
 /*L3r*/{L3r,Ni,Nt},/*L2r*/{L2r,Ni,Nt},/*L1r*/{L1r,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Lvf,Ni,Lt},
 /*Lvf*/{Lvf,Ni,Nt},
 /*L3f*/{L3f,Ni,Nt},/*L2f*/{L2f,Ni,Nt},/*L1f*/{L1f,Ni,Nt},
 /*Mtf*/{Mtf,Ni,Nt}
 },

 { /* Gid_rcv_leaveall_range */
 /*Inn*/{Lv, Ni,Lt},
 /*Lv */{Lv, Ni,Nt},
 /*L3 */{L3, Ni,Nt},/*L2 */{L2, Ni,Nt},/*L1 */{L1, Ni,Nt},
 /*Mt */{Mt, Ni,Nt},
 /*Inr*/{Lvr,Ni,Lt},
 /*Lvr*/{Lvr,Ni,Nt},
 /*L3r*/{L3r,Ni,Nt},/*L2r*/{L2r,Ni,Nt},/*L1r*/{L1r,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Lvf,Ni,Lt},
 /*Lvf*/{Lvf,Ni,Nt},
 /*L3f*/{L3f,Ni,Nt},/*L2f*/{L2f,Ni,Nt},/*L1f*/{L1f,Ni,Nt},
 /*Mtf*/{Mtf,Ni,Nt}
 },

 { /* Gid_rcv_empty */
 /*Inn*/{Inn,Ni,Nt},
 /*Lv */{Lv, Ni,Nt},
 /*L3 */{L3, Ni,Nt},/*L2 */{L2, Ni,Nt},/*L1 */{L1, Ni,Nt},
 /*Mt */{Mt, Ni,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Lvr,Ni,Nt},
 /*L3r*/{L3r,Ni,Nt},/*L2r*/{L2r,Ni,Nt},/*L1r*/{L1r,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Inf,Ni,Nt},
 /*Lvf*/{Lvf,Ni,Nt},
 /*L3f*/{L3f,Ni,Nt},/*L2f*/{L2f,Ni,Nt},/*L1f*/{L1f,Ni,Nt},
 /*Mtf*/{Mtf,Ni,Nt}
 },

 { /* Gid_rcv_joinempty */
 /*Inn*/{Inn,Ni,Nt},
 /*Lv */{Inn,Ni,Nt},
 /*L3 */{Inn,Ni,Nt},/*L2 */{Inn,Ni,Nt},/*L1 */{Inn,Ni,Nt},
 /*Mt */{Inn,Ji,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Inr,Ni,Nt},
 /*L3r*/{Inr,Ni,Nt},/*L2r*/{Inr,Ni,Nt},/*L1r*/{Inr,Ni,Nt},
 /*Mtr*/{Inr,Ni,Nt},
 /*Inf*/{Inf,Ni,Nt},
 /*Lvf*/{Inf,Ni,Nt},
 /*L3f*/{Inf,Ni,Nt},/*L2f*/{Inf,Ni,Nt},/*L1f*/{Inf,Ni,Nt},
 /*Mtf*/{Inf,Ni,Nt}
 },

 { /* Gid_rcv_joinin */
 /*Inn*/{Inn,Ni,Nt},
 /*Lv */{Inn,Ni,Nt},
 /*L3 */{Inn,Ni,Nt},/*L2 */{Inn,Ni,Nt},/*L1 */{Inn,Ni,Nt}, 
 /*Mt */{Inn,Ji,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Inr,Ni,Nt},
 /*L3r*/{Inr,Ni,Nt},/*L2r*/{Inr,Ni,Nt},/*L1r*/{Inr,Ni,Nt},
 /*Mtr*/{Inr,Ni,Nt},
 /*Inf*/{Inf,Ni,Nt},
 /*Lvf*/{Inf,Ni,Nt},
 /*L3f*/{Inf,Ni,Nt},/*L2f*/{Inf,Ni,Nt},/*L1f*/{Inf,Ni,Nt},
 /*Mtf*/{Inf,Ni,Nt}
 },

 { /* Gid_join :added by rhliu */
 /*Inn*/{Inn,Ni,Nt},
 /*Lv */{Lv,Ni,Nt},
 /*L3 */{L3,Ni,Nt},/*L2 */{L2,Ni,Nt},/*L1 */{L1,Ni,Nt},
 /*Mt */{Mt,Ni,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Lvr,Ni,Nt},
 /*L3r*/{L3r,Ni,Nt},/*L2r*/{L2r,Ni,Nt},/*L1r*/{L1r,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Inf,Ni,Nt},
 /*Lvf*/{Lvf,Ni,Nt},
 /*L3f*/{L3f,Ni,Nt},/*L2f*/{L2f,Ni,Nt},/*L1f*/{L1f,Ni,Nt},
 /*Mtf*/{Mtf,Ni,Nt}
 },

 { /* Gid_leave :added by rhliu */
 /*Inn*/{Inn,Ni,Nt},
 /*Lv */{Lv,Ni,Nt},
 /*L3 */{L3,Ni,Nt},/*L2 */{L2,Ni,Nt},/*L1 */{L1,Ni,Nt},
 /*Mt */{Mt,Ni,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Lvr,Ni,Nt},
 /*L3r*/{L3r,Ni,Nt},/*L2r*/{L2r,Ni,Nt},/*L1r*/{L1r,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Inf,Ni,Nt},
 /*Lvf*/{Lvf,Ni,Nt},
 /*L3f*/{L3f,Ni,Nt},/*L2f*/{L2f,Ni,Nt},/*L1f*/{L1f,Ni,Nt},
 /*Mtf*/{Mtf,Ni,Nt}
 },

 { /* Gid_normal_operation, same as Gid_null for the Registrar */
 /*In */{Inn,Ni,Nt},
 /*Lv */{Lv, Ni,Nt},
 /*L3 */{L3, Ni,Nt},/*L2 */{L2, Ni,Nt},/*L1 */{L1, Ni,Nt},
 /*Mt */{Mt, Ni,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Lvr,Ni,Nt},
 /*L3r*/{L3r,Ni,Nt},/*L2r*/{L2r,Ni,Nt},/*L1r*/{L1r,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Inf,Ni,Nt},
 /*Lvf*/{Lvf,Ni,Nt},
 /*L3f*/{L3f,Ni,Nt},/*L2f*/{L2f,Ni,Nt},/*L1f*/{L1f,Ni,Nt},
 /*Mtf*/{Mtf,Ni,Nt}
 },

 { /* Gid_no_protocol, same as Gid_null for the Registrar */
 /*In */{Inn,Ni,Nt},
 /*Lv */{Lv, Ni,Nt},
 /*L3 */{L3, Ni,Nt},/*L2 */{L2, Ni,Nt},/*L1 */{L1, Ni,Nt},
 /*Mt */{Mt, Ni,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Lvr,Ni,Nt},
 /*L3r*/{L3r,Ni,Nt},/*L2r*/{L2r,Ni,Nt},/*L1r*/{L1r,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Inf,Ni,Nt},
 /*Lvf*/{Lvf,Ni,Nt},
 /*L3f*/{L3f,Ni,Nt},/*L2f*/{L2f,Ni,Nt},/*L1f*/{L1f,Ni,Nt},
 /*Mtf*/{Mtf,Ni,Nt}
 },

 { /* Gid_normal_registration */
 /*Inn*/{Inn,Ni,Nt},
 /*Lv */{Lv, Ni,Nt},
 /*L3 */{L3, Ni,Nt},/*L2 */{L2, Ni,Nt},/*L1 */{L1, Ni,Nt},
 /*Mt */{Mt, Ni,Nt},
 /*Inr*/{Inn,Ni,Nt},
 /*Lvr*/{Lv, Ni,Nt},
 /*L3r*/{L3, Ni,Nt},/*L2r*/{L2, Ni,Nt},/*L1r*/{L1, Ni,Nt},
 /*Mtr*/{Mt, Li,Nt},
 /*Inf*/{Inn,Ji,Nt},
 /*Lvf*/{Lv, Ji,Nt},
 /*L3f*/{L3, Ji,Nt},/*L2f*/{L2, Ji,Nt},/*L1f*/{L1, Ji,Nt},
 /*Mtf*/{Mt, Ni,Nt}
 },

 { /* Gid_fix_registration */
 /*Inn*/{Inr,Ni,Nt},
 /*Lv */{Lvr,Ni,Nt},
 /*L3 */{L3r,Ni,Nt},/*L2 */{L2r,Ni,Nt},/*L1 */{L1r,Ni,Nt},
 /*Mt */{Mtr,Ji,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{Lvr,Ni,Nt},
 /*L3r*/{L3r,Ni,Nt},/*L2r*/{L2r,Ni,Nt},/*L1r*/{L1r,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Inr,Ji,Nt},
 /*Lvf*/{Lvr,Ji,Nt},
 /*L3f*/{L3r,Ji,Nt},/*L2f*/{L2r,Ji,Nt},/*L1f*/{L1r,Ji,Nt},
 /*Mtf*/{Mtr,Ji,Nt}
 },

 { /* Gid_forbid_registration */
 /*Inn*/{Inf,Li,Nt},
 /*Lv */{Lvf,Li,Nt},
 /*L3 */{L3f,Li,Nt},/*L2 */{L2f,Li,Nt},/*L1 */{L1f,Li,Nt},
 /*Mt */{Mtf,Ni,Nt},
 /*Inr*/{Inr,Li,Nt},
 /*Lvr*/{Lvr,Li,Nt},
 /*L3r*/{L3r,Li,Nt},/*L2r*/{L2r,Li,Nt},/*L1r*/{L1r,Li,Nt},
 /*Mtr*/{Mtr,Li,Nt},
 /*Inf*/{Inf,Ni,Nt},
 /*Lvf*/{Lvf,Ni,Nt},
 /*L3f*/{L3f,Ni,Nt},/*L2f*/{L2f,Ni,Nt},/*L1f*/{L1f,Ni,Nt},
 /*Mtf*/{Mtf,Ni,Nt}
 } 

};

/******************************************************************************
* GIDTT : GID PROTOCOL : APPLICANT TRANSMIT TABLE
*******************************************************************************/
Applicant_txtt_entry 
applicant_txtt[Number_of_applicant_states] =
{
 /*Va */{Aa, Jm,Jt},/*Aa */{Qa, Jm,Nt},/*Qa */{Qa, Nm,Nt},/*La */{Vo, Lm,Nt},
 /*Vp */{Aa, Jm,Jt},/*Ap */{Qa, Jm,Nt},/*Qp */{Qp, Nm,Nt},
 /*Vo */{Vo, Nm,Nt},/*Ao */{Ao, Nm,Nt},/*Qo */{Qo, Nm,Nt},/*Lo */{Vo, Nm,Nt},
 /*Von*/{Von,Nm,Nt},/*Aon*/{Aon,Nm,Nt},/*Qon*/{Qon,Nm,Nt}
};



/******************************************************************************
* GIDTT : GID PROTOCOL : REGISTRAR LEAVE TIMER TABLE
*******************************************************************************/
Registrar_leave_timer_entry
registrar_leave_timer_table[Number_of_registrar_states] =
{
 /*Inn*/{Inn, Ni,Nt},
 /*Lv */{L3, Ni,Lt},/*L3 */{L2, Ni,Lt},/*L2 */{L1, Ni,Lt},/*L1 */{Mt, Li,Nt},
 /*Mt */{Mt, Ni,Nt},
 /*Inr*/{Inr,Ni,Nt},
 /*Lvr*/{L3r,Ni,Nt},/*L3r*/{L2r,Ni,Nt},/*L2r*/{L1r,Ni,Nt},/*L1r*/{Mtr,Ni,Nt},
 /*Mtr*/{Mtr,Ni,Nt},
 /*Inf*/{Inf,Ni,Nt},
 /*Lvf*/{L3f,Ni,Lt},/*L3f*/{L2f,Ni,Lt},/*L2f*/{L1f,Ni,Lt},/*L1f*/{Mtf,Ni,Lt},
 /*Mtf*/{Mtf,Ni,Nt}
};

/******************************************************************************
* GIDTT : GID PROTOCOL : STATE REPORTING TABLES
*******************************************************************************/
Gid_applicant_state applicant_state_table[Number_of_applicant_states] =
{
 /*Va */Very_anxious,/*Aa */Anxious,/*Qa */Quiet,/*La */Leaving,
 /*Vp */Very_anxious,/*Ap */Anxious,/*Qp */Quiet,
 /*Vo */Very_anxious,/*Ao */Anxious,/*Qo */Quiet,/*Lo */Leaving,
 /*Von*/Very_anxious,/*Aon*/Anxious,/*Qon*/Quiet
};

Gid_applicant_mgt applicant_mgt_table[Number_of_applicant_states] =
{
 /*Va */Normal,/*Aa */Normal,/*Qa */Normal,
 /*La */Normal,
 /*Vp */Normal,/*Ap */Normal,/*Qp */Normal,
 /*Vo */Normal,/*Ao */Normal,/*Qo */Normal,
 /*Lo */Normal,
 /*Von*/No_protocol,/*Aon*/No_protocol,/*Qon*/No_protocol
};

Gid_registrar_state registrar_state_table[Number_of_registrar_states] =
{
 /*Inn*/In,
 /*Lv */Leave,/*L3 */Leave,/*L2 */Leave,/*L1 */Leave,/*Mt */Empty,
 /*Inr*/In,
 /*Lvr*/Leave,/*L3r*/Leave,/*L2r*/Leave,/*L1r*/Leave,/*Mtr*/Empty,
 /*Inf*/In,
 /*Lvf*/Leave,/*L3f*/Leave,/*L2f*/Leave,/*L1f*/Leave,/*Mtf*/Empty
};

Gid_registrar_mgt registrar_mgt_table[Number_of_registrar_states] =
{
 /*Inn*/Normal_registration,
 /*Lv */Normal_registration,/*L3 */Normal_registration,
 /*L2 */Normal_registration,/*L1 */Normal_registration,
 /*Mt */Normal_registration,
 /*Inr*/Registration_fixed,
 /*Lvr*/Registration_fixed,/*L3r*/Registration_fixed,
 /*L2r*/Registration_fixed,/*L1r*/Registration_fixed,
 /*Mtr*/Registration_fixed,
 /*Inf*/Registration_forbidden,
 /*Lvf*/Registration_forbidden,/*L3f*/Registration_forbidden,
 /*L2f*/Registration_forbidden,/*L1f*/Registration_forbidden,
 /*Mtf*/Registration_forbidden
};

Tbool registrar_in_table[Number_of_registrar_states] =
{
 /*Inn*/True,/*Lv */False,/*L3 */False,/*L2 */False,/*L1 */False,
 /*Mt */False,
 /*Inr*/True,/*Lvr*/True,/*L3r*/True,/*L2r*/True,/*L1r*/True,
 /*Mtr*/True,
 /*Inf*/False,/*Lvf*/False,/*L3f*/False,/*L2f*/False,/*L1f*/False,
 /*Mtf*/False
};


/******************************************************************************
* GIDTT : GID PROTOCOL : RECEIVE EVENTS, USER REQUESTS, & MGT PROCESSING
*******************************************************************************/
/* Handles receive events and join or leave requests. */

Gid_event gidtt_event(Gid *my_port, Gid_machine *machine, Gid_event event)
{ 
 Applicant_tt_entry *atransition;
 Registrar_tt_entry *rtransition;
 Gid_machine old_machine;

#if 0
  if(gvrp_debug_level & GVRP_DEBUG_GEN)
   printk(" gidtt_event: port_no=%d event=%d Old: applicant=%d registrar=%d \n",my_port->port_no,event,machine->applicant,machine->registrar);
#endif   

 old_machine=*machine; //save original machine state
 
 atransition = &applicant_tt[event][machine->applicant];
 rtransition = &registrar_tt[event][machine->registrar];

 machine->applicant = atransition->new_app_state;
 machine->registrar = rtransition->new_reg_state;

#if 0
  if(gvrp_debug_level & GVRP_DEBUG_GEN)
   printk(" gidtt_event: port_no=%d event=%d New: applicant=%d registrar=%d \n",my_port->port_no,event,machine->applicant,machine->registrar);
#endif      

 if ((event == Gid_join ) && (atransition->cstart_join_timer))
   my_port->cschedule_tx_now = True;

  
 my_port->cstart_join_timer = my_port->cstart_join_timer
                             || atransition->cstart_join_timer;
 

 my_port->cstart_leave_timer = my_port->cstart_leave_timer
                             || rtransition->cstart_leave_timer;

 //reference to Table 12-4 Registrar state table in 802.1d spec.
 if( ((old_machine.registrar==Lv) ||
     (old_machine.registrar==L3) ||
     (old_machine.registrar==L2) ||
     (old_machine.registrar==L1)) &&
     (event==Gid_rcv_joinempty || event==Gid_rcv_joinin) )
 {
   systime_stop_timer( &my_port->sLeave_timer);
   my_port->leave_timer_running=False;
 }
 
 switch (rtransition->indications)
 {
  case Ji: 
        return(Gid_join);
	break;
  case Li: 
  	return(Gid_leave); 
	break;
  case Ni:
  default: 
        return(Gid_null);
 }
}

/* Returns True if the Registrar is in, or if registration is fixed.*/
// fixed registration is always in the IN state
Tbool gidtt_in(Gid_machine *machine)
{
 return(registrar_in_table[machine->registrar]);
}


/******************************************************************************
* GIDTT : GID PROTOCOL TRANSITION TABLES : TRANSMIT MESSAGES
*******************************************************************************/

Gid_event gidtt_tx(Gid *my_port,Gid_machine *machine)
{
 Tuint32 msg=0;

 msg = applicant_txtt[machine->applicant].msg_to_transmit;
 my_port->cstart_join_timer = my_port->cstart_join_timer
	|| applicant_txtt[machine->applicant].cstart_join_timer;
 
 machine->applicant=applicant_txtt[machine->applicant].new_app_state;//added by rhliu
    
 
 switch (msg)
 {
  case Jm: 
      return (gidtt_in(machine) ? Gid_tx_joinin : Gid_tx_joinempty);
      break;
  case Lm: 
      return Gid_tx_leaveempty;
      break;
  case Em: 
      return(Gid_tx_empty);
      break;
  case Nm:
  default: 
      return(Gid_null);
 }
}

/******************************************************************************
* GIDTT : GID PROTOCOL TRANSITION TABLES : LEAVE TIMER PROCESSING
*******************************************************************************/
Gid_event gidtt_leave_timer_expiry(Gid *my_port,Gid_machine *machine)
{
 
 Registrar_leave_timer_entry *rtransition;
 rtransition = &registrar_leave_timer_table[machine->registrar];
 
 machine->registrar = rtransition->new_reg_state;

 my_port->cstart_leave_timer = my_port->cstart_leave_timer
                              || rtransition->cstart_leave_timer;

 return((rtransition->leave_indication == Li) ? Gid_leave : Gid_null);
}

/******************************************************************************
* GIDTT : GID PROTOCOL TRANSITION TABLES : STATE REPORTING
*******************************************************************************/
/*
* Returns False if the Registrar is Normal registration, Empty, and the
* Application is Normal membership, Very Anxious Observer.
*/
Tbool gidtt_machine_active(Gid_machine *machine)
{
 if ( (machine->applicant == Vo) && (machine->registrar == Mt))
    return(False);
 else
    return(True);
}

/*
* Reports the GID machine state : Gid_applicant_state,
* Gid_applicant_mgt, Gid_registrar_state, Gid_registrar_mgt.
*/
void gidtt_states(Gid_machine *machine, Gid_states *state)
{
 state->applicant_state = applicant_state_table[machine->applicant];//V/A/Q/Leaving
 state->applicant_mgt = applicant_mgt_table[machine->applicant]; //normal/no_protocol 
 state->registrar_state = registrar_state_table[machine->registrar];//In/leave/empty
 state->registrar_mgt = registrar_mgt_table[machine->registrar];// normal/fix/forbidden
}


/*
* Reports the state machine of attributes 
*/
void gidtt_machine_info(Garp *application, Gid_machine  garp_machine[])
{
  Tuint32	i=0,j=0;
  Gid		*my_gid;

  for(i=1; i<= application->last_gid_used && application->gid !=NULL ;i++) {
          
	  for(j=0 ; j < MAX_LOGIC_PORT ; j++) {
  	      if(!gid_find_port( application->gid, j , &my_gid))  {
		  garp_machine[(i-1)*MAX_LOGIC_PORT+j].applicant=Vo;
		  garp_machine[(i-1)*MAX_LOGIC_PORT+j].registrar=Mt;
	      } else {
	          garp_machine[(i-1)*MAX_LOGIC_PORT+j].applicant=my_gid->machines[i].applicant;
		  garp_machine[(i-1)*MAX_LOGIC_PORT+j].registrar=my_gid->machines[i].registrar;
	      }
	  }
  }

  //end of mark
  garp_machine[(i-1)*MAX_LOGIC_PORT].applicant=GID_MACHINE_MAX;
  garp_machine[(i-1)*MAX_LOGIC_PORT].registrar=GID_MACHINE_MAX;

}
