/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  THIS IS AN UNPUBLISHED WORK WHICH CONTAINS CONFIDENTIAL INFORMATION
 *  FROM CCL, ITRI.  NO PART OF THIS WORK MAY BE USED IN ANY WAY WITHOUT
 *  THE PRIOR WRITTEN PERMISSION.  ANY UNAUTHORIZED USE COULD SUBJECT THE
 *  PERPETRATOR TO CRIMINAL AND CIVIL LIABILITY.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************/

// ********************************************
// Name   : gid.c
// Author : Liu, Ren Hao (N300, CCL/ITRI)
// Date   : 2003/06/26
// Note   : GARP Information Declaration 
// ********************************************

#include "common.h"
#include "garp_common.h"

#ifdef __GVRP__
#include "gvrp_common.h"
extern   Gvr *my_gvr;
#endif

#ifdef __GMRP__
#include "gmrp_common.h"
extern   Gmr *my_gmr;
extern   TstGmrpMcst 	astStaticMcstGrp[GMRP_DB_SIZE];
#endif


TstGarpTrk astTrkInfoDB[8];

extern Tuint32		Gid_default_join_time;  
extern Tuint32 		Gid_default_leave_time;
extern Tuint32 		Gid_default_hold_time;  
extern Tuint32 		Gid_default_leaveall_time; 
extern Tuint32 		Gid_leaveall_count;
extern Tuint8 		aucGvrpPortEbl[MAX_LOGIC_PORT];
extern Tuint8		aucGmrpPortEbl[MAX_LOGIC_PORT];
extern TstLPortMask 	astStaticVlanGrp[4096];
extern garp_statistics  gvrp_statistics[MAX_LOGIC_PORT],gmrp_statistics[MAX_LOGIC_PORT];
extern Tbool 		gvrp_enable,gmrp_enable;

/******************************************************************************
* GID : GARP INFORMATION DISTRIBUTION PROTOCOL : CREATION, DESTRUCTION
*******************************************************************************/

/*Creates a new instance of GID.*/
static Tbool gid_create_gid(Garp *application, Tuint32 port_no, Gid **gid)
{ 
 Gid *my_port;
 Tuint32 index=0;
 
 if (!sysmalloc(sizeof(Gid), (void **)&my_port))
	goto gid_creation_failure;

 my_port->application 		= application;
 my_port->port_no 		= port_no;
 my_port->next_in_port_ring 	= my_port;
 my_port->next_in_connected_ring= my_port;
 my_port->is_enabled		= False;
 my_port->is_connected 		= False;
 my_port->is_point_to_poTuint32 = True;
 my_port->cschedule_tx_now 	= False;
 my_port->cstart_join_timer	= False;
 my_port->cstart_leave_timer 	= False;
 my_port->tx_now_scheduled 	= False;
 my_port->join_timer_running	= False;
 my_port->leave_timer_running	= False;
 my_port->hold_tx 		= False;
 my_port->join_timeout 		= Gid_default_join_time;
 my_port->hold_timeout 		= Gid_default_hold_time;
 
 my_port->sLeaveAll_timer.data=(Tuint32)&my_port->sLa_parm;
 my_port->sLeave_timer.data=(Tuint32)&my_port->sL_parm;
 my_port->sJoin_timer.data=(Tuint32)&my_port->sJi_parm;
 my_port->sHold_timer.data=(Tuint32)&my_port->sH_parm;
 
 if (!sysmalloc(sizeof(Gid_machine)*(application->max_gid_index + 2),(void **)&my_port->machines)){
	goto gid_mcreation_failure;
 }

 //setting applicant/registrar initial state
 for(index=0;index < application->max_gid_index;index++) {
   my_port->machines[index].applicant= Vo;
   my_port->machines[index].registrar= Mt;
 }

 my_port->leaveall_countdown = Gid_leaveall_count;
 my_port->leaveall_timeout_n = Gid_default_leaveall_time/Gid_leaveall_count;

 systime_start_timer(my_port->application->process_id,gid_leaveall_timer_expired,
          	     my_port->port_no,my_port->leaveall_timeout_n,&my_port->sLeaveAll_timer);
 
 my_port->tx_pending = True; 
 my_port->last_transmitted = application->last_gid_used;
 my_port->last_to_transmit = application->last_gid_used;
 my_port->untransmit_machine = application->max_gid_index;//changed by rhliu
 *gid = my_port; 

 return(True);
 
 gid_mcreation_failure: sysfree(my_port);
 gid_creation_failure: return(False);
}


//If attribute's registrar state not equal to Mt state -> return True
Tbool gid_registered_here(Gid *gid, Tuint32 gid_index)
{
  if(gid->machines[gid_index].registrar != Mt){
     return (True);
  }
  else {
     return (False);
  }
}



/*
* Destroys the instance of GID, releasing previously allocated space.
* Sends leave indications to the application for previously registered attributes.
*/
static void gid_destroy_gid(Gid *gid)
{ 
 Tuint32 gid_index;
   
 // unregister all attributes maintained by this gid
 for (gid_index = 1; gid_index <= gid->application->last_gid_used; gid_index++) {
    if (gid_registered_here(gid, gid_index)){
        gid->application->leave_indication_fn(gid->application,gid, gid_index);
    }
 }
   
 sysfree(gid->machines);
 sysfree(gid);
}

/* Adds new gid to the next_in_port_ring.*/
/* ex: existing_port ->> existing_next_port
       => existing_port ->> new_port ->> existing_next_port */
static Gid *gid_add_port(Gid *existing_ports, Gid *new_port)
{ 
 Gid *prior;
 Gid *next;
 Tuint32 new_port_no;

 if (existing_ports != NULL) {
 
  new_port_no = new_port->port_no;
  next = existing_ports;

  for(;;) {
  
    prior = next; 
    next = prior->next_in_port_ring;
      
    if (prior->port_no <= new_port_no) {
     if ((next->port_no <= prior->port_no)|| (next->port_no > new_port_no)){
        break;
     }
    }

    else { /* if (prior_>port_no > new_port_no) */
     if ((next->port_no <= prior->port_no)&& (next->port_no > new_port_no)){ 
         break;
     }
    }

  }//for

  if(prior->port_no == new_port_no) {
      syserr_panic();
  }
  
  prior->next_in_port_ring = new_port;
  new_port->next_in_port_ring = next;
 }//if

 new_port->is_enabled = True;
 return(new_port);
}
 


/* Remove my_port from the next_in_port_ring.*/ 
// prior = The gid before my_port
static Gid *gid_remove_port(Gid *my_port)
{
 Gid *prior;
 Gid *next;
 
 prior = my_port;

 // find prior gid
 while((next = prior->next_in_port_ring) != my_port){
        prior = next;
 }

 prior->next_in_port_ring = my_port->next_in_port_ring;
 
 if (prior == my_port){
    return(NULL);
 }
 else {
    return(prior);
 }

}



/* Is port already in the port ring?
No-> 1)create gid and add into port ring
     2)call added port function
*/
Tbool gid_create_port(Garp *application, Tuint32 port_no)
{
 Gid *my_port;

 if(application==NULL) {
   return (False);
 }
 
 if (!gid_find_port(application->gid, port_no, &my_port)) {
  if(gid_create_gid(application, port_no, &my_port)) {
     application->gid = gid_add_port(application->gid, my_port);
     application->added_port_fn(application, port_no);
     return(True);
  }
 }

 return(False);
}


/* Is port already in the port ring?
yes-> 1)destroy gid and remove from port ring
      2)call removed port function
*/
void gid_destroy_port(Garp *application, Tuint32 port_no)
{
 Gid *my_port;
 
 if (gid_find_port(application->gid, port_no, &my_port)) {
   gip_disconnect_port(application, port_no);// remove gid GIP propagation ring
   application->gid = gid_remove_port(my_port);//remove gid from next_in_port_ring
   gid_destroy_gid(my_port); // destroys the instance of GID
 }
 
}
     
 

/******************************************************************************
* GID : GARP INFORMATION DISTRIBUTION PROTOCOL : USEFUL FUNCTIONS
*******************************************************************************/
/*find gid based on port_no*/
Tbool gid_find_port(Gid *first_port, Tuint32 port_no, Gid **gid)
{
 Gid *next_port = first_port;

 if(first_port==NULL){
   return (False);
 }
  
 while (next_port->port_no != port_no) {
    if ((next_port = next_port->next_in_port_ring) == first_port){
         return(False);
    }
 }
 
 *gid = next_port; 
 return(True);
}
 

/* Get this_port's next port form next_in_port_ring*/
Gid *gid_next_port(Gid *this_port)
{
  return(this_port->next_in_port_ring);
}


/******************************************************************************
* GID : GARP INFORMATION DISTRIBUTION PROTOCOL : MGT
*******************************************************************************/
/* Get applicant_state, applicant_mgt, registrar_state, registrar_mgt
   variable from machines*/
/* There are many applicant and registrat states in the GID*/
/* each state has a index number*/
void gid_read_attribute_state(Gid *my_port, Tuint32 index, Gid_states *state)
{ 
  gidtt_states(&my_port->machines[index], state);
}


/*
* Changes the attributes management state on my_port. The directive
* can be Gid_normal_operation, Gid_no_protocol, Gid_normal_registration,
* Gid_fix_registration, or Gid_forbid_registration. If the change in
* management state causes a leave indication, this is sent to the user
* and propagated to other ports.
*/

/*
* Further management functions, including disabling and enabling GID ports,
* are to be provided. A significant part of the purpose of this reference
* implementation is to facilitate the unambiguous specification of such
* management operations.
*/

void gid_manage_attribute(Gid *my_port, Tuint32 index, Gid_event directive)
{ 
 Gid_machine *machine;
 Gid_event event;
 machine = &my_port->machines[index];
 event = gidtt_event(my_port, machine, directive);

 if (event == Gid_join) {
  my_port->application->join_indication_fn(my_port->application,my_port, index);
  gip_propagate_join(my_port, index);
 }

 else if (event == Gid_leave) {
  my_port->application->leave_indication_fn(my_port->application,my_port, index);
  gip_propagate_leave(my_port, index);
 } 

}

// find gid_index which has inactive machine
// all of mahines have to be Vo/Mt in every GID
Tbool gid_find_unused(Garp *application, Tuint32 from_index,Tuint32 *found_index)
{
 Tuint32 gid_index;
 Gid *check_port;
 
 gid_index = from_index;
 check_port = application->gid;


 for (;;) { // Is attribute in this gid active?
  if (gidtt_machine_active(&check_port->machines[gid_index])) {
    if(gid_index++ > application->last_gid_used){
        return(False);
    }

    check_port = application->gid;
  } 
  else if ((check_port = check_port->next_in_port_ring)== application->gid) {// Exit loop when go through port ring
     *found_index = gid_index;
     return (True);
  } 
 
 }//for

}

/******************************************************************************
* GID : GARP INFORMATION DISTRIBUTION PROTOCOL : EVENT PROCESSSING
*******************************************************************************/
/* called by leaveall_timer! or rleaveall */
void gid_leaveall(Gid *my_port)
{ 
 Tuint32 i;
 Garp *application=my_port->application;

 for (i = 1; i <= application->last_gid_used; i++) { //skip gid_index0
 (void)gidtt_event(my_port, &my_port->machines[i], Gid_rcv_leaveall);
 }

}


void gid_rcv_leaveall(Gid *my_port)
{
  if(my_port->leaveall_countdown != 0) {
    my_port->leaveall_countdown = Gid_leaveall_count;
    gid_leaveall(my_port);
  }
}


void gid_rcv_msg(Gid *my_port, Tuint32 index, Gid_event msg)
{
 Gid_machine *machine;
 Gid_event event;
 machine = &my_port->machines[index];
 
 event = gidtt_event(my_port, machine, msg);
 
 if (event == Gid_join) {
  my_port->application->join_indication_fn(my_port->application,my_port, index);
  gip_propagate_join(my_port, index);
 }
 else if (event == Gid_leave) {
  my_port->application->leave_indication_fn(my_port->application,my_port, index);
  gip_propagate_leave(my_port, index);
 }
}



void gid_join_request(Gid *my_port, Tuint32 gid_index)
{
 (void)(gidtt_event(my_port, &my_port->machines[gid_index], Gid_join));
}



void gid_leave_request(Gid *my_port, Tuint32 gid_index)
{
 (void)(gidtt_event(my_port, &my_port->machines[gid_index], Gid_leave));
}


// Is registrar state of this attribute in the IN state?
// yes: return True
// No: return False
Tbool gid_registrar_in(Gid_machine *machine)
{
 return(gidtt_in(machine));
}


/******************************************************************************
* GID : GARP INFORMATION DISTRIBUTION PROTOCOL : RECEIVE PROCESSING
*******************************************************************************/
/*
* If a GID instance for this application and port number is found and is
* enabled, pass the PDU to the application, which will parse it (using the
* applications own PDU formatting conventions) and call gid_rcv_msg() for
* each of the conceptual GID message components read from the PDU. Once
* the application is finished with the PDU, call gip_do_actions() to start
* timers as recorded in the GID scratchpad for this port and any ports to
* which it may have propagated joins or leaves.
*
* Finally release the received pdu.
*/

void gid_rcv_pdu(void *app, Tuint32 port_no, void *pdu)
{ 
 Gid *my_port;
 Garp *application=(Garp *)app;
 Tuint32 return_value;

 //find gid of this port
 if(gid_find_port(application->gid, port_no, &my_port)) {
  if(my_port->is_enabled) {

     return_value=application->receive_fn(NULL, my_port, pdu);
   
     if(return_value!=False){
        gip_do_actions(my_port);
     }
  }
 }

}



/******************************************************************************
* GID : GARP INFORMATION DISTRIBUTION PROTOCOL : TRANSMIT PROCESSSING
*******************************************************************************/
/*
* Check to see if a leaveall should be sent; if so, return Gid_tx_leaveall;
* otherwise, scan the GID machines for messsages that require transmission.
*
* Machines will be checked for potential transmission starting with the
* machine following last_transmitted and up to and including last_to_transmit.
* If all machines have transmitted, last_transmitted equals last_to_transmit
* and tx_pending is False (in this case tx_pending distinguished the
* case of all machines are yet to be checked for transmission from all have
* been checked.)
*
* If tx_pending is True and all machines are yet to be checked, transmission
* will start from the machine with GID index 0, rather than from immediately
* following last_transmitted.
*/

Gid_event gid_next_tx(Gid *my_port, Tuint32 *index)
{ 
 Tuint32 check_index;
 Tuint32 stop_after;
 Gid_event msg;

 if (my_port->hold_tx) {
    return(Gid_null);
 }
 
 if (my_port->leaveall_countdown == 0) {
 
  my_port->leaveall_countdown = Gid_leaveall_count;
  systime_start_timer(my_port->application->process_id,gid_leaveall_timer_expired,
  		      my_port->port_no,my_port->leaveall_timeout_n,&my_port->sLeaveAll_timer);
  
  return(Gid_tx_leaveall);
 }
 
 if (!my_port->tx_pending){ 
   return(Gid_null); 
 }
 
 //changed by rhliu
 check_index = my_port->last_to_transmit;
 stop_after = my_port->application->last_gid_used;
 

 if (stop_after < check_index){
   stop_after = my_port->application->last_gid_used;
 }
 
 for( ; ;check_index++) {
  
  if (check_index > stop_after) {
      my_port->tx_pending = False;
      return(Gid_null);
  }//if

  
  if ((msg = gidtt_tx(my_port, &my_port->machines[check_index]))!= Gid_null) {
   *index = my_port->last_transmitted = check_index;
   my_port->last_to_transmit=check_index+1; //added by rhliu
   
   my_port->machines[my_port->untransmit_machine].applicant =
   my_port->machines[check_index].applicant;
   my_port->tx_pending = (check_index != stop_after); //changed by rhliu check_index==stop_after
   return(msg);
  }
 
 }  /* end for(;;) */
}



void gid_untx(Gid *my_port)
{
 my_port->machines[my_port->last_transmitted].applicant =
	 my_port->machines[my_port->untransmit_machine].applicant;

 if (my_port->last_transmitted == 0){
    my_port->last_transmitted = my_port->application->last_gid_used;
 }
 else{
  my_port->last_transmitted--;
 }

 my_port->tx_pending = True;
 
}

/******************************************************************************
* GID : GARP INFORMATION DISTRIBUTION PROTOCOL : TIMER PROCESSING
*******************************************************************************/
/*
* Carries out scratchpad actions accumulated in this invocation of GID,
* and outstanding immediate transmissions and join timer starts that
* have been delayed by the operation of the hold timer. Note the way in
* which the hold timer works here. It could have been specified just to
* impose a minimum spacing on transmissions - and run in parallel with the
* join timer - with the effect that the longer of the hold timer and actual
* join timer values would have determined the actual transmission time.
* This approach was not taken because it could have led to bunching
* transmissions at the hold time.
*
* The procedure restarts the join timer if there are still transmissions
* pending (if leaveall_countdown is zero. a Leaveall is to be sent; if
* tx_pending is True, individual machines may have messages to send.)
*/
void gid_do_actions(Gid *my_port)
{ 
 
 if (my_port->cstart_join_timer) {
  my_port->last_to_transmit = my_port->last_transmitted = 1;//added by rhliu 
  my_port->tx_pending = True;
  my_port->cstart_join_timer = False; 
 }

  
 if (!my_port->hold_tx) {

  if (my_port->cschedule_tx_now) {
  
     if (!my_port->tx_now_scheduled) {
        if(my_port->join_timer_running) {
           systime_stop_timer(&my_port->sJoin_timer); 
           my_port->join_timer_running=False;
	}
      
	systime_schedule(my_port->application->process_id,
                       gid_join_timer_expired,
                       my_port->port_no,&my_port->sJoin_timer);
     }
      
     my_port->cschedule_tx_now = False;
  }
  else if((my_port->tx_pending || (my_port->leaveall_countdown == 0))
	   &&(!my_port->join_timer_running)) {
     systime_start_random_timer(my_port->application->process_id,
           gid_join_timer_expired,my_port->port_no, my_port->join_timeout ,&my_port->sJoin_timer);
 
     my_port->join_timer_running = True;
  }//else if 

 }//if


 if (my_port->cstart_leave_timer && (!my_port->leave_timer_running)) {
 
     systime_start_timer(my_port->application->process_id,gid_leave_timer_expired,
    	 	         my_port->port_no, Gid_default_leave_time/3 ,&my_port->sLeave_timer);

     my_port->leave_timer_running = True; //added by rhliu
 }

 my_port->cstart_leave_timer = False;

}

void gid_leave_timer_expired(Tuint32 data)
{
 Gid *my_port;
 Tuint32 gid_index;
 in_parm *parm=(in_parm *)data;
 Garp *application=(Garp *)parm->application;
 Tuint32 port_no=parm->port_no;

  
  if (gid_find_port(application->gid, port_no, &my_port)) {
  
      my_port->leave_timer_running = False; //added by rhliu 

      for(gid_index=1; gid_index <= my_port->application->last_gid_used;gid_index++) {//changed by rhliu 

	      if(gidtt_leave_timer_expiry(my_port,&my_port->machines[gid_index])== Gid_leave) {

		      my_port->application->leave_indication_fn(my_port->application, my_port, gid_index);
		      gip_propagate_leave(my_port, gid_index);

		      //if rcv attr has already timeout..delete it:added by rhliu 
		      if( my_port->application->gip[gid_index] <=0 ) {
#ifdef __GVRP__     
			      if(application->process_id==GVRP_PROCESS_ID && gid_index >1 ){
			         gvd_delete_entry(((Gvr *)application)->gvd,gid_index);
			      }
#endif

#ifdef __GMRP__
			      if(application->process_id==GMRP_PROCESS_ID && gid_index > 2){
				 gmd_delete_entry(((Gmr *)application)->gmd,gid_index);
			      }
#endif
			      my_port->application->gip[gid_index]=0; //reset
		      }
	      }//if
      }//for 
  } //if
}

void gid_leaveall_timer_expired(Tuint32 data)
{
 Gid *my_port;
 in_parm *parm=(in_parm *)data;
 Garp *application=(Garp *)parm->application;
 Tuint32 port_no=parm->port_no;

 if (gid_find_port(application->gid, port_no, &my_port)) {
  
   if (my_port->leaveall_countdown > 1){
      
       my_port->leaveall_countdown--;
       systime_start_timer(my_port->application->process_id,gid_leaveall_timer_expired,
               my_port->port_no,my_port->leaveall_timeout_n,&my_port->sLeaveAll_timer);//added by rhliu
   }
   else {

       if ((!my_port->join_timer_running) && (!my_port->hold_tx)) {
	       gid_leaveall(my_port); 
	       my_port->leaveall_countdown = 0; 
 
	       systime_start_random_timer(my_port->application->process_id,
			       gid_join_timer_expired,my_port->port_no,
			       my_port->join_timeout,&my_port->sJoin_timer);
      
	       my_port->cstart_join_timer = True;
	       my_port->join_timer_running = True;
       }
   }//else
 }//if
}

void gid_join_timer_expired(Tuint32 data)
{
 Gid *my_port;
 
 in_parm *parm=(in_parm *)data;
 Garp *application=parm->application;
 Tuint32 port_no=parm->port_no;
  
 if (gid_find_port(application->gid, port_no, &my_port)) {
 
   if (my_port->is_enabled) {

   my_port->join_timer_running=False;//added by rhliu
   application->transmit_fn(application, my_port);  

   if(!my_port->hold_timer_running){
     systime_start_timer(my_port->application->process_id,gid_hold_timer_expired,
                       my_port->port_no,Gid_default_hold_time ,&my_port->sHold_timer);
  
     my_port->hold_timer_running=True; 
   }
  }
 }
 
}

void gid_hold_timer_expired(Tuint32 data)
{
 Gid *my_port;
 
 in_parm *parm=(in_parm *)data;
 Garp *application=parm->application;
 Tuint32 port_no=parm->port_no;
 
 
 if (gid_find_port(application->gid, port_no, &my_port)) {  
   my_port->hold_timer_running=False; 
   my_port->hold_tx = False;
   gid_do_actions(my_port);
 }

}

Tbool gid_port_enable(Garp *application, TstLPortMask *logic_pm)
{
     Gid 	    *my_gid;
     Tuint32 	    i;
 
     //save port gvrp/gmrp setting
     for(i=0;i < MAX_LOGIC_PORT; i++) {
          
	  //if(logic_pm->ulMask[i/32] & (0x01 << (i%32))){
	  if(K_LPortMaskGetPort(logic_pm, i)) {
#ifdef __GVRP__		
  	     if(application->process_id==GVRP_PROCESS_ID) {
	        aucGvrpPortEbl[i] = True;
	     }
#endif
#ifdef __GMRP__
	     if(application->process_id==GMRP_PROCESS_ID) {
		aucGmrpPortEbl[i] = True;
	     }
#endif
	  }
	  else {
#ifdef __GVRP__
  	      if(application->process_id==GVRP_PROCESS_ID) {
	         aucGvrpPortEbl[i] = False;
	      }
#endif
#ifdef __GMRP__
	      if(application->process_id==GMRP_PROCESS_ID) {
	         aucGmrpPortEbl[i] = False;
	      }
#endif
	  }

     }
	 
     // if gvrp/gmrp not enable, user can't enable gvrp/gmrp for some ports
     if(application->process_id==GVRP_PROCESS_ID && gvrp_enable==False) {
	return False;
     }
     else if (application->process_id==GMRP_PROCESS_ID && gmrp_enable==False) {
	return False;
     }

     for(i=0;i < MAX_LOGIC_PORT; i++) {
          //if(logic_pm->ulMask[i/32] & (0x01 << (i%32))){
	  if(K_LPortMaskGetPort(logic_pm, i)) {
		  gid_create_port( application,i);
          }
	  else {
                if(gid_find_port(application->gid, i , &my_gid)) {
		   gid_stop_all_timer(my_gid);
		   gid_destroy_port( application,i);           
		
#ifdef __GVRP__
		   if(application->process_id==GVRP_PROCESS_ID) {
		      memset(&gvrp_statistics[i],0,sizeof(garp_statistics)); 
		   }
#endif
#ifdef __GMRP__
	    	   if(application->process_id==GMRP_PROCESS_ID) {
		      memset(&gmrp_statistics[i],0,sizeof(garp_statistics)); 
		   }
#endif
		}
	  }
      }

      return True;
}


#ifdef __GMRP__
Tbool gid_add_edit_static_multicast(Mac_addr addr, TstLPortMask *stPortMask)
{
     Garp 	*application= &my_gmr->garp;
     Gid 	*my_gid;
     Tuint32 	gid_index;
     Tuint32 	i;


     if(my_gmr->gmd==NULL){
         return False;
     }

      if(!gmd_find_entry(my_gmr->gmd, addr ,&gid_index)) {
	
	gmd_create_entry(my_gmr->gmd, addr , &gid_index);
        for(i=0;i< MAX_LOGIC_PORT;i++) {
	
	    if(!gid_find_port( application->gid, i, &my_gid)){
               continue;
	    }
           
	    //if(stPortMask->ulMask[i/32] & (0x01<<(i%32)) ) {
	    if(K_LPortMaskGetPort(&stPortMask, i)) {
	       gid_machine_set(my_gid,gid_index,Va,Inr);
	       gip_propagate_join(my_gid,gid_index);
	    }
	}
      }
      else {
	    application->gip[gid_index]=0; //reset
	    for(i=0;i< MAX_LOGIC_PORT;i++) {
	    
               if(!gid_find_port( application->gid, i , &my_gid)){
                   continue;
	       }

	       //if(stPortMask->ulMask[i/32] & (0x01<<(i%32)) ) {
	       if(K_LPortMaskGetPort(&stPortMask, i)) {
	          gid_machine_set(my_gid,gid_index,Va,Inr);
	          gip_propagate_join(my_gid,gid_index);
	       }
	       else {
                  gid_machine_set(my_gid,gid_index,Vo,Mt);
	       }
	    }
      }

      return True;

}

Tbool gid_del_static_multicast(Mac_addr addr) 
{
      Garp *application= &my_gmr->garp;
      Gid *my_gid;
      Tuint32 gid_index;
      Tuint32 i;

      if(my_gmr->gmd==NULL){
         return False;
      }

      if(gmd_find_entry(my_gmr->gmd,addr ,&gid_index)){
        gmd_delete_entry(my_gmr->gmd, gid_index);
	application->gip[gid_index]=0; //reset
      }

      for(i=0;i< MAX_LOGIC_PORT;i++) {
          
	  if(!gid_find_port(  application->gid, i, &my_gid)){
              continue;
	  }
	  
	  gid_machine_set(my_gid,gid_index,Vo,Mt);
      }

      return True;
}

#endif

#ifdef __GVRP__
Tbool gid_add_edit_static_vlan(Tuint16 vid, TstLPortMask *pstLPortMask)
{
     Garp *application= &my_gvr->garp;
     Gid *my_gid;
     Tuint32 gid_index;
     Tuint32 i;

     if(my_gvr->gvd==NULL){
         return False;
     }

     if(!gvd_find_entry(my_gvr->gvd,vid,&gid_index)){
        gvd_create_entry(my_gvr->gvd, vid , &gid_index);

        for(i=0;i< MAX_LOGIC_PORT;i++)
	{
         if(!gid_find_port(application->gid, i , &my_gid)){
            continue;
	 }

 	 //if(pstLPortMask->ulMask[i/32] & (0x01<< (i%32) ) )
	 if(K_LPortMaskGetPort(pstLPortMask, i)) {
            gid_machine_set(my_gid,gid_index,Va,Inr);
            gip_propagate_join(my_gid,gid_index);
	 }
	}
     }
     else {
        application->gip[gid_index]=0; //reset
     
        for(i=0;i< MAX_LOGIC_PORT;i++)
	{
         if(!gid_find_port(application->gid, i , &my_gid)){
            continue;
	 }
    
         //if(pstLPortMask->ulMask[i/32] & (0x01<< (i%32) ) )
	 if(K_LPortMaskGetPort(pstLPortMask, i)) {
            gid_machine_set(my_gid,gid_index,Va,Inr);
            gip_propagate_join(my_gid,gid_index);
	 }
 	 else {
            gid_machine_set(my_gid,gid_index,Vo,Mt);
	 }
	}
     }
    
     reset_gvrp_db();
     set_gvrp_db_from_reg();
     return True;
 
}


Tbool gid_del_static_vlan(Tuint16 vid)
{
      Garp *application= &my_gvr->garp;
      Gid *my_gid;
      Tuint32 gid_index;
      Tuint32 i;

      if(my_gvr->gvd==NULL){
	 return False;
      }

      if(gvd_find_entry(my_gvr->gvd,vid ,&gid_index)) {
         gvd_delete_entry(my_gvr->gvd, gid_index);
	 application->gip[gid_index]=0; //reset
      }

      for(i=0;i<MAX_LOGIC_PORT;i++) {

          if(!gid_find_port(application->gid, i , &my_gid)){
             continue;
	  }

	  gid_machine_set(my_gid,gid_index,Vo,Mt);
      }

      reset_gvrp_db();
      set_gvrp_db_from_reg();
      return True;
}
#endif

Tbool gid_trunk_add(Garp *application, TstGarpTrk stTrkInfo )
{
	Gid 	*my_gid;
	Tuint32	i;
	Tbool   port_ebl=False;
	Tuint8  ucPortId = stTrkInfo.ucTrkId + MAX_PHY_N_CPU_PORT;
#ifdef __GVRP__
	Tuint16 vid;
#endif
	TstLPortMask	stLPortMask;

	astTrkInfoDB[stTrkInfo.ucTrkId].ucTrkId=stTrkInfo.ucTrkId;
	memcpy( &astTrkInfoDB[stTrkInfo.ucTrkId].stPortMask, &stTrkInfo.stPortMask,sizeof(stTrkInfo.stPortMask));

	//save static vlan/multicast setting
#ifdef __GVRP__
	if(application->process_id==GVRP_PROCESS_ID) {
   	   for(vid=0; vid < 4096; vid++) {
	       // for example: port1 and port2 are member port of vlan30
	       // if port1 and port2 become trunk1, we have to modify the portmask for vlan30
	       // (trun off port1 port2 bit and trun on trk1 bit)
	       //if(astStaticVlanGrp[vid].ulMask[0] & stTrkInfo.stPortMask.ulMask[0]) { 
	       K_LPortMaskAnd(&stLPortMask, &astStaticVlanGrp[vid], &stTrkInfo.stPortMask);
               if(!K_LPortMaskIsZero(&stLPortMask)) {
	          //astStaticVlanGrp[vid].ulMask[0] &= ~(stTrkInfo.stPortMask.ulMask[0]);	
                  K_LPortMaskClrPorts(&astStaticVlanGrp[vid], &stTrkInfo.stPortMask);
	          //astStaticVlanGrp[vid].ulMask[ucPortId/32] |= (0x01 << (ucPortId%32));	
                  K_LPortMaskSetPort(&astStaticVlanGrp[vid], ucPortId);
	       }
	   }
	}
#endif	
#ifdef __GMRP__	
	if(application->process_id==GMRP_PROCESS_ID) {
   	   for(i=0; i < GMRP_DB_SIZE; i++) {
               //if(astStaticMcstGrp[i].stPortMask.ulMask[0] & stTrkInfo.stPortMask.ulMask[0]) {
	       K_LPortMaskAnd(&stLPortMask, &astStaticMcstGrp[i].stPortMask, &stTrkInfo.stPortMask);
               if(!K_LPortMaskIsZero(&stLPortMask)) {
                  //astStaticMcstGrp[i].stPortMask.ulMask[0] &= ~(stTrkInfo.stPortMask.ulMask[0]);
                  K_LPortMaskClrPorts(&astStaticMcstGrp[i].stPortMask, &stTrkInfo.stPortMask);
                  //astStaticMcstGrp[i].stPortMask.ulMask[ucPortId/32] |= (0x01 << (ucPortId%32));
                  K_LPortMaskSetPort(&astStaticMcstGrp[i].stPortMask, ucPortId);
               }
           }
        }
#endif	

	//make sure what gid have to release or create based on trunk information
	//for example: port1 and port2 are normal port, if they become trunk1...
	//Step1: Relase port1's and port2's gid
	//Step2: Create trunk1's gid
	for(i=0;i < MAX_PORT; i++) {
          //if(stTrkInfo.stPortMask.ulMask[i/32] & (0x01<< (i%32) )){
	  if(K_LPortMaskGetPort(&stTrkInfo.stPortMask, i)) {
              if(gid_find_port(application->gid, i , &my_gid)) {
	         gid_stop_all_timer(my_gid);
                 gid_destroy_port( application,i);
		 port_ebl=True; //One of trunk's member port already enabled GVRP protocol
#ifdef __GVRP__
		 if(application->process_id==GVRP_PROCESS_ID) {
		    aucGvrpPortEbl[i] = False;
		 }
#endif
#ifdef __GMRP__
	  	 if(application->process_id==GMRP_PROCESS_ID) {
		    aucGmrpPortEbl[i] = False;
		 }
#endif
	      }
	  }
	}

	//for example: port1 enabled GVRP protocol, but port2 not enabled GVRP protocol
	//if port1 and port2 become trunk1, trunk1 will enable GVRP because port1 already
	//enable GVRP.
	if(port_ebl){
		gid_create_port( application,stTrkInfo.ucTrkId+MAX_PHY_N_CPU_PORT);
		
#ifdef __GVRP__
		if(application->process_id==GVRP_PROCESS_ID) {
		    aucGvrpPortEbl[stTrkInfo.ucTrkId+MAX_PHY_N_CPU_PORT] = True;
		}
#endif
#ifdef __GMRP__
		if(application->process_id==GMRP_PROCESS_ID) {
		    aucGmrpPortEbl[stTrkInfo.ucTrkId+MAX_PHY_N_CPU_PORT] = True;
		}
#endif
		return True;
	}
	
	return False;	
} 


Tbool gid_trunk_del(Garp *application,  Tuint8 trunk_id )
{
	Gid *my_gid;
	Tuint32	i;
	Tuint8  ucPortId = trunk_id + MAX_PHY_N_CPU_PORT;
#ifdef __GVRP__
	Tuint16 vid;
#endif

	//save static vlan/multicast setting
#ifdef __GVRP__	
	if(application->process_id==GVRP_PROCESS_ID) {
	   for(vid=0; vid < 4096; vid++) {
	       //if(astStaticVlanGrp[vid].ulMask[ucPortId/32] & (0x01 << (ucPortId%32) )) { 
               if(K_LPortMaskGetPort(&astStaticVlanGrp[vid], ucPortId)) {
	          //astStaticVlanGrp[vid].ulMask[0] |= astTrkInfoDB[trunk_id].stPortMask.ulMask[0];	
                  K_LPortMaskOr(&astStaticVlanGrp[vid], &astStaticVlanGrp[vid], &astTrkInfoDB[trunk_id].stPortMask);
	          //astStaticVlanGrp[vid].ulMask[ucPortId/32] &= ~(0x01 << (ucPortId%32));	
                  K_LPortMaskClrPort(&astStaticVlanGrp[vid], ucPortId);
	       }
	   }
	}
#endif
#ifdef __GMRP__
	if(application->process_id==GMRP_PROCESS_ID) {
   	   for(i=0; i < GMRP_DB_SIZE; i++) {
	       //if(astStaticMcstGrp[i].stPortMask.ulMask[ucPortId/32] & (0x01 << (ucPortId%32) )) { 
               if(K_LPortMaskGetPort(&astStaticMcstGrp[i].stPortMask, ucPortId)) {
	          //astStaticMcstGrp[i].stPortMask.ulMask[0] |= astTrkInfoDB[trunk_id].stPortMask.ulMask[0];	
                  K_LPortMaskOr(&astStaticMcstGrp[i].stPortMask, &astStaticMcstGrp[i].stPortMask, &astTrkInfoDB[trunk_id].stPortMask);
	          //astStaticMcstGrp[i].stPortMask.ulMask[ucPortId/32] &= ~(0x01 << (ucPortId%32));	
                  K_LPortMaskClrPort(&astStaticMcstGrp[i].stPortMask, ucPortId);
	       }
	   }
	}
#endif	


	//make sure what gid have to release or create based on trunk information
	//for example: port1 and port2 are trunk1's member port. if trunk1 be deleted..
	//Step1: Relase trunk1's gid
	//Step2: Create port1's and port2's gid
	if(gid_find_port(application->gid, trunk_id+MAX_PHY_N_CPU_PORT , &my_gid)) {
               gid_stop_all_timer(my_gid);
               gid_destroy_port( application,trunk_id+MAX_PHY_N_CPU_PORT);
	        
#ifdef __GVRP__
	    if(application->process_id==GVRP_PROCESS_ID) {
	       aucGvrpPortEbl[trunk_id+MAX_PHY_N_CPU_PORT] = False;
	    }
#endif
#ifdef __GMRP__
	    if(application->process_id==GMRP_PROCESS_ID) {
	       aucGmrpPortEbl[trunk_id+MAX_PHY_N_CPU_PORT] = False;
	    }
#endif            
	    for(i=0;i < MAX_PORT; i++) {
	       //if(astTrkInfoDB[trunk_id].stPortMask.ulMask[i/32] & (0x01<< (i%32) )){
	       if(K_LPortMaskGetPort(&astTrkInfoDB[trunk_id].stPortMask, i)){
	          gid_create_port( application,i);
		       
#ifdef __GVRP__
	          if(application->process_id==GVRP_PROCESS_ID) {
		     aucGvrpPortEbl[i] = True;
		  }
#endif
#ifdef __GMRP__
		  if(application->process_id==GMRP_PROCESS_ID) {
		     aucGmrpPortEbl[i] = True;
		  }
#endif
	       }
	    }
	
	    astTrkInfoDB[trunk_id].ucTrkId=0;
	    memset( &astTrkInfoDB[trunk_id].stPortMask, 0, sizeof(astTrkInfoDB[trunk_id].stPortMask));
	    return True;
	}
	
	return False;
}

// stop all timer for one gid
void gid_stop_all_timer(Gid *my_gid)
{
  systime_stop_timer(&my_gid->sLeaveAll_timer);
  systime_stop_timer(&my_gid->sLeave_timer);
  systime_stop_timer(&my_gid->sJoin_timer);
  systime_stop_timer(&my_gid->sHold_timer);

}

// set applicant and registrar state for one gid
void gid_machine_set(Gid *my_gid, Tuint32 gid_index, enum Applicant_states app, enum Registrar_states reg)
{
  my_gid->machines[gid_index].applicant= app;
  my_gid->machines[gid_index].registrar= reg;
}
