#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>

#include <asm/uaccess.h>

#include "ccl.h"
#include "ccl_mx.h"
#include "cclmx_netfilter.h"

#include "krnmac.h"
#include "krnmod.h"
#include "krnport.h"
#include "krnportmask.h"
#include "krnstp.h"
#include "krntrk.h"

/* CCLMX_PERIODIC_CHECK_PORT_LINK:
     1.	Constant refresh switch controller (by calling K_PortGetLink(),
	K_PortGetSpeedDuplex(), and K_PortGetFlowCtrl() ).
     2.	When port link goes down (by calling K_PortGetLink() ),
	clear dynamic MAC entries of that port.
     3. When trunk member port link goes down, reset its hash value
	to ensure traffic is not lost. */
#define	CCLMX_PERIODIC_CHECK_PORT_LINK

struct nf_sockopt_ops cclmx_sockopts =
{
list:		{NULL, NULL},
pf:		PF_INET,
set_optmin:	CCLMX_BASE_CTL,
set_optmax:	CCLMX_GET_MAX,
set:			do_cclmx_set_ctl,
compat_set: 	NULL,
get_optmin:	CCLMX_BASE_CTL,
get_optmax:	CCLMX_SET_MAX,
get:			do_cclmx_get_ctl,
compat_get: 	NULL,
use:		0,
cleanup_task:	NULL
};


unsigned int ccl_port_map[CCL_SLOT_NUM][MAX_PORT]= CCL_PORT_MAP;
unsigned int ccl_slot_port_num[CCL_SLOT_NUM] = CCL_SLOT_PORT_NUM;
TstPPortMask astPhyMaskTbl[MAX_TRK_PORT];

struct CCL_MX_reg
{
	unsigned char mac[6];
	unsigned char mask[6];

	int (*ccl_mx_callback)(struct sk_buff *skb); 
	struct CCL_MX_reg *next;	
};

int (*ccl_mx_hook)(struct sk_buff *skb) = NULL;
static struct CCL_MX_reg *header;

void (*gvrp_stp_port_forwarding)(int )=NULL;
void (*gvrp_stp_port_disabled)(int )=NULL;

void (*gmrp_stp_port_forwarding)(int )=NULL;
void (*gmrp_stp_port_disabled)(int )=NULL;

void (*igmp_stp_port_forwarding)(int )=NULL;
void (*igmp_stp_port_disabled)(int )=NULL;

void (*mstp_vlan_add)(int,unsigned long )=NULL;
void (*mstp_vlan_del)(int,unsigned long )=NULL;


MODULE_AUTHOR ("PinChuan Flash Liu <flash@itri.org.tw>");
MODULE_DESCRIPTION ("VIA 24+2G packet Ethernet driver Multiplexer");
MODULE_LICENSE("GPL");



int do_cclmx_get_ctl (struct sock *sk, int cmd, void *user, int *len)
{
	int i;
	TstPPortMask stPMask;
	TstLPortMask stLMask;
	Tuint8 p, sp, du;

	switch (cmd) {
	/* VVV for testing only VVV */
	case CCLMX_GET_TRK:
		for (i = 0; i < MAX_TRK_PORT; i++)
			printk ("trk %d: %08lx\n", i, astPhyMaskTbl[i].ulMask[0]);
		break;
	case CCLMX_GET_L2P:
		copy_from_user (&stLMask, user, sizeof (TstLPortMask));
		cclmx_LMask2PMask (&stPMask, &stLMask);
		printk ("logic: %08lx,%08lx -> phy: %08lx\n", stLMask.ulMask[1],
				stLMask.ulMask[0], stPMask.ulMask[0]);
		break;
	case CCLMX_GET_P2L:
		copy_from_user (&stPMask, user, sizeof (TstPPortMask));
		cclmx_PMask2LMask (&stLMask, &stPMask);
		printk ("phy: %08lx -> logic: %08lx,%08lx\n", stPMask.ulMask[0],
				stLMask.ulMask[1], stLMask.ulMask[0]);
		break;
	case CCLMX_GET_SPEED:
		copy_from_user (&p, user, sizeof (Tuint8));
		cclmx_PortGetSpeedDuplex (p, &sp, &du);
		printk ("p%d: sp=%d, du=%d\n", p, sp, du);
		break;
	/* ^^^ for testing only ^^^ */
	}
	return 0;
}


int do_cclmx_set_ctl (struct sock *sk, int cmd, void *user, unsigned int len)
{
	Tuint8 ucId;
	TstTrkMem stTrk;

	switch (cmd) {
	case CCLMX_SET_ADD_TRK:
		copy_from_user (&stTrk, user, sizeof(TstTrkMem));
		cclmx_InsTrk (stTrk.ucTrkId, stTrk.stPhyMask);
		break;
	case CCLMX_SET_DEL_TRK:
		copy_from_user (&ucId, user, sizeof(Tuint8));
		cclmx_DelTrk (ucId);
		break;
	}
	return 0;
}


int ccl_mx_register(char *mac, char *mask, int(*callback)(struct sk_buff *skb))
{
	struct CCL_MX_reg *ptr;
	int i;

	if(header == NULL)
	{
		header=(struct CCL_MX_reg *) kmalloc(sizeof(struct CCL_MX_reg), GFP_KERNEL);	
		if(header == NULL) // allocate fail
		{
			printk("CCL_MX allocate memory error !!\n");
			return 1;
		}
		for(i=0;i<6;i++) 
		{
			header->mac[i]=mac[i]; 
			header->mask[i]=mask[i];
		}
		header->ccl_mx_callback=callback;
		header->next=NULL;
	}	
	else
	{
		ptr=header;

		while(ptr!=NULL && ptr->next!=NULL)
		{
			ptr=ptr->next;	
		}

		ptr->next=(struct CCL_MX_reg *) kmalloc(sizeof(struct CCL_MX_reg), GFP_KERNEL);	
		if(ptr->next == NULL) // allocate fail
		{
			printk("CCL_MX allocate memory error !!\n");
			return 1;
		}
		for(i=0;i<6;i++) 
		{
			ptr->next->mac[i]=mac[i]; 
			ptr->next->mask[i]=mask[i];
		}
		ptr->next->ccl_mx_callback=callback;
		ptr->next->next=NULL;

	}
#if 0
	printk("CCL_MX : One registed %02X:%02X:%02X:%02X:%02X:%02X mask %02X:%02X:%02X:%02X:%02X:%02X callback function %X\n",mac[0] ,mac[1] ,mac[2] ,mac[3] ,mac[4] ,mac[5],mask[0],mask[1],mask[2],mask[3],mask[4],mask[5],callback);
#endif
	return 0;
}


int ccl_mx_unregister(char *mac, char *mask)
{
	struct CCL_MX_reg *ptr, *ptr1;

	ptr=header;
	ptr1=header;
	while(ptr!=NULL)
	{
		if( ((ptr->mac[0] & ptr->mask[0]) == (mac[0] & mask[0])) &&	
				((ptr->mac[1] & ptr->mask[1]) == (mac[1] & mask[1])) &&	
				((ptr->mac[2] & ptr->mask[2]) == (mac[2] & mask[2])) &&	
				((ptr->mac[3] & ptr->mask[3]) == (mac[3] & mask[3])) &&	
				((ptr->mac[4] & ptr->mask[4]) == (mac[4] & mask[4])) &&	
				((ptr->mac[5] & ptr->mask[5]) == (mac[5] & mask[5])) 	
		  ) 
		{ 
			//this entry	
			ptr1->next=ptr->next;
			if(header==ptr)
				header = NULL;
			kfree(ptr);
			break; 	
		}
		else
		{
			ptr1=ptr;	
			ptr=ptr->next; 
		}     	
	}

	//printk("CCL_MX : One unregisted %02X:%02X:%02X:%02X:%02X:%02X mask %02X:%02X:%02X:%02X:%02X:%02X\n",mac[0],mac[1],mac[2],mac[3],mac[4],mac[5],mask[0],mask[1],mask[2],mask[3],mask[4],mask[5]);

	return 0; 
}


/****************************************************************/
static int ccl_mx(struct sk_buff *skb)
{
	struct CCL_MX_reg *ptr;
	unsigned char mac[6];
	int iret,i;
	static char lacp_mac[6] = {0x01,0x80,0xc2,0x00,0x00,0x02};

	Tuint8 p, criteria;

	for(i=0;i<6;i++) {
#ifdef __LINUX_2_6_19__
		mac[i]=eth_hdr(skb)->h_dest[i];
#else
		mac[i]=skb->mac.ethernet->h_dest[i];
#endif
	}

	//printk("ccl_mx(%2X:%2X:%2X:%2X:%2X:%2X)\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
	ptr=header;
	while(ptr!=NULL)
	{
		if( ((ptr->mac[0] & ptr->mask[0]) == (mac[0] & ptr->mask[0])) &&	
		    ((ptr->mac[1] & ptr->mask[1]) == (mac[1] & ptr->mask[1])) &&	
		    ((ptr->mac[2] & ptr->mask[2]) == (mac[2] & ptr->mask[2])) &&	
		    ((ptr->mac[3] & ptr->mask[3]) == (mac[3] & ptr->mask[3])) &&	
		    ((ptr->mac[4] & ptr->mask[4]) == (mac[4] & ptr->mask[4])) &&	
		    ((ptr->mac[5] & ptr->mask[5]) == (mac[5] & ptr->mask[5])) 	
		  ) 
		{ 
			if (lacp_mac[0] == mac[0] && lacp_mac[1] == mac[1] &&
			    lacp_mac[2] == mac[2] && lacp_mac[3] == mac[3] &&
			    lacp_mac[4] == mac[4] && lacp_mac[5] == mac[5])
				;			//lacp need to know physical port info
			else {
				/* check admission criteria for port to prevent loop */
				p = CCL_P(skb)->bySrcPortId;
				K_TrkGetPortAdmCri (p, &criteria);
				if (criteria == 0x80)
					return CCL_MX_DROP;
				cclmx_RxPkt (skb);	//convert CCL_P to CCL_LP
			}
			//this entry	
			iret=ptr->ccl_mx_callback(skb);
			if(iret == CCL_MX_ACCEPT)
				return CCL_MX_ACCEPT;	//done
			else if(iret == CCL_MX_CONTINUE)
				ptr=ptr->next;	        //continue to do this packet
			else if(iret == CCL_MX_DROP)
				return CCL_MX_DROP;	//drop the packet
			else if(iret == CCL_MX_NONE)
				return CCL_MX_NONE;	//send to upper layer
		}
		else
		{
			ptr=ptr->next; 
		}     	
	}

	return CCL_MX_NONE; 
}	


/*
 * InsTrk and DelTrk:
 *     for user api or ccldriver to manipulate trunk database
 *       range of db trunk id: 0 ~ MAX_TRK_PORT - 1
 */
int cclmx_InsTrk (Tuint8 ucTrkId, TstPPortMask stPMask)
{
	if (ucTrkId > MAX_TRK_PORT - 1)
		return 1;

	K_PPortMaskCopy (&astPhyMaskTbl[ucTrkId], &stPMask);
	return 0;
}


int cclmx_DelTrk (Tuint8 ucTrkId)
{
	if (ucTrkId > MAX_TRK_PORT - 1)
		return 1;

	K_PPortMaskClrAll (&astPhyMaskTbl[ucTrkId]);
	return 0;
}


int cclmx_AddPortToTrk (Tuint8 ucTrkId, Tuint8 ucPortId)
{
	if (ucTrkId > MAX_TRK_PORT - 1)
		return 1;

	if (ucPortId > MAX_PHY_PORT - 1)
		return 1;

	K_PPortMaskSetPort (&astPhyMaskTbl[ucTrkId], ucPortId);
	return 0;
}


int cclmx_DelPortFromTrk (Tuint8 ucPortId)
{
	int tid;

	if (ucPortId > MAX_PHY_PORT - 1)
		return 1;

	for (tid = 0; tid < MAX_TRK_PORT; tid++) {
		if (K_PPortMaskGetPort (&astPhyMaskTbl[tid], ucPortId)) {
			K_PPortMaskClrPort (&astPhyMaskTbl[tid], ucPortId);
			break;
		}
	}
	return 0;
}


/*
 * if the given port is a member of some trunk
 *     return (trunk_id + 1)
 * else
 *     return 0
 *
 * so that we could identify error 0 and trunk 0
 *
 * Note: ucPhyId must be demapped already
 */
Tuint8 cclmx_PortInTrunk (Tuint8 ucPhyId)
{
	int tid;

	if (ucPhyId >= MAX_PHY_PORT)
		return 0;
	for (tid = 0; tid < MAX_TRK_PORT; tid++) {
		if (K_PPortMaskGetPort (&astPhyMaskTbl[tid], ucPhyId))
			return (tid + 1); /* bcz trunk id starts from 0 */
	}
	return 0;
}


/* physical to logical port id mapping for the difference between panel and programs */
inline Tuint8 cclmx_MapPortId (Tuint8 ucPhyId)
{
	return ucPhyId;
}

/* logical to physical port id mapping for the difference between panel and programs */
inline Tuint8 cclmx_DemapPortId (Tuint8 ucLogicId)
{
	return ucLogicId;
}


Tuint8 cclmx_PId2LId (Tuint8 ucPhyId)
{
	Tuint8 i;

	for (i = 0; i < MAX_TRK_PORT; i++) {
		if (K_PPortMaskGetPort (&astPhyMaskTbl[i], ucPhyId)) /* port in trunk */
			return (Tuint8) cclmx_MapPortId (i + MAX_PHY_N_CPU_PORT);
	}
	return cclmx_MapPortId (ucPhyId);
}


Tuint8 cclmx_LId2PId (Tuint8 ucLogicId)
{
	Tuint8  i, trk, max = 0;

	if (ucLogicId < MAX_PHY_PORT)
		return cclmx_DemapPortId (ucLogicId);
	else if (MAX_PHY_PORT <= ucLogicId && ucLogicId < MAX_PHY_N_CPU_PORT)
		return ucLogicId;
	else if (MAX_PHY_N_CPU_PORT <= ucLogicId && ucLogicId < MAX_LOGIC_PORT) {
		trk = ucLogicId - MAX_PHY_N_CPU_PORT;
		if (K_PPortMaskIsZero (&astPhyMaskTbl[trk])) /* trunk dosen't exist */
			return (Tuint8)0xff; /* should I return 0 ? */
		for (i = 0; i < MAX_PHY_PORT; i++) {
			if (K_PPortMaskGetPort (&astPhyMaskTbl[trk], i)) { /* port in trunk */
				Tbool link;
				max = i;
				K_PortGetLink (i, &link);
				if (link)
					return cclmx_DemapPortId (i);
			}
		}
		return cclmx_DemapPortId (max);
	}
	else
		return (Tuint8)0xff;
}


int cclmx_LId2PMask (TstPPortMask * pstPMask, Tuint8 ucLogicId)
{
	Tuint8 trk;

	K_PPortMaskClrAll (pstPMask);
	if (ucLogicId < MAX_PHY_PORT) {
		trk = cclmx_PortInTrunk (ucLogicId);
		if (trk)
			K_PPortMaskSetPorts (pstPMask, &astPhyMaskTbl[trk - 1]);
		else
			K_PPortMaskSetPort (pstPMask, ucLogicId);
	}
	else if (MAX_PHY_PORT <= ucLogicId && ucLogicId < MAX_PHY_N_CPU_PORT) {
		K_PPortMaskSetPort (pstPMask, ucLogicId);
	}
	else if (MAX_PHY_N_CPU_PORT <= ucLogicId && ucLogicId < MAX_LOGIC_PORT) {
		trk = ucLogicId - MAX_PHY_N_CPU_PORT;
		K_PPortMaskSetPorts (pstPMask, &astPhyMaskTbl[trk]);
	}
	else
		return 1;
	return 0;
}


/*
 * physical port mask to logical port mask converter
 */
int cclmx_PMask2LMask (TstLPortMask * pstLMask, TstPPortMask * pstPMask)
{
	int p;
	Tuint8 trk;

	K_LPortMaskClrAll (pstLMask);

	/* convert: physical -> logical */
	for (p = 0; p < MAX_PHY_PORT; p++) {
		if (!K_PPortMaskGetPort (pstPMask, p))
			continue;

		trk = cclmx_PortInTrunk (p);
		if (trk)
			K_LPortMaskSetPort (pstLMask, trk - 1 + MAX_PHY_N_CPU_PORT);
		/* no need MapPortId here ? */
		else
			K_LPortMaskSetPort (pstLMask, cclmx_MapPortId (p));
	}
	for (p = MAX_PHY_PORT; p < MAX_PHY_N_CPU_PORT; p++) {
		if (!K_PPortMaskGetPort (pstPMask, p))
			continue;
		/* no need to check cpu port in trunk */
		K_LPortMaskSetPort (pstLMask, p); /* no mapping for cpu port */
	}
	return 0;
}


/*
 * logical port mask to physical port mask converter
 */
int cclmx_LMask2PMask (TstPPortMask * pstPMask, TstLPortMask * pstLMask)
{
	Tuint8 l;

	K_PPortMaskClrAll (pstPMask);

	/* convert: logical -> physical */
	for (l = 0; l < MAX_PHY_PORT; l++) {
		if (!K_LPortMaskGetPort (pstLMask, l))
			continue;
		if (cclmx_PortInTrunk (cclmx_DemapPortId (l)))
			continue; /* don't set this bit in logical mask */
		else
			K_PPortMaskSetPort (pstPMask, cclmx_DemapPortId (l));
	}
	for (l = MAX_PHY_PORT; l < MAX_PHY_N_CPU_PORT; l++) {
		if (K_LPortMaskGetPort (pstLMask, l))
			K_PPortMaskSetPort (pstPMask, l); /* cpu port currently no remap */
	}
	for (l = MAX_PHY_N_CPU_PORT; l < MAX_LOGIC_PORT; l++) {
		if (K_LPortMaskGetPort (pstLMask, l))
			K_PPortMaskSetPorts (pstPMask, &astPhyMaskTbl[l - MAX_PHY_N_CPU_PORT]);
		/* no need DemapPortId here ? */
	}
	return 0;
}


int cclmx_GetActPortMask (TstPPortMask * pstPMask)
{
	Tuint8 p, trk;
	Tbool link;
	Tuint8 criteria;

	K_PPortMaskClrAll (pstPMask);

	for (p = 0; p < MAX_PHY_PORT; p++) {
		if (cclmx_PortInTrunk (p))
			continue;
		K_PortGetLink (p, &link);
		if (link)
			K_PPortMaskSetPort (pstPMask, p);
	}

	for (trk = 0; trk < MAX_TRK_PORT; trk++) {
		if (K_PPortMaskIsZero (&astPhyMaskTbl[trk]))
			continue;
		for (p = 0; p < MAX_PHY_PORT; p++) {
			if (!K_PPortMaskGetPort (&astPhyMaskTbl[trk], p))
				continue;
			else {
				K_PortGetLink (p, &link);
				K_TrkGetPortAdmCri (p, &criteria);
				criteria &= 0x0F;
				if (link & criteria) {
					K_PPortMaskSetPort (pstPMask, p);
					break;
				}
			}
		}
	}
	return 0;
}


int cclmx_PortGetLink (Tuint8 ucLogicId)
{
	Tbool link;
	Tuint8 p;

	if (ucLogicId < MAX_PHY_PORT) {
		if (cclmx_PortInTrunk (cclmx_DemapPortId (ucLogicId)))
			return CCLMX_PORT_INTRUNK;
		K_PortGetLink (cclmx_DemapPortId (ucLogicId), &link);
		if (link)
			return CCLMX_PORT_LINKUP;
		else
			return CCLMX_PORT_LINKDOWN;
	}
	else if (MAX_PHY_PORT <= ucLogicId && ucLogicId < MAX_PHY_N_CPU_PORT)
		return CCLMX_PORT_LINKDOWN;		/* cpu always down */
	else if (MAX_PHY_N_CPU_PORT <= ucLogicId && ucLogicId < MAX_LOGIC_PORT) {
		if (K_PPortMaskIsZero (&astPhyMaskTbl[ucLogicId - MAX_PHY_N_CPU_PORT]))
			return CCLMX_PORT_LINKDOWN;	/* no such trunk */
		for (p = 0; p < MAX_PHY_PORT; p++) {	/* run through each port in the trunk */
			if (K_PPortMaskGetPort (&astPhyMaskTbl[ucLogicId - MAX_PHY_N_CPU_PORT], p)) {
				K_PortGetLink (p, &link);
				if (link) {		/* one of the ports in the trunk is up */
					return CCLMX_PORT_LINKUP;
				}
			}
		}
		return CCLMX_PORT_LINKDOWN;		/* all phy ports in the trunk are down */
	}
	else
		return CCLMX_PORT_LINKDOWN;
}


int cclmx_PortGetEbl (Tuint8 ucLogicId)
{
	Tbool enabled;
	Tuint8 criteria;
	Tuint8 p;

	if (ucLogicId < MAX_PHY_PORT) {
		if (cclmx_PortInTrunk (cclmx_DemapPortId (ucLogicId)))
			return CCLMX_PORT_OFF;		/* port in trunk */
		K_PortGetEbl (cclmx_DemapPortId (ucLogicId), &enabled);
		if (enabled)
			return CCLMX_PORT_ON;
		else
			return CCLMX_PORT_OFF;
	}
	else if (MAX_PHY_PORT <= ucLogicId && ucLogicId < MAX_PHY_N_CPU_PORT)
		return CCLMX_PORT_OFF;			/* cpu always down */
	else if (MAX_PHY_N_CPU_PORT <= ucLogicId && ucLogicId < MAX_LOGIC_PORT) {
		if (K_PPortMaskIsZero (&astPhyMaskTbl[ucLogicId - MAX_PHY_N_CPU_PORT]))
			return CCLMX_PORT_OFF;		/* no such trunk */
		for (p = 0; p < MAX_PHY_PORT; p++) {	/* run through each port in the trunk */
			if (K_PPortMaskGetPort (&astPhyMaskTbl[ucLogicId - MAX_PHY_N_CPU_PORT], p)) {
				K_PortGetEbl (p, &enabled);
				K_TrkGetPortAdmCri (p, &criteria);
				criteria &= 0x0F;
				if (enabled && criteria) { /* enabled and cri != 0 */
					return CCLMX_PORT_ON;
				}
			}
		}
		return CCLMX_PORT_OFF;			/* all phy ports in the trunk are down */
	}
	else
		return CCLMX_PORT_OFF;
}


Tuint8 cclmx_PortGetStpState (Tuint8 ucLogicId)
{
	Tuint8 p, state;

	if (ucLogicId < MAX_PHY_PORT) {
		if (cclmx_PortInTrunk (cclmx_DemapPortId (ucLogicId)))
			return STP_PORT_BLOCKING;
		K_StpGetPort (cclmx_DemapPortId (ucLogicId), &state);
		return state;
	}
	else if (MAX_PHY_PORT <= ucLogicId && ucLogicId < MAX_PHY_N_CPU_PORT)
		return STP_PORT_BLOCKING;		/* cpu always blocking */
	else if (MAX_PHY_N_CPU_PORT <= ucLogicId && ucLogicId < MAX_LOGIC_PORT) {
		if (K_PPortMaskIsZero (&astPhyMaskTbl[ucLogicId - MAX_PHY_N_CPU_PORT]))
			return STP_PORT_BLOCKING;	/* no such trunk */
		for (p = 0; p < MAX_PHY_PORT; p++) {	/* run through each port in the trunk */
			if (K_PPortMaskGetPort (&astPhyMaskTbl[ucLogicId - MAX_PHY_N_CPU_PORT], p)) {
				K_StpGetPort (p, &state);
				if (STP_PORT_FORWARDING == state)
					/* one of the ports in the trunk is enabled */
					return STP_PORT_FORWARDING;
			}
		}
		return state;				/* all phy ports in the trunk are disabled */
	}
	else
		return STP_PORT_BLOCKING;
}


void cclmx_PortSetStpState (Tuint8 ucLogicId, Tuint8 ucState)
{
	Tuint8 p;

	if (ucLogicId < MAX_PHY_PORT) {
		if (cclmx_PortInTrunk (cclmx_DemapPortId (ucLogicId)))
			return; /* protocol should not set this */
		K_StpSetPort (cclmx_DemapPortId (ucLogicId), ucState);
	}
	else if (MAX_PHY_PORT <= ucLogicId && ucLogicId < MAX_PHY_N_CPU_PORT)
		return;		/* do nothing to cpu port */
	else if (MAX_PHY_N_CPU_PORT <= ucLogicId && ucLogicId < MAX_LOGIC_PORT) {
		if (K_PPortMaskIsZero (&astPhyMaskTbl[ucLogicId - MAX_PHY_N_CPU_PORT]))
			return;	/* no such trunk */
		for (p = 0; p < MAX_PHY_PORT; p++) {	/* run through each port in the trunk */
			if (K_PPortMaskGetPort (&astPhyMaskTbl[ucLogicId - MAX_PHY_N_CPU_PORT], p)) {
				K_StpSetPort (p, ucState);
			}
		}
	}
	return;
}


void cclmx_PortSetMStpState (Tuint16 usInst,  TstVlanMask *pstVlanMask, Tuint8 ucLogicId, Tuint8 ucState)
{
	Tuint8 p;

	if (ucLogicId < MAX_PHY_PORT) {
		if (cclmx_PortInTrunk (cclmx_DemapPortId (ucLogicId)))
			return; /* protocol should not set this */
		K_StpSetInstPort (usInst, pstVlanMask, cclmx_DemapPortId (ucLogicId), ucState);
	}
	else if (MAX_PHY_PORT <= ucLogicId && ucLogicId < MAX_PHY_N_CPU_PORT)
		return;		/* do nothing to cpu port */
	else if (MAX_PHY_N_CPU_PORT <= ucLogicId && ucLogicId < MAX_LOGIC_PORT) {
		if (K_PPortMaskIsZero (&astPhyMaskTbl[ucLogicId - MAX_PHY_N_CPU_PORT]))
			return;	/* no such trunk */
		for (p = 0; p < MAX_PHY_PORT; p++) {	/* run through each port in the trunk */
			if (K_PPortMaskGetPort (&astPhyMaskTbl[ucLogicId - MAX_PHY_N_CPU_PORT], p)) {
				K_StpSetInstPort (usInst, pstVlanMask, p, ucState);
			}
		}
	}
	return;
}


int cclmx_PortGetSpeedDuplex (Tuint8 ucLogicId, Tuint8 * pucSpeed, Tuint8 * pucDuplex)
{
	Tuint8 speed, duplex, p, trk;

	if (ucLogicId < MAX_PHY_PORT) {
		trk = cclmx_PortInTrunk (cclmx_DemapPortId (ucLogicId));
		if (trk) { /* port in trunk */
			*pucSpeed = CCLMX_PORT_SPEED_0;
			*pucDuplex = CCLMX_PORT_DUPLEX_UNKNOWN;
			return 0;
		}
		K_PortGetSpeedDuplex (cclmx_DemapPortId (ucLogicId), &speed, &duplex);
		switch (speed)
		{
		case PORT_SPEED_10:   *pucSpeed = CCLMX_PORT_SPEED_10;   break;
		case PORT_SPEED_100:  *pucSpeed = CCLMX_PORT_SPEED_100;  break;
		case PORT_SPEED_1000: *pucSpeed = CCLMX_PORT_SPEED_1000; break;
		}
		if (duplex == PORT_DUPLEX_HALF)
			*pucDuplex = CCLMX_PORT_DUPLEX_HALF;
		else
			*pucDuplex = CCLMX_PORT_DUPLEX_FULL;
		return 0;
	}
	else if (MAX_PHY_PORT <= ucLogicId && ucLogicId < MAX_PHY_N_CPU_PORT) {
		*pucSpeed = CCLMX_PORT_SPEED_100;
		*pucDuplex = CCLMX_PORT_DUPLEX_FULL;
		return 0;
	}
	else if (MAX_PHY_N_CPU_PORT <= ucLogicId && ucLogicId < MAX_LOGIC_PORT) {
		*pucSpeed = CCLMX_PORT_SPEED_0;
		trk = ucLogicId - MAX_PHY_N_CPU_PORT;
		for (p = 0; p < MAX_PHY_PORT; p++) {
			if (K_PPortMaskGetPort (&astPhyMaskTbl[trk], p)) { /* is trunk member */
				Tbool link;
				K_PortGetLink (p, &link);
				if (!link)
					continue;
				K_PortGetSpeedDuplex (p, &speed, &duplex);
				if (*pucSpeed == CCLMX_PORT_SPEED_0) {
					switch (speed)
					{
					case PORT_SPEED_10:   *pucSpeed = CCLMX_PORT_SPEED_10;   break;
					case PORT_SPEED_100:  *pucSpeed = CCLMX_PORT_SPEED_100;  break;
					case PORT_SPEED_1000: *pucSpeed = CCLMX_PORT_SPEED_1000; break;
					}
				}
				else
					CCLMX_PORT_SPEED_NEXT (*pucSpeed);
			}
		}
		*pucDuplex = CCLMX_PORT_DUPLEX_FULL;
		return 0;
	}
	*pucSpeed = CCLMX_PORT_SPEED_0;
	*pucDuplex = CCLMX_PORT_DUPLEX_UNKNOWN;
	return 0;
}


/*
 * input: skb with cb as CCL_P()
 * function: translate CCL_P into CCL_LP for protocol callback functions
 *           (L stands for logical)
 */
int cclmx_RxPkt (struct sk_buff * skb)
{
	TstPPortMask    stPDstMask;
	TstPPortMask    stPTagMask;
	unsigned short  wSize         = CCL_P(skb)->wSize;
	unsigned short  wVid          = CCL_P(skb)->wVid;
	unsigned char   byPriority    = CCL_P(skb)->byPriority;
	unsigned char   bySrcPortId   = CCL_P(skb)->bySrcPortId; 
	unsigned char   byFwdReason   = CCL_P(skb)->byFwdReason;
	unsigned char   bIfTagged     = CCL_P(skb)->bIfTagged;

	stPDstMask.ulMask[0] = CCL_P(skb)->dwDstPortMsk;
	stPTagMask.ulMask[0] = CCL_P(skb)->dwTagPortMsk;

	cclmx_PMask2LMask (&CCL_LP(skb)->dwDstPortMsk, &stPDstMask);
	cclmx_PMask2LMask (&CCL_LP(skb)->dwTagPortMsk, &stPTagMask);
	CCL_LP(skb)->bySrcPortId = cclmx_PId2LId (bySrcPortId);

	CCL_LP(skb)->wSize = wSize;
	CCL_LP(skb)->wVid = wVid;
	CCL_LP(skb)->byPriority = byPriority;
	CCL_LP(skb)->byFwdReason = byFwdReason;
	CCL_LP(skb)->bIfTagged = bIfTagged;

	return 0;
}


/*
 * input: skb with cb as CCL_LP()
 * function: translate CCL_LP into CCL_P and send it to driver for transmission
 *           (L stands for logical)
 */
int cclmx_TxPkt (struct sk_buff * skb)
{
	TstPPortMask    stDstMask;
	TstPPortMask    stTagMask;
	unsigned short  wSize         = CCL_LP(skb)->wSize;
	unsigned short  wVid          = CCL_LP(skb)->wVid;
	unsigned char   byPriority    = CCL_LP(skb)->byPriority;
	unsigned char   bySrcPortId   = 0; 
	unsigned char   byFwdReason   = CCL_LP(skb)->byFwdReason;
	unsigned char   bIfTagged     = CCL_LP(skb)->bIfTagged;

	Tuint8 l, p, trk;
	Tbool link, enabled;
	Tuint8 criteria;

	K_PPortMaskClrAll (&stDstMask);
	K_PPortMaskClrAll (&stTagMask);

	for (l = 0; l < MAX_PHY_PORT; l++) {
		if (K_LPortMaskGetPort (&CCL_LP(skb)->dwDstPortMsk, l)) {
			trk = cclmx_PortInTrunk (cclmx_DemapPortId (l));
			if (trk)
				; /* protocols should not do this!! */
			else
				K_PPortMaskSetPort (&stDstMask , cclmx_DemapPortId (l));
		}
	}
	for (l = MAX_PHY_PORT; l < MAX_PHY_N_CPU_PORT; l++)
		; /* don't set cpu bits, either */
	for (l = MAX_PHY_N_CPU_PORT; l < MAX_LOGIC_PORT; l++) {
		trk = l - MAX_PHY_N_CPU_PORT;
		if (K_LPortMaskGetPort (&CCL_LP(skb)->dwDstPortMsk, l)) {
			for (p = 0; p < MAX_PHY_PORT; p++) {
				if (K_PPortMaskGetPort (&astPhyMaskTbl[trk], p)) {
					K_PortGetLink (p, &link);
					K_PortGetEbl (p, &enabled);
					K_TrkGetPortAdmCri (p, &criteria);
					criteria &= 0x0F;
					if (link && enabled && criteria) {
						K_PPortMaskSetPort (&stDstMask, p);
						break; /* set 1 bit is ok */
					}
				}
			}
		}
	}

	bySrcPortId = cclmx_LId2PId (CCL_LP(skb)->bySrcPortId);
	//CCL_P(skb)->dwDstPortMsk = stDstMask.ulMask[0]; /* ugly implementation */
	K_PPortMask2UL(&CCL_P(skb)->dwDstPortMsk, &stDstMask);
	//CCL_P(skb)->dwTagPortMsk = stTagMask.ulMask[0]; /* ugly implementation */
	K_PPortMask2UL(&CCL_P(skb)->dwTagPortMsk, &stTagMask);

	CCL_P(skb)->wSize        = wSize;
	CCL_P(skb)->wVid         = wVid;
	CCL_P(skb)->byPriority   = byPriority;
	CCL_P(skb)->bySrcPortId  = bySrcPortId;
	CCL_P(skb)->byFwdReason  = byFwdReason;
	CCL_P(skb)->bIfTagged    = bIfTagged;

	skb->dev = dev_get_by_name ("eth0");
	dev_queue_xmit (skb); //need to check portmask? free skb?

	return 0;
}


#ifdef CCLMX_PERIODIC_CHECK_PORT_LINK
void cclmx_sleep (unsigned howlong)
{
	if (in_interrupt ()) {
		return;
	}
	current->state = TASK_INTERRUPTIBLE;
	schedule_timeout (howlong);
}

void cclmx_refresh_trk (Tuint8 ucTrkId)
{
#if 0
	Tuint8		ucAggrId;
	TstPPortMask	stPMask;
#endif
	Tuint8		ucLinkedPorts = 0;
	Tuint8		ucPorts[MAX_PHY_PORT];
	Tuint8		ucPortId;
	Tbool		bLink;

#if 0
	/* Rebuild the trunk group */
	cclmx_DelTrk (ucTrkId - 1);
	K_PPortMaskClrAll (&stPMask);
	for (ucPortId = 0; ucPortId < MAX_PHY_PORT; ++ucPortId) {
		K_TrkGetPortAggrId (ucPortId, &ucAggrId);
		if (ucTrkId - 1 == ucAggrId) {
			K_PPortMaskSetPort(&stPMask, ucPortId);
		}
	}
	cclmx_InsTrk (ucTrkId - 1, stPMask);
#endif

	/* Find linked member ports of the trunk group */
	for (ucPortId = 0; ucPortId < MAX_PHY_PORT; ++ucPortId) {
		if (K_PPortMaskGetPort (&astPhyMaskTbl[ucTrkId-1], ucPortId)) {
			K_PortGetLink (ucPortId, &bLink);
			if (bLink) {
				ucPorts[ucLinkedPorts++] = ucPortId;
			}
		}
	}

	/* Call K_TrkSetPortAdmCri() to refresh linked member ports */
	for (ucPortId = 0; (ucPortId < ucLinkedPorts) && (ucPortId < MAX_PORTS_PER_TRK); ++ucPortId) {
		K_TrkSetPortAdmCri (ucPorts[ucPortId], KRN_TRK_PORT_ADM_CRI[ucLinkedPorts-1][ucPortId]);
	}
}

Tbool	cclmx_abPortLink[MAX_PHY_PORT];

int cclmx_loop (void * vp)
{
	Tuint8	ucPortId;
	Tbool	bLink;
	Tuint8	ucSpeed;
	Tuint8	ucDuplex;
	Tuint8	ucFlowCtrl;
	Tuint8	ucTrkId;

	while (1) {
		for (ucPortId = 0; ucPortId < MAX_PHY_PORT; ++ucPortId) {
			K_PortGetLink (ucPortId, &bLink);
			if (cclmx_abPortLink[ucPortId] != bLink) {
				cclmx_abPortLink[ucPortId] = bLink;
				if (bLink) { /* link goes up */
					ucTrkId = cclmx_PortInTrunk (ucPortId);
					if (ucTrkId) {
						cclmx_refresh_trk (ucTrkId);
					}
				}
				else { /* link goes down */
					K_MacClrPort (ucPortId);

					ucTrkId = cclmx_PortInTrunk (ucPortId);
					if (ucTrkId) {
						K_TrkSetPortAdmCri (ucPortId, 0);
						cclmx_refresh_trk (ucTrkId);
					}
				}
			}
			K_PortGetSpeedDuplex (ucPortId, &ucSpeed, &ucDuplex);
			K_PortGetFlowCtrl (ucPortId, &ucFlowCtrl);
		}
		cclmx_sleep (500); /* 5 seconds */
	}

	return 0;
}
#endif


static int __init ccl_mx_init_module(void)
{
	int i;
#ifdef CCLMX_PERIODIC_CHECK_PORT_LINK
	Tuint8	ucPortId;
#endif

	for (i = 0; i < MAX_TRK_PORT; i++)
		K_PPortMaskClrAll (&astPhyMaskTbl[i]);

	if(nf_register_sockopt(&cclmx_sockopts))
		printk("cclmx: cannot register sockopt \n");

#if 0	
	printk("================= CCL MX ================\n");	
	printk("CCL Packet Multiplexer\n");
	printk("=========================================\n");    	
	{
		for(i=0;i<CCL_SLOT_NUM;i++)
		{
			printk("   Slot %d : %d ports\n",i+1,ccl_slot_port_num[i]);
			printk("     Map");
			for(j=0;j<ccl_slot_port_num[i];j++)
				printk(" %d",CCL_PORT(i+1,j+1));
			printk("     \n");
		}
	}
#endif
	ccl_mx_hook = ccl_mx;
	header = NULL;

#ifdef CCLMX_PERIODIC_CHECK_PORT_LINK
	for (ucPortId = 0; ucPortId < MAX_PHY_PORT; ++ucPortId) {
		cclmx_abPortLink[ucPortId] = 0;
	}
	kernel_thread ((int (*)(void *))cclmx_loop, NULL, 0);
#endif

	return 0;
}	


static void __exit ccl_mx_cleanup_module(void)
{
	ccl_mx_hook = NULL;
	header = NULL;

	printk("================= CCL MX ================\n");	
	printk("Remove CCL Packet Multiplexer\n");
	printk("=========================================\n");    	

}


module_init(ccl_mx_init_module);
module_exit(ccl_mx_cleanup_module);

