/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  THIS IS AN UNPUBLISHED WORK WHICH CONTAINS CONFIDENTIAL INFORMATION
 *  FROM CCL, ITRI.  NO PART OF THIS WORK MAY BE USED IN ANY WAY WITHOUT
 *  THE PRIOR WRITTEN PERMISSION.  ANY UNAUTHORIZED USE COULD SUBJECT THE
 *  PERPETRATOR TO CRIMINAL AND CIVIL LIABILITY.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************/

// ********************************************
// Name   : 8021x_radius.c
// Author : Liu, Ren Hao (N300, CCL/ITRI)
// Date   : 2003/12/04
// Note   : 802.1x radius related function
// ********************************************
// Programmer   : Arunesh Mishra
// Copyright (c) Arunesh Mishra 2002
// All rights reserved.
// Maryland Information and Systems Security Lab
// University of Maryland, College Park.

#include "krndef.h"
#include "krntype.h"
#include "krnerr.h"
#include "krnmib_rmon.h"
#include "8021x.h"

//used in radius message
#define CONNECT_INFO  "CONNECT Ethernet 802.3"

extern Tbyte dot1x_my_mac[MAC_ADDR_LEN];
extern Global_Params *port_sm[MAX_PORT];
extern Radius_srv_info *radsrv_info;
extern Radius_acc_info *radacc_info;

//-------------------------------------------------
//   parsing eap attribute in radius message
//--------------------------------------------------
Tuint8 auth_radius_handle(Global_Params *global, Tuint8 id, Tuint8 code, Tuint16 len, Tbyte *pkt)
{
	Auth_Pae   *auth_pae = global->Auth;
	Bauth_SM   *bauth_sm = global->Auth->bauth_sm;
	bauth_sm->idFromServer = get_unaligned((unsigned char *)(pkt+1));
	 
	switch(code)
	{
	case X_EAP_REQUEST:

		bauth_sm->aReq = True;
	  	
		//update length for different size
		memcpy(auth_pae->fromsvr.pkt,pkt,len);
		auth_pae->fromsvr.length = len;
		auth_pae->fromsvr.valid = 1;
		break;
	
	case X_EAP_SUCCESS:
		
		bauth_sm->aSuccess = True;

		//Send Radius Account Start packet
                rad_send_acc_info(global,RADPKT_ACC_STATUS_START,0);

		break;
	
	case X_EAP_FAILURE:
	
		bauth_sm->aFail = True;
	
		if( global->Auth->portMode == pmt_Auto && global->portStatus == pst_Authorized )
		{
		  //Send Radius Account Stop packet
  		  rad_send_acc_info(global,RADPKT_ACC_STATUS_STOP,RADPKT_ACC_TERM_SESSIONOUT);
		}
		break;
	}
	return KRN_RET_OK;
}


//---------------------------------------------------------
//  receive radius message with eap attribute.
//----------------------------------------------------------
void radius_rx_eap_attr(Tuint8 nPortId, Tuint8 id, Tuint8 code, Tuint16 len, Tbyte *pkt)
{
  if(port_sm[nPortId]!=NULL)
   {
    auth_radius_handle(port_sm[nPortId],id,code,len,pkt);
   }
}



// receive eapol pkt will trigger radius pkt be sent
Tuint32 eap_rx2radius_tx( Tuint8 ucPort, Auth_Pae *pstAuthPae, Tuint8 *pucEapPkt)
{
    Tbool        bSendStateAttr = False ;
    Tuint8       aucMessAuth[20], ucIdentifier; 
    Tint16       usCopySize, usIdCopySize;
    Tuint32      ulNasPortType=0, ulNasPort=0, ulFrameMtu=0, ulPortNo=0;
    Tuint32      ulServiceType = htonl(RADPKT_ATTR_STYPE_FRAMED); 
    
    struct EAPHeader_T   *pstEapHdr;
    struct EAPCodeType_T *pstEapCodeTypeHdr;
    struct rad_port_info *pstRadPortInfo;
    struct rad_pkt_info  *pstRadPktConst;
 
    ulNasPort = ucPort;
    ulNasPortType = htonl(RADPKT_ATTR_PTYPE_ETHERNET) ;

    if(pucEapPkt!=NULL)
     pstEapHdr = (struct EAPHeader_T *) (pucEapPkt) ;
    else
     return False;

    ucIdentifier = pstEapHdr->ucIdentifier ; 
    ulFrameMtu = htonl(RADPKT_MAX_PKT_LEN) ;
     
    // check code field
    switch( pstEapHdr->ucCode ) {
        case EAP_CODE_RESPONSE:
            break;
        default:
            return False;
            /* error case */
    }
    
    // check EAP type field
    pstEapCodeTypeHdr = (struct EAPCodeType_T *) (pucEapPkt + X_EAP_HDRLEN);
    switch( pstEapCodeTypeHdr->ucType ) {
        case EAP_RR_TYPE_IDENTITY:
        case EAP_RR_TYPE_MD5_CHALLENGE:
        case EAP_RR_TYPE_TLS:    
        case EAP_RR_TYPE_PEAP:    
        case EAP_RR_TYPE_TTLS:    
        case EAP_RR_TYPE_NAK:
             break;
        default:
            return False;
    }    
   
 
    pstRadPortInfo = &(radsrv_info[ucPort].aucSWPort) ;
    if( pstEapCodeTypeHdr->ucType == EAP_RR_TYPE_IDENTITY ) {
      
       usCopySize = ntohs(pstEapHdr->usLen);
       
      if (usCopySize > RADPKT_MAX_PKT_LEN){
      	   return False;
      }
       
      memcpy( pstRadPortInfo->aucEAPMessFromSupp, pucEapPkt, usCopySize);

      if( usCopySize > (X_EAP_HDRLEN + 1))
        {
         usIdCopySize = ntohs(pstEapHdr->usLen) - X_EAP_HDRLEN - 1; //1=type field
      
      	 if( usIdCopySize > RADPKT_MAX_USERNAME_LEN){
	    return False;
	 }
	    
         if( memcmp( (Tuint8 *)pstEapCodeTypeHdr+1, pstRadPortInfo->aucUserName, usIdCopySize) == 0 )
            bSendStateAttr = True;
       
         memcpy( pstRadPortInfo->aucUserName, (Tuint8 *)pstEapCodeTypeHdr+1, usIdCopySize);
         pstRadPortInfo->ucUserNameLen = usIdCopySize;
        } 
      else
        {
          memset(pstRadPortInfo->aucUserName, 0x0, RADPKT_MAX_USERNAME_LEN );
          pstRadPortInfo->ucUserNameLen = 0;
        }
	
        pstRadPortInfo->aucEAPMessTypeFromSupp = EAP_RR_TYPE_IDENTITY;
        pstRadPortInfo->usEAPMessLenFromSupp = usCopySize;
        
        radsrv_info[ulNasPort].ucIdentifier += 2; 
        pstRadPortInfo->ucIdentifier = radsrv_info[ulNasPort].ucIdentifier ;
      
        memcpy(pstRadPortInfo->aucConnectInfo, CONNECT_INFO,strlen(CONNECT_INFO));       
        create_reqauth(pstRadPortInfo->aucReqAuthenticator, 1);
       
        // construct radius access request pkt
        memset(pstRadPortInfo->aucUDPSendToServer, 0x0, RADPKT_MAX_PKT_LEN);
        pstRadPktConst = create_radius_common_hdr( pstRadPortInfo->aucUDPSendToServer, RADPKT_TYPE_ACCESS_REQUEST, pstRadPortInfo->ucIdentifier, pstRadPortInfo->aucReqAuthenticator);
                
	if(pstRadPktConst==NULL){
           return False;
	}

        if( pstRadPortInfo->ucUserNameLen != 0 ){ 
            radconst_addattr( pstRadPktConst, RADPKT_ATTR_USER_NAME, pstRadPortInfo->ucUserNameLen,
	    		      pstRadPortInfo->aucUserName);
	}
        
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_NAS_IP_ADDRESS, 0x4, (Tuint8 *) &radsrv_info[0].ipNas );
        
        ulPortNo = htonl(ulNasPort + 1);
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_NAS_PORT, 0x4, (Tuint8 *) &ulPortNo );
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_NAS_IDENTIFIER, (Tuint8)(strlen(radsrv_info[0].aucNASIdentifier)),
			  (Tuint8 *)radsrv_info[0].aucNASIdentifier );
        
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_SERVICE_TYPE, 0x4, (Tuint8 *)&ulServiceType );

        
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_CALLED_STID, RADPKT_MAX_CALLED_ID_LEN,
			  (Tuint8 *)radsrv_info[0].aucCalledStationID);

        radconst_addattr( pstRadPktConst, RADPKT_ATTR_CALLING_STID,RADPKT_MAX_CALLING_ID_LEN, 
			  (Tuint8 *)pstAuthPae->calling_id );
    
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_FRAME_MTU, 0x4, (Tuint8 *) &ulFrameMtu );
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_NAS_PORT_TYPE, 0x4, (Tuint8 *) &ulNasPortType );
        pstRadPktConst->pulNASPortType = &ulNasPortType ;
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_CONNECT_INFO, strlen(pstRadPortInfo->aucConnectInfo),
		 	  pstRadPortInfo->aucConnectInfo );

        if( pstRadPortInfo->bStateAvailable && bSendStateAttr ){
            radconst_addattr( pstRadPktConst, RADPKT_ATTR_STATE, pstRadPortInfo->usStateLen,
	    		      pstRadPortInfo->aucStateAttr );
	}

        radconst_addEAPMessAttr( pstRadPktConst, pstRadPortInfo->usEAPMessLenFromSupp, 
				 pstRadPortInfo->aucEAPMessFromSupp);
				 
        memset( aucMessAuth, 0x0, 20);                
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_MESSAGE_AUTH, 0x10, aucMessAuth);//reserv 18bytes
        radconst_calradlength( pstRadPktConst );
        create_messauth( pstRadPktConst, pstRadPktConst->pMessageAuthenticator );        

        p8021x_send_radius_pkt(pstRadPktConst->pPkt, pstRadPktConst->usPktLength,radsrv_info[0].usServerPort );
        p8021x_free( pstRadPktConst );        
        
    } else {
        
        usCopySize = ntohs(pstEapHdr->usLen);
        
        if (usCopySize > RADPKT_MAX_PKT_LEN){
      	   return False;
	}
	
	memcpy( pstRadPortInfo->aucEAPMessFromSupp, pucEapPkt, usCopySize);
  
        if(pstEapCodeTypeHdr->ucType==EAP_RR_TYPE_MD5_CHALLENGE)
         pstRadPortInfo->aucEAPMessTypeFromSupp = EAP_RR_TYPE_MD5_CHALLENGE;
      
        else if(pstEapCodeTypeHdr->ucType==EAP_RR_TYPE_TLS)
         pstRadPortInfo->aucEAPMessTypeFromSupp = EAP_RR_TYPE_TLS;

        else if(pstEapCodeTypeHdr->ucType==EAP_RR_TYPE_PEAP)
         pstRadPortInfo->aucEAPMessTypeFromSupp = EAP_RR_TYPE_PEAP;
       
        else if(pstEapCodeTypeHdr->ucType==EAP_RR_TYPE_TTLS)
         pstRadPortInfo->aucEAPMessTypeFromSupp = EAP_RR_TYPE_TTLS;
        
	else if(pstEapCodeTypeHdr->ucType== EAP_RR_TYPE_NAK)
         pstRadPortInfo->aucEAPMessTypeFromSupp = EAP_RR_TYPE_NAK;

        pstRadPortInfo->usEAPMessLenFromSupp = usCopySize;
      	  
        radsrv_info[ulNasPort].ucIdentifier += 2; 
        pstRadPortInfo->ucIdentifier = radsrv_info[ulNasPort].ucIdentifier ;
        
        create_reqauth(pstRadPortInfo->aucReqAuthenticator, 1);

        // create the access-request pkt to server
        memset(pstRadPortInfo->aucUDPSendToServer, 0x0, RADPKT_MAX_PKT_LEN);
        pstRadPktConst = create_radius_common_hdr( pstRadPortInfo->aucUDPSendToServer, RADPKT_TYPE_ACCESS_REQUEST, pstRadPortInfo->ucIdentifier, pstRadPortInfo->aucReqAuthenticator);
         

	if(pstRadPktConst==NULL){
           return False;
	}

        if( pstRadPortInfo->ucUserNameLen != 0 ){ 
            radconst_addattr( pstRadPktConst, RADPKT_ATTR_USER_NAME, pstRadPortInfo->ucUserNameLen,
	    	              pstRadPortInfo->aucUserName);
	}

        radconst_addattr( pstRadPktConst, RADPKT_ATTR_NAS_IP_ADDRESS, 0x4, (Tuint8 *)&radsrv_info[0].ipNas );
        
        ulPortNo = htonl(ulNasPort + 1);          
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_NAS_PORT, 0x4, (Tuint8 *)&ulPortNo );

        radconst_addattr( pstRadPktConst, RADPKT_ATTR_CALLED_STID, RADPKT_MAX_CALLED_ID_LEN,
		          (Tuint8 *)radsrv_info[0].aucCalledStationID);
        
	radconst_addattr( pstRadPktConst, RADPKT_ATTR_CALLING_STID, RADPKT_MAX_CALLING_ID_LEN,
		          (Tuint8 *)pstAuthPae->calling_id );

        radconst_addattr( pstRadPktConst, RADPKT_ATTR_NAS_IDENTIFIER, (Tuint8)(strlen(radsrv_info[0].aucNASIdentifier)), (Tuint8 *)radsrv_info[0].aucNASIdentifier );

        radconst_addattr( pstRadPktConst, RADPKT_ATTR_SERVICE_TYPE, 0x4, (Tuint8 *)&ulServiceType );
        

        radconst_addattr( pstRadPktConst, RADPKT_ATTR_NAS_PORT_TYPE, 0x4, (Tuint8 *) &ulNasPortType );
        pstRadPktConst->pulNASPortType = &ulNasPortType ;
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_CONNECT_INFO,  strlen(pstRadPortInfo->aucConnectInfo),
			  pstRadPortInfo->aucConnectInfo );
			  
        if( pstRadPortInfo->bStateAvailable ){
            radconst_addattr( pstRadPktConst, RADPKT_ATTR_STATE, pstRadPortInfo->usStateLen,
	    	              pstRadPortInfo->aucStateAttr );
	}
			      
        radconst_addEAPMessAttr( pstRadPktConst, pstRadPortInfo->usEAPMessLenFromSupp, 
				 pstRadPortInfo->aucEAPMessFromSupp);
				 
        memset( aucMessAuth, 0x0, sizeof(aucMessAuth));                
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_MESSAGE_AUTH,   0x10, aucMessAuth);
        radconst_calradlength( pstRadPktConst );
        create_messauth( pstRadPktConst, pstRadPktConst->pMessageAuthenticator );        
        
        p8021x_send_radius_pkt(pstRadPktConst->pPkt, pstRadPktConst->usPktLength,radsrv_info[0].usServerPort );
        p8021x_free( pstRadPktConst );        
    }   

    return True;
}        


// receive radius packet will trigger eapol pkt be sent
Tuint32 radius_rx2eap_tx(Tuint8 *pucEapPkt)
{
    struct  rad_header      *pstRadiusPktHeader;
    struct  rad_attr_header *pstRadiusAttrHeader;

    Tuint8  ucPortId, ucSuppPort=0, ucNewEapPktLen=0, ucEapCode;
    static Tuint8  aucTmpPktBuf[RADPKT_MAX_PKT_LEN+RADPKT_MAX_SHARE_KEY_LEN];
    static Tuint8  auc_ChkAuth[RADPKT_AUTHENTICATOR_LEN];
    static Tuint8  eap_buf[RADPKT_MAX_PKT_LEN];
    Tuint8 *eap_msg=eap_buf;
    
    Tuint16 usRadPktLen,usRadHdrLen;
    Tuint32 ulAccInterimInterval, ulIdleTimeout, ulTerminationAction;
    Tuint32 ulOffset=0, ulShareKeyLen,ulMessPos=0,ulEapPos=0;
    Tbool   bFind = False, bIsEapMess=False;
    MD5_CTX st_MD5Context;

    if(pucEapPkt!=NULL){
      pstRadiusPktHeader=(struct rad_header *) pucEapPkt;
    }
    else {
      return  False;
    }

    usRadHdrLen=get_unaligned(&(pstRadiusPktHeader->usLength));
    usRadHdrLen=ntohs(usRadHdrLen);

    if(usRadHdrLen > RADPKT_MAX_PKT_LEN){
	    return False;
    }
      
    for( ucPortId=0 ; ucPortId<MAX_PORT ; ucPortId++ ) {
        if( radsrv_info[ucPortId].aucSWPort.ucIdentifier == pstRadiusPktHeader->ucIdentifier ) {
            // check response authenticator 
            ulShareKeyLen = strlen((char *)radsrv_info[0].aucShareKey);
            memset(aucTmpPktBuf, 0x0, sizeof(aucTmpPktBuf));
            memcpy(aucTmpPktBuf, pucEapPkt, usRadHdrLen);
            memcpy(aucTmpPktBuf+4, radsrv_info[ucPortId].aucSWPort.aucReqAuthenticator, RADPKT_AUTHENTICATOR_LEN);
            memcpy(aucTmpPktBuf+usRadHdrLen, radsrv_info[0].aucShareKey, ulShareKeyLen );
            MD5Init(&st_MD5Context);
            MD5Update(&st_MD5Context, (Tuint8 *)aucTmpPktBuf, (Tuint32 )(usRadHdrLen+ ulShareKeyLen));
            MD5Final(auc_ChkAuth, &st_MD5Context);
	    
            if( memcmp(auc_ChkAuth, pstRadiusPktHeader->aucAuthenticator, RADPKT_AUTHENTICATOR_LEN)!= 0)
                return False ;
		
            ucSuppPort = ucPortId ;    
            bFind = True ;
            break;
        } 
    }
    
    if( !bFind ) 
        return False ;

    switch( pstRadiusPktHeader->ucCode ) {
        case RADPKT_TYPE_ACCESS_REQUEST:
        case RADPKT_TYPE_ACC_REQUEST:
        case RADPKT_TYPE_ACC_RESPONSE:
            return False;
            
        case RADPKT_TYPE_ACCESS_ACCEPT:
            ucEapCode = EAP_CODE_SUCCESS;
            break;

        case RADPKT_TYPE_ACCESS_REJECT:            
            ucEapCode = EAP_CODE_FAILURE;
            break;

        case RADPKT_TYPE_ACCESS_CHALLENGE:
            ucEapCode = EAP_CODE_REQUEST;
            break;
            
        default:
            return False;
    }

    usRadPktLen = usRadHdrLen - RADPKT_PKT_HEADER_LEN ;

    while( usRadPktLen > 2 ) {//Type & Len field
        
        pstRadiusAttrHeader = (struct rad_attr_header *)( pucEapPkt+(RADPKT_PKT_HEADER_LEN + ulOffset));
        
        if( pstRadiusAttrHeader->ucType == RADPKT_ATTR_STATE ) {
            
            ulOffset += pstRadiusAttrHeader->ucLength ;
            memcpy((Tuint8 *)radsrv_info[ucSuppPort].aucSWPort.aucStateAttr, 
	    	   (Tuint8 *)pstRadiusAttrHeader+2, pstRadiusAttrHeader->ucLength - 2);
            radsrv_info[ucSuppPort].aucSWPort.bStateAvailable = True ;
            radsrv_info[ucSuppPort].aucSWPort.usStateLen = pstRadiusAttrHeader->ucLength - 2;

        } else if( pstRadiusAttrHeader->ucType == RADPKT_ATTR_EAP_MESSAGE ) {

            bIsEapMess = True ;
            ulMessPos = ulOffset + RADPKT_PKT_HEADER_LEN + RADPKT_ATTR_HEAD_LEN ;
            ucNewEapPktLen = pstRadiusAttrHeader->ucLength - RADPKT_ATTR_HEAD_LEN ;
            ulOffset += pstRadiusAttrHeader->ucLength ;

           // added by rhliu
           // EAP msg is devided into many fragments (because Length filed only 8bit->Max value is 255)
	   // Each fragment is TLV format. We combine every fragment into buffer, and send to supplicant
          
	   //+2 means "skip type & value field" for each fragment 
	   memcpy(eap_msg+ulEapPos,(Tuint8 *)pstRadiusAttrHeader+2 , ucNewEapPktLen);
           ulEapPos += ucNewEapPktLen;

        } else if( pstRadiusAttrHeader->ucType == RADPKT_ATTR_SESSION_TIMEOUT ) { 

            // maintain session-timeout timer            
            memcpy( (Tuint8 *)&ulAccInterimInterval, (Tuint8 *)pstRadiusAttrHeader + 2, 4);
            ulOffset += pstRadiusAttrHeader->ucLength ;
            
        
        } else if( pstRadiusAttrHeader->ucType == RADPKT_ATTR_IDLE_TIMEOUT ) {

            //  maintain idel-timeout timer            
            memcpy((Tuint8 *) &ulIdleTimeout, (Tuint8 *)pstRadiusAttrHeader + 2, 4);
            ulOffset += pstRadiusAttrHeader->ucLength ;        
        
        } else if( pstRadiusAttrHeader->ucType == RADPKT_ATTR_TERMINATION_ACTION ) {
            
            //  maintain termination action
            memcpy( (Tuint8 *)&ulTerminationAction, (Tuint8 *)pstRadiusAttrHeader + 2, 4);
            ulOffset += pstRadiusAttrHeader->ucLength ;        
            
        } else if( pstRadiusAttrHeader->ucType == RADPKT_ATTR_INTERVAL ) {
            
            //  Interval Timer
            memcpy( (Tuint8 *)&ulAccInterimInterval, (Tuint8 *)pstRadiusAttrHeader + 2, 4);
            ulOffset += pstRadiusAttrHeader->ucLength ;
            
        } else if( pstRadiusAttrHeader->ucType == RADPKT_ATTR_CLASS ) {

            // Class Attribute 
            radacc_info[ucPortId].ucClassLen = pstRadiusAttrHeader->ucLength - 2;
            memcpy( radacc_info[ucSuppPort].aucClass, (Tuint8 *)pstRadiusAttrHeader + 2, 
	    	    radacc_info[ucPortId].ucClassLen );
            
            ulOffset += pstRadiusAttrHeader->ucLength ;
        
        } else {
            ulOffset += pstRadiusAttrHeader->ucLength ;
        }
        usRadPktLen -= pstRadiusAttrHeader->ucLength ;
    }

    if( bIsEapMess ) {

        radius_rx_eap_attr(ucSuppPort, radsrv_info[ucSuppPort].aucSWPort.ucIdentifier,
			   ucEapCode, ulEapPos, eap_buf );        
               
        if( pstRadiusPktHeader->ucCode == RADPKT_TYPE_ACCESS_ACCEPT )
	{
            memcpy( radacc_info[ucSuppPort].aucUserName, radsrv_info[ucSuppPort].aucSWPort.aucUserName, radsrv_info[ucSuppPort].aucSWPort.ucUserNameLen);
            radacc_info[ucSuppPort].ucUserNameLen = radsrv_info[ucSuppPort].aucSWPort.ucUserNameLen ;            
        }
        
        return True;    
    
    } 
    return False ;   
}



int p8021x_send_radius_pkt(Tuint8 *paucRadPkt, Tuint16 usRadPktLen, Tuint16 usUdpPort)
{
    Tuint32   ulRet;
   
    ulRet = udp_send(usUdpPort, paucRadPkt,usRadPktLen);

    return ulRet;    
    
}

//--------------------------------------------------   
 // create the request authenticator field for the radius packets.   
 //  depends only on rinfo -> identifier from the auth_pae   
 //  and stores it into rinfo -> req_authenticator   
 //--------------------------------------------------   
 void create_reqauth(Tuint8 *paucReqAuthenticator, Tbool bFlag)   
 {   
     MD5_CTX    st_MD5Context;   
     Tuint8     buff[16];   
     int 	random_num=net_random();   
    
     sprintf(buff,"%d",random_num);   
    
     MD5Init(&st_MD5Context);   
     MD5Update(&st_MD5Context, (Tuint8 *)buff, (Tuint32)(strlen(buff)));   
     MD5Final(paucReqAuthenticator, &st_MD5Context);   
    
 } 



/*
 *  create a new template radius pkt with header( code, identifier, authenticator) not length
 */
struct rad_pkt_info *create_radius_common_hdr( Tuint8 *pucpkt, Tuint8 ucCode, Tuint8 ucIdentifier, Tuint8 *pstReqAuthenticator)
{
	struct rad_pkt_info *pstRadConst;
        
        if(p8021x_malloc(sizeof(struct rad_pkt_info),(void **)&pstRadConst))
          {
            pstRadConst->pPkt = pucpkt ;
            pstRadConst->usPktLength = RADPKT_PKT_HEADER_LEN ;
	    pstRadConst->prad_header = (struct rad_header *)pucpkt;
	    pstRadConst->prad_header->ucCode = ucCode;
	    pstRadConst->prad_header->ucIdentifier = ucIdentifier;
	    memcpy( pstRadConst->prad_header->aucAuthenticator, pstReqAuthenticator, RADPKT_AUTHENTICATOR_LEN);
	    return pstRadConst;
	  }  

	return NULL;  
}

/*
 *  insert an attribute to radius pkt(not include eap message attribute)
 */
void radconst_addattr( struct rad_pkt_info * pstRadPkt, Tuint8 ucAttrType, Tuint8 ucAttrLen, Tuint8 *pucAttrData)
{
    struct rad_attr_header *pstRadAttr;
    
    pstRadAttr = (struct rad_attr_header *) &( pstRadPkt->pPkt[pstRadPkt->usPktLength]);
    pstRadAttr->ucType = ucAttrType ;
    pstRadAttr->ucLength = ucAttrLen + RADPKT_ATTR_HEAD_LEN ;
    memcpy( (Tuint8 *)pstRadAttr + RADPKT_ATTR_HEAD_LEN, pucAttrData, ucAttrLen);
    pstRadPkt->usPktLength += pstRadAttr->ucLength ;
    
    if( ucAttrType == RADPKT_ATTR_MESSAGE_AUTH )
        pstRadPkt->pMessageAuthenticator = ((Tuint8 *)pstRadAttr) + RADPKT_ATTR_HEAD_LEN ;
    if( ucAttrType == RADPKT_ATTR_NAS_PORT_TYPE )
        pstRadPkt->pulNASPortType = (Tuint32 *)(pstRadAttr + RADPKT_ATTR_HEAD_LEN ) ;
}


//---------------------------------------------
//add eap message attribute in radius packet 
//---------------------------------------------
void radconst_addEAPMessAttr( struct rad_pkt_info * pstRadPkt, Tint16 usAttrLen, Tuint8 *pucAttrData)
{
    struct  rad_attr_header *pstRadAttr;
    Tuint8  *paucDataPtr;
    Tint16  usActualLen;
    
    paucDataPtr = pucAttrData;
    for(/*do nothing*/  ; usAttrLen > 0 ; usAttrLen -= RADPKT_MAX_EAP_ATTR_LEN ) {
        if( usAttrLen>RADPKT_MAX_EAP_ATTR_LEN ) 
            usActualLen = RADPKT_MAX_EAP_ATTR_LEN ;
        else 
            usActualLen = usAttrLen ;
        
        pstRadAttr = (struct rad_attr_header *) &( pstRadPkt->pPkt[pstRadPkt->usPktLength]);
        pstRadAttr->ucType = RADPKT_ATTR_EAP_MESSAGE ;
        pstRadAttr->ucLength = usActualLen + RADPKT_ATTR_HEAD_LEN ;
        memcpy( (Tuint8 *) pstRadAttr+ RADPKT_ATTR_HEAD_LEN, paucDataPtr, usActualLen);
        paucDataPtr += usActualLen ;
        pstRadPkt->usPktLength += pstRadAttr->ucLength ;
    }
}    

//------------------------------------------------
// Calculate the Packet length of Radius message
//------------------------------------------------ 
void radconst_calradlength( struct rad_pkt_info * pstRadPkt)
{
    struct rad_header *pstPktHdr;
    
    pstPktHdr =(struct rad_header *) pstRadPkt->pPkt ;
    pstPktHdr->usLength = htons(pstRadPkt->usPktLength);

}


//--------------------------------------------------
// create the message authenticator field for the 
// radius packets.
//--------------------------------------------------
void create_messauth( struct rad_pkt_info *pstRadConst, Tuint8 *pucMessAuth )
{
  hmac_md5( pstRadConst->pPkt, pstRadConst->usPktLength, radsrv_info[0].aucShareKey, 
   	    strlen(radsrv_info[0].aucShareKey), pucMessAuth ); 
}


Tuint32 radius_info_init()
{
   Tuint8    ucPortId;
    
   
   for( ucPortId=0 ; ucPortId<MAX_PORT ; ucPortId++ ) 
   {
        radsrv_info[ucPortId].ucIdentifier = 0 ;
        radsrv_info[ucPortId].aucSWPort.bStatus = False ;
        radsrv_info[ucPortId].aucSWPort.ucIdentifier = 0 ;

        memset(radsrv_info[ucPortId].aucSWPort.aucReqAuthenticator, 0x0, RADPKT_AUTHENTICATOR_LEN);
        memset(radsrv_info[ucPortId].aucSWPort.aucMessAuthenticator, 0x0, RADPKT_AUTHENTICATOR_LEN);

        memset(radsrv_info[ucPortId].aucSWPort.aucEAPMessFromSupp, 0x0, RADPKT_MAX_PKT_LEN);
        radsrv_info[ucPortId].aucSWPort.aucEAPMessTypeFromSupp = 0;        
        radsrv_info[ucPortId].aucSWPort.usEAPMessLenFromSupp = 0;
        
        memset(radsrv_info[ucPortId].aucSWPort.aucEAPMessFromServer, 0x0, RADPKT_MAX_PKT_LEN);
        radsrv_info[ucPortId].aucSWPort.aucEAPMessTypeFromServer = 0;        
        radsrv_info[ucPortId].aucSWPort.usEAPMessLenFromServer = 0;

        memset(radsrv_info[ucPortId].aucSWPort.aucUDPSendToServer, 0x0, RADPKT_MAX_PKT_LEN);
        radsrv_info[ucPortId].aucSWPort.usUPDLenToServer = 0;
        
        memset(radsrv_info[ucPortId].aucSWPort.aucEAPSendToSupp, 0x0, RADPKT_MAX_PKT_LEN);
        radsrv_info[ucPortId].aucSWPort.usEAPLenToSupp = 0;
            
        memset(radsrv_info[ucPortId].aucSWPort.aucUserName, 0x0, RADPKT_MAX_USERNAME_LEN);
        radsrv_info[ucPortId].aucSWPort.ucUserNameLen=0;        
        
        radsrv_info[ucPortId].aucSWPort.bStateAvailable = False ;
        radsrv_info[ucPortId].aucSWPort.usStateLen = 0;
        memset( radsrv_info[ucPortId].aucSWPort.aucStateAttr, 0x0, RADPKT_MAX_STATE_ATTR_LEN);
        
        radsrv_info[ucPortId].aucSWPort.bClassAvailable = False ;
        radsrv_info[ucPortId].aucSWPort.usClassLen = 0;
        memset( radsrv_info[ucPortId].aucSWPort.aucClassAttr, 0x0, RADPKT_MAX_CLASS_ATTR_LEN);

        memset( radsrv_info[ucPortId].aucSWPort.aucConnectInfo, 0x0, RADPKT_MAX_CONNECT_INFO_LEN);
        memcpy(radsrv_info[ucPortId].aucSWPort.aucConnectInfo, CONNECT_INFO, strlen(CONNECT_INFO));     
        
        radacc_info[ucPortId].ucIdentifier = 0;
        memset( radacc_info[ucPortId].aucUserName, 0x0, RADPKT_MAX_USERNAME_LEN );
        radacc_info[ucPortId].ucUserNameLen = 0;
        memset( radacc_info[ucPortId].aucReqAuthenticator, 0x0, RADPKT_AUTHENTICATOR_LEN );
        memset( radacc_info[ucPortId].aucClass, 0x0, RADPKT_MAX_CLASS_ATTR_LEN );
        radacc_info[ucPortId].ucClassLen = 0;
    }        

    return True;
}


//---------------------------------------
//  Sending Radius Accounting message
//---------------------------------------
Tuint32 radius_acc_pkt_tx(Tuint8 ucPort, RadAccInfo_T *pstRadAccInfo)
{
    static Tuint8   auc_pkt[RADPKT_MAX_PKT_LEN];
    static Tuint8   auc_tmp_pkt[RADPKT_MAX_PKT_LEN]; 
    Tuint8   aucMessAuth[20];
    Tuint8   ucSharedKeyLen;
    Tuint32  ulPortNo, ulDelayTime = 0;
    Tuint32  ulServiceType = htonl(RADPKT_ATTR_STYPE_FRAMED);
    Tuint32  ulAuthentic   = htonl(RADPKT_ACC_AUTH_RADIUS) ;
    Tuint32  ulStatusType  = htonl(pstRadAccInfo->ulStatusType);
    struct   rad_pkt_info  *pstRadPktConst;
    MD5_CTX  st_MD5Context;        
            
    memset( auc_pkt, 0x0, RADPKT_MAX_PKT_LEN );
    memset( auc_tmp_pkt, 0x0, RADPKT_MAX_PKT_LEN );
    memset( aucMessAuth, 0x0, 20);
 
    radsrv_info[ucPort].ucIdentifier += 2; 
    radacc_info[ucPort].ucIdentifier =  radsrv_info[ucPort].ucIdentifier ;
    
    pstRadPktConst = create_radius_common_hdr(auc_pkt, RADPKT_TYPE_ACC_REQUEST, radsrv_info[ucPort].ucIdentifier, aucMessAuth);

    if(pstRadPktConst==NULL)
       return False;
    
    // User Name(1)
    radconst_addattr( pstRadPktConst, RADPKT_ATTR_USER_NAME, radsrv_info[ucPort].aucSWPort.ucUserNameLen , radsrv_info[ucPort].aucSWPort.aucUserName); 

    // NAS IP Address(4)
    radconst_addattr( pstRadPktConst, RADPKT_ATTR_NAS_IP_ADDRESS, 0x4, (Tuint8 *) &(radsrv_info[0].ipNas) );

    // NAS Port (5)
    ulPortNo = htonl(ucPort + 1) ;
    radconst_addattr( pstRadPktConst, RADPKT_ATTR_NAS_PORT, 0x4, (Tuint8 *)&ulPortNo );

    // Service Type(6)
    radconst_addattr( pstRadPktConst, RADPKT_ATTR_SERVICE_TYPE, 0x4, (Tuint8 *)&ulServiceType );

  
    // Class(25)        
    if( radacc_info[ucPort].ucClassLen != 0 ){ 
        radconst_addattr( pstRadPktConst, RADPKT_ATTR_CLASS, (Tuint8)(strlen(radacc_info[ucPort].aucClass)), (Tuint8 *)radacc_info[ucPort].aucClass );    
    }

    // Called Station ID(30)
    radconst_addattr( pstRadPktConst, RADPKT_ATTR_CALLED_STID, RADPKT_MAX_CALLED_ID_LEN, 
    		      (Tuint8 *)radsrv_info[0].aucCalledStationID);

    // Calling Station ID(31)
    radconst_addattr( pstRadPktConst, RADPKT_ATTR_CALLING_STID, RADPKT_MAX_CALLING_ID_LEN,
    		      (Tuint8 *)port_sm[ucPort]->Auth->calling_id );

    // NAS Identifer(32)
    radconst_addattr( pstRadPktConst, RADPKT_ATTR_NAS_IDENTIFIER, (Tuint8)(strlen(radsrv_info[0].aucNASIdentifier)), (Tuint8 *)radsrv_info[0].aucNASIdentifier );
        
    // Accounting Status Type(40)
    radconst_addattr( pstRadPktConst, RADPKT_ACC_ATTR_STATUS_TYPE, 0x4,(Tuint8 *)&(ulStatusType) );       
        
    // Accounting Delay Time(41)
    radconst_addattr( pstRadPktConst, RADPKT_ACC_ATTR_DELAY_TIME, 0x4, (Tuint8 *)&ulDelayTime );       

    // Input Octets(42)
    radconst_addattr( pstRadPktConst, RADPKT_ACC_ATTR_INPUT_OCTETS, 0x4, (Tuint8 *)&(pstRadAccInfo->ulInputOctets) );       

    // Output Octets(43)
    radconst_addattr( pstRadPktConst, RADPKT_ACC_ATTR_OUTPUT_OCTETS, 0x4, (Tuint8 *)&(pstRadAccInfo->ulOutputOctets) );       
    //  Authentic(45)    
    radconst_addattr( pstRadPktConst, RADPKT_ACC_ATTR_AUTHENTIC, 0x4, (Tuint8 *)&ulAuthentic );       
    
    // Session time(46)    
    radconst_addattr( pstRadPktConst, RADPKT_ACC_ATTR_SESSION_TIME, 0x4, (Tuint8 *)&(pstRadAccInfo->ulSessionTime) );       
    // Input Packets(47)    
    radconst_addattr( pstRadPktConst, RADPKT_ACC_ATTR_INPUT_PKTS, 0x4, (Tuint8 *)&(pstRadAccInfo->ulInputPkts) );       
    
    // Output Packets(48)    
    radconst_addattr( pstRadPktConst, RADPKT_ACC_ATTR_OUTPUT_PKTS, 0x4, (Tuint8 *)&(pstRadAccInfo->ulOutputPkts) );       

    //  Terminate Casue(49)   
    if( pstRadAccInfo->ulStatusType == RADPKT_ACC_STATUS_STOP ) {       
        radconst_addattr( pstRadPktConst, RADPKT_ACC_ATTR_TM_COUSE, 0x4, (Tuint8 *)&(pstRadAccInfo->ulTerminateCause) );
    }

    radconst_calradlength( pstRadPktConst );    
    
    // calculate request authenticator 
    memcpy( auc_tmp_pkt, pstRadPktConst->pPkt, pstRadPktConst->usPktLength );
    ucSharedKeyLen = strlen(radsrv_info[0].aucShareKey) ;
    memcpy( &auc_tmp_pkt[pstRadPktConst->usPktLength], radsrv_info[0].aucShareKey, ucSharedKeyLen );    
    MD5Init(&st_MD5Context);    
    MD5Update(&st_MD5Context, (Tuint8 *)auc_tmp_pkt, (pstRadPktConst->usPktLength + ucSharedKeyLen));
    MD5Final(&auc_pkt[4], &st_MD5Context);
    memcpy( radacc_info[ucPort].aucReqAuthenticator, &auc_pkt[4], 0x10);
    
    p8021x_send_radius_pkt(pstRadPktConst->pPkt, pstRadPktConst->usPktLength,radsrv_info[0].usAuthPort );
    p8021x_free( pstRadPktConst );        
    
    return True;
} 
 
//--------------------------------------------------
//   send radius account pkt
//--------------------------------------------------
Tbool rad_send_acc_info(Global_Params *global,Tuint8 nType,Tuint32 nStopCause)
{
        static  Tuint32 session_id=0;
	RadAccInfo_T stRadAcc;
        Tuint8  ucPhyPortId;
        Tuint32 ulRxPkt64,ulRxPkt65to127,ulRxPkt128to255,ulRxPkt256to511;
	Tuint32 ulRxPkt512to1023,ulRxPkt1024to1518;
        Tuint32 ulRxOctet,ulTxOctet, ulTxNUcastPkt, ulTxUcastPkt;

	ucPhyPortId = cclmx_LId2PId(global->nPortId);

        K_MibGetEtherStatsPkts64Octets(ucPhyPortId, &ulRxPkt64);
	K_MibGetEtherStatsPkts65to127Octets(ucPhyPortId, &ulRxPkt65to127);
	K_MibGetEtherStatsPkts128to255Octets(ucPhyPortId, &ulRxPkt128to255);
	K_MibGetEtherStatsPkts256to511Octets(ucPhyPortId, &ulRxPkt256to511);
	K_MibGetEtherStatsPkts512to1023Octets(ucPhyPortId, &ulRxPkt512to1023);
	K_MibGetEtherStatsPkts1024to1518Octets(ucPhyPortId, &ulRxPkt1024to1518);
	K_MibGetEtherStatsOctets(ucPhyPortId, &ulRxOctet);
        K_MibGetEtherStatsTxUcastPkts(ucPhyPortId, &ulTxUcastPkt);
        K_MibGetEtherStatsTxNUcastPkts(ucPhyPortId, &ulTxNUcastPkt);
        K_MibGetEtherStatsTxOctets(ucPhyPortId, &ulTxOctet);


	//record start info
	global->dwSessId       = session_id++;
	global->dwSessTime     = jiffies; 
	global->dwSessRxOct    = ulRxOctet;
	global->dwSessTxOct    = ulTxOctet;
	global->dwSessRxPkt    = ulRxPkt64 + ulRxPkt65to127 + ulRxPkt128to255 +
		   	         ulRxPkt256to511 + ulRxPkt512to1023 + ulRxPkt1024to1518;
	global->dwSessTxPkt    = ulTxNUcastPkt + ulTxUcastPkt;


	if(nType == RADPKT_ACC_STATUS_START)
	  {
     	    stRadAcc.ulStatusType = RADPKT_ACC_STATUS_START;
          }
	else if(nType == RADPKT_ACC_STATUS_UPDATE)
	  {
    	    stRadAcc.ulStatusType = RADPKT_ACC_STATUS_UPDATE;
   	  }
        else if(nType == RADPKT_ACC_STATUS_STOP)
          {
	    stRadAcc.ulStatusType     = RADPKT_ACC_STATUS_STOP;
	    stRadAcc.ulTerminateCause = htonl(nStopCause);
 	  }

        stRadAcc.ucPortNo           = htonl(global->nPortId);
	stRadAcc.ulSessionId        = htonl(global->dwSessId);
	stRadAcc.ulSessionTime      = htonl(global->dwSessTime);
	stRadAcc.ulInputOctets      = htonl(global->dwSessRxOct);
	stRadAcc.ulOutputOctets     = htonl(global->dwSessTxOct);
	stRadAcc.ulInputPkts        = htonl(global->dwSessRxPkt);
	stRadAcc.ulOutputPkts       = htonl(global->dwSessTxPkt);

	//send Radius Account function
	radius_acc_pkt_tx(global->nPortId,&stRadAcc);

	return True;
}

/*
 *  initialize port configuration info
 */
Tuint32 radius_port_info_init(Tuint8 ucPortId)
{
    radsrv_info[ucPortId].aucSWPort.bStatus = False ;

    memset(radsrv_info[ucPortId].aucSWPort.aucReqAuthenticator, 0x0, RADPKT_AUTHENTICATOR_LEN);
    memset(radsrv_info[ucPortId].aucSWPort.aucMessAuthenticator, 0x0, RADPKT_AUTHENTICATOR_LEN);

    memset(radsrv_info[ucPortId].aucSWPort.aucEAPMessFromSupp, 0x0, RADPKT_MAX_PKT_LEN);
    radsrv_info[ucPortId].aucSWPort.aucEAPMessTypeFromSupp = 0;        
    radsrv_info[ucPortId].aucSWPort.usEAPMessLenFromSupp = 0;
    
    memset(radsrv_info[ucPortId].aucSWPort.aucEAPMessFromServer, 0x0, RADPKT_MAX_PKT_LEN);
    radsrv_info[ucPortId].aucSWPort.aucEAPMessTypeFromServer = 0;        
    radsrv_info[ucPortId].aucSWPort.usEAPMessLenFromServer = 0;

    memset(radsrv_info[ucPortId].aucSWPort.aucUDPSendToServer, 0x0, RADPKT_MAX_PKT_LEN);
    radsrv_info[ucPortId].aucSWPort.usUPDLenToServer = 0;
    
    memset(radsrv_info[ucPortId].aucSWPort.aucEAPSendToSupp, 0x0, RADPKT_MAX_PKT_LEN);
    radsrv_info[ucPortId].aucSWPort.usEAPLenToSupp = 0;
        
    memset(radsrv_info[ucPortId].aucSWPort.aucUserName, 0x0, RADPKT_MAX_USERNAME_LEN);
    radsrv_info[ucPortId].aucSWPort.ucUserNameLen=0;        
    
    radsrv_info[ucPortId].aucSWPort.bStateAvailable = False ;
    radsrv_info[ucPortId].aucSWPort.usStateLen = 0;
    memset( radsrv_info[ucPortId].aucSWPort.aucStateAttr, 0x0, RADPKT_MAX_STATE_ATTR_LEN);
    
    radsrv_info[ucPortId].aucSWPort.bClassAvailable = False ;
    radsrv_info[ucPortId].aucSWPort.usClassLen = 0;
    memset( radsrv_info[ucPortId].aucSWPort.aucClassAttr, 0x0, RADPKT_MAX_CLASS_ATTR_LEN);

    memset( radsrv_info[ucPortId].aucSWPort.aucConnectInfo, 0x0, RADPKT_MAX_CONNECT_INFO_LEN);
    memcpy( radsrv_info[ucPortId].aucSWPort.aucConnectInfo, CONNECT_INFO, strlen(CONNECT_INFO));         

    radacc_info[ucPortId].ucClassLen = 0 ;
    memset( radacc_info[ucPortId].aucClass , 0x0, RADPKT_MAX_CLASS_ATTR_LEN);

   return True;
}


/* Radius Pkt rx handler.  */
Tuint32 radius_pkt_rx(Pdu *pkt, void * data)
{
    Tuint8 *pucRadiusPkt;
    struct iphdr *iph = pkt->nh.iph;

    pucRadiusPkt = (Tuint8 *)(pkt->data + (iph->ihl*4) + 8) ;    
    radius_rx2eap_tx(pucRadiusPkt);
    
    return True;         
}


