/***************************************************************************
 *
 *  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
 *
 *  THIS IS AN UNPUBLISHED WORK WHICH CONTAINS CONFIDENTIAL INFORMATION
 *  FROM CCL, ITRI.  NO PART OF THIS WORK MAY BE USED IN ANY WAY WITHOUT
 *  THE PRIOR WRITTEN PERMISSION.  ANY UNAUTHORIZED USE COULD SUBJECT THE
 *  PERPETRATOR TO CRIMINAL AND CIVIL LIABILITY.
 *
 *  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
 *  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *
 ***************************************************************************/

// ********************************************
// Name   : 8021x_bauth_sm.c
// Author : Liu, Ren Hao (N300, CCL/ITRI)
// Date   : 2003/12/04
// Note   : 802.1x Backend Authentication State Machine (BAUTHSM)
// ********************************************
// Programmer   : Arunesh Mishra
// Copyright (c) Arunesh Mishra 2002
// All rights reserved.
// Maryland Information and Systems Security Lab
// University of Maryland, College Park.

#include "8021x.h"

//--------------------------------------------------
// Initialize it.
//--------------------------------------------------
void bauthsm_init(Bauth_SM *bauth_sm)
{	
  bauth_sm->state	 = basm_Initialize;
  bauth_sm->reqCount     = 0;
  bauth_sm->rxResp       = False;
  bauth_sm->aSuccess     = False;
  bauth_sm->aFail        = False;
  bauth_sm->aReq         = False;
  bauth_sm->idFromServer = -1;  // Neg value = invalid value

  // These are the constants initialized to their respective default values.
  bauth_sm->suppTimeout   = 0;
  bauth_sm->serverTimeout = 0;
  bauth_sm->maxReq        = 0; 
}

//--------------------------------------------------
//  This function implements a single transition for the Backend Authentication
//  State Machine. .. but not the initialization !!
//--------------------------------------------------
Tbool trans_bauthsm(Global_Params *global)
{
  Auth_Pae   *auth_pae = global->Auth;
  Bauth_SM   *bauth_sm = global->Auth->bauth_sm;
	

  // Global Transitions first !
  if ( (global->portControl != pmt_Auto) || (global->initialize) || (global->authAbort) ) 
  {
	  bauth_sm->state = basm_Initialize;
	  bauthsm_initialize(global);
	  return True;
  }

  switch (bauth_sm->state)
  {
    case basm_Initialize: 

	    bauth_sm -> state = basm_Idle ; 	// Unconditional Transfer !
	    bauthsm_idle(global);
	    return True;

    case basm_Request:

	    if(bauth_sm->rxResp) 
	    {
		    bauth_sm->state = basm_Response;
		    bauthsm_response(global);
		    return True;
	    }

	    if ( (auth_pae->port_timers->aWhile == 0) && 
			    (bauth_sm->reqCount != bauth_sm->maxReq) )
	    {

		    bauth_sm->state = basm_Request;
		    bauthsm_request(global);
		    return True;
	    }

	    if ( (auth_pae->port_timers->aWhile == 0) && 
			    (bauth_sm->reqCount >= bauth_sm->maxReq) )
	    {
		    if(auth_pae->fromsvr.valid != 0) {
			    auth_pae->fromsvr.valid    = 0;
			    auth_pae->fromsvr.length = 0;			
		    }

		    radius_port_info_init(global->nPortId);

		    bauth_sm -> state = basm_Timeout;
		    bauthsm_timeout(global);
		    return True;
	    }
	    break;

    case basm_Response:

	    if(bauth_sm->aReq) 
	    {
		    bauth_sm->state = basm_Request;
		    bauthsm_request(global);	
		    return True;
	    }

	    if(auth_pae->port_timers->aWhile == 0)
	    {
		    radius_port_info_init(global->nPortId);
		    bauth_sm -> state = basm_Timeout;
		    bauthsm_timeout(global);
		    return True;		
	    } 

	    if (bauth_sm->aFail)
	    {
		    bauth_sm->state = basm_Fail;
		    bauthsm_fail(global);
		    return True;
	    }

	    if ( bauth_sm->aSuccess )
	    {
		    bauth_sm->state = basm_Success;
		    bauthsm_success(global);
		    return True;
	    }
	    break;

    case basm_Success:

	    bauth_sm->state = basm_Idle; 	// Unconditional Transfer !
	    bauthsm_idle(global);
	    return True;

    case basm_Timeout:

	    bauth_sm->state = basm_Idle ; 	// Unconditional Transfer !
	    bauthsm_idle(global);
	    return True;

    case basm_Fail:

	    bauth_sm -> state = basm_Idle ; 	// Unconditional Transfer !
	    bauthsm_idle(global);
	    return True;

    case basm_Idle:

	    if (global->authStart)
	    {
		    bauth_sm->state = basm_Response;
		    bauthsm_response(global);
		    return True;
	    }
	    break;
  }
  return False;
}

//--------------------------------------------------
//  Initialize state inits.
//--------------------------------------------------
void bauthsm_initialize(Global_Params *global)
{
  bauthsm_abortAuth(global); 
  global->authAbort = False;
}

//--------------------------------------------------
//  Inits for the request state.
//--------------------------------------------------
void bauthsm_request(Global_Params * global)
{
  Auth_Pae   *auth_pae = global->Auth;
  Bauth_SM   *bauth_sm = global->Auth->bauth_sm;

  global->currentId = bauth_sm->idFromServer ;
  bauthsm_txReq(global); 
  auth_pae->port_timers->aWhile = bauth_sm->suppTimeout;
  bauth_sm->reqCount++;

}

//--------------------------------------------------
// Response state inits.
//--------------------------------------------------
void bauthsm_response(Global_Params *global)
{
  Auth_Pae *auth_pae = global->Auth;
  Bauth_SM *bauth_sm = global->Auth->bauth_sm;

  bauth_sm->aReq			  = False; 
  bauth_sm->aSuccess			  = False;
  global->authTimeout          		  = False;
  bauth_sm->rxResp			  = False;
  bauth_sm->aFail			  = False;
  auth_pae->port_timers->aWhile		  = bauth_sm->serverTimeout;
  bauth_sm->reqCount			  = 0;

  bauthsm_sendRespToServer(global);	
}

//--------------------------------------------------
// Success state inits.
//--------------------------------------------------
void bauthsm_success(Global_Params *global)
{
  Bauth_SM *bauth_sm = global->Auth->bauth_sm;

  global->currentId = bauth_sm->idFromServer;
  auth_txCannedSuccess(global);
  global->authSuccess = True;
}

//--------------------------------------------------
// Fail state inits.
//--------------------------------------------------
void bauthsm_fail(Global_Params *global)
{	
  Bauth_SM *bauth_sm = global->Auth->bauth_sm;

  global->currentId = bauth_sm->idFromServer;
  auth_txCannedFail(global);
  global->authFail = True;
}

//--------------------------------------------------
// Timeout state inits.
//--------------------------------------------------
void bauthsm_timeout(Global_Params *global)
{
  if(global->portStatus == pst_Unauthorized) {
     auth_txCannedFail(global);
  }
  global->authTimeout = True;
}

//---------------------------------------------------
//Idle state inits.
//--------------------------------------------------
void bauthsm_idle(Global_Params *global)
{
  Bauth_SM *bauth_sm = global->Auth->bauth_sm;

  global->authStart = False;
  bauth_sm->reqCount = 0;
}

//--------------------------------------------------
// bauthsm releases any system resources and
// informs auth sm of auth abort.
//--------------------------------------------------
void bauthsm_abortAuth(Global_Params *global)
{
  Auth_Pae *auth_pae = global->Auth;
	
  auth_pae->fromsupp.valid   = 0;
  auth_pae->fromsupp.length  = 0;
  auth_pae->fromsvr.valid    = 0;
  auth_pae->fromsvr.length   = 0;

}

//----------------------------------------
// txReq(x): EAPOL frame of type 
// EAP Request
//----------------------------------------
Tuint8 bauthsm_txReq(Global_Params *global)
{
  Auth_Pae 	   *auth_pae = global->Auth;
  struct eapol_pkt eapol={0};
  static Tbyte     pBuf[1600];
  Tuint16 	   size=0;


  if(auth_pae->fromsvr.valid != 0)
  {
    eapol.protocol_version   = X_EAPOL_VER;
    eapol.packet_type        = X_EAPOL_EAPPKT;
    eapol.packet_body_length = htons(auth_pae->fromsvr.length);

    size = X_EAPOL_HDRLEN + auth_pae->fromsvr.length;

    if(size > ETHER_MAXSIZE)
    {
      return KRN_RET_ERR_DOT1X_VALUE_INVALID;
    }

    memcpy(pBuf ,&eapol,X_EAPOL_HDRLEN);
    memcpy(pBuf+X_EAPOL_HDRLEN,auth_pae->fromsvr.pkt, auth_pae->fromsvr.length);
    //printk("bauthsm_txReq -> ");
    eapol_tx_handler(global->nPortId,pBuf,size);
    
    //buffer is ready to receive another packet from radius server
    auth_pae->fromsvr.valid  = 0;
    auth_pae->fromsvr.length = 0;			
  }
  else
  {
     return KRN_RET_ERR_DOT1X_EMPTY_BUFFER;
  }

  return KRN_RET_OK;
}

//--------------------------------------------------
// sendRespToServer(x): frame of type
// EAP Response to server
//--------------------------------------------------
Tuint8 bauthsm_sendRespToServer(Global_Params *global)
{
  Auth_Pae *auth_pae = global->Auth;

  if(auth_pae->fromsupp.valid != 0)
    {
      eap_rx2radius_tx(global->nPortId,auth_pae,auth_pae->fromsupp.pkt);

      //buffer is ready to receive another packet from supplicant
      auth_pae->fromsupp.valid    = 0;
      auth_pae->fromsupp.length = 0;
      return KRN_RET_OK;
    }
  else
  {
      return KRN_RET_ERR_DOT1X_EMPTY_BUFFER;
  }
}

