/***************************************************************************
*
*  Copyright (C) 2003-2005 CCL, ITRI.  All Rights Reserved.
*
*  THIS IS AN UNPUBLISHED WORK WHICH CONTAINS CONFIDENTIAL INFORMATION
*  FROM CCL, ITRI.  NO PART OF THIS WORK MAY BE USED IN ANY WAY WITHOUT
*  THE PRIOR WRITTEN PERMISSION.  ANY UNAUTHORIZED USE COULD SUBJECT THE
*  PERPETRATOR TO CRIMINAL AND CIVIL LIABILITY.
*
*  CCL, ITRI IS NOT RESPONSIBLE OR LIABLE FOR ANY DIRECT, INDIRECT,
*  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES THAT MAY RESULT FROM
*  THE USE, OR INABILITY TO USE OF THIS WORK.  ANY EXPRESSED OR IMPLIED
*  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
*  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
*
***************************************************************************/

// ********************************************
// Name   : 8021x_auth_pae.c
// Author : Liu, Ren Hao (N300, CCL/ITRI)
// Date   : 2003/12/04
// Note   : 802.1x Port Access Entity function (PAE)
// ********************************************
// Programmer   : Arunesh Mishra
// Copyright (c) Arunesh Mishra 2002
// All rights reserved.
// Maryland Information and Systems Security Lab
// University of Maryland, College Park.

#include "krndef.h"
#include "krnerr.h"
#include "krnport.h"
#include "krntype.h"

#include "8021x.h"

//--------------------------------------------------
// auth_init:
//   
//  Initialize various other state machines and construct a
//  complete authenticator. The Authenticator requires the 
//  following state machines :
// 	1. Port Timers
// 	2. Auth PAE
// 	3. Reauth Timer
// 	4. Backend Auth
// 	5. Controlled Dir
//--------------------------------------------------
Tuint8 auth_init(Global_Params *global)
{
  Auth_Pae    *auth_pae    = NULL;
  PT_SM	      *port_timers = NULL;	
  Reauth_SM   *reauth_sm   = NULL;  
  Bauth_SM    *bauth_sm    = NULL;

  // Init Authenticator PAE
  if(!p8021x_malloc(sizeof(Auth_Pae),(void **)&auth_pae))
	  return KRN_RET_ERR_DOT1X_VALUE_INVALID;

  global->Auth = auth_pae;

  // 1. Init port timers
  if(!p8021x_malloc(sizeof(PT_SM),(void **)&port_timers))
	  return KRN_RET_ERR_DOT1X_VALUE_INVALID;

  auth_pae->port_timers = port_timers;

  //used by authenticator.
  port_timers->aWhile     =  port_timers->quietWhile = 0; 
  port_timers->reAuthWhen =  port_timers->txWhen     = 0;    

  // 2. Init Reauthentication State Machine
  if(!p8021x_malloc(sizeof(Reauth_SM),(void **)&reauth_sm))
	  return KRN_RET_ERR_DOT1X_VALUE_INVALID;

  auth_pae->reauth_sm = reauth_sm;
  rauthsm_init(reauth_sm);

  // 3. Init Backend Authentication State Machine
  if(!p8021x_malloc(sizeof(Bauth_SM),(void **)&bauth_sm))
	  return KRN_RET_ERR_DOT1X_VALUE_INVALID;

  auth_pae->bauth_sm = bauth_sm;
  bauthsm_init(bauth_sm);

  // 4. Initialize Authenticator Port Access Entity .. state machine also !
  auth_pae->state       = apsm_Connecting;
  auth_pae->eapLogoff   = False;
  auth_pae->eapStart    = False;
  auth_pae->portMode    = pmt_None;
  auth_pae->rxRespId    = False;
  auth_pae->reAuthCount = 0;
  auth_pae->quietPeriod = 0; 
  auth_pae->reAuthMax   = 0;
  auth_pae->txPeriod    = 0;

  auth_pae ->fromsupp.valid   = 0;
  auth_pae ->fromsupp.length  = 0; 
  auth_pae ->fromsvr.valid    = 0;
  auth_pae ->fromsvr.length   = 0; 

  return KRN_RET_OK;
}



//--------------------------------------------------
//   Count "Port Timers State Machine" timer        
//--------------------------------------------------
void auth_timer_check(Global_Params *global)
{
  PT_SM *ptsm = global->Auth->port_timers;

  DEC(ptsm->aWhile);
  DEC(ptsm->quietWhile);
  DEC(ptsm->txWhen);
  DEC(ptsm->reAuthWhen);
}

//--------------------------------------------------
// This function does an authenticator's job .. i.e. runs
// the state machines .. one transition.
// Call this in a loop !
//--------------------------------------------------
void auth_state_check(Global_Params *global)
{
  // first the auth_pae state machine
  trans_authsm(global);

  // Backend Authentication State Machine
  trans_bauthsm(global);

  // Reauthentication Timer State Machine.
  trans_rauthsm(global);

  if(global->initialize)
     global->initialize = False;
	
}

//--------------------------------------------------
// trans_authsm : 
// 
// This function transitions the auth pae state 
// machine.
//--------------------------------------------------
Tbool trans_authsm(Global_Params *global)
{
  Auth_Pae *auth_pae = global->Auth;
	
  // Check Global Conditions Here.

  // Condition 1:
  if ( (global->portControl == pmt_Auto) && 
		  ( (auth_pae->portMode != global->portControl) ||
		    (global->initialize) || !global->portEnabled ))
  {
    auth_pae->state = apsm_Initialize;
    authsm_initialize(global);		
    return True;
  }

  // Condition 2:
  if ( (global->portControl == pmt_ForceAuthorized) &&
       (auth_pae->portMode != global->portControl) &&
       !(global->initialize || !global->portEnabled) )
  {	
     auth_pae->state    = apsm_Force_Auth;
     authsm_force_auth(global);
     return True;
  }

  // Condition 3:
  if ( (global->portControl == pmt_ForceUnauthorized) &&
       (auth_pae->portMode != global->portControl) &&
       !(global->initialize || !global->portEnabled) )
  {	
      auth_pae->state = apsm_Force_Unauth;
      authsm_force_unauth(global);
      return True;
  }
	

  switch(auth_pae->state)
  {
    case apsm_Initialize:

	 auth_pae->state = apsm_Disconnected;	// Unconditional transfer 
	 authsm_disconnected(global);
	 return True;

    case apsm_Disconnected:

	 auth_pae->state = apsm_Connecting;	// Unconditional transfer
	 //printk("apsm_Disconnected: ");
	 authsm_connecting(global);
	 return True;

    case apsm_Connecting:

	 radius_port_info_init(global->nPortId);

	 //Condition 1.
	 if ( auth_pae->rxRespId && (auth_pae->reAuthCount <= auth_pae->reAuthMax) )
	 {
	    auth_pae->state = apsm_Authenticating;
	    authsm_authenticating(global);
	    return True;
	 }

	 //Condition 2.		
	 if ( ( (auth_pae->port_timers->txWhen == 0 ) || auth_pae->eapStart  || global -> reAuthenticate )
			 && (auth_pae->reAuthCount <= auth_pae->reAuthMax) )
	 {
   	     auth_pae->state = apsm_Connecting;
	     //printk("apsm_Connecting: ");
	     authsm_connecting(global);
	     return True;		
	 }     

	 //Condition 3.
	 if( auth_pae->eapLogoff || (auth_pae->reAuthCount > auth_pae->reAuthMax) )
	 {
    	     auth_pae->state = apsm_Disconnected;
	     authsm_disconnected(global);
	     return True;
	 }
	 break;

    case apsm_Authenticating:

	 //Condition 1.
	 if (global->authSuccess)
	 {
 	      auth_pae->state = apsm_Authenticated;

	      //when port x become authenticated, reset re-authenticate timer for port x
	      //prevent all ports have to re-authenticate at the same time.
	      auth_pae->port_timers->reAuthWhen = auth_pae->reauth_sm->reAuthPeriod;

	      authsm_authenticated(global);
	      return True;
	 }

	 //Condition 2.
	 if (global->authFail)
	 {
  	       auth_pae->state = apsm_Held;
	       authsm_held(global);
	       return True;
	 }

	 //Condition 3.
	 if ( global->reAuthenticate || auth_pae->eapStart ||
  	      auth_pae->eapLogoff || global->authTimeout )
	 {
	        auth_pae->state = apsm_Aborting;
		authsm_aborting(global);
		return True;
	 }

	 break;

    case apsm_Authenticated:

	 //Condition 1.
	 if ( auth_pae->eapStart || global->reAuthenticate )
	 {
		 auth_pae->state = apsm_Connecting;
		 //printk("apsm_Authenticated: ");
		 authsm_connecting(global);
		 return True;
	 }

	 //Condition 2.
	 if ( auth_pae->eapLogoff ) 
	 {
		 rad_send_acc_info(global,RADPKT_ACC_STATUS_STOP,RADPKT_ACC_TERM_USERREQ); 
		 auth_pae->state = apsm_Disconnected;   
		 authsm_disconnected(global); 
		 return True;
	 }
	 break;

    case apsm_Aborting:

	 //Condition 1.
	 if ( auth_pae->eapLogoff && !global->authAbort )
	 {
		 auth_pae->state = apsm_Disconnected;
		 authsm_disconnected(global);
		 return True;
	 }

	 //Condition 2.
	 if ( !auth_pae->eapLogoff && !global->authAbort )
	 {
		 auth_pae->state = apsm_Connecting;
		 //printk("apsm_Aborting: ");
		 authsm_connecting(global);
		 return True;
	 }

	 break;

    case apsm_Held:

	 if ( auth_pae->port_timers->quietWhile == 0 )
	 {
		 auth_pae->state = apsm_Connecting;
		 //printk("apsm_Held: ");
		 authsm_connecting(global);
		 return True;
	 }
	 break;

    case apsm_Force_Auth:

	 if (auth_pae->eapStart)
	 {
		 auth_pae->state = apsm_Force_Auth;
		 authsm_force_auth(global);
		 return True;
	 }
	 break;

    case apsm_Force_Unauth:

	 if (auth_pae->eapStart)
	 {
		 auth_pae->state = apsm_Force_Unauth;
		 authsm_force_unauth(global);
		 return True;
	 }
	 break;
  }	

  return False;
}

void authsm_initialize(Global_Params *global)
{
  global->currentId  = 0;
  global->Auth->portMode = pmt_Auto;
}


void authsm_disconnected(Global_Params *global)
{
  Auth_Pae *auth_pae = global->Auth;

  global->portStatus    = pst_Unauthorized;
  auth_pae->eapLogoff   = False;
  auth_pae->reAuthCount = 0;
  auth_txCannedFail(global);	
  INC(global->currentId);	
}

void authsm_connecting(Global_Params * global)
{
  Auth_Pae *auth_pae = global->Auth;

  auth_pae->eapStart = False;
  global->reAuthenticate = False; 

  auth_pae->port_timers->txWhen = auth_pae -> txPeriod;
  auth_pae->rxRespId = False; 
  auth_txReqId(global);
  INC(auth_pae->reAuthCount);
}

void authsm_authenticated(Global_Params * global)
{
  Auth_Pae * auth_pae = global->Auth;

  global->portStatus = pst_Authorized;
  auth_pae->reAuthCount = 0;
  INC(global->currentId);
}

void authsm_authenticating(Global_Params * global)
{
  global->authSuccess = False;
  global->authFail    = False;
  global->authTimeout = False;
  global->authStart   = True;
}

void authsm_aborting(Global_Params *global)
{
  global->authAbort = True;
  INC(global->currentId);
}

void authsm_force_unauth(Global_Params *global)
{
  if( global->Auth->portMode == pmt_Auto && 
      global->portStatus == pst_Authorized )
  {
    //Sending Radius Account Stop packet
    rad_send_acc_info(global,RADPKT_ACC_STATUS_STOP,RADPKT_ACC_TERM_ADMINRESET);
  }

  global->portStatus     = pst_Unauthorized;
  global->Auth->portMode = pmt_ForceUnauthorized;
  global->Auth->eapStart = False;
  auth_txCannedFail(global);
  INC(global->currentId);
}

void authsm_force_auth(Global_Params *global)
{
  if( global->Auth->portMode == pmt_Auto && 
      global->portStatus == pst_Authorized )
  {
    //Sending Radius account stop packet
    rad_send_acc_info(global,RADPKT_ACC_STATUS_STOP,RADPKT_ACC_TERM_ADMINRESET);
  }

  global->portStatus     = pst_Authorized;
  global->Auth->portMode = pmt_ForceAuthorized;
  global->Auth->eapStart = False;
  auth_txCannedSuccess(global);
  INC(global->currentId);
}

void authsm_held(Global_Params * global)
{
  Auth_Pae *auth_pae = global->Auth;

  global->portStatus = pst_Unauthorized;
  auth_pae->port_timers->quietWhile = auth_pae->quietPeriod;
  auth_pae->eapLogoff = False;
  INC(global->currentId);
}

//--------------------------------------------------
//  Send a success EAP packet to supplicant.
//--------------------------------------------------
int auth_txCannedSuccess(Global_Params *global)
{
	struct eapol_pkt eapol;
	Tbool   bSuccess = True;
	Tuint16 size=0;
	Tuint8  phy_port;

	eapol.protocol_version   = X_EAPOL_VER;
	eapol.packet_type        = X_EAPOL_EAPPKT;
	eapol.packet_body_length = htons(X_EAP_HDRLEN);

  eapol.code       = X_EAP_SUCCESS;
  eapol.identifier = global->currentId;
  eapol.length     = htons(X_EAP_HDRLEN);

  size = X_EAPOL_HDRLEN + X_EAP_HDRLEN;

  //printk("auth_txCannedSuccess -> ");
  bSuccess = eapol_tx_handler(global->nPortId,(Tbyte*)&eapol,size);

  phy_port =  cclmx_LId2PId (global->nPortId);
  //printk("auth_txCannedSuccess -> ");
  K_PortSetSecurity(phy_port, False);
  return 0;

}

//--------------------------------------------------
//  Send a request/identity packet
//--------------------------------------------------
void auth_txReqId(Global_Params *global)
{
  struct eapol_pkt eapol;
  Tbool   bSuccess = True;
  Tuint16 size = 0;
  Tint32 tv=net_random();

  eapol.protocol_version   = X_EAPOL_VER;
  eapol.packet_type        = X_EAPOL_EAPPKT;
  eapol.packet_body_length = htons(X_EAP_HDRLEN + X_EAP_RRLEN) ;


  // random generation identifier 
  if(tv < 0)
	  global->currentId = ((-tv) % 255);
  else
	  global->currentId = (tv % 255);


  eapol.code       = X_EAP_REQUEST;
  eapol.identifier = global->currentId;
  eapol.length     = htons(X_EAP_HDRLEN + X_EAP_RRLEN) ;
  eapol.type       = X_EAP_RRIDENTITY;

  size = X_EAPOL_HDRLEN + X_EAP_HDRLEN + X_EAP_RRLEN ;
  //printk("auth_txReqId -> ");
  bSuccess = eapol_tx_handler(global->nPortId,(Tbyte*)&eapol,size);

}	


//--------------------------------------------------
//  Send a fail EAP packet to supplicant.
//--------------------------------------------------
int auth_txCannedFail(Global_Params *global)
{
  struct eapol_pkt eapol;
  Tbool   bSuccess = True;
  Tuint16 size = 0;
  Tuint8  phy_port;

  eapol.protocol_version   = X_EAPOL_VER;
  eapol.packet_type        = X_EAPOL_EAPPKT;
  eapol.packet_body_length = htons(X_EAP_HDRLEN);

  eapol.code       = X_EAP_FAILURE;
  eapol.identifier = global->currentId;
  eapol.length     = htons(X_EAP_HDRLEN);

  size = X_EAPOL_HDRLEN + X_EAP_HDRLEN;
  //printk("auth_txCannedFail -> ");
  bSuccess = eapol_tx_handler(global->nPortId,(Tbyte*)&eapol,size);

  phy_port =  cclmx_LId2PId (global->nPortId);
  //printk("auth_txCannedFail -> ");
  K_PortSetSecurity(phy_port, True);
  return 0;

}

