/*
 * iptables module to match IP address ranges
 *
 * (C) 2003 Jozsef Kadlecsik <kadlec@blackhole.kfki.hu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/ip.h>
#include <linux/netfilter_ipv4/ip_tables.h>
#include <linux/netfilter_ipv4/ipt_iprange.h>

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jozsef Kadlecsik <kadlec@blackhole.kfki.hu>");
MODULE_DESCRIPTION("iptables arbitrary IP range match module");

#if 0
#define DEBUGP printk
#else
#define DEBUGP(format, args...)
#endif

static int
match(const struct sk_buff *skb,
      const struct net_device *in,
      const struct net_device *out,
      const struct xt_match *match,
      const void *matchinfo,
      int offset, unsigned int protoff, int *hotdrop)
{
	const struct ipt_iprange_info *info = matchinfo;
	const struct iphdr *iph = skb->nh.iph;

	if (info->flags & IPRANGE_SRC) {
		if (((ntohl(iph->saddr) < ntohl(info->src.min_ip))
			  || (ntohl(iph->saddr) > ntohl(info->src.max_ip)))
			 ^ !!(info->flags & IPRANGE_SRC_INV)) {
			DEBUGP("src IP %u.%u.%u.%u NOT in range %s"
			       "%u.%u.%u.%u-%u.%u.%u.%u\n",
				NIPQUAD(iph->saddr),
			        info->flags & IPRANGE_SRC_INV ? "(INV) " : "",
				NIPQUAD(info->src.min_ip),
				NIPQUAD(info->src.max_ip));
			return 0;
		}
	}
	if (info->flags & IPRANGE_DST) {
		if (((ntohl(iph->daddr) < ntohl(info->dst.min_ip))
			  || (ntohl(iph->daddr) > ntohl(info->dst.max_ip)))
			 ^ !!(info->flags & IPRANGE_DST_INV)) {
			DEBUGP("dst IP %u.%u.%u.%u NOT in range %s"
			       "%u.%u.%u.%u-%u.%u.%u.%u\n",
				NIPQUAD(iph->daddr),
			        info->flags & IPRANGE_DST_INV ? "(INV) " : "",
				NIPQUAD(info->dst.min_ip),
				NIPQUAD(info->dst.max_ip));
			return 0;
		}
	}
	return 1;
}

#ifdef RTL865X_IPTABLES_RULE_2_ACL
static int iprange_match2acl(const char *tablename,
			  const void *ip,
			  const struct xt_match *match,
			  void *matchinfo,
			  rtl865x_AclRule_t *rule,
			  unsigned int *invflags)
{

	const struct ipt_iprange_info *info = matchinfo;

	if(ip == NULL || matchinfo == NULL || rule == NULL)
		return 1;

	switch(rule->ruleType_)
	{
		case		RTL865X_ACL_TCP:
			rule->ruleType_ = RTL865X_ACL_TCP_IPRANGE;
			break;
		case		RTL865X_ACL_UDP:
			rule->ruleType_ = RTL865X_ACL_UDP_IPRANGE;
			break;
		case		RTL865X_ACL_IGMP:
			rule->ruleType_ = RTL865X_ACL_ICMP_IPRANGE;
			break;
		case		RTL865X_ACL_ICMP:
			rule->ruleType_ = RTL865X_ACL_IGMP_IPRANGE;
			break;
		case		RTL865X_ACL_SRCFILTER:
			rule->ruleType_ = RTL865X_ACL_SRCFILTER_IPRANGE;
			break;
		case		RTL865X_ACL_MAC:
		case		RTL865X_ACL_IP:
			rule->ruleType_ = RTL865X_ACL_IP_RANGE;
			break;
		default:
			return 1;
	}

	rule->srcIpAddrLB_ = rule->dstIpAddrLB_ = 0;
	rule->srcIpAddrUB_ = rule->dstIpAddrUB_ = 0xffffffff;
	if (info->flags & IPRANGE_SRC) {
		rule->srcIpAddrLB_ = ntohl(info->src.min_ip);
		rule->srcIpAddrUB_ = ntohl(info->src.max_ip);
	}
	if (info->flags & IPRANGE_DST) {
		rule->dstIpAddrLB_ = ntohl(info->dst.min_ip);
		rule->dstIpAddrUB_ = ntohl(info->dst.max_ip);
	}
	
	return 0;
}
#endif


static struct ipt_match iprange_match = {
	.name		= "iprange",
	.match		= match,
	.matchsize	= sizeof(struct ipt_iprange_info),
	.destroy	= NULL,
	.me		= THIS_MODULE, 
#ifdef RTL865X_IPTABLES_RULE_2_ACL
	.match2acl	= iprange_match2acl,
#endif

};

static int __init ipt_iprange_init(void)
{
	return ipt_register_match(&iprange_match);
}

static void __exit ipt_iprange_fini(void)
{
	ipt_unregister_match(&iprange_match);
}

module_init(ipt_iprange_init);
module_exit(ipt_iprange_fini);
