/* Masquerade.  Simple mapping which alters range to a local IP address
   (depending on route). */

/* (C) 1999-2001 Paul `Rusty' Russell
 * (C) 2002-2004 Netfilter Core Team <coreteam@netfilter.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/types.h>
#include <linux/inetdevice.h>
#include <linux/ip.h>
#include <linux/timer.h>
#include <linux/module.h>
#include <linux/netfilter.h>
#include <net/protocol.h>
#include <net/ip.h>
#include <net/checksum.h>
#include <net/route.h>
#include <linux/netfilter_ipv4.h>
#include <linux/netfilter_ipv4/ip_nat_rule.h>
#include <linux/netfilter_ipv4/ip_tables.h>

#ifdef CONFIG_RTL865X_HARDWARE_NAT 
/*2007-12-19*/
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
#include "../../../drivers/net/re865x/common/rtl_types.h"
#include "../../../drivers/net/re865x/l3Driver/rtl865x_ip.h"
#include "../../../drivers/net/re865x/l4Driver/rtl865x_nat.h"
#else
#include "../../../drivers/net/re865x/common/rtl865x_common.h"
#include "../../../drivers/net/re865x/l3Driver/rtl865x_layer3.h"
#include "../../../drivers/net/re865x/l4Driver/rtl865x_layer4.h"
#endif

extern char masq_if[];
extern unsigned int hw_napt_ip;
#endif
#ifdef	CONFIG_IP_NF_LOOSEUDP
#include <linux/udp.h>
#endif

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Netfilter Core Team <coreteam@netfilter.org>");
MODULE_DESCRIPTION("iptables MASQUERADE target module");

#if 0
#define DEBUGP printk
#else
#define DEBUGP(format, args...)
#endif

/* Lock protects masq region inside conntrack */
static DEFINE_RWLOCK(masq_lock);

/* FIXME: Multiple targets. --RR */
static int
masquerade_check(const char *tablename,
		 const void *e,
		 const struct xt_target *target,
		 void *targinfo,
		 unsigned int hook_mask)
{
	const struct ip_nat_multi_range_compat *mr = targinfo;

	if (mr->range[0].flags & IP_NAT_RANGE_MAP_IPS) {
		DEBUGP("masquerade_check: bad MAP_IPS.\n");
		return 0;
	}
	if (mr->rangesize != 1) {
		DEBUGP("masquerade_check: bad rangesize %u.\n", mr->rangesize);
		return 0;
	}
	return 1;
}

static unsigned int
masquerade_target(struct sk_buff **pskb,
		  const struct net_device *in,
		  const struct net_device *out,
		  unsigned int hooknum,
		  const struct xt_target *target,
		  const void *targinfo)
{
	struct ip_conntrack *ct;
	enum ip_conntrack_info ctinfo;
	const struct ip_nat_multi_range_compat *mr;
	struct ip_nat_range newrange;
	struct rtable *rt;
	__be32 newsrc;
#ifdef CONFIG_IP_NF_LOOSEUDP
	int ret;
	/* Imported from net/ipv4/sysctl_net_ipv4.c for looseUDP support */
	extern int sysctl_ip_masq_udp_dloose;
	struct sk_buff *skb=*pskb;
	struct iphdr *iph = NULL;
	struct udphdr *udph = NULL;
	unsigned char *check_teredo;
	unsigned char key_teredo[10];
#endif

	IP_NF_ASSERT(hooknum == NF_IP_POST_ROUTING);

	ct = ip_conntrack_get(*pskb, &ctinfo);
	IP_NF_ASSERT(ct && (ctinfo == IP_CT_NEW || ctinfo == IP_CT_RELATED
	                    || ctinfo == IP_CT_RELATED + IP_CT_IS_REPLY));

	/* Source address is 0.0.0.0 - locally generated packet that is
	 * probably not supposed to be masqueraded.
	 */
	if (ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple.src.ip == 0)
		return NF_ACCEPT;

	mr = targinfo;
	rt = (struct rtable *)(*pskb)->dst;
	newsrc = inet_select_addr(out, rt->rt_gateway, RT_SCOPE_UNIVERSE);
	if (!newsrc) {
		printk("MASQUERADE: %s ate my IP address\n", out->name);
		return NF_DROP;
	}

	write_lock_bh(&masq_lock);
	ct->nat.masq_index = out->ifindex;
	write_unlock_bh(&masq_lock);

	/* Transfer from original range. */
	newrange = ((struct ip_nat_range)
		{ mr->range[0].flags | IP_NAT_RANGE_MAP_IPS,
		  newsrc, newsrc,
		  mr->range[0].min, mr->range[0].max });

#ifdef	CONFIG_IP_NF_LOOSEUDP
	/* John Ho modified to support LooseUDP -- 03/19/2004 */
	ret = ip_nat_setup_info(ct, &newrange, hooknum);
	if( sysctl_ip_masq_udp_dloose ){
		struct ip_conntrack_tuple loose_reply;
		loose_reply = ct->tuplehash[IP_CT_DIR_REPLY].tuple;
		if( loose_reply.dst.protonum == IPPROTO_UDP ){
			/* We do not care about the reply's src pair */
				skb->h.raw = skb->nh.raw = skb->data;
				iph = skb->nh.iph;
				udph = (struct udphdr *)((void *)iph + iph->ihl * 4);
				if(3544==udph->dest)
				{
					check_teredo = (unsigned char *)(iph);
					memcpy(key_teredo, check_teredo+59, 6);
					key_teredo[6]='\0';
					/*printk("key_teredo=%s\n", key_teredo);*/
					if(strcmp(	key_teredo, "TEREDO")){
						loose_reply.src.ip = INADDR_BROADCAST;
						loose_reply.src.u.udp.port = 0xFFFF;
						ip_conntrack_alter_reply(ct, &loose_reply);
					}				
				}else{
					loose_reply.src.ip = INADDR_BROADCAST;
					loose_reply.src.u.udp.port = 0xFFFF;
				}
				ret = NF_ACCEPT;
		}
	}
	
	return ret;
#else
	/* Hand modified range to generic setup. */
	return ip_nat_setup_info(ct, &newrange, hooknum);
#endif
}

static inline int
device_cmp(struct ip_conntrack *i, void *ifindex)
{
	int ret;

	read_lock_bh(&masq_lock);
	ret = (i->nat.masq_index == (int)(long)ifindex);
	read_unlock_bh(&masq_lock);

	return ret;
}

static int masq_device_event(struct notifier_block *this,
			     unsigned long event,
			     void *ptr)
{
	struct net_device *dev = ptr;

	if (event == NETDEV_DOWN) {
		/* Device was downed.  Search entire table for
		   conntracks which were associated with that device,
		   and forget them. */
		IP_NF_ASSERT(dev->ifindex != 0);

		ip_ct_iterate_cleanup(device_cmp, (void *)(long)dev->ifindex);
	}

	return NOTIFY_DONE;
}

static int masq_inet_event(struct notifier_block *this,
			   unsigned long event,
			   void *ptr)
{
	struct net_device *dev = ((struct in_ifaddr *)ptr)->ifa_dev->dev;

	if (event == NETDEV_DOWN) {
		/* IP address was deleted.  Search entire table for
		   conntracks which were associated with that device,
		   and forget them. */
		IP_NF_ASSERT(dev->ifindex != 0);

		ip_ct_iterate_cleanup(device_cmp, (void *)(long)dev->ifindex);
	}

#ifdef CONFIG_RTL865X_HARDWARE_NAT  
/*2007-12-19*/
	#ifdef CONFIG_HARDWARE_NAT_DEBUG
	/*2007-12-19*/
	printk("%s:%d\n",__FUNCTION__,__LINE__);
	printk("ptr->ifa_dev->dev->name is %s\n",((struct in_ifaddr *)ptr)->ifa_dev->dev->name);
	printk("masq_if is %s\n",masq_if);
	#endif


	{
		struct in_ifaddr *ina = ptr;
		int rc;
		if (masq_if[0]!='\0' && !memcmp(masq_if, ina->ifa_dev->dev->name, IFNAMSIZ) ) {

			if (event == NETDEV_UP ) {
	#ifdef CONFIG_RTL865X_LAYERED_DRIVER
				rc = 0;
	#ifdef CONFIG_RTL865X_LAYERED_DRIVER_L3
				rc = rtl865x_addIp(0,(u32)(ina->ifa_local),IP_TYPE_NAPT);
	#endif
	#else
				rc = rtl865x_addNaptIp((u32)(ina->ifa_local));
	#endif
				if (rc == SUCCESS)
					hw_napt_ip = ina->ifa_local;
				#ifdef CONFIG_HARDWARE_NAT_DEBUG
				/*2007-12-19*/
				printk("masq_inet_event:(%s): %u.%u.%u.%u, errno=%d\n", "add_natip",
						NIPQUAD(ina->ifa_local), rc);
				#endif
			}
			if (event == NETDEV_DOWN) {
				
	#ifdef CONFIG_RTL865X_LAYERED_DRIVER
				rc = 0;
	#ifdef CONFIG_RTL865X_LAYERED_DRIVER_L3
				rc = rtl865x_delIp(ina->ifa_local);
	#endif
				if(rc==SUCCESS)
				{
	#ifdef CONFIG_RTL865X_LAYERED_DRIVER_L4
					rtl865x_nat_init();
	#endif
					//rtl865x_natInit();
					hw_napt_ip = 0;
				}
	#else
				rc = rtl865x_delNaptIp(ina->ifa_local);

				if(rc==RTL_SUCCESS)
				{
					rtl865x_natInit();					
					hw_napt_ip = 0;
				}
				
	#endif
				
				#ifdef CONFIG_HARDWARE_NAT_DEBUG
				/*2007-12-19*/
				printk("masq_inet_event:(%s): %u.%u.%u.%u, errno=%d\n", "del_natip", 
						NIPQUAD(ina->ifa_local), rc);
				#endif
			}
		}
	}
#endif

	return NOTIFY_DONE;
}

static struct notifier_block masq_dev_notifier = {
	.notifier_call	= masq_device_event,
};

static struct notifier_block masq_inet_notifier = {
	.notifier_call	= masq_inet_event,
};

static struct ipt_target masquerade = {
	.name		= "MASQUERADE",
	.target		= masquerade_target,
	.targetsize	= sizeof(struct ip_nat_multi_range_compat),
	.table		= "nat",
	.hooks		= 1 << NF_IP_POST_ROUTING,
	.checkentry	= masquerade_check,
	.me		= THIS_MODULE,
};

static int __init ipt_masquerade_init(void)
{
	int ret;

	ret = ipt_register_target(&masquerade);

	if (ret == 0) {
		/* Register for device down reports */
		register_netdevice_notifier(&masq_dev_notifier);
		/* Register IP address change reports */
		register_inetaddr_notifier(&masq_inet_notifier);
	}

	return ret;
}

static void __exit ipt_masquerade_fini(void)
{
	ipt_unregister_target(&masquerade);
	unregister_netdevice_notifier(&masq_dev_notifier);
	unregister_inetaddr_notifier(&masq_inet_notifier);	
}

module_init(ipt_masquerade_init);
module_exit(ipt_masquerade_fini);
