/* NAT for netfilter; shared with compatibility layer. */

/* (C) 1999-2001 Paul `Rusty' Russell
 * (C) 2002-2004 Netfilter Core Team <coreteam@netfilter.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/timer.h>
#include <linux/skbuff.h>
#include <linux/netfilter_ipv4.h>
#include <linux/vmalloc.h>
#include <net/checksum.h>
#include <net/icmp.h>
#include <net/ip.h>
#include <net/tcp.h>  /* For tcp_prot in getorigdst */
#include <linux/icmp.h>
#include <linux/udp.h>
#include <linux/jhash.h>

#include <linux/netfilter_ipv4/ip_conntrack.h>
#include <linux/netfilter_ipv4/ip_conntrack_core.h>
#include <linux/netfilter_ipv4/ip_conntrack_protocol.h>
#include <linux/netfilter_ipv4/ip_nat.h>
#include <linux/netfilter_ipv4/ip_nat_protocol.h>
#include <linux/netfilter_ipv4/ip_nat_core.h>
#include <linux/netfilter_ipv4/ip_nat_helper.h>
#include <linux/netfilter_ipv4/ip_conntrack_helper.h>

#ifdef CONFIG_RTL865X_LAYERED_DRIVER
#include <rtl_nic.h>
#include <l4Driver/rtl865x_nat.h>
#else
#include <common/rtl865x_common.h>
#ifdef CONFIG_RTL865X_HARDWARE_NAT 
/*2007-12-19*/
#include <l3Driver/rtl865x_layer3.h>
#include <l4Driver/rtl865x_layer4.h>
#endif
#endif /*CONFIG_RTL865X_LAYERED_DRIVER*/

/*
  * In order to support disable HW Qos support for 8196b, 
  * Always include rtl865x_outputQueue.h
#if defined(CONFIG_RTL865X_HW_QOS_SUPPORT)
*/
#include <l2Driver/rtl865x_outputQueue.h>
/*
#endif
*/

#ifdef CONFIG_RTL865X_HW_PPTPL2TP
#include <linux/netfilter_ipv4/ip_conntrack_pptp.h>
#include "../fastpath/fastpath_core.h"
#include <linux/ppp_channel.h>
#endif

#if defined(CONFIG_PROC_FS)
#include <linux/proc_fs.h>
#include <linux/ctype.h>
#include <linux/string.h>
#endif

#if defined(CONFIG_NET_QOS) 
__DRAM_GEN int gQosEnabled=0;
#endif

#ifndef __KERNEL__
#define __KERNEL__
#include <linux/inetdevice.h>
#endif

#if 0
#define DEBUGP printk
#else
#define DEBUGP(format, args...)
#endif

DEFINE_RWLOCK(ip_nat_lock);

/* Calculated at init based on memory size */
static unsigned int ip_nat_htable_size;

static struct list_head *bysource;

#define MAX_IP_NAT_PROTO 256
static struct ip_nat_protocol *ip_nat_protos[MAX_IP_NAT_PROTO];


#ifdef CONFIG_RTL865X_HARDWARE_NAT
__DRAM_GEN int gHwNatEnabled;

int rtl865x_handle_nat(struct ip_conntrack *ct, int act)
{
	struct ip_nat_info *info=&ct->nat.info;
	u_int32_t sip, dip, gip;
	u_int16_t sp, dp, gp, proto=0;
	u_int32_t timeval;
	int rc=0;

	if (gHwNatEnabled!=1)
		return -1;

	proto = (ct->tuplehash[0].tuple.dst.protonum==IPPROTO_TCP)? 1: 0;

	if (ct->status & IPS_SRC_NAT)
	{ /* outbound flow */
		sip	= ct->tuplehash[0].tuple.src.ip;
		dip 	= ct->tuplehash[0].tuple.dst.ip;
		gip 	= ct->tuplehash[1].tuple.dst.ip;
		sp  	= (proto)? ct->tuplehash[0].tuple.src.u.tcp.port: ct->tuplehash[0].tuple.src.u.udp.port;
		dp  	= (proto)? ct->tuplehash[0].tuple.dst.u.tcp.port: ct->tuplehash[0].tuple.dst.u.udp.port;
		gp  	= (proto)? ct->tuplehash[1].tuple.dst.u.tcp.port: ct->tuplehash[1].tuple.dst.u.udp.port;
	} 
	else if (ct->status & IPS_DST_NAT)
	{ /* inbound flow */
		sip	= ct->tuplehash[1].tuple.src.ip;
		dip 	= ct->tuplehash[1].tuple.dst.ip;
		gip 	= ct->tuplehash[0].tuple.dst.ip;
		sp  	= (proto)? ct->tuplehash[1].tuple.src.u.tcp.port: ct->tuplehash[1].tuple.src.u.udp.port;
		dp  	= (proto)? ct->tuplehash[1].tuple.dst.u.tcp.port: ct->tuplehash[1].tuple.dst.u.udp.port;
		gp  	= (proto)? ct->tuplehash[0].tuple.dst.u.tcp.port: ct->tuplehash[0].tuple.dst.u.udp.port;
	}
	else 
		return -1;

	/* do not add hardware NAPT table if protocol is UDP and source IP address is equal to gateway IP address */
	if ((act == 1) && (proto == 0) && (sip == gip))
		return -1;

	/* for TZO DDNS */
	if ((act == 1) && (proto == 1) && (dp == 21347)) {
		return -1;
	}

	if (act == 2) {
		/* query for idle */
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
		timeval = 0;
#ifdef CONFIG_RTL865X_LAYERED_DRIVER_L4
		timeval = rtl865x_naptSync(proto, sip, sp, gip, gp, dip, dp, 0);
#endif
#else
		timeval = rtl865x_naptSync(proto, sip, sp, gip, gp, dip, dp, 0);
#endif //end CONFIG_RTL865X_LAYERED_DRIVER
		if (timeval > 0)
			return 0;
		else
			return -1;
	}
	else if (act == 0) {
		/* delete */
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
		rc = 0;
#ifdef CONFIG_RTL865X_LAYERED_DRIVER_L4
		rc = rtl865x_delNaptConnection(proto, sip, sp, gip, gp, dip, dp);
#endif
#else
		rc = rtl865x_delNaptConnection(proto, sip, sp, gip, gp, dip, dp);
#endif // enf CONFIG_RTL865X_LAYERED_DRIVER
	}
	else {
		/* add */
#if defined(CONFIG_PROC_FS) && defined(CONFIG_NET_QOS)
		static int checkQosSetting(u_int16_t proto, u_int32_t sip, u_int32_t dip,u_int16_t sp, u_int16_t dp);
		if ((gQosEnabled == 0) 
			#if defined(CONFIG_RTL865X_HW_QOS_SUPPORT)
			|| 1
			#elif !defined(CONFIG_QOS_UI_BY_BANDWIDTH)
			|| checkQosSetting(proto, sip, dip, sp, dp)
			#endif
			)
		{
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
			rc = 0;
#ifdef CONFIG_RTL865X_LAYERED_DRIVER_L4
			rc =  rtl865x_addNaptConnection(proto, sip, sp, gip, gp, dip, dp, 0);
#endif
#else
			rc =  rtl865x_addNaptConnection(proto, sip, sp, gip, gp, dip, dp, 0);
#endif //endif CONFIG_RTL865X_LAYERED_DRIVER
		}
		else
		{
			act = 0;
		}
#else
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
			rc = 0;
#ifdef CONFIG_RTL865X_LAYERED_DRIVER_L4
			rc =  rtl865x_addNaptConnection(proto, sip, sp, gip, gp, dip, dp, 0);
#endif
#else
			rc =  rtl865x_addNaptConnection(proto, sip, sp, gip, gp, dip, dp, 0);
#endif //end CONFIG_RTL865X_LAYERED_DRIVER

#endif
       }
    
	if (!rc && act == 1) /* mark it as an asic entry */
		info->hw_acc = 1;
	if (!rc && act == 0) /* unmark it */
		info->hw_acc = 0;

	#ifdef CONFIG_HARDWARE_NAT_DEBUG
	/*2007-12-19*/
	DEBUGP("%s:%d:(%s): errno=%d\n %s (%u.%u.%u.%u:%u -> %u.%u.%u.%u:%u) g:(%u.%u.%u.%u:%u)\n",
			__FUNCTION__,__LINE__,((is_add)?"add_nat": "del_nat"), rc, ((proto)? "tcp": "udp"), 
			NIPQUAD(sip), sp, NIPQUAD(dip), dp, NIPQUAD(gip), gp);	
	#endif

	return 0;	
}


#endif

#ifdef CONFIG_RTL865X_HW_PPTPL2TP

//__DRAM_GEN struct pptp_acc_info pptpAccInfo = { 0 };
extern __DRAM_GEN struct pptp_acc_info pptpAccInfo;

int rtl865x_addPppInfoChannel(struct net_device *pppDev, struct ppp_channel *pppChan)
{
	if (!pppDev || !pppChan)
		return -1;

	pptpAccInfo.pppDev = pppDev;
	pptpAccInfo.pppChan = pppChan;
	return 0;
}

int rtl865x_delPppInfoChannel(struct net_device *pppDev, struct ppp_channel *pppChan)
{
	if (!pppDev || !pppChan)
		return -1;

	if (pptpAccInfo.pppDev == pppDev) {
		pptpAccInfo.pppDev = NULL;
		pptpAccInfo.pppChan = NULL;
	}
	return 0;
}

int rtl865x_filterPptp(struct sk_buff *skb)
{
	return 0;
}

__IRAM_EXTDEV int rtl865x_acceleratePptpToWanSync(struct sk_buff *skb)
{
	struct iphdr ip, *iph = (struct iphdr *)(skb->data + ETH_HLEN);
	struct pptp_gre_hdr gre, *greh = (struct pptp_gre_hdr *)((char *)iph + (iph->ihl << 2));

	if (gHwNatEnabled != 1)
		return -1;

#if 0
	if (!pptpAccInfo.valid || !pptpAccInfo.pppDev || !pptpAccInfo.wanDev) {
		return -1;
	}
#endif

	if (iph->protocol == IPPROTO_GRE && skb->len >= (ETH_HLEN + (iph->ihl << 2) + sizeof(struct pptp_gre_hdr) - 8)) {
		if ((greh->version & 7) == GRE_VERSION_PPTP &&
			ntohs(greh->protocol) == GRE_PROTOCOL_PPTP) {
			if (skb->dev == pptpAccInfo.wanDev &&
				iph->saddr == pptpAccInfo.ourIp && 
				iph->daddr == pptpAccInfo.peerIp && 
				greh->call_id == pptpAccInfo.peerCallID) {
				/* Matched */
				if (GRE_IS_S(greh->flags)) {
					/* Others go head !? */
					unsigned int seq = ntohl(greh->seq);
					if (time_after(seq, pptpAccInfo.tx_seqno)) {
						pptpAccInfo.tx_seqno = seq + 1;	
					}
					else {
	   					greh->seq = htonl(pptpAccInfo.tx_seqno++);	
	   				}
				}			
				if (GRE_IS_A(greh->version)) {
					/* Others go head !? */
					unsigned int ack = ntohl(greh->ack);
					if (time_after(ack, pptpAccInfo.rx_seqno)) {
						pptpAccInfo.rx_seqno = ack;	
					}
					else {
	   					greh->ack = htonl(pptpAccInfo.rx_seqno);	
	   				}					
				}
			}
		}
	}			

	return 0;
}

__IRAM_EXTDEV static inline int rtl865x_acceleratePptpToWAN(struct sk_buff *skb)
{
	int	headroom;
	struct ethhdr *eth;
	struct iphdr ip, *iph;
	struct pptp_gre_hdr	gre, *greh;	
	unsigned char tos = skb->nh.iph->tos;

	headroom = sizeof(struct ethhdr) + sizeof(struct iphdr) + sizeof(struct pptp_gre_hdr);
	if (skb_headroom(skb) < headroom || skb_cloned(skb) || skb_shared(skb)) {	
		struct sk_buff *new_skb = skb_realloc_headroom(skb, headroom);				
		if (!new_skb) {
			return -1;
		}									
		dev_kfree_skb(skb);
		skb = new_skb;
	}			

	eth = (struct ethhdr *)skb_push(skb, headroom);
	memcpy(eth->h_dest, pptpAccInfo.peerMac, ETH_ALEN);
	memcpy(eth->h_source, pptpAccInfo.ourMac, ETH_ALEN);
	eth->h_proto = htons(0x0800);

	iph = (struct iphdr *)((char *)eth + sizeof(struct ethhdr));
	if (((unsigned long)iph) & 0x03) {
		ip.version  = 4;
		ip.ihl      = sizeof(struct iphdr) >> 2;
		ip.frag_off = 0;			
		ip.protocol = IPPROTO_GRE;
		ip.tos      = tos;
		ip.daddr    = pptpAccInfo.peerIp;
		ip.saddr    = pptpAccInfo.ourIp;
		ip.ttl      = IPDEFTTL;   
		ip.tot_len  = htons(skb->len - sizeof(struct ethhdr));		
		ip.id       = htons(++pptpAccInfo.tx_ipID);
	   	ip.check    = 0;
	   	/*ip.check    = ip_fast_csum((unsigned char *)&ip, ip.ihl);*/
		memcpy(iph, &ip, sizeof(struct iphdr));
	}
	else {
		iph->version  = 4;
		iph->ihl      = sizeof(struct iphdr) >> 2;
		iph->frag_off = 0;			
		iph->protocol = IPPROTO_GRE;
		iph->tos      = tos;
		iph->daddr    = pptpAccInfo.peerIp;
		iph->saddr    = pptpAccInfo.ourIp;
		iph->ttl      = IPDEFTTL;   
		iph->tot_len  = htons(skb->len - sizeof(struct ethhdr));		
		iph->id       = htons(++pptpAccInfo.tx_ipID);
	   	iph->check    = 0;
	   	/*iph->check    = ip_fast_csum((unsigned char *)iph, iph->ihl);*/
	}		
	/*skb->ip_summed = CHECKSUM_NONE;*/
	/*skb->ip_summed = CHECKSUM_PARTIAL;*/

	greh = (struct pptp_gre_hdr *)((char *)iph + sizeof(struct iphdr));
	if (((unsigned long)greh) & 0x03) {
	   	gre.flags       = GRE_FLAG_K | GRE_FLAG_S;
	   	gre.version     = GRE_VERSION_PPTP | GRE_FLAG_A;
	   	gre.protocol    = htons(GRE_PROTOCOL_PPTP);
	   	gre.payload_len = htons(skb->len - headroom);    	
	   	gre.call_id     = pptpAccInfo.peerCallID;			
	   	gre.seq         = htonl(pptpAccInfo.tx_seqno++);		
	   	gre.ack         = htonl(pptpAccInfo.rx_seqno);
		memcpy(greh, &gre, sizeof(struct pptp_gre_hdr));
	}
	else {
	   	greh->flags       = GRE_FLAG_K | GRE_FLAG_S;
	   	greh->version     = GRE_VERSION_PPTP | GRE_FLAG_A;
	   	greh->protocol    = htons(GRE_PROTOCOL_PPTP);
	   	greh->payload_len = htons(skb->len - headroom);    	
	   	greh->call_id     = pptpAccInfo.peerCallID;			
	   	greh->seq         = htonl(pptpAccInfo.tx_seqno++);		
	   	greh->ack         = htonl(pptpAccInfo.rx_seqno);
	}		

   	skb->dev = pptpAccInfo.wanDev;
	/* dev_queue_xmit(skb);*/

	{
		rtl865x_acceleratePptpToWanSync(skb);
	}

	re865x_pptp_xmit(skb, skb->dev, FALSE);

	return 0;
}

__IRAM_EXTDEV int rtl865x_acceleratePptpToLAN(struct sk_buff *skb)
{
	struct iphdr ip, *iph = skb->nh.iph;
	struct pptp_gre_hdr gre, *greh = (struct pptp_gre_hdr *)(skb->nh.raw + (iph->ihl << 2));
	int offset;
	unsigned char *p;

#if 0
	if (!pptpAccInfo.valid || !pptpAccInfo.pppDev || !pptpAccInfo.wanDev) {
		return NET_RX_SUCCESS;
	}
#endif

	/* PPTP GRE */
	if ((greh->version & 7) == GRE_VERSION_PPTP &&
		ntohs(greh->protocol) == GRE_PROTOCOL_PPTP) {
		/* Unalignment !? */
		if (((unsigned long)iph) & 0x03) {
			memcpy(&ip, iph, sizeof(struct iphdr));
			iph = &ip;
		}
		if (((unsigned long)greh) & 0x03) {
			memcpy(&gre, greh, sizeof(struct pptp_gre_hdr));
			greh = &gre;
		}

#if 0
		DEBUGP("com1: dev[%s], dip[0x%x], sip[0x%x], call_id[%d].\n", skb->dev->name, 
			iph->daddr, iph->saddr, greh->call_id);
		DEBUGP("com2: dev[%s], dip[0x%x], sip[0x%x], call_id[%d][%d].\n", pptpAccInfo.wanDev->name, 
			pptpAccInfo.ourIp, pptpAccInfo.peerIp, pptpAccInfo.ourCallID,
			pptpAccInfo.peerCallID);
#endif
		if (skb->dev == pptpAccInfo.wanDev &&
			iph->daddr == pptpAccInfo.ourIp && 
			iph->saddr == pptpAccInfo.peerIp && 
			greh->call_id == pptpAccInfo.ourCallID) {
			/* Matched */
			if (greh->payload_len == 0)
			{
				return NET_RX_SUCCESS;
			}

			offset = (iph->ihl << 2) + sizeof(struct pptp_gre_hdr) - 8;				
			if (GRE_IS_S(greh->flags)) {	
				pptpAccInfo.rx_seqno = ntohl(greh->seq);						
				offset += sizeof(greh->seq);
			}	
			if (GRE_IS_A(greh->version)) {
				offset += sizeof(greh->ack);
			}
			
			/* strip address/control field if present */
			p = skb->nh.raw + offset;
			if (p[0] == 0xff /* PPP_ALLSTATIONS */ && p[1] == 0x03 /* PPP_UI */) {			
				/* chop off address/control */
				if (greh->payload_len < 3)
				{
					return NET_RX_SUCCESS;
				}
				offset += 2;
				p += 2;
			}
			/* decompress protocol field if compressed */
			if (p[0] & 1) {
				/* protocol is compressed */
				offset -= 1;
				skb_pull(skb, offset)[0] = 0;
			} 
			else {
				if (greh->payload_len < 2)
				{
					return NET_RX_SUCCESS;
				}
				skb_pull(skb, offset);
			}

//			skb->cb[0] = 'P'; skb->cb[1] = 'P'; skb->cb[2] = 'P';
			ppp_input(pptpAccInfo.pppChan, skb);
			memset(skb->cb, '\x0', 3);
			return NET_RX_DROP;
		}
	}

	memset(skb->cb, '\x0', 3);
	return NET_RX_SUCCESS;
}

#if 1
__IRAM_EXTDEV int rtl865x_pppAcceleratePptpL2tpToWAN(struct sk_buff *skb, struct net_device *pppDev)
{
	if (gHwNatEnabled != 1)
		return -1;

#if 0
	if (!pptpAccInfo.valid || !pptpAccInfo.pppDev || !pptpAccInfo.wanDev || pptpAccInfo.pppDev != pppDev) {
		return -1;
	}
#endif
	return (rtl865x_acceleratePptpToWAN(skb));
}
#endif

__IRAM_EXTDEV int rtl865x_pppAcceleratePptpL2tpToLAN(struct sk_buff *skb, struct net_device *pppDev)
{
	struct ethhdr *eth;

	if (gHwNatEnabled != 1)
		return -1;

#if 0
	if ( (gHwNatEnabled != 1) || (!pptpAccInfo.valid || !pptpAccInfo.pppDev || !pptpAccInfo.wanDev || pptpAccInfo.pppDev != skb->dev))
	{
		return -1;
	}
#endif

	{
		struct iphdr *iph;
		struct tcphdr *tcphupuh;  //just keep one , don't care tcp or udp //

		iph = (struct iphdr*)(skb->data);
		tcphupuh = (struct tcphdr*)((__u32 *)iph + iph->ihl);

#if 0
		if ((iph->protocol==IPPROTO_TCP)&&(tcphupuh->fin||tcphupuh->rst||tcphupuh->syn))
			DEBUGP("fin %d rst %d syn %d.\n", tcphupuh->fin, tcphupuh->rst, tcphupuh->syn);
#endif
		if (!((iph->protocol==IPPROTO_UDP) || 
			((iph->protocol==IPPROTO_TCP)&&!(tcphupuh->fin||tcphupuh->rst||tcphupuh->syn))))
		{
#if 0
			DEBUGP("iph 0x%p: %u.%u.%u.%u==>%u.%u.%u.%u [0x%x]  %c.%c.%c\n", 
				iph, NIPQUAD(iph->saddr), NIPQUAD(iph->daddr), iph->protocol, 
				skb->cb[0], skb->cb[1], skb->cb[2]);
#endif
//			skb->cb[0] = skb->cb[1] = skb->cb[2] = 0;
			memset(skb->cb, '\x0', 3);
			netif_rx(skb);
			pppDev->last_rx = jiffies;
			return SUCCESS;
		}
	}
	
	if (skb_headroom(skb) < sizeof(struct ethhdr) || skb_cloned(skb) || skb_shared(skb)) {
		/*DEBUGP("header room too small: %d:%d.\n", skb_headroom(skb), sizeof(struct ethhdr));*/
		struct sk_buff *new_skb = skb_realloc_headroom(skb, sizeof(struct ethhdr));				
		if (!new_skb) {				
			return -1;
		}
		dev_kfree_skb(skb);
		skb = new_skb;
	}
		
	skb_push(skb, sizeof(struct ethhdr));
	eth = (struct ethhdr *)skb->data;
	memcpy(eth->h_dest, pptpAccInfo.ourMac, ETH_ALEN);
	memcpy(eth->h_source, pptpAccInfo.peerMac, ETH_ALEN);
	eth->h_proto = htons(0x0800);
	re865x_pptp_xmit(skb, skb->dev, TRUE);
	return 0;
}

__IRAM_FWD int rtl865x_acceleratePptpL2tp(struct sk_buff **pskb) 
{
	struct sk_buff *skb = *pskb;
	struct iphdr *iph;
	
	if (gHwNatEnabled != 1)
		return NET_RX_SUCCESS;
		
	if (ntohs(((struct ethhdr*)skb->mac.raw)->h_proto) != 0x0800) {
		return NET_RX_SUCCESS;
	}

	skb->h.raw = skb->nh.raw = skb->data;
#if 0	
	if (pptpAccInfo.pppDev!=NULL)
	{
		/* pptp wan type */
		if (!memcmp(skb->dev->name, "eth0", 4))
		{
			/* from lan -> wan */
			pptpAccInfo.pppDev->hard_start_xmit(skb, pptpAccInfo.pppDev);
			return NET_RX_DROP;
		}
		else if(!memcmp(skb->dev->name, "eth1", 4))
		{
			/* from wan -> lan */
			iph = skb->nh.iph;
			if (iph->protocol == IPPROTO_GRE && skb->len >= ((iph->ihl << 2) + sizeof(struct pptp_gre_hdr))) {
				return (rtl865x_acceleratePptpToLAN(skb));
			}
		}
	}

#else
//	if (skb->cb[0] == 'P' && skb->cb[1] == 'P' && skb->cb[2] == 'P')
	if (!memcmp(skb->cb, "PPP", 3))
	{
//		if (pptpAccInfo.pppDev)
		{
			pptpAccInfo.pppDev->hard_start_xmit(skb, pptpAccInfo.pppDev);
			memset(skb->cb, '\x0', 3);
			return NET_RX_DROP;
		}
	}
//	else if (skb->cb[0] == 'N' && skb->cb[1] == 'A' && skb->cb[2] == 'T')
	else if (!memcmp(skb->cb, "NNN", 3))
	{
		iph = skb->nh.iph;
		if (iph->protocol == IPPROTO_GRE && skb->len >= ((iph->ihl << 2) + sizeof(struct pptp_gre_hdr))) {
			return (rtl865x_acceleratePptpToLAN(skb));
		}
	}
#endif	
	return NET_RX_SUCCESS;	
}	
#endif

static inline struct ip_nat_protocol *
__ip_nat_proto_find(u_int8_t protonum)
{
	return ip_nat_protos[protonum];
}

struct ip_nat_protocol *
ip_nat_proto_find_get(u_int8_t protonum)
{
	struct ip_nat_protocol *p;

	/* we need to disable preemption to make sure 'p' doesn't get
	 * removed until we've grabbed the reference */
	preempt_disable();
	p = __ip_nat_proto_find(protonum);
	if (!try_module_get(p->me))
		p = &ip_nat_unknown_protocol;
	preempt_enable();

	return p;
}
EXPORT_SYMBOL_GPL(ip_nat_proto_find_get);

void
ip_nat_proto_put(struct ip_nat_protocol *p)
{
	module_put(p->me);
}
EXPORT_SYMBOL_GPL(ip_nat_proto_put);

/* We keep an extra hash for each conntrack, for fast searching. */
static inline unsigned int
hash_by_src(const struct ip_conntrack_tuple *tuple)
{
	/* Original src, to ensure we map it consistently if poss. */
	return jhash_3words((__force u32)tuple->src.ip, tuple->src.u.all,
			    tuple->dst.protonum, 0) % ip_nat_htable_size;
}

/* Noone using conntrack by the time this called. */
static void ip_nat_cleanup_conntrack(struct ip_conntrack *conn)
{
	if (!(conn->status & IPS_NAT_DONE_MASK))
		return;

	write_lock_bh(&ip_nat_lock);
	list_del(&conn->nat.info.bysource);
	write_unlock_bh(&ip_nat_lock);
}

/* Is this tuple already taken? (not by us) */
int
ip_nat_used_tuple(const struct ip_conntrack_tuple *tuple,
		  const struct ip_conntrack *ignored_conntrack)
{
	/* Conntrack tracking doesn't keep track of outgoing tuples; only
	   incoming ones.  NAT means they don't have a fixed mapping,
	   so we invert the tuple and look for the incoming reply.

	   We could keep a separate hash if this proves too slow. */
	struct ip_conntrack_tuple reply;

	invert_tuplepr(&reply, tuple);
	return ip_conntrack_tuple_taken(&reply, ignored_conntrack);
}
EXPORT_SYMBOL(ip_nat_used_tuple);

/* If we source map this tuple so reply looks like reply_tuple, will
 * that meet the constraints of range. */
static int
in_range(const struct ip_conntrack_tuple *tuple,
	 const struct ip_nat_range *range)
{
	struct ip_nat_protocol *proto = 
				__ip_nat_proto_find(tuple->dst.protonum);

	/* If we are supposed to map IPs, then we must be in the
	   range specified, otherwise let this drag us onto a new src IP. */
	if (range->flags & IP_NAT_RANGE_MAP_IPS) {
		if (ntohl(tuple->src.ip) < ntohl(range->min_ip)
		    || ntohl(tuple->src.ip) > ntohl(range->max_ip))
			return 0;
	}

	if (!(range->flags & IP_NAT_RANGE_PROTO_SPECIFIED)
	    || proto->in_range(tuple, IP_NAT_MANIP_SRC,
			       &range->min, &range->max))
		return 1;

	return 0;
}

static inline int
same_src(const struct ip_conntrack *ct,
	 const struct ip_conntrack_tuple *tuple)
{
	return (ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple.dst.protonum
		== tuple->dst.protonum
		&& ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple.src.ip
		== tuple->src.ip
		&& ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple.src.u.all
		== tuple->src.u.all);
}

/* Only called for SRC manip */
static int
find_appropriate_src(const struct ip_conntrack_tuple *tuple,
		     struct ip_conntrack_tuple *result,
		     const struct ip_nat_range *range)
{
	unsigned int h = hash_by_src(tuple);
	struct ip_conntrack *ct;

	read_lock_bh(&ip_nat_lock);
	list_for_each_entry(ct, &bysource[h], nat.info.bysource) {
		if (same_src(ct, tuple)) {
			/* Copy source part from reply tuple. */
			invert_tuplepr(result,
				       &ct->tuplehash[IP_CT_DIR_REPLY].tuple);
			result->dst = tuple->dst;

			if (in_range(result, range)) {
				read_unlock_bh(&ip_nat_lock);
				return 1;
			}
		}
	}
	read_unlock_bh(&ip_nat_lock);
	return 0;
}

/* For [FUTURE] fragmentation handling, we want the least-used
   src-ip/dst-ip/proto triple.  Fairness doesn't come into it.  Thus
   if the range specifies 1.2.3.4 ports 10000-10005 and 1.2.3.5 ports
   1-65535, we don't do pro-rata allocation based on ports; we choose
   the ip with the lowest src-ip/dst-ip/proto usage.
*/
static void
find_best_ips_proto(struct ip_conntrack_tuple *tuple,
		    const struct ip_nat_range *range,
		    const struct ip_conntrack *conntrack,
		    enum ip_nat_manip_type maniptype)
{
	__be32 *var_ipp;
	/* Host order */
	u_int32_t minip, maxip, j;

	/* No IP mapping?  Do nothing. */
	if (!(range->flags & IP_NAT_RANGE_MAP_IPS))
		return;

	if (maniptype == IP_NAT_MANIP_SRC)
		var_ipp = &tuple->src.ip;
	else
		var_ipp = &tuple->dst.ip;

	/* Fast path: only one choice. */
	if (range->min_ip == range->max_ip) {
		*var_ipp = range->min_ip;
		return;
	}

	/* Hashing source and destination IPs gives a fairly even
	 * spread in practice (if there are a small number of IPs
	 * involved, there usually aren't that many connections
	 * anyway).  The consistency means that servers see the same
	 * client coming from the same IP (some Internet Banking sites
	 * like this), even across reboots. */
	minip = ntohl(range->min_ip);
	maxip = ntohl(range->max_ip);
	j = jhash_2words((__force u32)tuple->src.ip, (__force u32)tuple->dst.ip, 0);
	*var_ipp = htonl(minip + j % (maxip - minip + 1));
}

/* Manipulate the tuple into the range given.  For NF_IP_POST_ROUTING,
 * we change the source to map into the range.  For NF_IP_PRE_ROUTING
 * and NF_IP_LOCAL_OUT, we change the destination to map into the
 * range.  It might not be possible to get a unique tuple, but we try.
 * At worst (or if we race), we will end up with a final duplicate in
 * __ip_conntrack_confirm and drop the packet. */
static void
get_unique_tuple(struct ip_conntrack_tuple *tuple,
		 const struct ip_conntrack_tuple *orig_tuple,
		 const struct ip_nat_range *range,
		 struct ip_conntrack *conntrack,
		 enum ip_nat_manip_type maniptype)
{
	struct ip_nat_protocol *proto;

	/* 1) If this srcip/proto/src-proto-part is currently mapped,
	   and that same mapping gives a unique tuple within the given
	   range, use that.

	   This is only required for source (ie. NAT/masq) mappings.
	   So far, we don't do local source mappings, so multiple
	   manips not an issue.  */
	if (maniptype == IP_NAT_MANIP_SRC) {
		if (find_appropriate_src(orig_tuple, tuple, range)) {
			DEBUGP("get_unique_tuple: Found current src map\n");
			if (!ip_nat_used_tuple(tuple, conntrack))
				return;
		}
	}

	/* 2) Select the least-used IP/proto combination in the given
	   range. */
	*tuple = *orig_tuple;
	find_best_ips_proto(tuple, range, conntrack, maniptype);

	/* 3) The per-protocol part of the manip is made to map into
	   the range to make a unique tuple. */

	proto = ip_nat_proto_find_get(orig_tuple->dst.protonum);

	/* Only bother mapping if it's not already in range and unique */
	if ((!(range->flags & IP_NAT_RANGE_PROTO_SPECIFIED)
	     || proto->in_range(tuple, maniptype, &range->min, &range->max))
	    && !ip_nat_used_tuple(tuple, conntrack)) {
		ip_nat_proto_put(proto);
		return;
	}

	/* Last change: get protocol to try to obtain unique tuple. */
	proto->unique_tuple(tuple, range, maniptype, conntrack);

	ip_nat_proto_put(proto);
}

unsigned int
ip_nat_setup_info(struct ip_conntrack *conntrack,
		  const struct ip_nat_range *range,
		  unsigned int hooknum)
{
	struct ip_conntrack_tuple curr_tuple, new_tuple;
	struct ip_nat_info *info = &conntrack->nat.info;
	int have_to_hash = !(conntrack->status & IPS_NAT_DONE_MASK);
	enum ip_nat_manip_type maniptype = HOOK2MANIP(hooknum);

	IP_NF_ASSERT(hooknum == NF_IP_PRE_ROUTING
		     || hooknum == NF_IP_POST_ROUTING
		     || hooknum == NF_IP_LOCAL_IN
		     || hooknum == NF_IP_LOCAL_OUT);
	BUG_ON(ip_nat_initialized(conntrack, maniptype));

	/* What we've got will look like inverse of reply. Normally
	   this is what is in the conntrack, except for prior
	   manipulations (future optimization: if num_manips == 0,
	   orig_tp =
	   conntrack->tuplehash[IP_CT_DIR_ORIGINAL].tuple) */
	invert_tuplepr(&curr_tuple,
		       &conntrack->tuplehash[IP_CT_DIR_REPLY].tuple);

	get_unique_tuple(&new_tuple, &curr_tuple, range, conntrack, maniptype);

	if (!ip_ct_tuple_equal(&new_tuple, &curr_tuple)) {
		struct ip_conntrack_tuple reply;

		/* Alter conntrack table so will recognize replies. */
		invert_tuplepr(&reply, &new_tuple);
		ip_conntrack_alter_reply(conntrack, &reply);

		/* Non-atomic: we own this at the moment. */
		if (maniptype == IP_NAT_MANIP_SRC)
			conntrack->status |= IPS_SRC_NAT;
		else
			conntrack->status |= IPS_DST_NAT;
	}

	/* Place in source hash if this is the first time. */
	if (have_to_hash) {
		unsigned int srchash
			= hash_by_src(&conntrack->tuplehash[IP_CT_DIR_ORIGINAL]
				      .tuple);
		write_lock_bh(&ip_nat_lock);
		list_add(&info->bysource, &bysource[srchash]);
		write_unlock_bh(&ip_nat_lock);
	}

	/* It's done. */
	if (maniptype == IP_NAT_MANIP_DST)
		set_bit(IPS_DST_NAT_DONE_BIT, &conntrack->status);
	else
		set_bit(IPS_SRC_NAT_DONE_BIT, &conntrack->status);

	return NF_ACCEPT;
}
EXPORT_SYMBOL(ip_nat_setup_info);

/* Returns true if succeeded. */
static int
manip_pkt(u_int16_t proto,
	  struct sk_buff **pskb,
	  unsigned int iphdroff,
	  const struct ip_conntrack_tuple *target,
	  enum ip_nat_manip_type maniptype)
{
	struct iphdr *iph;
	struct ip_nat_protocol *p;

	if (!skb_make_writable(pskb, iphdroff + sizeof(*iph)))
		return 0;

	iph = (void *)(*pskb)->data + iphdroff;

	/* Manipulate protcol part. */
	p = ip_nat_proto_find_get(proto);
	if (!p->manip_pkt(pskb, iphdroff, target, maniptype)) {
		ip_nat_proto_put(p);
		return 0;
	}
	ip_nat_proto_put(p);

	iph = (void *)(*pskb)->data + iphdroff;

	if (maniptype == IP_NAT_MANIP_SRC) {
		iph->check = nf_csum_update(~iph->saddr, target->src.ip,
					    iph->check);
		iph->saddr = target->src.ip;
	} else {
		iph->check = nf_csum_update(~iph->daddr, target->dst.ip,
					    iph->check);
		iph->daddr = target->dst.ip;
	}
	return 1;
}

/* Do packet manipulations according to ip_nat_setup_info. */
unsigned int ip_nat_packet(struct ip_conntrack *ct,
			   enum ip_conntrack_info ctinfo,
			   unsigned int hooknum,
			   struct sk_buff **pskb)
{
	enum ip_conntrack_dir dir = CTINFO2DIR(ctinfo);
	unsigned long statusbit;
	enum ip_nat_manip_type mtype = HOOK2MANIP(hooknum);

#ifdef CONFIG_RTL865X_HARDWARE_NAT  
/*2007-12-19*/
	u_int16_t proto=(*pskb)->nh.iph->protocol;
	int is_tcp =( proto == IPPROTO_TCP);
#endif	

	if (mtype == IP_NAT_MANIP_SRC)
		statusbit = IPS_SRC_NAT;
	else
		statusbit = IPS_DST_NAT;

	/* Invert if this is reply dir. */
	if (dir == IP_CT_DIR_REPLY)
		statusbit ^= IPS_NAT_MASK;

	/* Non-atomic: these bits don't change. */
	if (ct->status & statusbit) {
		struct ip_conntrack_tuple target;

		/* We are aiming to look like inverse of other direction. */
		invert_tuplepr(&target, &ct->tuplehash[!dir].tuple);

		if (!manip_pkt(target.dst.protonum, pskb, 0, &target, mtype))
			return NF_DROP;
	}
	
#ifdef CONFIG_RTL865X_HARDWARE_NAT  
/*2007-12-19*/
	if (((ct->status&IPS_DST_NAT_DONE_BIT) && (ct->status&IPS_SRC_NAT_DONE_BIT) )&&
		((proto==IPPROTO_TCP) || (proto==IPPROTO_UDP)) &&
		(!is_tcp||ct->proto.tcp.state==TCP_CONNTRACK_ESTABLISHED) &&(!(ct->helper)))
		rtl865x_handle_nat(ct, 1);
#endif

	return NF_ACCEPT;
}
EXPORT_SYMBOL_GPL(ip_nat_packet);

/* Dir is direction ICMP is coming from (opposite to packet it contains) */
int ip_nat_icmp_reply_translation(struct ip_conntrack *ct,
				  enum ip_conntrack_info ctinfo,
				  unsigned int hooknum,
				  struct sk_buff **pskb)
{
	struct {
		struct icmphdr icmp;
		struct iphdr ip;
	} *inside;
	struct ip_conntrack_tuple inner, target;
	int hdrlen = (*pskb)->nh.iph->ihl * 4;
	enum ip_conntrack_dir dir = CTINFO2DIR(ctinfo);
	unsigned long statusbit;
	enum ip_nat_manip_type manip = HOOK2MANIP(hooknum);

	if (!skb_make_writable(pskb, hdrlen + sizeof(*inside)))
		return 0;

	inside = (void *)(*pskb)->data + (*pskb)->nh.iph->ihl*4;

	/* We're actually going to mangle it beyond trivial checksum
	   adjustment, so make sure the current checksum is correct. */
	if (nf_ip_checksum(*pskb, hooknum, hdrlen, 0))
		return 0;

	/* Must be RELATED */
	IP_NF_ASSERT((*pskb)->nfctinfo == IP_CT_RELATED ||
		     (*pskb)->nfctinfo == IP_CT_RELATED+IP_CT_IS_REPLY);

	/* Redirects on non-null nats must be dropped, else they'll
           start talking to each other without our translation, and be
           confused... --RR */
	if (inside->icmp.type == ICMP_REDIRECT) {
		/* If NAT isn't finished, assume it and drop. */
		if ((ct->status & IPS_NAT_DONE_MASK) != IPS_NAT_DONE_MASK)
			return 0;

		if (ct->status & IPS_NAT_MASK)
			return 0;
	}

	DEBUGP("icmp_reply_translation: translating error %p manp %u dir %s\n",
	       *pskb, manip, dir == IP_CT_DIR_ORIGINAL ? "ORIG" : "REPLY");

	if (!ip_ct_get_tuple(&inside->ip, *pskb, (*pskb)->nh.iph->ihl*4 +
	                     sizeof(struct icmphdr) + inside->ip.ihl*4,
	                     &inner,
			     __ip_conntrack_proto_find(inside->ip.protocol)))
		return 0;

	/* Change inner back to look like incoming packet.  We do the
	   opposite manip on this hook to normal, because it might not
	   pass all hooks (locally-generated ICMP).  Consider incoming
	   packet: PREROUTING (DST manip), routing produces ICMP, goes
	   through POSTROUTING (which must correct the DST manip). */
	if (!manip_pkt(inside->ip.protocol, pskb,
		       (*pskb)->nh.iph->ihl*4
		       + sizeof(inside->icmp),
		       &ct->tuplehash[!dir].tuple,
		       !manip))
		return 0;

	if ((*pskb)->ip_summed != CHECKSUM_PARTIAL) {
		/* Reloading "inside" here since manip_pkt inner. */
		inside = (void *)(*pskb)->data + (*pskb)->nh.iph->ihl*4;
		inside->icmp.checksum = 0;
		inside->icmp.checksum = csum_fold(skb_checksum(*pskb, hdrlen,
							       (*pskb)->len - hdrlen,
							       0));
	}

	/* Change outer to look the reply to an incoming packet
	 * (proto 0 means don't invert per-proto part). */
	if (manip == IP_NAT_MANIP_SRC)
		statusbit = IPS_SRC_NAT;
	else
		statusbit = IPS_DST_NAT;

	/* Invert if this is reply dir. */
	if (dir == IP_CT_DIR_REPLY)
		statusbit ^= IPS_NAT_MASK;

	if (ct->status & statusbit) {
		invert_tuplepr(&target, &ct->tuplehash[!dir].tuple);
		if (!manip_pkt(0, pskb, 0, &target, manip))
			return 0;
	}

	return 1;
}
EXPORT_SYMBOL_GPL(ip_nat_icmp_reply_translation);

unsigned int _br0_ip;
unsigned int _br0_mask;
static void get_br0_ip_mask(void)
{
	struct in_device *in_dev;	
	struct net_device *landev;
	struct in_ifaddr *ifap = NULL;
	
      	if ((landev = __dev_get_by_name("br0")) != NULL){
		in_dev=(struct net_device *)(landev->ip_ptr);
		if (in_dev != NULL) {
			for (ifap=in_dev->ifa_list; ifap != NULL; ifap=ifap->ifa_next) {
				if (strcmp("br0", ifap->ifa_label) == 0){
					_br0_ip = ifap->ifa_address;
					_br0_mask = ifap->ifa_mask;
					return; 
				}
			}
			
		}
	}	
}

#if defined(CONFIG_PROC_FS) && defined(CONFIG_RTL865X_HARDWARE_NAT )
static struct proc_dir_entry *proc_hw_nat=NULL;
static char gHwNatSetting[16];
//extern unsigned int ldst, lmask, wdst, wmask;

static int hw_nat_read_proc(char *page, char **start, off_t off,
		     int count, int *eof, void *data)
{
	int len;

	len = sprintf(page, "%s\n", gHwNatSetting);

	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;
	return len;
}

static int hw_nat_write_proc(struct file *file, const char *buffer,
		      unsigned long count, void *data)
{
	if (count < 2) 
		return -EFAULT;
      
	if (buffer && !copy_from_user(&gHwNatSetting, buffer, 8)) {
		write_lock(&ip_nat_lock);
		if (gHwNatSetting[0] == '0') { /* hardware NAT disabled, operation mode = gateway */
			
			rtl865x_changeOpMode(GATEWAY_MODE);
			rtl8651_setAsicOperationLayer(2);
			gHwNatEnabled = 0;
		}
		else if (gHwNatSetting[0] == '1') { /* hardware NAT enabled, operation mode = gateway */
			
			rtl865x_changeOpMode(GATEWAY_MODE);
			rtl8651_setAsicOperationLayer(4);
			gHwNatEnabled = 1;
		}
		else if (gHwNatSetting[0] == '2') { /* hardware NAT enabled, operation mode = bridge mode*/
			//rtl865x_delRoute(wdst, wmask);
			//rtl865x_delRoute(ldst, lmask);
			rtl865x_changeOpMode(BRIDGE_MODE);
			gHwNatEnabled = 2;
		}
		else if(gHwNatSetting[0] == '3'){ /* hardware NAT enabled, operation mode = bridge mode */
			rtl865x_changeOpMode(WISP_MODE);
			gHwNatEnabled = 3;
		}
		#ifdef CONFIG_RTL865X_LAYERED_DRIVER
		else if(gHwNatSetting[0] == '4')/* hardware NAT disabled, operation mode = multiple vlan bridge/WISP */
		{
			rtl865x_changeOpMode(MULTIPLE_VLAN_BRIDGE_MODE);
			gHwNatEnabled = 4;
		}
		else if(gHwNatSetting[0] == '5')/* hardware NAT disabled, operation mode = multiple vlan bridge/WISP */
		{
			rtl865x_changeOpMode(MULTIPLE_VLAN_WISP_MODE);
			gHwNatEnabled = 5;
		}
		#endif
		else if (gHwNatSetting[0] == '8') {
			if (gHwNatSetting[1] == '6') /* L2TP */
				gHwNatEnabled = 0;
		}
		else if (gHwNatSetting[0] == '9') {
			get_br0_ip_mask();
		}
		write_unlock(&ip_nat_lock);
		return count;
	}
	return -EFAULT;
}
#endif

#if defined(CONFIG_PROC_FS) && !defined(CONFIG_RTL865X_HARDWARE_NAT)
static struct proc_dir_entry *proc_sw_nat=NULL;
static char gSwNatSetting[16];

static int sw_nat_read_proc(char *page, char **start, off_t off,
		     int count, int *eof, void *data)
{
	int len;

	len = sprintf(page, "%s\n", gSwNatSetting);

	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;
	return len;
}

static int sw_nat_write_proc(struct file *file, const char *buffer,
		      unsigned long count, void *data)
{
	if (count < 2) 
		return -EFAULT;
      
	if (buffer && !copy_from_user(&gSwNatSetting, buffer, 8)) {
		write_lock(&ip_nat_lock);
		if (gSwNatSetting[0] == '0'){  /* operation mode = GATEWAY */
			//SoftNAT_OP_Mode(2);
			rtl865x_changeOpMode(GATEWAY_MODE);
			rtl8651_setAsicOperationLayer(2);
		}
		else if (gSwNatSetting[0] == '1'){  /* operation mode = BRIDGE*/
			//SoftNAT_OP_Mode(1);
			rtl865x_changeOpMode(BRIDGE_MODE);
			rtl8651_setAsicOperationLayer(2);
		}
		else if(gSwNatSetting[0] == '2'){ /* operation mode = WISP */
			rtl865x_changeOpMode(WISP_MODE);
			rtl8651_setAsicOperationLayer(2);
		}
		#ifdef CONFIG_RTL865X_LAYERED_DRIVER
		else if(gSwNatSetting[0] == '3'){/* operation mode = BRIDGE mode with multiple VLAN */
			rtl865x_changeOpMode(MULTIPLE_VLAN_BRIDGE_MODE);
			rtl8651_setAsicOperationLayer(2);
		}
		else if (gSwNatSetting[0] == '4')/* operation mode = WISP mode with multiple VLAN */
		{
			rtl865x_changeOpMode(MULTIPLE_VLAN_WISP_MODE);
			rtl8651_setAsicOperationLayer(2);
		}
		#endif
		else if(gSwNatSetting[0] == '9'){
			get_br0_ip_mask();
		}
		write_unlock(&ip_nat_lock);
		return count;
	}
	return -EFAULT;
}
#endif
/* Protocol registration. */
int ip_nat_protocol_register(struct ip_nat_protocol *proto)
{
	int ret = 0;

	write_lock_bh(&ip_nat_lock);
	if (ip_nat_protos[proto->protonum] != &ip_nat_unknown_protocol) {
		ret = -EBUSY;
		goto out;
	}
	ip_nat_protos[proto->protonum] = proto;
 out:
	write_unlock_bh(&ip_nat_lock);
	return ret;
}
EXPORT_SYMBOL(ip_nat_protocol_register);

/* Noone stores the protocol anywhere; simply delete it. */
void ip_nat_protocol_unregister(struct ip_nat_protocol *proto)
{
	write_lock_bh(&ip_nat_lock);
	ip_nat_protos[proto->protonum] = &ip_nat_unknown_protocol;
	write_unlock_bh(&ip_nat_lock);

	/* Someone could be still looking at the proto in a bh. */
	synchronize_net();
}
EXPORT_SYMBOL(ip_nat_protocol_unregister);

#if defined(CONFIG_IP_NF_CONNTRACK_NETLINK) || \
    defined(CONFIG_IP_NF_CONNTRACK_NETLINK_MODULE)
int
ip_nat_port_range_to_nfattr(struct sk_buff *skb, 
			    const struct ip_nat_range *range)
{
	NFA_PUT(skb, CTA_PROTONAT_PORT_MIN, sizeof(__be16),
		&range->min.tcp.port);
	NFA_PUT(skb, CTA_PROTONAT_PORT_MAX, sizeof(__be16),
		&range->max.tcp.port);

	return 0;

nfattr_failure:
	return -1;
}

int
ip_nat_port_nfattr_to_range(struct nfattr *tb[], struct ip_nat_range *range)
{
	int ret = 0;
	
	/* we have to return whether we actually parsed something or not */

	if (tb[CTA_PROTONAT_PORT_MIN-1]) {
		ret = 1;
		range->min.tcp.port = 
			*(__be16 *)NFA_DATA(tb[CTA_PROTONAT_PORT_MIN-1]);
	}
	
	if (!tb[CTA_PROTONAT_PORT_MAX-1]) {
		if (ret) 
			range->max.tcp.port = range->min.tcp.port;
	} else {
		ret = 1;
		range->max.tcp.port = 
			*(__be16 *)NFA_DATA(tb[CTA_PROTONAT_PORT_MAX-1]);
	}

	return ret;
}
EXPORT_SYMBOL_GPL(ip_nat_port_nfattr_to_range);
EXPORT_SYMBOL_GPL(ip_nat_port_range_to_nfattr);
#endif

#if defined(CONFIG_PROC_FS) && defined(CONFIG_NET_QOS)

#define	QOS_CLASSIFY_INFO_LEN	16
typedef struct {
	/*	classify	*/
	unsigned int protocol;
	struct in_addr local_ip;
	struct in_addr  remote_ip;
	unsigned short lo_port;
	unsigned short re_port;

	/*	tc	*/
	uint32		mark;
	unsigned char	prio;
} rtl865x_qos_cache_t;

static struct proc_dir_entry *proc_qos=NULL;

static char *gQosSetting = NULL;
#ifdef CONFIG_FAST_PATH_MODULE
EXPORT_SYMBOL(gQosEnabled);
#endif
#if (!defined(CONFIG_QOS_UI_BY_BANDWIDTH))
/*	Currently, only record the qos info for wan interface	*/
__DRAM_GEN static rtl865x_qos_entry_t gQosTbl[MAX_QOS_RULE_NUM];
static rtl865x_qos_t	gQosInfo;
__DRAM_GEN rtl865x_qos_cache_t		gQosCache;
#endif

#if (!defined(CONFIG_QOS_UI_BY_BANDWIDTH))
static unsigned long atoi_dec(char *s)
{
	unsigned long k = 0;

	k = 0;
	while (*s != '\0' && *s >= '0' && *s <= '9') {
		k = 10 * k + (*s - '0');
		s++;
	}
	return k;
}

/* 
 * Check whether "cp" is a valid ascii representation
 * of an Internet address and convert to a binary address.
 * Returns 1 if the address is valid, 0 if not.
 * This replaces inet_addr, the return value from which
 * cannot distinguish between failure and a local broadcast address.
 */
typedef unsigned long in_addr_t; 
static int inet_aton(const char *cp, struct in_addr *addr)
{
	register in_addr_t val;
	register int base, n;
	register char c;
	unsigned int parts[4];
	register unsigned int *pp = parts;

	c = *cp;
	for (;;) {
		/*
		 * Collect number up to ``.''.
		 * Values are specified as for C:
		 * 0x=hex, 0=octal, isdigit=decimal.
		 */
		if (!isdigit(c))
			return (0);
		val = 0; base = 10;
		if (c == '0') {
			c = *++cp;
			if (c == 'x' || c == 'X')
				base = 16, c = *++cp;
			else
				base = 8;
		}
		for (;;) {
			if (isascii(c) && isdigit(c)) {
				val = (val * base) + (c - '0');
				c = *++cp;
			} else if (base == 16 && isascii(c) && isxdigit(c)) {
				val = (val << 4) |
					(c + 10 - (islower(c) ? 'a' : 'A'));
				c = *++cp;
			} else
				break;
		}
		if (c == '.') {
			/*
			 * Internet format:
			 *	a.b.c.d
			 *	a.b.c	(with c treated as 16 bits)
			 *	a.b	(with b treated as 24 bits)
			 */
			if (pp >= parts + 3)
				return (0);
			*pp++ = val;
			c = *++cp;
		} else
			break;
	}
	/*
	 * Check for trailing characters.
	 */
	if (c != '\0' && (!isascii(c) || !isspace(c)))
		return (0);
	/*
	 * Concoct the address according to
	 * the number of parts specified.
	 */
	n = pp - parts + 1;
	switch (n) {

	case 0:
		return (0);		/* initial nondigit */

	case 1:				/* a -- 32 bits */
		val <<=24;
		break;

	case 2:				/* a.b -- 8.24 bits */
		val <<=16;
		if ((val > 0xffffff) || (parts[0] > 0xff))
			return (0);
		val |= parts[0] << 24;
		break;

	case 3:				/* a.b.c -- 8.8.16 bits */
		val <<=8;
		if ((val > 0xffff) || (parts[0] > 0xff) || (parts[1] > 0xff))
			return (0);
		val |= (parts[0] << 24) | (parts[1] << 16);
		break;

	case 4:				/* a.b.c.d -- 8.8.8.8 bits */
		if ((val > 0xff) || (parts[0] > 0xff) || (parts[1] > 0xff) || (parts[2] > 0xff))
			return (0);
		val |= (parts[0] << 24) | (parts[1] << 16) | (parts[2] << 8);
		break;
	}
	if (addr)
		addr->s_addr = htonl(val);
	return (1);
}
#endif

static int qos_read_proc(char *page, char **start, off_t off,
		     int count, int *eof, void *data)
{
      int len;

      len = sprintf(page, "%s\n", gQosSetting);

      if (len <= off+count) *eof = 1;
      *start = page + off;
      len -= off;
      if (len>count) len = count;
      if (len<0) len = 0;
      return len;
}

static int qos_write_proc(struct file *file, const char *buffer,
		      unsigned long count, void *data)
{
#if (!defined(CONFIG_QOS_UI_BY_BANDWIDTH))
      int i;
      char *str, *token;
      char *tmpbuf;
#endif

      if ( gQosSetting==NULL || count < 2) 
	    return -EFAULT;

      if (buffer && !copy_from_user(gQosSetting, buffer, count)) {
          gQosSetting[count-1] = 0; /* remove 0x0a */
          if (gQosSetting[0] == '0')
                gQosEnabled = 0;
          else
                gQosEnabled = 1;

#if (!defined(CONFIG_QOS_UI_BY_BANDWIDTH))
	tmpbuf = kmalloc(1024, GFP_KERNEL);
	if (!tmpbuf)
		return -EFAULT;

          strcpy(tmpbuf, gQosSetting);

          str = tmpbuf+2;
          memset(gQosTbl, 0, MAX_QOS_RULE_NUM * sizeof(QOS_T));
	   memset(&gQosInfo, 0, sizeof(rtl865x_qos_t));
	   memset(&gQosCache, 0, sizeof(rtl865x_qos_cache_t));
	   
          token = strtok(str,",");
	   memcpy(gQosInfo.ifname, token, sizeof(token));
	   token = strtok(NULL,",");
	   gQosInfo.bandwidth = atoi_dec(token);
	   token = strtok(NULL,",");
	   token = strtok(NULL,",");

          for (i=0 ; (i<MAX_QOS_RULE_NUM) && (token != NULL); i++, token = strtok(NULL,","))
          {
             gQosTbl[i].protocol = atoi_dec(token);
	      	if((token = strtok(NULL,","))==NULL)
	        	break;
		if ( !inet_aton(token, &(gQosTbl[i].local_ip_start)))
                    break;             
	      	if((token = strtok(NULL,","))==NULL)
	        	break;
		if ( !inet_aton(token, &(gQosTbl[i].local_ip_end)))
                    break;
	      	if((token = strtok(NULL,","))==NULL)
	        	break;
		if ( !inet_aton(token, &(gQosTbl[i].remote_ip_start)))
                    break;
	      	if((token = strtok(NULL,","))==NULL)
	        	break;
		if ( !inet_aton(token, &(gQosTbl[i].remote_ip_end)))
                    break;        

	      	if((token = strtok(NULL,","))==NULL)
	        	break;
             gQosTbl[i].lo_port_start = atoi_dec(token);
	      	if((token = strtok(NULL,","))==NULL)
	        	break;
             gQosTbl[i].lo_port_end = atoi_dec(token);
	      	if((token = strtok(NULL,","))==NULL)
	        	break;
             gQosTbl[i].re_port_start = atoi_dec(token);
	      	if((token = strtok(NULL,","))==NULL)
	        	break;
             gQosTbl[i].re_port_end = atoi_dec(token);
		if((token = strtok(NULL,","))==NULL)
	        	break;
             gQosTbl[i].mark= atoi_dec(token);
		if((token = strtok(NULL,","))==NULL)
	        	break;
             gQosTbl[i].rate= atoi_dec(token);
          }

          /* let gQosEnabled = 0 if no any QoS rule be set */
          if (gQosTbl[0].protocol == 0) 
              gQosEnabled = 0;
	    kfree(tmpbuf);
#endif
	    return count;
      }
      return -EFAULT;
}

#if (!defined(CONFIG_QOS_UI_BY_BANDWIDTH))
/* proto: 1 TCP, 0 UDP */
static int checkQosSetting( u_int16_t proto,u_int32_t sip, u_int32_t dip, u_int16_t sp, u_int16_t dp)
{
#if defined(CONFIG_RTL865X_HW_QOS_SUPPORT)
	return 1; /* can be added to H/W NAT table */
#else
    int i;
    unsigned long v1, v2, v3, v4;   
    
    for (i=0; i<MAX_QOS_RULE_NUM; i++)
    {
        if (gQosTbl[i].protocol == 0) 
            break;
    
        v1 = *((unsigned long *)&(gQosTbl[i].local_ip_start));
        v2 = *((unsigned long *)&(gQosTbl[i].local_ip_end));
        v3 = *((unsigned long *)&(gQosTbl[i].remote_ip_start));
        v4 = *((unsigned long *)&(gQosTbl[i].remote_ip_end));
    
        if (((proto == 1 && gQosTbl[i].protocol != IPPROTO_UDP) || (proto == 0 && gQosTbl[i].protocol != IPPROTO_TCP)) &&
            (sip >= v1 && sip <= v2) && (dip >= v3 && dip <= v4) ) {
            if ((sp >= gQosTbl[i].lo_port_start && sp <= gQosTbl[i].lo_port_end) && (dp >= gQosTbl[i].re_port_start && dp <= gQosTbl[i].re_port_end)) {
                return 0; /* do not add to H/W NAT table */
            }
        }        
    }
    return 1; /* can be added to H/W NAT table */
#endif
}
#endif
#endif

static int __init ip_nat_init(void)
{
	size_t i;

	/* Leave them the same for the moment. */
	ip_nat_htable_size = ip_conntrack_htable_size;

	/* One vmalloc for both hash tables */
	bysource = vmalloc(sizeof(struct list_head) * ip_nat_htable_size);
	if (!bysource)
		return -ENOMEM;

	/* Sew in builtin protocols. */
	write_lock_bh(&ip_nat_lock);
	for (i = 0; i < MAX_IP_NAT_PROTO; i++)
		ip_nat_protos[i] = &ip_nat_unknown_protocol;
	ip_nat_protos[IPPROTO_TCP] = &ip_nat_protocol_tcp;
	ip_nat_protos[IPPROTO_UDP] = &ip_nat_protocol_udp;
	ip_nat_protos[IPPROTO_ICMP] = &ip_nat_protocol_icmp;
	write_unlock_bh(&ip_nat_lock);

	for (i = 0; i < ip_nat_htable_size; i++) {
		INIT_LIST_HEAD(&bysource[i]);
	}

	/* FIXME: Man, this is a hack.  <SIGH> */
	IP_NF_ASSERT(ip_conntrack_destroyed == NULL);
	ip_conntrack_destroyed = &ip_nat_cleanup_conntrack;

	/* Initialize fake conntrack so that NAT will skip it */
	ip_conntrack_untracked.status |= IPS_NAT_DONE_MASK;

#if defined(CONFIG_PROC_FS) && defined(CONFIG_RTL865X_HARDWARE_NAT)
	proc_hw_nat = create_proc_entry("hw_nat", 0, NULL);
	if (proc_hw_nat) {
	    proc_hw_nat->read_proc = hw_nat_read_proc;
	    proc_hw_nat->write_proc = hw_nat_write_proc;
      }  
#endif


#if defined(CONFIG_PROC_FS) && defined(CONFIG_NET_QOS)
	proc_qos = create_proc_entry("qos", 0, NULL);
	if (proc_qos) {
	    proc_qos->read_proc = qos_read_proc;
	    proc_qos->write_proc = qos_write_proc;
      }  
#endif

#if defined(CONFIG_PROC_FS) && !defined(CONFIG_RTL865X_HARDWARE_NAT)
	proc_sw_nat = create_proc_entry("sw_nat", 0, NULL);
	if (proc_sw_nat) {
		proc_sw_nat->read_proc = sw_nat_read_proc;
		proc_sw_nat->write_proc = sw_nat_write_proc;
	}  
#endif

#ifdef CONFIG_RTL865X_HARDWARE_NAT
	gHwNatEnabled=1;
#endif
#if defined(CONFIG_PROC_FS) && defined(CONFIG_NET_QOS)
	gQosSetting = kmalloc(1024, GFP_KERNEL);
#endif
	return 0;
}

/* Clear NAT section of all conntracks, in case we're loaded again. */
static int clean_nat(struct ip_conntrack *i, void *data)
{
	memset(&i->nat, 0, sizeof(i->nat));
	i->status &= ~(IPS_NAT_MASK | IPS_NAT_DONE_MASK | IPS_SEQ_ADJUST);
	return 0;
}

static void __exit ip_nat_cleanup(void)
{
	ip_ct_iterate_cleanup(&clean_nat, NULL);
	ip_conntrack_destroyed = NULL;
	vfree(bysource);
#if defined(CONFIG_PROC_FS) && defined(CONFIG_NET_QOS)
	kfree(gQosSetting);
#endif

}

MODULE_LICENSE("GPL");

module_init(ip_nat_init);
module_exit(ip_nat_cleanup);
