#include <linux/module.h>
#include <linux/netfilter.h>
#include <linux/ip.h>
#include <net/checksum.h>
#include <net/tcp.h>
#include <net/udp.h>

#include <linux/netfilter_ipv4/ip_conntrack_helper.h>
#include <linux/netfilter_ipv4/ip_conntrack_protocol.h>
#include <linux/netfilter_ipv4/ip_conntrack_ipsec.h>	
#ifdef CONFIG_IP_NF_ALG_ONOFF
#define PROCFS_NAME "algonoff_ipsec"
ALGONOFF_INIT(ipsec)
#endif
/*
 * Prototypes of IPSec NAT hook
 */
unsigned int (*ip_nat_ipsec_inbound_hook) 
				(struct sk_buff **pskb,
				struct ip_conntrack *ct,
				 enum ip_conntrack_info ctinfo,
				 struct ip_conntrack_expect *exp);

EXPORT_SYMBOL_GPL(ip_nat_ipsec_inbound_hook);


unsigned int (*ip_nat_ipsec_outbound_hook) 
				(struct sk_buff **pskb,
				struct ip_conntrack *ct,
				 enum ip_conntrack_info ctinfo,
				 struct ip_conntrack_expect *exp);

EXPORT_SYMBOL_GPL(ip_nat_ipsec_outbound_hook);

unsigned int (*ip_nat_esp_hook)	
				(struct sk_buff *skb,
				struct ip_conntrack *ct,	
				 enum ip_conntrack_info ctinfo,
				 struct ip_conntrack_expect *exp);

EXPORT_SYMBOL_GPL(ip_nat_esp_hook);

#if 0
#define DEBUGP printk
#else
#define DEBUGP(format, args...)
#endif


static int ports[MAX_PORTS];
static int ports_n_c = 0;

extern char ipsec_flag;

unsigned long ip_ct_esp_timeout = 30*HZ;
unsigned long ip_ct_esp_timeout_stream = 180*HZ;


static int esp_pkt_to_tuple(const struct sk_buff *skb, unsigned int dataoff,
			    struct ip_conntrack_tuple *tuple)
{
	struct udphdr _hdr, *hdr;

	hdr = skb_header_pointer(skb, dataoff, sizeof(_hdr), &_hdr);
	if(hdr == NULL)
		return 0;

	tuple->src.u.all = hdr->source;
	tuple->dst.u.all = hdr->dest;

	DEBUGP("%s: tuple->src = src %x, tuple->dst = dst %x\n", __func__, tuple->src.u.all, tuple->dst.u.all);
	
	return 1;
}

static int esp_invert_tuple(struct ip_conntrack_tuple *tuple,
			    const struct ip_conntrack_tuple *orig)
{
	tuple->src.u.all = orig->dst.u.all;
	tuple->dst.u.all = orig->src.u.all;

	DEBUGP("%s: tuple->src = dst %x, dst = src %x\n", __func__, orig->dst.u.all, orig->src.u.all);
	return 1;
}

/* Print out the per-protocol part of the tuple. */
static int esp_print_tuple(struct seq_file *buffer,
					const struct ip_conntrack_tuple *tuple)
{

	return seq_printf(buffer, "sport=%hu dport=%hu ",

		       ntohs(tuple->src.u.all),
		       ntohs(tuple->dst.u.all));
}

/* Print out the private part of the conntrack. */
static int esp_print_conntrack(struct seq_file *buffer,
					const struct ip_conntrack *conntrack)
{
	return 0;
}

/* Returns verdict for packet, and may modify conntracktype */
#ifdef CONFIG_RTL867X_KERNEL_MIPS16_NET
__NOMIPS16
#endif
static int esp_packet(struct ip_conntrack *conntrack,      
		const struct sk_buff *skb,
		enum ip_conntrack_info conntrackinfo)
{


	if (ip_nat_esp_hook) {
		if (!ip_nat_esp_hook(skb, conntrack, conntrackinfo, NULL)) 
			return  NF_DROP;
	}

	/* If we've seen traffic both ways, this is some kind of UDP
	   stream.  Extend timeout. */

	if(conntrackinfo== IP_CT_IS_REPLY){
		ip_ct_refresh(conntrack, skb, ip_ct_esp_timeout_stream);	
		DEBUGP("%s: refresh ip_ct on kinda UDP stream.\n", __func__);

		/* Also, more likely to be important, and not a probe */
		set_bit(IPS_ASSURED_BIT, &conntrack->status);
		ip_conntrack_event_cache(IPCT_STATUS, skb);

	} else
		ip_ct_refresh(conntrack, skb, ip_ct_esp_timeout);
		DEBUGP("%s: refresh ip_ct with shorter expected timeout.\n", __func__);

	return NF_ACCEPT;
}

/* Called when a new connection for this protocol found. */
static int esp_new(struct ip_conntrack *conntrack,
			     const struct sk_buff *skb)
{	
	return 1;
}


static int esp_error(struct sk_buff *skb, enum ip_conntrack_info *ctinfo,
		     unsigned int hooknum)
{
	return 1;
}

struct ip_conntrack_protocol ip_conntrack_protocol_esp = {
	.proto		 = IPPROTO_ESP,
	.name		 = "esp",
	.pkt_to_tuple	 = esp_pkt_to_tuple,
	.invert_tuple	 = esp_invert_tuple,
	.print_tuple	 = esp_print_tuple,
	.print_conntrack = esp_print_conntrack,
	.packet		 = esp_packet,
	.new		 = esp_new,
	.error 		 = esp_error,
	.destroy	 = NULL,
	.me 		 = THIS_MODULE,
#if defined(CONFIG_IP_NF_CONNTRACK_NETLINK) || \
    defined(CONFIG_IP_NF_CONNTRACK_NETLINK_MODULE)
      //Fix me later- shlee
	.tuple_to_nfattr = NULL,
	.nfattr_to_tuple = NULL,
#endif
};


/* FIXME: This should be in userspace.  Later. */
static int help(struct sk_buff **pskb,
		struct ip_conntrack *ct, enum ip_conntrack_info ctinfo)
{
#ifdef CONFIG_IP_NF_ALG_ONOFF
	if(!algonoff_ipsec) {
		printk("DROP ipsec PACKET!!!\n");
		return NF_DROP;
	}
#endif
	struct ip_conntrack_expect * exp;
	int ret = NF_ACCEPT;

	//workaround- refresh udp:500 conntrack with 10 seconds for 2st client connection.
	ip_ct_refresh_acct(ct, ctinfo, *pskb, 10*HZ);

	//Allocate expectation on next reply, just care bout src ip, dst ip and protocol

	exp = ip_conntrack_expect_alloc(ct);
	if (exp == NULL)
		return NF_DROP;

	exp->tuple = ct->tuplehash[IP_CT_DIR_REPLY].tuple;
	exp->mask.src.ip = htonl(0xffffffff);
	exp->mask.src.u.udp.port = 0;	
	exp->mask.dst.ip = htonl(0xffffffff);
	exp->mask.dst.u.udp.port = 0;	
	exp->mask.dst.protonum = 0xff;
	exp->expectfn = NULL;
	exp->flags = 0;

	DEBUGP("expect: ");
	DUMP_TUPLE(&exp->tuple);
	DUMP_TUPLE(&exp->mask);

	//call outbound hook to record and update the track of ipsec 
	if (ip_nat_ipsec_outbound_hook) 
		if (!ip_nat_ipsec_outbound_hook(pskb, ct, ctinfo, exp)) 
			return  NF_DROP;


	// register expect tuple, ip_conntrack_in will hit the expected pkt and update the conntrack state
	if (ip_conntrack_expect_related(exp) != 0)
		ret = NF_DROP;
	ip_conntrack_expect_put(exp);


//	if(ipsec_flag == '1') 
		/* tcplen not negative guaranteed by ip_conntrack_tcp.c */
		ip_conntrack_protocol_register(&ip_conntrack_protocol_esp);

		
	return ret;
}

static struct ip_conntrack_helper ipsec_helpers[MAX_PORTS];

static void fini(void);



static int alginit_ipsec(void)
{
int i ;
#ifdef CONFIG_IP_NF_ALG_ONOFF
ports_n_c=0;
#endif

	/* If no port given, default to standard irc port */

	if (ports[0] == 0)
		ports[0] = 500;

	for (i = 0; (i < MAX_PORTS) && ports[i]; i++) {
		memset(&ipsec_helpers[i], 0,
		       sizeof(struct ip_conntrack_helper));
		ipsec_helpers[i].tuple.src.u.udp.port = htons(ports[i]);
		ipsec_helpers[i].tuple.dst.protonum = IPPROTO_UDP;
		ipsec_helpers[i].mask.src.u.udp.port = 0xFFFF;
		ipsec_helpers[i].mask.dst.protonum = 0xFF;
		//init timeout for expected or kernel will bitch it in helper registering - shlee
		ipsec_helpers[i].timeout = ip_ct_esp_timeout_stream;;
		ipsec_helpers[i].me = THIS_MODULE;
		ipsec_helpers[i].max_expected = 2;
		ipsec_helpers[i].name = "ipsec";
		ipsec_helpers[i].help = help;

		if( ip_conntrack_helper_register(&ipsec_helpers[i])){
			printk("ip_conntrack_ipsec: ERROR registering port %d\n",
				ports[i]);
			fini();
			return;
		}
		ports_n_c++;
	}
	printk("ip_conntrack_ipsec loaded\n");  

}

static int __init init(void)
{
#ifdef CONFIG_IP_NF_ALG_ONOFF
	Alg_OnOff_init_ipsec();
#endif
	alginit_ipsec();
#ifdef CONFIG_IP_NF_ALG_ONOFF
	algonoff_ipsec=1;	
#endif
	return 0;
}

/* This function is intentionally _NOT_ defined as __exit, because 
 * it is needed by the init function */
static void fini(void)
{
	int i;
	for (i = 0; (i < MAX_PORTS) && ports[i]; i++) {
		ip_conntrack_helper_unregister(&ipsec_helpers[i]);
	}
}

module_init(init);
module_exit(fini);
