/*
 *	Handle incoming frames
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	$Id: br_input.c,v 1.1.1.1 2009/02/17 09:44:45 anderson Exp $
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/netfilter_bridge.h>
#include "br_private.h"

#ifdef IGMP_SNOOPING
/*2008-01-15,for porting igmp snooping to linux kernel 2.6*/
#include <linux/ip.h>
#include <linux/in.h>
#include <linux/igmp.h>
#include <net/checksum.h>
extern int igmpsnoopenabled;
#ifdef CONFIG_BRIDGE_IGMPV3_SNOOPING
#include "../../drivers/net/re865x/igmpsnooping/rtl865x_igmpsnooping_glue.h"
#include "../../drivers/net/re865x/igmpsnooping/rtl865x_igmpsnooping.h"
extern unsigned int brIgmpModuleIndex;
#endif
#endif

/* Bridge group multicast address 802.1d (pg 51). */
const u8 br_group_address[ETH_ALEN] = { 0x01, 0x80, 0xc2, 0x00, 0x00, 0x00 };

//plus add ;2008-0609
#ifdef	MCAST_TO_UNICAST
extern int IGMPProxyOpened;

#ifdef	IPV6_MCAST_TO_UNICAST
#include <linux/ipv6.h>
#include <linux/in6.h>
#include <linux/icmpv6.h>
//#define	DBG_ICMPv6	//enable it to debug icmpv6 check
static int ICMPv6_check(struct sk_buff *skb , unsigned char *gmac);
#endif	//end of IPV6_MCAST_TO_UNICAST

#endif

#ifdef IGMP_SNOOPING
/*2008-01-15,for porting igmp snooping to linux kernel 2.6*/
static char igmp_type_check(struct sk_buff *skb, unsigned char *gmac);
static void br_update_igmp_snoop_fdb(unsigned char op, struct net_bridge *br, struct net_bridge_port *p, unsigned char *gmac
#ifdef	MCAST_TO_UNICAST
	, struct sk_buff *skb
#endif
);
#endif

static void br_pass_frame_up(struct net_bridge *br, struct sk_buff *skb)
{
	struct net_device *indev;

#ifdef IGMP_SNOOPING
#ifdef CONFIG_BRIDGE_IGMPV3_SNOOPING
/*2008-01-15,for porting igmp snooping to linux kernel 2.6*/
	unsigned char *dest;
	struct net_bridge_port *p;
	struct iphdr *iph;
	unsigned char proto=0;  
                          
       iph = skb->nh.iph;
	proto =  iph->protocol;    
	dest =  eth_hdr(skb)->h_dest;
	
       p = skb->dev->br_port;
	 if (igmpsnoopenabled && MULTICAST_MAC(dest) && (eth_hdr(skb)->h_proto == ETH_P_IP)){
                if (proto== IPPROTO_IGMP){
			uint32 fwdPortMask;
			#ifdef CONFIG_BRIDGE_IGMPV3_SNOOPING_DEBUG
			printk("%s:%d,the len is %d\n", __FUNCTION__,__LINE__,skb->len);
			printk("%s:%d,the port number is %d\n", __FUNCTION__,__LINE__,p->port_no);
			printk("%s:%d,DMAC is 0x%x-%x-%x-%x-%x-%x\n", __FUNCTION__,__LINE__, skb->mac.raw[0], skb->mac.raw[1], skb->mac.raw[2], skb->mac.raw[3], skb->mac.raw[4], skb->mac.raw[5]);
			#endif
			rtl_igmpMldProcess(brIgmpModuleIndex, skb->mac.raw, p->port_no, &fwdPortMask);
		}
        }
#else

/*2008-01-15,for porting igmp snooping to linux kernel 2.6*/
        unsigned char *dest;
        struct net_bridge_port *p;
        unsigned char mac[6];
        unsigned char operation;
        struct iphdr *iph;
	unsigned char proto=0;  
                          
       iph = skb->nh.iph;
	proto =  iph->protocol;    
	dest =  eth_hdr(skb)->h_dest;
	
       p = skb->dev->br_port;
	
	 if (igmpsnoopenabled && MULTICAST_MAC(dest) && (eth_hdr(skb)->h_proto == ETH_P_IP)){
                if (proto== IPPROTO_IGMP){
			if ((operation=igmp_type_check(skb, mac)) > 0) {
#ifdef	MCAST_TO_UNICAST
               	            br_update_igmp_snoop_fdb(operation, br, p, mac,skb);
#else
                             br_update_igmp_snoop_fdb(operation, br, p, mac);
#endif
                          }
		}
        }
	#ifdef	MCAST_TO_UNICAST	
	#ifdef	IPV6_MCAST_TO_UNICAST
	else if(igmpsnoopenabled 
		&& IPV6_MULTICAST_MAC(dest) 
		&& (skb->mac.ethernet->h_proto == ETH_P_IPV6) )
	{		
		operation = ICMPv6_check(skb , macAddr);
		if (operation > 0) {
				#ifdef	DBG_ICMPv6
				if( operation == 1)
					printk("ICMPv6 mac add (from frame_up)\n");
				else if(operation == 2)
					printk("ICMPv6 mac del (from frame_up)\n");	
				#endif
               	br_update_igmp_snoop_fdb(operation, br, p, macAddr,skb);
		}
	}
	#endif	//end of IPV6_MCAST_TO_UNICAST
	#endif
#endif	 
#endif

	br->statistics.rx_packets++;
	br->statistics.rx_bytes += skb->len;

	indev = skb->dev;
	skb->dev = br->dev;

	NF_HOOK(PF_BRIDGE, NF_BR_LOCAL_IN, skb, indev, NULL,
		netif_receive_skb);
}

/* note: already called with rcu_read_lock (preempt_disabled) */
int br_handle_frame_finish(struct sk_buff *skb)
{
	const unsigned char *dest = eth_hdr(skb)->h_dest;
	struct net_bridge_port *p = rcu_dereference(skb->dev->br_port);
	struct net_bridge *br;
	struct net_bridge_fdb_entry *dst;
	int passedup = 0;

	if (!p || p->state == BR_STATE_DISABLED)
		goto drop;

	/* insert into forwarding database after filtering to avoid spoofing */
	br = p->br;
	br_fdb_update(br, p, eth_hdr(skb)->h_source);

	if (p->state == BR_STATE_LEARNING)
		goto drop;

	if (br->dev->flags & IFF_PROMISC) {
		struct sk_buff *skb2;

		skb2 = skb_clone(skb, GFP_ATOMIC);
		if (skb2 != NULL) {
			passedup = 1;
			br_pass_frame_up(br, skb2);
		}
	}

	if (is_multicast_ether_addr(dest)) {
		br->statistics.multicast++;
#ifdef IGMP_SNOOPING
		struct iphdr *iph;
		unsigned int ipaddr=0;
		unsigned char proto=0;
		unsigned char reserved=0;
		unsigned char mac[6];
		unsigned char operation;

		if (!(br->dev->flags & IFF_PROMISC) &&
                    MULTICAST_MAC(dest) && (eth_hdr(skb)->h_proto == ETH_P_IP)){
	  
			
			iph = skb->nh.iph;
			ipaddr =  iph->daddr;
			proto =  iph->protocol;  
	                if (proto == IPPROTO_IGMP) {	
                	        if ((operation=igmp_type_check(skb, mac)) > 0){
#ifdef	MCAST_TO_UNICAST
					br_update_igmp_snoop_fdb(operation, br, p, mac,skb);
#else
					br_update_igmp_snoop_fdb(operation, br, p, mac);
#endif		
                                 }
                	}
	        }
		#ifdef	IPV6_MCAST_TO_UNICAST
		else if(!(br->dev->flags & IFF_PROMISC) 
				&& IPV6_MULTICAST_MAC(dest)
				&& (eth_hdr(skb)->h_proto == ETH_P_IPV6))
		{
								
				operation = ICMPv6_check(skb , mac);
				if(operation > 0){
					#ifdef	DBG_ICMPv6
					if( operation == 1)
						printk("icmpv6 add from frame finish\n");
					else if(operation == 2)
						printk("icmpv6 del from frame finish\n");	
					#endif
					br_update_igmp_snoop_fdb(operation, br, p, mac,skb);
				}
				
		}
		#endif	
#ifdef CONFIG_BRIDGE_IGMPV3_SNOOPING
/*2008-01-15,for porting igmp snooping to linux kernel 2.6*/
		if (!(br->dev->flags & IFF_PROMISC) &&
                    MULTICAST_MAC(dest) && (eth_hdr(skb)->h_proto == ETH_P_IP)){
			iph = skb->nh.iph;
			ipaddr =  iph->daddr;
			proto =  iph->protocol; 
			if (proto == IPPROTO_IGMP) 
			{
				uint32 fwdPortMask;
				#ifdef CONFIG_BRIDGE_IGMPV3_SNOOPING_DEBUG
				printk("%s:%d,the len is %d\n", __FUNCTION__,__LINE__,skb->len);
				printk("%s:%d,the port number is %d\n", __FUNCTION__,__LINE__,p->port_no);
				printk("%s:%d,DMAC is 0x%x-%x-%x-%x-%x-%x\n", __FUNCTION__,__LINE__, skb->mac.raw[0], skb->mac.raw[1], skb->mac.raw[2], skb->mac.raw[3], skb->mac.raw[4], skb->mac.raw[5]);
				#endif
				rtl_igmpMldProcess(brIgmpModuleIndex, skb->mac.raw, p->port_no, &fwdPortMask);	
				
                	}
	        }
	        
	        if((ipaddr&0xFFFFFF00)==0xE0000000)
	        	reserved=1;
	    
		if(igmpsnoopenabled && MULTICAST_MAC(dest) && proto !=IPPROTO_IGMP && ipaddr != 0xEFFFFFFA && reserved ==0) 
		{	
			int retVal=FAILED;
			unsigned int fwdPortMask=0;
			struct rtl_multicastDataInfo multicastDataInfo;
			
			iph = skb->nh.iph;
			multicastDataInfo.ipVersion=4;
			multicastDataInfo.sourceIp[0]=  (uint32)(iph->saddr);
			multicastDataInfo.groupAddr[0]=  (uint32)(iph->daddr);
			
			retVal= rtl_getMulticastDataFwdPortMask(brIgmpModuleIndex, &multicastDataInfo, &fwdPortMask);
			#ifdef CONFIG_BRIDGE_IGMPV3_SNOOPING_DEBUG
			printk("%s:%d,fwdPortMask is 0x%x\n",__FUNCTION__,__LINE__,fwdPortMask);
			#endif
			if(retVal==SUCCESS)
			{
				#ifdef CONFIG_RTL865X_HARDWARE_MULTICAST
				unsigned int srcPort=skb->srcPort;
				unsigned int srcVlanId=skb->srcVlanId;
				#endif
			
				br_multicast_forward(br, fwdPortMask, skb, !passedup);
			
				#ifdef CONFIG_RTL865X_HARDWARE_MULTICAST
				if((srcVlanId!=0) && (srcPort!=0xFFFF))
				{
					rtl865x_ipMulticastHardwareAccelerate(br, fwdPortMask,srcPort,srcVlanId, multicastDataInfo.sourceIp[0], multicastDataInfo.groupAddr[0]);
				}	
				#endif
			}
			else
			{
				br_flood_forward(br, skb, !passedup);
			}
		}
		else 
		{ 
			// broadcast
			br_flood_forward(br, skb, !passedup);
		}
#else
/*2008-01-15,for porting igmp snooping to linux kernel 2.6*/
		struct iphdr *iph;
		unsigned int ipaddr=0;
		unsigned char proto=0;
		unsigned char reserved=0;
		#if 1
		if (!(br->dev->flags & IFF_PROMISC) &&
                    MULTICAST_MAC(dest) && (eth_hdr(skb)->h_proto == ETH_P_IP)){
		#else
		if (!(br->dev.flags & IFF_PROMISC) &&
                    MULTICAST_MAC(dest) && (skb->mac.ethernet->h_proto == ETH_P_IP)){
              #endif      
			unsigned char mac[6];
			unsigned char operation;
			
			iph = skb->nh.iph;
			ipaddr =  iph->daddr;
			proto =  iph->protocol;  
	                if (proto == IPPROTO_IGMP) {	
                	        if ((operation=igmp_type_check(skb, mac)) > 0){
#ifdef	MCAST_TO_UNICAST
					br_update_igmp_snoop_fdb(operation, br, p, mac,skb);
#else
					br_update_igmp_snoop_fdb(operation, br, p, mac);
#endif		
                                 }
                	}
	        }
	        
	        if((ipaddr&0xFFFFFF00)==0xE0000000)
	        	reserved=1;
		
		if(igmpsnoopenabled && MULTICAST_MAC(dest) && proto !=IPPROTO_IGMP && ipaddr != 0xEFFFFFFA && reserved ==0) {	
                        if ((dst = br_fdb_get(br, dest)) != NULL){
                                br_multicast_forward(br, dst, skb, !passedup);
                                br_fdb_put(dst);
                        }else{
                        	br_flood_forward(br, skb, !passedup);
                    	}
                }
                else { // broadcast
			br_flood_forward(br, skb, !passedup);
                }
		
#endif		
#else
		br_flood_forward(br, skb, !passedup);
#endif
		if (!passedup)
			br_pass_frame_up(br, skb);
		goto out;
	}

	dst = __br_fdb_get(br, dest);
	if (dst != NULL && dst->is_local) {
		if (!passedup)
			br_pass_frame_up(br, skb);
		else
			kfree_skb(skb);
		goto out;
	}

	if (dst != NULL) {
		br_forward(dst->dst, skb);
		goto out;
	}

	br_flood_forward(br, skb, 0);

out:
	return 0;
drop:
	kfree_skb(skb);
	goto out;
}

/* note: already called with rcu_read_lock (preempt_disabled) */
static int br_handle_local_finish(struct sk_buff *skb)
{
	struct net_bridge_port *p = rcu_dereference(skb->dev->br_port);

	if (p && p->state != BR_STATE_DISABLED)
		br_fdb_update(p->br, p, eth_hdr(skb)->h_source);

	return 0;	 /* process further */
}

/* Does address match the link local multicast address.
 * 01:80:c2:00:00:0X
 */
static inline int is_link_local(const unsigned char *dest)
{
	return memcmp(dest, br_group_address, 5) == 0 && (dest[5] & 0xf0) == 0;
}

/*
 * Called via br_handle_frame_hook.
 * Return 0 if *pskb should be processed furthur
 *	  1 if *pskb is handled
 * note: already called with rcu_read_lock (preempt_disabled) 
 */
int br_handle_frame(struct net_bridge_port *p, struct sk_buff **pskb)
{
	struct sk_buff *skb = *pskb;
	const unsigned char *dest = eth_hdr(skb)->h_dest;

	if (!is_valid_ether_addr(eth_hdr(skb)->h_source))
		goto err;

	if (unlikely(is_link_local(dest))) {
		skb->pkt_type = PACKET_HOST;
		return NF_HOOK(PF_BRIDGE, NF_BR_LOCAL_IN, skb, skb->dev,
			       NULL, br_handle_local_finish) != 0;
	}

	if (p->state == BR_STATE_FORWARDING || p->state == BR_STATE_LEARNING) {
		if (br_should_route_hook) {
			if (br_should_route_hook(pskb)) 
				return 0;
			skb = *pskb;
			dest = eth_hdr(skb)->h_dest;
		}

		if (!compare_ether_addr(p->br->dev->dev_addr, dest))
			skb->pkt_type = PACKET_HOST;

		NF_HOOK(PF_BRIDGE, NF_BR_PRE_ROUTING, skb, skb->dev, NULL,
			br_handle_frame_finish);
		return 1;
	}

err:
	kfree_skb(skb);
	return 1;
}

#ifdef IGMP_SNOOPING

#ifdef	IPV6_MCAST_TO_UNICAST
/*Convert  MultiCatst IPV6_Addr to MAC_Addr*/
static void CIPV6toMac
	(unsigned char* icmpv6_McastAddr, unsigned char *gmac )
{
	/*ICMPv6 valid addr 2^32 -1*/
	gmac[0] = 0x33;
	gmac[1] = 0x33;
	gmac[2] = icmpv6_McastAddr[12];
	gmac[3] = icmpv6_McastAddr[13];
	gmac[4] = icmpv6_McastAddr[14];
	gmac[5] = icmpv6_McastAddr[15];			
}



static int ICMPv6_check(struct sk_buff *skb , unsigned char *gmac)
{
	
	struct ipv6hdr *ipv6h;
	char* protoType;	
	
	/* check IPv6 header information */
	ipv6h = skb->nh.ipv6h;
	if(ipv6h->version != 6){	
		//printk("ipv6h->version != 6\n");
		return -1;
	}


	/*Next header: IPv6 hop-by-hop option (0x00)*/
	if(ipv6h->nexthdr == 0)	{
		protoType = (unsigned char*)( (unsigned char*)ipv6h + sizeof(struct ipv6hdr) );	
	}else{
		//printk("ipv6h->nexthdr != 0\n");
		return -1;
	}

	if(protoType[0] == 0x3a){
		
		//printk("recv icmpv6 packet\n");
		struct icmp6hdr* icmpv6h = (struct icmp6hdr*)(protoType + 8);
		unsigned char* icmpv6_McastAddr ;
	
		if(icmpv6h->icmp6_type == 0x83){
			
			icmpv6_McastAddr = (unsigned char*)((unsigned char*)icmpv6h + 8);
			#ifdef	DBG_ICMPv6					
			printk("Type: 0x%x (Multicast listener report) \n",icmpv6h->icmp6_type);
			#endif

		}else if(icmpv6h->icmp6_type == 0x8f){		
		
			icmpv6_McastAddr = (unsigned char*)((unsigned char*)icmpv6h + 8 + 4);
			#ifdef	DBG_ICMPv6					
			printk("Type: 0x%x (Multicast listener report v2) \n",icmpv6h->icmp6_type);
			#endif			
		}else if(icmpv6h->icmp6_type == 0x84){
		
			icmpv6_McastAddr = (unsigned char*)((unsigned char*)icmpv6h + 8 );			
			#ifdef	DBG_ICMPv6					
			printk("Type: 0x%x (Multicast listener done ) \n",icmpv6h->icmp6_type);
			#endif			
		}
		else{
			#ifdef	DBG_ICMPv6
			printk("Type: 0x%x (unknow type)\n",icmpv6h->icmp6_type);
			#endif			
			return -1;
		}				

		#ifdef	DBG_ICMPv6			
		printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
			icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
			icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
			icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
			icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
		#endif

		CIPV6toMac(icmpv6_McastAddr, gmac);
		
		#ifdef	DBG_ICMPv6					
		printk("group_mac [%02x:%02x:%02x:%02x:%02x:%02x] \n",
			gmac[0],gmac[1],gmac[2],
			gmac[3],gmac[4],gmac[5]);
		#endif
			


		if(icmpv6h->icmp6_type == 0x83){

			return 1;//icmpv6 listener report (add)
		}
		else if(icmpv6h->icmp6_type == 0x8f){
			return 1;//icmpv6 listener report v2 (add) 
		}
		else if(icmpv6h->icmp6_type == 0x84){
			return 2;//icmpv6 Multicast listener done (del)
		}
	}		
	else{
		//printk("protoType[0] != 0x3a\n");		
		return -1;//not icmpv6 type
	}
		

}

#endif	//end of IPV6_MCAST_TO_UNICAST

/*2008-01-15,for porting igmp snooping to linux kernel 2.6*/
static void ConvertMulticatIPtoMacAddr(__u32 group, unsigned char *gmac)
{
	__u32 u32tmp, tmp;
	int i;

	u32tmp = group & 0x007FFFFF;
	gmac[0]=0x01; gmac[1]=0x00; gmac[2]=0x5e;
	for (i=5; i>=3; i--) {
		tmp=u32tmp&0xFF;
		gmac[i]=tmp;
		u32tmp >>= 8;
	}
}
static char igmp_type_check(struct sk_buff *skb, unsigned char *gmac)
{
        struct iphdr *iph;
	__u8 hdrlen;
	struct igmphdr *igmph;

#ifdef	MCAST_TO_UNICAST	
	unsigned int IGMP_Group;// add  for fit igmp v3
#endif
	
	/* check IP header information */
	iph = skb->nh.iph;
	hdrlen = iph->ihl << 2;
	if ((iph->version != 4) &&  (hdrlen < 20))
		return -1;
	if (ip_fast_csum((u8 *)iph, iph->ihl) != 0)
		return -1;
	{ /* check the length */
	__u32 len = ntohs(iph->tot_len);
	if (skb->len < len || len < hdrlen)
		return -1; 
	}
	/* parsing the igmp packet */
	igmph = (struct igmphdr *)((u8*)iph+hdrlen);

#ifdef	MCAST_TO_UNICAST
	/*IGMP-V3 type Report*/
	if(igmph->type == IGMPV3_HOST_MEMBERSHIP_REPORT)
	{
		//printk("rec v3 report 1\n");
		/*in 11n seem need no care igmpProxy is opened or not,plus 2008-0612*/
		#if	0
		if(IGMPProxyOpened==0){
			IGMP_Group = *(unsigned int*)((unsigned int*)igmph + 3);
			
			//printk("v3_group:%02X:%02X:%02X:%02X\n",
			//IGMP_Group>>24,(IGMP_Group<<8)>>24,(IGMP_Group<<16)>>24,(IGMP_Group<<24)>>24);
		}else{
			return -1;//don't care v3 report
		}
		#else
			IGMP_Group = *(unsigned int*)((unsigned int*)igmph + 3);
		#endif
		
	}else{	//4 V2 or V1
		//printk("igmph->group:%04X\n",igmph->group);	
		IGMP_Group = igmph->group;
	}
#endif	
#ifdef	MCAST_TO_UNICAST

	/*check if it's protocol reserved group */
	if(!IN_MULTICAST(IGMP_Group))
	{			
			return -1;
	}
	//Brad disable 20080619
	/*
	if((IGMP_Group&0xFFFFFF00)==0xE0000000){			
			return -1;
	}
	*/
	ConvertMulticatIPtoMacAddr(IGMP_Group, gmac);
	
#else
	if(!IN_MULTICAST(igmph->group))
			return -1;
	/* check if it's protocol reserved group */
	if((igmph->group&0xFFFFFF00)==0xE0000000)
			return -1;
				
	ConvertMulticatIPtoMacAddr(igmph->group, gmac);
#endif

	#if 1
	if (
#ifdef	MCAST_TO_UNICAST		
		(igmph->type==IGMPV3_HOST_MEMBERSHIP_REPORT)||	//for support igmp v3
#endif	
            (igmph->type==IGMP_HOST_MEMBERSHIP_REPORT) ||
	    (igmph->type==IGMPV2_HOST_MEMBERSHIP_REPORT)) {
		return 1; /* report and add it */
	}else if (igmph->type==IGMP_HOST_LEAVE_MESSAGE){
		return 2; /* leave and delete it */
	}
	#else
	if ((igmph->type==IGMP_HOST_MEMBERSHIP_REPORT) ||
	    (igmph->type==IGMP_HOST_NEW_MEMBERSHIP_REPORT)) {
		return 1; /* report and add it */
	}else if (igmph->type==IGMP_HOST_LEAVE_MESSAGE){
		return 2; /* leave and delete it */
	}
	#endif
	
	
	return -1;
}


static void br_update_igmp_snoop_fdb(unsigned char op, struct net_bridge *br, struct net_bridge_port *p, unsigned char *dest 
#ifdef	MCAST_TO_UNICAST
, struct sk_buff *skb
#endif
)
{
#ifdef	MCAST_TO_UNICAST
	if((!MULTICAST_MAC(dest)
#ifdef	IPV6_MCAST_TO_UNICAST
		&& !IPV6_MULTICAST_MAC(dest)
#endif	
        ) || dest==NULL){		
		return;
	}
#endif
	struct net_bridge_fdb_entry *dst;
//	unsigned char i=0;
	unsigned char exist=0;
	unsigned short del_group_src=0;
	/* check whether entry exist */
	dst = br_fdb_get(br, dest);
	if (dst != NULL) {
		if((dst->group_src & (1<< p->port_no))== (1 << p->port_no)){
				exist = 1;
		}		
	
	}
	if (op == 1) /* add */
	{	

#ifdef	MCAST_TO_UNICAST
		struct net_device *dev = __dev_get_by_name("wlan0");	
		if (dev) {			
			unsigned char StaMacAndGroup[20];
			memcpy(StaMacAndGroup, dest, 6);
			memcpy(StaMacAndGroup+6, skb->mac.raw+ETH_ALEN, 6);				
			if (dev->do_ioctl != NULL)
				dev->do_ioctl(dev, (struct ifreq*)StaMacAndGroup, 0x8B80);				
		}
#endif

		if (dst) {
			if (exist == 0)	{
	        	        dst->group_src = dst->group_src | (1 << p->port_no);
			}
			// update the timer to prevent from timeout
			dst->ageing_timer = jiffies;
		}
		else {
			/* insert one fdb entry */
			#if 1
			br_fdb_insert(br, p, dest);
			#else
			br_fdb_insert(br, p, dest, 0);
			#endif
			dst = br_fdb_get(br, dest);
			if(dst !=NULL){
				dst->group_src = dst->group_src | (1 << p->port_no);
			}
		}
	}
	else if (op == 2) /* delete */
	{
		if (dst && exist){
			del_group_src = ~(1 << p->port_no);
			dst->group_src = dst->group_src & del_group_src;

#ifdef	MCAST_TO_UNICAST
			struct net_device *dev = __dev_get_by_name("wlan0");	
			if (dev) {			
				unsigned char StaMacAndGroup[12];
				memcpy(StaMacAndGroup, dest , 6);
				memcpy(StaMacAndGroup+6, skb->mac.raw+ETH_ALEN, 6);
				if (dev->do_ioctl != NULL) 
					dev->do_ioctl(dev, (struct ifreq*)StaMacAndGroup, 0x8B81);							
			}
#endif	
			
		}
	}
}

#endif // IGMP_SNOOPING

