/*
 *	Device handling code
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	$Id: br_device.c,v 1.1.1.1 2009/02/17 09:44:45 anderson Exp $
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ethtool.h>

#include <asm/uaccess.h>
#include "br_private.h"

#ifdef IGMP_SNOOPING
/*2008-01-15,for porting igmp snooping to linux kernel 2.6*/
#include <linux/ip.h>
#include <linux/in.h>
#include <linux/igmp.h>
#include <net/checksum.h>
#ifdef CONFIG_BRIDGE_IGMPV3_SNOOPING
#include "../../drivers/net/re865x/igmpsnooping/rtl865x_igmpsnooping_glue.h"
#include "../../drivers/net/re865x/igmpsnooping/rtl865x_igmpsnooping.h"
#ifdef CONFIG_RTL865X_HARDWARE_MULTICAST
#include "../../drivers/net/re865x/l3Driver/rtl865x_multicast.h"
#endif
extern unsigned int brIgmpModuleIndex;
#endif
#endif

static struct net_device_stats *br_dev_get_stats(struct net_device *dev)
{
	struct net_bridge *br = netdev_priv(dev);
	return &br->statistics;
}

/* net device transmit always called with no BH (preempt_disabled) */
int br_dev_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct net_bridge *br = netdev_priv(dev);
	const unsigned char *dest = skb->data;
	struct net_bridge_fdb_entry *dst;
#ifdef IGMP_SNOOPING	
/*2008-01-15,for porting igmp snooping to linux kernel 2.6*/
	struct iphdr *iph;
	unsigned int ipaddr=0;
	unsigned char proto=0;
	unsigned char reserved=0;
	extern int igmpsnoopenabled;
	
	#ifdef CONFIG_BRIDGE_IGMPV3_SNOOPING
	struct rtl_multicastDataInfo multicastDataInfo;
	#endif
	
	iph = skb->nh.iph;
	ipaddr =  iph->daddr;
	proto =  iph->protocol;
	 if((ipaddr&0xFFFFFF00)==0xE0000000)
	        reserved=1;
	 
	#ifdef CONFIG_BRIDGE_IGMPV3_SNOOPING
	multicastDataInfo.ipVersion=4;
	multicastDataInfo.sourceIp[0]=  (uint32)(iph->saddr);
	multicastDataInfo.groupAddr[0]=  (uint32)(iph->daddr);
	#ifdef CONFIG_BRIDGE_IGMPV3_SNOOPING_DEBUG
	printk("source ip is 0x%x, dest ip is 0x%x\n",multicastDataInfo.sourceIp[0],multicastDataInfo.groupAddr[0]);
	#endif
	#endif
	
#endif	

	br->statistics.tx_packets++;
	br->statistics.tx_bytes += skb->len;

	skb->mac.raw = skb->data;
	skb_pull(skb, ETH_HLEN);

#ifdef IGMP_SNOOPING
/*2008-01-15,for porting igmp snooping to linux kernel 2.6*/
	if (dest[0] & 1)
	{
		if(igmpsnoopenabled && MULTICAST_MAC(dest) && proto !=IPPROTO_IGMP && ipaddr != 0xEFFFFFFA && reserved ==0) 
		{	
			#ifdef CONFIG_BRIDGE_IGMPV3_SNOOPING
			int retVal=FAILED;
			unsigned int brFwdPortMask=0;
			#ifdef CONFIG_RTL865X_HARDWARE_MULTICAST
			unsigned int srcPort=skb->srcPort;
			unsigned int srcVlanId=skb->srcVlanId;
			#endif
			retVal= rtl_getMulticastDataFwdPortMask(brIgmpModuleIndex, &multicastDataInfo, &brFwdPortMask);
			if(retVal==SUCCESS)
			{
				#ifdef CONFIG_RTL865X_HARDWARE_MULTICAST
				//printk("%s:%d,skb->srcPort  is %d,skb->srcVlanId is %d,skb->dev->name is %s,skb->input_dev->name is %s\n",__FUNCTION__,__LINE__,skb->srcPort,skb->srcVlanId,skb->dev->name,skb->input_dev->name);
				#endif
			
				br_multicast_deliver(br, brFwdPortMask, skb, 0);

				#ifdef CONFIG_RTL865X_HARDWARE_MULTICAST
				if((srcVlanId!=0) && (srcPort!=0xFFFF))
				{
					rtl865x_ipMulticastHardwareAccelerate(br, brFwdPortMask,srcPort,srcVlanId, multicastDataInfo.sourceIp[0], multicastDataInfo.groupAddr[0]);
				}
				#endif		
			}
			else
			{
				br_flood_deliver(br, skb, 0);
			}
			#else
			if ((dst = br_fdb_get(br, dest)) != NULL)
			{
				br_multicast_deliver(br, dst, skb, 0);
				br_fdb_put(dst);
			}
			else
			{
				br_flood_deliver(br, skb, 0);
			}
			#endif
		}
		else
		{ 
			// broadcast
			br_flood_deliver(br, skb, 0);
		}	
	
	}
	else if ((dst = __br_fdb_get(br, dest)) != NULL)
	{
		br_deliver(dst->dst, skb);
	}
	else
	{
		br_flood_deliver(br, skb, 0);
	}
#else
	if (dest[0] & 1) 
		br_flood_deliver(br, skb, 0);
	else if ((dst = __br_fdb_get(br, dest)) != NULL)
		br_deliver(dst->dst, skb);
	else
		br_flood_deliver(br, skb, 0);
#endif

	return 0;
}

static int br_dev_open(struct net_device *dev)
{
	struct net_bridge *br = netdev_priv(dev);

	br_features_recompute(br);
	netif_start_queue(dev);
	br_stp_enable_bridge(br);

	return 0;
}

static void br_dev_set_multicast_list(struct net_device *dev)
{
}

static int br_dev_stop(struct net_device *dev)
{
	br_stp_disable_bridge(netdev_priv(dev));

	netif_stop_queue(dev);

	return 0;
}

static int br_change_mtu(struct net_device *dev, int new_mtu)
{
	if (new_mtu < 68 || new_mtu > br_min_mtu(netdev_priv(dev)))
		return -EINVAL;

	dev->mtu = new_mtu;
	return 0;
}

/* Allow setting mac address of pseudo-bridge to be same as
 * any of the bound interfaces
 */
static int br_set_mac_address(struct net_device *dev, void *p)
{
	struct net_bridge *br = netdev_priv(dev);
	struct sockaddr *addr = p;
	struct net_bridge_port *port;
	int err = -EADDRNOTAVAIL;

	spin_lock_bh(&br->lock);
	list_for_each_entry(port, &br->port_list, list) {
		if (!compare_ether_addr(port->dev->dev_addr, addr->sa_data)) {
			br_stp_change_bridge_id(br, addr->sa_data);
			err = 0;
			break;
		}
	}
	spin_unlock_bh(&br->lock);

	return err;
}

static void br_getinfo(struct net_device *dev, struct ethtool_drvinfo *info)
{
	strcpy(info->driver, "bridge");
	strcpy(info->version, BR_VERSION);
	strcpy(info->fw_version, "N/A");
	strcpy(info->bus_info, "N/A");
}

static int br_set_sg(struct net_device *dev, u32 data)
{
	struct net_bridge *br = netdev_priv(dev);

	if (data)
		br->feature_mask |= NETIF_F_SG;
	else
		br->feature_mask &= ~NETIF_F_SG;

	br_features_recompute(br);
	return 0;
}

static int br_set_tso(struct net_device *dev, u32 data)
{
	struct net_bridge *br = netdev_priv(dev);

	if (data)
		br->feature_mask |= NETIF_F_TSO;
	else
		br->feature_mask &= ~NETIF_F_TSO;

	br_features_recompute(br);
	return 0;
}

static int br_set_tx_csum(struct net_device *dev, u32 data)
{
	struct net_bridge *br = netdev_priv(dev);

	if (data)
		br->feature_mask |= NETIF_F_NO_CSUM;
	else
		br->feature_mask &= ~NETIF_F_ALL_CSUM;

	br_features_recompute(br);
	return 0;
}

static struct ethtool_ops br_ethtool_ops = {
	.get_drvinfo = br_getinfo,
	.get_link = ethtool_op_get_link,
	.get_sg = ethtool_op_get_sg,
	.set_sg = br_set_sg,
	.get_tx_csum = ethtool_op_get_tx_csum,
	.set_tx_csum = br_set_tx_csum,
	.get_tso = ethtool_op_get_tso,
	.set_tso = br_set_tso,
};

void br_dev_setup(struct net_device *dev)
{
	memset(dev->dev_addr, 0, ETH_ALEN);

	ether_setup(dev);

	dev->do_ioctl = br_dev_ioctl;
	dev->get_stats = br_dev_get_stats;
	dev->hard_start_xmit = br_dev_xmit;
	dev->open = br_dev_open;
	dev->set_multicast_list = br_dev_set_multicast_list;
	dev->change_mtu = br_change_mtu;
	dev->destructor = free_netdev;
	SET_MODULE_OWNER(dev);
 	SET_ETHTOOL_OPS(dev, &br_ethtool_ops);
	dev->stop = br_dev_stop;
	dev->tx_queue_len = 0;
	dev->set_mac_address = br_set_mac_address;
	dev->priv_flags = IFF_EBRIDGE;

 	dev->features = NETIF_F_SG | NETIF_F_FRAGLIST | NETIF_F_HIGHDMA |
 			NETIF_F_TSO | NETIF_F_NO_CSUM | NETIF_F_GSO_ROBUST;
}


#ifdef CONFIG_RTL865X_HARDWARE_MULTICAST
int rtl865x_ipMulticastHardwareAccelerate(struct net_bridge *br, unsigned int brFwdPortMask,
												unsigned int srcPort,unsigned int srcVlanId, 
												unsigned int srcIpAddr, unsigned int destIpAddr)
{
	int ret;
	int fwdDescCnt;
	struct net_bridge_port *prev;
	struct net_bridge_port *p, *n;
	unsigned short port_bitmask=0;

	rtl_multicastDeviceInfo_t devInfo;
	unsigned int	devModuleIndex;
	struct rtl_multicastDataInfo multicastDataInfo;
	
	rtl865x_tblDrv_mCast_t * existMulticastEntry;
	rtl865x_mcast_fwd_descriptor_t  fwdDescriptor;
	
	//printk("%s:%d,srcPort is %d,srcVlanId is %d,srcIpAddr is 0x%x,destIpAddr is 0x%x\n",__FUNCTION__,__LINE__,srcPort,srcVlanId,srcIpAddr,destIpAddr);

	if(strcmp(br->dev->name,"br0")!=0)
	{
		return -1;
	}
	
	/*first scan forwarding devices, to check wireless lan device's existance */
	list_for_each_entry_safe(p, n, &br->port_list, list) 
	{
		port_bitmask = (1 << p->port_no);
		if (port_bitmask & brFwdPortMask) 
		{
			if(	(p->dev->name[0]=='w')&&
				(p->dev->name[1]=='l')&&
				(p->dev->name[2]=='a')&&
				(p->dev->name[3]=='n'))
				
			{
				return 0;
			}
                                                                                                                       
               }
		
	}

	existMulticastEntry=rtl865x_findMCastEntry(destIpAddr, srcIpAddr, (unsigned short)srcVlanId, (unsigned short)srcPort);
	if(existMulticastEntry!=NULL)
	{
		/*it's already in asic */
		if(existMulticastEntry->inAsic)
		{
			return 0;
		}
	}

	multicastDataInfo.ipVersion=4;
	multicastDataInfo.sourceIp[0]=  srcIpAddr;
	multicastDataInfo.groupAddr[0]=  destIpAddr;

	/*add hardware multicast entry*/
	fwdDescCnt=0;
	prev=NULL;
	list_for_each_entry_safe(p, n, &br->port_list, list) 
	{
		port_bitmask = (1 << p->port_no);
		if (port_bitmask & brFwdPortMask)  	 
		{
			if(p->state == BR_STATE_FORWARDING)
			{
				if (prev != NULL)
				{
					/*search control plane*/
					strcpy(devInfo.devName,p->dev->name);
					ret=rtl865x_getDeviceIgmpSnoopingModuleIndex(&devInfo, &devModuleIndex);
					if(ret!=0)
					{
						return -1;
					}
					memset(&fwdDescriptor, 0, sizeof(rtl865x_mcast_fwd_descriptor_t ));

					
					ret= rtl_getMulticastDataFwdPortMask(devModuleIndex, &multicastDataInfo, &fwdDescriptor.fwdPortMask);
					if(ret!=0)
					{
						return -1;
					}
					/*add multicast data flow into asic*/
					strcpy(fwdDescriptor.netifName,p->dev->name);

					/*to flush old descriptor chain when add first one*/
					if(fwdDescCnt==0)
					{
						rtl865x_addMulticastEntry(destIpAddr, srcIpAddr, (unsigned short)srcVlanId, (unsigned short)srcPort, 
						&fwdDescriptor, 1, 0, 1, 0);
					}
					else
					{	
						/*incremental adding*/
						ret=rtl865x_addMulticastEntry(destIpAddr, srcIpAddr, (unsigned short)srcVlanId, (unsigned short)srcPort, 
						&fwdDescriptor, 0, 0, 1, 0);
					}
					
					fwdDescCnt++;
					
				}
				
				prev = p;
			}
                                                                                                                      
               }
	}
	
	//printk("%s:%d,fwdDescCnt is %d\n",__FUNCTION__,__LINE__,fwdDescCnt);
	if (prev != NULL) 
	{
		/*search control plane*/
		strcpy(devInfo.devName,prev->dev->name);
		ret=rtl865x_getDeviceIgmpSnoopingModuleIndex(&devInfo, &devModuleIndex);
		if(ret!=0)
		{
			return -1;
		}
		memset(&fwdDescriptor, 0, sizeof(rtl865x_mcast_fwd_descriptor_t ));

		
		ret= rtl_getMulticastDataFwdPortMask(devModuleIndex, &multicastDataInfo, &fwdDescriptor.fwdPortMask);
		if(ret!=0)
		{
			return -1;
		}
		/*add multicast data flow into asic*/
		strcpy(fwdDescriptor.netifName,prev->dev->name);

		/*the first one and only one forwarding device exist*/
		if(fwdDescCnt==0)
		{
			rtl865x_addMulticastEntry(destIpAddr, srcIpAddr, (unsigned short)srcVlanId, (unsigned short)srcPort, 
						&fwdDescriptor, 1, 0, 0, 0);
		}
		else
		{
			/*the last one, incremental adding */
			rtl865x_addMulticastEntry(destIpAddr, srcIpAddr, (unsigned short)srcVlanId, (unsigned short)srcPort, 
						&fwdDescriptor, 0, 0, 0, 0);
		}
	}
	return 0;
}
#endif



