
#include "rtl_types.h"
#include "mbuf.h"
#include "assert.h"

#include "rtl865xc_swNic.h"
#include "common/types.h"
#include "common/rtl8651_hwPatch.h"		/* define for chip related spec */
#include <common/rtl8651_aclLocal.h>
#include "rtl865xC_tblAsicDrv.h"

#include "rtl_queue.h"
#include "rtl865x_common.h"
#include "rtl865x_common_local.h"
#include "rtl865x_layer2.h"
#include "rtl865x_layer2_local.h"
#include "rtl_types.h"
#include "rtl865x_layer3.h"
#include "rtl865x_layer3_local.h"
#include "../common/rtl8651_debug.h" /*debug msg*/
#include <../net/ipv4/fastpath/fastpath_core.h>
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
#include "common/rtl865x_vlan.h"
#include "common/rtl865x_netif.h"
#include "common/rtl865x_netif_local.h"
#endif

static struct if_entry *lr_pppoe_lookup(uint32, uint32 *);
static int32 lr_pppoe_init(void);

static int32 lr_route_init(void);
static rtl865x_tblAsicDrv_routingParam_t *	lr_route_lookup(ipaddr_t, ipaddr_t, uint32 *);

static int32 lr_arp_init(void);
static int32 lr_arp_hash(ipaddr_t );
static rtl865x_tblAsicDrv_arpParam_t *lr_arp_lookup(ipaddr_t );
static int32 lr_arp_tbl_alloc(struct if_entry *);
static int32 lr_arp_tbl_free(struct if_entry *);
void rt_arp_sync(ipaddr_t ip, uint32 row, uint32 column, enum SYNC_FLAG dowhat);

static int32 lr_natip_init(void);
static rtl865x_tblAsicDrv_extIntIpParam_t * lr_natip_lookup(ipaddr_t, uint32 *);
static int32 lr_natip_add(ipaddr_t);
static int32 lr_natip_del(ipaddr_t);


/*======================================
 *  ARP Table - arpt_tbl
 *======================================*/
struct arp_table arpt_tbl = {
	arp_init:		lr_arp_init,				/* Initialize ARP table */
	arp_add:			rtl865x_addArp,			/* Add an arp entry */
	arp_del:			rtl865x_delArp,			/* Remove an arp entry */
	arp_lookup:		lr_arp_lookup,				/* ARP entry lookup */
	arp_hash:		lr_arp_hash,				/* Get a specified ARP entry position in ASIC */


	arp_tbl_alloc:		lr_arp_tbl_alloc,			/* Allocate ARP space for a network */
	arp_tbl_free:		lr_arp_tbl_free,			/* Release allocated ARP space */


	arp_asic_set:		rtl8651_setAsicArp,		/* ASIC Interface: add asic arp */
	arp_asic_get:		rtl8651_getAsicArp,		/* ASIC Interface: query asic arp */
	arp_asic_del:		rtl8651_delAsicArp,			/* ASIC Interface: remove asic arp */
};

static int32 lr_arp_init(void)
{
	memset(TBLFIELD(arpt_tbl, arp_mask), 0, 64);
	return 0;
}



static rtl865x_tblAsicDrv_arpParam_t *lr_arp_lookup(ipaddr_t ip)
{
	rtl865x_tblAsicDrv_arpParam_t *arpe;
	int32 hash;
	
	if ((hash=TBLFIELD(arpt_tbl, arp_hash)(ip)) == -1)
		return (rtl865x_tblAsicDrv_arpParam_t*)0;
	arpe = &TBLFIELD(arpt_tbl, __arpbuff);
	if (TBLFIELD(arpt_tbl, arp_asic_get)(hash, arpe))
		return (rtl865x_tblAsicDrv_arpParam_t*)0;
	return &TBLFIELD(arpt_tbl, __arpbuff);
}

static int32 lr_arp_hash(ipaddr_t ip)
{
	struct if_entry *ife;

	ife = TBLFIELD(if_tbl, if_lookup)(NULL, ip);
	if (!ife)
		return -1;
	return ((ife->arp_start_<<3)+(ip&~ife->mask_));
}

static int32 lr_arp_tbl_alloc(struct if_entry *ife)
{
#if 1
	uint32 i;

	if (ife->isWan) {
		ife->arp_start_ = 32;
		ife->arp_end_  = 63;
	}
	else {
		ife->arp_start_ = 0;
		ife->arp_end_  = 31;
	}
	for(i=ife->arp_start_; i<=ife->arp_end_; i++) 
		TBLFIELD(arpt_tbl, arp_mask)[i] = ife-TBLFIELD(if_tbl, if_hash) + 1;
	return 0;
#else
	uint32 netSize, entry, bestSize=0, bestStartPos=0xffffffff;
	uint32 curSize, curStartPos;

	for(entry=0; entry<32; entry++)
		if(ife->mask_ & (1<<entry))	
			break;
	if ((netSize = (1<<entry)) > 1) {
		curStartPos = bestSize = curSize = 0;
		for(entry = 0; entry <= 64; entry++) {
			if(entry == 64 || TBLFIELD(arpt_tbl, arp_mask)[entry]) {
				if(curSize > bestSize) {
					bestStartPos = curStartPos;
					bestSize = curSize;
				}
				curStartPos = entry+1;
				curSize = 0;
			} else curSize++;
		}
	} 
	if (netSize>1 && (bestSize<<3) >= netSize) {
		ife->arp_start_ = bestStartPos;
		ife->arp_end_	  = bestStartPos + (netSize>>3) - ((netSize&0x7)==0? 1: 0);
		for(entry=ife->arp_start_; entry<=ife->arp_end_; entry++)
			TBLFIELD(arpt_tbl, arp_mask)[entry] = ife-TBLFIELD(if_tbl, if_hash) + 1;
		return 0;
	}
	return -1;
#endif	
}


static int32 lr_arp_tbl_free(struct if_entry *ife)
{
	rtl865x_tblAsicDrv_arpParam_t arpe;
	uint32 index;
	int32 i, j, rc;

	if (!IF_UP(ife))
		return -1;
	for(i=ife->arp_start_; i<=ife->arp_end_; i++) {
		TBLFIELD(arpt_tbl, arp_mask)[i] = 0;
		for(j=0; j<8; j++) {
			index = (i<<3)+j;
			rc = TBLFIELD(arpt_tbl, arp_asic_get)(index, &arpe);
			if (!rc) {
				TBLFIELD(arpt_tbl, arp_asic_del)(index);
				TBLFIELD(fdb_tbl, fdb_asic_del)(arpe.nextHopRow, arpe.nextHopColumn);
			}
		}
	}
	return 0;
}


/*======================================
 *  PPPoE Table - pppoe_tbl
 *======================================*/
struct pppoe_table pppoe_tbl = {
	pppoe_init:		lr_pppoe_init,				/* Initialize PPPoE table */
	pppoe_lookup:	lr_pppoe_lookup,			/* PPPoE session lookup */

	pppoe_asic_set:	rtl8651_setAsicPppoe,		/* ASIC Interface: configure asic pppoe session */
	pppoe_asic_get:	rtl8651_getAsicPppoe,		/* ASIC Interface: query asic pppoe session */
};

static int32 lr_pppoe_init(void)
{
	memset(TBLFIELD(pppoe_tbl, ppps), 0, sizeof(struct if_entry *)*RTL8651_PPPOETBL_SIZE);
	return 0;
}


static struct if_entry *lr_pppoe_lookup(uint32 sid, uint32 *index)
{
	struct if_entry *psif;
	uint32 entry;

	for(entry=0; entry<RTL8651_PPPOETBL_SIZE; entry++) {
		psif = TBLFIELD(pppoe_tbl, ppps)[entry];
		if (!psif)
			continue;
		if (psif->sid_ == sid) {
			if (index) *index=entry;
			return psif;
		}
	}
	return (struct if_entry *)0;
}



/*======================================
 *  Routing Table - rt_tbl
 *======================================*/
struct rt_table rt_tbl = {
	route_init:		lr_route_init,				/* Initialize routing table */
	route_add:		rtl865x_addRoute,			/* Add a routing entry */
	route_del:		rtl865x_delRoute,			/* Remove a routing entry */
	route_lookup:		lr_route_lookup,			/* Routing entry lookup, not perform longest-prefix match */

	rt_asic_set:		rtl8651_setAsicRouting,		/* ASIC Interface: add asic route */
	rt_asic_get:		rtl8651_getAsicRouting,		/* ASIC Interface: query asic route */
	rt_asic_del:		rtl8651_delAsicRouting,		/* ASIC Interface: remove asic route */
};

static int32 lr_route_init(void)
{
	memset(TBLFIELD(rt_tbl, pendgw), 0, sizeof(ipaddr_t)*RTL8651_ROUTINGTBL_SIZE);
	memset(&TBLFIELD(rt_tbl, __l3buff), 0, sizeof(rtl865x_tblAsicDrv_routingParam_t));
	
	/* Set 7th route entry to toCPU if no default route is present */
	TBLFIELD(rt_tbl, __l3buff).process = 0x04; /* trap toCPU */
	TBLFIELD(rt_tbl, rt_asic_set)(RTL8651_ROUTINGTBL_SIZE-1, &TBLFIELD(rt_tbl, __l3buff));
	return 0;
}


/*
*	dst/mask		gw		dev		Routing type
*	=======		==		===		========================================
*		0		 0		 0		(X) Invalid route
*		0		 0		 1		PPP type default route 
*		0		 1		 0		ether type default route
*		0		 1		 1		ppp/ether type default route, depends on dev type
*		1		 0		 0		(X) Invalid route
*		1		 0 		 1		ppp or interface route, depends on dev type
*		1		 1		 0		ether type route
*		1		 1		 1		ppp/ether type route, depends on dev type
*
*/

enum RTL_RESULT rtl865x_addRoute( ipaddr_t ip, ipaddr_t mask, ipaddr_t gateway, uint8* ifname, enum RTL_ROUTE_FLAGS flags )
{
	rtl865x_tblAsicDrv_routingParam_t rth, rth0;
	rtl865x_tblAsicDrv_arpParam_t *arpe;
	rtl865x_tblAsicDrv_l2Param_t *fdbe;
	struct if_entry *ife0, *ife1;
	int32 rtidx=0;
	uint32 way, pindex;
#ifdef CONFIG_RTL865X_HW_PPTPL2TP
	uint32 vid = 0;
#endif

#ifdef CONFIG_RTL865X_LAYERED_DRIVER
	int32 fid = 0;
#endif

	memset(&rth, 0, sizeof(rtl865x_tblAsicDrv_routingParam_t));
	ife0 = TBLFIELD(if_tbl, if_lookup)(ifname, 0);
	ife1 = TBLFIELD(if_tbl, if_lookup)(NULL, gateway);
	if (!ife0 && !ife1)
		return RTL_INVIF;
#ifdef CONFIG_RTL865X_HW_PPTPL2TP
	if (ife0 && ife0->if_type == IF_PPP) {
		if (IF_UP(ife0)) {
			if (ife0->sid_) {
				/* PPPoE */
				ife1 = TBLFIELD(pppoe_tbl, pppoe_lookup)(ife0->sid_, &pindex);
				fdbe = TBLFIELD(fdb_tbl, fdb_lookup)(GET_IF_VID(ife0), &ife0->pmac_, FDB_STATIC, &way);
				if (IF_UP(ife0) && fdbe && ife1) {
					rth.process			= 0x0;
					rth.pppoeIdx		= pindex;
					rth.nextHopColumn	= way;
#ifdef CONFIG_RTL865XC
					rth.nextHopRow = TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife0->pmac_, GET_IF_VID(ife0));
#else
					rth.nextHopRow = TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife0->pmac_);
#endif
					for (rtidx = RTL8651_ROUTINGTBL_SIZE-2; rtidx >= 0; rtidx--)
						if (TBLFIELD(rt_tbl, rt_asic_get)(rtidx, &rth0))
							break;
					if (rtidx < 0)
						return LR_NOBUFFER;
					else {
						vid = 8; /* hack to eth1 */
						goto add_rt;
					}
				}
				return RTL_RTUNREACH;			
			}
			else {
				/* PPTP/L2TP */
				fdbe = TBLFIELD(fdb_tbl, fdb_lookup)(GET_IF_VID(ife0), &ife0->pmac_, FDB_STATIC, &way);
				rth.process = 0x1;
				rth.nextHopColumn = way;
#ifdef CONFIG_RTL865XC
				rth.nextHopRow = TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife0->pmac_, GET_IF_VID(ife0));
#else
				rth.nextHopRow = TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife0->pmac_);
#endif
				for (rtidx = RTL8651_ROUTINGTBL_SIZE-2; rtidx >= 0; rtidx--)
					if (TBLFIELD(rt_tbl, rt_asic_get)(rtidx, &rth0))
						break;
				if (rtidx < 0)
					return RTL_NOBUFFER;
				else 
					goto add_rt;
			}
		}
		else
			return RTL_IFDOWN;
	}
#else	
	if (ife0 && ife0->if_type == IF_PPPOE) {
		ife1 = TBLFIELD(pppoe_tbl, pppoe_lookup)(ife0->sid_, &pindex);
		//fdbe = TBLFIELD(fdb_tbl, fdb_lookup)(ife0->fid, &ife0->pmac_, FDB_STATIC, &way);
		
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
		{
			rtl865x_vlan_entry_t *vlan;
			vlan = _rtl8651_getVlanTableEntry(ife0->vid);
			if(vlan == NULL)
				printk("%s(%d): can't get vlan by vid(%d)....\n",__FUNCTION__,__LINE__,ife0->vid);
			
			fid = vlan->fid;
		}
		fdbe = TBLFIELD(fdb_tbl, fdb_lookup)(fid, &ife0->pmac_, FDB_STATIC, &way);
#else
		fdbe = TBLFIELD(fdb_tbl, fdb_lookup)(GET_IF_VID(ife0), &ife0->pmac_, FDB_STATIC, &way);
#endif
		if (IF_UP(ife0) && fdbe && ife1) {
			rth.process		= 0x0; /*0x0:means pppoe*/
			rth.pppoeIdx		= pindex;
			rth.nextHopColumn	= way;
#ifdef CONFIG_RTL865XC

#ifdef CONFIG_RTL865X_LAYERED_DRIVER
			rth.nextHopRow	= TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife0->pmac_, fid);
#else
			rth.nextHopRow	= TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife0->pmac_, GET_IF_VID(ife0));
#endif
#else
			rth.nextHopRow	= TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife0->pmac_);
#endif
			for(rtidx=RTL8651_ROUTINGTBL_SIZE-2; rtidx>=0; rtidx--)
				if (TBLFIELD(rt_tbl, rt_asic_get)(rtidx, &rth0))
					break;
			if (rtidx<0)
				return RTL_NOBUFFER;
			else goto add_rt;
		}
		return RTL_RTUNREACH;
	}
#endif
	if ((ife0&&ife0->if_type==IF_ETHER) || (ife1&&ife1->if_type==IF_ETHER)) {
		if (ife0 && ife1 && ife0!=ife1)
			return RTL_RTUNREACH;
		
		if (ife0 && !gateway) { 
			if (!(ip&mask) || TBLFIELD(if_tbl, if_up)(ifname, ip, mask, 0, NULL))
				return RTL_INVAPARAM;
			rth.process	= 0x2; /* interface route */
			rth.arpStart	= ife0->arp_start_;
			rth.arpEnd	= ife0->arp_end_;
			for(rtidx=0; rtidx<RTL8651_ROUTINGTBL_SIZE-1; rtidx++)
				if (TBLFIELD(rt_tbl, rt_asic_get)(rtidx, &rth0)) 
					break;
			if (rtidx==RTL8651_ROUTINGTBL_SIZE-1)
				return RTL_NOBUFFER;
			else goto add_rt;
		}
		ife0 = (!ife0)? ife1: ife0;
		arpe = TBLFIELD(arpt_tbl, arp_lookup)(gateway);
		if (arpe) {
			rth.process		= 0x1; /* direct route */
			rth.nextHopRow	= arpe->nextHopRow;
			rth.nextHopColumn	= arpe->nextHopColumn;
		} else rth.process = 0x4; /* CPU */
		for(rtidx=RTL8651_ROUTINGTBL_SIZE-2; rtidx>=0; rtidx--)
			if (TBLFIELD(rt_tbl, rt_asic_get)(rtidx, &rth0))
				break;
		if (rtidx < 0) return RTL_NOBUFFER;
	}
		
add_rt: /* set ASIC RT entry */
	rtidx = (ip&mask)?  rtidx: RTL8651_ROUTINGTBL_SIZE-1;	
	rth.ipAddr		= ip&mask;
	rth.ipMask		= mask;
#ifdef CONFIG_RTL865XC

#ifdef CONFIG_RTL865X_HW_PPTPL2TP
	rth.vidx			= vid ? rtl8651_vlanTableIndex(vid) : rtl8651_vlanTableIndex(ife0->vid);
#else
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
	rth.vidx			= _rtl865x_getNetifIdxByVid(ife0->vid);
#else
	rth.vidx			= rtl8651_vlanTableIndex(ife0->vid);
#endif
#endif

	rth.internal            = ife0->isWan? 0:1;
#if 0
/*2007-12-19*/
	rtlglue_printf("%s:%d, ife0->vid is %d,rth.process is %d\n",__FUNCTION__,__LINE__, ife0->vid,rth.process);
	rtlglue_printf("rtidx is %d,rth.ipAddr is 0x%x,rth.ipMask is 0x%x,rth.vidx=%d,rth.internal=%d,rth.pppoeIdx=%d\n",rtidx,rth.ipAddr,rth.ipMask,rth.vidx,rth.internal,rth.pppoeIdx);
#endif	
#else
	rth.vidx			= ife0->vid & 0x07;
#endif
	TBLFIELD(rt_tbl, rt_asic_set)(rtidx, &rth);
	TBLFIELD(rt_tbl, pendgw)[rtidx] = (rth.process=0x04)? gateway: 0;	
	return RTL_SUCCESS;
}



enum RTL_RESULT rtl865x_delRoute( ipaddr_t ip, ipaddr_t mask )
{
	rtl865x_tblAsicDrv_routingParam_t *rth;
	struct if_entry *ife;
	uint32 rtidx;

	if (!(rth=TBLFIELD(rt_tbl, route_lookup)(ip, mask, &rtidx)))
		return RTL_NOTFOUND;
	if (rth->process == 2) { /* Interface route */
		ife = TBLFIELD(if_tbl, if_lookup)(NULL, ip);
		assert( ife );
		TBLFIELD(if_tbl, if_down)(ife->name);
	}
	TBLFIELD(rt_tbl, rt_asic_del)(rtidx);
	TBLFIELD(rt_tbl, pendgw)[rtidx] = 0;
	/* The removed route is default route, add 7th entry to toCPU */
	if (rtidx == RTL8651_ROUTINGTBL_SIZE-1) {
		memset(&TBLFIELD(rt_tbl, __l3buff), 0, 
			sizeof(rtl865x_tblAsicDrv_routingParam_t));
		TBLFIELD(rt_tbl, __l3buff).process = 0x04; /* trap toCPU */
		TBLFIELD(rt_tbl, rt_asic_set)(RTL8651_ROUTINGTBL_SIZE-1, 
			&TBLFIELD(rt_tbl, __l3buff));
	}
	return RTL_SUCCESS;
}

static rtl865x_tblAsicDrv_routingParam_t *	lr_route_lookup(ipaddr_t ip, ipaddr_t mask, uint32 *rthidx) 
{
	int32 rtidx, rc;
	
	for(rtidx=0; rtidx<RTL8651_ROUTINGTBL_SIZE; rtidx++) {
		rc = TBLFIELD(rt_tbl, rt_asic_get)(rtidx, &TBLFIELD(rt_tbl, __l3buff));
		if (rc) continue;
		/* The 7th entry should be taken care of specially */
		if (rtidx == RTL8651_ROUTINGTBL_SIZE-1) {
			if (TBLFIELD(rt_tbl, __l3buff).process == 0x04 &&
			    TBLFIELD(rt_tbl, pendgw)[RTL8651_ROUTINGTBL_SIZE-1] == 0 )
			    continue;
			else	TBLFIELD(rt_tbl, __l3buff).ipMask = 0;
		}
		if ( TBLFIELD(rt_tbl, __l3buff).ipAddr==ip && 
		     	TBLFIELD(rt_tbl, __l3buff).ipMask==mask ) {
		     	assert(rthidx);
			*rthidx = rtidx;
			return &TBLFIELD(rt_tbl, __l3buff);
		}
	}
	return (rtl865x_tblAsicDrv_routingParam_t *)0;
}



/*======================================
 *  External IP Table - extip_tbl
 *======================================*/
struct natip_table natip_tbl = {
	natip_init:		lr_natip_init,				/* Initialize external IP table */
	natip_lookup:		lr_natip_lookup,			/* External IP table lookup */
	natip_add:		lr_natip_add,
	natip_del:		lr_natip_del,
	
	natip_asic_set:	rtl8651_setAsicExtIntIpTable,
	natip_asic_get:	rtl8651_getAsicExtIntIpTable,
	natip_asic_del:	rtl8651_delAsicExtIntIpTable,
#ifndef CONFIG_RTL865XC
	gidx_asic_set:		rtl8651_setAsicGidxRegister,
	gidx_asic_get:		rtl8651_getAsicGidxRegister,
#endif
};



static int32 lr_natip_init(void)
{
	memset(&TBLFIELD(natip_tbl, __natipbuff), 0, sizeof(rtl865x_tblAsicDrv_extIntIpParam_t));
	TBLFIELD(natip_tbl, natip_no) = 0;
#ifndef CONFIG_RTL865XC
	/* IC Bug: always use the first IP table entry, set GIDX=0 */
	TBLFIELD(natip_tbl, gidx_asic_set)(0);
#endif
	return 0;
}

static rtl865x_tblAsicDrv_extIntIpParam_t * lr_natip_lookup(ipaddr_t natip, uint32 *pos)
{
	rtl865x_tblAsicDrv_extIntIpParam_t *natbuf;
	uint32 entry;

	natbuf = &TBLFIELD(natip_tbl, __natipbuff);
	for(entry=0; entry<RTL8651_IPTABLE_SIZE; entry++) {
		if (TBLFIELD(natip_tbl, natip_asic_get)(entry, natbuf))
			continue;
		if (natbuf->extIpAddr == natip) {
			if (pos) *pos = entry;
			return natbuf;
		}
	}
	return (rtl865x_tblAsicDrv_extIntIpParam_t *)0;
}

static int32 lr_natip_add(ipaddr_t natip)
{
	rtl865x_tblAsicDrv_extIntIpParam_t *natbuf;
	uint32 entry;

	if (TBLFIELD(natip_tbl, natip_lookup)(natip, NULL))
		return -1;
	/* Currently we only support one NAPT IP */
	if (TBLFIELD(natip_tbl,natip_no) > 0)
		return -1;
	natbuf = &TBLFIELD(natip_tbl, __natipbuff);
	for(entry=0; entry<RTL8651_IPTABLE_SIZE; entry++) 
		if (TBLFIELD(natip_tbl, natip_asic_get)(entry, natbuf))
			break;	
	if (entry < RTL8651_IPTABLE_SIZE) {
		memset(natbuf, 0, sizeof(*natbuf));
		natbuf->extIpAddr		= natip;
		natbuf->intIpAddr		= 0;
		natbuf->localPublic	= 0;
		natbuf->nat			= 0;
		natbuf->nhIndex		= 0;
		TBLFIELD(natip_tbl, natip_asic_set)(entry, natbuf);
		TBLFIELD(natip_tbl,natip_no) ++;
		LR_INIT_CHECK(rtl8651_setAsicOperationLayer(4));
		return 0;
	}
	return -1;	
}


static int32 lr_natip_del(ipaddr_t natip)
{
	uint32 pos;

	if (TBLFIELD(natip_tbl, natip_lookup)(natip, &pos)) {
		TBLFIELD(natip_tbl, natip_asic_del)(pos);
		TBLFIELD(natip_tbl,natip_no) --;
		if (!TBLFIELD(natip_tbl,natip_no))
			LR_INIT_CHECK(rtl8651_setAsicOperationLayer(2));
		return 0;
	}
	return -1;
}

enum RTL_RESULT rtl865x_addNaptIp(ipaddr_t natip)
{
	if (TBLFIELD(natip_tbl, natip_add)(natip) < 0)
		return RTL_FAILED;
	return RTL_SUCCESS;
}


enum RTL_RESULT rtl865x_delNaptIp(ipaddr_t natip)
{
	uint32 ipidx;
	if (!TBLFIELD(natip_tbl, natip_lookup)(natip, &ipidx))
		return RTL_INVEXTIP;
	if (TBLFIELD(natip_tbl, natip_del)(natip) < 0)
		return RTL_FAILED;
	return RTL_SUCCESS;
}


void rt_arp_sync(ipaddr_t ip, uint32 row, uint32 column, enum SYNC_FLAG dowhat)
{
	rtl865x_tblAsicDrv_routingParam_t l3entry;
	int32 rtidx, rc;
	
	for(rtidx=0; rtidx<RTL8651_ROUTINGTBL_SIZE; rtidx++) {
		if (dowhat == SYNC_DEL) {
			rc = TBLFIELD(rt_tbl, rt_asic_get)(rtidx, &l3entry);
			if (rc || l3entry.process!=0x01) 
				continue;
			if (l3entry.nextHopRow!=row || 
				l3entry.nextHopColumn!=column)
				continue;
			/* update the route: lack gw info. */
			l3entry.process = 0x04;
			TBLFIELD(rt_tbl, rt_asic_set)(rtidx, &l3entry);
			TBLFIELD(rt_tbl, pendgw)[rtidx] = ip;
		} 
		else if (dowhat == SYNC_ADD) {
			if (TBLFIELD(rt_tbl, pendgw)[rtidx]!=ip) 
				continue;
			rc = TBLFIELD(rt_tbl, rt_asic_get)(rtidx, &l3entry);
			assert(!rc && l3entry.process==0x04);
			l3entry.process 		= 0x01;
			l3entry.nextHopRow	= row;
			l3entry.nextHopColumn	= column;
			TBLFIELD(rt_tbl, rt_asic_set)(rtidx, &l3entry);
			TBLFIELD(rt_tbl, pendgw)[rtidx] = 0;
		}
	}
}


enum RTL_RESULT rtl865x_addArp( ipaddr_t ip, ether_addr_t * mac, enum RTL_ARP_FLAGS flags )
{
	rtl865x_tblAsicDrv_l2Param_t *fdbe;
	rtl865x_tblAsicDrv_arpParam_t arpe;
	uint32 fdb_type[]={ FDB_STATIC, FDB_DYNAMIC };
	struct if_entry *ife;
	uint32 hash,way;
	int32 rc;
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
	int32 fid = 0;
#endif

	if (TBLFIELD(arpt_tbl, arp_lookup)(ip))
		return RTL_DUPENTRY;
	ife = TBLFIELD(if_tbl, if_lookup)(NULL, ip);
	if (ife==NULL)
		return RTL_INVAPARAM;
	if (!IF_UP(ife))
		return RTL_IFDOWN;
	hash = TBLFIELD(arpt_tbl, arp_hash)(ip);
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
	{
		rtl865x_vlan_entry_t *vlan;
		vlan = _rtl8651_getVlanTableEntry(ife->vid);
		if(vlan == NULL)
			printk("%s(%d): can't get vlan by vid(%d)....\n",__FUNCTION__,__LINE__,ife->vid);
		
		fid = vlan->fid;
	}
#endif
	
	for(rc=0; rc<2; rc++) {
		//fdbe = TBLFIELD(fdb_tbl, fdb_lookup)(ife->fid, mac, fdb_type[rc], &way);
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
		fdbe = TBLFIELD(fdb_tbl, fdb_lookup)(fid, mac, fdb_type[rc], &way);
#else
		fdbe = TBLFIELD(fdb_tbl, fdb_lookup)(GET_IF_VID(ife), mac, fdb_type[rc], &way);
#endif
		if (fdbe) {/* present an fdb entry in the ASIC */
			arpe.nextHopColumn = way;
#ifdef CONFIG_RTL865XC
			arpe.aging    =    300;

#ifdef CONFIG_RTL865X_LAYERED_DRIVER
		arpe.nextHopRow = TBLFIELD(fdb_tbl, fdb_asic_hash)(mac, fid);
#else
		arpe.nextHopRow = TBLFIELD(fdb_tbl, fdb_asic_hash)(mac, GET_IF_VID(ife));
#endif

			
#else
			arpe.nextHopRow = TBLFIELD(fdb_tbl, fdb_asic_hash)(mac);
#endif
			TBLFIELD(arpt_tbl, arp_asic_set)(hash, &arpe);
			TBLFIELD(fdb_tbl, FDB)[(arpe.nextHopRow<<2)+arpe.nextHopColumn] ++;
			if (fdb_type[rc]==FDB_DYNAMIC) {
				/* force dynamic entry to static entry */
#if 0
				fdbe->isStatic = 1;
#else
				fdbe->nhFlag  = 1;
#endif
				fdbe->ageSec = L2_AGING_TIME;
				TBLFIELD(fdb_tbl, fdb_asic_set)(arpe.nextHopRow, arpe.nextHopColumn, fdbe);
			}
			rt_arp_sync(ip, arpe.nextHopRow, arpe.nextHopColumn, SYNC_ADD);
			return RTL_SUCCESS;
		}
	}	
	
	/* No specified mac address presents in fdb table, give up !! */
	return RTL_FAILED;
}


enum RTL_RESULT rtl865x_delArp( ipaddr_t ip )
{
	rtl865x_tblAsicDrv_arpParam_t *arpe;
	uint32 arp_hash, l2pos;
	
	arpe = TBLFIELD(arpt_tbl, arp_lookup)(ip);
	if (arpe == NULL)
		return RTL_NOTFOUND;
	arp_hash = TBLFIELD(arpt_tbl, arp_hash)(ip);
	TBLFIELD(arpt_tbl, arp_asic_del)(arp_hash);
	l2pos = (arpe->nextHopRow<<2) + arpe->nextHopColumn;
	if (--TBLFIELD(fdb_tbl, FDB)[l2pos] == 0) {
		rt_arp_sync(ip, arpe->nextHopRow, arpe->nextHopColumn, SYNC_DEL);
		TBLFIELD(fdb_tbl, fdb_asic_del)(arpe->nextHopRow, arpe->nextHopColumn);
	}
	return RTL_SUCCESS;
}


uint32 rtl865x_arpSync( ipaddr_t ip, uint32 refresh )
{
	rtl865x_tblAsicDrv_arpParam_t *arpe;
	rtl865x_tblAsicDrv_l2Param_t l2entry;
	uint32 age;

	arpe = TBLFIELD(arpt_tbl, arp_lookup)(ip);
	if (!arpe)
		return 0;
	TBLFIELD(fdb_tbl, fdb_asic_get)(arpe->nextHopRow, 
			arpe->nextHopColumn, &l2entry);
	age = l2entry.ageSec;
	if (refresh) {
		l2entry.ageSec = 300;
		TBLFIELD(fdb_tbl, fdb_asic_set)(arpe->nextHopRow, 
				arpe->nextHopColumn, &l2entry);
	}
	return age;
}

int32 rtl865x_addPppoeSession( uint8* ifname, ether_addr_t *mac, uint32 sessionId, uint32 flags )
{
	uint32 fdb_type[]={ FDB_STATIC, FDB_DYNAMIC };
	rtl865x_tblAsicDrv_pppoeParam_t pppoe;
	rtl865x_tblAsicDrv_l2Param_t *fdbe = NULL;
	struct if_entry *ife;
	enum RTL_RESULT rc;
	uint32 idx, way, hash, pppidx;
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
	int32 fid = 0;
#endif


	if (!(ife=TBLFIELD(if_tbl, if_lookup)(ifname, 0)))
		return RTL_NOTFOUND;
	if (ife->if_type != IF_PPPOE)
		return RTL_INVAPARAM;
	if (TBLFIELD(pppoe_tbl, pppoe_lookup)(sessionId, NULL))
		return RTL_EXIST;	
	if ((rc=TBLFIELD(if_tbl, if_up)(ifname, 0, 0, sessionId, mac)))
		return rc;
	for(pppidx=0; pppidx<RTL8651_PPPOETBL_SIZE; pppidx++)
		if (!TBLFIELD(pppoe_tbl, ppps[pppidx]))
			break;
	if (pppidx == RTL8651_PPPOETBL_SIZE)
		return RTL_NOBUFFER;
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
	{
		rtl865x_vlan_entry_t *vlan;
		vlan = _rtl8651_getVlanTableEntry(ife->vid);
		if(vlan == NULL)
			printk("%s(%d): can't get vlan by vid(%d)....\n",__FUNCTION__,__LINE__,ife->vid);
		
		fid = vlan->fid;
	}
#endif


	
#ifdef CONFIG_RTL865XC
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
	hash = TBLFIELD(fdb_tbl, fdb_asic_hash)(mac, fid);
#else
	hash = TBLFIELD(fdb_tbl, fdb_asic_hash)(mac, GET_IF_VID(ife));
#endif
#else
	hash = TBLFIELD(fdb_tbl, fdb_asic_hash)(mac);
#endif
	for(idx=0; idx<2; idx++) {

#ifdef CONFIG_RTL865X_LAYERED_DRIVER
		fdbe = TBLFIELD(fdb_tbl, fdb_lookup)(fid, mac, fdb_type[idx], &way);
#else
		fdbe = TBLFIELD(fdb_tbl, fdb_lookup)(GET_IF_VID(ife), mac, fdb_type[idx], &way);
#endif

		
		if (fdbe) {
			TBLFIELD(fdb_tbl, FDB)[(hash<<2)+way] ++;
			if (fdb_type[idx] == FDB_DYNAMIC) {
				/* force dynamic entry to static entry */
				fdbe->isStatic = 1;
				fdbe->ageSec = L2_AGING_TIME;
				TBLFIELD(fdb_tbl, fdb_asic_set)(hash, way, fdbe);
			} break;
		}
	}
	if (!fdbe) {
		TBLFIELD(if_tbl, if_down)(ifname);
		return RTL_FAILED;
	}
	memset(&pppoe, 0, sizeof(rtl865x_tblAsicDrv_pppoeParam_t));
	pppoe.age		= 300;
	pppoe.sessionId	= sessionId;
	TBLFIELD(pppoe_tbl, ppps[pppidx]) = ife;
	TBLFIELD(pppoe_tbl, pppoe_asic_set)(pppidx, &pppoe);
	return RTL_SUCCESS;
}


int32 rtl865x_delPppoeSession( uint8* ifname )
{
	rtl865x_tblAsicDrv_routingParam_t rth;
	rtl865x_tblAsicDrv_pppoeParam_t pppoe;
	struct if_entry *ife, *ife1;
	uint32 idx, hash, way;
	int32 ret;
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
	int32 fid = 0;
#endif

	ife = TBLFIELD(if_tbl, if_lookup)(ifname, 0);
	if (ife == NULL)
		return RTL_NOTFOUND;
	
	if(ife->if_type != IF_PPPOE)
		return RTL_INVAPARAM;

#ifdef CONFIG_RTL865X_LAYERED_DRIVER
	{
		rtl865x_vlan_entry_t *vlan;
		vlan = _rtl8651_getVlanTableEntry(ife->vid);
		if(vlan == NULL)
			printk("%s(%d): can't get vlan by vid(%d)....\n",__FUNCTION__,__LINE__,ife->vid);
		
		fid = vlan->fid;
	}
#endif


	for(idx=0; idx<RTL8651_ROUTINGTBL_SIZE; idx++) 
	{
		if ((ret =TBLFIELD(rt_tbl, rt_asic_get)(idx, &rth)) != SUCCESS)
			continue;
		ife1 = TBLFIELD(pppoe_tbl, ppps[rth.pppoeIdx]);
		if (rth.process==0x00 && ife==ife1)
			return RTL_INUSE;
	}
	
	memset(&pppoe, 0, sizeof(rtl865x_tblAsicDrv_pppoeParam_t));
	TBLFIELD(if_tbl, if_down)(ifname);

	if (TBLFIELD(pppoe_tbl, pppoe_lookup)(ife->sid_, &idx)) 
	{
		TBLFIELD(pppoe_tbl, pppoe_asic_set)(idx, &pppoe);
		TBLFIELD(pppoe_tbl, ppps[idx]) = NULL;
	}
	
#ifdef CONFIG_RTL865X_LAYERED_DRIVER
	if (TBLFIELD(fdb_tbl, fdb_lookup)(fid, &ife->pmac_, FDB_STATIC, &way)) 
#else
	if (TBLFIELD(fdb_tbl, fdb_lookup)(GET_IF_VID(ife), &ife->pmac_, FDB_STATIC, &way)) 
#endif
	{
#ifdef CONFIG_RTL865XC

#ifdef CONFIG_RTL865X_LAYERED_DRIVER
		hash = TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife->pmac_, fid);
#else
		hash = TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife->pmac_, GET_IF_VID(ife));
#endif
#else
		hash = TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife->pmac_);
#endif
		idx = (hash<<2) + way;
		if (--TBLFIELD(fdb_tbl, FDB)[idx] == 0)
			TBLFIELD(fdb_tbl, fdb_asic_del)(hash, way);
	}
	return RTL_SUCCESS;
}

#if defined(CONFIG_RTL865X_HW_PPTPL2TP) || defined(CONFIG_RTL865X_PPPOE_HWACC)
int32 rtl865x_addPppSession(uint8 *ifname, ether_addr_t *mac, uint32 sessionId, int32 type)
{
#if defined(CONFIG_RTL865X_HW_PPTPL2TP)	
	rtl865x_tblAsicDrv_l2Param_t *fdbe;
	uint32 way;
#endif
	struct if_entry *ife;
	enum LR_RESULT rc;
	
#if 0	
	struct rtl865x_vlanConfig		vlanConfig;
#endif
	
	if (type == SE_PPPOE)
		return rtl865x_addPppoeSession(ifname, mac, sessionId, SE_NONE);
#if 0	
	if (! rc = TBLFIELD(vlan_tbl, vlan_create)(vlanConfig))
		return rc
#endif
	if (!(ife = TBLFIELD(if_tbl, if_lookup)(ifname, 0)))
		return RTL_NOTFOUND;
	if (ife->if_type != IF_PPPOE)
		return RTL_INVAPARAM;

	if ((rc = TBLFIELD(if_tbl, if_up)(ifname, 0, 0, 0, mac)))
		return rc;

#if defined(CONFIG_RTL865X_HW_PPTPL2TP)	
	fdbe = TBLFIELD(fdb_tbl, fdb_lookup)(GET_IF_VID(ife), mac, FDB_STATIC, &way);
	if (!fdbe) {
		TBLFIELD(fdb_tbl, fdb_add)(GET_IF_VID(ife), mac, RTL865X_PPTP_HWACC_PORTMASK, FDB_FWD);
	}
#endif
	RTL_DEBUG(RTL_MSG_PPPOE, "LR(addPppSession):  %s \n", ifname);
	return RTL_SUCCESS;
}

int32 rtl865x_delPppSession(uint8 *ifname, int32 type)
{
	struct if_entry *ife;
#if defined(CONFIG_RTL865X_HW_PPTPL2TP)
	uint32 idx, hash, way;
#endif
	if (type == SE_PPPOE)
		return rtl865x_delPppoeSession(ifname);
		
	if (!(ife = TBLFIELD(if_tbl, if_lookup)(ifname, 0)))
		return LR_NOTFOUND;
	
	if (ife->if_type != IF_PPPOE)
		return RTL_INVAPARAM;
	TBLFIELD(if_tbl, if_down)(ifname);

#if defined(CONFIG_RTL865X_HW_PPTPL2TP)
	if (TBLFIELD(fdb_tbl, fdb_lookup)(GET_IF_VID(ife), &ife->pmac_, FDB_STATIC, &way)) {
#ifdef CONFIG_RTL865XC
		hash = TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife->pmac_, GET_IF_VID(ife));
#else
		hash = TBLFIELD(fdb_tbl, fdb_asic_hash)(&ife->pmac_);
#endif
		idx = (hash<<2) + way;
		if (--TBLFIELD(fdb_tbl, FDB)[idx] == 0)
			TBLFIELD(fdb_tbl, fdb_asic_del)(hash, way);
	}
#endif	
	RTL_DEBUG(RTL_MSG_PPPOE, "LR(delPppSession):  %s \n", ifname);
	return RTL_SUCCESS;
}

#endif


