/*
* Copyright c                  Realtek Semiconductor Corporation, 2002  
* All rights reserved.
* 
* Program : rtl8651_layer2asic.c
* Abstract : RTL8651 Home gateway controller Layer2 asic table driver
* Author : Chun-Feng Liu(cfliu@realtek.com.tw)
* $Id: rtl8651_layer2asic.c,v 1.1 2010/05/19 11:14:57 jackey Exp $
*/

/*	@doc RTL8651_LAYER2ASIC_API

	@module rtl8651_layer2asic.c - RTL8651 Home gateway controller Layer2 ASIC related  API documentation	|
	This document explains the internal and external API interface of the layer 2 ASIC module. Functions with _rtl8651 prefix
	are internal functions and functions with rtl8651_ prefix are external functions.
	@normal Gateway Team (gwhp@realtek.com.tw) <date>

	Copyright <cp>2006 Realtek<tm> Semiconductor Cooperation, All Rights Reserved.

 	@head3 List of Symbols |
 	Here is a list of all functions and variables in this module.

 	@index | RTL8651_LAYER2_API
*/


#include "rtl_types.h"
#include "rtl_utils.h"
#include "assert.h"
#include "types.h"
#ifndef CONFIG_RTL865X_LIGHT_ROMEDRV
#include "rtl8651_layer2.h"
#include "rtl8651_tblAsicDrv.h"
#include "rtl8651_tblDrvLocal.h"
#else
#define RTL8651_TBLDRV_LOCAL_H
#include "rtl8651_tblAsicDrv.h"
#include "rtl865x_lightrome.h"
#endif
#include "asicRegs.h"
#ifdef __TO_BE_DELETED__
#include "rtl_bdinfo.h"
#include "board.h"
#endif

#ifdef RTL865X_TEST
#include <stdio.h>
#ifndef CONFIG_RTL865X_LIGHT_ROMEDRV
#include "drvTest.h"
#endif
int8						*pVirtualSWReg;
int8						*pVirtualSysReg;
int8						*pVirtualHsb;
int8						*pVirtualHsa;
int8						*pVirtualSWTable;
#endif /* RTL865X_TEST */

#ifdef RTL865X_MODEL_USER
#include <stdio.h>
#include <string.h>
#include "rtl_glue.h"
#include "icModel.h"
int8						*pVirtualSWReg;
int8						*pVirtualSysReg;
int8						*pVirtualHsb;
int8						*pVirtualHsa;
int8						*pVirtualSWTable;
#endif

#ifdef RTL865X_MODEL_KERNEL
int8						*pVirtualSWReg = (int8*)REAL_SWCORE_BASE;
int8						*pVirtualSysReg = (int8*)REAL_BD01_BASE;
int8						*pVirtualHsb = (int8*)REAL_HSB_BASE;
int8						*pVirtualHsa = (int8*)REAL_HSA_BASE;
int8						*pVirtualSWTable = (int8*)REAL_SWTBL_BASE;
#endif

#ifndef CONFIG_RTL865X_LIGHT_ROMEDRV
#include "rtl8651_layer2fwdLocal.h"
#endif

int8 rtl8651_layer2Asic_Id[] = "$Id: rtl8651_layer2asic.c,v 1.1 2010/05/19 11:14:57 jackey Exp $";

#ifndef CONFIG_RTL865X_LIGHT_ROMEDRV

/*=========================================
  * Internal Data structure
  *=========================================*/
typedef struct rtl8651_tblAsic_hardwiredProtoTrapTable_s {
	uint16 trapContent;
	uint8  trapType;
	int8   isTrapOn;
	uint32 trapFlag;
} rtl8651_tblAsichardwiredProtoTrapTable_t;


/* 
* Initialize Hardwired Protocol Trap Table:
* =====================================================================
*		EtherType  IPProtocol  Layer4Port  Protocol
*		0x0806		*			*			ARP
*		0x8035		*			*			RARP
*		0x8863		*			*			PPPoE Discover phase
*		0x0800		2			*			IGMP, DVMRP, PIMv1
*		0x0800		UDP			67			BOOTPs/DHCP (Bug in RTL8651)
*		0x0800		UDP			68			BOOTPc/DHCP (Bug in RTL8651)
*		0x0800		89			*			OSPF
*		0x0800		UDP			520			RIP
*
*		(PS: RTL8651 ASIC implemets BOOTP as IP protocol, its a wrong implementation.
*		     Hence the following table set BOOTP to IP protocol)
*/
rtl8651_tblAsichardwiredProtoTrapTable_t hardwiredProtoTrap[] =
{
	{ 0x0806, RTL8651_PROTOTRAP_ETHER,   	FALSE, 	EN_ARP_TRAP   		}, /* ARP */
	{ 0x8035, RTL8651_PROTOTRAP_ETHER,   	FALSE, 	EN_RARP_TRAP  		}, /* RARP */
	{ 0x8863, RTL8651_PROTOTRAP_ETHER,   	FALSE, 	EN_PPPOE_TRAP 		}, /* PPPoE Discover phase */
	{ 0x0002, RTL8651_PROTOTRAP_IP, 		FALSE, 	EN_IGMP_TRAP  		}, /* IGMP, DVMRP, PIMv1 */
	{ 0x0043, RTL8651_PROTOTRAP_IP, 		FALSE, 	EN_DHCP_TRAP1 		}, /* BOOTPs/DHCP (Bug in RTL8651) */
	{ 0x0044, RTL8651_PROTOTRAP_IP, 		FALSE, 	EN_DHCP_TRAP2 		}, /* BOOTPc/DHCP (Bug in RTL8651) */
	{ 0x0059, RTL8651_PROTOTRAP_IP, 	   	FALSE, 	EN_OSPF_TRAP  		}, /* OSPF */
	{ 0x0208, RTL8651_PROTOTRAP_UDPPORT, 	FALSE, 	EN_RIP_TRAP 		  	}, /* RIP */
};
#endif


/*=========================================
  * Static Function Prototype
  *=========================================*/
static void _rtl8651_clearSpecifiedAsicTable(uint32 type, uint32 count);
static void _rtl8651_initialRead(void);

#ifdef CONFIG_RTL865X_LIGHT_ROMEDRV
typedef struct rtl8651_tblAsic_ethernet_s {
	uint8 linkUp:1;
} rtl8651_tblAsic_ethernet_t;

#endif

rtl8651_tblAsic_ethernet_t 	rtl8651AsicEthernetTable[9];//RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum
uint32 	linkChangePendingCount;
int32 	miiPhyAddress;
int32 	wlan_acc_debug = 0;

int32		rtl8651_totalExtPortNum=0; //this replaces all RTL8651_EXTPORT_NUMBER defines
int32		rtl8651_allExtPortMask=0; //this replaces all RTL8651_EXTPORTMASK defines



#ifndef CONFIG_RTL865X_LIGHT_ROMEDRV
/*=========================================
  * ASIC DRIVER API: PROTOCOL TRAP TABLE
  *=========================================*/
#define RTL865X_ASIC_DRIVER_PROTOTRAP_TBL_API

int32 rtl8651_turnOnHardwiredProtoTrap(uint8 protoType, uint16 protoContent) {
//when WLAN acceleration flag is compiled, ASIC protocol trap is disabled. CPU would receive a copy
//of any bcast pkt.

	rtl8651_tblAsichardwiredProtoTrapTable_t *ptPtr;
	int32 index;

	ptPtr = &hardwiredProtoTrap[0];
	for(index=0; index<RTL8651_HARDWIRED_PROTOTRAP_SIZE; index++, ptPtr++)
		if (protoType==ptPtr->trapType && protoContent==ptPtr->trapContent) {
			//write to ASIC only when WLAN acceleration is NOT enabled.
			WRITE_MEM32(PTRAPCR,READ_MEM32(PTRAPCR)|ptPtr->trapFlag);
			ptPtr->isTrapOn = TRUE;
			return SUCCESS;
		}
		
	return FAILED;
}

int32 rtl8651_turnOffHardwiredProtoTrap(uint8 protoType, uint16 protoContent) {
	rtl8651_tblAsichardwiredProtoTrapTable_t *ptPtr;
	int32 index;

	ptPtr = &hardwiredProtoTrap[0];
	for(index=0; index<RTL8651_HARDWIRED_PROTOTRAP_SIZE; index++, ptPtr++)
		if (protoType==ptPtr->trapType && protoContent==ptPtr->trapContent) {
			WRITE_MEM32(PTRAPCR,READ_MEM32(PTRAPCR)&~ptPtr->trapFlag);
			ptPtr->isTrapOn = FALSE;
			return SUCCESS;
		}
	return FAILED;
}

int32 rtl8651_getHardwiredProtoTrap(uint8 protoType, uint16 protoContent, int8 *isEnable) {
	rtl8651_tblAsichardwiredProtoTrapTable_t *ptPtr;
	int32 index;

	if (isEnable == NULL)
		return FAILED;
	ptPtr = &hardwiredProtoTrap[0];
	for(index=0; index<RTL8651_HARDWIRED_PROTOTRAP_SIZE; index++, ptPtr++)
		if (protoType==ptPtr->trapType && protoContent==ptPtr->trapContent) {
			*isEnable = (READ_MEM32(PTRAPCR)&ptPtr->trapFlag)? TRUE: FALSE;
			//*isEnable = ptPtr->isTrapOn;
			return index;
		}
	return FAILED;
}
#endif



/*=========================================
  * ASIC DRIVER API: SWITCH MODE
  *=========================================*/
#define RTL865X_ASIC_DRIVER_SWITCH_MODE_API

 int32 rtl8651_operationlayer=0;
int32 rtl8651_setAsicOperationLayer(uint32 layer) {
	if(layer<1 || layer>4)
		return FAILED;

	/*   for bridge mode ip multicast patch
		When  in bridge mode,
		only one vlan(8) is available,
		if rtl8651_operationlayer is set less than 3,
		(pelase refer to rtl8651_setAsicVlan() )
		the "enable routing bit" of VLAN table will be set to 0 according to rtl8651_operationlayer.
		On the one hand, multicast data is flooded in vlan(8) by hardware, 
		on the other hand,it will be also trapped to cpu.
		In romedriver process,
		it will do _rtl8651_l2PhysicalPortRelay() in _rtl8651_l2switch(),
		and results in same multicast packet being flooded twice: 
		one is by hareware, the other is by romedriver.
		so the minimum  rtl8651_operationlayer will be set 3.
	*/
	if(layer<=3)
	{
		layer=3;
	}
	
	if(layer == 1) {
		WRITE_MEM32(MSCR,READ_MEM32(MSCR)&~(EN_L2|EN_L3|EN_L4));
		WRITE_MEM32(MSCR,READ_MEM32(MSCR)&~(EN_IN_ACL));
		WRITE_MEM32(MSCR,READ_MEM32(MSCR)&~(EN_OUT_ACL));
	}else{
		WRITE_MEM32(MSCR,READ_MEM32(MSCR)|(EN_IN_ACL));
		WRITE_MEM32(MSCR,READ_MEM32(MSCR)|(EN_OUT_ACL));

		if(layer == 2) {
			WRITE_MEM32(MSCR,READ_MEM32(MSCR)|(EN_L2));
			WRITE_MEM32(MSCR,READ_MEM32(MSCR)&~(EN_L3|EN_L4));
		}
		else
		{	// options for L3/L4 enable
			//WRITE_MEM32(MISCCR,READ_MEM32(MISCCR)|FRAG2CPU); //IP fragment packet need to send to CPU when multilayer is enabled
			/*
			  * Only for RTL8650B:
			  * It is no need to trap fragment packets to CPU in pure rouitng mode while ACL is enabled, hence we should 
			  * turn this bit (FRAG2CPU) off.
			  * NOTE: if we do this, we should also turn ENFRAG2ACLPT on.     
			  *														-chhuang, 7/30/2004
			  */
			WRITE_MEM32(MISCCR,READ_MEM32(MISCCR) & ~FRAG2CPU);
		
			if(layer == 3) {
				WRITE_MEM32(MSCR,READ_MEM32(MSCR)|(EN_L2|EN_L3));
				WRITE_MEM32(MSCR,READ_MEM32(MSCR)&~(EN_L4));
			}
			else {	// layer 4
				WRITE_MEM32(MSCR,READ_MEM32(MSCR)|(EN_L2|EN_L3|EN_L4));
			}
		}
	}
	if(layer == 1)
		rtl8651_setAsicAgingFunction(FALSE, FALSE);
	else if (layer == 2 || layer == 3)
		rtl8651_setAsicAgingFunction(TRUE, FALSE);
	else
		rtl8651_setAsicAgingFunction(TRUE, TRUE);
	rtl8651_operationlayer	=layer;
	return SUCCESS;
}

int32 rtl8651_getAsicOperationLayer(void) {
#ifdef RTL865X_DEBUG
	uint32 regValue, layer=0;

	regValue = READ_MEM32(MSCR);
	switch(regValue & (EN_L2|EN_L3|EN_L4)) {
		case 0:
			layer = 1;
		break;
		case EN_L2:
			layer = 2;
		break;
		case (EN_L2|EN_L3):
			layer = 3;
		break;
		case (EN_L2|EN_L3|EN_L4):
			layer = 4;
		break;
		default:
			assert(0);//ASIC should not have such value
	}
	if(layer!=rtl8651_operationlayer){
		return -1;
	}
		
#endif
	return  rtl8651_operationlayer;
}




/*=========================================
  * ASIC DRIVER API: SPANNING TREE
  *=========================================*/
#define RTL865X_ASIC_DRIVER_SPANNING_TREE_API

#if 0
//portState reference to RTL8651_PORTSTA_xxx
int32 rtl8651_getAsicMulticastSpanningTreePortStateEx(uint32 port, uint32 *portState, int link_id) {
	uint32 reg;
	uint32 offset;
#if defined(CONFIG_BRIDGE) ||defined(CONFIG_BRIDGE_MODULE)
	_rtl8651_extDevice_t *extDev;
#endif

	if(port >= (RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum) || portState == NULL)
		return FAILED;

	
	if(port < RTL8651_PORT_NUMBER && link_id <= 0)   //phy port
	{
		offset = 7;
		reg=(uint32)SWTMCR;
		WRITE_MEM32(reg,READ_MEM32(reg)&~(0x3 << (offset+(port<<1))));
		switch((READ_MEM32(reg) >> (offset+(port<<1)) )&0x3) {
			case 0:
				*portState = RTL8651_PORTSTA_DISABLED;
				break;
			case 1:
				*portState = RTL8651_PORTSTA_BLOCKING;
				break;
			case 2:
				*portState = RTL8651_PORTSTA_LEARNING;
				break;
			case 3:
				*portState = RTL8651_PORTSTA_FORWARDING;
				break;
			default:
				assert(0);
		}
		return SUCCESS;
	}

	if (link_id > 0)
	//if (port >= RTL8651_PORT_NUMBER && link_id > 0)	
	{
#if defined(CONFIG_BRIDGE) ||defined(CONFIG_BRIDGE_MODULE)		
		extDev = (_rtl8651_extDevice_t *)rtl8651_fwdEngineGetExtDevFromLinkid(link_id);

		if (extDev == NULL)
			return FAILED;

		*portState = extDev->portStat;
#endif
		return SUCCESS;
	}
	return FAILED;
}
#endif

//portState reference to RTL8651_PORTSTA_xxx
int32 rtl8651_setAsicMulticastSpanningTreePortState(uint32 port, uint32 portState) {
	uint32 reg;
	uint32 offset;
	if(port >= (RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum) || portState > RTL8651_PORTSTA_FORWARDING)
		return FAILED;

	if(port>=RTL8651_PORT_NUMBER){
		//extension port.
		reg=(uint32)MISCCR;
		offset = 13;
		port -=RTL8651_PORT_NUMBER;
	}else{
		offset = 7;
		reg=(uint32)SWTMCR;
	}
		
	WRITE_MEM32(reg,READ_MEM32(reg)&~(0x3 << (offset+(port<<1))));
	switch(portState) {
		case RTL8651_PORTSTA_DISABLED://0
			break;
		case RTL8651_PORTSTA_BLOCKING:
		case RTL8651_PORTSTA_LISTENING:
			WRITE_MEM32(reg,READ_MEM32(reg)|(0x1 << (offset+(port<<1))));
			break;
		case RTL8651_PORTSTA_LEARNING:
			WRITE_MEM32(reg,READ_MEM32(reg)|(0x2 << (offset+(port<<1))));
			break;
		case RTL8651_PORTSTA_FORWARDING:
			WRITE_MEM32(reg,READ_MEM32(reg)|(0x3 << (offset+(port<<1))));
			break;
		default:
			assert(0);
	}
	return SUCCESS;
}

//portState reference to RTL8651_PORTSTA_xxx
int32 rtl8651_getAsicMulticastSpanningTreePortState(uint32 port, uint32 *portState) {
	uint32 reg;
	uint32 offset;

	if(port >= (RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum) || portState == NULL)
		return FAILED;
	if(port>=RTL8651_PORT_NUMBER){
		//extension port.
		reg=(uint32)MISCCR;
		offset = 13;
		port -=RTL8651_PORT_NUMBER;
	}else{
		offset = 7;
		reg=(uint32)SWTMCR;
	}

	switch((READ_MEM32(reg) >> (offset+(port<<1)) )&0x3) {
		case 0:
			*portState = RTL8651_PORTSTA_DISABLED;
			break;
		case 1:
			*portState = RTL8651_PORTSTA_BLOCKING;
			break;
		case 2:
			*portState = RTL8651_PORTSTA_LEARNING;
			break;
		case 3:
			*portState = RTL8651_PORTSTA_FORWARDING;
			break;
		default:
			assert(0);
	}
	return SUCCESS;
}



#ifdef RTL865XB_DCUT_SWVLAN
int32 rtl8651_setAsicUnknownVidToCpu(int8 enabled) {
	if (enabled == TRUE)
		WRITE_MEM32(MSCR, READ_MEM32(MSCR) | (EN_UKNOWNVID_TO_CPU));
	else
		WRITE_MEM32(MSCR, READ_MEM32(MSCR) & ~(EN_UKNOWNVID_TO_CPU));
	return SUCCESS;
}
#endif

int32 rtl8651_setAsicSpanningEnable(int8 spanningTreeEnabled) {
	if(spanningTreeEnabled == TRUE)
	{
		WRITE_MEM32(MSCR,READ_MEM32(MSCR)|(EN_STP));
		WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)|(BRIDGE_PKT_TO_CPU));
	}else
	{
		WRITE_MEM32(MSCR,READ_MEM32(MSCR)&~(EN_STP));
		WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)&~(BRIDGE_PKT_TO_CPU));
	}
	return SUCCESS;
}

int32 rtl8651_getAsicSpanningEnable(int8 *spanningTreeEnabled) {
	if(spanningTreeEnabled == NULL)
		return FAILED;
	*spanningTreeEnabled = (READ_MEM32(MSCR)&(EN_STP)) == (EN_STP)? TRUE: FALSE;
	return SUCCESS;
}





/*=========================================
  * ASIC DRIVER API: BROADCAST REG
  *=========================================*/
#define RTL865X_ASIC_DRIVER_BROADCAST_REG_API

int32 rtl8651_setBroadCastStormReg(int8 enable) {
	uint32 flag;

	if (enable == TRUE)
	/*The LSB of BSCR should be 0 to enable broadcast storm control */
	{
		flag = 0xFFFFFFFE;
		WRITE_MEM32(BSCR, flag & READ_MEM32(BSCR));
	}
	else
	{
		 flag = 0x00;
		WRITE_MEM32(BSCR, flag | READ_MEM32(BSCR));
	}
	return SUCCESS;
}

int32 rtl8651_getBroadCastSTormReg(int8 *enable) {
	*enable = (READ_MEM32(BSCR)&0x01)==0x01? TRUE: FALSE;
	return SUCCESS;
}



/*=========================================
  * ASIC DRIVER API: PATTERN MATCH
  *=========================================*/
#define RTL865X_ASIC_DRIVER_PATTERN_MATCH_API

int32 rtl8651_setAsicPortPatternMatch(uint32 port, uint32 pattern, uint32 patternMask, int32 operation) {
	//not for ext port
	if(port>=RTL8651_PORT_NUMBER)
		return FAILED;

	if(pattern==0&&patternMask==0){
		if((READ_MEM32(PPMAR)&0x2000)==0) //system pattern match not enabled.
			return SUCCESS;
		WRITE_MEM32(PPMAR,READ_MEM32(PPMAR)&~(1<<(port+26)));
		if((READ_MEM32(PPMAR)&0xfc000000)==0)
			WRITE_MEM32(PPMAR,READ_MEM32(PPMAR)&~(1<<13)); //turn off system pattern match switch.

		return SUCCESS;
	}
	if(operation>3)
		return FAILED; //valid operations: 0(drop), 1(mirror to cpu),2(fwd to cpu), 3(to mirror port) 
	WRITE_MEM32(PPMAR,READ_MEM32(PPMAR)|((1<<(port+26))|(1<<13))); //turn on system pattern match and turn on pattern match on indicated port.
	WRITE_MEM32(PPMAR,READ_MEM32(PPMAR)|(operation<<(14+2*port)));   //specify operation
	WRITE_MEM32(PATP0+4*port,pattern);
	WRITE_MEM32(MASKP0+4*port,patternMask);
	return SUCCESS;
}

int32 rtl8651_getAsicPortPatternMatch(uint32 port, uint32 *pattern, uint32 *patternMask, int32 *operation) {
	//not for ext port
	if(port>=RTL8651_PORT_NUMBER)
		return FAILED;
	if(((READ_MEM32(PPMAR)& (1<<13))==0)||((READ_MEM32(PPMAR)& (1<<(26+port)))==0))
		return FAILED;
	if(pattern)
		*pattern=READ_MEM32(PATP0+4*port);
	if(patternMask)
		*patternMask=READ_MEM32(MASKP0+4*port);
	if(operation)
		*operation=(READ_MEM32(PPMAR)>>(14+2*port))&0x3;
	return SUCCESS;		
}



#ifndef CONFIG_RTL865X_LIGHT_ROMEDRV

/*=========================================
  * ASIC DRIVER API: PORT MIRROR
  *=========================================*/
#define RTL865X_ASIC_DRIVER_PORT_MIRROR_TBL_API

int32 rtl8651_setAsicPortMirror(uint32 mTxMask, uint32 mRxMask, uint32 mPortMask) {
	uint32 i, port;
	uint32 extRxMask, extTxMask, rxMask, txMask, extPortMask, portmask;
	mTxMask &= RTL8651_ALLPORTMASK; mRxMask &= RTL8651_ALLPORTMASK;	
	/* For Tx Mirror, only one Mirror Tx port is allowed */
	for(i=0, port=0; i<RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum; i++)
		if (mTxMask & (1<<i))
			port++;
	if (port > 1) 
		return FAILED;
	extRxMask=mRxMask>>RTL8651_MAC_NUMBER;
	extTxMask=mTxMask>>RTL8651_MAC_NUMBER;
	rxMask =mRxMask&RTL8651_PHYSICALPORTMASK;
	txMask = mTxMask&RTL8651_PHYSICALPORTMASK;
	extPortMask = mPortMask>>RTL8651_MAC_NUMBER;
	portmask = mPortMask & RTL8651_PHYSICALPORTMASK;
	WRITE_MEM32(PMCR, ((portmask<<26)|(rxMask<<20)|(txMask<<14)|(extPortMask<<11)|(extRxMask<<8)|(extTxMask<<5)));
	return SUCCESS;
}

int32 rtl8651_getAsicPortMirror(uint32 *mRxMask, uint32 *mTxMask, uint32 *mPortMask) {
	uint32 cache = READ_MEM32(PMCR);

	if (mPortMask) *mPortMask = (((cache>>11)&0x07)<<6)|(cache>>26);
	if (mRxMask) *mRxMask = (((cache>>8)&0x07)<<6)|((cache>>20)&0x3f);
	if (mTxMask) *mTxMask = (((cache>>5)&0x07)<<6)|((cache>>14)&0x3f);
	return SUCCESS;
}

#endif



/*=========================================
  * ASIC DRIVER API: ETHERNET MII
  *=========================================*/
#define RTL865X_ASIC_DRIVER_ETHERNET_MII_API

int32 rtl8651_setAsicEthernetMII(uint32 phyAddress, int32 mode, int32 enabled){
	if(enabled){
		if(miiPhyAddress>=0){//all enabled.
			if(miiPhyAddress!=phyAddress)
				return FAILED;
		}	
		miiPhyAddress=phyAddress;
	}else{
		miiPhyAddress=-1;
	}
	REG32(MISCCR)|=	mode << P5_LINK_OFFSET;	
	if(mode==P5_LINK_MII_PHY){
		/*    Bug: Modification of driver level link status information can't let mii port be added into VLAN member. 
		*      Therefore, MII port cannot work at all. 
		*/
		#if 0
		rtl8651AsicEthernetTable[RTL8651_MII_PORTNUMBER].linkUp=1;
		#endif

		/* Make link change, then MII port will be added into the corresponding VLAN active member. */
		rtl8651AsicEthernetTable[RTL8651_MII_PORTNUMBER].linkUp=0;
		rtl8651_setAsicEthernetLinkStatus(RTL8651_MII_PORTNUMBER, TRUE);
	}
	return SUCCESS;
}

int32 rtl8651_getAsicEthernetMII(uint32 *phyAddress){
	*phyAddress=miiPhyAddress;
	return SUCCESS;
}



/*=========================================
  * ASIC DRIVER API: PHY LOOPBACK
  *=========================================*/
#define RTL865X_ASIC_DRIVER_PHY_LOOPBACK_TBL_API

int32 rtl8651_setAsicEthernetPHYLoopback(uint32 port, int32 enabled) {
	uint32 value;
	if(port >= RTL8651_PORT_NUMBER) //not for extension ports
		return FAILED;

	if(port==RTL8651_PORT_NUMBER-1){//config MII port. must be a 802.3 PHY
		WRITE_MEM32(MDCIOCR,(miiPhyAddress<<24)); //read PHY address 0
#ifndef RTL865X_TEST
		while((READ_MEM32(MDCIOSR)&0x80000000)!=0);//wait until command complete
#endif /* RTL865X_TEST */
		value=READ_MEM32(MDCIOSR)&0xffff;//keep last 16 bits
		if(enabled)
			WRITE_MEM32(MDCIOCR,0x80000000 | (miiPhyAddress<<24)|value|0x4000); //write to PHY reg 0 to enable digital loopback
		else
			WRITE_MEM32(MDCIOCR,0x80000000 | (miiPhyAddress<<24)|(value & ~0x4000)); //write to PHY reg 0 to disable digital loopback
		goto out;
	}
	value=READ_MEM32(PORT0_PHY_CONTROL+ (port<<5));
	WRITE_MEM32(SWTAA, (uint32)(PHY_BASE + (port<<5)));
	if(enabled){
		value|= ((1<<13)|(1<<14));//enable didital loopback and 100Mbps
		value &=~(1<<12); //disbale N-way 
		WRITE_MEM32(TCR0,value);
	}
	else
		WRITE_MEM32(TCR0, value & (~(1<<14)));//disable didital loopback
	//rtlglue_printf("\ntaa: %x  tcr0: %x, value: %x", SWTAA, TCR0, value|(1<<14));
 	WRITE_MEM32(SWTACR, ACTION_START | CMD_FORCE);//Activate  command
#ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */
	//rtlglue_printf("reg: %x\n", READ_MEM32(PORT0_PHY_CONTROL+ (port<<5)));
	//rtlglue_printf("Port %d digital loopback enabled\n", port);

out:
#ifndef CONFIG_RTL865X_LIGHT_ROMEDRV
        rtl8651_setEthernetPortLinkStatus(port, TRUE);
#endif
	return SUCCESS;

}

int32 rtl8651_getAsicEthernetPHYLoopback(uint32 port, int32 *flag) {
	//not for ext port.
	if(port >= RTL8651_PORT_NUMBER||!flag)
		return FAILED;
	
	if(port==RTL8651_PORT_NUMBER-1){//read MII port. must be a 802.3 PHY
		uint32 value;
		if(miiPhyAddress<0||miiPhyAddress>31)
			return FAILED; //not yet enabled.

		WRITE_MEM32(MDCIOCR,(miiPhyAddress<<24)); //read PHY address 0
#ifndef RTL865X_TEST
		while((READ_MEM32(MDCIOSR)&0x80000000)!=0);//wait until command complete
#endif /* RTL865X_TEST */
		value=READ_MEM32(MDCIOSR)&0xffff;//keep last 16 bits
		if(value&0x4000)
			*flag=TRUE;
		else 
			*flag= FALSE;
		return SUCCESS;
	}
	*flag = (READ_MEM32(PHY_BASE + (port<<5)) & ENABLE_LOOPBACK)? TRUE: FALSE;
	return SUCCESS;
}




/*=========================================
  * ASIC DRIVER API: ETHERNET PHY
  *=========================================*/
#define RTL865X_ASIC_DRIVER_ETHERNET_PHY_API

int32 rtl8651_setAsicEthernetPHY(uint32 port, int8 autoNegotiation, uint32 advCapability, uint32 speed, int8 fullDuplex) {
	uint32 ctrlReg;
	int32 flag;
	//not for ext port
	if(port >= RTL8651_PORT_NUMBER)
		return FAILED;

	if(port==RTL8651_PORT_NUMBER-1){//config MII port. must be a 802.3 PHY
		uint32 capability;
		if(miiPhyAddress<0||miiPhyAddress>31)
			return FAILED; //not yet enabled.
		if(READ_MEM32(MISCCR)&(P5_LINK_MII_PHY << P5_LINK_OFFSET))
			return FAILED; //if set to PHY mode, there is no external PHY for config
		rtl8651_getAsicEthernetPHYLoopback(port, &flag);
		if(flag==TRUE)
			ctrlReg = ENABLE_LOOPBACK; //auto neg and loopback is mutually exclusive
		else if(autoNegotiation == TRUE) 
			ctrlReg = ENABLE_AUTONEGO;//Henry suggestion not to restart it....So, remark this following parts | RESTART_AUTONEGO;
		else
			ctrlReg = 0;
		if(speed == 1) // 100Mbps, default assume 10Mbps
			ctrlReg |= SPEED_SELECT_100M;
		if(fullDuplex == TRUE)
			ctrlReg |= SELECT_FULL_DUPLEX;
		//rtlglue_printf("MII reg 0 =%x",0x80000000 | (miiPhyAddress<<24)|ctrlReg);
		WRITE_MEM32(MDCIOCR,0x80000000 | (miiPhyAddress<<24)|ctrlReg); //write to PHY reg 0

		if(advCapability == RTL8651_ETHER_AUTO_100FULL)
			capability = CAPABLE_PAUSE | CAPABLE_100BASE_TX_FD | CAPABLE_100BASE_TX_HD| CAPABLE_10BASE_TX_FD | CAPABLE_10BASE_TX_HD | 0x1;
		else if(advCapability == RTL8651_ETHER_AUTO_100HALF)
			capability = CAPABLE_PAUSE | CAPABLE_100BASE_TX_HD| CAPABLE_10BASE_TX_FD | CAPABLE_10BASE_TX_HD | 0x1;
		else	if(advCapability == RTL8651_ETHER_AUTO_10FULL)
			capability = CAPABLE_PAUSE | CAPABLE_10BASE_TX_FD | CAPABLE_10BASE_TX_HD | 0x1;
		else if(advCapability == RTL8651_ETHER_AUTO_10HALF)
			capability = CAPABLE_PAUSE | CAPABLE_10BASE_TX_HD | 0x1;
		else
			return FAILED;//Parameter of RTL8651_ETHER_AUTO_ not matched
		//rtlglue_printf(",reg 4 =%x \n",0x80000000 | (miiPhyAddress<<24)|(4<<16)|capability);
		WRITE_MEM32(MDCIOCR,0x80000000 | (miiPhyAddress<<24)|(4<<16)|capability); //write to PHY reg 4
		
		return SUCCESS;
	}

	WRITE_MEM32(SWTAA, (uint32)(PHY_BASE + (port<<5) + 16));

	/* For RTL865xB cut-d & after that. */
	if (RtkHomeGatewayChipRevisionID >= 0x02)
	{
	 	/* chhuang: WHY always turn on flow control????? It doesn't make sense!!!*/
		if(advCapability == RTL8651_ETHER_AUTO_100FULL)
			WRITE_MEM32(TCR0,CAPABLE_PAUSE | CAPABLE_100BASE_TX_FD | CAPABLE_100BASE_TX_HD| CAPABLE_10BASE_TX_FD | CAPABLE_10BASE_TX_HD | 0x1);
		else if(advCapability == RTL8651_ETHER_AUTO_100HALF)
			WRITE_MEM32(TCR0,CAPABLE_PAUSE | CAPABLE_100BASE_TX_HD| CAPABLE_10BASE_TX_FD | CAPABLE_10BASE_TX_HD | 0x1);
		else	if(advCapability == RTL8651_ETHER_AUTO_10FULL)
			WRITE_MEM32(TCR0,CAPABLE_PAUSE | CAPABLE_10BASE_TX_FD | CAPABLE_10BASE_TX_HD | 0x1);
		else if(advCapability == RTL8651_ETHER_AUTO_10HALF)
			WRITE_MEM32(TCR0,CAPABLE_PAUSE | CAPABLE_10BASE_TX_HD | 0x1);
		else
			return FAILED;//Parameter of RTL8651_ETHER_AUTO_ not matched
	}
	/* For RTL865xB cut-b & before that. */
	else	
	{
		/* chhuang: patch for IC bug & priority issue , 2004/4/7 */
		ctrlReg = READ_MEM32((PHY_BASE + (port<<5) + 16)) & ~(0xf<<5);
		if(advCapability == RTL8651_ETHER_AUTO_100FULL)
			ctrlReg |= (CAPABLE_100BASE_TX_FD | CAPABLE_100BASE_TX_HD| CAPABLE_10BASE_TX_FD | CAPABLE_10BASE_TX_HD);
		else if(advCapability == RTL8651_ETHER_AUTO_100HALF)
			ctrlReg |= (CAPABLE_100BASE_TX_HD| CAPABLE_10BASE_TX_FD | CAPABLE_10BASE_TX_HD);
		else if(advCapability == RTL8651_ETHER_AUTO_10FULL)
			ctrlReg |= (CAPABLE_10BASE_TX_FD | CAPABLE_10BASE_TX_HD);
		else if(advCapability == RTL8651_ETHER_AUTO_10HALF)
			ctrlReg |= (CAPABLE_10BASE_TX_HD);
		else if (advCapability == RTL8651_ETHER_FORCE_100FULL)
			ctrlReg |= 	CAPABLE_100BASE_TX_FD;	
		else if (advCapability == RTL8651_ETHER_FORCE_100HALF)
			ctrlReg |= CAPABLE_100BASE_TX_HD;
		else if (advCapability == RTL8651_ETHER_FORCE_10FULL)
			ctrlReg |= CAPABLE_10BASE_TX_FD;
		else if (advCapability == RTL8651_ETHER_FORCE_10HALF)
			ctrlReg |= CAPABLE_10BASE_TX_HD;
		else return FAILED;//Parameter of RTL8651_ETHER_AUTO_ not matched
		WRITE_MEM32(TCR0, ctrlReg);
	}	

 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);//Activate add command
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */

	if(autoNegotiation == TRUE) 
		ctrlReg = ENABLE_AUTONEGO;//Henry suggestion not to restart it....So, remark this following parts | RESTART_AUTONEGO;
	else
		ctrlReg = 0;
	if(speed == 1) // 100Mbps, default assume 10Mbps
		ctrlReg |= SPEED_SELECT_100M;
	if(fullDuplex == TRUE)
		ctrlReg |= SELECT_FULL_DUPLEX;
	WRITE_MEM32(SWTAA, (uint32)(PHY_BASE + (port<<5)));
	WRITE_MEM32(TCR0,ctrlReg);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);//Activate add command
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */

	//if ( (READ_MEM32(SWTASR) & TABSTS_MASK) == TABSTS_SUCCESS )//Check status
	//	return SUCCESS;

	/* After setting, we still need to Restart N-Way when auto-negotiation is enabled */
	WRITE_MEM32(SWTAA, PHY_BASE + (port<<5));
	WRITE_MEM32(TCR0, (READ_MEM32(PHY_BASE+(port<<5)) | 0x200));
	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);//Activate add command
#ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */
	return SUCCESS;
}

int32 rtl8651_getAsicEthernetPHY(uint32 port, int8 *autoNegotiation, uint32 *advCapability, uint32 *speed, int8 *fullDuplex) {
	uint32 count = 0;
	if(port >= RTL8651_PORT_NUMBER || autoNegotiation == NULL || advCapability == NULL || speed == NULL || fullDuplex == NULL)
		return FAILED;
		
	if(port==RTL8651_PORT_NUMBER-1){//read MII port. must be a 802.3 PHY
		uint32 status;
		if(miiPhyAddress<0||miiPhyAddress>31)
			return FAILED; //not yet enabled.
		status=READ_MEM32(MDCIOSR); //read MII port status.
		*autoNegotiation = (status&0x800000)? TRUE: FALSE;
		if(status&0x50000)
		*advCapability = RTL8651_ETHER_AUTO_100FULL;
		else if(status&0x10000)
		*advCapability = RTL8651_ETHER_AUTO_100HALF;
		else if(status&0x40000)
		*advCapability = RTL8651_ETHER_AUTO_10FULL;
		else if((status&0x70000)==0)
		*advCapability = RTL8651_ETHER_AUTO_10HALF;
		*speed = (status&0x10000)? 1: 0;
		*fullDuplex = (status&0x40000)? TRUE: FALSE;
		return SUCCESS;
	}

	//make sure Auto-Negotiation Complete, if auto-negotiation is disabed, just
	//read PHY register
	while (count < 5 && !(READ_MEM32(PHY_BASE+(port<<5)+0x04)  & 0x20))
		count ++;
	
	*autoNegotiation = (READ_MEM32(PHY_BASE + (port<<5)) & ENABLE_AUTONEGO)? TRUE: FALSE;
	/* === shliu: why not read PHY reg. directly? === */
	if(READ_MEM32(TCR0) & CAPABLE_100BASE_TX_FD)
		*advCapability = RTL8651_ETHER_AUTO_100FULL;
	else if(READ_MEM32(TCR0) & CAPABLE_100BASE_TX_HD)
		*advCapability = RTL8651_ETHER_AUTO_100HALF;
	else if(READ_MEM32(TCR0) & CAPABLE_10BASE_TX_FD)
		*advCapability = RTL8651_ETHER_AUTO_10FULL;
	else if(READ_MEM32(TCR0) & CAPABLE_10BASE_TX_HD)
		*advCapability = RTL8651_ETHER_AUTO_10HALF;
	/* ================================ */
	*speed = (READ_MEM32(PHY_BASE + (port<<5)) & SPEED_SELECT_100M)? 1: 0;
	*fullDuplex = (READ_MEM32(PHY_BASE + (port<<5)) & SELECT_FULL_DUPLEX)? TRUE: FALSE;
	
	return SUCCESS;
}


int32 rtl8651_asicEthernetCableMeterInit(void){
	uint32 old_value;
	//set PHY6 Reg0 TxD latch internal clock phase
	WRITE_MEM32(SWTAA, 0xbc8020c0);
	WRITE_MEM32(TCR0,0x56);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */
	//set PHY7 Reg 5
	WRITE_MEM32(SWTAA, 0xbc8020f4);
	WRITE_MEM32(TCR0,0x4b68);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */
	//set PHY7 Reg 6
	WRITE_MEM32(SWTAA, 0xbc8020f8);
	WRITE_MEM32(TCR0,0x0380);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */
	old_value=READ_MEM32(0xbc8020fc);
	//set PHY7 Reg 7
	WRITE_MEM32(SWTAA, 0xbc8020fc);
	WRITE_MEM32(TCR0,old_value|0x300);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */
return SUCCESS;
}


int32 rtl8651_asicEthernetCableMeter(uint32 port, int32 *rxStatus, int32 *txStatus) {
	uint32 i;

	//not for ext port
	if((port >= RTL8651_PHY_NUMBER)||!rxStatus || !txStatus)
		return FAILED;

	//set target PHY port
	WRITE_MEM32(SWTAA, 0xbc8020fc);
	WRITE_MEM32(TCR0,port);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */

	//Prepare to test Rx pair, write 0
	WRITE_MEM32(SWTAA, 0xbc8020d8);
	WRITE_MEM32(TCR0,0x0000);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */

	//Start to test Rx pair, write 1
	WRITE_MEM32(SWTAA, 0xbc8020d8);
	WRITE_MEM32(TCR0,0x1000);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */

	//Wait and Check status
	for(i=0;i<10000000;i++);
	*rxStatus=READ_MEM32(0xbc8020d8)&0x7ff;
	
	//Reset cable meter
	WRITE_MEM32(SWTAA, 0xbc8020d8);
	WRITE_MEM32(TCR0,0x0000);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */

///////////////////////////////////////////////////////////////

	//set target PHY port
	WRITE_MEM32(SWTAA, 0xbc8020fc);
	WRITE_MEM32(TCR0,port);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */

	//Prepare to test Tx pair, write 0
	WRITE_MEM32(SWTAA, 0xbc8020d8);
	WRITE_MEM32(TCR0,0x8000);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */

	//Start to test Tx pair, write 1
	WRITE_MEM32(SWTAA, 0xbc8020d8);
	WRITE_MEM32(TCR0,0x9000);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */

	//Wait and Check status
	for(i=0;i<10000000;i++);
	*txStatus=READ_MEM32(0xbc8020d8)&0x7ff;

	//Reset cable meter
	WRITE_MEM32(SWTAA, 0xbc8020d8);
	WRITE_MEM32(TCR0,0x0000);
 	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);
 #ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */
	return SUCCESS;
}




/*=========================================
  * ASIC DRIVER API: BANDWIDTH CONTROL
  *=========================================*/
#define RTL865X_ASIC_DRIVER_BANDWIDTH_CONTROL_API

int32 rtl8651_setAsicEthernetBandwidthControl(uint32 port, int8 input, uint32 rate) {//RTL8651_ASICBC_xxx
	if(port >= RTL8651_PORT_NUMBER || rate > BW_8M)
		return FAILED;
	if(port<4) {
		if(input == TRUE)
			WRITE_MEM32(BCR1,(READ_MEM32(BCR1) & ~(0xf << (port<<3)) ) | (rate<< (port<<3)));
		else
			WRITE_MEM32(BCR1,(READ_MEM32(BCR1) & ~(0xf0 << (port<<3)) ) | (rate<< ((port<<3)+4)));
	}
	else {
		if(input == TRUE)
			WRITE_MEM32(BCR0,(READ_MEM32(BCR0) & ~(0xf << ((port-4)<<3)) ) | (rate<< (((port-4)<<3))));
		else
			WRITE_MEM32(BCR0,(READ_MEM32(BCR0) & ~(0xf0 << ((port-4)<<3)) ) | (rate<< (((port-4)<<3)+4)));
	}
	return SUCCESS;
}

int32 rtl8651_getAsicEthernetBandwidthControl(uint32 port, int8 input, uint32 *rate) {//RTL8651_ASICBC_xxx
	if(port >= RTL8651_PORT_NUMBER || rate == NULL)
		return FAILED;

	if(port<4) {
		if(input == TRUE)
			*rate = (READ_MEM32(BCR1) >> (port<<3)) & 0xf;
		else
			*rate = (READ_MEM32(BCR1) >> ((port<<3)+4)) & 0xf;
	}
	else {
		if(input == TRUE)
			*rate = (READ_MEM32(BCR0) >> ((port-4)<<3)) & 0xf;
		else
			*rate = (READ_MEM32(BCR0) >> (((port-4)<<3)+4)) & 0xf;
	}
	return SUCCESS;
}

int32 rtl8651_setAsicEthernetBandwidthControlX4(int8 enable) {

	if(enable == TRUE)
		WRITE_MEM32(SWTECR,(READ_MEM32(SWTECR) | (1 << 15) ));
	else
		WRITE_MEM32(SWTECR,(READ_MEM32(SWTECR) & ~(1 << 15) ));
		
	return SUCCESS;
}

int32 rtl8651_getAsicEthernetBandwidthControlX4(int8 *enable) {
	if((READ_MEM32(SWTECR) & (1 << 15)))
		*enable = TRUE;
	else
		*enable = FALSE;
	return SUCCESS;
}

int32 rtl8651_setAsicEthernetBandwidthControlX8(int8 enable) {

	if(enable == TRUE)
		WRITE_MEM32(SWTECR,(READ_MEM32(SWTECR) | (1 << 14) ));
	else
		WRITE_MEM32(SWTECR,(READ_MEM32(SWTECR) & ~(1 << 14) ));
		
	return SUCCESS;
}

int32 rtl8651_getAsicEthernetBandwidthControlX8(int8 *enable) {
	if((READ_MEM32(SWTECR) & (1 << 14)))
		*enable = TRUE;
	else
		*enable = FALSE;
	return SUCCESS;
}


/*=========================================
  * ASIC DRIVER API: PORT INTERNAL
  *=========================================*/
#define RTL865X_ASIC_DRIVER_PORT_INTERNAL_API

int32 rtl8651_setAsicMulticastPortInternal(uint32 port, int8 isInternal) {
	if(port >= RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum)
		return FAILED;
	if(port >= RTL8651_PORT_NUMBER){
		//extension port.
		port-=(RTL8651_PORT_NUMBER);
		WRITE_MEM32(QOSCR,READ_MEM32(QOSCR)&~(0x1 << (port)));
		if(isInternal == FALSE)
			WRITE_MEM32(QOSCR,READ_MEM32(QOSCR)|(0x1 << (port)));
		return SUCCESS;
	}	
	WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)&~(0x1 << (MCAST_PORT_EXT_MODE_OFFSET+port)));
	if(isInternal == FALSE)
		WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)|(0x1 << (MCAST_PORT_EXT_MODE_OFFSET+port)));
	return SUCCESS;
}

int32 rtl8651_getAsicMulticastPortInternal(uint32 port, int8 *isInternal) {
	uint32 reg;
	uint32 offset;
	if(port >= (RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum) || isInternal == NULL)
		return FAILED;
	if(port>=RTL8651_PORT_NUMBER){
		//extension port.
		reg=(uint32)QOSCR;
		offset = 0;
		port -=RTL8651_PORT_NUMBER;
	}else{
		offset = 19;
		reg=(uint32)SWTMCR;
	}

	if((READ_MEM32(reg)>>(offset+port)) & 0x1)
		*isInternal = TRUE;
	else
		*isInternal = FALSE;
	return SUCCESS;
}

/*
@func int32		|	rtl8651_setAsicMulticastMTU 	| Set MTU for ASIC IP multicast forwarding
@parm uint32		|	mcastMTU	| MTU used by HW IP multicast forwarding.
@rvalue FAILED	|	Failed
@rvalue SUCCESS	|	Success
@comm
To set the MTU for ASIC IP multicast forwarding.
Its independent from other packet forwarding because IP multicast would include L2/L3/L4 at one time.
*/
int32 rtl8651_setAsicMulticastMTU(uint32 mcastMTU)
{
	if (mcastMTU & ~(MultiCastMTU_MASK) )
	{	/* multicast MTU overflow */
		return FAILED;
	}

	UPDATE_MEM32(MISCCR, mcastMTU, MultiCastMTU_MASK, MultiCastMTU_OFFSET);

	return SUCCESS;
}

/*
@func int32		|	rtl8651_setAsicMulticastMTU 	| Get MTU for ASIC IP multicast forwarding
@parm uint32*	|	mcastMTU	| Pointer to get MTU used by HW IP multicast forwarding.
@rvalue FAILED	|	Failed
@rvalue SUCCESS	|	Success
@comm
To get the MTU value for ASIC IP multicast forwarding.
Its independent from other packet forwarding because IP multicast would include L2/L3/L4 at one time.
*/
int32 rtl8651_getAsicMulticastMTU(uint32 *mcastMTU)
{
	if (mcastMTU == NULL)
	{
		return FAILED;
	}

	*mcastMTU = GET_MEM32_VAL(MISCCR, MultiCastMTU_MASK, MultiCastMTU_OFFSET);

	return SUCCESS;
}


/*=========================================
  * ASIC DRIVER API: L2 TABLE
  *=========================================*/
#define RTL865X_ASIC_DRIVER_L2_TBL_API

int32 rtl8651_setAsicL2Table(uint32 row, uint32 column, rtl865x_tblAsicDrv_l2Param_t *l2p) {
	rtl8651_tblAsic_l2Table_t entry;

	if((row >= RTL8651_L2TBL_ROW) || (column >= RTL8651_L2TBL_COLUMN) || (l2p == NULL))
		return FAILED;
	if(l2p->macAddr.octet[5] != ((row ^ l2p->macAddr.octet[0] ^ l2p->macAddr.octet[1] ^ l2p->macAddr.octet[2] ^ l2p->macAddr.octet[3] ^ l2p->macAddr.octet[4]) & 0xff))
		return FAILED;

	memset(&entry, 0,sizeof(entry));
	entry.mac47_40 = l2p->macAddr.octet[0];
	entry.mac39_24 = (l2p->macAddr.octet[1] << 8) | l2p->macAddr.octet[2];
	entry.mac23_8 = (l2p->macAddr.octet[3] << 8) | l2p->macAddr.octet[4];

#if 1 //chhuang: #ifdef CONFIG_RTL8650B
	if( l2p->memberPortMask  > RTL8651_PHYSICALPORTMASK) //this MAC is on extension port
		entry.extMemberPort = (l2p->memberPortMask >>RTL8651_PORT_NUMBER);   
#endif /* CONFIG_RTL8650B */

	entry.memberPort = l2p->memberPortMask & RTL8651_PHYSICALPORTMASK;
	entry.hPriority = 0;
	entry.toCPU = l2p->cpu==TRUE? 1: 0;
	entry.isStatic = l2p->isStatic==TRUE? 1: 0;
	entry.nxtHostFlag = l2p->nhFlag==TRUE? 1: 0;
	entry.agingTime = (l2p->ageSec>200)? 0x02: (l2p->ageSec<=200 && l2p->ageSec>100)? 0x03: (l2p->ageSec<=100 && l2p->ageSec>0)? 0x01: 0x00;
	entry.srcBlock = (l2p->srcBlk==TRUE)? 1: 0;
	return _rtl8651_forceAddAsicEntry(TYPE_L2_SWITCH_TABLE, row<<2 | column, &entry);
}

int32 rtl8651_delAsicL2Table(uint32 row, uint32 column) {
	rtl8651_tblAsic_l2Table_t entry;

	if(row >= RTL8651_L2TBL_ROW || column >= RTL8651_L2TBL_COLUMN)
		return FAILED;

	bzero(&entry, sizeof(entry));
	entry.agingTime = 0;
	entry.nxtHostFlag = 0;
	return _rtl8651_forceAddAsicEntry(TYPE_L2_SWITCH_TABLE, row<<2 | column, &entry);
}


ether_addr_t cachedDA;
static uint32 cachedMbr;
unsigned int rtl8651_asicL2DAlookup(uint8 *dmac){
	uint32 column;
	rtl8651_tblAsic_l2Table_t   entry;
	
	unsigned int row = dmac[0]^dmac[1]^dmac[2]^dmac[3]^dmac[4]^dmac[5];
	//rtlglue_printf("mac %02x %02x %02x %02x %02x %02x \n",	mac[0],mac[1],mac[2],mac[3],mac[4],mac[5]);

#if 0 //chhuang:
	//cache hit...
	if(memcmp(&cachedDA, dmac, 6)==0)
		return cachedMbr;
#endif

	//cache miss...
	cachedMbr=0;
	for(column=0;column<RTL8651_L2TBL_COLUMN; column++) {
		WRITE_MEM32(TEATCR,READ_MEM32(TEATCR)|0x1);/*ASIC patch: disable L2 Aging while reading L2 table */
		_rtl8651_readAsicEntry(TYPE_L2_SWITCH_TABLE, row<<2 | column, &entry);
		WRITE_MEM32(TEATCR,READ_MEM32(TEATCR)&~0x1);/*ASIC patch: enable L2 Aging aftrer reading L2 table */
		if(entry.agingTime == 0 && entry.isStatic == 0)
			continue;
		if(	dmac[0]==entry.mac47_40 &&
		    	dmac[1]==(entry.mac39_24>>8) &&
		    	dmac[2]==(entry.mac39_24 & 0xff)&&
		    	dmac[3]==(entry.mac23_8 >> 8)&&
		    	dmac[4]==(entry.mac23_8 & 0xff)&&
			dmac[5]== (row ^dmac[0]^dmac[1]^dmac[2]^dmac[3]^dmac[4])){

			cachedDA=*((ether_addr_t *)dmac);
			cachedMbr =(entry.extMemberPort<<RTL8651_PORT_NUMBER) | entry.memberPort;
			return cachedMbr;
		}
	}
	if(column==RTL8651_L2TBL_COLUMN)
		return 0xffffffff;//can't find this MAC, broadcast it.
	return cachedMbr;
}



int32 rtl8651_getAsicL2Table(uint32 row, uint32 column, rtl865x_tblAsicDrv_l2Param_t *l2p) {
	rtl8651_tblAsic_l2Table_t   entry;

	if((row >= RTL8651_L2TBL_ROW) || (column >= RTL8651_L2TBL_COLUMN) || (l2p == NULL))
		return FAILED;

	WRITE_MEM32(TEATCR,READ_MEM32(TEATCR)|0x1);/*ASIC patch: disable L2 Aging while reading L2 table */
	_rtl8651_readAsicEntry(TYPE_L2_SWITCH_TABLE, row<<2 | column, &entry);
	WRITE_MEM32(TEATCR,READ_MEM32(TEATCR)&~0x1);/*ASIC patch: enable L2 Aging aftrer reading L2 table */

	if(entry.agingTime == 0 && entry.isStatic == 0 && entry.nxtHostFlag == 0)
		return FAILED;
	l2p->macAddr.octet[0] = entry.mac47_40;
	l2p->macAddr.octet[1] = entry.mac39_24 >> 8;
	l2p->macAddr.octet[2] = entry.mac39_24 & 0xff;
	l2p->macAddr.octet[3] = entry.mac23_8 >> 8;
	l2p->macAddr.octet[4] = entry.mac23_8 & 0xff;
	l2p->macAddr.octet[5] = row ^ l2p->macAddr.octet[0] ^ l2p->macAddr.octet[1] ^ l2p->macAddr.octet[2] ^ l2p->macAddr.octet[3] ^ l2p->macAddr.octet[4];
	l2p->cpu = entry.toCPU==1? TRUE: FALSE;
	l2p->srcBlk = entry.srcBlock==1? TRUE: FALSE;
	l2p->nhFlag = entry.nxtHostFlag==1? TRUE: FALSE;
	l2p->isStatic = entry.isStatic==1? TRUE: FALSE;
#if 1 //chhuang: #ifdef CONFIG_RTL8650B
	l2p->memberPortMask = (entry.extMemberPort<<RTL8651_PORT_NUMBER) | entry.memberPort;
#else
	l2p->memberPortMask = entry.memberPort;
#endif /* CONFIG_RTL8650B */
	l2p->ageSec = ( entry.agingTime == 0x02 )? 300: ( entry.agingTime == 0x03 )? 200: ( entry.agingTime == 0x01 )? 100: 0;//Maximum to be 5 min

	return SUCCESS;
}




/*=========================================
  * ASIC DRIVER API: PORT TRUNKING
  *=========================================*/
#define RTL865X_ASIC_DRIVER_TRUNKING_API

//if portmask=0: clear all aggregation port mappings.
//if portmask<0: clear specified port from agregator
//if portmask>0: set specified port to aggregator and rearrange them.
//rearrange policy is round-robin. ie. if port a,b,c is in portmask, then hash block 0~7's port number is a,b,c,a,b,c,a,b
int32  rtl8651_updateAsicLinkAggregatorLMPR(int32 portmask) {
	uint32 i,j, temp;
	uint32 reg1, reg2;
	uint32 PTCRvalue,PTRAPCRvalue;	
	if(portmask==0){
		uint32 oldPTRAPCR;
		WRITE_MEM32(PTCR,0);
		oldPTRAPCR=READ_MEM32(PTRAPCR);
		oldPTRAPCR &= ~((1<<24)-1); //clear all extension port mappings and portmask.
		WRITE_MEM32(PTRAPCR,oldPTRAPCR);
		return SUCCESS;
	}
	//First read current aggregated ports from register.
	reg1=READ_MEM32(PTCR);
	reg1 = reg1&((1<<RTL8651_PORT_NUMBER)-1);
	reg2=READ_MEM32(PTRAPCR);	
	reg2 = reg2&((1<<rtl8651_totalExtPortNum)-1);
	temp=  reg1|reg2 ;
	if(!temp)
		return SUCCESS;//continue only when there are already some ports aggregated.

	//Now determine what ports stay in aggregation set.
	portmask &= RTL8651_ALLPORTMASK; //clear out all unrelated bits.
	if(portmask<0){ 
		//deleting ports
		if(((-portmask) & temp) !=portmask)
			return FAILED;//Not all deleting ports are in aggregated port list. Delete none of them
		//otherwise, mask out deleted ports
		temp &= ~ (-portmask);
	}else{
		//adding ports
		temp |= portmask;
	}
	
	PTRAPCRvalue=reg2;
	PTCRvalue=0;
	
	//Now update ASIC registers, i is port number, j is hash result index
	for(i=0,j=0;j<8;i++, i=i%RTL8651_AGGREGATOR_NUMBER){
		if(i<RTL8651_AGGREGATOR_NUMBER-1){ 
			if( temp & (1<<i)){//port is aggregated
				PTCRvalue |= (  i<<((j+2)*3)   /*port number*/|(1<<i)/*portmask*/);
				j++; //go to next hash result index.
			}
		}else{
			if( temp & (1<<i)){//port is aggregated
				PTCRvalue |= (  0<<((j+2)*3)   /*port number*/);
				PTRAPCRvalue|=( ((i & (1<<3)) <<j  ) /*port number*/|(1<<(i-RTL8651_PORT_NUMBER))/*portmask*/);
				j++; //go to next hash result index.
			}
		}
	}
	return SUCCESS;
#if 0
	int8 candidate[RTL8651_AGGREGATOR_NUMBER], trueConfig;
	uint32 trunk1PortMask, lmpr;

	for(i=0; i<RTL8651_AGGREGATOR_NUMBER; i++)
		candidate[i] = FALSE;
	trunk1PortMask = READ_MEM32(PTCR)&0x3f;
	for(i=0, trueConfig = FALSE; i<RTL8651_PORT_NUMBER; i++) {
		if((trunk1PortMask & (1<<i)) && rtl8651AsicEthernetTable[i].linkUp == 1) {
			candidate[i] = TRUE;
			trueConfig = TRUE;
		}
		else
			candidate[i] = FALSE;
	}

	lmpr = 0;
	if(trueConfig == TRUE) 
		for(i=0, idx = 0; i<8; i++) {
			while(candidate[idx] == FALSE)
				idx = (idx+1)%RTL8651_PORT_NUMBER;
			lmpr |= idx<<i*3;
			idx = (idx+1)%RTL8651_PORT_NUMBER;
		}
	WRITE_MEM32(PTCR,(lmpr << 6) | trunk1PortMask);
#endif	
}


int32 rtl8651_setAsicLinkAggregator(uint32 portMask) {
	if(portMask >= 1<<RTL8651_AGGREGATOR_NUMBER)
		return FAILED;
	if(portMask > (1<<RTL8651_PORT_NUMBER)-1){
		//extensioin ports use part of PTRAPCR
		uint32 extPorts = portMask >> RTL8651_PORT_NUMBER, oldPTRAPCR;
		oldPTRAPCR=READ_MEM32(PTRAPCR);
		WRITE_MEM32(PTRAPCR,oldPTRAPCR|extPorts);
	}
	portMask &= 1<<RTL8651_PORT_NUMBER;
	WRITE_MEM32(PTCR,portMask);
	rtl8651_updateAsicLinkAggregatorLMPR(portMask);
	return SUCCESS;
}

int32 rtl8651_getAsicLinkAggregator(uint32 * portMask, uint32 *mapping) {
	uint32 PTRAPCRvalue, PTCRvalue, map=0, i, mappedPort;
	if(portMask == NULL || mapping == NULL)
		return FAILED;
	PTRAPCRvalue=READ_MEM32(PTRAPCR);
	PTCRvalue=READ_MEM32(PTCR);

	for(i=0;i<8;i++){
		mappedPort = (((PTRAPCRvalue&(rtl8651_totalExtPortNum+i))<<rtl8651_totalExtPortNum)  |((PTCRvalue>>(RTL8651_PORT_NUMBER+3*i))&7));
		map |= mappedPort<<(4*i);
	}
	*mapping = map;
	*portMask = ( PTRAPCRvalue & ((1<<rtl8651_totalExtPortNum)-1) ) | 
	            ( PTCRvalue & ((1<<RTL8651_PORT_NUMBER)-1) );

	return SUCCESS;
}



#ifndef CONFIG_RTL865X_LIGHT_ROMEDRV

int32 rtl8651_setAsicProtoTrap(uint32 index, rtl865x_tblAsicDrv_protoTrapParam_t *protoTrapp) {
    rtl8651_tblAsic_protoTrapTable_t entry;

    if ((index >= RTL8651_PROTOCOLTRAP_SIZE) || (protoTrapp == NULL))
        return FAILED;
        
    bzero(&entry, sizeof(entry));
    switch(protoTrapp->type) {
    case RTL8651_PROTOTRAP_ETHER: /* Trap ether type */
         entry.trapProtocol = 0x00;         
         break;
    case RTL8651_PROTOTRAP_IP: /* Trap IP protocol */
         entry.trapProtocol = 0x02;
         break;
    case RTL8651_PROTOTRAP_TCPPORT: /* Trap TCP port number */
         entry.trapProtocol = 0x05;
         break;
    case RTL8651_PROTOTRAP_UDPPORT: /* Trap UDP port number */
         entry.trapProtocol = 0x06;
         break;
    default: return FAILED;
    }
    entry.trapContent = protoTrapp->content;
    entry.valid = 1;
    return _rtl8651_forceAddAsicEntry(TYPE_PROTOCOL_TRAP_TABLE, index, &entry);
}

int32 rtl8651_delAsicProtoTrap(uint32 index) {
    rtl8651_tblAsic_protoTrapTable_t entry;

    if (index >= RTL8651_PROTOCOLTRAP_SIZE)
        return FAILED;
        
    bzero(&entry, sizeof(entry));
    entry.valid = 0;
    return _rtl8651_forceAddAsicEntry(TYPE_PROTOCOL_TRAP_TABLE, index, &entry);
}

int32 rtl8651_getAsicProtoTrap(uint32 index, rtl865x_tblAsicDrv_protoTrapParam_t *protoTrapp) {
    rtl8651_tblAsic_protoTrapTable_t entry;

    if ((index >= RTL8651_PROTOCOLTRAP_SIZE) || (protoTrapp == NULL))
        return FAILED;
    _rtl8651_readAsicEntry(TYPE_PROTOCOL_TRAP_TABLE, index, &entry);
    if (entry.valid == 0)
        return FAILED;
    protoTrapp->content = entry.trapContent;
    switch((uint8)entry.trapProtocol) {
    case 0x00: /* Trap Ether type */
         protoTrapp->type = RTL8651_PROTOTRAP_ETHER;
         break;
    case 0x02: /* Trap IP protocol */
         protoTrapp->type = RTL8651_PROTOTRAP_IP;
         break;
    case 0x05: /* Trap TCP port number */
         protoTrapp->type = RTL8651_PROTOTRAP_TCPPORT;
         break;
    case 0x06: /* Trap UDP port number */
         protoTrapp->type = RTL8651_PROTOTRAP_UDPPORT;
         break;
    default: assert(0);
    }
    return SUCCESS;
}

#endif


/*=========================================
  * ASIC DRIVER API: PVID 
  *=========================================*/
#define RTL865X_ASIC_DRIVER_PVID_API

//extension port ready.
int32 rtl8651_setAsicPvid(uint32 port, uint32 pvidx) {
	uint32 regValue;
	rtl8651_tblAsic_vlanTable_t entry;
	
	if(port>=RTL8651_AGGREGATOR_NUMBER || pvidx>=RTL8651_VLAN_NUMBER)
		return FAILED;;
	_rtl8651_readAsicEntry(TYPE_VLAN_TABLE, pvidx, &entry);
	if(entry.valid == 0)
		return FAILED;

	regValue = READ_MEM32(PVCR);
	regValue = (regValue & ~(0x7<<port*3)) | (pvidx<<port*3);
	WRITE_MEM32(PVCR,regValue);	
	return SUCCESS;
}

//extension port ready.
int32 rtl8651_getAsicPvid(uint32 port, uint32 *pvidx) {

	if(port>=RTL8651_AGGREGATOR_NUMBER || pvidx == NULL)
		return FAILED;
	*pvidx = (READ_MEM32(PVCR) >> port*3)&0x7;
	return SUCCESS;
}



/*=========================================
  * ASIC DRIVER API: VLAN TABLE
  *=========================================*/
#define RTL865X_ASIC_DRIVER_VLAN_TBL_API

int32 rtl8651_setAsicVlan(rtl865x_tblAsicDrv_vlanParam_t *vlanp) {
	uint32 i, vidx = rtl8651_vlanTableIndex(vlanp->vid);
	rtl8651_tblAsic_vlanTable_t entry;

	if(vlanp == NULL)
		return FAILED;

	bzero(&entry, sizeof(entry));
	entry.vhid = vlanp->vid>>3;
	entry.mac47_32 = (vlanp->macAddr.octet[0]<<8) | vlanp->macAddr.octet[1];
	entry.mac31_16 = (vlanp->macAddr.octet[2]<<8) | vlanp->macAddr.octet[3];
	entry.mac15_0 = (vlanp->macAddr.octet[4]<<8) | vlanp->macAddr.octet[5];

#if 1 //chhuang: #ifdef CONFIG_RTL8650B
	if(vlanp->memberPortMask > RTL8651_PHYSICALPORTMASK )
		entry.extMemberPort = vlanp->memberPortMask >> RTL8651_PORT_NUMBER;
	if(vlanp->untagPortMask > RTL8651_PHYSICALPORTMASK )
		entry.extEgressUntag = vlanp->untagPortMask >> RTL8651_PORT_NUMBER;
#endif /* CONFIG_RTL8650B */

	entry.memberPort = vlanp->memberPortMask & RTL8651_PHYSICALPORTMASK;

	entry.inACLStart = vlanp->inAclStart;
	entry.inACLEnd = vlanp->inAclEnd;
	entry.outACLStart = vlanp->outAclStart;
	entry.outACLEnd = vlanp->outAclEnd;
	entry.isInternal = vlanp->internal==TRUE? 1: 0;
	entry.enHWRoute = (rtl8651_getAsicOperationLayer()>2)?
		(vlanp->enableRoute==TRUE? 1: 0):0;
	entry.STPStatus = 0;

	for(i=0; i<RTL8651_PORT_NUMBER; i++) {
		switch(vlanp->portState[i]) {
		case RTL8651_PORTSTA_DISABLED:
			break;
		case RTL8651_PORTSTA_BLOCKING:
		case RTL8651_PORTSTA_LISTENING:
			entry.STPStatus |= (0x1<<(i*2));
			break;
		case RTL8651_PORTSTA_LEARNING:
			entry.STPStatus |= (0x2<<(i*2));
			break;
		case RTL8651_PORTSTA_FORWARDING:
			entry.STPStatus |= (0x3<<(i*2));
			break;
		default: assert(0);
		}
	}

	for(i=0; i<rtl8651_totalExtPortNum; i++) {
		switch(vlanp->portState[RTL8651_PORT_NUMBER+i]) {
		/*
			Always set forward state to extension ports.
			This is because, in current extension port structure, there might be many different
			extension devices aggregated in single extension port. And HW can not distinguish them
			according to Extension Port. The dispatch need to be processed in SW.
		*/
#if 0
		case RTL8651_PORTSTA_DISABLED:
			break;
		case RTL8651_PORTSTA_BLOCKING:
		case RTL8651_PORTSTA_LISTENING:
			entry.extSTPStatus |= (0x1<<(i*2));
			break;
		case RTL8651_PORTSTA_LEARNING:
			entry.extSTPStatus |= (0x2<<(i*2));
			break;
		case RTL8651_PORTSTA_FORWARDING:
			entry.extSTPStatus |= (0x3<<(i*2));
			break;
		//default: assert(0);
#endif
		default: //set forwarding
			entry.extSTPStatus |= (0x3<<(i*2));
			break;
		}
	}

	entry.bcastToCPU = vlanp->broadcastToCpu==TRUE? 1: 0;
	entry.promiscuous = vlanp->promiscuous == TRUE? 1: 0;
	entry.egressUntag = vlanp->untagPortMask & RTL8651_PHYSICALPORTMASK;

#if 1 //chhuang: #ifdef CONFIG_RTL8650B	
	if(vlanp->macAddrNumber==0)
	{
		#ifdef CONFIG_RTL865XB_ENRT		/* refer to _rtl8651_addVlan() for the meaning of this compile flag */
		entry.macNotExist = vlanp->macNonExist;
		#else
		entry.macNotExist=1;
		#endif
	}
	else
	{
		#ifdef CONFIG_RTL865XB_ENRT		/* refer to _rtl8651_addVlan() for the meaning of this compile flag */
		entry.macNotExist = vlanp->macNonExist;
		#else
		entry.macNotExist=0;
		#endif
#endif /* CONFIG_RTL8650B */		
		switch(vlanp->macAddrNumber) {
		case 0:
		case 1:
		    entry.macMask = 0;
		break;
		case 2:
		    entry.macMask = 1;
		break;
		case 4:
		    entry.macMask = 2;
		break;
		case 8:
		    entry.macMask = 3;
		break;
		default:
		    return FAILED;//Not permitted macNumber value
		}
#if 1 //chhuang: #ifdef CONFIG_RTL8650B
	}
#endif /* CONFIG_RTL8650B */
	entry.mtuH = vlanp->mtu >> 8;
	entry.mtuL = vlanp->mtu & 0xff;
	entry.valid = vlanp->valid;
#if 1 //chhuang: #ifdef CONFIG_RTL8650B
	entry.isDMZ = vlanp->DMZFlag;
#endif /* CONFIG_RTL8650B */


	return _rtl8651_forceAddAsicEntry(TYPE_VLAN_TABLE, vidx, &entry);
}

int32 rtl8651_delAsicVlan(uint16 vid) {
	uint32 vidx = rtl8651_vlanTableIndex(vid);
	rtl8651_tblAsic_vlanTable_t entry;

	bzero(&entry, sizeof(entry));
	entry.valid = 0;
	return _rtl8651_forceAddAsicEntry(TYPE_VLAN_TABLE, vidx, &entry);
}

int32 rtl8651_getAsicVlan(uint16 vid, rtl865x_tblAsicDrv_vlanParam_t *vlanp) {
	uint32 i, vidx = rtl8651_vlanTableIndex(vid);
	rtl8651_tblAsic_vlanTable_t entry;

	if(vlanp == NULL)
		return FAILED;
        
	_rtl8651_readAsicEntry(TYPE_VLAN_TABLE, vidx, &entry);
	if(entry.valid == 0 || entry.vhid != (vid>>3))
	{
		vlanp->valid = entry.valid; /* return the entry usage */
		return FAILED;
	}
	vlanp->valid=1;
	vlanp->vid=vid;
	vlanp->macAddr.octet[0] = entry.mac47_32 >> 8;
	vlanp->macAddr.octet[1] = entry.mac47_32 & 0xff;
	vlanp->macAddr.octet[2] = entry.mac31_16 >> 8;
	vlanp->macAddr.octet[3] = entry.mac31_16 & 0xff;
	vlanp->macAddr.octet[4] = entry.mac15_0 >> 8;
	vlanp->macAddr.octet[5] = entry.mac15_0 & 0xff;
	vlanp->memberPortMask = (entry.extMemberPort<<RTL8651_PORT_NUMBER) | entry.memberPort;
	vlanp->untagPortMask = (entry.extEgressUntag<<RTL8651_PORT_NUMBER) |entry.egressUntag;
	vlanp->inAclStart = entry.inACLStart;
	vlanp->inAclEnd = entry.inACLEnd;
	vlanp->outAclStart = entry.outACLStart;
	vlanp->outAclEnd = entry.outACLEnd;
	vlanp->internal = entry.isInternal==1? TRUE: FALSE;
	vlanp->enableRoute = entry.enHWRoute==1? TRUE: FALSE;
	#ifdef CONFIG_RTL865XB_ENRT		/* refer to _rtl8651_addVlan() for the meaning of this compile flag */
	vlanp->macNonExist = entry.macNotExist == 1? TRUE: FALSE;
	#endif
	
	for(i=0; i<RTL8651_PORT_NUMBER; i++) {
		switch((entry.STPStatus>>(i*2))&0x3) {
		case 0x0:
			vlanp->portState[i] = RTL8651_PORTSTA_DISABLED;
			break;
		case 0x1:
			vlanp->portState[i] = RTL8651_PORTSTA_BLOCKING;
			break;
		case 0x2:
			vlanp->portState[i] = RTL8651_PORTSTA_LEARNING;
			break;
		case 0x3:
			vlanp->portState[i] = RTL8651_PORTSTA_FORWARDING;
			break;
		}
	}

	for(i=0; i<rtl8651_totalExtPortNum; i++) {
		uint32 portNum=RTL8651_PORT_NUMBER+i;
		switch((entry.extSTPStatus>>(i*2))&0x3) {
		case 0x0:
			vlanp->portState[portNum] = RTL8651_PORTSTA_DISABLED;
			break;
		case 0x1:
			vlanp->portState[portNum] = RTL8651_PORTSTA_BLOCKING;
			break;
		case 0x2:
			vlanp->portState[portNum] = RTL8651_PORTSTA_LEARNING;
			break;
		case 0x3:
			vlanp->portState[portNum] = RTL8651_PORTSTA_FORWARDING;
			break;
		}
	}
    
	vlanp->broadcastToCpu = entry.bcastToCPU==1? TRUE: FALSE;
	vlanp->promiscuous = entry.promiscuous==1? TRUE: FALSE;
#ifdef CONFIG_RTL865XB_ENRT		/* refer to _rtl8651_addVlan() for the meaning of this compile flag */
	vlanp->macAddrNumber = (1 << entry.macMask);
#else
	vlanp->macAddrNumber = entry.macNotExist?0:(1<<entry.macMask);
#endif
	vlanp->mtu = (entry.mtuH<<8) | entry.mtuL;
	return SUCCESS;
}


/*=========================================
  * ASIC DRIVER API: ASIC Counter
  *=========================================*/
#define RTL865X_ASIC_DRIVER_COUNTER_API

static void _rtl8651_initialRead(void) {//RTL8651 read counter for the first time will get value -1 and this is meaningless
	uint32 i;
	for(i=0; i<=0xac; i+=0x4)
		rtl8651_returnAsicCounter(i);
}

int32 rtl8651_returnAsicCounter(uint32 offset) {
	if(offset & 0x3)
		return 0;
	return  READ_MEM32(MIB_COUNTER_BASE + offset);
}

int32 rtl8651_clearAsicCounter(void) {
	uint32 reg;
	//Clear mib counter and enable MIB counter as system counter
	reg = READ_MEM32(MIB_CONTROL) & 0x3FE00000;
	WRITE_MEM32(MIB_CONTROL,IN_COUNTER_RESTART | OUT_COUNTER_RESTART);
	WRITE_MEM32(MIB_CONTROL, reg);
	reg = READ_MEM32(MIB_CONTROL_1);
	WRITE_MEM32(MIB_CONTROL_1,  IN_COUNTER_RESTART | OUT_COUNTER_RESTART);
	WRITE_MEM32(MIB_CONTROL_1, reg);
	reg = READ_MEM32(MIB_CONTROL_2);
	WRITE_MEM32(MIB_CONTROL_2, IN_COUNTER_RESTART | OUT_COUNTER_RESTART);
	WRITE_MEM32(MIB_CONTROL_2, reg);
	_rtl8651_initialRead();
	return SUCCESS;
}

/*
@func int32 | rtl8651_clearAsicCounter | Clear specified ASIC counter to zero
@parm	uint32	|	counterIdx |  Specify the counter to clear
@rvalue SUCCESS | When counter index is valid, return SUCCESS
@rvalue FAILED | When counter index is invalid, return FAILED
@comm
	When specify a vlid counter, the corresponding counter will be reset to zero and read the counter once
	for guarantee following read can get correct value. 
*/
int32 rtl8651_clearAsicSpecifiedCounter(uint32 counterIdx) {
	uint32 reg;
	switch(counterIdx) {
		case 0:
		reg = READ_MEM32(MIB_CONTROL) & 0x3FE00000;
		WRITE_MEM32(MIB_CONTROL,IN_COUNTER_RESTART | OUT_COUNTER_RESTART);
		WRITE_MEM32(MIB_CONTROL, reg);
		break;
		case 1:
		reg = READ_MEM32(MIB_CONTROL_1) & 0x3FE00000;
		WRITE_MEM32(MIB_CONTROL_1,  IN_COUNTER_RESTART | OUT_COUNTER_RESTART);
		WRITE_MEM32(MIB_CONTROL_1, reg);
		break;
		case 2:
		reg = READ_MEM32(MIB_CONTROL_2) & 0x3FE00000;
		WRITE_MEM32(MIB_CONTROL_2, IN_COUNTER_RESTART | OUT_COUNTER_RESTART);
		WRITE_MEM32(MIB_CONTROL_2, reg);
		break;
		default:	
			return FAILED;//counter index out of range

	}
	_rtl8651_initialRead();
	return SUCCESS;
}

/*
@func int32 | rtl8651_resetAsicCounterMemberPort | Clear the specified counter value and its member port
@parm	uint32	|	counterIdx |  Specify the counter to clear
@rvalue SUCCESS | When counter index is valid, return SUCCESS
@rvalue FAILED | When counter index is invalid, return FAILED
@comm
	When specify a vlid counter, the member port of the specified counter will be cleared to null set. 
*/
int32 rtl8651_resetAsicCounterMemberPort(uint32 counterIdx){

	switch(counterIdx) {
		case 0:
		WRITE_MEM32(MIB_CONTROL, 0x0);
		break;
		case 1:
		WRITE_MEM32(MIB_CONTROL_1, 0x0);
		break;
		case 2:
		WRITE_MEM32(MIB_CONTROL_2, 0x0);
		break;
		default:	
			return FAILED;//counter index out of range

	}
	_rtl8651_initialRead();
	return SUCCESS;
}

/*
@func int32 | rtl8651_addAsicCounterMemberPort | The specified counter value add the specified port port into counter monitor member
@parm	uint32	|	counterIdx |  Specify the counter to add member port
@parm	uint32	|	port |  The added member port
@rvalue SUCCESS | When counter index is valid, return SUCCESS
@rvalue FAILED | When counter index is invalid, return FAILED
@comm
	When specify a vlid counter and a valid port number, the specified port will be added to the counter coverage. 
*/
int32 rtl8651_addAsicCounterMemberPort(uint32 counterIdx, uint32 port) {
	uint32 reg, portMask;

	if(port <RTL8651_PORT_NUMBER)
		portMask = 1<<(port + PORT_FOR_COUNTING_OFFSET);
	else if (port < RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum)
		portMask = 1<<(port - RTL8651_PORT_NUMBER + EXT_PORT_FOR_COUNTING_OFFSET);
	else
		return FAILED;//Port number out of range
	switch(counterIdx) {
		case 0:
		reg = READ_MEM32(MIB_CONTROL) & 0x3FE00000;
		WRITE_MEM32(MIB_CONTROL, reg|portMask);
		break;
		case 1:
		reg = READ_MEM32(MIB_CONTROL_1) & 0x3FE00000;
		WRITE_MEM32(MIB_CONTROL_1, reg|portMask);
		break;
		case 2:
		reg = READ_MEM32(MIB_CONTROL_2) & 0x3FE00000;
		WRITE_MEM32(MIB_CONTROL_2, reg|portMask);
		break;
		default:	
			return FAILED;//counter index out of range
	}
	return SUCCESS;
}


/*
@func int32 | rtl8651_resetAsicCounterMemberPort | Clear the specified counter value and its member port
@parm	uint32	|	counterIdx |  Specify the counter to clear
@rvalue SUCCESS | When counter index is valid, return SUCCESS
@rvalue FAILED | When counter index is invalid, return FAILED
@comm
	When specify a vlid counter, the corresponding counter will be reset to zero, its member port will be cleared to null set and read the counter once
	for guarantee following read can get correct value. 
*/
int32 rtl8651_delAsicCounterMemberPort(uint32 counterIdx, uint32 port) {
	uint32 reg, portMask;

	if(port <RTL8651_PORT_NUMBER)
		portMask = 1<<(port + PORT_FOR_COUNTING_OFFSET);
	else if (port < RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum)
		portMask = 1<<(port - RTL8651_PORT_NUMBER + EXT_PORT_FOR_COUNTING_OFFSET);
	else
		return FAILED;//Port number out of range
	switch(counterIdx) {
		case 0:
		reg = READ_MEM32(MIB_CONTROL) & 0x3FE00000;
		WRITE_MEM32(MIB_CONTROL, reg&~portMask);
		break;
		case 1:
		reg = READ_MEM32(MIB_CONTROL_1) & 0x3FE00000;
		WRITE_MEM32(MIB_CONTROL_1, reg&~portMask);
		break;
		case 2:
		reg = READ_MEM32(MIB_CONTROL_2) & 0x3FE00000;
		WRITE_MEM32(MIB_CONTROL_2, reg&~portMask);
		break;
		default:	
			return FAILED;//counter index out of range
	}
	return SUCCESS;
}


/*
@func int32 | rtl8651_resetAsicCounterMemberPort | Clear the specified counter value and its member port
@parm	uint32	|	counterIdx |  Specify the counter to clear
@rvalue SUCCESS | When counter index is valid, return SUCCESS
@rvalue FAILED | When counter index is invalid, return FAILED
@comm
	When specify a vlid counter, the corresponding counter will be reset to zero, its member port will be cleared to null set and read the counter once
	for guarantee following read can get correct value. 
*/
int32 rtl8651_getAsicCounter(uint32 counterIdx, rtl865x_tblAsicDrv_basicCounterParam_t * basicCounter) {

	_rtl8651_initialRead();
	switch(counterIdx) {
		case 0:
				basicCounter->rxBytes = READ_MEM32(MIB_COUNTER_BASE);
				basicCounter->rxPackets = READ_MEM32(MIB_COUNTER_BASE+0x14) +	//Unicast
						READ_MEM32(MIB_COUNTER_BASE+0x18) + 	//Multicast
						READ_MEM32(MIB_COUNTER_BASE+0x1c);	//Broadcast
				basicCounter->rxErrors = READ_MEM32(MIB_COUNTER_BASE+0x8) +	//CRC error and Alignment error
						READ_MEM32(MIB_COUNTER_BASE+0xc) +	//Fragment error
						READ_MEM32(MIB_COUNTER_BASE+010);	//Jabber error
				basicCounter->drops = READ_MEM32(MIB_COUNTER_BASE+0x4);
				basicCounter->cpus = READ_MEM32(MIB_COUNTER_BASE+0x74);
				basicCounter->txBytes = READ_MEM32(MIB_COUNTER_BASE+0x48);
				basicCounter->txPackets = READ_MEM32(MIB_COUNTER_BASE+0x4c) +	//Unicast
						READ_MEM32(MIB_COUNTER_BASE+0x50) +	//Multicast
						READ_MEM32(MIB_COUNTER_BASE+0x54);	//Broadcast
				basicCounter->mbr = (READ_MEM32(MIB_CONTROL)&PORT_FOR_COUNTING_MASK)>>PORT_FOR_COUNTING_OFFSET | 
								((READ_MEM32(MIB_CONTROL)&EXT_PORT_FOR_COUNTING_MASK)>>EXT_PORT_FOR_COUNTING_OFFSET)<<6;
		break;
		case 1:
				basicCounter->rxBytes = READ_MEM32(MIB_COUNTER_BASE+0x80);
				basicCounter->rxPackets = READ_MEM32(MIB_COUNTER_BASE+0x84);
				basicCounter->rxErrors = READ_MEM32(MIB_COUNTER_BASE+0x88);
				basicCounter->drops = READ_MEM32(MIB_COUNTER_BASE+0x88);
				basicCounter->cpus = READ_MEM32(MIB_COUNTER_BASE+0x8c);
				basicCounter->txBytes = READ_MEM32(MIB_COUNTER_BASE+0xa0);
				basicCounter->txPackets = READ_MEM32(MIB_COUNTER_BASE+0xa4);
				basicCounter->mbr = (READ_MEM32(MIB_CONTROL_1)&PORT_FOR_COUNTING_MASK)>>PORT_FOR_COUNTING_OFFSET | 
								((READ_MEM32(MIB_CONTROL_1)&EXT_PORT_FOR_COUNTING_MASK)>>EXT_PORT_FOR_COUNTING_OFFSET)<<6;
		break;
		case 2:
				basicCounter->rxBytes = READ_MEM32(MIB_COUNTER_BASE+0x90);
				basicCounter->rxPackets = READ_MEM32(MIB_COUNTER_BASE+0x94);
				basicCounter->rxErrors = READ_MEM32(MIB_COUNTER_BASE+0x98);
				basicCounter->drops = READ_MEM32(MIB_COUNTER_BASE+0x98);
				basicCounter->cpus = READ_MEM32(MIB_COUNTER_BASE+0x9c);
				basicCounter->txBytes = READ_MEM32(MIB_COUNTER_BASE+0xa8);
				basicCounter->txPackets = READ_MEM32(MIB_COUNTER_BASE+0xac);
				basicCounter->mbr = (READ_MEM32(MIB_CONTROL_2)&PORT_FOR_COUNTING_MASK)>>PORT_FOR_COUNTING_OFFSET | 
								((READ_MEM32(MIB_CONTROL_2)&EXT_PORT_FOR_COUNTING_MASK)>>EXT_PORT_FOR_COUNTING_OFFSET)<<6;
		break;
		default:	
			return FAILED;//counter index out of range

	}
	return SUCCESS;
}

void rtl8651_updateLinkChangePendingCount(void) {
	linkChangePendingCount++;
}



/*=========================================
  * ASIC DRIVER API: LINK STATUS
  *=========================================*/
#define RTL865X_ASIC_DRIVER_LINK_STATUS_API

static void (*linkChangeNotifier)(uint32, int8) = NULL;

int32 rtl8651_regLinkChangeCallBackFun(void (*callBackFun)(uint32, int8))
{
	linkChangeNotifier = callBackFun;
	return SUCCESS;
}


int32 rtl8651_setAsicEthernetLinkStatus(uint32 port, int8 linkUp) {
	int8 notify;
	if(port >= RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum)
		return FAILED;
	if(rtl8651AsicEthernetTable[port].linkUp != (linkUp==TRUE? 1: 0)) 
		notify = TRUE;
	else
		notify = FALSE;
	rtl8651AsicEthernetTable[port].linkUp = linkUp == TRUE? 1: 0;
	
	//Add/remove port from asic aggregator if this port is a member
	rtl8651_updateAsicLinkAggregatorLMPR(linkUp?port: -port);
#ifndef CONFIG_RTL865X_LIGHT_ROMEDRV	
	if(notify==TRUE)
		rtl8651_setEthernetPortLinkStatus(port, linkUp);
#else
	/*
	 * Hard code to notify upper layer is an ugly way. Recommend to use callback function
	 * registration to achieve link-change notification.
	 */
	if (notify == TRUE && linkChangeNotifier)
		linkChangeNotifier(port, linkUp);

#endif
	return SUCCESS;
}

int32 rtl8651_getAsicEthernetLinkStatus(uint32 port, int8 *linkUp) {
	if(port >= (RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum) || linkUp == NULL)
		return FAILED;
	*linkUp = rtl8651AsicEthernetTable[port].linkUp == 1? TRUE: FALSE;
	return SUCCESS;
}


uint32 prePortState[RTL8651_PHY_NUMBER];

int32 rtl8651_updateLinkStatus(void) {

	/* =====================================
		For RTL865xB cut-d & after that.
	    =====================================*/
	if (RtkHomeGatewayChipRevisionID >= 0x02)
	{
		uint32 i, patchReg, fullDuplex;


		if(linkChangePendingCount>0) {
			linkChangePendingCount = 0;
			/*
			  * The following disabled for-loop is for 8650/8651 bug fixed, now we remove it. 
			  * But we still need to read PHY Reg. 1 again.  This is because the bit n.2 of PHY Reg. 1 (link status)
			  * has a behavior of holding old link status. Hence twice reading is needed.
			  */
			//for(patchLoop=0; patchLoop<100; patchLoop++){
				for(i=0; i<RTL8651_PHY_NUMBER; i++) //rtl8651 only have 5 phy
					patchReg = READ_MEM32(PHY_BASE + (i<<5) + 4);
				patchReg &=0; //dummy line to avoid compile warning
			//}

			for(i=0; i<RTL8651_PHY_NUMBER; i++) {
				uint32 dummy;
#ifndef CONFIG_RTL865X_LIGHT_ROMEDRV
				int32 flag=FALSE;
				
				rtl8651_getAsicEthernetPHYLoopback(i, &flag);
				if(flag==TRUE){
					continue; //don't touch loopbacked ports since it is manually configured by user
				}
#endif
				
				dummy = READ_MEM32(PHY_BASE+(i<<5) + 0x4); /* To get the correct link status, this register must be read twice. */
				if(READ_MEM32(PHY_BASE+(i<<5) + 0x4) & 0x4) 
				{ 
					/* link is up */
					rtl8651_setAsicEthernetLinkStatus(i, TRUE);

					/* Louis notes:
					 *   According to David Lu's command, we modify the decision flow of flow control.
					 *                                                                         2005/03/16
					 */
#if 1
					{
						uint32 ability;

						/* First, we check the full/half status of two party's highest ability.
						 *  ( PHY_CR4[8:5] AND PHY_CR5[8:5] )
						 */
						ability = READ_MEM32(PHY_BASE+(i<<5)+0x10) & READ_MEM32(PHY_BASE+(i<<5)+0x14);
						if ( ability & CAPABLE_100BASE_TX_FD )
							fullDuplex = TRUE;
						else if ( ability & CAPABLE_100BASE_TX_HD )
							fullDuplex = FALSE;
						else if ( ability & CAPABLE_10BASE_TX_FD )
							fullDuplex = TRUE;
						else/* if ( ability & CAPABLE_10BASE_TX_HD )*/
							fullDuplex = FALSE;

						/* Second, enable/disable flow control according to half or full detected. */
						if ( !fullDuplex )
						{
							/* half mode ==> see if back presure is enabled. */
							if ( READ_MEM32(MACCR) & EN_BACK_PRESSURE )
								WRITE_MEM32(FCREN,READ_MEM32(FCREN)|(0x81<<(18+i))); /* Enable Flow Control */
							else
								WRITE_MEM32(FCREN,READ_MEM32(FCREN)&~(0x81<<(18+i))); /* Disable Flow Control */ 
						}
						else
						{
							/* full mode ==> see nWay result. */
							if ( ( READ_MEM32(PHY_BASE+(i<<5)+0x10) & CAPABLE_PAUSE ) &&
							     ( READ_MEM32(PHY_BASE+(i<<5)+0x14) & CAPABLE_PAUSE ) )
								WRITE_MEM32(FCREN,READ_MEM32(FCREN)|(0x81<<(18+i))); /* Enable Flow Control */
							else
								WRITE_MEM32(FCREN,READ_MEM32(FCREN)&~(0x81<<(18+i))); /* Disable Flow Control */ 
						}
					}
						
#else
					if ((READ_MEM32(PHY_BASE + (i<<5)) & 0x1000)) 
					{
						/* Auto-Negotiation is Enabled. */
						fullDuplex = (READ_MEM32(PHY_BASE + (i<<5)) & 0x100);
						if (fullDuplex || ((!fullDuplex)&&(READ_MEM32(MACCR)&EN_BACK_PRESSURE))) {
							/* Whether enable or disable flow control depends on the result of (Reg4.10 & Reg5.10). */
							if ((READ_MEM32(PHY_BASE+(i<<5) + 0x14) & 0x400) && (READ_MEM32(PHY_BASE+(i<<5) + 0x10) & 0x400))
								/* Enable Flow Control */
								WRITE_MEM32(FCREN,READ_MEM32(FCREN)|(0x81<<(18+i)));
							else /* Disable Flow Control */ 
								WRITE_MEM32(FCREN,READ_MEM32(FCREN)&~(0x81<<(18+i)));
						}
						else /* Disable Flow Control */ 
							WRITE_MEM32(FCREN,READ_MEM32(FCREN)&~(0x81<<(18+i)));
					}
					else /* Auto-Negotiation is Disabled, alwlays disabled Flow Control */
						WRITE_MEM32(FCREN,READ_MEM32(FCREN)&~(0x81<<(18+i)));
#endif
	 

#if 0				
					if((READ_MEM32(PHY_BASE + (i<<5)) & 0x1000) && (READ_MEM32(PHY_BASE+(i<<5) + 0x14) & 0x400)) {//Auto-negotiation enabled and Link partner can enable flow control
						WRITE_MEM32(FCREN,READ_MEM32(FCREN)|(0x81<<(18+i)));
					} 
					else if((READ_MEM32(PHY_BASE + (i<<5)) & 0x100) == 0 && READ_MEM32(MACCR)&EN_BACK_PRESSURE)// Half duplex mode no matter Auto-negotiation enabled or not 
						WRITE_MEM32(FCREN,READ_MEM32(FCREN)|(0x81<<(18+i)));
					else
						WRITE_MEM32(FCREN,READ_MEM32(FCREN)&~(0x81<<(18+i)));
#endif
						
				}
				else 
				{
					/* link is down */
					rtl8651_setAsicEthernetLinkStatus(i, FALSE);
					WRITE_MEM32(FCREN,READ_MEM32(FCREN)&~(0x81<<(18+i)));
				}
			}

#if 1 /* Louis notes: Mii will never link up/down in run-time. */
			//check the MII interface
			if(miiPhyAddress>=0){//MII Phy address already configured.
				uint32 status=READ_MEM32(MDCIOSR);
				if((status&0x100000)){ //link is up
					uint32 partner;
					rtl8651_setAsicEthernetLinkStatus(RTL8651_PHY_NUMBER, TRUE);
					WRITE_MEM32(MDCIOCR,miiPhyAddress<<24|5<<16); //read PHY reg 5 on MII
#ifndef RTL865X_TEST
					while((READ_MEM32(MDCIOSR)&0x80000000)!=0); //wait until commd finished.
#endif /* RTL865X_TEST */
					partner=READ_MEM32(MDCIOSR);
					if((status&0x800000) && (partner & 0x400))//Auto-negotiation enabled and Link partner can enable flow control
						WRITE_MEM32(FCREN,READ_MEM32(FCREN)|(0x81<<(18+RTL8651_PHY_NUMBER)));
					else if((status&0x04) == 0 && READ_MEM32(MACCR)&EN_BACK_PRESSURE)// Half duplex mode no matter Auto-negotiation enabled or not 
						WRITE_MEM32(FCREN,READ_MEM32(FCREN)|(0x81<<(18+RTL8651_PHY_NUMBER)));
					//else if((READ_MEM32(PHY_BASE + (i<<5)) & 0x100) == 1 && (READ_MEM32(PHY_BASE+(i<<5) + 0x14) & 0x400)) //forced mode
					else
						WRITE_MEM32(FCREN,READ_MEM32(FCREN)&~(0x81<<(18+RTL8651_PHY_NUMBER)));
				}else{
					rtl8651_setAsicEthernetLinkStatus(RTL8651_PHY_NUMBER, FALSE);
					WRITE_MEM32(FCREN,READ_MEM32(FCREN)&~(0x81<<(18+RTL8651_PHY_NUMBER)));
				}
			}
#endif

			return SUCCESS;
		}
		return FAILED;//No link change
	}
	/* =====================================
		For RTL865xB cut-b & before that.
	    =====================================*/
	else
	{
		uint32 port, REG0, value;

		
#define DEBIG_INFO(p, v)
#define FLOW_CTRL(v)		( (READ_MEM32(PHY_BASE+(v<<5)+0x10)&(1<<10)) && (READ_MEM32(PHY_BASE+(v<<5)+0x14)&(1<<10)) )
#define FULL_DUPLEX(v)	( v&(1<<6) )
#define IS_100MBPS(v)	( v&(1<<5) )
#define LINK_UP(v)		(  v&(1<<4) )



		if (linkChangePendingCount == 0)
			return FAILED; /* No link change */

		linkChangePendingCount = 0;	
		for(port=0; port<RTL8651_PHY_NUMBER; port++) {
			value = (port==4)? READ_MEM32(PSIR1): (READ_MEM32(PSIR0)>>(port*8));
			if ((LINK_UP(value)>>4) != prePortState[port]) { /* check link status */
				/* link status=0: link change */
				if ( (READ_MEM32(PHY_BASE+(port<<5))&(1<<12))==0 ){ /* This port was set to force mode */
					if (LINK_UP(value) == 0)
						rtl8651_setAsicEthernetLinkStatus(port, FALSE);
					else rtl8651_setAsicEthernetLinkStatus(port, TRUE);
					/* Enable N-Way & Restart N-Way */
					WRITE_MEM32(SWTAA, (uint32)(PHY_BASE + (port<<5)) );
					WRITE_MEM32(TCR0, (READ_MEM32((PHY_BASE+(port<<5))) | (1<<12)));
					WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);//Activate add command
					while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
					WRITE_MEM32(SWTAA, (uint32)(PHY_BASE + (port<<5)) );
					WRITE_MEM32(TCR0, (READ_MEM32((PHY_BASE+(port<<5))) | (1<<9)));
					WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);//Activate add command
					while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
					DEBIG_INFO(port, 5);
					prePortState[port] = 0;
				}
				else {
					if (LINK_UP(value)) {
						rtl8651_setAsicEthernetLinkStatus(port, TRUE);
						//value = (port==4)? READ_MEM32(PSIR1): (READ_MEM32(PSIR0)>>(port*8));

						if ( FULL_DUPLEX(value) && (FLOW_CTRL(port)==0) ) {
							/* set to force mode */
							REG0 = READ_MEM32((PHY_BASE+(port<<5))) & ~(1<<12);
							REG0 = FULL_DUPLEX(value)? (REG0|(1<<8)) : (REG0&~(1<<8));
							REG0 = IS_100MBPS(value)?  (REG0|(1<<13)): (REG0&~(1<<13));
							WRITE_MEM32(SWTAA, (uint32)(PHY_BASE + (port<<5)) );
							WRITE_MEM32(TCR0, REG0);
							WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);//Activate add command
							while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done

							WRITE_MEM32(SWTAA, (uint32)(PHY_BASE + (port<<5)+0x14) );
							WRITE_MEM32(TCR0, READ_MEM32((PHY_BASE+(port<<5))+0x14)|(1<<10));
							WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);//Activate add command
							while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
							DEBIG_INFO(port, 4);
							prePortState[port] = 1;
						} else {DEBIG_INFO(port, 3); prePortState[port] =1;}
					}
					else {
						rtl8651_setAsicEthernetLinkStatus(port, FALSE);
						DEBIG_INFO(port, 2);
						prePortState[port] = 0;
					}
				}
				//prePortState[port]=LINK_UP(value)?1: 0;
			}
			else {
				//prePortState[port]=LINK_UP(value)?1:0;
				if (prePortState[port]) 	
					rtl8651_setAsicEthernetLinkStatus(port, TRUE);
				else rtl8651_setAsicEthernetLinkStatus(port, FALSE);
				DEBIG_INFO(port, 1);

			}
		}

		/* check the MII interface */
		if ( miiPhyAddress >= 0 )
		{
			uint32 status = READ_MEM32(MDCIOSR);
			if ( (status & 0x100000 ) )	/* link is up : Bit.21 of MDC/MDIO status register shows link status of mii port */
			{
				rtl8651_setAsicEthernetLinkStatus( RTL8651_MII_PORTNUMBER, TRUE );
				/* always turn on flow control */
				WRITE_MEM32( FCREN, READ_MEM32(FCREN) | ( 0x81 << ( 18 + RTL8651_MII_PORTNUMBER ) ) );
			}
			else
			{
				rtl8651_setAsicEthernetLinkStatus( RTL8651_MII_PORTNUMBER, FALSE );
				WRITE_MEM32( FCREN, READ_MEM32(FCREN) & ~( 0x81<<( 18 + RTL8651_MII_PORTNUMBER ) ) );
			}
		}

		return SUCCESS;

	}
}



/*=========================================
  * ASIC DRIVER API: SYSTEM INIT
  *=========================================*/
#define RTL865X_ASIC_DRIVER_SYSTEM_INIT_API

static void _rtl8651_clearSpecifiedAsicTable(uint32 type, uint32 count) {
	rtl8651_tblAsic_aclTable_t entry;
	uint32 idx;
	
	bzero(&entry, sizeof(entry));
	for (idx=0; idx<count; idx++)// Write into hardware
		_rtl8651_forceAddAsicEntry(type, idx, &entry);

}

int32 rtl8651_clearAsicAllTable(void) {
	_rtl8651_clearSpecifiedAsicTable(TYPE_L2_SWITCH_TABLE, RTL8651_L2TBL_ROW*RTL8651_L2TBL_COLUMN);
       _rtl8651_clearSpecifiedAsicTable(TYPE_ARP_TABLE, RTL8651_ARPTBL_SIZE);
       _rtl8651_clearSpecifiedAsicTable(TYPE_L3_ROUTING_TABLE, RTL8651_ROUTINGTBL_SIZE);
       _rtl8651_clearSpecifiedAsicTable(TYPE_MULTICAST_TABLE, RTL8651_IPMULTICASTTBL_SIZE);
#ifndef CONFIG_RTL865X_LIGHT_ROMEDRV
	   _rtl8651_clearSpecifiedAsicTable(TYPE_PROTOCOL_TRAP_TABLE, RTL8651_PROTOCOLTRAP_SIZE);
#endif
       _rtl8651_clearSpecifiedAsicTable(TYPE_VLAN_TABLE, RTL8651_VLAN_NUMBER);
       _rtl8651_clearSpecifiedAsicTable(TYPE_EXT_INT_IP_TABLE, RTL8651_IPTABLE_SIZE);
       _rtl8651_clearSpecifiedAsicTable(TYPE_ALG_TABLE, RTL8651_ALGTBL_SIZE);
       _rtl8651_clearSpecifiedAsicTable(TYPE_SERVER_PORT_TABLE, RTL8651_SERVERPORTTBL_SIZE);
       _rtl8651_clearSpecifiedAsicTable(TYPE_L4_TCP_UDP_TABLE, RTL8651_TCPUDPTBL_SIZE);
       _rtl8651_clearSpecifiedAsicTable(TYPE_L4_ICMP_TABLE, RTL8651_ICMPTBL_SIZE);
       _rtl8651_clearSpecifiedAsicTable(TYPE_PPPOE_TABLE, RTL8651_PPPOE_NUMBER);
       _rtl8651_clearSpecifiedAsicTable(TYPE_ACL_RULE_TABLE, RTL8651_ACLTBL_SIZE);
       _rtl8651_clearSpecifiedAsicTable(TYPE_NEXT_HOP_TABLE, RTL8651_NEXTHOPTBL_SIZE);
       _rtl8651_clearSpecifiedAsicTable(TYPE_RATE_LIMIT_TABLE, RTL8651_RATELIMITTBL_SIZE);

	return SUCCESS;
}

unsigned int GetSysClockRateForAsicDrv(void)
{
	unsigned int SysClkRate;
	
    REG32(MACMR) = REG32(MACMR) & ~SYS_CLK_MASK;
    switch ( REG32(MACMR) & SYS_CLK_MASK )
    {
        case SYS_CLK_100M:
            SysClkRate = 100000000;
            break;
        case SYS_CLK_90M:
            SysClkRate = 90000000;
            break;
        case SYS_CLK_85M:
            SysClkRate = 85000000;
            break;
        case SYS_CLK_96M:
            SysClkRate = 96000000;
            break;
        case SYS_CLK_80M:
            SysClkRate = 80000000;
            break;
        case SYS_CLK_75M:
            SysClkRate = 75000000;
            break;
        case SYS_CLK_70M:
            SysClkRate = 70000000;
            break;
        case SYS_CLK_50M:
            SysClkRate = 50000000;
            break;
        default:
            while(1);
    }

    return SysClkRate;
}

/*cfliu: Remove legacy patch code for 8650Bv1 !!*/
#if 0

/*
 * Configure Timer 1 to be a counter to emulate 10ms.
 */

#define DIVISOR 4096   /*change from 0xe to 4096 for watchdog*/
#define HZ 100
void tick_Delay10ms( uint32 count )
{
#ifndef RTL865X_TEST
	uint32 passed;
	uint32 timerData = (((GetSysClockRateForAsicDrv()/DIVISOR)/HZ)+1) << TCD_OFFSET;
	
	for( passed = 0; passed < count; passed++ )
	{
		/* set counter initial value */
		WRITE_MEM32( TC1DATA, timerData );	
		
		/* Enable timer1 */
		WRITE_MEM32( TCCNR, READ_MEM32(TCCNR) | TC1EN | TC1MODE_COUNTER );

		/* Wait until the counter reaches */
		while( READ_MEM32( TC1CNT ) >= ( 2 << TCD_OFFSET ) ); /* The counter will stop counting at 1. */

		/* Disable timer1 */
		WRITE_MEM32( TCCNR, READ_MEM32(TCCNR) & ~TC1EN );
	}
#endif
}

void tick_Delay100ms( uint32 count )
{
	tick_Delay10ms( count * 10 );
}



void FullAndSemiReset( void )
{
	int c;

	/* Perform full-reset for sw-core. */
	WRITE_MEM32( MACCR, READ_MEM32(MACCR) | FULL_RST );

	/* Wait BISTing bit become 1. */
	c = 0;
	while( ((READ_MEM32( BISTCR )) & 0x40000000) == 0 )
	{
		tick_Delay10ms(1);
		c++;
		if(c>100) break;
	}

	/* Wait BISTing bit become 0. */
	c = 0;
	while( READ_MEM32( BISTCR ) & 0x40000000 )
	{
		tick_Delay10ms(1);
		c++;
		if(c>100) break;
	}

	/* Delay 100ms after BISTing bit became 0. */
	tick_Delay100ms(1);

	/* Disable TRXRDY */
	WRITE_MEM32( BISTCR, READ_MEM32( BISTCR ) & ~TRXRDY);

	/* Semi-Reset */
	WRITE_MEM32( MACCR, READ_MEM32( MACCR ) | SEMI_RST );

	/* Enable TRXRDY */
	WRITE_MEM32( BISTCR, READ_MEM32( BISTCR ) | TRXRDY );
	
	/* Wait QOK and COK bit all become 1. */
	c = 0;
	while( ( READ_MEM32( BISTCR ) & 0x30000000 ) != 0x30000000 )
	{
		tick_Delay10ms(1);
		c++;
		if(c>100) break;
	}

	/* clear bits */
	REG32(MACCR) &=  ~(FULL_RST|SEMI_RST);
}
#endif


#if defined(RTL865X_TEST) || defined(RTL865X_MODEL_USER) || defined(RTL865X_MODEL_KERNEL)
int32 _rtl8651_mapToVirtualRegSpace( void )
{
	static int8 virtualSwReg[VIRTUAL_SWCORE_REG_SIZE*2];
	static int8 virtualSysReg[VIRTUAL_SYSTEM_REG_SIZE*2];
	static int8 virtualHsb[HSB_SIZE*2];
	static int8 virtualHsa[HSA_SIZE*2];
	static int8 virtualSwTable[VIRTUAL_SWCORE_TBL_SIZE];
	
	pVirtualSWReg = (int8*)(((uint32)virtualSwReg+VIRTUAL_SWCORE_REG_SIZE-1)&~(VIRTUAL_SWCORE_REG_SIZE-1));
	pVirtualSysReg = (int8*)(((uint32)virtualSysReg+VIRTUAL_SYSTEM_REG_SIZE-1)&~(VIRTUAL_SYSTEM_REG_SIZE-1));
	pVirtualHsb = virtualHsb;
	pVirtualHsa = virtualHsa;
	pVirtualSWTable = virtualSwTable;
	
	return SUCCESS;
}

int32 _rtl8651_mapToRealRegSpace( void )
{
	pVirtualSWReg = (int8*)REAL_SWCORE_BASE;
	pVirtualSysReg = (int8*)REAL_BD01_BASE;
	pVirtualHsb = (int8*)REAL_HSB_BASE;
	pVirtualHsa = (int8*)REAL_HSA_BASE;
	pVirtualSWTable = (int8*)REAL_SWTBL_BASE;
	return SUCCESS;
}
#endif

/*
	Initiate ASIC configurating parameter.
*/
static int32 _rtl8651_initAsicPara( rtl8651_tblAsic_InitPara_t *para )
{
	memset(&rtl8651_tblAsicDrvPara, 0, sizeof(rtl8651_tblAsic_InitPara_t));

	if ( para )
	{
		/* Parameter != NULL, check its correctness */

		/* NOTHING yet */

		/* ============= Check passed : set it =============  */
		rtl8651_memcpy(&rtl8651_tblAsicDrvPara, para, sizeof(rtl8651_tblAsic_InitPara_t));
	}

	return SUCCESS;
}

int32 rtl8651_getChipVersion(int8 *name,uint32 size, int32 *rev)
{

	uint32 id = READ_MEM32(CRMR);
	if(!id){
		id = READ_MEM32(CHIPID);
		if((uint16)(id>>16)==0x8650)
			strncpy(name,"8650",size);
		else
			strncpy(name,"8651",size);
		if(rev)
			*rev=0;
	}else if((uint16)id==0x5788){
		int revId;
		revId = (id>>16)&0xf;
#if 1
		/* chenyl: in 865xB rev.C, we can't and don't distinguish the difference between 8650B and 8651B */
		if (revId >= 2 /* rev.C */)
		{
			strncpy(name, "865xB", size);
		}else
#endif
		/* in 865xB rev.A and rev.B, we can distinguish the difference between 8650B and 8651B */
		{
			if(id&0x02000000)
				strncpy(name,"8651B",size);
			else
				strncpy(name,"8650B",size);
		}

		if(rev)
			*rev=revId;
	}else
		snprintf(name, size, "%08x", id);
	//rtlglue_printf("Read CRMR=%08x, CHIP=%08x\n", READ_MEM32(CRMR),READ_MEM32(CHIPID));
	return SUCCESS;
}

int32 rtl8651_initAsic(rtl8651_tblAsic_InitPara_t *para) 
{
	int32 index;
	uint16 flowTblIdx;
	rtl865x_tblAsicDrv_naptTcpUdpParam_t naptTcpUdp;
	rtl865x_tblAsicDrv_naptIcmpParam_t naptIcmp;

#if defined(RTL865X_TEST)
	char chipVersion[16];
	int rev;

	_rtl8651_mapToVirtualRegSpace();
	rtl8651_getChipVersion(chipVersion, sizeof(chipVersion), &rev);
	bzero(pVirtualSWReg, VIRTUAL_SWCORE_REG_SIZE);
	bzero(pVirtualSysReg, VIRTUAL_SYSTEM_REG_SIZE);
	bzero(pVirtualHsb, HSB_SIZE);
	bzero(pVirtualHsa, HSA_SIZE);
	bzero(pVirtualSWTable, VIRTUAL_SWCORE_TBL_SIZE);
	rtl8651_setChipVersion(chipVersion,  &rev);

#elif defined(RTL865X_MODEL_USER)
	_rtl8651_mapToVirtualRegSpace(); /* for cleshell.c:main() only ! */
	modelIcSetDefaultValue();
#elif defined(RTL865X_MODEL_KERNEL)
#endif

	ASICDRV_INIT_CHECK(_rtl8651_initAsicPara(para));

	rtl8651_getChipVersion(RtkHomeGatewayChipName, sizeof(RtkHomeGatewayChipName), &RtkHomeGatewayChipRevisionID);
	rtl8651_getChipNameID(&RtkHomeGatewayChipNameID); /* Get the chip ID for further process */

#ifndef RTL865X_TEST
	rtlglue_printf("chip name: %s, chip revid: %d\n", RtkHomeGatewayChipName, RtkHomeGatewayChipRevisionID);
#endif


	/* 
	  * Turn on ENFRAG2ACLPT for Rate Limit. For those packets which need to be trapped to CPU, we turn on
	  * this bit to tell ASIC ACL and Protocol Trap to process these packets. If this bit is not turnned on, packets
	  * which need to be trapped to CPU will not be processed by ASIC ACL and Protocol Trap.
	  * NOTE: 	If this bit is turned on, the backward compatible will disable.
	  *																- chhuang
	  */
	/* 	2006.3.8
		We also turn on bit.29 (ENATT2LOG). 

		* Current implementation of unnumbered pppoe in multiple session
		When wan type is multiple-session, and one session is unnumbered pppoe, WAN to unnumbered LAN is RP --> NPI.
		And user adds ACL rule to trap dip = unnumbered LAN to CPU.

		However, when pktOpApp of this ACL rule is set, it seems that this toCPU ACL does not work.
		Therefore, we turn on this bit (ENATT2LOG) to trap pkts (WAN --> unnumbered LAN) to CPU.
		
	*/
	WRITE_MEM32(SWTECR, 0x60000000);

#if 0	// RTL865xB d-cut testing hash-1
	// ---- Testing ----
	WRITE_MEM32(SWTECR, 0xC0000000);
	// ------------------
#endif
	
	/* 
	  * Cannot turn on EnNAP8651B due to:
	  * If turn on, NAT/LP/ServerPort will reference nexthop. This will result in referecing wrong L2 entry when
	  * the destination host is in the same subnet as WAN.
	  */

//#if 0	// If testing RTL865xB d-cut testing hash-1, need to mark this line.
	WRITE_MEM32(SWTECR, (READ_MEM32(SWTECR) & 0x7fffffff));
//#endif
	
	/*Although chip is in 8650 compatible mode, 
	some 865XB features are independent to compatibility register*/
	/*Initialize them here if needed*/

	{
		unsigned int rev;
		char chipVersion[16];
		rtl8651_getChipVersion(chipVersion, sizeof(chipVersion), &rev);
		if(chipVersion[strlen(chipVersion)-1]=='B'){
			rtl8651_totalExtPortNum=3; //this replaces all RTL8651_EXTPORT_NUMBER defines
			rtl8651_allExtPortMask = 0x7<<RTL8651_MAC_NUMBER; //this replaces all RTL8651_EXTPORTMASK defines
#if !defined(RTL865X_TEST) && !defined(RTL865X_MODEL_USER)
			rtl8651_asicEthernetCableMeterInit();
#endif
		}
		
	}

	memset( &rtl8651AsicEthernetTable[0], 0, ( RTL8651_PORT_NUMBER + rtl8651_totalExtPortNum ) * sizeof(rtl8651_tblAsic_ethernet_t) );

	//Disable layer2, layer3 and layer4 function
	//Layer 2 enabled automatically when a VLAN is added
	//Layer 3 enabled automatically when a network interface is added.
	//Layer 4 enabled automatically when an IP address is setup.
	rtl8651_setAsicOperationLayer(1);
	rtl8651_clearAsicAllTable();//MAY BE OMITTED. FULL_RST clears all tables already.
	rtl8651_setAsicSpanningEnable(FALSE);

#ifdef RTL865XB_DCUT_SWVLAN
	rtl8651_setAsicUnknownVidToCpu(TRUE);
#endif


#ifdef RTL865XB_URL_FILTER
	WRITE_MEM32(SWTECR, READ_MEM32(SWTECR)|(1<<21));//use 8650B's new reason bit definition
#endif

#if !defined(RTL865X_TEST) && !defined(RTL865X_MODEL_USER)
	WRITE_MEM32(0xbd012064,READ_MEM32(0xbd012064)|0xf0000000);//Enable Lexra bus timeout interrupt and timeout limit
	#if 0
	// shliu: Why we set PORT5_PHY_CONTROL to value "0x2c7"?? Should set "0x2c2"?!
	WRITE_MEM32(0xbc800020,0x000002c7);
	WRITE_MEM32(0xbc800008,0xbc8020a0);
	WRITE_MEM32(0xbc800000,0x00000009);
	#endif
#endif
#if 0	/* Don't config LED mode. Leave this responsibility to users. */
	//Init PHY Control Register: port 5
	#if 0	// shliu: Why we set PORT5_PHY_CONTROL to value "0x2c7"?? Should set "0x2c2"?!
	WRITE_MEM32(PORT5_PHY_CONTROL, 0x2c7);
	#endif

	//WRITE_MEM32(PORT5_PHY_CONTROL, 0x2c2);
	/* config LED mode */
	WRITE_MEM32(SWTAA, PORT5_PHY_CONTROL);
	WRITE_MEM32(TCR0, 0x000002C2);
	WRITE_MEM32(SWTACR, CMD_FORCE | ACTION_START); // force add
#endif

	//MAC Control (0xBC803000)
	WRITE_MEM32(MACCR,READ_MEM32(MACCR)&~DIS_IPG);//Set IFG range as 96+-4bit time
	WRITE_MEM32(MACCR,READ_MEM32(MACCR)&~EN_INT_CAM);//disable Internal CAM
	WRITE_MEM32(MACCR,READ_MEM32(MACCR)&~NORMAL_BACKOFF);//Normal backoff
	WRITE_MEM32(MACCR,READ_MEM32(MACCR)&~BACKOFF_EXPONENTIAL_3);//Exponential parameter is 9
	WRITE_MEM32(MACCR,READ_MEM32(MACCR)|INFINITE_PAUSE_FRAMES);//send pause frames infinitely.
	WRITE_MEM32(MACCR,READ_MEM32(MACCR)|DIS_MASK_CGST);
	WRITE_MEM32(MACCR,READ_MEM32(MACCR)&~ACPT_MAXLEN_1552);//Max length 1536
	WRITE_MEM32(MACCR,READ_MEM32(MACCR)&~(EN_FX_P4 | EN_FX_P3 | EN_FX_P2 | EN_FX_P1 | EN_FX_P0));//Disable FX mode (UTP mode)

	/* Initialize MIB counters */
	rtl8651_clearAsicCounter();

	/* Patch for RTL865xB cut-d only: the bit of MII_enForceMode must inverse when read it. */
	if (RtkHomeGatewayChipRevisionID == 0x03) { 
		uint32 value = READ_MEM32(VLANTCR);
		value = ((~value)&MII_ENFORCE_MODE)  | (value&(~MII_ENFORCE_MODE));	// We should NOT bit.4 when reading that reg.
		WRITE_MEM32(VLANTCR, value&~VLAN_TAG_ONLY);
	}
	/* For not RTL865xB cut-d */
	else
		WRITE_MEM32(VLANTCR,READ_MEM32(VLANTCR)&~VLAN_TAG_ONLY);//port based VLAN

	WRITE_MEM32(FCRTH,0x3c203c20);	//David Lu: change to 0x3c203c20 to fix QoS/flow control problem.
	//WRITE_MEM32(FCRTH,0x10081008);	/* Flow-control: 0x20102010 for EnQoS=0, 0x10081008 for EnQoS=1 */
	WRITE_MEM32(FCPTR,0x006c006c);	/* Flow-control prime threshold register*/

	/* For RTL865xB cut-d & after that. */
	if (RtkHomeGatewayChipRevisionID >= 0x02)
	{
		WRITE_MEM32(FCREN,0x810005e0);	/* Enable flow-control: Only enable CPU */
	}
	/* For RTL865xB cut-b & before that. */
	else
	{
		/* Should always enable internal flow control */
	       WRITE_MEM32(FCREN, 0xfffc05e0);
	}

	WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)&~NAPTR_NOT_FOUND_DROP);//When reverse NAPT entry not found, CPU process it.
	rtl8651_setAsicNaptAutoAddDelete(FALSE, TRUE);
#ifdef CONFIG_RTL8316S
	#define EN_UKNOWNVID_TO_CPU	(1 << 6)		/* Enable unknown vlan-tag (vid) to CPU: only exists in RTL865xB d-cut */
	WRITE_MEM32(MSCR, READ_MEM32(MSCR) | (EN_UKNOWNVID_TO_CPU));
#else
	WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)|EN_VLAN_INGRESS_FILTER);
#endif
	WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)&~WAN_ROUTE_MASK);//Set WAN route toEnable (Allow traffic from WAN port to WAN port)
	WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)|NAPTF2CPU);//When packet destination to switch. Just send to CPU
	WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)|EN_MCAST); //Enable multicast table
	WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)&~MCAST_TO_CPU);
	WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)|EN_BCAST);//Enable special broadcast handling
	WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)&~BCAST_TO_CPU);//When EN_BCAST enables, this bit is invalid
	WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)&~BRIDGE_PKT_TO_CPU);//Current no bridge protocol is supported

	WRITE_MEM32(SWTMCR,READ_MEM32(SWTMCR)|(uint32)EN_PPPOE);//enable PPPoE auto encapsulation
	

#ifdef CONFIG_RTL8316S
	WRITE_MEM32( CPUICR, READ_MEM32(CPUICR) & ~( EXCLUDE_CRC) );
	
	WRITE_MEM32(CSCR,READ_MEM32(CSCR)&ALLOW_L2_CHKSUM_ERR);// allow chcksum error pkt be forwarded.
	WRITE_MEM32(CSCR,READ_MEM32(CSCR)&ALLOW_L3_CHKSUM_ERR);
	WRITE_MEM32(CSCR,READ_MEM32(CSCR)&ALLOW_L4_CHKSUM_ERR);
#else
	WRITE_MEM32(CSCR,READ_MEM32(CSCR)&~ALLOW_L2_CHKSUM_ERR);//Don't allow chcksum error pkt be forwarded.	
	WRITE_MEM32(CSCR,READ_MEM32(CSCR)&~ALLOW_L3_CHKSUM_ERR);
	WRITE_MEM32(CSCR,READ_MEM32(CSCR)&~ALLOW_L4_CHKSUM_ERR);
	WRITE_MEM32(CSCR,READ_MEM32(CSCR)|EN_ETHER_L3_CHKSUM_REC); //Enable L3 checksum Re-calculation
	WRITE_MEM32(CSCR,READ_MEM32(CSCR)|EN_ETHER_L4_CHKSUM_REC); //Enable L4 checksum Re-calculation
#endif
	

	WRITE_MEM32(BSCR,0x1|READ_MEM32(BSCR));//Disable broadcast storm prevention mechanism
	WRITE_MEM32(MISCCR,READ_MEM32(MISCCR)&~FRAG2CPU);//IP fragment packet does not need to send to CPU when initial ASIC
	WRITE_MEM32(MISCCR,READ_MEM32(MISCCR)&~MULTICAST_L2_MTU_MASK);
	WRITE_MEM32(MISCCR,READ_MEM32(MISCCR)|L4_HASH);
	WRITE_MEM32(MISCCR,READ_MEM32(MISCCR)|(1522&MULTICAST_L2_MTU_MASK));//Multicast packet layer2 size 1522 at most


	//Hardwired Protocol trap: Default turn off
	WRITE_MEM32(PTRAPCR,READ_MEM32(PTRAPCR)&~EN_ARP_TRAP);
	WRITE_MEM32(PTRAPCR,READ_MEM32(PTRAPCR)&~EN_RARP_TRAP);
	WRITE_MEM32(PTRAPCR,READ_MEM32(PTRAPCR)&~EN_PPPOE_TRAP);
	WRITE_MEM32(PTRAPCR,READ_MEM32(PTRAPCR)&~EN_IGMP_TRAP);
	WRITE_MEM32(PTRAPCR,READ_MEM32(PTRAPCR)&~EN_DHCP_TRAP1);
	WRITE_MEM32(PTRAPCR,READ_MEM32(PTRAPCR)&~EN_DHCP_TRAP2);
	WRITE_MEM32(PTRAPCR,READ_MEM32(PTRAPCR)&~EN_OSPF_TRAP);
	WRITE_MEM32(PTRAPCR,READ_MEM32(PTRAPCR)&~EN_RIP_TRAP);

	//Set all Protocol-Based Reg. to 0
	WRITE_MEM32(PBVCR,  0x00000000);
	WRITE_MEM32(PBVCR2, 0x00000000);
	WRITE_MEM32(PBVR1,  0x00000000);	/* IPX */
	WRITE_MEM32(PBVR1E, 0x00000000);
	WRITE_MEM32(PBVR2,  0x00000000);	/* NetBIOS */
	WRITE_MEM32(PBVR2E, 0x00000000);
	WRITE_MEM32(PBVR3,  0x00000000);	/* PPPoE Control */
	WRITE_MEM32(PBVR3E, 0x00000000);
	WRITE_MEM32(PBVR4,  0x00000000);	/* PPPoE Session */
	WRITE_MEM32(PBVR4E, 0x00000000);
	WRITE_MEM32(PBVR5,  0x00000000);	/* User-defined 1 */
	WRITE_MEM32(PBVR5E, 0x00000000);
	WRITE_MEM32(PBVR6,  0x00000000);	/* User-defined 2 */
	WRITE_MEM32(PBVR6E, 0x00000000);
	
	//Enable TTL-1
	WRITE_MEM32(TTLCR,READ_MEM32(TTLCR)|(uint32)EN_TTL1);//Don't hide this router. enable TTL-1 when routing on this gateway.
	/* 
		In RTL865xB d-cut, bit.30 in TTLCR register is used to enable/disable rate limit table hardware aging. 
		Therefore, if it is RTL865xB d-cut, TBLDRV turns this bit on.
	*/
	if (RtkHomeGatewayChipName[strlen(RtkHomeGatewayChipName) - 1] == 'B' && RtkHomeGatewayChipRevisionID == 0x03)
		WRITE_MEM32(TTLCR, READ_MEM32(TTLCR) | (uint32)EN_RATELIMIT_HWAGING);

	//QoS setting: Disable QoS
	WRITE_MEM32(QOSCR, READ_MEM32(QOSCR) | 0xc0000000);
	

	//Read all counter value once to get rid of meaningless counter values returned
	_rtl8651_initialRead();


	for (index=0; index<RTL8651_PORT_NUMBER+rtl8651_totalExtPortNum; index++) {
		if(	rtl8651_setAsicMulticastPortInternal(index, TRUE)||

			rtl8651_setAsicMulticastSpanningTreePortState(index, RTL8651_PORTSTA_FORWARDING))
			return FAILED;
		rtl8651_setAsicEthernetBandwidthControl(index, TRUE, RTL8651_BC_FULL);
		rtl8651_setAsicEthernetBandwidthControl(index, FALSE, RTL8651_BC_FULL);
	}

      /* Enable back pressure and PHY */
	WRITE_MEM32(BISTCR,READ_MEM32(BISTCR)|TRXRDY);	// 16.2 The bist control register,  BISTCR.TRXRDY Set 1 to start normal Tx and Rx

#if !defined(RTL865X_TEST) && !defined(RTL865X_MODEL_USER)
#ifndef FPGA_BOARD
    /* Check BIST until all packet buffers are reset */
    while ( ( READ_MEM32(BISTCR) & BIST_TRXRDY_PATTERN ) != BIST_TRXRDY_PATTERN );

	/* Check Queue Memory is OK */
	if ( ( READ_MEM32(BISTCR) & BIST_QUEUE_MEMORY_FAIL_PATTERN ) == BIST_QUEUE_MEMORY_FAIL_PATTERN )
	{
		rtlglue_printf("swCore_init(): BIST: Queue Memory failed, BISTCR=0x%08x\n", READ_MEM32(BISTCR) );
		while(1);
	}

	/* Check Packet Buffer is OK */
	if ( ( READ_MEM32(BISTCR) & BIST_PACKET_BUFFER_FAIL_PATTERN ) == BIST_PACKET_BUFFER_FAIL_PATTERN )
	{
		rtlglue_printf("swCore_init(): BIST: Packet Buffer failed, BISTCR=0x%08x\n", READ_MEM32(BISTCR) );
		while(1);
	}

#endif
#endif

     	WRITE_MEM32(MACCR,READ_MEM32(MACCR)|(EN_BACK_PRESSURE | EN_PHY_P4 | EN_PHY_P3 | EN_PHY_P2 | EN_PHY_P1 | EN_PHY_P0));

	/* 100M TxD latch clock phase */
	WRITE_MEM32(SWTAA, (uint32)PORT6_PHY_CONTROL);
	WRITE_MEM32(TCR0, 0x00000056);
	WRITE_MEM32(SWTACR, CMD_FORCE | ACTION_START); // force add
#if !defined(RTL865X_TEST) && !defined(RTL865X_MODEL_USER)
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE ); /* Wait for command done */
#endif
	
	// TCPUDP Table: Set collision bit to 1
	memset( &naptTcpUdp, 0, sizeof(naptTcpUdp) );
	naptTcpUdp.isCollision = 1;
	naptTcpUdp.isCollision2 = 1;
	for(flowTblIdx=0; flowTblIdx<RTL8651_TCPUDPTBL_SIZE; flowTblIdx++)
		rtl8651_setAsicNaptTcpUdpTable( TRUE, flowTblIdx, &naptTcpUdp );

	// ICMP Table: Set collision bit to 1
	memset( &naptIcmp, 0, sizeof(naptIcmp) );
	naptIcmp.isCollision = 1;
	for(flowTblIdx=0; flowTblIdx<RTL8651_ICMPTBL_SIZE; flowTblIdx++)
		rtl8651_setAsicNaptIcmpTable( TRUE, flowTblIdx, &naptIcmp );	


	/* ======================================
		Enable Flow control for ALL ports
	     ====================================== */
	{
		uint32 port;

		for ( port = 0 ; port < RTL8651_PHY_NUMBER ; port ++ )
		{
			rtl8651_setAsicFlowControlRegister(port, TRUE);
		}
	}

	return SUCCESS;
}


/*=========================================
  * ASIC DRIVER API: HASHING FUNCTION
  *=========================================*/
#define RTL865X_ASIC_DRIVER_HASHING_FUNCTION_API

uint32 rtl8651_filterDbIndex(ether_addr_t * macAddr) {
    return ( macAddr->octet[0] ^ macAddr->octet[1] ^
                    macAddr->octet[2] ^ macAddr->octet[3] ^
                    macAddr->octet[4] ^ macAddr->octet[5]) & 0xFF;
}

//uint32 rtl8651_vlanTableIndex(uint16 vid) {
//	return (vid& (RTL8651_VLAN_NUMBER-1));
//}



int32 rtl8651_setAsicAgingFunction(int8 l2Enable, int8 l4Enable) {
	WRITE_MEM32(TEATCR,(READ_MEM32(TEATCR) & ~0x3) | (l2Enable == TRUE? 0x0: 0x1) | (l4Enable == TRUE? 0x0 : 0x2));
	return SUCCESS;
}

int32 rtl8651_getAsicAgingFunction(int8 * l2Enable, int8 * l4Enable) {
	if(l2Enable == NULL || l4Enable == NULL)
		return FAILED;

	*l2Enable = (READ_MEM32(TEATCR) & 0x1)? FALSE: TRUE;
	*l4Enable = (READ_MEM32(TEATCR) & 0x2)? FALSE: TRUE;
	return SUCCESS;
}


/*=========================================
  * ASIC DRIVER API: FLOW CONTROL
  *=========================================*/
#define RTL865X_ASIC_DRIVER_FLOW_CONTROL_API

/*
	When user uses rtl8651_setAsicFlowControlRegister() to turn off the flow control,
	we record it in "fake_DisableFlowControl" array instead of turning off flow control of PHY.
	When user uses rtl8651_getAsicFlowControlRegister() to read the flow control setting,
	we report it according "fake_DisableFlowControl" array.
*/
uint32 fake_DisableFlowControl[RTL8651_PHY_NUMBER];

int32 rtl8651_setAsicFlowControlRegister(uint32 port, int8 enable)
{
	uint32 bitMask;

	/* Patch for RTL865xB cut-b & before that. */
	if (RtkHomeGatewayChipRevisionID < 0x02) {
		if (!enable)
			fake_DisableFlowControl[port] = 1;
		else
			fake_DisableFlowControl[port] = 0;
		return SUCCESS;
	}

	/* port 5: MII,   port 7: CPU */
	if (port > 4)
		return FAILED;
	bitMask = (0x02040000 << port);
	
	/*check if bit setted as to be set.if so just return.avoiding link state change*/
	if((TRUE == enable && (READ_MEM32(PHY_BASE+(port<<5)+0x10) & 0x400))
		||(FALSE == enable && !(READ_MEM32(PHY_BASE+(port<<5)+0x10) & 0x400)))
		return SUCCESS;
	/* Set PHY Control Reg 4: Pause, (n.10) */
	WRITE_MEM32(SWTAA, (uint32)(PHY_BASE + (port<<5) + 0x10));
	if (enable == FALSE) 
		WRITE_MEM32(TCR0, (READ_MEM32((PHY_BASE+(port<<5)+0x10)) & ~0x400)); /* Disable n.10 */
	else WRITE_MEM32(TCR0, (READ_MEM32((PHY_BASE+(port<<5)+0x10)) | 0x400)); /* Enable n.10 */
	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);//Activate add command
#if !defined(RTL865X_TEST) && !defined(RTL865X_MODEL_USER)
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#else
	*((uint32 *)(PHY_BASE + (port<<5) + 0x10)) = READ_MEM32(TCR0);
#endif	

	/* 
	 * Try to modify MAC's Flow Control state, though, when N-Way restart, a link change
	 * interrupt will be issued and the rtl8651_updateLinkStatus() will try to modify MAC' Flow
	 * Control state.
	 */
	if (enable == FALSE)
		/* Disable MAC Flow Control Reg. */
		WRITE_MEM32(FCREN, (READ_MEM32(FCREN) & ~bitMask));	
	else /* Enable MAC Flow Control Reg. */
		WRITE_MEM32(FCREN, (READ_MEM32(FCREN) | bitMask));

	/* Restart N-Way */
	WRITE_MEM32(SWTAA, PHY_BASE + (port<<5));
	WRITE_MEM32(TCR0, (READ_MEM32(PHY_BASE+(port<<5)) | 0x200));
	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);//Activate add command
#if !defined(RTL865X_TEST) && !defined(RTL865X_MODEL_USER)
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif

	return SUCCESS;





#if 0
	//Set PHY Control Reg 4: Pause
	WRITE_MEM32(SWTAA, (uint32)(PHY_BASE + (port<<5) + 0x10));
	if (enable) 
		WRITE_MEM32(TCR0, (READ_MEM32((PHY_BASE+(port<<5)+0x10)) | 0x400));
	else 	WRITE_MEM32(TCR0, (READ_MEM32((PHY_BASE+(port<<5)+0x10)) & ~0x400));
	WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);//Activate add command
#ifndef RTL865X_TEST
	while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */	

#if 1
	if (enable)
		WRITE_MEM32(FCREN, (READ_MEM32(FCREN) | bitMask));
	else WRITE_MEM32(FCREN, (READ_MEM32(FCREN) & ~bitMask));	

#else
	if (READ_MEM32(PHY_BASE+(port<<5)+0x04) & 0x08) { //auto-Negotiation ability
		//Restart Auto-Negoitation
		WRITE_MEM32(SWTAA, PHY_BASE + (port<<5));
		WRITE_MEM32(TCR0, (READ_MEM32(PHY_BASE+(port<<5)) | 0x200));
		WRITE_MEM32(SWTACR,ACTION_START | CMD_FORCE);//Activate add command
#ifndef RTL865X_TEST
		while ( (READ_MEM32(SWTACR) & ACTION_MASK) != ACTION_DONE );//Wait for command done
#endif /* RTL865X_TEST */	

	}	
	else { //no auto-negotiation ability
		if (enable)
			WRITE_MEM32(FCREN, (READ_MEM32(FCREN) | bitMask));
		else WRITE_MEM32(FCREN, (READ_MEM32(FCREN) & ~bitMask));	
	}
#endif
	return SUCCESS;
#endif

}

int32 rtl8651_getAsicFlowControlRegister(uint32 *fcren)
{
	if (fcren == NULL)
		return FAILED;

	/* For RTL865xB cut-b & before that. Return fake flow control reg value to user. */
	if (RtkHomeGatewayChipRevisionID < 0x02) {
		uint32 i;

		*fcren = 0xfffc05e0;
		for (i = 0; i < RTL8651_PHY_NUMBER; i++)
			if (fake_DisableFlowControl[i] == 1)
				*fcren &= (~0x02040000 << i);
		return SUCCESS;
	}

	*fcren = READ_MEM32(FCREN);
	return SUCCESS;
}



/*=========================================
  * ASIC DRIVER API: QoS
  *=========================================*/
#define RTL865X_ASIC_DRIVER_QOS_API


int32 rtl8651_setAsicHLQueueWeight(uint32 weight)
{
	if (weight > 0x03)
		return FAILED;
	WRITE_MEM32(QOSCR, ((READ_MEM32(QOSCR)&(~0xc0000000))|(weight<<30)));
	return SUCCESS;
}


int32 rtl8651_setAsicPortPriority(uint32 port, int8 highPriority)
{	

	
	if (port > 5)
		return FAILED;
	if (highPriority == TRUE)
		WRITE_MEM32(QOSCR, (READ_MEM32(QOSCR) | (1 << (port+21))));
	else  WRITE_MEM32(QOSCR, (READ_MEM32(QOSCR) & ~(1 << (port+21))));
	return SUCCESS;
}


int32 rtl8651_getAsicQoSControlRegister(uint32 *qoscr)
{
	if (qoscr == NULL)
		return FAILED;
	*qoscr = READ_MEM32(QOSCR);
	return SUCCESS;
}


int32 rtl8651_setAsicDiffServReg(uint32 dscp, int8 highPriority)
{
	if (dscp > 63)
		return FAILED;
	if (dscp > 31) {
		if (highPriority == TRUE)
			WRITE_MEM32(DSCR1, (READ_MEM32(DSCR1) | (1<<(dscp-32))));
		else WRITE_MEM32(DSCR1, (READ_MEM32(DSCR1) & ~(1<<(dscp-32))));
	}
	else {
		if (highPriority == TRUE)
			WRITE_MEM32(DSCR0, (READ_MEM32(DSCR0) | (1<<dscp)));
		else WRITE_MEM32(DSCR0, (READ_MEM32(DSCR0) & ~(1<<dscp)));
	}
	return SUCCESS;
}

int32 rtl8651_getAsicDiffServReg(uint32 *dscr0, uint32 *dscr1)
{
	if (dscr0 == NULL || dscr1 == NULL)
		return FAILED;
	*dscr0 = READ_MEM32(DSCR0);
	*dscr1 = READ_MEM32(DSCR1);
	return SUCCESS;
}


/*=========================================
 * ASIC DRIVER API: Protocol-based VLAN
 *=========================================*/

/*
@func int32 | rtl8651_defineProtocolBasedVLAN | configure user-definable protocol-based VLAN
@parm uint32 | ruleNo        | Valid values: RTL8651_PBV_RULE_USR1 and RTL8651_PBV_RULE_USR2
@parm uint8  | ProtocolType  | 00:ethernetII, 01:RFC-1042, 10: LLC-Other, 11:reserved
@parm uint16 | ProtocolValue | ethernetII:ether type, RFC-1042:ether type, LLC-Other:PDSAP(8),SSAP(8)}
@rvalue SUCCESS | 
@comm
 */
int32 rtl8651_defineProtocolBasedVLAN( uint32 ruleNo, uint8 ProtocolType, uint16 ProtocolValue )
{
	assert( ruleNo == RTL8651_PBV_RULE_USR1 ||
	        ruleNo == RTL8651_PBV_RULE_USR2 );

	if ( ruleNo == RTL8651_PBV_RULE_USR1 )
	{
		WRITE_MEM32( PBVCR, ( ProtocolType << PBVCR_PROTO_TYPE_OFFSET ) |
		                    ( ProtocolValue << PBVCR_PROTO_VALUE_OFFSET ) );
	}
	else if ( ruleNo == RTL8651_PBV_RULE_USR2 )
	{
		WRITE_MEM32( PBVCR2, ( ProtocolType << PBVCR_PROTO_TYPE_OFFSET ) |
		                     ( ProtocolValue << PBVCR_PROTO_VALUE_OFFSET ) );
	}

	return SUCCESS;
}


/*
@func int32 | rtl8651_queryProtocolBasedVLAN | Query user-definable protocol-based VLAN
@parm uint32 | ruleNo        | Valid values: RTL8651_PBV_RULE_USR1 and RTL8651_PBV_RULE_USR2
@parm uint8*  | ProtocolType  | 00:ethernetII, 01:RFC-1042, 10: LLC-Other, 11:reserved
@parm uint16* | ProtocolValue | ethernetII:ether type, RFC-1042:ether type, LLC-Other:PDSAP(8),SSAP(8)}
@rvalue SUCCESS | 
@comm
 */
int32 rtl8651_queryProtocolBasedVLAN( uint32 ruleNo, uint8* ProtocolType, uint16* ProtocolValue )
{
	uint32 regValue=0;
	
	assert( ruleNo == RTL8651_PBV_RULE_USR1 ||
	        ruleNo == RTL8651_PBV_RULE_USR2 );

	if ( ruleNo == RTL8651_PBV_RULE_USR1 )
	{
		regValue = READ_MEM32( PBVCR );
	}
	else if ( ruleNo == RTL8651_PBV_RULE_USR2 )
	{
		regValue = READ_MEM32( PBVCR2 );
	}

	if ( ProtocolType ) *ProtocolType = ( regValue & PBVCR_PROTO_TYPE_MASK ) >> PBVCR_PROTO_TYPE_OFFSET;
	if ( ProtocolValue ) *ProtocolValue = ( regValue & PBVCR_PROTO_VALUE_MASK ) >> PBVCR_PROTO_VALUE_OFFSET;

	return SUCCESS;
}


/*
@func int32 | rtl8651_setProtocolBasedVLAN | set corresponding table index of protocol-based VLAN
@parm uint32 | ruleNo  | rule Number (1~6)
@parm uint8  | port    | 0~4:PHY  5:MII  6~8:ExtPort
@parm uint8  | vlanIdx | VLAN Table index (0~7)
@rvalue SUCCESS | 
@comm
 */
int32 rtl8651_setProtocolBasedVLAN( uint32 ruleNo, uint32 port, uint8 valid, uint8 vlanIdx )
{
	uint32 mask;
	uint32 value;
	
	assert( ruleNo > 0 && ruleNo < RTL8651_PBV_RULE_MAX );
	assert( vlanIdx < RTL8651_VLAN_NUMBER );
	assert( port < RTL8651_AGGREGATOR_NUMBER );

	valid = valid ? TRUE : FALSE;
	if ( valid == FALSE )
	{
		vlanIdx = 0; // clear it for looking pretty.
	}

	if ( port < RTL8651_PORT_NUMBER )
	{
		// Port0 ~ Port5
		value = ( vlanIdx<<1 | valid ) << ( port*4 );
		mask = 0x0000000f << (port*4);
		WRITE_MEM32( PBVCR+ruleNo*8, READ_MEM32(PBVCR+ruleNo*8) & ~mask );
		WRITE_MEM32( PBVCR+ruleNo*8, READ_MEM32(PBVCR+ruleNo*8) | value );
	}
	else
	{
		// ExtPort0 ~ ExtPort2
		port -= RTL8651_PORT_NUMBER;
		value = ( vlanIdx<<1 | valid ) << ( port*4 );
		mask = 0x0000000f << (port*4);
		WRITE_MEM32( PBVCR+ruleNo*8+4, READ_MEM32(PBVCR+ruleNo*8+4) & ~mask );
		WRITE_MEM32( PBVCR+ruleNo*8+4, READ_MEM32(PBVCR+ruleNo*8+4) | value );
	}

	return SUCCESS;
}

/*
@func int32 | rtl8651_getProtocolBasedVLAN | get corresponding table index of protocol-based VLAN
@parm uint32 | ruleNo  | rule Number (1~6)
@parm uint8* | port    | (output) 0~4:PHY  5:MII  6~8:ExtPort
@parm uint8* | vlanIdx | (output) VLAN Table index (0~7)
@rvalue SUCCESS | 
@comm
 */
int32 rtl8651_getProtocolBasedVLAN( uint32 ruleNo, uint32 port, uint8* valid, uint8* vlanIdx )
{
	uint32 mask;
	uint32 value;

	assert( ruleNo > 0 && ruleNo < RTL8651_PBV_RULE_MAX );
	assert( port < RTL8651_AGGREGATOR_NUMBER );

	if ( port < RTL8651_PORT_NUMBER )
	{
		// Port0 ~ Port5
		mask = 0x0000000f << (port*4);
		value = ( READ_MEM32(PBVCR+ruleNo*8) & mask ) >> (port*4);

		if ( valid ) *valid = value & 1;
		if ( vlanIdx ) *vlanIdx = value >> 1;
	}
	else
	{
		// ExtPort0 ~ ExtPort2
		port -= RTL8651_PORT_NUMBER;
		mask = 0x0000000f << (port*4);
		value = ( READ_MEM32(PBVCR+ruleNo*8+4) & mask ) >> (port*4);

		if ( valid ) *valid = value & 1;
		if ( vlanIdx ) *vlanIdx = value >> 1;
	}

	assert( *vlanIdx < RTL8651_VLAN_NUMBER );

	return SUCCESS;
}



/*=========================================
  * ASIC DRIVER API: Auto MDI/MDIX 
  *=========================================*/
#define RTL865X_ASIC_DRIVER_AUTO_MDI_MDIX_API

int32 rtl8651_autoMdiMdix(uint32 port, uint32 isEnable)
{
	uint32 value;
	
	if (port > 4)
		return FAILED;
	value = READ_MEM32(VLANTCR);

	/* Patch for RTL865xB cut-d only: the bit of MII_enForceMode must inverse when read it. */
	if (RtkHomeGatewayChipRevisionID == 0x03) 
		value = ((~value)&MII_ENFORCE_MODE)  | (value&(~MII_ENFORCE_MODE));
	if (isEnable == FALSE)
		WRITE_MEM32(VLANTCR, value | (1<<(port+27))); /* 1: disable */
	else WRITE_MEM32(VLANTCR, value & ~(1<<(port+27))); /* 0: enable */

#if 0
	if (isEnable == FALSE)
		WRITE_MEM32(VLANTCR, READ_MEM32(VLANTCR) | (1<<(port+27))); /* 1: disable */
	else WRITE_MEM32(VLANTCR, READ_MEM32(VLANTCR) & ~(1<<(port+27))); /* 0: enable */
#endif

	return SUCCESS;
}

int32 rtl8651_getAutoMdiMdix(uint32 port, uint32 *isEnable)
{
	uint32 value;
	if (port > 4 || isEnable == NULL)
		return FAILED;
	value = READ_MEM32(VLANTCR);
	/* Patch for RTL865xB cut-d only: the bit of MII_enForceMode must inverse when read it. */
	if (RtkHomeGatewayChipRevisionID == 0x03) 
		value = ((~value)&MII_ENFORCE_MODE)  | (value&(~MII_ENFORCE_MODE));
	*isEnable = (value & (1<<(port+27)))? FALSE: TRUE;
	return SUCCESS;
}

int32 rtl8651_selectMdiMdix(uint32 port, uint32 isMdi)
{
	uint32 value;
	
	if (port > 4)
		return FAILED;
	value = READ_MEM32(VLANTCR);
	/* Patch for RTL865xB cut-d only: the bit of MII_enForceMode must inverse when read it. */
	if (RtkHomeGatewayChipRevisionID == 0x03) 
		value = ((~value)&MII_ENFORCE_MODE)  | (value&(~MII_ENFORCE_MODE));
	if (isMdi == TRUE)
		WRITE_MEM32(VLANTCR, value | (1<<(port+22)));
	else WRITE_MEM32(VLANTCR, value & ~(1<<(port+22)));

#if 0
	if (isMdi == TRUE)
		WRITE_MEM32(VLANTCR, READ_MEM32(VLANTCR) | (1<<(port+22)));
	else WRITE_MEM32(VLANTCR, READ_MEM32(VLANTCR) & ~(1<<(port+22)));
#endif

	return SUCCESS;
}

int32 rtl8651_getSelectMdiMdix(uint32 port, uint32 *isMdi)
{
	uint32 value;
	if (port > 4 || isMdi == NULL)
		return FAILED;
	value = READ_MEM32(VLANTCR);
	/* Patch for RTL865xB cut-d only: the bit of MII_enForceMode must inverse when read it. */
	if (RtkHomeGatewayChipRevisionID == 0x03) 
		value = ((~value)&MII_ENFORCE_MODE)  | (value&(~MII_ENFORCE_MODE));
	*isMdi = (value & (1<<(port+22)))? TRUE: FALSE;
	return SUCCESS;
}



