#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <unistd.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#if HAVE_LIBGEN_H
#include <libgen.h>
#endif

#include "sfile.h"
#include "vlist.h"
#include "defs.h"
#include "invoke.h"
#include "fgetline.h"

struct ctsinfo {
	char *filename;
	int pmask;
} _ctslist[] = {
    {PATH_NAME_LIST, 0},
    {PATH_ROUTE_LIST, 0},
    {PATH_RULE_CTS, 0},
    {PATH_BIND_LIST, 0},
    {NULL, -1},
};

static int sfileWriteLine(const char *path, const char *mode, const char *str)
{
    FILE *fp = (FILE*) 0;

    if (! (fp = fopen(path, mode)))
        return -1;

    if (fprintf(fp, "%s\n", str) < 0) {
        fclose(fp);
        return -1;
    }
    if (fclose(fp) != 0)
        return -1;
    return 0;
}

int sfileAppend(const char *path, const char *append)
{
    return sfileWriteLine(path, "a", append);
}

int sfileInstead(const char *path, const char *instead)
{
    return sfileWriteLine(path, "w", instead);
}

int sfileCopy(const char *from, const char *to)
{
    char **argv = (char**) 0;
    int nargs = 0;
    int ret = 0;

    if (! (argv = argvNew(PS_COPY, &nargs)))
        return -1;
    if (! argvAdd(&argv, "-f", &nargs) ||
        ! argvAdd(&argv, from, &nargs) ||
        ! argvAdd(&argv, to, &nargs)) {
        argvFree(argv);
        return -1;
    }

    ret = invokeQuiet(PS_COPY, argv);
    argvFree(argv);

    return ret;
}

int sfileMove(const char *from, const char *to)
{
    char **argv = (char**) 0;
    int nargs = 0;
    int ret = 0;

    if (! (argv = argvNew(PS_MV, &nargs)))
        return -1;
    if (! argvAdd(&argv, "-f", &nargs) ||
        ! argvAdd(&argv, from, &nargs) ||
        ! argvAdd(&argv, to, &nargs)) {
        argvFree(argv);
        return -1;
    }

    ret = invokeQuiet(PS_MV, argv);
    argvFree(argv);

    return ret;
}

int sfileRemove(const char *path, const char *str)
{
    int changed = 0;
    int rr = -1;
    FILE *fp, *tempfile;
    char temppath[MAXPATHLEN + 1];
    char buffer[MAX_CONFIG_LINE + 1];
    int tfd = -1;
    int len;

    len = strlen(str);

    snprintf(temppath, MAXPATHLEN, "/tmp/rfw.XXXXXX");
    if ((tfd = mkstemp(temppath)) < 0)
        return -1;
    if (chmod(temppath, 0644) < 0) {
        close(tfd);
        unlink(temppath);
        return -1;
    }
    if (! (tempfile = fopen(temppath, "w+"))) {
        close(tfd);
        unlink(temppath);
        return -1;
    }
    if (! (fp = fopen(path, "r+"))) {
        fclose(tempfile);
        unlink(temppath);
        return -1;
    }

    for (; (rr = fgetline(buffer, MAX_CONFIG_LINE, fp)) != -1; ) {
        if (rr) {
            if (strncmp(buffer, str, len)) {
                if (fprintf(tempfile, "%s\n", buffer) < 0) {
                    fclose(fp);
                    fclose(tempfile);
                    unlink(temppath);
                    return -1;
                }
            }else
                changed = 1;
        }
    }
    if (ferror(fp)) {
        fclose(fp);
        fclose(tempfile);
        unlink(temppath);
        return -1;
    }
    fclose(fp);
    fclose(tempfile);
    if (changed) {
        if (sfileMove(temppath, path) < 0) {
            unlink(temppath);
            return -1;
        }
    }else
        unlink(temppath);
    return 0;
}

int sfileRemoveLine(const char *path, int lineno)
{
    int changed = 0;
    int rr = -1;
    int cur = 1;
    FILE *fp, *tempfile;
    char temppath[MAXPATHLEN + 1];
    char buffer[MAX_CONFIG_LINE + 1];
    int tfd = -1;

    snprintf(temppath, MAXPATHLEN, "/tmp/rfw.XXXXXX");
    if ((tfd = mkstemp(temppath)) < 0)
        return -1;
    if (chmod(temppath, 0644) < 0) {
        close(tfd);
        unlink(temppath);
        return -1;
    }
    if (! (tempfile = fopen(temppath, "w+"))) {
        close(tfd);
        unlink(temppath);
        return -1;
    }
    if (! (fp = fopen(path, "r+"))) {
        fclose(tempfile);
        unlink(temppath);
        return -1;
    }

    for (; (rr = fgetline(buffer, MAX_CONFIG_LINE, fp)) != -1; ) {
        if (cur != lineno && rr) {
            if (fprintf(tempfile, "%s\n", buffer) < 0) {
                fclose(fp);
                fclose(tempfile);
                unlink(temppath);
                return -1;
            }
        }else
            changed = 1;
        cur++;
    }
    if (ferror(fp)) {
        fclose(fp);
        fclose(tempfile);
        unlink(temppath);
        return -1;
    }
    fclose(fp);
    fclose(tempfile);
    if (changed) {
        if (sfileMove(temppath, path) < 0) {
            unlink(temppath);
            return -1;
        }
    }else
        unlink(temppath);
    return 0;
}

int sfileExist(const char *path, const char *str)
{
    int ret = 0;
    FILE *fp = (FILE*) 0;
    char buffer[MAX_CONFIG_LINE + 1];
    int len = 0;

    len = strlen(str);

    if (! (fp = fopen(path, "r")))
        return 0;

    for (; (ret = fgetline(buffer, MAX_CONFIG_LINE, fp)) != -1; ) {
        if (ret && ! strncmp(buffer, str, len)) {
            fclose(fp);
            return 1;
        }
    }
    return 0;
}

int sfileListInit(const char *path, vlist *vl)
{
    FILE *fp = (FILE*) 0;
    char *line = (char*)0;
    char buf[MAX_CONFIG_LINE + 1];
    int ret = 0;

    vlistInit(vl);
    
    if (! (fp = fopen(path, "r")))
        return 0;
    for (; (ret = fgetline(buf, MAXPATHLEN, fp)) != -1; ) {
        if (! ret)
            continue;
        if (! (line = strdup(buf))) {
            fclose(fp);
            vlistDestroy(vl, free);
            return -1;
        }
        if (! vlistInsertLast(vl, line)) {
            fclose(fp);
            vlistDestroy(vl, free);
            return -1;
        }
    }
    if (ferror(fp)) {
        fclose(fp);
        vlistDestroy(vl, free);
        return -1;
    }
    fclose(fp);
    return 0;
}

int sfileListOutput(const char *path, vlist *vl)
{
    FILE *fp = (FILE*) 0;
    char *line = (char*) 0;
    vlistEntry *iter;

    if (! (fp = fopen(path, "w")))
        return -1;
    
    for (iter = vlistFirstEntry(vl); iter; iter = vlistNextEntry(iter)) {
        if ((line = (char*)iter->elem)) {
            if (fprintf(fp, "%s\n", line) < 0) {
                fclose(fp);
                return -1;
            }
        }
    }
    return 0;
}

int mountDisk()
{
    char **argv = (char**) argv;
    int nargs = 0;
    int ret = 0;
    
    if (! (argv = argvNew(PS_MOUNT, &nargs)))
        return -1;
    if (! argvAdd(&argv, FLUSH_SLICE, &nargs) ||
        ! argvAdd(&argv, FLUSH_MOUNT_POINT, &nargs)) {
        argvFree(argv);
        return -1;
    }
    ret = invokeQuiet(PS_MOUNT, argv);
    argvFree(argv);
    return (ret == 0) ? 0 : -1;
}

int umountDisk()
{
    char **argv = (char**) argv;
    int nargs = 0;
    int ret = 0;
    
    if (! (argv = argvNew(PS_UNMOUNT, &nargs)))
        return -1;
    if (! argvAdd(&argv, FLUSH_MOUNT_POINT, &nargs)) {
        argvFree(argv);
        return -1;
    }
    ret = invokeQuiet(PS_UNMOUNT, argv);
    argvFree(argv);
    return (ret == 0) ? 0 : -1;
}

int sfileTouch(const char *path)
{
    char **argv = (char**) argv;
    int nargs = 0;
    int ret = 0;
    
	if (! path)
		return -1;

    if (! (argv = argvNew(PS_TOUCH, &nargs)))
        return -1;
    if (! argvAdd(&argv, path, &nargs)) {
        argvFree(argv);
        return -1;
    }
    ret = invokeQuiet(PS_TOUCH, argv);
    argvFree(argv);
    return (ret == 0) ? 0 : -1;
}

int sfileDelete(const char *path)
{
    char **argv = (char**) argv;
    int nargs = 0;
    int ret = 0;
    
	if (! path)
		return -1;

    if (! (argv = argvNew(PS_RM, &nargs)))
        return -1;
    if (! argvAdd(&argv, "-f", &nargs) ||
			! argvAdd(&argv, path, &nargs)) {
        argvFree(argv);
        return -1;
    }
    ret = invokeQuiet(PS_RM, argv);
    argvFree(argv);
    return (ret == 0) ? 0 : -1;
}

