#include <sys/types.h>
#include <sys/param.h>
#include <limits.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#include "ruleset.h"
#include "fgetline.h"
#include "defs.h"

static char _bufName[MAX_NAMESIZE * 2];
static char _bufRule[MAXPATHLEN + 1];
static char _bufBind[MAXPATHLEN + 1];
static char _bufUnBind[MAXPATHLEN + 1];
static Rule *_lastRule = NULL;
static RuleGroup *_lastGroup = NULL;
static char *_tableName;
static char sMASK[64], sLIMIT[64];
static const char *_method;

static void ruleDestroy(void *p)
{
    Rule *rule = (Rule*) p;

    if (! rule)
        return;

    if (rule->rule)
        free(rule->rule);
    vlistDestroy(&(rule->binds), free);
    vlistDestroy(&(rule->unbinds), free);
    free(rule);
}

static Rule *ruleAlloc(const char *rl)
{
    Rule *rule = (Rule*) 0;

    rule = (Rule *) malloc(sizeof(Rule));
    if (rule) {
        vlistInit(&(rule->binds));
        vlistInit(&(rule->unbinds));

        rule->rule = strdup(rl);
        if (NULL == rule->rule) {
            free(rule);
            rule = NULL;
        }
    }
    return rule;
}

static RuleGroup *ruleGroupAlloc(const char *name)
{
    RuleGroup *group = (RuleGroup*) 0;

    group = (RuleGroup *) malloc(sizeof(RuleGroup));
    if (group) {
        vlistInit(&(group->rules));
        group->name = strdup(name);
        if (NULL == group->name) {
            free(group);
            group = NULL;
        }
    }
    return group;
}

static void ruleGroupDestroy(void *p)
{
    RuleGroup *group = (RuleGroup*) p;

    if (! group)
        return;

    if (group->name)
        free(group->name);
    vlistDestroy(&(group->rules), ruleDestroy);
}

static int ruleOpAdd(Rule *rule, const char *op, int bind)
{
    int ret = -1;
    char *cp = strdup(op);

    if (! rule || ! op || ! *op)
        return -1;

    if (cp) {
        if (bind) {
            if (! vlistInsertLast(&(rule->binds), cp)) {
                free(cp);
                return -1;
            }
        }else {
            if (! vlistInsertLast(&(rule->unbinds), cp)) {
                free(cp);
                return -1;
            }
        }
        ret = 0;
    }
    return ret;
}

static int parserGroup(Ruleset *rs, FILE *fp)
{
    int ret = -1;
    int ch = '[';
    int pos = 0;

    for (; ']' != ch;) {
        ch = fgetc(fp);
        if (ferror(fp) || feof(fp) || '\n' == ch)
            return -1;

        if (']' == ch)
            _bufName[pos] = '\0';
        else {
            if (MAX_NAMESIZE * 2 <= pos)
                return -1;
            _bufName[pos++] = ch;
            if (0 > ret)
                ret = 0;
        }
    }
    if (ret == 0) {
        _lastRule = NULL;
        if (! (_lastGroup = ruleGroupAlloc(_bufName)))
            return -1;
        if (! vlistInsertLast(rs, _lastGroup)) {
            ruleGroupDestroy(_lastGroup);
            return -1;
        }
    }

    return ret;
}

static int parserLine(Ruleset *rs, FILE *fp)
{
    int ch = '\0';

    ch = fgetc(fp);
    if (ferror(fp))
        return -1;
    if (feof(fp) || '\n' == ch)
        return 0;

    if ('[' == ch)
        return parserGroup(rs, fp);

    if (ungetc(ch, fp) == EOF)
        return -1;

    if (fgetline(_bufRule, MAXPATHLEN, fp) != 1)
        return -1;
    if (*_bufRule == '+' || *_bufRule == '-') {
        if (! _lastRule ||
            ruleOpAdd(_lastRule, _bufRule + 1, (*_bufRule == '+') ? 1 : 0) < 0)
            return -1;
    }else {
        if (! _lastGroup || ! (_lastRule = ruleAlloc(_bufRule)))
            return -1;
        if (! vlistInsertLast(&(_lastGroup->rules), _lastRule)) {
            ruleDestroy(_lastRule);
            return -1;
        }
    }
    return 0;
}

int rulesetInit(Ruleset *rs)
{
    FILE *fp = NULL;

    vlistInit(rs);

    if (! (fp = fopen(PATH_RULE_CTS, "r")))
        return 0;

    for (; ! feof(fp);) {
        if (parserLine(rs, fp) < 0) {
            fclose(fp);
            rulesetDestroy(rs);
            return -1;
        }
    }
    fclose(fp);
    return 0;
}

void rulesetDestroy(Ruleset *rs)
{
    vlistDestroy(rs, ruleGroupDestroy);
}

static int rulesetInsertRule(Ruleset *rs, const char *name, Rule *rule)
{
    vlistEntry *iter1 = NULL;
    RuleGroup *group = NULL;

    iter1 = vlistFirstEntry(rs);
    for (; iter1; iter1 = vlistNextEntry(iter1)) {
        group = (RuleGroup*) iter1->elem;
        if (! strcmp(name, group->name))
            break;
    }
    if (! iter1) {              /* ½ */
        if (! (group = ruleGroupAlloc(name)))
            return -1;
        if (! vlistInsertLast(rs, group)) {
            ruleGroupDestroy(group);
            return -1;
        }
    }
    if (! vlistInsertLast(&(group->rules), rule))
        return -1;
    return 0;
}

int rulesetWrite(Ruleset *rs)
{
    FILE *fp = (FILE*) 0;
    vlistEntry *iter1, *iter2, *iter3;
    RuleGroup *group = NULL;
    Rule *rule = NULL;

    if (! (fp = fopen(PATH_RULE_CTS, "w")))
        return -1;

    for (iter1 = vlistFirstEntry(rs); iter1; iter1 = vlistNextEntry(iter1)) {
        group = (RuleGroup*) iter1->elem;
        if (fprintf(fp, "[%s]\n", group->name) < 0) {
            fclose(fp);
            return -1;
        }
        for (iter2 = vlistFirstEntry(& (group->rules));
             iter2;
             iter2 = vlistNextEntry(iter2)) {
            rule = (Rule*) iter2->elem;
            if (fprintf(fp, "%s\n", rule->rule) < 0) {
                fclose(fp);
                return -1;
            }
            for (iter3 = vlistFirstEntry(& (rule->binds));
                 iter3;
                 iter3 = vlistNextEntry(iter3)) {
                if (fprintf(fp, "+%s\n", (char*) iter3->elem) < 0) {
                    fclose(fp);
                    return -1;
                }
            }
            for (iter3 = vlistFirstEntry(& (rule->unbinds));
                 iter3;
                 iter3 = vlistNextEntry(iter3)) {
                if (fprintf(fp, "-%s\n", (char*) iter3->elem) < 0) {
                    fclose(fp);
                    return -1;
                }
            }
        }
    }
    return fclose(fp);
}

int rulesetRemove(Ruleset *rs, const char *name, int pos)
{
    int i = 0;
    vlistEntry *iter1, *iter2;
    RuleGroup *group = NULL;

    if (! rs || pos < 0)
        return -1;

    if (! name) {
        vlistDestroy(rs, ruleGroupDestroy);
        return 0;
    }

    iter1 = vlistFirstEntry(rs);
    for (; iter1; iter1 = vlistNextEntry(iter1)) {
        group = (RuleGroup*) iter1->elem;
        if (! strcmp(group->name, name)) {
            if (! pos) {
                vlistRemoveEntry(rs, iter1, ruleGroupDestroy);
                return 0;
            }else {
                iter2 = vlistFirstEntry(& (group->rules));
                for (; iter2; iter2 = vlistNextEntry(iter2))
                    if (++i == pos) {
                        vlistRemoveEntry(&(group->rules), iter2, ruleDestroy);
                        if (vlistEmpty(&(group->rules)))
                            vlistRemoveEntry(rs, iter1, ruleGroupDestroy);
                        return 0;
                    }
            }
        }
    }
    return -1;
}

int rulesetList(Ruleset *rs, const char *name, FILE *fp)
{
    vlistEntry *iter1, *iter2;
    RuleGroup *group = NULL;
    Rule *rule = NULL;
    int i = 1;

    if (! rs || !fp)
        return -1;

    iter1 = vlistFirstEntry(rs);
    for (; iter1; iter1 = vlistNextEntry(iter1)) {
        group = (RuleGroup*) iter1->elem;
        if (name && strcmp(name, group->name))
            continue;
        if (fprintf(fp, "%s\n", group->name) < 0)
            return -1;
        iter2 = vlistFirstEntry(& (group->rules));
        i = 1;
        for (; iter2; iter2 = vlistNextEntry(iter2)) {
            rule = (Rule*) iter2->elem;
            if (fprintf(fp, "%5d   %s\n", i++, rule->rule) < 0)
                return -1;
        }
        if (fprintf(fp, "\n") < 0)
            return -1;
    }
    return 0;
}

int rulesetExist(Ruleset *rs, const char *name, int pos)
{
    int i = 0;
    vlistEntry *iter1, *iter2;
    RuleGroup *group = NULL;

    if (! rs)
        return 0;
    if (! name)
        return 1;

    iter1 = vlistFirstEntry(rs);
    for (; iter1; iter1 = vlistNextEntry(iter1)) {
        group = (RuleGroup*) iter1->elem;
        if (! strcmp(group->name, name)) {
            if (! pos)
                return 1;
            iter2 = vlistFirstEntry(& (group->rules));
            for (; iter2; iter2 = vlistNextEntry(iter2)) {
                if (++i == pos)
                    return 1;
            }
        }
    }
    return 0;
}


/* =========== rfwrule﷨ӹ =========== */

#include "cmd-inc.h"
#include "rule.h"
#include "test.h"
#include "fgetline.h"

static char *_alldays[8] = {
    NULL, "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"
};

static char* iptAction(const char *act)
{
    char *action = (char*) 0;
    static char *accept = "ACCEPT";
    static char *deny = "DROP";
    static char *reset =  "RETURN";
    static char conn_limit[MAX_CONFIG_LINE + 1];

    if (! strcmp("allow", act))
        action = accept;
    else if (! strcmp("deny", act))
        action = deny;
    else if (! strcmp("reset", act))
        action = reset;
    
    return action;
}

static int RULE_TEST_IP(const char *ip)
{
    if (! strcmp("any", ip) || test_ip(ip) || test_ipmask(ip))
        return 1;
    return 0;
}

static int RULE_TEST_PORT(const char *port)
{
    if (! strcmp("any", port) || test_portrange(port))
        return 1;
    return 0;
}

static void instOptionAddress(char *cmd, char *what, char *opt, int test)
{
    if (what && strcmp("any", what)) {
        strcat(cmd, " ");
        strcat(cmd, opt);
        if (test)
            strcat(cmd, " \\! ");
        else
            strcat(cmd, " ");
        strcat(cmd, what);
    }
}

static char* iptDays(char *wday)
{
    char *start, *stop;
    int nstart, nstop;
    static char days[30];
    
    days[0] = '\0';
    start = wday;
	nstart = nstop = 0;

    stop = strchr(start, '-');
    if (stop) {
        *stop = '\0';
        stop++;
        nstop = atoi(stop);
    }
    nstart = atoi(start);
    if (stop) {
        int i = 0;

        if (nstart > nstop) {
            for (i = nstart; i <= 7; i++) {
                strcat(days, _alldays[i]);
                strcat(days, ",");
            }
            for (i = 1; i <= nstop; i++) {
                strcat(days, _alldays[i]);
                if (i != nstop)
                    strcat(days, ",");
            }
        }else {
            for (i = nstart; i <= nstop; i++) {
                strcat(days, _alldays[i]);
                if (i != nstop)
                    strcat(days, ",");
            }
        }
    }else {
        strcpy(days, _alldays[nstart]);
    }
    return days;
}

static int instOptionTcpFlags(char *cmd, const char *tcpflags)
{
    char *dup = NULL;
    char *mask = NULL;
    char *deli = NULL;

    if (! tcpflags)
        return 0;
    if (! (dup = strdup(tcpflags)))
        return -1;
    mask = dup;
    if (! (deli = strchr(mask, '/'))) {
        free(dup);
        return -1;
    }
    *deli = '\0'; deli++;
    strcat(cmd, " --tcp-flags ");
    if (*dup == '!') {
        mask++;
        strcat(cmd, "! ");
    }
    strcat(cmd, mask);
    strcat(cmd, " ");
    strcat(cmd, deli);
    free(dup);
    return 0;
}

static int instOptionOptional(char *cmd, const char *wday, const char *time, const char *state)
{
    char cpday[64], cptime[64];
    
    if (wday)
        strcpy(cpday, wday);
    if (time)
        strcpy(cptime, time);

    if (state) {
        strcat(cmd, " -m state --state ");
        strcat(cmd, state);
    }

    if (wday || time) {
        strcat(cmd, " -m time");
        if (wday) {
            strcat(cmd, " --weekdays ");
            strcat(cmd, iptDays(cpday));
        }
        if (time) {
            char *t_start, *t_stop;
            
            t_start = cptime;
            t_stop = strchr(t_start, '-');
            if (t_stop) {
                *t_stop = '\0';
                t_stop ++;
                strcat(cmd, " --timestart "); strcat(cmd, t_start);
                strcat(cmd, " --timestop "); strcat(cmd, t_stop);
            }
        }
    }
    return 0;
}

#ifndef parserAddress
#define parserAddress(first, value, bflag)              \
do {                                                    \
    if ((first)) {                                      \
        (value) = strtok_r(args, ARGS_DELIM, &brkp);    \
    }else {                                             \
        (value) = strtok_r(NULL, ARGS_DELIM, &brkp);    \
    }                                                   \
    if (! (value))                                      \
        return -1;                                      \
    if (*(value) == '!') {                              \
        (value)++;                                      \
        (bflag) = 1;                                    \
    }                                                   \
} while (0)
#endif

static int parserOptional(char *args, char **oday, char **otime, char **ostate, char **tcpflags)
{
    char *key, *value, *brkp;
    key = value = brkp = (char*) 0;

    *oday = *otime = *ostate = (char*) 0;
    
    for (key = strtok_r(args, ARGS_DELIM, &brkp);
         key;
         key = strtok_r(NULL, ARGS_DELIM, &brkp)) {
        if (! (value = strtok_r(NULL, ARGS_DELIM, &brkp)))
            return -1;
        
        if (tcpflags) {
            if (! strcmp("tcpflags", key)) {
                if (! test_tcpflags(value))
                    return -1;
                *tcpflags = value;
                continue;
            }
        }
        if (! strcmp("state", key)) {
            if (! test_state(value))
                return -1;
            *ostate = value;
        }else if (! strcmp("day", key)) {
            if (! test_wdayrange(value))
                return -1;
            *oday = value;
            
        }else if (! strcmp("time", key)){
            if (! test_timerange(value))
                return -1;
            *otime = value;
        }else {
            return -1;
        }
    }
    return 0;
}

#ifndef COMBIN_OPTIONAL
#define COMBIN_OPTIONAL()                       \
do {                                            \
    if (state) {                                \
        strcat(formated, " state ");            \
        strcat(formated, state);                \
    }                                           \
    if (wday) {                                 \
        strcat(formated, " day ");              \
        strcat(formated, wday);                 \
    }                                           \
    if (time) {                                 \
        strcat(formated, " time ");             \
        strcat(formated, time);                 \
    }                                           \
}while(0)
#endif

#ifndef parserMRange
#define parserMRange(first,proto,value,sflag)		\
do {							\
    if (!strcmp("ip", (proto))) {			\
	if (ippGet((first), "ip",(value)) == 0)		\
		(sflag) = 1;				\
    }							\
    if (!strcmp("tcp", (proto))) {			\
	if (ippGet((first), "tcp_port", (value)) == 0)	\
		(sflag) = 1;				\
    }							\
    if (!strcmp("udp", (proto))) {			\
	if (ippGet((first), "udp_port", (value)) == 0)	\
		(sflag) = 1;				\
    }							\
} while(0)
#endif

/* policy portfw NAME (tcp|udp) wanPort lanIP lanPort */
static int subRulePF(const char *group,
                    const char *protocol,
                    char *args)
{
    Ruleset rs;
    Rule *rl = NULL;
    char *wanIP, *wanPort, *lanIP, *lanPort, *brkp, *modifier;
    char *tcpflags;
    char *wday, *time,  *state;
    char formated[MAX_CONFIG_LINE + 1];
    int flag;
    int inet_sock;
    struct ifreq ifr;
    const char *iface1 = NULL;
    
    state = wday = time = (char*) 0;
    
    if (! args)
        return -1;

    parserAddress(1, wanPort, flag);
    if (! test_port(wanPort)) {
	return -1;   /* wan˿ڴ */
    }
    flag = 0;
    parserAddress(0, lanIP, flag);
    if (! test_ip(lanIP)) {  
	return -1;   /* ĿIP */
    }
    
    flag = 0;

    parserAddress(0, lanPort, flag);
    if (! test_port(lanPort)) {
            return -1;   /* ĿĶ˿ڴ */
    }

    /*  */
    snprintf(formated, MAX_CONFIG_LINE, "%s: %s %s %s %s",
	     _method,
             protocol, wanPort, lanIP, lanPort);
    if (! (rl = ruleAlloc(formated)))
        return -1;

    if (! (iface1 = ifnByAlias("outside")))
	        return rpsErrorString("outside interface nonexist");
    
    inet_sock = socket(AF_INET, SOCK_DGRAM, 0);
    strcpy(ifr.ifr_name, iface1);
    if (ioctl(inet_sock, SIOCGIFADDR, &ifr) < 0)
	    return rpsErrorString("error while get wan IP address.");
    wanIP = strdup(inet_ntoa(((struct sockaddr_in*)&(ifr.ifr_addr))->sin_addr));
             

    /* iptables */
    sprintf(_bufBind, "%s -I %s -t nat -p %s -d %s --dport %s -j DNAT --to %s:%s", 
		    PS_IPTABLES, _tableName, protocol,
		    wanIP, wanPort, lanIP, lanPort);
    sprintf(_bufUnBind, "%s -D %s -t nat -p %s -d %s --dport %s -j DNAT --to %s:%s", 
		    PS_IPTABLES, _tableName, protocol,
		    wanIP, wanPort, lanIP, lanPort);

    if (ruleOpAdd(rl, _bufBind, 1) < 0 || ruleOpAdd(rl, _bufUnBind, 0) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInit(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInsertRule(&rs, group, rl) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetWrite(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    return 0;
}

/* policy portfw_lan NAME (tcp|udp) wanPort lanIP lanPort lan_net */
static int subRulePFLAN(const char *group,
                    const char *protocol,
                    char *args)
{
    Ruleset rs;
    Rule *rl = NULL;
    char *laninIP, *wanPort, *lanIP, *lanPort, *brkp, *modifier, *lan_net;
    char *tcpflags;
    char *wday, *time,  *state;
    char formated[MAX_CONFIG_LINE + 1];
    int flag;
    int inet_sock;
    struct ifreq ifr;
    const char *iface2 = NULL;
    
    state = wday = time = (char*) 0;
    
    if (! args)
        return -1;

    parserAddress(1, wanPort, flag);
    if (! test_port(wanPort)) {
	return -1;   /* wan˿ڴ */
    }
    flag = 0;
    parserAddress(0, lanIP, flag);
    if (! test_ip(lanIP)) {  
	return -1;   /* ĿIP */
    }
    
    flag = 0;

    parserAddress(0, lanPort, flag);
    if (! test_port(lanPort)) {
            return -1;   /* ĿĶ˿ڴ */
    }

    flag = 0;

    parserAddress(0, lan_net, flag);
    if (! (test_ip(lan_net) || test_ipmask(lan_net))) {
            return -1;   /* lanô */
    }

    /*  */
    snprintf(formated, MAX_CONFIG_LINE, "%s: %s %s %s %s %s",
	     _method,
             protocol, wanPort, lanIP, lanPort, lan_net);
    if (! (rl = ruleAlloc(formated)))
        return -1;

    if (! (iface2 = ifnByAlias("inside")))
	        return rpsErrorString("inside interface nonexist");
    
    inet_sock = socket(AF_INET, SOCK_DGRAM, 0);
    strcpy(ifr.ifr_name, iface2);
    if (ioctl(inet_sock, SIOCGIFADDR, &ifr) < 0)
	    return rpsErrorString("error while get Lan_in IP address.");
    laninIP = strdup(inet_ntoa(((struct sockaddr_in*)&(ifr.ifr_addr))->sin_addr));
             
    /* iptables */
    sprintf(_bufBind, "%s -I POSTROUTING -t nat -p %s -d %s -s %s --dport %s -j SNAT --to %s", 
		    PS_IPTABLES, protocol, lanIP, lan_net, lanPort, laninIP);
    sprintf(_bufUnBind, "%s -D POSTROUTING -t nat -p %s -d %s -s %s --dport %s -j SNAT --to %s", 
		    PS_IPTABLES, protocol, lanIP, lan_net, lanPort, laninIP);


    if (ruleOpAdd(rl, _bufBind, 1) < 0 || ruleOpAdd(rl, _bufUnBind, 0) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInit(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInsertRule(&rs, group, rl) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetWrite(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    return 0;
}

/* policy trigger NAME (tcp|udp) matchMPort openMPort */
static int subRuleTRIG(const char *group,
                    const char *protocol,
                    char *args)
{
    Ruleset rs;
    Rule *rl = NULL;
    char *matchMPort, *openMPort, *brkp, *modifier,*mproto, *tproto, *matchMPort2, *ptt;
    char formated[MAX_CONFIG_LINE + 1];
    int flag = 0;
    const char *iface = NULL;
    
    if (! args)
        return -1;

    if (strcmp(protocol, "tcp") && strcmp(protocol, "udp") && strcmp(protocol, "all"))
	    return -1;
    if (!strcmp(protocol, "all")) {
	    mproto = strdup(" ");
    } else {
	    mproto = strdup("-p ");
	    strcat(mproto, protocol);
    }
		    

    parserAddress(1, matchMPort, flag);
    if (! test_portrange(matchMPort)) {
	return -1;   /* match˿ڴ */
    }
    flag = 0;

    parserAddress(0, tproto, flag);
    if (strcmp(tproto, "tcp") && strcmp(tproto, "udp") && strcmp(tproto, "all"))
	    return -1;
    flag = 0;

    parserAddress(0, openMPort, flag);
    if (! test_portrange(openMPort)) {
            return -1;   /* open˿ڴ */
    }

    /*  */
    snprintf(formated, MAX_CONFIG_LINE, "%s: %s %s %s %s",
	     _method,
             protocol, matchMPort, tproto, openMPort);
    if (! (rl = ruleAlloc(formated)))
        return -1;

    if (! (iface = ifnByAlias("outside")))
	        return rpsErrorString("interface nonexist");

    matchMPort2 = strdup(matchMPort);
    ptt = (char *) 0;
    ptt = strchr(matchMPort2, '-');
    if (ptt)
	    *ptt = ':';

    

    /* iptables */
    if (!strcmp(protocol, "all")) {
        sprintf(_bufBind, "%s -I %s -o %s -p tcp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s;%s -I %s -o %s -p udp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s", 
		    PS_IPTABLES, _tableName, iface, matchMPort2, tproto, matchMPort, openMPort,
		    PS_IPTABLES, _tableName, iface, matchMPort2, tproto, matchMPort, openMPort);
        sprintf(_bufUnBind, "%s -D %s -o %s -p tcp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s;%s -D %s -o %s -p udp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s", 
		    PS_IPTABLES, _tableName, iface, matchMPort2, tproto, matchMPort, openMPort,
		    PS_IPTABLES, _tableName, iface, matchMPort2, tproto, matchMPort, openMPort);
    } else {
        sprintf(_bufBind, "%s -I %s -o %s %s --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s", 
		    PS_IPTABLES, _tableName, iface, mproto, matchMPort2, tproto, matchMPort, openMPort);
        sprintf(_bufUnBind, "%s -D %s -o %s %s --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s", 
		    PS_IPTABLES, _tableName, iface, mproto, matchMPort2, tproto, matchMPort, openMPort);
    }

    if (ruleOpAdd(rl, _bufBind, 1) < 0 || ruleOpAdd(rl, _bufUnBind, 0) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInit(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInsertRule(&rs, group, rl) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetWrite(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    return 0;
}

static int subRuleProto(const char *group, char *args) {
	Ruleset rs;
	Rule *rl = NULL;
        char formated[MAX_CONFIG_LINE + 1];
	char *sProto, *brkp;
	int protonum;
        int flag = 0;

        if (! args)
        	return -1;

	if (strcmp(group, "ipv4"))
		return -1;
	
        parserAddress(1, sProto, flag);
	protonum = atoi(sProto);

	if ((protonum < 1) || (protonum > 254))   // ЭŴ
		return -1;
	
        /*  */
        snprintf(formated, MAX_CONFIG_LINE, "%s: %d", _method, protonum);
        if (! (rl = ruleAlloc(formated)))
        	return -1;
        /* iptables */
	sprintf(_bufBind, "%s -A FORWARD -p %d -j ACCEPT", 
		    PS_IPTABLES, protonum);
	sprintf(_bufUnBind, "%s -D FORWARD -p %d -j ACCEPT", 
		    PS_IPTABLES, protonum);

	if (ruleOpAdd(rl, _bufBind, 1) < 0 || ruleOpAdd(rl, _bufUnBind, 0) < 0) {
		ruleDestroy(rl);
		return -1;
	}
	if (rulesetInit(&rs) < 0) {
		ruleDestroy(rl);
		return -1;
	}
	if (rulesetInsertRule(&rs, group, rl) < 0) {
		ruleDestroy(rl);
		return -1;
	}
	if (rulesetWrite(&rs) < 0) {
		ruleDestroy(rl);
		return -1;
	}
	return 0;
}

/* policy trigger NAME (tcp|udp) matchMPort openMPort */
static int subRuleDomainBlock(const char *group,
                    char *args)
{
    Ruleset rs;
    Rule *rl = NULL;
    char *webPort, *domain, *brkp, *modifier;
    char formated[MAX_CONFIG_LINE + 1];
    int flag = 0;
    
    if (! args)
        return -1;

    parserAddress(1, webPort, flag);
    if (! test_port(webPort)) {
	return -1;   /* web˿ڴ */
    }

    if (!(domain = strtok_r(NULL, ARGS_DELIM, &brkp)))
	    return -1;

    /*  */
    snprintf(formated, MAX_CONFIG_LINE, "%s: %s %s", _method, webPort, domain);
    if (! (rl = ruleAlloc(formated)))
        return -1;

    /* iptables */
    sprintf(_bufBind, "%s -I %s -p tcp --dport %s -m webstr --host %s -j DROP", 
		    PS_IPTABLES, _tableName, webPort, domain);
    sprintf(_bufUnBind, "%s -D %s -p tcp --dport %s -m webstr --host %s -j DROP", 
		    PS_IPTABLES, _tableName, webPort, domain);

    if (ruleOpAdd(rl, _bufBind, 1) < 0 || ruleOpAdd(rl, _bufUnBind, 0) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInit(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInsertRule(&rs, group, rl) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetWrite(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    return 0;
}

/* access-list KEY (permit|deny|reset) (tcp|udp) */
/*             [~]A.B.C.D/M [~]<1-65535>[:<1-65535>] */
/*             [~]A.B.C.D/M [~]<1-65535>[:<1-65535>] */
/*             [day WD-WD] [time HH:MM-HH:MM] */
static int subRule1(const char *group,
                    const char *action,
                    const char *protocol,
                    char *args)
{
    Ruleset rs;
    Rule *rl = NULL;
    char *srcIP, *srcPort, *dstIP, *dstPort, *brkp, *modifier;
    char msrcIP[MAX_CONFIG_LINE + 1], msrcPort[MAX_CONFIG_LINE + 1], mdstIP[MAX_CONFIG_LINE + 1], mdstPort[MAX_CONFIG_LINE + 1];
    int msrcFlagExIP, msrcFlagExPort, mdstFlagExIP, mdstFlagExPort; 
    int srcFlagExIP, srcFlagExPort, dstFlagExIP, dstFlagExPort;
    char *tcpflags;
    char *wday, *time,  *state;
    char formated[MAX_CONFIG_LINE + 1];
    
    tcpflags = srcIP = srcPort = dstIP = dstPort = brkp = modifier = (char*) 0;
    srcFlagExIP = srcFlagExPort = dstFlagExIP = dstFlagExPort = 0;
    msrcFlagExIP = msrcFlagExPort = mdstFlagExIP = mdstFlagExPort = 0;
    state = wday = time = (char*) 0;
    
    if (! args)
        return -1;

    parserAddress(1, srcIP, srcFlagExIP);
    if (! RULE_TEST_IP(srcIP)) { 
	parserMRange(srcIP, "ip", msrcIP, msrcFlagExIP);
	if (msrcFlagExIP == 0) {
			return -1;   /* ԴIP */
	}
    }

    parserAddress(0, srcPort, srcFlagExPort);
    if (! RULE_TEST_PORT(srcPort)) {
	parserMRange(srcPort, protocol, msrcPort, msrcFlagExPort);
	if (msrcFlagExPort == 0)
	    return -1;   /* Դ˿ڴ */
    }

    parserAddress(0, dstIP, dstFlagExIP);
    if (! RULE_TEST_IP(dstIP)) {  
	parserMRange(dstIP, "ip", mdstIP, mdstFlagExIP);
	if (mdstFlagExIP == 0) {
			return -1;   /* ĿIP */
	}
    }

    parserAddress(0, dstPort, dstFlagExPort);
    if (! RULE_TEST_PORT(dstPort)) {
	parserMRange(dstPort, protocol, mdstPort, mdstFlagExPort);
	if (mdstFlagExPort == 0)
            return -1;   /* ĿĶ˿ڴ */
    }

    if (! strcmp("tcp", protocol)) {
        if (parserOptional(brkp, &wday, &time, &state, &tcpflags) < 0)
            return -1;              /* ѡ */
    }else {
        if (parserOptional(brkp, &wday, &time, &state, NULL) < 0)
            return -1;
    }

    /*  */
    snprintf(formated, MAX_CONFIG_LINE, "%s: %s %s %s%s %s%s %s%s %s%s",
	     _method,
             action, protocol,
             srcFlagExIP ? "!" : "", srcIP, srcFlagExPort ? "!" : "", srcPort,
             dstFlagExIP ? "!" : "", dstIP, dstFlagExPort ? "!" : "", dstPort);
    if (tcpflags) {
        strcat(formated, " tcpflags ");
        strcat(formated, tcpflags);
    }
    COMBIN_OPTIONAL();
    if (! (rl = ruleAlloc(formated)))
        return -1;
             
    /* iptables */
    sprintf(_bufBind, "%s -I %s -p %s", PS_IPTABLES, _tableName, protocol);
    sprintf(_bufUnBind, "%s -D %s -p %s", PS_IPTABLES, _tableName, protocol);

    if (msrcFlagExIP == 0) {
        instOptionAddress(_bufBind, srcIP, "-s", srcFlagExIP);
        instOptionAddress(_bufUnBind, srcIP, "-s", srcFlagExIP);
    } else if (msrcFlagExIP != 0) {
	    if (!strchr(msrcIP, ',')) {
		    if (!strchr(msrcIP, '-')) {
			    instOptionAddress(_bufBind, srcIP, "-s", srcFlagExIP);
			    instOptionAddress(_bufUnBind, srcIP, "-s", srcFlagExIP);
		    } else {
			    strcat(_bufBind, " -m iprange");
			    strcat(_bufBind, srcFlagExIP ? " ! " : " ");
			    strcat(_bufBind, "--src-range ");
			    strcat(_bufBind, msrcIP);

			    strcat(_bufUnBind, " -m iprange");
			    strcat(_bufUnBind, srcFlagExIP ? " ! " : " ");
			    strcat(_bufUnBind, "--src-range ");
			    strcat(_bufUnBind, msrcIP);
		    }
	    }
    }
    
    if (msrcFlagExPort == 0) {
    	instOptionAddress(_bufBind, srcPort, "--source-port", srcFlagExPort);
        instOptionAddress(_bufUnBind, srcPort, "--source-port", srcFlagExPort);
    } else {
    	instOptionAddress(_bufBind, msrcPort, "-m multiport --source-port", srcFlagExPort);
    	instOptionAddress(_bufUnBind, msrcPort, "-m multiport --source-port", srcFlagExPort);
    }    

    if (mdstFlagExIP == 0) {
    	instOptionAddress(_bufBind, dstIP, "-d", dstFlagExIP);
        instOptionAddress(_bufUnBind, dstIP, "-d", dstFlagExIP);
    } else if (msrcFlagExIP != 0) {
	    if (!strchr(mdstIP, ',')) {
		    if (!strchr(mdstIP, '-')) {
			    instOptionAddress(_bufBind, dstIP, "-d", dstFlagExIP);
    			    instOptionAddress(_bufUnBind, dstIP, "-d", dstFlagExIP);
		    } else {
			    strcat(_bufBind, " -m iprange");
			    strcat(_bufBind, dstFlagExIP ? " ! " : " ");
			    strcat(_bufBind, "--dst-range ");
			    strcat(_bufBind, mdstIP);

			    strcat(_bufUnBind, " -m iprange");
			    strcat(_bufUnBind, dstFlagExIP ? " ! " : " ");
			    strcat(_bufUnBind, "--dst-range ");
			    strcat(_bufUnBind, mdstIP);
		    }
	    }
    }

    if (mdstFlagExPort == 0) {
    	instOptionAddress(_bufBind, dstPort, "--destination-port", dstFlagExPort);
	instOptionAddress(_bufUnBind, dstPort, "--destination-port", dstFlagExPort);
    } else {
    	instOptionAddress(_bufBind, mdstPort, "-m multiport --destination-port", dstFlagExPort);
    	instOptionAddress(_bufUnBind, mdstPort, "-m multiport --destination-port", dstFlagExPort);
    }

    instOptionTcpFlags(_bufBind, tcpflags);
    instOptionOptional(_bufBind, wday, time, state);
    
    instOptionTcpFlags(_bufUnBind, tcpflags);
    instOptionOptional(_bufUnBind, wday, time, state);
        
    strcat(_bufBind, " -j ");
    strcat(_bufBind, iptAction(action));
    strcat(_bufUnBind, " -j ");
    strcat(_bufUnBind, iptAction(action));
    if (ruleOpAdd(rl, _bufBind, 1) < 0 || ruleOpAdd(rl, _bufUnBind, 0) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInit(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInsertRule(&rs, group, rl) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetWrite(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    return 0;
}


/* access-list KEY (permit|deny) icmp type (ping | pong | source-quench | */
/*             destination-unreach | redirect | router-advertisement | */
/*             router-solicitation | ttl-exceeded | parameter-problem | */
/*             timestamp-request | timestamp-reply | address-mask-request | */
/*             address-mask-reply) [~]A.B.C.D/M [~]A.B.C.D/M */
/*             [day WD-WD] [time HH:MM-HH:MM] */
static int subRule2(const char *group,
                    const char *action,
                    char *args)
{
    Ruleset rs;
    Rule *rl = NULL;
    char *modifier, *type;
    char *srcIP, *dstIP, *brkp;
    int srcFlagExIP, dstFlagExIP, msrcFlagExIP, mdstFlagExIP;
    char *wday, *time, *state;
    char formated[MAX_CONFIG_LINE + 1];
    char msrcIP[MAX_CONFIG_LINE + 1], mdstIP[MAX_CONFIG_LINE + 1];

    srcIP = dstIP = modifier = type = state = wday = time = brkp = (char*) 0;
    srcFlagExIP = dstFlagExIP = msrcFlagExIP = mdstFlagExIP = 0;
    
    if (! args ||
        ! (modifier = strtok_r(args, ARGS_DELIM, &brkp)) ||
        strcmp("type", modifier) ||
        ! (type = strtok_r(NULL, ARGS_DELIM, &brkp)))
        return -1;
    if (! test_icmptype(type))
        return -1;

    parserAddress(0, srcIP, srcFlagExIP);
    if (! RULE_TEST_IP(srcIP)) {
	    parserMRange(srcIP, "ip", msrcIP, msrcFlagExIP);
	    if (msrcFlagExIP == 0) {
			    return -1;
	    }
    }

    parserAddress(0, dstIP, dstFlagExIP);
    if (! RULE_TEST_IP(dstIP)) {
	    parserMRange(dstIP, "ip", mdstIP, mdstFlagExIP);
	    if (mdstFlagExIP == 0) {
			    return -1;
	    }
    }

	if (parserOptional(brkp, &wday, &time, &state, NULL) < 0)
		return -1;

    snprintf(formated, MAX_CONFIG_LINE, "%s: %s icmp type %s %s%s %s%s",
	     _method,
             action, type,
             srcFlagExIP ? "!" : "", srcIP,
             dstFlagExIP ? "!" : "", dstIP);
    COMBIN_OPTIONAL();

    if (! (rl = ruleAlloc(formated)))
        return -1;
    
    /* iptables */
    sprintf(_bufBind, "%s -I %s -p icmp --icmp-type %s", PS_IPTABLES, _tableName, type);
    sprintf(_bufUnBind, "%s -D %s -p icmp --icmp-type %s", PS_IPTABLES, _tableName, type);
	

    if (msrcFlagExIP == 0) {
	instOptionAddress(_bufBind, srcIP, "-s", srcFlagExIP);
	instOptionAddress(_bufUnBind, srcIP, "-s", srcFlagExIP);
    } else if (msrcFlagExIP !=0 ) {
	    if (!strchr(msrcIP, ',')) {
                    if (!strchr(msrcIP, '-')) {
			    instOptionAddress(_bufBind, srcIP, "-s", srcFlagExIP);
			    instOptionAddress(_bufUnBind, srcIP, "-s", srcFlagExIP);
		    } else {
                            strcat(_bufBind, " -m iprange");
                            strcat(_bufBind, srcFlagExIP ? " ! " : " ");
                            strcat(_bufBind, "--src-range ");
                            strcat(_bufBind, msrcIP);

                            strcat(_bufUnBind, " -m iprange");
                            strcat(_bufUnBind, srcFlagExIP ? " ! " : " ");
                            strcat(_bufUnBind, "--src-range ");
                            strcat(_bufUnBind, msrcIP);
                    }
	    }
    }

    if (mdstFlagExIP == 0) {
	instOptionAddress(_bufBind, dstIP, "-d", dstFlagExIP);
	instOptionAddress(_bufUnBind, dstIP, "-d", dstFlagExIP);
    } else if (msrcFlagExIP != 0) {
	    if (!strchr(mdstIP, ',')) {
		    if (!strchr(mdstIP, '-')) {
			    instOptionAddress(_bufBind, dstIP, "-d", dstFlagExIP);
		    } else {
			    strcat(_bufBind, " -m iprange");
			    strcat(_bufBind, dstFlagExIP ? " ! " : " ");
			    strcat(_bufBind, "--dst-range ");
			    strcat(_bufBind, mdstIP);

			    strcat(_bufUnBind, " -m iprange");
			    strcat(_bufUnBind, dstFlagExIP ? " ! " : " ");
			    strcat(_bufUnBind, "--dst-range ");
			    strcat(_bufUnBind, mdstIP);
		    }
	    }
    }

    instOptionOptional(_bufBind, wday, time, state);
        
    instOptionOptional(_bufUnBind, wday, time, state);
        
    strcat(_bufBind, " -j ");
    strcat(_bufBind, iptAction(action));
    strcat(_bufUnBind, " -j ");
    strcat(_bufUnBind, iptAction(action));
    if (ruleOpAdd(rl, _bufBind, 1) < 0 || ruleOpAdd(rl, _bufUnBind, 0) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInit(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInsertRule(&rs, group, rl) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetWrite(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    return 0;
}



/* access-list KEY (permit|deny) (ip | icmp | igmp | ipip | ahp | esp) */
/*             [~]A.B.C.D/M [~]A.B.C.D/M */
/*             [day WD-WD] [time HH:MM-HH:MM] */
static int subRule3(const char *group,
                    const char *action,
                    const char *protocol,
                    char *args)
{
    Ruleset rs;
    Rule *rl = NULL;
    char *srcIP, *dstIP, *brkp;
    int srcFlagExIP, dstFlagExIP, msrcFlagExIP, mdstFlagExIP;
    char *wday, *time, *state;
    char formated[MAX_CONFIG_LINE + 1], msrcIP[MAX_CONFIG_LINE + 1], mdstIP[MAX_CONFIG_LINE + 1];

    srcIP = dstIP = brkp = state = wday = time = (char*) 0;
    srcFlagExIP = dstFlagExIP = msrcFlagExIP = mdstFlagExIP = 0;
    
    if (! args)
        return -1;

    parserAddress(1, srcIP, srcFlagExIP);
    if (! RULE_TEST_IP(srcIP)) {
	    parserMRange(srcIP, "ip", msrcIP, msrcFlagExIP);
	    if (msrcFlagExIP == 0) {
			    return -1;
	    }
    }
    
    parserAddress(0, dstIP, dstFlagExIP);
    if (! RULE_TEST_IP(dstIP)) {
	    parserMRange(dstIP, "ip", mdstIP, mdstFlagExIP);
	    if (mdstFlagExIP == 0) {
			    return -1;
	    }
    }

    if (parserOptional(brkp, &wday, &time, &state, NULL) < 0)
        return -1;

    snprintf(formated, MAX_CONFIG_LINE, "%s: %s %s %s%s %s%s",
	     _method,
             action, protocol,
             srcFlagExIP ? "!" : "", srcIP,
             dstFlagExIP ? "!" : "", dstIP);
    COMBIN_OPTIONAL();
    if (! (rl = ruleAlloc(formated)))
        return -1;
    
    /* iptables */
    sprintf(_bufBind, "%s -I %s -p %s", PS_IPTABLES, _tableName, protocol);
    sprintf(_bufUnBind, "%s -D %s -p %s", PS_IPTABLES, _tableName, protocol);

    if (msrcFlagExIP == 0) {
	instOptionAddress(_bufBind, srcIP, "-s", srcFlagExIP);
	instOptionAddress(_bufUnBind, srcIP, "-s", srcFlagExIP);
    } else if (msrcFlagExIP != 0) {
            if (!strchr(msrcIP, ',')) {
                    if (!strchr(msrcIP, '-')) {
                            instOptionAddress(_bufBind, srcIP, "-s", srcFlagExIP);
                    } else {
                            strcat(_bufBind, " -m iprange");
                            strcat(_bufBind, srcFlagExIP ? " ! " : " ");
                            strcat(_bufBind, "--src-range ");
                            strcat(_bufBind, msrcIP);

			    strcat(_bufUnBind, " -m iprange");
			    strcat(_bufUnBind, srcFlagExIP ? " ! " : " ");
			    strcat(_bufUnBind, "--src-range ");
			    strcat(_bufUnBind, msrcIP);
                    }
            }
    }

    if (mdstFlagExIP == 0) {
        instOptionAddress(_bufBind, dstIP, "-d", dstFlagExIP);
	instOptionAddress(_bufUnBind, dstIP, "-d", dstFlagExIP);
    } else if (msrcFlagExIP != 0) {
            if (!strchr(mdstIP, ',')) {
                    if (!strchr(mdstIP, '-')) {
                            instOptionAddress(_bufBind, dstIP, "-d", dstFlagExIP);
                            instOptionAddress(_bufUnBind, dstIP, "-d", dstFlagExIP);
                    } else {
                            strcat(_bufBind, " -m iprange");
                            strcat(_bufBind, dstFlagExIP ? " ! " : " ");
                            strcat(_bufBind, "--dst-range ");
                            strcat(_bufBind, mdstIP);

                            strcat(_bufUnBind, " -m iprange");
                            strcat(_bufUnBind, dstFlagExIP ? " ! " : " ");
                            strcat(_bufUnBind, "--dst-range ");
                            strcat(_bufUnBind, mdstIP);
                    }
            }
    }

    instOptionOptional(_bufBind, wday, time, state);
        
    instOptionOptional(_bufUnBind, wday, time, state);
        
    strcat(_bufBind, " -j ");
    strcat(_bufBind, iptAction(action));
    strcat(_bufUnBind, " -j ");
    strcat(_bufUnBind, iptAction(action));
    if (ruleOpAdd(rl, _bufBind, 1) < 0 || ruleOpAdd(rl, _bufUnBind, 0) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInit(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetInsertRule(&rs, group, rl) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    if (rulesetWrite(&rs) < 0) {
        ruleDestroy(rl);
        return -1;
    }
    return 0;
}

int rulesetAdd(const char *what, char *rfwrule)
{
	char *group, *action, *protocol, *brkp;

	_method = what;
	if (! _method)
		return -1;

	if (! strcmp("portfw", _method)) {
		_tableName = "PREROUTING";
	}else {
		_tableName = "FORWARD";
	}

	if (! rfwrule || ! (group = strtok_r(rfwrule, ARGS_DELIM, &brkp)))
		return -1;

	if (! strcmp("permproto", _method)) {
		return subRuleProto(group, brkp);
	}

	if (! strcmp("domainblock", _method)) {
		return subRuleDomainBlock(group, brkp);
	}
		
	/*  allow, deny */
	if (! (action = strtok_r(NULL, ARGS_DELIM, &brkp)))
		return -1;              /* ûָ */

	if (! strcmp("portfw", _method)) {
		protocol = action;
		return subRulePF(group, protocol, brkp);
	}
		
	if (! strcmp("portfw_lan", _method)) {
		protocol = action;
		return subRulePFLAN(group, protocol, brkp);
	}
		
	if (! strcmp("trigger", _method)) {
		protocol = action;
		return subRuleTRIG(group, protocol, brkp);
	}

	if (bindGetInterface(group, (char*) 0) == 0)
		return -1;              /* Ѿ */


	if (strcmp("allow", action) && strcmp("deny", action) && strcmp("reset", action))
		return -1;              /* Ķ */
	
	if (! (protocol = strtok_r(NULL, ARGS_DELIM, &brkp)))
		return -1;              /* ûָЭ */

	if (! strcmp("tcp", protocol) || ! strcmp("udp", protocol)) {
		return subRule1(group, action, protocol, brkp);
	} else if (! strcmp("ip", protocol)) {
		if (! brkp)
			return rpsErrorArguments();
		return subRule3(group, action, protocol, brkp);

	} else if (! strcmp("icmp", protocol)) {
		if (! brkp)
			return rpsErrorArguments();
		if (! strstr(brkp, "type"))
			return subRule3(group, action, protocol, brkp);
		return subRule2(group, action, brkp);

	} else if (! strcmp("igmp", protocol) || ! strcmp("aph", protocol) ||
              ! strcmp("ipip", protocol) || ! strcmp("esp", protocol)) {
		return subRule3(group, action, protocol, brkp);
	}

    return -1;                  /* ﷨ */
}

static int strReplace(char *str, const char *from, const char *to)
{
    char *p = NULL;
    char part2[MAXPATHLEN + 1];
    
    if (! str || ! from || !to)
        return -1;
    p = strstr(str, from);
    if (p) {
        *p = '\0'; p += strlen(from);
        strcpy(part2, p);
        strcat(str, to);
        strcat(str, part2);
    }
    return 0;
}

static int rulesetRuleCall(Ruleset *rs, const char *name, const char *iface, int bind)
{
    vlistEntry *iter1, *iter2, *iter3;
    RuleGroup *group = NULL;
    Rule *rule = NULL;
    vlist *cmdlist = NULL;
    char cmd[MAXPATHLEN + 1];

    if (! rs || ! name || ! iface)
        return -1;

    iter1 = vlistFirstEntry(rs);
    for (; iter1; iter1 = vlistNextEntry(iter1)) {
        group = (RuleGroup*) iter1->elem;
        if (! strcmp(group->name, name)) {
            iter2 = vlistFirstEntry(& (group->rules));
            for (; iter2; iter2 = vlistNextEntry(iter2)) {
                rule = (Rule*) iter2->elem;
		if (bind)
                	cmdlist = &(rule->binds);
		else
			cmdlist = &(rule->unbinds);
                iter3 = vlistFirstEntry(cmdlist);
                for (; iter3; iter3 = vlistNextEntry(iter3)) {
                    strcpy(cmd, (char*)iter3->elem);
                    
                    system(cmd);
                    
                }
            }
			return 0;
        }
    }
    return -1;
}

int rulesetBind(Ruleset *rs, const char *name, const char *iface)
{
    return rulesetRuleCall(rs, name, iface, 1);
}

int rulesetUnBind(Ruleset *rs, const char *name, const char *iface)
{
    return rulesetRuleCall(rs, name, iface, 0);
}

int rulesetMove(const char *name, int from, int to)
{
	Ruleset rs;
	vlistEntry *iter1 = NULL;
	RuleGroup *group = NULL;
	
	if (rulesetInit(&rs) < 0)
		return -1;
	iter1 = vlistFirstEntry(&rs);
	for (; iter1; iter1 = vlistNextEntry(iter1)) {
		group = (RuleGroup*) iter1->elem;
		if (! strcmp(group->name, name))
			break;
	}
	if (! iter1) {
		rulesetDestroy(&rs);
		return -1;
	}
	if (vlistMove(&(group->rules), from, to) < 0) {
		rulesetDestroy(&rs);
		return -1;
	}
	if (rulesetWrite(&rs) < 0) {
		rulesetDestroy(&rs);
		return -1;
	}
	rulesetDestroy(&rs);
	return 0;
}
