#include <limits.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "parser.h"
#include "alstr.h"

static char _bufContent[PARSER_CONTENT_LENGTH + 1];
static char _bufKey[PARSER_KEY_LENGTH + 1];
static char _bufValue[PARSER_VALUE_LENGTH + 1];
static char *_currContent = NULL;

static int _newEntry = 0;
static int _lineno = 1;
char *defaultContent = "global";

static void parserEntryDestroy(void *ventry)
{
    parserEntry *entry = (parserEntry *) ventry;

    if (entry->key)
        free(entry->key);
    if (entry->value)
        free(entry->value);

    free(entry);
}

static parserEntry *parserEntryAlloc(const char *key, const char *value)
{
    parserEntry *entry = NULL;
    entry = (parserEntry *) malloc(sizeof(parserEntry));
    if (entry) {
        int err = 0;

        entry->key = NULL;
        entry->value = NULL;

        if (! err && key) {
            entry->key = strdup(key);
            if (NULL == entry->key)
                err = 1;
        }

        if (! err && value) {
            entry->value = strdup(value);
            if (NULL == entry->value)
                err = 1;
        }

        if (err) {
            parserEntryDestroy(entry);
            entry = NULL;
        }
    }
    return entry;
}

static void parserContentDestroy(void *vct)
{
    parserContent *ct = (parserContent *)vct;

    if (ct->content)
        free(ct->content);
    vlistDestroy(&(ct->entries), parserEntryDestroy);
    free(ct);
}

static parserContent *parserContentAlloc(const char *content)
{
    parserContent *ct = NULL;

    ct = (parserContent *) malloc(sizeof(parserContent));
    if (ct) {
        vlistInit(&(ct->entries));

        ct->content = strdup(content);
        if (NULL == ct->content) {
            parserContentDestroy(ct);
            ct = NULL;
        }
    }
    return ct;
}

static int parserInsertLast(parser *psr, const char *content, const char *key, const char *value)
{
    int ret = -1, nct = 1;
    vlistEntry *iter = NULL;
    parserContent *lastct = NULL;
    parserEntry *ent = NULL;

    iter = vlistLastEntry(psr);
    if (iter) {
        lastct = (parserContent *) iter->elem;
        if (! strcmp(lastct->content, content))
            nct = 0;
    }

    ent = parserEntryAlloc(key, value);
    if (ent) {
        if (nct) {
            lastct = parserContentAlloc(content);
            if (lastct) {
                if (! vlistInsertLast(psr, lastct)) {
                    parserContentDestroy(lastct);
                    lastct = NULL;
                }
            }
        }
        if (lastct) {
            if (vlistInsertLast(&(lastct->entries), ent))
                ret = 0;
            else
                parserEntryDestroy(ent);
        }
    }
    return ret;
}

static int fpSkipCommentLine(FILE *fp)
{
    int ch = '\0';

    do {
        ch = fgetc(fp);
    }while (! feof(fp) && ! ferror(fp) && '\n' != ch);

    return ferror(fp) ? -1 : 0;
}

static int fpSkipBlankLine(FILE *fp)
{
    int ch = '\0';

    for(;;) {
        ch = fgetc(fp);
        if (ferror(fp))
            return -1;
        else if (feof(fp))
            return 0;
        else if ('\n' == ch)
            break;
        if (' ' != ch && '\t' != ch)
            return -1;
    }
    return 0;
}

static int fpDecodeContent(FILE *fp)
{
    int ret = -1;
    int ch = '[';
    int pos = 0;

    for (; ']' != ch;) {
        ch = fgetc(fp);
        if (ferror(fp) || feof(fp) || '\n' == ch)
            return -1;
        if (']' == ch)
            _bufContent[pos] = '\0';
        else {
            if (PARSER_CONTENT_LENGTH < pos)
                return -1;
            _bufContent[pos++] = ch;
            if (0 > ret)
                ret = 0;
        }
    }
    _currContent = _bufContent;

    return (0 == ret) ? fpSkipBlankLine(fp) : -1;
}

static int fpDecodeKey(FILE *fp, int deli, int nosptab)
{
    int ret = -1;
    int ch = '\0';
    int pos = 0;

    for (; deli != ch;) {
        ch = fgetc(fp);
        if (feof(fp) || ferror(fp) || '\n' == ch)
            return -1;

        else if (deli == ch)
            _bufKey[pos] = '\0';

        else {
            if (PARSER_KEY_LENGTH < pos)
                return -1;
            _bufKey[pos++] = ch;
            if (0 > ret)
                ret = 0;
        }
    }
    if (0 == ret && nosptab) {
        srms(_bufKey);
    }
    return ret;
}

static int fpDecodeValue(FILE *fp, int nosptab)
{
    int ch = '\0';
    int pos = 0;

    for (;;) {
        ch = fgetc(fp);

        if (ferror(fp))
            return -1;

        else if (feof(fp) || '\n' == ch)
            break;

        else {
            if (PARSER_VALUE_LENGTH < pos)
                return -1;
            _bufValue[pos++] = ch;
        }
    }
    _newEntry = 1;
    _bufValue[pos] = '\0';
    if (nosptab)
        srmps(_bufValue);
    return 0;
}

static int fpDecodeEntry(FILE *fp, int deli, int nosptab)
{
    if (fpDecodeKey(fp, deli, nosptab) < 0)
        return -1;
    return fpDecodeValue(fp, nosptab);
}

static int parserLine(FILE *fp, int deli, int nosptab)
{
    int ch = '\0';

    ch = fgetc(fp);
    if (ferror(fp))
        return -1;
    if (feof(fp) || '\n' == ch)
        return 0;

    else if ('#' == ch)
        return fpSkipCommentLine(fp);

    else if (' ' == ch || '\t' == ch)
        return fpSkipBlankLine(fp);

    else if (deli == ch)
        return -1;

    else if ('[' == ch)
        return fpDecodeContent(fp);

    if (ungetc(ch, fp) == EOF)
        return -1;

    return fpDecodeEntry(fp, deli, nosptab);
}

int parserInit(parser *psr, const char *path, int deli, int nosptab)
{
    int err = 0;
    FILE *fp = NULL;

    _lineno = 1;
    _currContent = defaultContent;
    vlistInit(psr);

    fp = fopen(path, "r");
    if (NULL != fp) {
        for (; ! err && ! feof(fp);) {
            _newEntry = 0;
            err = parserLine(fp, deli, nosptab);
            if (0 == err && _newEntry) {
                if (parserInsertLast(psr, _currContent, _bufKey, _bufValue) < 0)
                    err = 1;
            }
            if (0 == err)
                _lineno++;
        } /* for (;;) */
        fclose(fp);
    }else /* NULL != fp */
        err = 1;

    if (err)
        parserDestroy(psr);
    return err ? _lineno : 0;
}

int parserFirstContent(parser *psr, char **strval)
{
    int ret = -1;
    vlistEntry *contentIter = NULL;

    contentIter = vlistFirstEntry(psr);
    if (contentIter) {
        parserContent *ct = (parserContent *) contentIter->elem;
        *strval = ct->content;
        ret = 0;
    }
    return ret;
}

int parserNextContent(parser *psr, char **strval)
{
    int ret = -1;
    vlistEntry *contentIter = NULL;
    parserContent *ct = NULL;

    if (! (*strval))
        return parserFirstContent(psr, strval);

    for (contentIter = vlistFirstEntry(psr);
         contentIter;
         contentIter = vlistNextEntry(contentIter)) {
        ct = (parserContent *)contentIter->elem;
        if (! strcmp(*strval, ct->content))
            break;
    }
    if (contentIter) {
        contentIter = vlistNextEntry(contentIter);
        if (contentIter) {
            ct = (parserContent *)contentIter->elem;
            *strval = ct->content;
            ret = 0;
        }
    }
    return ret;
}

int parserStr(parser *psr, const char *content, const char *key, char **strval)
{
    int ret = -1;
    vlistEntry *contentIter = NULL;
    vlistEntry *entryIter = NULL;
    parserContent *ct = NULL;
    parserEntry *ent = NULL;

    if (strval)
        *strval = NULL;

    if (! content)
        content = defaultContent;

    for (contentIter = vlistFirstEntry(psr);
         contentIter;
         contentIter = vlistNextEntry(contentIter)) {
        ct = (parserContent *) contentIter->elem;
        if (! strcmp(ct->content, content))
            break;
    }

    if (contentIter) {
        /* content found, search entries */
        for (entryIter = vlistFirstEntry(&(ct->entries));
             entryIter && 0 != ret;
             entryIter = vlistNextEntry(entryIter)) {
            ent = (parserEntry *) entryIter->elem;
            if (! strcmp(ent->key, key)) {
                ret = 0;
                if (strval)
                    *strval = ent->value;
            }
        }
    }
    return ret;
}

int parserLong(parser *psr, const char *content, const char *key, long *longval)
{
    int ret = -1;
    char *strval = NULL;

    ret = parserStr(psr, content, key, &strval);
    if (0 == ret) {
        char *endc = NULL;
        long val = LONG_MIN;

        val = strtol(strval, &endc, 10);
        if (LONG_MIN != val && LONG_MAX != val &&
            endc != strval && *endc == '\0') {
            if (longval)
                *longval = val;
        }
    }
    return ret;
}

void parserDestroy(parser *psr)
{
    vlistDestroy(psr, parserContentDestroy);
}

int parserWrite(parser *psr, const char *path, int deli)
{
    FILE *fp = (FILE*) 0;
    vlistEntry *iter1, *iter2;
    parserContent *pctnt;
    parserEntry *pent;

    if (! (fp = fopen(path, "w")))
        return -1;

    for (iter1 = vlistFirstEntry(psr); iter1; iter1 = vlistNextEntry(iter1)) {
        pctnt = (parserContent*) iter1->elem;
        if (pctnt->content) {
            if (fprintf(fp, "[%s]\n", pctnt->content) < 0) {
                fclose(fp);
                return -1;
            }
        }
        /* write content item */
        for (iter2 = vlistFirstEntry(& pctnt->entries); iter2; iter2 = vlistNextEntry(iter2)) {
            pent = iter2->elem;
            if (fprintf(fp, "%s%c%s\n", pent->key, (char)deli, pent->value ? pent->value : "") < 0) {
                fclose(fp);
                return -1;
            }
        }
    }
    return fclose(fp);
}

int parserSetLong(parser *psr, const char *content, const char *key, long longval)
{
    char buf[65];
    
    snprintf(buf, 64, "%ld", longval);
    return parserSetStr(psr, content, key, buf);
}

int parserSetStr(parser *psr, const char *content, const char *key, const char *strval)
{
    vlistEntry *iter1, *iter2;
    parserContent *pctnt;
    parserEntry *pent;
    
    if (! content)
        content = defaultContent;
    for (iter1 = vlistFirstEntry(psr); iter1; iter1 = vlistNextEntry(iter1)) {
        pctnt = iter1->elem;
        if (! strcmp(content, pctnt->content))
            break;
    }
    if (! iter1) {              /* create content */
        if (! (pctnt = parserContentAlloc(content)))
            return -1;
        if (! vlistInsertLast(psr, pctnt)) {
            parserContentDestroy(pctnt);
            return -1;
        }
    }
    for (iter2 = vlistFirstEntry(&pctnt->entries); iter2; iter2 = vlistNextEntry(iter2)) {
        pent = iter2->elem;
        if (! strcmp(key, pent->key))
            break;
    }
    if (! iter2) {              /* create item */
        pent = parserEntryAlloc(key, strval);
        if (! vlistInsertLast(&pctnt->entries, pent)) {
            parserEntryDestroy(pent);
            return -1;
        }
    }else {
        if (pent->value) {
            free(pent->value);
            pent->value = (char*) 0;
        }
        if (strval)
            pent->value = strdup(strval);
    }
    
    return 0;
}

int parserRemove(parser *psr, const char *content, const char *key)
{
    vlistEntry *iter1;
    parserContent *pctnt;
    
    if (! content)
        content = defaultContent;
    for (iter1 = vlistFirstEntry(psr); iter1; iter1 = vlistNextEntry(iter1)) {
        pctnt = iter1->elem;
        if (! strcmp(content, pctnt->content))
            break;
    }
    if (! iter1)                /* content not found */
        return -1;
    if (! key) {
        /* remove all key below this content */
        vlistRemoveEntry(psr, iter1, parserContentDestroy);
    }else {
        vlistEntry *iter2;
        parserEntry *pent;
        
        for (iter2 = vlistFirstEntry(&pctnt->entries); iter2; iter2 = vlistNextEntry(iter2)) {
            pent = iter2->elem;
            if (! strcmp(key, pent->key))
                break;
        }
        if (! iter2)            /* key not found */
            return -1;
        vlistRemoveEntry(&pctnt->entries, iter2, parserEntryDestroy);
        if (vlistEmpty(&pctnt->entries))
            vlistRemoveEntry(psr, iter1, parserContentDestroy);
    }
    return 0;
}
