/*
 * Be created for GUI operation of Static Route, Dynamic Route and Routing List
 * By U-Media Ricky Cao on Mar. 05 2008
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <netdb.h>
#include <netinet/in.h>
#include <net/route.h>
#include <net/if.h>

#include "nvram.h"
#include "utils.h"
#include "webs.h"
#include "uemf.h"
#include "internet.h"
#include "routing.h"

/****************** Following Code is for CGI of "Routing" page operation ********************/

static int isPPPWan(void)
{
	char *wanType;
	
	wanType = nvram_bufget(RT2860_NVRAM, "wanConnectionMode");
	if(strcmp(wanType, "STATIC")&&strcmp(wanType, "DHCP")){
		return 1;
	}

	return 0;
}

static int isNumOnly(char *str){
	int i, len = strlen(str);
	for(i=0; i<len; i++){
		if((str[i] >= '0' && str[i] <= '9'))
			continue;
		return 0;
	}
	return 1;
}

static int isAllNumAndSlash(char *str){
	int i, len = strlen(str);
	printf("str = %s\n", str);
	for(i=0; i<len; i++){
		if( (str[i] >= '0' && str[i] <= '9') || str[i] == '.' || str[i] == '/' )
			continue;
		return 0;
	}
	return 1;
}

static int isOnlyOneSlash(char *str)
{
	int i, count=0;
	int len = strlen(str);
	for(i=0; i<len; i++)
		if( str[i] == '/')
			count++;
	return count <= 1 ? 1 : 0;
}

static int isIpNetmaskValid(char *s)
{
	char str[32];
	char *slash;
	struct in_addr addr;    // for examination

	if(!s || !strlen(s)){
		return 0;
	}

	strncpy(str, s, 32);

    if( (!strcmp("any", str)) || (!strcmp("any/0", str)))
        return 1;

	if (!isAllNumAndSlash(str)){
		return 0;
	}

	if(!isOnlyOneSlash(str)){
		return 0;
	}

	slash = strchr(str, '/');
	if(slash){
		int mask;

		*slash = '\0';
		slash++;
		if(!strlen(slash)){
			return 0;
		}

		if(!isNumOnly(slash)){
			return 0;
		}

		mask = atoi(slash);
		if(mask < 0 || mask > 32){
			return 0;
		}
	}

	if(! (inet_aton(str, &addr))){
        printf("isIpNetmaskValid(): %s is not a valid IP address.\n", str);
        return 0;
    }
    return 1;
}

static int isIpValid(char *str)
{
	struct in_addr addr;	// for examination
	if( (! strcmp(T("any"), str)) || (! strcmp(T("any/0"), str)))
		return 1;

	if(! (inet_aton(str, &addr))){
		printf("isIpValid(): %s is not a valid IP address.\n", str);
		return 0;
	}
	return 1;
}

//deleteRouteCommand() is used to delete the static route add command in "/var/staticRoute"
//StaticRoute is created for re-add static route after wan port cable re-connect
int deleteRouteCommand(char *destType, char *ip, char *netmask, char *gateway, char *metric, char *viaIf)
{
	char *command[512] = {0};
	char tmp_destType[8]={0}, tmp_ip[16]={0}, tmp_netmask[16]={0}, tmp_gateway[16]={0}, tmp_metric[8]={0}, tmp_viaIf[2]={0};
	int parameterCount = 0;
	FILE *fptr, *tmpfptr;

	fptr = fopen("/var/staticRoute", "r");
	tmpfptr = fopen("/var/temp_staticRoute", "w+");

	if(fptr==NULL || tmpfptr==NULL){
		return -1;
	}

	fputs("#!/bin/sh\n", tmpfptr);
	fgets(command, 512, fptr);//the first line is "#!/bin/sh", just skip it
	//printf("Read variables form static route list: %s %s %s %s %s %s\n", destType, ip, netmask, gateway, metric, viaIf);
	
	while(1){
		memset(command, 0, 512);
		fgets(command, 512, fptr);
		parameterCount = sscanf(command, "route add %s %s netmask %s gw %s metric %s dev %s", tmp_destType, tmp_ip, tmp_netmask, tmp_gateway, tmp_metric, tmp_viaIf);
		
		if(parameterCount==6){
			//printf("Read variables form /var/staticRoute: %s %s %s %s %s %s\n", tmp_destType, tmp_ip, tmp_netmask, tmp_gateway, tmp_metric, tmp_viaIf);
			if(strcmp(destType, tmp_destType) || strcmp(ip, tmp_ip) || strcmp(netmask, tmp_netmask) ||
				strcmp(gateway, tmp_gateway) || strcmp(metric, tmp_metric) || strcmp(viaIf, tmp_viaIf)){
				memset(command, 0, 512);
				sprintf(command, "route add %s %s netmask %s gw %s metric %s dev %s\n", tmp_destType, tmp_ip, tmp_netmask, tmp_gateway, tmp_metric, tmp_viaIf);
				//printf("command = %s", command);
				fputs(command, tmpfptr);
			}	
		}else{
			if(parameterCount<0&&feof(fptr)){
				break;
			}
		}
	}

	fclose(fptr);
	fclose(tmpfptr);
	system("rm -f /var/staticRoute");
	system("mv /var/temp_staticRoute /var/staticRoute");

	return 0;
}

int deleteRouteRecord(char *routeRecord, int i)
{
	char ip[16]={0}, netmask[16]={0}, gateway[16]={0}, metric[8]={0}, viaIf[2]={0};
	char viaIfStr[8]={0};
	char routeCmd[128] = {0};
	char destType[8] = {0};

	//1. get the "ip" setting in static route record
	if((getNthValueSafe(0, routeRecord, ',', ip, 16) == -1) || !isIpValid(ip)){
		printf("Incorrect ip setting: %s in %dth record of staticRoutList\n", ip, i);
		return 1;
	}

	//2. get the "netmask" setting in static route record
	if((getNthValueSafe(1, routeRecord, ',', netmask, 16) == -1) || !isIpNetmaskValid(netmask)){
		printf("Incorrect netmask setting: %s in %dth record of staticRoutList\n", netmask, i);
		return 1;
	}

	//3. get the "gateway" setting in static route record
		if((getNthValueSafe(2, routeRecord, ',', gateway, 16) == -1) || !isIpValid(gateway)){
		printf("Incorrect gateway setting: %s in %dth record of staticRoutList\n", gateway, i);
		return 1;
	}

	//4. get the "metric" setting in static route record
	if(getNthValueSafe(3, routeRecord, ',', metric, 8) == -1){
		printf("Incorrect metric setting: %s in %dth record of staticRoutList\n", metric, i);
		return 1;		
	}

	//5. get the "interface" setting in static route record, there are two value can be set: 0=wan, 1=lan/wlan, 2=ppp0
	if((getNthValueSafe(4, routeRecord, ',', viaIf, 2) == -1) || (strcmp(viaIf, "0")&&strcmp(viaIf, "1")&&strcmp(viaIf, "2"))){
		printf("Incorrect interface setting: %s in %dth record of staticRoutList\n", viaIf, i);
		return 1;
	}

	if(!strcmp(viaIf, "0")){
		strcpy(viaIfStr, "eth2.2");		
	}else if(!strcmp(viaIf, "1")){
		strcpy(viaIfStr, "br0");	
	}else if(!strcmp(viaIf, "2")){
		strcpy(viaIfStr, "ppp0");
	}

	if(strcmp(netmask, "0.0.0.0")){
		strcpy(destType, "-net");
	}else{
		strcpy(destType, "-host");
	}

	sprintf(routeCmd, "route del %s %s netmask %s gw %s metric %s dev %s", destType, ip, netmask, gateway, metric,  viaIfStr);
	//printf("del route command is: %s\n", routeCmd);
	doSystem(routeCmd);

	deleteRouteCommand(destType, ip, netmask, gateway, metric,  viaIfStr);

	return 0;
}

int deleteNthRouteMulti(int index[], int count, char *value, char delimit)
{
	char *begin, *end;
	int i=0,j=0;
	int need_check_flag=0;
	char staticRouteEntry[64] = {0}; 
	char *buf = strdup(value);

	begin = buf;

	end = strchr(begin, delimit);
	while(end){
		if(i == index[j]){
			memset(staticRouteEntry, 0, 64);
			if(i==0){
				memcpy(staticRouteEntry, begin, end-begin);
			}else{
				memcpy(staticRouteEntry, begin+1, end-(begin+1));
			}
			deleteRouteRecord(staticRouteEntry, j);
			memset(begin, 0, end - begin );
			if(index[j] == 0)
				need_check_flag = 1;
			j++;
			if(j >=count)
				break;
		}
		begin = end;

		end = strchr(begin+1, delimit);
		i++;
	}

	if(!end && index[j] == i){
		memset(staticRouteEntry, 0, 64);
		if(i==0){
			memcpy(staticRouteEntry, begin, strlen(begin));
		}else{
			memcpy(staticRouteEntry, begin+1, strlen(begin+1));
		}
		deleteRouteRecord(staticRouteEntry, j);
		memset(begin, 0, strlen(begin));
	}

	if(need_check_flag){
		for(i=0; i<strlen(value); i++){
			if(buf[i] == '\0')
				continue;
			if(buf[i] == ';')
				buf[i] = '\0';
			break;
		}
	}

	for(i=0, j=0; i<strlen(value); i++){
		if(buf[i] != '\0'){
			value[j++] = buf[i];
		}
	}
	value[j] = '\0';

	free(buf);
	return 0;
}

static int getNums(char *value, char delimit)
{
	char *pos = value;
    	int count=1;
		
    	if(!pos)
    		return 0;
		while( (pos = strchr(pos, delimit))){
			pos = pos+1;
			count++;
		}
		
	return count;
}

inline int getStaticRouteNums(char *Route){
	return getNums(Route, ';');
}

/* numeric: & 0x8000: default instead of *,
 *          & 0x4000: host instead of net,
 *          & 0x0fff: don't resolve
 */
char *INET_rresolve(struct sockaddr_in *s_in, int numeric, uint32_t netmask)
{
	/* addr-to-name cache */
	struct addr {
		struct addr *next;
		struct sockaddr_in addr;
		int host;
		char name[1];
	};
	static struct addr *cache = NULL;

	struct addr *pn;
	char *name;
	uint32_t ad, host_ad;
	int host = 0;

	if (s_in->sin_family != AF_INET) {
		printf("rresolve: unsupported address family %d!\n", s_in->sin_family);
		return NULL;
	}
	ad = s_in->sin_addr.s_addr;
	//printf("rresolve: %08x, mask %08x, num %08x\n", (unsigned)ad, netmask, numeric);
	if (ad == INADDR_ANY) {
		if ((numeric & 0x0FFF) == 0) {
			if (numeric & 0x8000)
				return strdup("default");
			return strdup("*");
		}
	}
	if (numeric & 0x0FFF)
		return strdup(inet_ntoa(s_in->sin_addr));

	if ((ad & (~netmask)) != 0 || (numeric & 0x4000))
		host = 1;
	pn = cache;
	while (pn) {
		if (pn->addr.sin_addr.s_addr == ad && pn->host == host) {
			printf("rresolve of routing.c: found %s %08x in cache\n",
					  (host ? "host" : "net"), (unsigned)ad);
			return strdup(pn->name);
		}
		pn = pn->next;
	}

	host_ad = ntohl(ad);
	name = NULL;
	if (host) {
		struct hostent *ent;
		ent = gethostbyaddr((char *) &ad, 4, AF_INET);
		if (ent)
			name = strdup(ent->h_name);
	} 
	
	if (!name)
		name = strdup(inet_ntoa(s_in->sin_addr));
	pn = malloc(sizeof(*pn) + strlen(name)); /* no '+ 1', it's already accounted for */
	pn->next = cache;
	pn->addr = *s_in;
	pn->host = host;
	strcpy(pn->name, name);
	cache = pn;
	return name;
}


 /****************** Following Code is the CGI of "Routing" page ********************/
static int exposeWanOptionASP(int eid, webs_t wp, int argc, char_t **argv)
{
	char if_addr[16] = {0};

	websWrite(wp, T("<option value=\"0\" >WAN</option>"));
	websWrite(wp, T("<option value=\"1\" >LAN/WLAN</option>"));
	if(!getIfLive("ppp0") && (getIfIp("ppp0", if_addr)!=-1)){
		websWrite(wp, T("<option value=\"2\" >PPP</option>"));
	}

	return 0;
}

 //Export the routing table to GUI
static int showRoutingTableASP(int eid, webs_t wp, int argc, char_t **argv)
{
	char devname[64], /*flags[16],*/ *sdest, *sgw;
	unsigned long d, g, m;
	int flgs, ref, use, metric, mtu, win, ir;
	struct sockaddr_in s_addr;
	struct in_addr mask;
	int routeEntryCount=0;

	FILE *fp = fopen("/proc/net/route", "r");

	//printf("Kernel IP routing table\n"
	//       "Destination     Gateway         Genmask         Flags %s Iface\n", "Metric Ref    Use\n");

	if (fscanf(fp, "%*[^\n]\n") < 0) { /* Skip the first line. */
		goto ERROR;		   /* Empty or missing line, or read error. */
	}
	while (1) {
		int r;
		r = fscanf(fp, "%63s%lx%lx%X%d%d%d%lx%d%d%d\n",
				   devname, &d, &g, &flgs, &ref, &use, &metric, &m,
				   &mtu, &win, &ir);
		if (r != 11) {
			if ((r < 0) && feof(fp)) { /* EOF with no (nonspace) chars read. */
				fclose(fp);
				break;
			}
 ERROR:
 			fclose(fp);
			break;
		}

		if (!(flgs & 0x0001)) { /* Skip interfaces that are down. */
			continue;
		}

		memset(&s_addr, 0, sizeof(struct sockaddr_in));
		s_addr.sin_family = AF_INET;
		s_addr.sin_addr.s_addr = d;
		sdest = INET_rresolve(&s_addr, 0x0FFF, m); /* 'default' instead of '*' */
		s_addr.sin_addr.s_addr = g;
		sgw = INET_rresolve(&s_addr, 0x0FFF, m); /* Host instead of net */
		mask.s_addr = m;
		/* "%15.15s" truncates hostnames, do we really want that? */
		//printf("%-15.15s %-15.15s %-16s%-6s", sdest, sgw, inet_ntoa(mask), flags);
		//printf("%-6d %-2d %7d %s\n", metric, ref, use, devname);

		routeEntryCount++;
		
		//export this record to GUI
		websWrite(wp, T("<tr>\n"));
		websWrite(wp, T("	<td class=\"form_list_content\" align=center> %s </td>"), sdest); //export the ip setting
		websWrite(wp, T("	<td class=\"form_list_content\" align=center> %s </td>"), inet_ntoa(mask)); //export the netmask setting
		websWrite(wp, T("	<td class=\"form_list_content\" align=center> %s </td>"), sgw); //export the gateway setting
		websWrite(wp, T("	<td class=\"form_list_content\" align=center> %d </td>"), metric); //export the metric setting
		websWrite(wp, T("	<td class=\"form_list_content\" align=center> %s </td>"), (strcmp(devname, "eth2.2")&&strcmp(devname, "ppp0"))?"LAN/WLAN":(!strcmp(devname, "eth2.2")?"WAN":"PPP")); //export the metric setting
		websWrite(wp, T("</tr>\n"));

		free(sdest);
		free(sgw);
	}
	
	return 0;
}

//Export the static route list to GUI
static int showStaticRouteListASP(int eid, webs_t wp, int argc, char_t **argv)
{
	int i=0;
	char routeRecord[64];
	char ip[16]={0}, netmask[16]={0}, gateway[16]={0}, metric[8]={0}, viaIf[2]={0};
	char *staticRouteList = nvram_bufget(RT2860_NVRAM, "staticRouteList");

	if(!staticRouteList || !strlen(staticRouteList)){
		printf("staticRoutList is NULL\n");
		return 0;
	}

	//printf("staticRouteList = %s\n", staticRouteList);
	
	//the fomat of route record as below
	//[ip],[netmask],[gateway],[metric],[viaIf],;
	while(getNthValueSafe(i++, staticRouteList, ';', routeRecord, 64) != -1){
		//1. get the "ip" setting in static route record
		if((getNthValueSafe(0, routeRecord, ',', ip, 16) == -1) || !isIpValid(ip)){
			printf("Incorrect ip setting: %s in %dth record of staticRoutList\n", ip, i);
			continue;
		}

		//2. get the "netmask" setting in static route record
		if((getNthValueSafe(1, routeRecord, ',', netmask, 16) == -1) || !isIpNetmaskValid(netmask)){
			printf("Incorrect netmask setting: %s in %dth record of staticRoutList\n", netmask, i);
			continue;
		}

		//3. get the "gateway" setting in static route record
		if((getNthValueSafe(2, routeRecord, ',', gateway, 16) == -1) || !isIpValid(gateway)){
			printf("Incorrect gateway setting: %s in %dth record of staticRoutList\n", gateway, i);
			continue;
		}

		//4. get the "metric" setting in static route record
		if(getNthValueSafe(3, routeRecord, ',', metric, 8) == -1){
			printf("Incorrect metric setting: %s in %dth record of staticRoutList\n", metric, i);
			continue;		
		}

		//5. get the "interface" setting in static route record, there are two value can be set: 0=wan, 1=lan/wlan
		if((getNthValueSafe(4, routeRecord, ',', viaIf, 2) == -1) || (strcmp(viaIf, "0")&&strcmp(viaIf, "1")&&strcmp(viaIf, "2"))){
			printf("Incorrect interface setting: %s in %dth record of staticRoutList\n", viaIf, i);
			continue;
		}

		//export this record to GUI
		websWrite(wp, T("<tr>\n"));
		websWrite(wp, T("	<td class=\"form_list_content\"> %d&nbsp; <input type=\"checkbox\" id=\"delRouteID%d\" name=\"delRoute%d\" style=\"margin:0 0 0 0;\"> </td>"), i, i, i-1 );
		websWrite(wp, T("	<td id=\"routeEntry%d_ip\" class=\"form_list_content\" align=center> %s </td>"), i, ip); //export the ip setting
		websWrite(wp, T("	<td id=\"routeEntry%d_netmask\" class=\"form_list_content\" align=center> %s </td>"), i, netmask); //export the netmask setting
		websWrite(wp, T("	<td id=\"routeEntry%d_gateway\" class=\"form_list_content\" align=center> %s </td>"), i, gateway); //export the gateway setting
		websWrite(wp, T("	<td id=\"routeEntry%d_metric\" class=\"form_list_content\" align=center> %s </td>"), i, metric); //export the metric setting
		websWrite(wp, T("	<td id=\"routeEntry%d_If\" class=\"form_list_content\" align=center> %s </td>"), i, !strcmp(viaIf, "1")?"LAN/WLAN":(!strcmp(viaIf, "0")?"WAN":"PPP")); //export the metric setting
		websWrite(wp, T("</tr>\n"));
	}
	websWrite(wp, T("	<input type=\"hidden\" name=\"routeEntryNums\" value=\"%d\">"), --i);

	return 0;
}

//Add the static route entry
static void addStaticRoute(webs_t wp, char_t *path, char_t *query)
{
	char *ip, *netmask, *gateway, *metric, *viaIf, *destType, *wanName;
	char *staticRoutList; 
	char staticRouteList_new[1024] = {0};
	int viaIf_int = 0;
	char routeCmd[128] = {0};
	FILE *fptr;

	ip = websGetVar(wp, T("staticRoute_IP_setting"), T(""));
	netmask = websGetVar(wp, T("staticRoute_Netmask_setting"), T(""));
	gateway = websGetVar(wp, T("staticRoute_Gateway_setting"), T(""));
	metric = websGetVar(wp, T("staticRoute_Metric_setting"), T(""));
	viaIf = websGetVar(wp, T("staticRoute_If_setting"), T(""));
	viaIf_int = atoi(viaIf);
	destType = websGetVar(wp, T("staticRoute_destType_setting"), T(""));
	wanName = websGetVar(wp, T("staticRoute_destType_setting"), T(""));

	//Check the destination IP in static route setting
	if(!ip && !strlen(ip))
		return;
	if(!isIpValid(ip))
		return;

	//Check the netmask in static route setting
	if(!netmask && !strlen(netmask))
		return;
	if(!isIpNetmaskValid(netmask))
		return;

	//Check the destination IP in static route setting
	if(!gateway && !strlen(gateway))
		return;
	if(!isIpValid(gateway))
		return;

	if(!metric && !strlen(metric))
		return;
	if(!isNumOnly(metric))
		return;

	if(( staticRoutList = nvram_bufget(RT2860_NVRAM, "staticRouteList")) && strlen( staticRoutList) )
		snprintf(staticRouteList_new, 1024, "%s;%s,%s,%s,%s,%s",  staticRoutList, ip, netmask, gateway, metric, viaIf);
	else
		snprintf(staticRouteList_new, 1024, "%s,%s,%s,%s,%s", ip, netmask, gateway, metric, viaIf);

	nvram_set(RT2860_NVRAM, "staticRouteList", staticRouteList_new);
	nvram_commit(RT2860_NVRAM);	

	sprintf(routeCmd, "route add %s %s netmask %s gw %s metric %s dev %s\n", destType, ip, netmask, gateway, metric,  (viaIf_int==1)?"br0":((viaIf_int==2)?"ppp0":"eth2.2"));
	doSystem(routeCmd);

	fptr=fopen("/var/staticRoute", "a");
	fputs(routeCmd, fptr);
	fclose(fptr);

	//websRedirect(wp, "/advanced/routing.asp");
	websRedirect(wp, "/applied.asp?url=/advanced/routing.asp");
}

//Delete the static route entry
static void delStaticRoute(webs_t wp, char_t *path, char_t *query)
{
	int staticRouteCount, i, j;
	char *value;
	int *deleArray;
	char_t name_buf[16];
	char *staticRouteList;

	staticRouteList = nvram_bufget(RT2860_NVRAM, "staticRouteList");
	if(!staticRouteList || !strlen(staticRouteList) )
       	return;
	
	staticRouteCount = getStaticRouteNums(staticRouteList);
	if(!staticRouteCount)
		return;

	deleArray = (int *)malloc(staticRouteCount * sizeof(int));

	for(i=0, j=0; i< staticRouteCount; i++){
		snprintf(name_buf, 16, "delRoute%d", i);
		value = websGetVar(wp, T(name_buf), NULL);
		if(value){
			deleArray[j++] = i;
		}
	}

   	 if(!j){
	 	free(deleArray);
       	websHeader(wp);
        	websWrite(wp, T("You didn't select any rules to delete.<br>\n"));
        	websFooter(wp);
        	websDone(wp, 200);
        	return;
    	}

	deleteNthRouteMulti(deleArray, staticRouteCount, staticRouteList, ';');
	nvram_set(RT2860_NVRAM, "staticRouteList", staticRouteList);
	nvram_commit(RT2860_NVRAM);

	//websRedirect(wp, "/advanced/routing.asp");
	websRedirect(wp, "/applied.asp?url=/advanced/routing.asp");

	free(deleArray);
}

//Define the cgi functions for GUI to use those functions
void formDefineRouting(void) {
	websAspDefine(T("exposeWanOptionASP"), exposeWanOptionASP);
	websAspDefine(T("showRoutingTableASP"), showRoutingTableASP);
	websAspDefine(T("showStaticRouteListASP"), showStaticRouteListASP);
	websFormDefine(T("addStaticRoute"), addStaticRoute);
	websFormDefine(T("delStaticRoute"), delStaticRoute);
}

int configureStaticRoute(char *staticRouteList){
	int i=0;
	char ip[16]={0}, netmask[16]={0}, gateway[16]={0}, metric[8]={0}, viaIf[2]={0};
	char routeRecord[64] = {0};
	char routeCmd[128] = {0};
	char destType[8] = "-net";
	FILE *fptr;
	fptr = fopen("/var/staticRoute", "w+");

	fputs("#!/bin/sh\n", fptr);
	
	//the fomat of route record as below
	//[ip],[netmask],[gateway],[metric],[viaIf],;
	while(getNthValueSafe(i++, staticRouteList, ';', routeRecord, 64) != -1){
		//1. get the "ip" setting in static route record
		if((getNthValueSafe(0, routeRecord, ',', ip, 16) == -1) || !isIpValid(ip)){
			printf("Incorrect ip setting: %s in %dth record of staticRoutList\n", ip, i);
			continue;
		}

		//2. get the "netmask" setting in static route record
		if((getNthValueSafe(1, routeRecord, ',', netmask, 16) == -1) || !isIpNetmaskValid(netmask)){
			printf("Incorrect netmask setting: %s in %dth record of staticRoutList\n", netmask, i);
			continue;
		}

		//3. get the "gateway" setting in static route record
		if((getNthValueSafe(2, routeRecord, ',', gateway, 16) == -1) || !isIpValid(gateway)){
			printf("Incorrect gateway setting: %s in %dth record of staticRoutList\n", gateway, i);
			continue;
		}

		//4. get the "metric" setting in static route record
		if(getNthValueSafe(3, routeRecord, ',', metric, 8) == -1){
			printf("Incorrect metric setting: %s in %dth record of staticRoutList\n", metric, i);
			continue;		
		}

		//5. get the "interface" setting in static route record, there are two value can be set: 0=wan, 1=lan/wlan
		if((getNthValueSafe(4, routeRecord, ',', viaIf, 2) == -1) || (strcmp(viaIf, "0")&&strcmp(viaIf, "1")&&strcmp(viaIf, "2"))){
			printf("Incorrect interface setting: %s in %dth record of staticRoutList\n", viaIf, i);
			continue;
		}

		if(strcmp(netmask, "0.0.0.0")){
			strcpy(destType, "-net");
		}else{
			strcpy(destType, "-host");
		}

		//Set up the static route entry
		sprintf(routeCmd, "route add %s %s netmask %s gw %s metric %s dev %s\n", destType, ip, netmask, gateway, metric,  !strcmp(viaIf, "1")?"br0":(!strcmp(viaIf, "0")?"eth2.2":"ppp0"));
		fputs(routeCmd, fptr);
		doSystem(routeCmd);
	}

	fclose(fptr);
	system("chmod 755 /var/staticRoute");
	return 0;
}

int initStaticRoute(void) {
	char *staticRouteList = nvram_bufget(RT2860_NVRAM, "staticRouteList"); 

	if(!staticRouteList || !strlen(staticRouteList)){
		if(staticRouteList){
			printf("There is not any static route be set up!\n");
		}
		return -1;
	}

	printf("Configure the static route\n");
	configureStaticRoute(staticRouteList);
		
	return 0;
}

