/* vi: set sw=4 ts=4 sts=4 fdm=marker: */
/*
 *	internet.c -- Internet Settings
 *
 *	Copyright (c) Ralink Technology Corporation All Rights Reserved.
 *
 */

#include	<stdlib.h>
#include	<sys/ioctl.h>
#include	<net/if.h>
#include	<net/route.h>
#include    <string.h>
#include    <dirent.h>
#include	"internet.h"
#include	"nvram.h"
#include	"webs.h"
#include	"utils.h"
#include 	"firewall.h"
#include	"management.h"
#include	"station.h"
#include	"wireless.h"
#include <signal.h>

#include	"linux/autoconf.h"  //kernel config
#include	"config/autoconf.h" //user config
#include	"user/busybox/include/autoconf.h" //busybox config

#ifdef CONFIG_RALINKAPP_SWQOS
#include      "qos.h"
#endif

//Tom.Hung 2010-1-27, include ralink_gpio.h for GPIO check dip switch, port from usr/sysd/sysd.c
#include "ralink_gpio.h"

#define GPIO_DEV	"/dev/gpio"
#ifdef CONFIG_DIP_SWITCH
  #define DIP_SWITCH		CONFIG_DIP_SWITCH_GPIO_NUM	//define in TEW672GR_config.vendor-2.4x
  #ifdef CONFIG_DIP_SWITCH_FULL_HI
	#define DIP_SWITCH_ON	1	//raido on: 1, raido off: 0
	#define DIP_SWITCH_OFF	0	//raido on: 1, raido off: 0
  #else
	#define DIP_SWITCH_ON	0	//raido on: 0, raido off: 1
	#define DIP_SWITCH_OFF	1	//raido on: 0, raido off: 1
  #endif
#endif
//Tom.Hung 2010-1-27

inline void zebraRestart(void);
void ripdRestart(void);

static void DHCPReserveAdd(webs_t wp, char_t *path, char_t *query)
{
	int rule_count;
	char rule[4096], rec[256];
	char *action, *ruleNameID, *ruleName, *ruleEnable, *ip, *mac;
	char oldRuleName[32], oldruleEnable[32], oldipaddress[32], oldmacaddress[32];
	char *rules = nvram_bufget(RT2860_NVRAM, "DHCPReserve");
	
	action = websGetVar(wp, T("action"), T(""));
	ruleNameID = websGetVar(wp, T("ruleNameID"), T("0"));
	ruleEnable = websGetVar(wp, T("ruleEnable"), T("0"));
	ruleName = websGetVar(wp, T("ruleName"), T(""));
	ip = websGetVar(wp, T("ipaddress"), T(""));
	mac = websGetVar(wp, T("macaddress"), T(""));
	
	if (!strcmp(action, "edit"))
	{
		int *deleArray;
		int i=0, j=0;
		rule_count = getRuleNums(rules);
		if(!rule_count)
			return;
		deleArray = (int *)malloc(rule_count * sizeof(int));
		if (ruleNameID)
		{
			deleArray[j++] = atoi(ruleNameID);
		}
		
		//get old rule name
		if (getNthValueSafe(atoi(ruleNameID), rules, ';', rec, sizeof(rec)) != -1 )
		{
			//printf("jacky - ruleNameID=%d rec:%s\n", atoi(ruleNameID), rec);
			if((getNthValueSafe(0, rec, ',', oldRuleName, sizeof(oldRuleName)) == -1)){
			}
		}
		
		deleteNthValueMulti(deleArray, rule_count, rules, ';');
		free(deleArray);
	}

	//Tom.Hung 1-Dec-2008, Check if the IP or MAC address is already set.
	if(rules && strlen(rules)) {
		int i=0;
		while(getNthValueSafe(i++, rules, ';', rec, sizeof(rec)) != -1 ){
			if((getNthValueSafe(0, rec, ',', oldRuleName, sizeof(oldRuleName)) == -1)){
				continue;
			}
			if((getNthValueSafe(1, rec, ',', oldruleEnable, sizeof(oldruleEnable)) == -1)){
				continue;
			}
			if((getNthValueSafe(2, rec, ',', oldmacaddress, sizeof(oldmacaddress)) == -1)){
				continue;
			}
			if((getNthValueSafe(3, rec, ',', oldipaddress, sizeof(oldipaddress)) == -1)){
				continue;
			}
			if(!strcmp(oldmacaddress, mac)) {
				websWrite(wp, T("<script language=\"JavaScript\" type=\"text/javascript\">"));
				websWrite(wp, T("alert(\"This MAC Address is already set a DHCP Reservations.\");"));
				websWrite(wp, T("location.href = \"/internet/lan.asp\";"));
				websWrite(wp, T("</script>"));
				websDone(wp, 200);
				return 0;
			}
			if(!strcmp(oldipaddress, ip)) {
				websWrite(wp, T("<script language=\"JavaScript\" type=\"text/javascript\">"));
				websWrite(wp, T("alert(\"This IP Address is already set for other MAC Address.\");"));
				websWrite(wp, T("location.href = \"/internet/lan.asp\";"));
				websWrite(wp, T("</script>"));
				websDone(wp, 200);
				return 0;
			}
		}
	}
	//Tom.Hung 1-Dec-2008

	if(rules && strlen( rules) ) {
		snprintf(rule, sizeof(rule), "%s;%s,%s,%s,%s",  rules, ruleName, ruleEnable, mac, ip);
	}
	else {
		snprintf(rule, sizeof(rule), "%s,%s,%s,%s", ruleName, ruleEnable, mac, ip);
	}
	
	nvram_set(RT2860_NVRAM, "DHCPReserve", rule);
	//doSystem("lan.sh");
	//websRedirect(wp, "/internet/lan.asp");
	websRedirect(wp, "/applied2.asp?url=/internet/lan.asp");
}

static void DHCPReserveDelete(webs_t wp, char_t *path, char_t *query)
{
	int i, j, rule_count;
	char_t name_buf[16];
	char_t *value, *ruleName, *action;
	int *deleArray;
	char iptablesRuleName[64];

	action = websGetVar(wp, T("action"), T(""));

	if (!strcmp(action, "delete"))
	{
    	char *rules = nvram_bufget(RT2860_NVRAM, "DHCPReserve");
    	if(!rules || !strlen(rules) )
        	return;
    
		rule_count = getRuleNums(rules);
		if(!rule_count)
			return;
	
		deleArray = (int *)malloc(rule_count * sizeof(int));
	
		value = websGetVar(wp, T("ruleNameID"), T(""));	

		i=j=0;
		if(value){
			snprintf(name_buf, 16, "ruleName%d", atoi(value));
			ruleName = websGetVar(wp, T(name_buf), T(""));
		
			deleArray[j++] = atoi(value);
		}

		deleteNthValueMulti(deleArray, rule_count, rules, ';');
		free(deleArray);
		
		nvram_set(RT2860_NVRAM, "DHCPReserve", rules);
	}
	//doSystem("lan.sh");
  //websRedirect(wp, "/internet/lan.asp");
	websRedirect(wp, "/applied2.asp?url=/internet/lan.asp");
}

static int showDHCPReserveList(int eid, webs_t wp, int argc, char_t **argv)
{
	
	int i=0;
	char rec[256];
	char ruleEnable[4], ruleName[32], ipaddress[32], macaddress[32];
	char *rules = nvram_bufget(RT2860_NVRAM, "DHCPReserve");
	if(!rules)
		return 0;
	if(!strlen(rules))
		return 0;
	int listnumber=0;

	while(getNthValueSafe(i++, rules, ';', rec, sizeof(rec)) != -1 ){
		// get Rule Name
		if((getNthValueSafe(0, rec, ',', ruleName, sizeof(ruleName)) == -1)){
			continue;
		}
		
		if((getNthValueSafe(1, rec, ',', ruleEnable, sizeof(ruleEnable)) == -1)){
			continue;
		}
		if((getNthValueSafe(2, rec, ',', macaddress, sizeof(macaddress)) == -1)){
			continue;
		}
		if((getNthValueSafe(3, rec, ',', ipaddress, sizeof(ipaddress)) == -1)){
			continue;
		}
		

		websWrite(wp, T("<tr>\n"));
		if (!strcmp(ruleEnable, "on"))
			websWrite(wp, T("<td align=left style=\"word-break : break-all;\"> %d&nbsp; <input type=\"checkbox\" id=\"delRule%d\" name=\"delRule%d\" disabled=\"disabled\" checked=\"checked\"> </td>"), i, i-1, i-1 );
		else
			websWrite(wp, T("<td align=left style=\"word-break : break-all;\"> %d&nbsp; <input type=\"checkbox\" id=\"delRule%d\" name=\"delRule%d\" disabled=\"disabled\"> </td>"), i, i-1, i-1 );

		websWrite(wp, T("<input type=\"hidden\" id=\"ruleName%d\" name=\"ruleName%d\" value=\"%s\">"), i-1, i-1, ruleName);
		if(strlen(ruleName))
			websWrite(wp, T("<td id=\"ruleName%d\" align=center style=\"word-break : break-all;\">%s</td>"), i-1, ruleName);
		else
			websWrite(wp, T("<td align=center style=\"word-break : break-all;\">&nbsp;</td>"));
		
		websWrite(wp, T("<td id=\"ipaddress%d\" align=center style=\"word-break : break-all;\">%s</td>"), i-1, ipaddress);
		websWrite(wp, T("<td id=\"macaddress%d\" align=center style=\"word-break : break-all;\">%s</td>"), i-1, macaddress);
		websWrite(wp, T("<td align=center><a href=\"javascript:submit_apply('edit', %d);\"><img src=\"/Images/img_edit.gif\"></a></td>"), i-1); //rulenameID
		websWrite(wp, T("<td align=center><a href=\"javascript:submit_apply('delete', %d);\"><img src=\"/Images/img_delete.gif\"></a></td>"), i-1); //rulenameID
		websWrite(wp, T("</tr>\n"));
		listnumber++;
	}
	websWrite(wp, T("<input type=\"hidden\" id=\"drnum\" name=\"drnum\" value=\"%d\">\n"), listnumber);
	return 0;
}

/*
 * arguments: ifname  - interface name
 * description: test the existence of interface through /proc/net/dev
 * return: -1 = fopen error, 1 = not found, 0 = found
 */
int getIfLive(char *ifname)
{
	FILE *fp;
	char buf[256], *p;
	int i;

	if (NULL == (fp = fopen("/proc/net/dev", "r"))) {
		error(E_L, E_LOG, T("getIfLive: open /proc/net/dev error"));
		return -1;
	}

	fgets(buf, 256, fp);
	fgets(buf, 256, fp);
	while (NULL != fgets(buf, 256, fp)) {
		i = 0;
		while (isspace(buf[i++]))
			;
		p = buf + i - 1;
		while (':' != buf[i++])
			;
		buf[i-1] = '\0';
		if (!strcmp(p, ifname)) {
			fclose(fp);
			return 0;
		}
	}
	fclose(fp);
	error(E_L, E_LOG, T("getIfLive: device %s not found"), ifname);
	return 1;
}

/*
 * arguments: ifname  - interface name
 *            if_addr - a 18-byte buffer to store mac address
 * description: fetch mac address according to given interface name
 */
int getIfMac(char *ifname, char *if_hw)
{
	struct ifreq ifr;
	char *ptr;
	int skfd;

	if((skfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		error(E_L, E_LOG, T("getIfMac: open socket error"));
		return -1;
	}

	strncpy(ifr.ifr_name, ifname, IF_NAMESIZE);
	if(ioctl(skfd, SIOCGIFHWADDR, &ifr) < 0) {
		close(skfd);
		//error(E_L, E_LOG, T("getIfMac: ioctl SIOCGIFHWADDR error for %s"), ifname);
		return -1;
	}

	ptr = (char *)&ifr.ifr_addr.sa_data;
	sprintf(if_hw, "%02X:%02X:%02X:%02X:%02X:%02X",
			(ptr[0] & 0377), (ptr[1] & 0377), (ptr[2] & 0377),
			(ptr[3] & 0377), (ptr[4] & 0377), (ptr[5] & 0377));

	close(skfd);
	return 0;
}

/*
 * arguments: ifname  - interface name
 *            if_addr - a 16-byte buffer to store ip address
 * description: fetch ip address, netmask associated to given interface name
 */
int getIfIp(char *ifname, char *if_addr)
{
	struct ifreq ifr;
	int skfd = 0;

	if((skfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		error(E_L, E_LOG, T("getIfIp: open socket error"));
		return -1;
	}

	strncpy(ifr.ifr_name, ifname, IF_NAMESIZE);
	if (ioctl(skfd, SIOCGIFADDR, &ifr) < 0) {
		//error(E_L, E_LOG, T("getIfIp: ioctl SIOCGIFADDR error for %s"), ifname);
		close(skfd); //Port from SDK 2.3.0.0 by Ricky Cao on Oct. 01 2008
		return -1;
	}
	strcpy(if_addr, inet_ntoa(((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr));

	close(skfd);
	return 0;
}

/*
 * arguments: ifname - interface name
 * description: return 1 if interface is up
 *              return 0 if interface is down
 */
int getIfIsUp(char *ifname)
{
	struct ifreq ifr;
	int skfd;

	skfd = socket(AF_INET, SOCK_DGRAM, 0);
	if (skfd == -1) {
		perror("socket");
		return -1;
	}
	strncpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
	if (ioctl(skfd, SIOCGIFFLAGS, &ifr) < 0) {
		perror("ioctl");
		close(skfd);
		return -1;
	}
	close(skfd);
	if (ifr.ifr_flags & IFF_UP)
		return 1;
	else
		return 0;
}

/*
 * arguments: ifname - interface name
 *            if_net - a 16-byte buffer to store subnet mask
 * description: fetch subnet mask associated to given interface name
 *              0 = bridge, 1 = gateway, 2 = wirelss isp
 */
int getIfNetmask(char *ifname, char *if_net)
{
	struct ifreq ifr;
	int skfd = 0;

	if((skfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		error(E_L, E_LOG, T("getIfNetmask: open socket error"));
		return -1;
	}

	strncpy(ifr.ifr_name, ifname, IF_NAMESIZE);
	if (ioctl(skfd, SIOCGIFNETMASK, &ifr) < 0) {
		//error(E_L, E_LOG, T("getIfNetmask: ioctl SIOCGIFNETMASK error for %s\n"), ifname);
		close(skfd);//Add by Ricky Cao on Apr. 30 2008
		return -1;
	}
	strcpy(if_net, inet_ntoa(((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr));
	close(skfd);
	return 0;
}

/*
 * description: return WAN interface name
 *              0 = bridge, 1 = gateway, 2 = wirelss isp
 */
char* getWanIfName(void)
{
	char *mode = nvram_bufget(RT2860_NVRAM, "OperationMode");
	static char *if_name = "br0";

	char *cm;
    cm = nvram_bufget(RT2860_NVRAM, "wanConnectionMode");
    if (!strncmp(cm, "PPPOE", 6) || !strncmp(cm, "L2TP", 5) || !strncmp(cm, "PPTP", 5) || !strncmp(cm, "3G", 3))
        return "ppp0";

	if (NULL == mode)
		return if_name;
	if (!strncmp(mode, "0", 2))
		if_name = "br0";
	else if (!strncmp(mode, "1", 2)) {
//Tim Wang, for RTL8366SR driver
//#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW
#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW || defined CONFIG_RTL8366SR		
		if_name = "eth2.2";
#else /* MARVELL & CONFIG_ICPLUS_PHY */
		if_name = "eth2";
#endif
	}
	else if (!strncmp(mode, "2", 2))
		if_name = "ra0";
	else if (!strncmp(mode, "3", 2))
		if_name = "apcli0";
	return if_name;
}

char* getWanIfNamePPP(void)
{
    char *cm;
    cm = nvram_bufget(RT2860_NVRAM, "wanConnectionMode");
    if (!strncmp(cm, "PPPOE", 6) || !strncmp(cm, "L2TP", 5) || !strncmp(cm, "PPTP", 5) 
#ifdef CONFIG_USER_3G
		|| !strncmp(cm, "3G", 3)
#endif
	){
        return "ppp0";
	}

    return getWanIfName();
}


/*
 * description: return LAN interface name
 */
char* getLanIfName(void)
{
	char *mode = nvram_bufget(RT2860_NVRAM, "OperationMode");
	static char *if_name = "br0";

	if (NULL == mode)
		return if_name;
	if (!strncmp(mode, "0", 2))
		if_name = "br0";
	else if (!strncmp(mode, "1", 2)) {
//Tim Wang, for RTL8366SR driver
//#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW
#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW || defined CONFIG_RTL8366SR		
		if_name = "br0";
#elif defined  CONFIG_ICPLUS_PHY && CONFIG_RT2860V2_AP_MBSS
		char *num_s = nvram_bufget(RT2860_NVRAM, "BssidNum");
		if(atoi(num_s) > 1)	// multiple ssid
			if_name = "br0";
		else
			if_name = "ra0";
#else
		if_name = "ra0";
#endif
	}
	else if (!strncmp(mode, "2", 2)) {
		if_name = "eth2";
	}
	else if (!strncmp(mode, "3", 2)) {
		if_name = "br0";
	}
	return if_name;
}

/*
 * description: get the value "WAN" or "LAN" the interface is belong to.
 */
char *getLanWanNamebyIf(char *ifname)
{
	char *mode = nvram_bufget(RT2860_NVRAM, "OperationMode");

	if (NULL == mode)
		return "Unknown";

	if (!strcmp(mode, "0")){	// bridge mode
		if(!strcmp(ifname, "br0"))
			return "LAN";
		return ifname;
	}

	if (!strcmp(mode, "1")) {	// gateway mode
#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW
		if(!strcmp(ifname, "br0"))
			return "LAN";
		if(!strcmp(ifname, "eth2.2") || !strcmp(ifname, "ppp0"))
			return "WAN";
		return ifname;
#elif defined  CONFIG_ICPLUS_PHY && CONFIG_RT2860V2_AP_MBSS
		char *num_s = nvram_bufget(RT2860_NVRAM, "BssidNum");
		if(atoi(num_s) > 1 && !strcmp(ifname, "br0") )	// multiple ssid
			return "LAN";
		if(atoi(num_s) == 1 && !strcmp(ifname, "ra0"))
			return "LAN";
		if (!strcmp(ifname, "eth2") || !strcmp(ifname, "ppp0"))
			return "WAN";
		return ifname;
#else
		if(!strcmp(ifname, "ra0"))
			return "LAN";
		return ifname;
#endif
	}else if (!strncmp(mode, "2", 2)) {	// ethernet convertor
		if(!strcmp("eth2", ifname))
			return "LAN";
		if(!strcmp("ra0", ifname))
			return "WAN";
		return ifname;
	}else if (!strncmp(mode, "3", 2)) {	// apcli mode
		if(!strcmp("br0", ifname))
			return "LAN";
		if(!strcmp("apcli0", ifname))
			return "WAN";
		return ifname;
	}
	return ifname;
}


/*
 * description: write DHCP client list
 */
static int getDhcpCliList(int eid, webs_t wp, int argc, char_t **argv)
{
	FILE *fp;
	struct dhcpOfferedAddr {
		unsigned char hostname[16];
		unsigned char mac[16];
		unsigned long ip;
		unsigned long expires;
	} lease;
	int i;
	struct in_addr addr;
	unsigned long expires;
	unsigned d, h, m;

	doSystem("killall -q -USR1 udhcpd");

	fp = fopen("/var/udhcpd.leases", "r");
	if (NULL == fp)
		return websWrite(wp, T(""));
	while (fread(&lease, 1, sizeof(lease), fp) == sizeof(lease)) {
		addr.s_addr = lease.ip;
		expires = ntohl(lease.expires);
		d = expires / (24*60*60); expires %= (24*60*60);
		h = expires / (60*60); expires %= (60*60);
		m = expires / 60; expires %= 60;

		//Tom.Hung 2009-3-11, fix display all 00 MAC, and expire time 0 issue.
		char tmp[12];
		sprintf(tmp, "%02X%02X%02X%02X%02X%02X", lease.mac[0], lease.mac[1], lease.mac[2], lease.mac[3], lease.mac[4], lease.mac[5]);
		if(!strcmp(tmp, "000000000000") || (expires == 0 && d == 0 && h == 0 && m == 0)) {
			printf("mac = %s", tmp);
			printf("expires = %02u, d=%u, h=%02u, m=%02u", expires, d, h, m);
			continue;
		}
		//Tom.Hung 2009-3-11

		websWrite(wp, T("<tr><td>%02X"), lease.mac[0]);
		for (i = 1; i < 6; i++)
			websWrite(wp, T(":%02X"), lease.mac[i]);

		websWrite(wp, T("</td><td>%s</td><td>"), inet_ntoa(addr));

		if (d) websWrite(wp, T("%u days "), d);
		websWrite(wp, T("%02u:%02u:%02u\n"), h, m, (unsigned)expires);
	}
	fclose(fp);
	return 0;
}

/*
 * arguments: type - 1 = write Primary DNS
 *                   2 = write Secondary DNS
 * description: write DNS ip address accordingly
 */
static int getDns(int eid, webs_t wp, int argc, char_t **argv)
{
	FILE *fp;
	char buf[80] = {0}, ns_str[11], dns[16] = {0};
	int type, idx = 0, req = 0;

	if (ejArgs(argc, argv, T("%d"), &type) == 1) {
		if (1 == type)
			req = 1;
		else if (2 == type)
			req = 2;
		else
			return websWrite(wp, T(""));
	}

	fp = fopen("/etc/resolv.conf", "r");
	if (NULL == fp)
		return websWrite(wp, T(""));
	//Jacky.Yang Begin 19-Sep-2007
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		sscanf(buf, "%s%s", ns_str, dns);		
		if (strstr(buf, "nameserver") > 0)
		{
			idx++;
			if (idx == req) {
				websWrite(wp, T("%s"), dns);
				break;
			}
		}
	}
	fclose(fp);
	//Jacky.Yang End 19-Sep-2007	
	return 0;
}

/*
 * arguments: 
 * description: return 1 if hostname is supported
 */
static int getHostSupp(int eid, webs_t wp, int argc, char_t **argv)
{
#ifdef GA_HOSTNAME_SUPPORT
	ejSetResult(eid, "1");
#else
	ejSetResult(eid, "0");
#endif
	return 0;
}

/*
 * arguments: name - interface name (ex. eth0, rax ..etc)
 * description: write the existence of given interface,
 *              0 = ifc dosen't exist, 1 = ifc exists
 */
static int getIfLiveWeb(int eid, webs_t wp, int argc, char_t **argv)
{
	char_t *name;
	char exist[2] = "0";

	if (ejArgs(argc, argv, T("%s"), &name) < 1) {
		websError(wp, 400, T("Insufficient args\n"));
		return -1;
	}
	exist[0] = (getIfLive(name) == 0)? '1' : '0';
	return websWrite(wp, T("%s"), exist);
}

/*
 * arguments: name - interface name (ex. eth0, rax ..etc)
 * description: write the existence of given interface,
 *              0 = ifc is down, 1 = ifc is up
 */
static int getIfIsUpWeb(int eid, webs_t wp, int argc, char_t **argv)
{
	char_t *name;
	char up[2] = "1";

	if (ejArgs(argc, argv, T("%s"), &name) < 1) {
		websError(wp, 400, T("Insufficient args\n"));
		return -1;
	}
	up[0] = (getIfIsUp(name) == 1)? '1' : '0';
	return websWrite(wp, T("%s"), up);
}

static int getIgmpProxyBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#ifdef CONFIG_USER_IGMP_PROXY
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getVPNBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_NF_CONNTRACK_PPTP || defined CONFIG_NF_CONNTRACK_PPTP_MODULE || \
    defined CONFIG_IP_NF_PPTP        || defined CONFIG_IP_NF_PPTP_MODULE
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getMeshBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_RT2860V2_AP_MESH || defined CONFIG_RT2860V2_STA_MESH
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getWDSBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_RT2860V2_AP_WDS
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getWSCBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_RT2860V2_AP_WSC || defined CONFIG_RT2860V2_STA_WSC
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getMBSSIDBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_RT2860V2_AP_MBSS
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getUSBBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_USB
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getStorageBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_USB_STORAGE && defined CONFIG_USER_STORAGE
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getFtpBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_USER_STUPID_FTPD
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getSmbBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_USER_SAMBA
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getMediaBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_USB && defined CONFIG_USER_USHARE
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getWebCamBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_USB && defined CONFIG_USER_UVC_STREAM
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getPrinterSrvBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_USB && defined CONFIG_USER_P910ND
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getUSBiNICBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_INIC_USB 
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getDynamicRoutingBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_USER_ZEBRA
    return websWrite(wp, T("1"));
#else
    return websWrite(wp, T("0"));
#endif
}

static int getSWQoSBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_RALINKAPP_SWQOS
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getDATEBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_DATE
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getDDNSBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_USER_INADYN
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getSysLogBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_LOGREAD && defined CONFIG_KLOGD
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getETHTOOLBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_USER_ETHTOOL
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

/*
 * description: write LAN ip address accordingly
 */
static int getLanIp(int eid, webs_t wp, int argc, char_t **argv)
{
	char if_addr[16];

	if (-1 == getIfIp(getLanIfName(), if_addr)) {
		//websError(wp, 500, T("getLanIp: calling getIfIp error\n"));
		return websWrite(wp, T(""));
	}
	return websWrite(wp, T("%s"), if_addr);
}

/*
 * description: write LAN MAC address accordingly
 */
static int getLanMac(int eid, webs_t wp, int argc, char_t **argv)
{
	char if_mac[18];

	if (-1 == getIfMac(getLanIfName(), if_mac)) {
		//websError(wp, 500, T("getLanIp: calling getIfMac error\n"));
		return websWrite(wp, T(""));
	}
	return websWrite(wp, T("%s"), if_mac);
}

/*
 * arguments: type - 0 = return LAN interface name (default)
 *                   1 = write LAN interface name
 * description: return or write LAN interface name accordingly
 */
static int getLanIfNameWeb(int eid, webs_t wp, int argc, char_t **argv)
{
	int type;
	char *name = getLanIfName();

	if (ejArgs(argc, argv, T("%d"), &type) == 1) {
		if (1 == type) {
			return websWrite(wp, T("%s"), name);
		}
	}
	ejSetResult(eid, name);
	return 0;
}

/*
 * description: write LAN subnet mask accordingly
 */
static int getLanNetmask(int eid, webs_t wp, int argc, char_t **argv)
{
	char if_net[16];

	if (-1 == getIfNetmask(getLanIfName(), if_net)) {
		//websError(wp, 500, T("getLanNetmask: calling getIfNetmask error\n"));
		return websWrite(wp, T(""));
	}
	return websWrite(wp, T("%s"), if_net);
}

static int getGWBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_LAN_WAN_SUPPORT || defined CONFIG_MAC_TO_MAC_MODE
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getDnsmasqBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#ifdef CONFIG_USER_DNSMASQ
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getLltdBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_USER_LLTD && defined CONFIG_RT2860V2_AP_LLTD
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getPppoeRelayBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#ifdef CONFIG_USER_RPPPPOE_RELAY
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getUpnpBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#ifdef CONFIG_USER_UPNP_IGD
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getRadvdBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#ifdef CONFIG_USER_RADVD
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

/*
 * description: write WAN ip address accordingly
 */
static int getWanIp(int eid, webs_t wp, int argc, char_t **argv)
{
/*	char if_addr[16];

	if (-1 == getIfIp(getWanIfNamePPP(), if_addr)) {
		//websError(wp, 500, T("getWanIp: calling getIfIp error\n"));
		return websWrite(wp, T(""));
	}
	return websWrite(wp, T("%s"), if_addr);*/

	/*
	 * Since the wan interface is ppp0 when the wan type is set to PPPoE, L2TP or PPTP, so, we should
	 * get the wan ip from ppp0 interface
	 * Revised by U-Media Ricky Cao on Mar. 20 2008
	 */
	char if_addr[16];
	char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");
	
	if(!strcmp(connType, "PPPOE") || !strcmp(connType, "L2TP")  || !strcmp(connType, "PPTP") || !strcmp(connType, "RussiaPPTP") || !strcmp(connType, "3G")){
		if (-1 == getIfIp("ppp0", if_addr)) {
			//websError(wp, 500, T("getWanIp: calling getIfIp error\n"));
			return websWrite(wp, T(""));
		}		
	}else{
		if (-1 == getIfIp(getWanIfName(), if_addr)) {
			//websError(wp, 500, T("getWanIp: calling getIfIp error\n"));
			return websWrite(wp, T(""));
		}
	}
	return websWrite(wp, T("%s"), if_addr);	
}

/*
 * description: write WAN MAC address accordingly
 */
static int getWanMac(int eid, webs_t wp, int argc, char_t **argv)
{
	char if_mac[18];

	if (-1 == getIfMac(getWanIfName(), if_mac)) {
		//websError(wp, 500, T("getLanIp: calling getIfMac error\n"));
		return websWrite(wp, T(""));
	}
	return websWrite(wp, T("%s"), if_mac);
}

/*
 * arguments: type - 0 = return WAN interface name (default)
 *                   1 = write WAN interface name
 * description: return or write WAN interface name accordingly
 */
static int getWanIfNameWeb(int eid, webs_t wp, int argc, char_t **argv)
{
	int type;
	char *name = getWanIfName();

	if (ejArgs(argc, argv, T("%d"), &type) == 1) {
		if (1 == type) {
			return websWrite(wp, T("%s"), name);
		}
	}
	ejSetResult(eid, name);
	return 0;
}

/*
 * description: write WAN subnet mask accordingly
 */
static int getWanNetmask(int eid, webs_t wp, int argc, char_t **argv)
{
	char if_net[16];
	char *cm;

	cm = nvram_bufget(RT2860_NVRAM, "wanConnectionMode");
	//Add 3G for 3G modem support - Ricky Cao on Jan. 15 2008
	if (!strncmp(cm, "PPPOE", 6) || !strncmp(cm, "L2TP", 5) || !strncmp(cm, "PPTP", 5) 
#ifdef CONFIG_USER_3G
			|| !strncmp(cm, "3G", 3)
#endif
	){ //fetch ip from ppp0
		if (-1 == getIfNetmask("ppp0", if_net)) {
			return websWrite(wp, T(""));
		}
	}
	else if (-1 == getIfNetmask(getWanIfName(), if_net)) {
		//websError(wp, 500, T("getWanNetmask: calling getIfNetmask error\n"));
		return websWrite(wp, T(""));
	}
	return websWrite(wp, T("%s"), if_net);
}

/*
 * description: write WAN default gateway accordingly
 */
static int getWanGateway(int eid, webs_t wp, int argc, char_t **argv)
{
	char   buff[256];
	int    nl = 0 ;
	struct in_addr dest;
	struct in_addr gw;
	int    flgs, ref, use, metric;
	unsigned long int d,g,m;
	int    find_default_flag = 0;

	char sgw[16];

	FILE *fp = fopen("/proc/net/route", "r");
	//Add by Ricky Cao on Apr. 30 2008
	if(!fp){
		return websWrite(wp, T(""));
	}
	while (fgets(buff, sizeof(buff), fp) != NULL) {
		if (nl) {
			int ifl = 0;
			while (buff[ifl]!=' ' && buff[ifl]!='\t' && buff[ifl]!='\0')
				ifl++;
			buff[ifl]=0;    /* interface */
			if (sscanf(buff+ifl+1, "%lx%lx%X%d%d%d%lx",
						&d, &g, &flgs, &ref, &use, &metric, &m)!=7) {
				fclose(fp);
				return websWrite(wp, T("format error"));
			}

			if (flgs&RTF_UP) {
				dest.s_addr = d;
				gw.s_addr   = g;
				strcpy(sgw, (gw.s_addr==0 ? "" : inet_ntoa(gw)));

				if (dest.s_addr == 0) {
					find_default_flag = 1;
					break;
				}
			}
		}
		nl++;
	}
	fclose(fp);

	if (find_default_flag == 1)
		return websWrite(wp, T("%s"), sgw);
	else
		return websWrite(wp, T(""));
}


#define DD printf("%d\n", __LINE__);fflush(stdout);

/*
 *
 */
int getIndexOfRoutingRule(char *dest, char *netmask, char *interface)
{
	int index=0;
	char *rrs, one_rule[256];
	char dest_f[32], netmask_f[32], interface_f[32];
	rrs = nvram_bufget(RT2860_NVRAM, "RoutingRules");
	if(!rrs || !strlen(rrs))
		return -1;

	while( getNthValueSafe(index, rrs, ';', one_rule, 256) != -1 ){
		if((getNthValueSafe(0, one_rule, ',', dest_f, sizeof(dest_f)) == -1)){
			index++;
			continue;
		}
		if((getNthValueSafe(1, one_rule, ',', netmask_f, sizeof(netmask_f)) == -1)){
			index++;
			continue;
		}
		if((getNthValueSafe(4, one_rule, ',', interface_f, sizeof(interface_f)) == -1)){
			index++;
			continue;
		}
		//printf("@@@@@ %s %s %s\n", dest_f, netmask_f, interface_f);
		//printf("----- %s %s %s\n", dest, netmask, interface);
		if( (!strcmp(dest, dest_f)) && (!strcmp(netmask, netmask_f)) && (!strcmp(interface, interface_f))){
			return index;
		}
		index++;
	}

	return -1;
}

static void removeRoutingRule(char *dest, char *netmask, char *ifname)
{
	char cmd[1024];
	strcpy(cmd, "route del ");
	
	// host or net?
	if(!strcmp(netmask, "255.255.255.255") )
		strcat(cmd, "-host ");
	else
		strcat(cmd, "-net ");

	// destination
	strcat(cmd, dest);
	strcat(cmd, " ");

	// netmask
	if(strcmp(netmask, "255.255.255.255"))
		sprintf(cmd, "%s netmask %s", cmd, netmask);

	//interface
	sprintf(cmd, "%s dev %s ", cmd, ifname);
	doSystem(cmd);
}

void staticRoutingInit(void)
{
	int index=0;
	char one_rule[256];
	char *rrs;
	struct in_addr dest_s, gw_s, netmask_s;
	char dest[32], netmask[32], gw[32], interface[32], true_interface[32], custom_interface[32], comment[32];
	int	flgs, ref, use, metric, nl=0;
	unsigned long int d,g,m;
	int isGatewayMode = (!strcmp("1", nvram_bufget(RT2860_NVRAM, "OperationMode"))) ? 1 : 0 ;

	// delete old user rules
	FILE *fp = fopen("/proc/net/route", "r");
	if(!fp)
		return;

	while (fgets(one_rule, sizeof(one_rule), fp) != NULL) {
		if (nl) {
			if (sscanf(one_rule, "%s%lx%lx%X%d%d%d%lx",
					interface, &d, &g, &flgs, &ref, &use, &metric, &m) != 8) {
				printf("format error\n");
				fclose(fp);
				return;
			}
			dest_s.s_addr = d;
			gw_s.s_addr = g;
			netmask_s.s_addr = m;

			strncpy(dest, inet_ntoa(dest_s), sizeof(dest));
			strncpy(gw, inet_ntoa(gw_s), sizeof(gw));
			strncpy(netmask, inet_ntoa(netmask_s), sizeof(netmask));

			// check if internal routing rules
			if( (index=getIndexOfRoutingRule(dest, netmask, interface)) != -1){
				removeRoutingRule(dest, netmask, interface);
			}
		}
		nl++;
	}
	fclose(fp);

	index = 0;
	rrs = nvram_bufget(RT2860_NVRAM, "RoutingRules");
	if(!rrs|| !strlen(rrs))
		return;

	while( getNthValueSafe(index, rrs, ';', one_rule, 256) != -1 ){
		char cmd[1024];

		if((getNthValueSafe(0, one_rule, ',', dest, sizeof(dest)) == -1)){
			index++;
			continue;
		}
		if((getNthValueSafe(1, one_rule, ',', netmask, sizeof(netmask)) == -1)){
			index++;
			continue;
		}
		if((getNthValueSafe(2, one_rule, ',', gw, sizeof(gw)) == -1)){
			index++;
			continue;
		}
		if((getNthValueSafe(3, one_rule, ',', interface, sizeof(interface)) == -1)){
			index++;
			continue;
		}
		if((getNthValueSafe(4, one_rule, ',', true_interface, sizeof(true_interface)) == -1)){
			index++;
			continue;
		}
		if((getNthValueSafe(5, one_rule, ',', custom_interface, sizeof(custom_interface)) == -1)){
			index++;
			continue;
		}
		if((getNthValueSafe(6, one_rule, ',', comment, sizeof(comment)) == -1)){
			index++;
			continue;
		}

		strcpy(cmd, "route add ");
		
		// host or net?
		if(!strcmp(netmask, "255.255.255.255") )
			strcat(cmd, "-host ");
		else
			strcat(cmd, "-net ");

		// destination
		strcat(cmd, dest);
		strcat(cmd, " ");

		// netmask
		if(strcmp(netmask, "255.255.255.255") )
			sprintf(cmd, "%s netmask %s", cmd, netmask);

		// gateway
		if(strcmp(gw, "0.0.0.0"))
			sprintf(cmd, "%s gw %s", cmd, gw);

		//interface
//		if (!strcmp(interface, "WAN")){
//			true_interface = getWanIfName();
//		}else if (!gstrcmp(interface, "Custom")){
//			true_interface = custom_interface;
//		}else	// LAN & unknown
//			true_interface = getLanIfName();

		sprintf(cmd, "%s dev %s ", cmd, true_interface);

		strcat(cmd, "2>&1 ");

		if(strcmp(interface, "WAN") || (!strcmp(interface, "WAN") && isGatewayMode)  ){
			doSystem(cmd);
		}else{
			printf("Skip WAN routing rule in the non-Gateway mode: %s\n", cmd);
		}

		index++;
	}
	return;
}

void dynamicRoutingInit(void)
{
	zebraRestart();
	ripdRestart();
}

void RoutingInit(void)
{
	staticRoutingInit();
	//dynamicRoutingInit();
}

static inline int getNums(char *value, char delimit)
{
    char *pos = value;
    int count=1;
    if(!pos || !strlen(pos))
        return 0;
    while( (pos = strchr(pos, delimit))){
        pos = pos+1;
        count++;
    }
    return count;
}

/*
 * description: get routing table
 */
static int getRoutingTable(int eid, webs_t wp, int argc, char_t **argv)
{
	char   result[4096] = {0};
	char   buff[512];
	int    nl = 0, index;
	char   ifname[32], interface[128];
	struct in_addr dest, gw, netmask;
	char   dest_str[32], gw_str[32], netmask_str[32], comment[32];
	int    flgs, ref, use, metric;
	int	   *running_rules = NULL;
	unsigned long int d,g,m;
	char *rrs;
	int  rule_count;
	FILE *fp = fopen("/proc/net/route", "r");
	if(!fp)
		return -1;

	rrs = nvram_bufget(RT2860_NVRAM, "RoutingRules");
	rule_count = getNums(rrs, ';');

	if(rule_count){
		running_rules = calloc(1, sizeof(int) * rule_count);
		if(!running_rules)
			return -1;
	}
		
	strncat(result, "\"", sizeof(result));
	while (fgets(buff, sizeof(buff), fp) != NULL) {
		if (nl) {
			if (sscanf(buff, "%s%lx%lx%X%d%d%d%lx",
					ifname, &d, &g, &flgs, &ref, &use, &metric, &m) != 8) {
				printf("format error\n");
				fclose(fp);
				return websWrite(wp, T(""));
			}
			dest.s_addr = d;
			gw.s_addr = g;
			netmask.s_addr = m;

			if(! (flgs & 0x1) )	// skip not usable
				continue;

			strncpy(dest_str, inet_ntoa(dest), sizeof(dest_str));
			strncpy(gw_str, inet_ntoa(gw), sizeof(gw_str));
			strncpy(netmask_str, inet_ntoa(netmask), sizeof(netmask_str));

			if(nl > 1)
				strncat(result, ";", sizeof(result));
			strncat(result, ifname, sizeof(result));		strncat(result, ",", sizeof(result));
			strncat(result, dest_str, sizeof(result));		strncat(result, ",", sizeof(result));
			strncat(result, gw_str, sizeof(result));			strncat(result, ",", sizeof(result));
			strncat(result, netmask_str, sizeof(result) );	strncat(result, ",", sizeof(result));
			snprintf(result, sizeof(result), "%s%d,%d,%d,%d,", result, flgs, ref, use, metric);

			// check if internal routing rules
			strcpy(comment, " ");
			if( (index=getIndexOfRoutingRule(dest_str, netmask_str, ifname)) != -1){
				char one_rule[256];

				if(index < rule_count)
					running_rules[index] = 1;
				else
					printf("fatal error in %s\n", __FUNCTION__);

				snprintf(result, sizeof(result), "%s%d,", result, index);
				if(rrs && strlen(rrs)){
					if( getNthValueSafe(index, rrs, ';', one_rule, sizeof(one_rule)) != -1){

						if( getNthValueSafe(3, one_rule, ',', interface, sizeof(interface)) != -1){
							strncat(result, interface, sizeof(result));
							strncat(result, ",", sizeof(result));
						}
						if( getNthValueSafe(6, one_rule, ',', comment, sizeof(comment)) != -1){
							// do nothing;
						}
					}
				}
			}else{
				strncat(result, "-1,", sizeof(result));
				strncat(result, getLanWanNamebyIf(ifname), sizeof(result));
				strncat(result, ",", sizeof(result));
			}
			strncat(result, "0,", sizeof(result));	// used rule
			strncat(result, comment, sizeof(result));
		}
		nl++;
	}

	for(index=0; index < rule_count; index++){
		char one_rule[256];

		if(running_rules[index])
			continue;

		if(getNthValueSafe(index, rrs, ';', one_rule, sizeof(one_rule)) == -1)
			continue;

		if(getNthValueSafe(0, one_rule, ',', dest_str, sizeof(dest_str)) == -1)
			continue;

		if(getNthValueSafe(1, one_rule, ',', netmask_str, sizeof(netmask_str)) == -1)
			continue;

		if(getNthValueSafe(2, one_rule, ',', gw_str, sizeof(gw_str)) == -1)
			continue;

		if(getNthValueSafe(3, one_rule, ',', interface, sizeof(interface)) == -1)
			continue;

		if(getNthValueSafe(4, one_rule, ',', ifname, sizeof(ifname)) == -1)
			continue;

		if(getNthValueSafe(6, one_rule, ',', comment, sizeof(comment)) == -1)
			continue;

		if(strlen(result))
			strncat(result, ";", sizeof(result));

		snprintf(result, sizeof(result), "%s%s,%s,%s,%s,0,0,0,0,%d,%s,1,%s", result, ifname, dest_str, gw_str, netmask_str, index, interface, comment);
	}

	strcat(result, "\"");
	websLongWrite(wp, result);
	fclose(fp);
	if(running_rules)
		free(running_rules);
	//printf("%s\n", result);
	return 0;
}

static void addRouting(webs_t wp, char_t *path, char_t *query)
{
	char_t *dest, *hostnet, *netmask, *gateway, *interface, *true_interface, *custom_interface, *comment;
	char cmd[256] = {0};
	char result[256] = {0};

	FILE *fp;

	dest = websGetVar(wp, T("dest"), T(""));
	hostnet = websGetVar(wp, T("hostnet"), T(""));
	netmask = websGetVar(wp, T("netmask"), T(""));	
	gateway = websGetVar(wp, T("gateway"), T(""));
	interface = websGetVar(wp, T("interface"), T(""));
	custom_interface = websGetVar(wp, T("custom_interface"), T(""));
	comment = websGetVar(wp, T("comment"), T(""));

	if( !dest)
		return;

	strcat(cmd, "route add ");
	
	// host or net?
	if(!gstrcmp(hostnet, "net"))
		strcat(cmd, "-net ");
	else
		strcat(cmd, "-host ");

	// destination
	strcat(cmd, dest);
	strcat(cmd, " ");

	// netmask
	if(gstrlen(netmask))
		sprintf(cmd, "%s netmask %s", cmd, netmask);
	else
		netmask = "255.255.255.255";

	//gateway
	if(gstrlen(gateway))
		sprintf(cmd, "%s gw %s", cmd, gateway);
	else
		gateway = "0.0.0.0";

	//interface
	if(gstrlen(interface)){
		if (!gstrcmp(interface, "WAN")){
			true_interface = getWanIfName();
		}else if (!gstrcmp(interface, "Custom")){
			if(!gstrlen(custom_interface))
				return;
			true_interface = custom_interface;
		}else	// LAN & unknown
			true_interface = getLanIfName();
	}else{
		interface = "LAN";
		true_interface = getLanIfName();
	}
	sprintf(cmd, "%s dev %s ", cmd, true_interface);

	strcat(cmd, "2>&1 ");

	printf("%s\n", cmd);
	fp = popen(cmd, "r");
	fgets(result, sizeof(result), fp);
	pclose(fp);


	if(!strlen(result)){
		// success, write down to the flash
		char tmp[1024];
		char *rrs = nvram_bufget(RT2860_NVRAM, "RoutingRules");
		if(!rrs || !strlen(rrs)){
			memset(tmp, 0, sizeof(tmp));
		}else{
			strncpy(tmp, rrs, sizeof(tmp));
		}
		if(strlen(tmp))
			strcat(tmp, ";");
		sprintf(tmp, "%s%s,%s,%s,%s,%s,%s,%s", tmp, dest, netmask, gateway, interface, true_interface, custom_interface, comment);
		nvram_bufset(RT2860_NVRAM, "RoutingRules", tmp);
		nvram_commit(RT2860_NVRAM);
	}else{
		websHeader(wp);		
		websWrite(wp, T("<h1>Add routing failed:<br> %s<h1>"), result);
		websFooter(wp);
		websDone(wp, 200);
		return;
	}

	//debug print
	websHeader(wp);
	websWrite(wp, T("<h3>Add routing table:</h3><br>\n"));
	if(strlen(result)){
		websWrite(wp, T("Success"));
	}else
		websWrite(wp, T("%s"), result);

	websWrite(wp, T("Destination: %s<br>\n"), dest);
	websWrite(wp, T("Host/Net: %s<br>\n"), hostnet);
	websWrite(wp, T("Netmask: %s<br>\n"), netmask);
	websWrite(wp, T("Gateway: %s<br>\n"), gateway);
	websWrite(wp, T("Interface: %s<br>\n"), interface);
	websWrite(wp, T("True Interface: %s<br>\n"), true_interface);
	if(strlen(custom_interface))
		websWrite(wp, T("Custom_interface %s<br>\n"), custom_interface);
	websWrite(wp, T("Comment: %s<br>\n"), comment);
	websFooter(wp);
	websDone(wp, 200);
}

static void delRouting(webs_t wp, char_t *path, char_t *query)
{
	int index, rule_count;
	char_t *value, dest[256], netmask[256], true_interface[256];
	char name_buf[16] = {0};
	char *rrs;
	int *deleArray, j=0;
	
	rrs = nvram_bufget(RT2860_NVRAM, "RoutingRules");
	if(!rrs || !strlen(rrs))
		return;

	rule_count = getNums(rrs, ';');
	if(!rule_count)
		return;

	if(!(deleArray = malloc(sizeof(int) * rule_count) ) )
		return;
		
	websHeader(wp);

	for(index=0; index< rule_count; index++){
		snprintf(name_buf, sizeof(name_buf), "DR%d", index);
		value = websGetVar(wp, name_buf, NULL);
		if(value){
			deleArray[j++] = index;
			if(strlen(value) > 256)
				continue;
			sscanf(value, "%s%s%s", dest, netmask, true_interface);
			removeRoutingRule(dest, netmask, true_interface);
			websWrite(wp, T("Delete entry: %s,%s,%s<br>\n"), dest, netmask, true_interface);
		}
	}

	if(j>0){
		deleteNthValueMulti(deleArray, j, rrs, ';');
		nvram_bufset(RT2860_NVRAM, "RoutingRules", rrs);
		nvram_commit(RT2860_NVRAM);
	}

	websFooter(wp);
	websDone(wp, 200);

	free(deleArray);
}

void ripdRestart(void)
{
	char lan_ip[16], wan_ip[16], lan_mask[16], wan_mask[16];

	char *opmode = nvram_bufget(RT2860_NVRAM, "OperationMode");
	char *password = nvram_bufget(RT2860_NVRAM, "Password");
	char *RIPEnable = nvram_bufget(RT2860_NVRAM, "RIPEnable");

	doSystem("killall -q ripd");

	if(!opmode||!strlen(opmode))
		return;
	if(!strcmp(opmode, "0"))	// bridge
		return;

	if(!RIPEnable || !strlen(RIPEnable) || !strcmp(RIPEnable,"0"))
        return;

	if(!password || !strlen(password))
		password = "rt2880";

	doSystem("echo \"hostname linux.router1\" > /etc/ripd.conf ");
	doSystem("echo \"password %s\" >> /etc/ripd.conf ", password);
	doSystem("echo \"router rip\" >> /etc/ripd.conf ");

	// deal with WAN
	if(getIfIp(getWanIfName(), wan_ip) != -1){
		if(getIfNetmask(getWanIfName(), wan_mask) != -1){
			doSystem("echo \"network %s/%d\" >> /etc/ripd.conf", wan_ip, netmask_aton(wan_mask));
			doSystem("echo \"network %s\" >> /etc/ripd.conf", getWanIfName());
		}else
			printf("ripdRestart(): The WAN IP is still undeterminated...\n");
	}else
		printf("ripdRestart(): The WAN IP is still undeterminated...\n");

	// deal with LAN
	if(getIfIp(getLanIfName(), lan_ip) != -1){
		if(getIfNetmask(getLanIfName(), lan_mask) != -1){
			doSystem("echo \"network %s/%d\" >> /etc/ripd.conf", lan_ip, netmask_aton(lan_mask));
			doSystem("echo \"network %s\" >> /etc/ripd.conf", getLanIfName());
		}
	}
	doSystem("echo \"version 2\" >> /etc/ripd.conf");
	doSystem("echo \"log syslog\" >> /etc/ripd.conf");
	doSystem("ripd -f /etc/ripd.conf -d");
}

inline void zebraRestart(void)
{
	char *opmode = nvram_bufget(RT2860_NVRAM, "OperationMode");
	char *password = nvram_bufget(RT2860_NVRAM, "Password");

	char *RIPEnable = nvram_bufget(RT2860_NVRAM, "RIPEnable");

	doSystem("killall -q zebra");

	if(!opmode||!strlen(opmode))
		return;
	if(!strcmp(opmode, "0"))	// bridge
		return;

	if(!RIPEnable || !strlen(RIPEnable) || !strcmp(RIPEnable,"0"))
		return;

	if(!password || !strlen(password))
		password = "rt2880";

	doSystem("echo \"hostname linux.router1\" > /etc/zebra.conf ");
	doSystem("echo \"password %s\" >> /etc/zebra.conf ", password);
	doSystem("echo \"enable password rt2880\" >> /etc/zebra.conf ");
	doSystem("echo \"log syslog\" >> /etc/zebra.conf ");
	doSystem("zebra -d -f /etc/zebra.conf");
}

static void dynamicRouting(webs_t wp, char_t *path, char_t *query)
{
	char_t *rip;
	char *RIPEnable;
	rip = websGetVar(wp, T("RIPSelect"), T(""));
	if(!rip || !strlen(rip))
		return;

	RIPEnable = nvram_bufget(RT2860_NVRAM, "RIPEnable");
	if(!RIPEnable || !strlen(RIPEnable))
		RIPEnable = "0";

	if(!gstrcmp(rip, "0") && !strcmp(RIPEnable, "0")){
		// nothing changed
	}else if(!gstrcmp(rip, "1") && !strcmp(RIPEnable, "1")){
		// nothing changed
	}else if(!gstrcmp(rip, "0") && !strcmp(RIPEnable, "1")){
		nvram_bufset(RT2860_NVRAM, "RIPEnable", rip);
		nvram_commit(RT2860_NVRAM);
		doSystem("killall -q ripd");
		doSystem("killall -q zebra");
	}else if(!gstrcmp(rip, "1") && !strcmp(RIPEnable, "0")){
		nvram_bufset(RT2860_NVRAM, "RIPEnable", rip);
		nvram_commit(RT2860_NVRAM);
		zebraRestart();
		ripdRestart();
	}else{
		return;
	}

	//debug print
	websHeader(wp);
	websWrite(wp, T("<h3>Dynamic Routing:</h3><br>\n"));
	websWrite(wp, T("RIPEnable %s<br>\n"), rip);
	websFooter(wp);
	websDone(wp, 200);
}

//Tom.Hung 2010-1-27, GPIO check function, port from usr/sysd/sysd.c
int gpio_read_bit(int idx, int *value)
{
	int fd, req;

	*value = 0;
	fd = open(GPIO_DEV, O_RDONLY);
	if (fd < 0) {
		perror(GPIO_DEV);
		printf("can not open GPIO_DEV\n");
		return -1;
	}
	if (0L <= idx && idx < RALINK_GPIO_DATA_LEN)
		req = RALINK_GPIO_READ_BIT | (idx << RALINK_GPIO_DATA_LEN);
	else {
		close(fd);
		printf("gpio_read_bit: index %d out of range\n", idx);
		return -1;
	}
	if (ioctl(fd, req, value) < 0) {
		perror("ioctl");
		close(fd);
		return -1;
	}
	close(fd);
}

int GPIO_Check(int pin)
{
	int d;

	gpio_read_bit(pin, &d);
	return d;
}
//Tom.Hung 2010-1-27

/*
 * description: setup internet according to nvram configurations
 *              (assume that nvram_init has already been called)
 *              return value: 0 = successful, -1 = failed
 */
//int initInternet(void)
int initInternet(char *initTag)
{
	FILE *fp;
#ifndef CONFIG_RALINK_RT2880
	char *auth_mode = nvram_bufget(RT2860_NVRAM, "AuthMode");
#endif
#if defined CONFIG_RT2860V2_STA || defined CONFIG_RT2860V2_STA_MODULE
	char *opmode;
#endif

	//Add for MAC cloning - U-Media on Apr. 11 2008
	char if_mac[18];
	char *mode = nvram_bufget(RT2860_NVRAM, "OperationMode");
	if (!strncmp(mode, "1", 2)) {	
		if(getIfMac("eth2", if_mac)!=-1){
			nvram_bufset(RT2860_NVRAM, "OriginalWANMac", if_mac);
		}
	}
	//U-Media on Apr. 11 2008

	//Add for status page - U-Media Ricky Cao on Mar. 19 2008
	//char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");
	//Jacky.Yang 22-Apr-2008, fix nvram get buffer value.
	FILE *wanStatusFile;
	char *connType = nvram_bufget(RT2860_NVRAM, "wanConnectionMode");
	wanStatusFile = fopen("/var/wan_status", "w+");
	if(!wanStatusFile){
		printf("open \"/var/wanstatus\" faile\n");
	}else{	
		if(strcmp(connType, "STATIC")){
			fputs("0", wanStatusFile);
		}else{
			if(getWanPortPhyStatus()){
				fputs("1", wanStatusFile);
			}else{
				fputs("0", wanStatusFile);
			}
		}
		fclose(wanStatusFile);	
	}
	//U-Media Ricky Cao on Mar. 19 2008

	//Jacky.Yang 3-Mar-2009, check goahead is first one init or reload it.
	if ((!strcmp(initTag, "first") && (fp = fopen("/var/goahead_init", "r")) != NULL))
	{
printf("Salim-Not calling internet.sh!!!!!\n");
		fclose(fp);
	}
	else
	{
printf("Salim-calling internet.sh!!!!!\n");
		//Tom.Hung 2010-1-27, when first time, detect dip switch is on/off. internet.sh will turn off radio if dip switch is off.
		system("gpio r");
		printf("First time initInternet, detect dip switch status: ");
		if (GPIO_Check(DIP_SWITCH) == DIP_SWITCH_ON) {
				printf("On\n");
				system("nvram_set 2860 RadioOff 0");
		} else if (GPIO_Check(DIP_SWITCH) == DIP_SWITCH_OFF) {
				printf("Off\n");
				system("nvram_set 2860 RadioOff 1");
		}
		//Tom.Hung 2010-1-27
		doSystem("internet.sh");
	}
	//Jacky.Yang 26-May-2008, resart 802.1x
	restart8021XDaemon(RT2860_NVRAM);	// in wireless.c

	//automatically connect to AP according to the active profile
#if defined CONFIG_RT2860V2_STA || defined CONFIG_RT2860V2_STA_MODULE
	opmode = nvram_bufget(RT2860_NVRAM, "OperationMode");
	if (!strcmp(opmode, "2") || (!strcmp(opmode, "0") &&
				!strcmp("1", nvram_get(RT2860_NVRAM, "ethConver")))) {
		if (-1 != initStaProfile())
			initStaConnection();
	}
#endif

#ifndef CONFIG_RALINK_RT2880
	if (!strcmp(auth_mode, "Disable") || !strcmp(auth_mode, "OPEN"))
		ledAlways(13, LED_OFF); //turn off security LED (gpio 13)
	else
		ledAlways(13, LED_ON); //turn on security LED (gpio 13)
#endif

#if defined (CONFIG_RT2860V2_AP) || defined (CONFIG_RT2860V2_AP_MODULE)
	restart8021XDaemon(RT2860_NVRAM);	// in wireless.c
#endif
#if defined (CONFIG_INIC_MII) || defined (CONFIG_INIC_PCI) || defined (CONFIG_INIC_USB) 
	restart8021XDaemon(RTINIC_NVRAM);	// in wireless.c
#endif
#if defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE)
	restart8021XDaemon(RT2561_NVRAM);	// in wireless.c
#endif

#ifdef CONFIG_RT2860V2_AP_ANTENNA_DIVERSITY
	AntennaDiversityInit();
#endif

	//Jacky.Yang 8-May-2008, disable it, becasue when the wan side get ip or have change that need to run firewall_init()
	//firewall_init();
	management_init();
	RoutingInit();
#ifdef CONFIG_RALINKAPP_SWQOS
	QoSInit();
#endif

	return 0;
}

static void getMyMAC(webs_t wp, char_t *path, char_t *query)
{
	char myMAC[32];

	arplookup(wp->ipaddr, myMAC);
	websWrite(wp, T("HTTP/1.1 200 OK\nContent-type: text/plain\nPragma: no-cache\nCache-Control: no-cache\n\n"));
	websWrite(wp, T("%s"), myMAC);
	websDone(wp, 200);
}

/* goform/setLan */
static void setLan(webs_t wp, char_t *path, char_t *query)
{
	char_t	*ip, *nm, *dhcp_tp, *stp_en, *lltd_en, *igmp_en, *upnp_en,
			*radvd_en, *pppoer_en, *dnsp_en;
	char_t	*gw = NULL, *pd = NULL, *sd = NULL;
	char_t *lan2enabled, *lan2_ip, *lan2_nm;
	char *oldLanIP = nvram_bufget(RT2860_NVRAM, "lan_ipaddr");
#ifdef GA_HOSTNAME_SUPPORT
	char_t	*host;
#endif
	char_t  *dhcp_s, *dhcp_e, *dhcp_m, *dhcp_pd, *dhcp_sd, *dhcp_g, *dhcp_l;
	char_t	*dhcp_sl1, *dhcp_sl2, *dhcp_sl3;
	char	*opmode = nvram_bufget(RT2860_NVRAM, "OperationMode");
	char	*wan_ip = nvram_bufget(RT2860_NVRAM, "wan_ipaddr");
	char	*ctype = nvram_bufget(RT2860_NVRAM, "connectionType");

	ip = websGetVar(wp, T("lanIp"), T(""));
	nm = websGetVar(wp, T("lanNetmask"), T(""));
	lan2enabled = websGetVar(wp, T("lan2enabled"), T(""));
	lan2_ip = websGetVar(wp, T("lan2Ip"), T(""));
	lan2_nm = websGetVar(wp, T("lan2Netmask"), T(""));
#ifdef GA_HOSTNAME_SUPPORT
	host = websGetVar(wp, T("hostname"), T("0"));
#endif
	dhcp_tp = websGetVar(wp, T("lanDhcpType"), T("DISABLE"));
	stp_en = websGetVar(wp, T("stpEnbl"), T("0"));
	lltd_en = websGetVar(wp, T("lltdEnbl"), T("0"));
	igmp_en = websGetVar(wp, T("igmpEnbl"), T("0"));
	upnp_en = websGetVar(wp, T("upnpEnbl"), T("0"));
	radvd_en = websGetVar(wp, T("radvdEnbl"), T("0"));
	pppoer_en = websGetVar(wp, T("pppoeREnbl"), T("0"));
	dnsp_en = websGetVar(wp, T("dnspEnbl"), T("0"));
	dhcp_s = websGetVar(wp, T("dhcpStart"), T(""));
	dhcp_e = websGetVar(wp, T("dhcpEnd"), T(""));
	dhcp_m = websGetVar(wp, T("dhcpMask"), T(""));
	//dhcp_pd = websGetVar(wp, T("dhcpPriDns"), T(""));
	//dhcp_sd = websGetVar(wp, T("dhcpSecDns"), T(""));
	dhcp_g = websGetVar(wp, T("dhcpGateway"), T(""));
	dhcp_l = websGetVar(wp, T("dhcpLease"), T("86400"));
	dhcp_sl1 = websGetVar(wp, T("dhcpStatic1"), T(""));
	dhcp_sl2 = websGetVar(wp, T("dhcpStatic2"), T(""));
	dhcp_sl3 = websGetVar(wp, T("dhcpStatic3"), T(""));

	/*
	 * check static ip address:
	 * lan and wan ip should not be the same except in bridge mode
	 */
	if (strncmp(ctype, "STATIC", 7)) {
		if (strcmp(opmode, "0") && !strncmp(ip, wan_ip, 15)) {
			websError(wp, 200, "IP address is identical to WAN");
			return;
		}
		if (!strcmp(lan2enabled, "1"))
		{
			if (strcmp(opmode, "0") && !strncmp(lan2_ip, wan_ip, 15)) {
				websError(wp, 200, "LAN2 IP address is identical to WAN");
				return;
			}
			else if (strcmp(opmode, "0") && !strncmp(lan2_ip, ip, 15)) {
				websError(wp, 200, "LAN2 IP address is identical to LAN1");
				return;
			}
		}
	}
	// configure gateway and dns (WAN) at bridge mode
	if (!strncmp(opmode, "0", 2)) {
		gw = websGetVar(wp, T("lanGateway"), T(""));
		pd = websGetVar(wp, T("lanPriDns"), T(""));
		sd = websGetVar(wp, T("lanSecDns"), T(""));
		nvram_bufset(RT2860_NVRAM, "wan_gateway", gw);
		nvram_bufset(RT2860_NVRAM, "wan_primary_dns", pd);
		nvram_bufset(RT2860_NVRAM, "wan_secondary_dns", sd);
	}
	nvram_bufset(RT2860_NVRAM, "lan_ipaddr", ip);
	nvram_bufset(RT2860_NVRAM, "lan_netmask", nm);
	nvram_bufset(RT2860_NVRAM, "Lan2Enabled", lan2enabled);
	nvram_bufset(RT2860_NVRAM, "lan2_ipaddr", lan2_ip);
	nvram_bufset(RT2860_NVRAM, "lan2_netmask", lan2_nm);
#ifdef GA_HOSTNAME_SUPPORT
	nvram_bufset(RT2860_NVRAM, "HostName", host);
#endif
	if (!strncmp(dhcp_tp, "DISABLE", 8))
		nvram_bufset(RT2860_NVRAM, "dhcpEnabled", "0");
	else if (!strncmp(dhcp_tp, "SERVER", 7)) {
		if (-1 == inet_addr(dhcp_s)) {
			nvram_commit(RT2860_NVRAM);
			websError(wp, 200, "invalid DHCP Start IP");
			return;
		}
		nvram_bufset(RT2860_NVRAM, "dhcpStart", dhcp_s);
		if (-1 == inet_addr(dhcp_e)) {
			nvram_commit(RT2860_NVRAM);
			websError(wp, 200, "invalid DHCP End IP");
			return;
		}
		nvram_bufset(RT2860_NVRAM, "dhcpEnd", dhcp_e);
		if (-1 == inet_addr(dhcp_m)) {
			nvram_commit(RT2860_NVRAM);
			websError(wp, 200, "invalid DHCP Subnet Mask");
			return;
		}
		nvram_bufset(RT2860_NVRAM, "dhcpMask", dhcp_m);
		nvram_bufset(RT2860_NVRAM, "dhcpEnabled", "1");
		//nvram_bufset(RT2860_NVRAM, "dhcpPriDns", dhcp_pd);
		//nvram_bufset(RT2860_NVRAM, "dhcpSecDns", dhcp_sd);
		nvram_bufset(RT2860_NVRAM, "dhcpGateway", dhcp_g);
		nvram_bufset(RT2860_NVRAM, "dhcpLease", dhcp_l);
		nvram_bufset(RT2860_NVRAM, "dhcpStatic1", dhcp_sl1);
		nvram_bufset(RT2860_NVRAM, "dhcpStatic2", dhcp_sl2);
		nvram_bufset(RT2860_NVRAM, "dhcpStatic3", dhcp_sl3);
	}
	nvram_bufset(RT2860_NVRAM, "stpEnabled", stp_en);
	nvram_bufset(RT2860_NVRAM, "lltdEnabled", lltd_en);
	nvram_bufset(RT2860_NVRAM, "igmpEnabled", igmp_en);
	nvram_bufset(RT2860_NVRAM, "upnpEnabled", upnp_en);
	nvram_bufset(RT2860_NVRAM, "radvdEnabled", radvd_en);
	nvram_bufset(RT2860_NVRAM, "pppoeREnabled", pppoer_en);
	nvram_bufset(RT2860_NVRAM, "dnsPEnabled", dnsp_en);
	nvram_commit(RT2860_NVRAM);

/* Tom.Hung 2010-1-6, only save to flash, let user need reboot manually to let setting take effect.
	//Jacky.Yang 8-May-2008, disable it, becasue when the wan side get ip or have change that need to run firewall_init()
	printf("jacky - oldLanIP:%s, ip:%s\n", oldLanIP, ip);
	if (strcmp(oldLanIP, ip)) //Jacky.Yang 26-Aug-2008, If Lan ip address had changed, we need run internet.sh for renew NAT table(nat.sh).
	{
		//doSystem("internet.sh");
		if (fork() == 0) {
			//websRedirect(wp, "/wait_page.asp");
		} else
		{
			//Jacky.Yang 17-Sep-2008, You can use the following command to reset the PHY without reboot the system necessary. This command include link down the PHY about 2 seconds then link up.
			//doSystem("reset_all_phys.sh");
			//doSystem("sleep 5 ; reboot");
		}
	}
	else
	{
		doSystem("lan.sh");

		updateFlash8021x(RT2860_NVRAM);
		doSystem("ralink_init make_wireless_config rt2860");
		restart8021XDaemon(RT2860_NVRAM);

		firewall_init();
		management_init();
		//RoutingInit(); //Skip it, Because we design static route control by ourself - Ricky Cao
		iptablesRemoteManagementRun();
		
		//websRedirect(wp, "/internet/lan.asp");
	}
Tom.Hung 2010-1-6 */
	websRedirect(wp, "/applied2.asp?url=/internet/lan.asp");
}

/* goform/setVpnPaThru */
static void setVpnPaThru(webs_t wp, char_t *path, char_t *query)
{
	char_t	*l2tp_pt, *ipsec_pt, *pptp_pt;

	l2tp_pt = websGetVar(wp, T("l2tpPT"), T("0"));
	ipsec_pt = websGetVar(wp, T("ipsecPT"), T("0"));
	pptp_pt = websGetVar(wp, T("pptpPT"), T("0"));
	
	nvram_bufset(RT2860_NVRAM, "l2tpPassThru", l2tp_pt);
	nvram_bufset(RT2860_NVRAM, "ipsecPassThru", ipsec_pt);
	nvram_bufset(RT2860_NVRAM, "pptpPassThru", pptp_pt);
	nvram_commit(RT2860_NVRAM);

	doSystem("vpn-passthru.sh");

	//debug print
	websHeader(wp);
	websWrite(wp, T("<h3>VPN Pass Through</h3><br>\n"));
	websWrite(wp, T("l2tp: %s<br>\n"), l2tp_pt);
	websWrite(wp, T("ipsec: %s<br>\n"), ipsec_pt);
	websWrite(wp, T("pptp: %s<br>\n"), pptp_pt);
	websFooter(wp);
	websDone(wp, 200);
}

/* goform/setWan */
static void setWan(webs_t wp, char_t *path, char_t *query)
{
	char_t	*ctype;
	char_t	*ip, *nm, *gw, *pd, *sd, *manuallyDNS;
	char_t	*eth, *user, *pass;
	char_t	*clone_en, *clone_mac;
	char_t  *pptp_srv, *pptp_mode;
	char_t  *l2tp_srv, *l2tp_mode;
#ifdef CONFIG_USER_3G
	char_t	*usb3g_dev;
#endif
	char	*opmode = nvram_bufget(RT2860_NVRAM, "OperationMode");
	char	*lan_ip = nvram_bufget(RT2860_NVRAM, "lan_ipaddr");
	char	*lan2enabled = nvram_bufget(RT2860_NVRAM, "Lan2Enabled");
	//Add for status page - U-Media Ricky Cao on Apr. 25 2008
	FILE *wanStatusFile;
	char wanStatus[8] = {0};
	//Add for set MTU - U-Media Ricky Cao on May 16 2008
	char *useDefault_mtu, *wan_mtu;
	//U-Media Ricky Cao
	char_t	*whost;
	//Add for wan page set DHCP hostname - U-Media Tom Hung Sep 19 2008
	char_t *ThreeGCountry, *ThreeGDialNum, *threeg_apn, *threeg_pin, *threeg_enable_pin; //Add for 3G modem - Ricky Cao

	ctype = ip = nm = gw = pd = sd = eth = user = pass = 
		clone_en = clone_mac = pptp_srv = pptp_mode = l2tp_srv = l2tp_mode = ThreeGDialNum /*Add for 3G modem - Ricky Cao*/ =
		NULL;

	ctype = websGetVar(wp, T("connectionType"), T("0")); 
	manuallyDNS = websGetVar(wp, T("manuallyDNS"), T("0"));
	nvram_bufset(RT2860_NVRAM, "manuallyDNS", manuallyDNS);
	pd = websGetVar(wp, T("staticPriDns"), T(""));
	sd = websGetVar(wp, T("staticSecDns"), T(""));
	nvram_bufset(RT2860_NVRAM, "wan_primary_dns", pd);
	nvram_bufset(RT2860_NVRAM, "wan_secondary_dns", sd);
	if (!strncmp(ctype, "STATIC", 7) || !strcmp(opmode, "0")) {
		//always treat bridge mode having static wan connection
		ip = websGetVar(wp, T("staticIp"), T(""));
		nm = websGetVar(wp, T("staticNetmask"), T("0"));
		gw = websGetVar(wp, T("staticGateway"), T(""));
		

		nvram_bufset(RT2860_NVRAM, "wanConnectionMode", ctype);
		if (-1 == inet_addr(ip)) {
			nvram_commit(RT2860_NVRAM);
			websError(wp, 200, "invalid IP Address");
			return;
		}
		/*
		 * lan and wan ip should not be the same except in bridge mode
		 */
		if (NULL != opmode && strcmp(opmode, "0") && !strncmp(ip, lan_ip, 15)) {
			nvram_commit(RT2860_NVRAM);
			websError(wp, 200, "IP address is identical to LAN");
			return;
		}
		if (!strcmp(lan2enabled, "1"))
		{
			char	*lan2_ip = nvram_bufget(RT2860_NVRAM, "lan2_ipaddr");
			if (NULL != opmode && strcmp(opmode, "0") && !strncmp(ip, lan2_ip, 15)) {
				nvram_commit(RT2860_NVRAM);
				websError(wp, 200, "IP address is identical to LAN2");
				return;
			}
		}
		nvram_bufset(RT2860_NVRAM, "wan_ipaddr", ip);
		if (-1 == inet_addr(nm)) {
			nvram_commit(RT2860_NVRAM);
			websError(wp, 200, "invalid Subnet Mask");
			return;
		}
		nvram_bufset(RT2860_NVRAM, "wan_netmask", nm);
		/*
		 * in Bridge Mode, lan and wan are bridged together and associated with
		 * the same ip address
		 */
		if (NULL != opmode && !strcmp(opmode, "0")) {
			nvram_bufset(RT2860_NVRAM, "lan_ipaddr", ip);
			nvram_bufset(RT2860_NVRAM, "lan_netmask", nm);
		}
		nvram_bufset(RT2860_NVRAM, "wan_gateway", gw);
	}
	else if (!strncmp(ctype, "DHCP", 5)) {
		whost = websGetVar(wp, T("whostname"), T("0"));
		nvram_bufset(RT2860_NVRAM, "wHostName", whost);
		//Add for wan page set DHCP hostname - U-Media Tom Hung Sep 19 2008		
		nvram_bufset(RT2860_NVRAM, "wanConnectionMode", ctype);
	}
	else if (!strncmp(ctype, "PPPOE", 6)) {
		char_t *pppoe_opmode, *pppoe_optime;

		user = websGetVar(wp, T("pppoeUser"), T(""));
		pass = websGetVar(wp, T("pppoePass"), T(""));
		pppoe_opmode = websGetVar(wp, T("pppoeOPMode"), T(""));
		if (0 == strcmp(pppoe_opmode, "OnDemand"))
			pppoe_optime = websGetVar(wp, T("pppoeIdleTime"), T(""));
		else 
			pppoe_optime = websGetVar(wp, T("pppoeRedialPeriod"), T(""));
		nvram_bufset(RT2860_NVRAM, "wan_pppoe_user", user);
		nvram_bufset(RT2860_NVRAM, "wan_pppoe_pass", pass);
		nvram_bufset(RT2860_NVRAM, "wanConnectionMode", ctype);
		nvram_bufset(RT2860_NVRAM, "wan_pppoe_opmode", pppoe_opmode);
		nvram_bufset(RT2860_NVRAM, "wan_pppoe_optime", pppoe_optime);
	}
	else if (!strncmp(ctype, "L2TP", 5)) {
		char_t *l2tp_opmode, *l2tp_optime;

		l2tp_srv = websGetVar(wp, T("l2tpServer"), T(""));
		user = websGetVar(wp, T("l2tpUser"), T(""));
		pass = websGetVar(wp, T("l2tpPass"), T(""));
		l2tp_mode = websGetVar(wp, T("l2tpMode"), T("0"));
		ip = websGetVar(wp, T("l2tpIp"), T(""));
		nm = websGetVar(wp, T("l2tpNetmask"), T(""));
		gw = websGetVar(wp, T("l2tpGateway"), T(""));
		l2tp_opmode = websGetVar(wp, T("l2tpOPMode"), T(""));
		if (0 == strcmp(l2tp_opmode, "OnDemand"))
			l2tp_optime = websGetVar(wp, T("l2tpIdleTime"), T(""));
		else
			l2tp_optime = websGetVar(wp, T("l2tpRedialPeriod"), T(""));
		nvram_bufset(RT2860_NVRAM, "wanConnectionMode", ctype);
		nvram_bufset(RT2860_NVRAM, "wan_l2tp_server", l2tp_srv);
		nvram_bufset(RT2860_NVRAM, "wan_l2tp_user", user);
		nvram_bufset(RT2860_NVRAM, "wan_l2tp_pass", pass);
		nvram_bufset(RT2860_NVRAM, "wan_l2tp_mode", l2tp_mode);
		nvram_bufset(RT2860_NVRAM, "wan_l2tp_opmode", l2tp_opmode);
		nvram_bufset(RT2860_NVRAM, "wan_l2tp_optime", l2tp_optime);
		if (!strncmp(l2tp_mode, "0", 2)) {
			nvram_bufset(RT2860_NVRAM, "wan_l2tp_ip", ip);
			nvram_bufset(RT2860_NVRAM, "wan_l2tp_netmask", nm);
			nvram_bufset(RT2860_NVRAM, "wan_l2tp_gateway", gw);
		}
	}
	else if (!strncmp(ctype, "PPTP", 5)) {
		char_t *pptp_opmode, *pptp_optime;

		pptp_srv = websGetVar(wp, T("pptpServer"), T(""));
		user = websGetVar(wp, T("pptpUser"), T(""));
		pass = websGetVar(wp, T("pptpPass"), T(""));
		pptp_mode = websGetVar(wp, T("pptpMode"), T("0"));
		ip = websGetVar(wp, T("pptpIp"), T(""));
		nm = websGetVar(wp, T("pptpNetmask"), T(""));
		gw = websGetVar(wp, T("pptpGateway"), T(""));
		pptp_opmode = websGetVar(wp, T("pptpOPMode"), T(""));
		if (0 == strcmp(pptp_opmode, "OnDemand"))
			pptp_optime = websGetVar(wp, T("pptpIdleTime"), T(""));
		else
			pptp_optime = websGetVar(wp, T("pptpRedialPeriod"), T(""));

		nvram_bufset(RT2860_NVRAM, "wanConnectionMode", ctype);
		nvram_bufset(RT2860_NVRAM, "wan_pptp_server", pptp_srv);
		nvram_bufset(RT2860_NVRAM, "wan_pptp_user", user);
		nvram_bufset(RT2860_NVRAM, "wan_pptp_pass", pass);
		nvram_bufset(RT2860_NVRAM, "wan_pptp_mode", pptp_mode);
		nvram_bufset(RT2860_NVRAM, "wan_pptp_opmode", pptp_opmode);
		nvram_bufset(RT2860_NVRAM, "wan_pptp_optime", pptp_optime);
		if (!strncmp(pptp_mode, "0", 2)) {
			nvram_bufset(RT2860_NVRAM, "wan_pptp_ip", ip);
			nvram_bufset(RT2860_NVRAM, "wan_pptp_netmask", nm);
			nvram_bufset(RT2860_NVRAM, "wan_pptp_gateway", gw);
		}
	}
#ifdef CONFIG_USER_3G
	//Jacky.Yang 3-Jul-2009, Drop 3.3.0.0 original design.
	/*else if (!strncmp(ctype, "3G", 3)) {
		usb3g_dev = websGetVar(wp, T("Dev3G"), T(""));

		nvram_bufset(RT2860_NVRAM, "wan_3g_dev", usb3g_dev);
		nvram_bufset(RT2860_NVRAM, "wanConnectionMode", ctype);
	}*/
	//Add for 3G modem - Ricky Cao on Dec. 25 .2008
	else if (!strncmp(ctype, "3G", 3)) {
		char_t *threeg_opmode, *threeg_optime;

		system("killall -9 config-chat.sh");
		sleep(1);
		system("killall -9 pppd");
		sleep(5);

		ThreeGCountry = websGetVar(wp, T("ThreeGCountry"), T(""));
		ThreeGDialNum = websGetVar(wp, T("ThreeGDialNum"), T(""));

		user = websGetVar(wp, T("ThreeGUsername"), T(""));
		pass = websGetVar(wp, T("ThreeGPassword"), T(""));

		threeg_apn = websGetVar(wp, T("ThreeGApn"), T(""));
		threeg_enable_pin = websGetVar(wp, T("ThreeGEnablePin"), T(""));
		if (0 == strcmp(threeg_enable_pin, "1"))
			threeg_pin = websGetVar(wp, T("ThreeGSimPin"), T(""));
		
		threeg_opmode = websGetVar(wp, T("ThreeGOPMode"), T(""));
		if (0 == strcmp(threeg_opmode, "OnDemand"))
			threeg_optime = websGetVar(wp, T("ThreeGIdleTime"), T(""));
		else 
			threeg_optime = websGetVar(wp, T("ThreeGRedialPeriod"), T(""));
		
		nvram_bufset(RT2860_NVRAM, "wanConnectionMode", ctype);
		nvram_bufset(RT2860_NVRAM, "ThreeGCountry", ThreeGCountry);
		nvram_bufset(RT2860_NVRAM, "ThreeGDialNum", ThreeGDialNum);		
		nvram_bufset(RT2860_NVRAM, "ThreeGUsername", user);
		nvram_bufset(RT2860_NVRAM, "ThreeGPassword", pass);
		nvram_bufset(RT2860_NVRAM, "ThreeGApn", threeg_apn);
		nvram_bufset(RT2860_NVRAM, "ThreeGEnablePinNum", threeg_enable_pin);
		if (0 == strcmp(threeg_enable_pin, "1")){
			nvram_bufset(RT2860_NVRAM, "ThreeGPinNum", threeg_pin);
		}
		nvram_bufset(RT2860_NVRAM, "ThreeGOpmode", threeg_opmode);
		nvram_bufset(RT2860_NVRAM, "ThreeGOptime", threeg_optime);		
	}
#endif
	else {
		websHeader(wp);
		websWrite(wp, T("<h2>Unknown Connection Type: %s</h2><br>\n"), ctype);
		websFooter(wp);
		websDone(wp, 200);
		return;
	}


	useDefault_mtu = websGetVar(wp, T("useDefaultMTU"), T("1"));
	nvram_bufset(RT2860_NVRAM, "wan_use_default_mtu", useDefault_mtu);
	if(!strncmp(useDefault_mtu, "0", 2)){
		wan_mtu = websGetVar(wp, T("wanMTU_Setting"), T("1500"));
		nvram_bufset(RT2860_NVRAM, "wan_mtu", wan_mtu);
	}

//Tom.Hung 2009-5-13, 3G use default MTU
	if (!strncmp(ctype, "3G", 3))	{
		nvram_bufset(RT2860_NVRAM, "wan_use_default_mtu", "1");
		if(!strncmp(useDefault_mtu, "0", 2)){
			wan_mtu = websGetVar(wp, T("wanMTU_Setting"), T("1500"));
			nvram_bufset(RT2860_NVRAM, "wan_mtu", wan_mtu);
		}
	}
//Tom.Hung 2009-5-13

	// mac clone
	clone_en = websGetVar(wp, T("macCloneEnbl"), T("0"));
	clone_mac = websGetVar(wp, T("macCloneMac"), T(""));
	nvram_bufset(RT2860_NVRAM, "macCloneEnabled", clone_en);
	if (!strncmp(clone_en, "1", 2))
		nvram_bufset(RT2860_NVRAM, "macCloneMac", clone_mac);
	nvram_commit(RT2860_NVRAM);

	//initInternet(); //Skip by Ricky Cao on Oct. 27 2008
	doSystem("wan.sh");
	doSystem("lan.sh setWan");
	doSystem("nat.sh");	

	//Port from initInternet() - Ricky Cao on Oct. 27 2008
#if defined (CONFIG_RT2860V2_AP) || defined (CONFIG_RT2860V2_AP_MODULE)
	restart8021XDaemon(RT2860_NVRAM);	// in wireless.c
#endif
#if defined (CONFIG_RT2880v2_INIC_MII) || defined (CONFIG_RT2880v2_INIC_MII_MODULE) || \
	defined (CONFIG_RT2880v2_INIC_PCI) || defined (CONFIG_RT2880v2_INIC_PCI_MODULE)
	restart8021XDaemon(RTINIC_NVRAM);	// in wireless.c
#endif
#if defined (CONFIG_RT2561_AP) || defined (CONFIG_RT2561_AP_MODULE)
	restart8021XDaemon(RT2561_NVRAM);	// in wireless.c
#endif

#ifdef CONFIG_RT2860V2_AP_ANTENNA_DIVERSITY
	AntennaDiversityInit();
#endif
	//Ricky Cao on Oct. 27 2008
	
	management_init();	
	firewall_init();
	//Port from initInternet() - Ricky Cao on Oct. 27 2008
#ifdef CONFIG_RALINKAPP_SWQOS
	QoSInit();
#endif
	//Ricky Cao on Oct. 27 2008
	
	//Add for status page - U-Media Ricky Cao on Apr. 25 2008
	if(!strcmp(ctype, "STATIC")){
		if(getWanPortPhyStatus()){
			wanStatusFile = fopen("/var/wan_status", "w+");
			if(wanStatusFile){
				fgets(wanStatus, 8, wanStatusFile);
				if(atoi(wanStatus)==0){
					fputs("1", wanStatusFile);
				}
				fclose(wanStatusFile);
			}else{
				printf("Open \"/var/wan_status\" fail");
			}
		}
		//Jacky.Yang 8-May-2008, disable it, becasue when the wan side get ip or have change that need to run firewall_init()
		//firewall_init();		
	}
	//U-Media Ricky Cao on Apr. 25 2008
	
	iptablesRemoteManagementRun();
	//websRedirect(wp, "/internet/wan.asp");
	websRedirect(wp, "/applied.asp?url=/internet/wan.asp");
}

//Port from SDK 2.3.0.0 - Ricky Cao on Oct. 01 2008
//Add for display wan status on GUI
int getWanPortPhyStatus(void)
{
//Tim Wang, for RTL8366SR driver
//#if defined CONFIG_RAETH_ROUTER && defined CONFIG_USER_ETHTOOL
//printf(">>>>>Calling getWanPortPhyStatus() ..\n");

#if (defined CONFIG_RAETH_ROUTER || defined CONFIG_RT_3052_ESW || defined CONFIG_RTL8366SR) && defined CONFIG_USER_ETHTOOL
	int port, rc;
	FILE *fp;
	char buf[1024];
	int wanPhyPort;
#ifdef CONFIG_WAN_AT_P4
	wanPhyPort = 4;
#else //CONFIG_WAN_AT_P4 - this is configured in menuconfig
	wanPhyPort = 1;
#endif
	//return 1;
//printf(">>>>>1 ..\n");

#if 0
	//Tim Wang, test I2C
	int i;
	for (i=0;i<10;i++) {
		for(port=0; port<5; port++){
			char *pos;
			char link = '0';
			int speed = 100;
			char duplex = 'F';
			FILE *proc_file = fopen("/proc/rt2880/gmac", "w");
			if(!proc_file){
				printf("fopen \"/proc/rt2880/gmac\" fail!!\n");
				return 0;
			}
			fprintf(proc_file, "%d", port);
			fclose(proc_file);

			if((fp = popen("ethtool eth2", "r")) == NULL){
				return 0;
			}
			rc = fread(buf, 1, 1024, fp);
			pclose(fp);
			if(rc == -1){
				printf("fread fail on getWanPortPhyStatus()!!\n");
			}
		}
	}
#endif	

	for(port=0; port<5; port++){
		char *pos;
		char link = '0';
		int speed = 100;
		char duplex = 'F';
		if(port!=wanPhyPort){
			continue;
		}
		FILE *proc_file = fopen("/proc/rt2880/gmac", "w");
		if(!proc_file){
			printf("fopen \"/proc/rt2880/gmac\" fail!!\n");
			return 0;
		}
		fprintf(proc_file, "%d", port);
		fclose(proc_file);

		if((fp = popen("ethtool eth2", "r")) == NULL){
			return 0;
		}
		rc = fread(buf, 1, 1024, fp);
		pclose(fp);
		if(rc == -1){
			printf("fread fail on getWanPortPhyStatus()!!\n");
			return 0;
		}else{
			// get Link status
			if(pos = strstr(buf, "Link detected: ")){
				pos += strlen("Link detected: ");
				if(*pos == 'y')
					return 1;
			}
			
		}
	}
	return 0;
#else
	//printf(">>>>>2 ..\n");
	return 0;
#endif
}

//Add for display wan status on GUI - Ricky Cao on Mar. 17 2008
static int getGatewayWanNetworkStatus(int eid, webs_t wp, int argc, char_t **argv)
{
	char *mode = nvram_get(RT2860_NVRAM, "OperationMode");
	char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");
	char wanStatus[8] = {0};
	FILE *wanStatusFile;
	static char if_name[8] = {0};
	char if_addr[16] = {0};
	int isUp = 0; //0:disconnect, 1:connect, 2:connecting
	int wanPhyPort;
	int pid=0;
	//Below is ported from TEW-672GR_TEW-639GR - Ricky CAO
	//Tom.Hung 2009-11-19, add for GUI verify ppp is on or down
	char pppStatus[8] = {0};
	FILE *pppStatusFile;
	//Tom.Hung 2009-11-19
	//Above is ported from TEW-672GR_TEW-639GR - Ricky CAO

	wanStatusFile = fopen("/var/wan_status", "r");
	if(wanStatusFile){
		fgets(wanStatus, 8, wanStatusFile);
		fclose(wanStatusFile);

		if(!strcmp(connType, "3G") || getWanPortPhyStatus()){
			if(!strcmp(connType, "STATIC")||!strcmp(connType, "DHCP")){
				if(atol(wanStatus)==0){
					if(!strcmp(connType, "DHCP")){
						isUp = 3;
					}else{
						isUp = 0;
					}
				}else{
					isUp = 1;
				}
			}else{
				strcpy(if_name, "ppp0");
				if(getIfLive(if_name)==1){ //Can not found the ppp0 interface in /proc/net/dev
					isUp = 0;
					if(getPsPid("pppd")){
						isUp = 3;
					}
				}else{
					if(getIfIp("ppp0", if_addr) != -1){
						isUp = 1;
					}else{
						isUp = 3;
					}
				}
			}
		}	
	}else{
		printf("Open \"/var/wan_status\" fail\n");
	}
	
	//Below is ported from TEW-672GR_TEW-639GR - Ricky CAO
	//Tom.Hung 2009-11-19, add for GUI verify ppp is on or down
	if(!(!strcmp(connType, "STATIC")||!strcmp(connType, "DHCP"))) { //If not STATIC or DHCP
		pppStatusFile = fopen("/var/ppp_status", "r");
		if(pppStatusFile){
			fgets(pppStatus, 8, pppStatusFile);
			fclose(pppStatusFile);
			if(isUp==1) {
				if(atol(pppStatus)==0)
					isUp = 0;
				else
					isUp = 1;
			}
		}else{
			//printf("Open \"/var/ppp_status\" fail");
		}
	}
	//Tom.Hung 2009-11-19
	//Above is ported from TEW-672GR_TEW-639GR - Ricky CAO
	
	return websWrite(wp, T("%s"), (isUp==1)?"Connected":(isUp==0?"Disconnected":"Connecting"));
}

//Add for display wan status on GUI - Ricky Cao on Mar. 19 2008
static void doStatusButton(webs_t wp, char_t *path, char_t *query)
{
	int pid = 0;
	char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");
	char *btn = websGetVar(wp, T("selectedBtnSetting"), T(""));
	FILE *wanStatusFile;
	FILE *pppStatusFile;
	char *pppReconnectMode=NULL;
	char pppStatus[8] = {0};
	//doSystem("lan.sh");
	

	if(!strcmp(connType, "DHCP")){
		pid = getPid("/var/run/udhcpc.pid");
		//printf("Got pid of dhcp = %d\n", pid);
	}else if(!strcmp(connType, "PPPOE")){
		pid = getPid("/var/run/ppp0.pid");
		//printf("Got pid of ppp0 = %d\n", pid);
	}

	if(!strcmp(btn, "Btn1")){
		//printf("status button 1 be pressed !!\n");
		if(!strcmp(connType, "DHCP")){
			if(pid!=0){
				kill(pid, SIGUSR1);
			}
		}else if(!strcmp(connType, "PPPOE") || !strcmp(connType, "L2TP") || !strcmp(connType, "PPTP") || !strcmp(connType, "3G")){
			//Ricky CAO: Below is added for prevent from connect button be click after ppp link is UP.
			pppStatusFile = fopen("/var/ppp_status", "r");
			if(pppStatusFile!=NULL)
				fgets(pppStatus, 8, pppStatusFile);
			//Ricky CAO: Above is added for prevent from connect button be click after ppp link is UP.

			if(pppStatusFile==NULL || atol(pppStatus)==0){
				if(!getPsPid("pppd")){
					doSystem("wan.sh");
				}
				
				//Ricky CAO: Below is for trigger ppp interface going up when user click to connect button
				//If pppd is exist on background, then, send several packets to gateway for bring ppp interface up
				if(!strcmp(connType, "PPPOE")){
					pppReconnectMode = nvram_get(RT2860_NVRAM, "wan_pppoe_opmode");
				}else if(!strcmp(connType, "L2TP")){
					pppReconnectMode = nvram_get(RT2860_NVRAM, "wan_l2tp_opmode");
				}else if(!strcmp(connType, "PPTP")){
					pppReconnectMode = nvram_get(RT2860_NVRAM, "wan_pptp_opmode");
				}
				
				if(!strncmp(pppReconnectMode, "OnDemand", 8)){
					system("killall -9 pppTrigger.sh > /dev/null 2>&1");
					system("/sbin/pppTrigger.sh &");
				}
				//Ricky CAO: Above is for trigger ppp interface going up when user click to connect button
			}

			if(pppStatusFile!=NULL)
				fclose(pppStatusFile);
		}
		//wan up will be detected by udhcpc and set it to up in udhcpc.sh - U-Media Ricky Cao	
		wanStatusFile = fopen("/var/wan_status", "w+");
		if(wanStatusFile){
			fputs("0", wanStatusFile);
			fclose(wanStatusFile);
		}else{
			printf("Open \"/var/wan_status\" fail");		
		}
	}else if(!strcmp(btn, "Btn2")){
		//printf("status button 2 be pressed !!\n");
		if(!strcmp(connType, "DHCP")){ //Release
			if(pid!=0){
				kill(pid, SIGUSR2);
			}
		}else if(!strcmp(connType, "PPPOE") || !strcmp(connType, "L2TP") || !strcmp(connType, "PPTP") || !strcmp(connType, "3G")){ //Disconnect
			if(!strcmp(connType, "3G")){
				system("killall config-chat.sh");
			}
			/*Salim:Bug:Disconnect L2TP connection from UI. When user sends traffic, connection is established.
			Kill l2tpd*/
			if(!strcmp(connType, "L2TP")){
				system("killall -q l2tpd");
			}
			pid = getPsPid("pppd");
			if(pid != 0){
				kill(pid, SIGTERM);
			}
		}
		wanStatusFile = fopen("/var/wan_status", "w+");
		if(wanStatusFile){		
			fputs("1", wanStatusFile);
			fclose(wanStatusFile);
		}else{
			printf("Open \"/var/wan_status\" fail");		
		}
	}

	usleep(100000);
	websRedirect(wp, "/adm/status.asp");
}

//Add for display the static wan setting on GUI - U-Media Ricky Cao on Mar. 25 2008
static int getPPPStaticWanIPSetting(int eid, webs_t wp, int argc, char_t **argv)
{
	//char_t *settingName;
	//char *setting;
	char if_addr[16] = {0};
	char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");

	/*if (ejArgs(argc, argv, T("%s"), &settingName) != 1) {
		return websWrite(wp, T(""));
	}

	setting = nvram_get(RT2860_NVRAM, settingName);*/

	if (-1 == getIfIp(getWanIfName(), if_addr)) {
		//websError(wp, 500, T("getWanIp: calling getIfIp error\n"));
		return websWrite(wp, T(""));
	}	

	return websWrite(wp, T("%s"), if_addr);
}

//Add for display the static wan setting on GUI - U-Media Ricky Cao on Mar. 25 2008
static int getStaticWanSetting(int eid, webs_t wp, int argc, char_t **argv)
{
	char_t *settingName;
	char *setting;

	if (ejArgs(argc, argv, T("%s"), &settingName) != 1) {
		return websWrite(wp, T(""));
	}

	setting = nvram_get(RT2860_NVRAM, settingName);

	return websWrite(wp, T("%s"), setting);
}

//Ricky Cao on Oct. 01 2008

static void advancedNetwork(webs_t wp, char_t *path, char_t *query)
{
	char_t *icmp, *upnp_en, *upnp_orig;
	
	upnp_orig = nvram_bufget(RT2860_NVRAM, "upnpEnabled");
	upnp_en = websGetVar(wp, T("upnpEnbl"), T("0"));
	nvram_bufset(RT2860_NVRAM, "upnpEnabled", upnp_en);
	
	//Jacky.Yang 30-Dec-2008, suppoprt icmp enabled.
	icmp = websGetVar(wp, T("icmpEnabled"), T("0"));
	nvram_bufset(RT2860_NVRAM, "icmpEnabled", icmp);
	nvram_commit(RT2860_NVRAM);
	
	doSystem("lan.sh advancedNetwork");
	icmpEnabled();
	
	/*Salim-Reboot if UPnP settings changed*/
	if(strcmp(upnp_orig, upnp_en))
		websRedirect(wp, "/applied2.asp?url=/advanced/network.asp");
	else
	//websRedirect(wp, "/advanced/network.asp");
		websRedirect(wp, "/applied.asp?url=/advanced/network.asp");
}

//Tom.Hung 2010-1-27, getDipSwitchStatus() for websAsp use.
static int getDipSwitchStatus(int eid, webs_t wp, int argc, char_t **argv)
{
		system("gpio r > /dev/null");
		if (GPIO_Check(DIP_SWITCH) == DIP_SWITCH_ON) {
				return websWrite(wp, T("1"));
		} else if (GPIO_Check(DIP_SWITCH) == DIP_SWITCH_OFF) {
				return websWrite(wp, T("0"));
		}
		return 0;
}
//Tom.Hung 2010-1-27

//Ricky Cao: Below is added for auto update information on status page
int getDnsAddr(int req, char *dns)
{
	FILE *fp;
	char buf[80] = {0}, ns_str[11];
	int type, idx = 0;

	fp = fopen("/etc/resolv.conf", "r");
	if (NULL == fp)
		return -1;
	
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		sscanf(buf, "%s%s", ns_str, dns);		
		if (strstr(buf, "nameserver") > 0)
		{
			idx++;
			if (idx == req) {
				break;
			}
		}
	}
	
	fclose(fp);
	
	return 0;
}

int getWanGatewayAddr(char *addr)
{
	char   buff[256];
	int    nl = 0 ;
	struct in_addr dest;
	struct in_addr gw;
	int    flgs, ref, use, metric;
	unsigned long int d,g,m;
	int    find_default_flag = 0;

	FILE *fp = fopen("/proc/net/route", "r");
	//Add by Ricky Cao on Apr. 30 2008
	if(!fp){
		return -1;
	}
	while (fgets(buff, sizeof(buff), fp) != NULL) {
		if (nl) {
			int ifl = 0;
			while (buff[ifl]!=' ' && buff[ifl]!='\t' && buff[ifl]!='\0')
				ifl++;
			buff[ifl]=0;    /* interface */
			if (sscanf(buff+ifl+1, "%lx%lx%X%d%d%d%lx",
						&d, &g, &flgs, &ref, &use, &metric, &m)!=7) {
				fclose(fp);
				return -1;
			}

			if (flgs&RTF_UP) {
				dest.s_addr = d;
				gw.s_addr   = g;
				strcpy(addr, (gw.s_addr==0 ? "" : inet_ntoa(gw)));

				if (dest.s_addr == 0) {
					find_default_flag = 1;
					break;
				}
			}
		}
		nl++;
	}
	fclose(fp);

	return 0;
}

//Return value 1:Connected, 0:Disconnect, 3:Connecting
static int getWanNetworkStatus()
{
	char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");
	char wanStatus[8] = {0};
	FILE *wanStatusFile;
	static char if_name[8] = {0};
	char if_addr[16] = {0};
	int isUp = 0; //0:disconnect, 1:connect, 2:connecting

	char pppStatus[8] = {0};
	FILE *pppStatusFile;

	wanStatusFile = fopen("/var/wan_status", "r");
	if(wanStatusFile){
		fgets(wanStatus, 8, wanStatusFile);
		fclose(wanStatusFile);

		if(!strcmp(connType, "3G") || getWanPortPhyStatus()){
			if(!strcmp(connType, "STATIC")||!strcmp(connType, "DHCP")){
				if(atol(wanStatus)==0){
					if(!strcmp(connType, "DHCP")){
						isUp = 3;
					}else{
						isUp = 0;
					}
				}else{
					isUp = 1;
				}
			}else{
				strcpy(if_name, "ppp0");
				if(getIfLive(if_name)==1){ //Can not found the ppp0 interface in /proc/net/dev
					isUp = 0;
					if(getPsPid("pppd")){
						isUp = 3;
					}
				}else{
					if(getIfIp("ppp0", if_addr) != -1){
						isUp = 1;
					}else{
						isUp = 3;
					}
				}
			}
		}	
	}else{
		printf("Open \"/var/wan_status\" fail\n");
	}

	if(!(!strcmp(connType, "STATIC")||!strcmp(connType, "DHCP"))) { //If not STATIC or DHCP
		pppStatusFile = fopen("/var/ppp_status", "r");
		if(pppStatusFile){
			fgets(pppStatus, 8, pppStatusFile);
			fclose(pppStatusFile);
			if(isUp==1) {
				if(atol(pppStatus)==0)
					isUp = 0;
				else
					isUp = 1;
			}
		}else{
			printf("Open \"/var/ppp_status\" fail");
		}
	}
	
	return isUp;
}

void updateInfoForStatusPage( webs_t wp, char_t *path, char_t *query)
{
	int connectionStatus=0;
	char wanIP[16]={0};
	char wanNetmask[16]={0};
	char wanGateway[16]={0};
	char wanDNS1[16]={0};
	char wanDNS2[16]={0};
	char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");

	connectionStatus=getWanNetworkStatus();
		
	if(connectionStatus == 1){
		getIfIp(getWanIfName(), wanIP);
		getIfNetmask(getWanIfName(), wanNetmask);
		getWanGatewayAddr(wanGateway);
		getDnsAddr(1, wanDNS1);
		getDnsAddr(2, wanDNS2);
	}

	websWrite(wp, T("HTTP/1.1 200 OK\nContent-type: text/plain\nPragma: no-cache\nCache-Control: no-cache\n\n"));

	websWrite(wp, T("%d,%s,%s,%s,%s,%s\n"),
								connectionStatus,
								wanIP,
								wanNetmask,
								wanGateway,
								wanDNS1,
								wanDNS2);

    	websDone(wp, 200);
		
	return;
}
//Ricky Cao: Above is added for auto update information on status page

void formDefineInternet(void) {
	websAspDefine(T("getDhcpCliList"), getDhcpCliList);
	websAspDefine(T("getDns"), getDns);
	websAspDefine(T("getHostSupp"), getHostSupp);
	websAspDefine(T("getIfLiveWeb"), getIfLiveWeb);
	websAspDefine(T("getIfIsUpWeb"), getIfIsUpWeb);
	websAspDefine(T("getIgmpProxyBuilt"), getIgmpProxyBuilt);
	websAspDefine(T("getVPNBuilt"), getVPNBuilt);
	websAspDefine(T("getLanIp"), getLanIp);
	websAspDefine(T("getLanMac"), getLanMac);
	websAspDefine(T("getLanIfNameWeb"), getLanIfNameWeb);
	websAspDefine(T("getLanNetmask"), getLanNetmask);
	websAspDefine(T("getDnsmasqBuilt"), getDnsmasqBuilt);
	websAspDefine(T("getGWBuilt"), getGWBuilt);
	websAspDefine(T("getLltdBuilt"), getLltdBuilt);
	websAspDefine(T("getPppoeRelayBuilt"), getPppoeRelayBuilt);
	websAspDefine(T("getUpnpBuilt"), getUpnpBuilt);
	websAspDefine(T("getRadvdBuilt"), getRadvdBuilt);
	websAspDefine(T("getWanIp"), getWanIp);
	websAspDefine(T("getWanMac"), getWanMac);
	websAspDefine(T("getWanIfNameWeb"), getWanIfNameWeb);
	websAspDefine(T("getWanNetmask"), getWanNetmask);
	websAspDefine(T("getWanGateway"), getWanGateway);
	websAspDefine(T("getRoutingTable"), getRoutingTable);
	websAspDefine(T("getMeshBuilt"), getMeshBuilt);
	websAspDefine(T("getWDSBuilt"), getWDSBuilt);
	websAspDefine(T("getWSCBuilt"), getWSCBuilt);
	websAspDefine(T("getMBSSIDBuilt"), getMBSSIDBuilt);
	websAspDefine(T("getUSBBuilt"), getUSBBuilt);
	websAspDefine(T("getStorageBuilt"), getStorageBuilt);
	websAspDefine(T("getFtpBuilt"), getFtpBuilt);
	websAspDefine(T("getSmbBuilt"), getSmbBuilt);
	websAspDefine(T("getMediaBuilt"), getMediaBuilt);
	websAspDefine(T("getWebCamBuilt"), getWebCamBuilt);
	websAspDefine(T("getPrinterSrvBuilt"), getPrinterSrvBuilt);
	websAspDefine(T("getUSBiNICBuilt"), getUSBiNICBuilt);
	websFormDefine(T("setLan"), setLan);
	websFormDefine(T("setVpnPaThru"), setVpnPaThru);
	websFormDefine(T("setWan"), setWan);
	websFormDefine(T("getMyMAC"), getMyMAC);
	websFormDefine(T("addRouting"), addRouting);
	websFormDefine(T("delRouting"), delRouting);
	websFormDefine(T("dynamicRouting"), dynamicRouting);
	websAspDefine(T("getDynamicRoutingBuilt"), getDynamicRoutingBuilt);
	websAspDefine(T("getSWQoSBuilt"), getSWQoSBuilt);
	websAspDefine(T("getDATEBuilt"), getDATEBuilt);
	websAspDefine(T("getDDNSBuilt"), getDDNSBuilt);
	websAspDefine(T("getSysLogBuilt"), getSysLogBuilt);
	websAspDefine(T("getETHTOOLBuilt"), getETHTOOLBuilt);
	websAspDefine(T("getGatewayWanNetworkStatus"), getGatewayWanNetworkStatus); //Port from SDK 2.3 for display wan status on GUI - Ricky Cao on Oct. 02 2008
	websFormDefine(T("doStatusButton"), doStatusButton); //Port from SDK 2.3 for display wan status on GUI - Ricky Cao on Oct. 02 2008
	websAspDefine(T("getPPPStaticWanIPSetting"), getPPPStaticWanIPSetting);//Add for display the static wan setting on GUI - U-Media Ricky Cao on Mar. 25 2008
	websAspDefine(T("getStaticWanSetting"), getStaticWanSetting);//Add for display the static wan setting on GUI - U-Media Ricky Cao on Mar. 25 2008
	websAspDefine(T("showDHCPReserveList"), showDHCPReserveList);
	websFormDefine(T("DHCPReserveAdd"), DHCPReserveAdd);
	websFormDefine(T("DHCPReserveDelete"), DHCPReserveDelete);
	websFormDefine(T("advancedNetwork"), advancedNetwork);
//Tom.Hung 2010-1-27, getDipSwitchStatus() for websAsp use.
	websAspDefine(T("getDipSwitchStatus"), getDipSwitchStatus);
//Tom.Hung 2010-1-27
	websFormDefine(T("updateInfoForStatusPage"), updateInfoForStatusPage); //Add for dynamic update status page information - Ricky CAO on Jan 28 2010
}
