/* 
 * ip_conntrack_proto_esp.c - Version $Revision$
 *
 * Connection tracking helper module for IPSEC ESP
 *
 * Based on the existing UDP helper
 *
 * (C) 2002 by Harald Welte <laforge@gnumonks.org>
 *
 * 2008-9-15: Porting to kernel 2.6.21.x by Arthur Tang
 * 		<tangjingbiao@gmail.com>
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/seq_file.h>
#include <linux/netfilter.h>
#include <linux/skbuff.h>
#include <linux/netfilter_ipv4.h>
#include <linux/netfilter_ipv6.h>
#include <linux/in.h>
#include <linux/ip.h>
#include <net/netfilter/nf_conntrack.h>
#include <net/netfilter/nf_conntrack_l4proto.h>

/* same timeouts as UDP */
//#define ESP_TIMEOUT (10*HZ)
#define ESP_STREAM_TIMEOUT (20*HZ)

#if 0
#define DEBUGP printk
#else
#define DEBUGP(format, args...)
#endif

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Harald Welte <laforge@gnumonks.org>");
MODULE_DESCRIPTION("netfilter connection tracking protocol helper for ESP");

static int esp_pkt_to_tuple(const struct sk_buff *skb, unsigned int dataoff,
                           struct nf_conntrack_tuple *tuple)
{
	struct iphdr *iph;
	const void *datah;

	iph = skb->nh.iph;
	datah = (void *)iph + iph->ihl * 4;

        const u_int16_t *hspi = datah;
	const u_int16_t *lspi = datah + 2;
	__be16 spi;

	spi = *hspi ^ *lspi;

        tuple->src.u.esp.spi = spi; // Arthur modified
        tuple->dst.u.esp.spi = spi;
	DEBUGP("nf_conntrack_l4proto_esp: %s: spi=%x\n", __FUNCTION__, 
			ntohl(tuple->dst.u.esp.spi));

        return 1;
}

static int esp_invert_tuple(struct nf_conntrack_tuple *tuple,
                           const struct nf_conntrack_tuple *orig)
{
        tuple->src.u.esp.spi = orig->dst.u.esp.spi;
        tuple->dst.u.esp.spi = orig->src.u.esp.spi;

	DEBUGP("nf_conntrack_proto_esp: %s: spi=%x\n", __FUNCTION__, 
			ntohl(tuple->dst.u.esp.spi));
        return 1;
}

/* Print out the per-protocol part of the tuple. */
static int esp_print_tuple(struct seq_file *s,
                           const struct nf_conntrack_tuple *tuple)
{
       return seq_printf(s, "spi=%u ", ntohl(tuple->src.u.esp.spi));
}

/* Print out the private part of the conntrack. */
static int esp_print_conntrack(struct seq_file *s,
                               const struct nf_conn *conntrack)
{
       return 0;
}

/* Returns verdict for packet, and may modify conntracktype */
static int esp_packet(struct nf_conn *conntrack,
                      const struct sk_buff *skb, unsigned int dataoff,
                      enum ip_conntrack_info ctinfo,
		      int pf, unsigned int hooknum)
{
       /* If we've seen traffic both ways, this is some kind of ESP
          stream.  Extend timeout. */
       if (conntrack->status & IPS_SEEN_REPLY) {
               nf_ct_refresh_acct(conntrack, ctinfo, skb, ESP_STREAM_TIMEOUT);
               /* Also, more likely to be important, and not a probe */
               set_bit(IPS_ASSURED_BIT, &conntrack->status);
       } else
               nf_ct_refresh_acct(conntrack, ctinfo, skb, ESP_TIMEOUT);

       return NF_ACCEPT;
}

/* Called when a new connection for this protocol found. */
static int esp_new(struct nf_conn *conntrack,
                   const struct sk_buff *skb, unsigned int dataoff)
{
	DEBUGP("ip_conntrack_proto_esp: %s\n", __FUNCTION__);
        return 1;
}

static void esp_destroy(struct nf_conn *ct)
{
	DEBUGP("esp destroy entering\n");
	return;
}

struct nf_conntrack_l4proto nf_ct_p_esp =
{
	.l3proto	 = PF_INET,
	.l4proto	 = IPPROTO_ESP,
	.name		 = "esp",
	.pkt_to_tuple    = esp_pkt_to_tuple,
	.invert_tuple    = esp_invert_tuple,
	.print_tuple     = esp_print_tuple,
	.print_conntrack = esp_print_conntrack,
	.packet		 = esp_packet,
	.new		 = esp_new,
	.destroy	 = esp_destroy,
	.me		 = THIS_MODULE,
};

static int __init nf_ct_proto_esp_init(void)
{
       int ret;

       if ((ret = nf_conntrack_l4proto_register(&nf_ct_p_esp))) {
               printk(KERN_ERR "unable to register conntrack protocol "
                               "helper for ESP: %d\n", ret);
               return -EIO;
       }
       return 0;
}

static void __exit nf_ct_proto_esp_fini(void)
{
       nf_conntrack_l4proto_unregister(&nf_ct_p_esp);
}

module_init(nf_ct_proto_esp_init);
module_exit(nf_ct_proto_esp_fini);
