#include <stdlib.h>

#include "vlist.h"

vlistEntry* listEntryAlloc(void *v)
{
    vlistEntry *entry = NULL;

    entry = (vlistEntry *) malloc(sizeof(vlistEntry));
    if (entry) {
        entry->elem = v;
        entry->prev = NULL;
        entry->next = NULL;
    }
    return entry;
}

void vlistEntryDestroy(vlistEntry *entry, void (*vdestroy)(void *))
{
    if (entry) {
        if (vdestroy && NULL != entry->elem)
            vdestroy(entry->elem);
        free(entry);
    }
}

void vlistInit(vlist *vl)
{
    vl->first = NULL;
    vl->last = NULL;
}

vlistEntry *vlistFirstEntry(vlist *vl)
{
    return vl->first;
}

vlistEntry *vlistLastEntry(vlist *vl)
{
    return vl->last;
}

vlistEntry *vlistPrevEntry(vlistEntry *ent)
{
    return ent->prev;
}

vlistEntry *vlistNextEntry(vlistEntry *ent)
{
    return ent->next;
}

vlistEntry *vlistInsertBefore(vlist *vl, void *v, vlistEntry *ent)
{
    vlistEntry *newent = NULL;

    newent = listEntryAlloc(v);
    if (newent) {
        if (ent) {
            vlistEntry *pvent = ent->prev;

            if (pvent)
                pvent->next = newent;
            else {
                /* ent is first entry */
                vl->first = newent;
            }

            newent->prev = pvent;
            newent->next = ent;
            ent->prev = newent;
        }else {
            /* should be a empty list */

            vl->first = vl->last = newent;
        }
    }
    return newent;
}

vlistEntry *vlistInsertAfter(vlist *vl, void *v, vlistEntry *ent)
{
    vlistEntry *newent = NULL;

    newent = listEntryAlloc(v);
    if (newent) {
        if (ent) {
            vlistEntry *nxent = ent->next;

            if (nxent)
                nxent->prev = newent;
            else {
                /* ent is last entry */
                vl->last = newent;
            }

            newent->prev = ent;
            newent->next = nxent;
            ent->next = newent;
        }else {
            /* should be a empty list */
            vl->first = vl->last = newent;
        }
    }
    return newent;
}

vlistEntry *vlistInsertFirst(vlist *vl, void *v)
{
    return vlistInsertBefore(vl, v, vlistFirstEntry(vl));
}

vlistEntry *vlistInsertLast(vlist *vl, void *v)
{
    return vlistInsertAfter(vl, v, vlistLastEntry(vl));
}

void vlistRemoveEntry(vlist *vl, vlistEntry *ent, void (*vdestroy)(void *))
{
    if (ent) {
        vlistEntry *pvent = ent->prev;
        vlistEntry *nxent = ent->next;

        if (pvent)
            pvent->next = nxent;
        else {
            /* is first entry */
            vl->first = nxent;
        }

        if (nxent)
            nxent->prev = pvent;
        else {
            /* is last entry */
            vl->last = pvent;
        }
        vlistEntryDestroy(ent, vdestroy);
    }
}

void vlistRemoveFirst(vlist *vl, void (*vdestroy)(void *))
{
    return vlistRemoveEntry(vl, vlistFirstEntry(vl), vdestroy);
}

void vlistRemoveLast(vlist *vl, void (*vdestroy)(void *))
{
    return vlistRemoveEntry(vl, vlistLastEntry(vl), vdestroy);
}

void vlistDestroy(vlist *vl, void (*vdestroy)(void *))
{
    if (vl->first) {
        vlistEntry *iter = NULL, *inxt = NULL;

        for (iter = vl->first; iter; iter = inxt) {
            inxt = iter->next;
            vlistEntryDestroy(iter, vdestroy);
        }

        vl->first = vl->last = NULL;
    }
}

int vlistEmpty(vlist *vl)
{
    return (NULL == vl->first) ? 1 : 0;
}

unsigned int vlistCount(vlist *vl)
{
    unsigned int count = 0;
    vlistEntry *iter = NULL;

    for (iter = vl->first; iter; iter = iter->next)
        count++;

    return count;
}

void vlistRemove(vlist *vl, void *item, void (*vdestroy)(void*), int (*compare)(const void*, const void*))
{
    vlistEntry *iter = NULL;
    void *p = 0;

    for (iter = vl->first; iter; iter = iter->next) {
        p = iter->elem;
        if (compare(p, item) == 0)
            break;
    }
    if (iter)
        vlistRemoveEntry(vl, iter, vdestroy);
}

int vlistExchange(vlist *vl, int pos1, int pos2)
{
	void **p1 = NULL, **p2 = NULL, *pt = NULL;
	int i = 1;
	vlistEntry *iter = NULL;

	if (pos1 < 1 || pos2 < 1 || pos1 == pos2)
		return -1;
	if (pos1 > pos2) {
		int tmp = pos1;
		pos1 = pos2;
		pos2 = tmp;
	}

	i = 0;
	iter = vl->first;
	for (; iter; iter = iter->next) {
		++i;
		if (i == pos1) {
			p1 = &(iter->elem);
		}else if (i == pos2) {
			p2 = &(iter->elem);
			break;
		}
	}
	if (i != pos2)
		return -1;

	pt = *p1;
	*p1 = *p2;
	*p2 = pt;
	return 0;
}

int vlistMove(vlist *vl, int from, int to)
{
	vlistEntry *pfrom = NULL, *pto = NULL, *iter = NULL;
	vlistEntry *pp = NULL, *pn = NULL;
	int i = 0;

	/* invalid arguments */
	if (!vl || from < 1 || to < 1)
		return -1;
	/* don't move if ...*/
	if (from == to || from + 1 == to)
		return 0;

	/* count of entry should bigger than 2 */
	for (iter = vl->first; iter && i < 2; iter = iter->next)
		i++;
	if (i < 2)
		return -1;

	/* search insert point */
	i = 0;
	for (iter = vl->first; iter && (! pfrom || ! pto); iter = iter->next) {
		++i;
		if (i == from)
			pfrom = iter;
		else if (i == to)
			pto = iter;
	}
	if (! pfrom)
		return -1;
	if (! pto) {
		if (to != i + 1)
			return -1;
	}

	/* break list */
	pp = pfrom->prev;
	pn = pfrom->next;
	if (pp)
		pp->next = pn;
	else
		vl->first = pn;
	if (pn)
		pn->prev = pp;
	else
		vl->last = pp;

	/* insert list */
	if (! pto) {
		/* insert to last */
		vlistEntry *lastent = vl->last;
		lastent->next = pfrom;
		pfrom->prev = lastent;
		pfrom->next = NULL;
	}else {
		/* insert before specifed point */
		pp = pto->prev;
		pfrom->prev = pp;
		pfrom->next = pto;
		pto->prev = pfrom;
		if (pp)
			pp->next = pfrom;
		else
			vl->first = pfrom;
	}
	return 0;
}
