#include <sys/socket.h>

#include <limits.h>

#include <unistd.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

static const unsigned char _maskbit[] = {
    (unsigned char) 0x00,       /* λδ */
    (unsigned char) 0x7F,
    (unsigned char) 0x3F,
    (unsigned char) 0x1F,
    (unsigned char) 0x0F,
    (unsigned char) 0x07,
    (unsigned char) 0x03,
    (unsigned char) 0x01
};


static in_addr_t inaddrt_maskit(in_addr_t in, int mask)
{
    int i,j,k;
    in_addr_t out = in;
    unsigned char *p = (unsigned char*) &out;
        
    i = mask / 8;
    j = mask % 8;

    for (k = 0; k < i; k++, p++)
        *p = (unsigned char)0x00;
    if (j)
        *p &= _maskbit[j];
    return out;
}

int test_ipmask(const char *ipmask)
{
    const in_addr_t WILL_BE = (in_addr_t) 0;
    char *ip, *mask;
    long nm;
    char *endp = (char*) 0;
    in_addr_t addr, maddr;

    ip = strdup(ipmask);
    if (! ip)
        return 0;
            
    if (! (mask = strchr(ip, '/'))) {
        free(ip);
        return 0;
    }
    *mask = '\0'; mask++;
    
    if (inet_pton(AF_INET, ip, &addr) != 1) {
        free(ip);
        return 0;
    }
        
    nm = strtol(mask, &endp, 10);
    if (nm == LONG_MAX || nm == LONG_MIN ||
        ! endp || *endp != '\0' ||
        nm < 0 || nm > 32) {
        free(ip);
        return 0;
    }
        
    maddr = inaddrt_maskit(addr, (int) nm);

    if (maddr != WILL_BE) {
        free(ip);
        return 0;
    }
    free(ip);
    return 1;
}

int test_ip(const char *ip)
{
    in_addr_t in;
    if (inet_pton(AF_INET, ip, &in) != 1)
        return 0;
    return 1;
}

int test_integer(const char *str, int low, int high)
{
    long num = 0;
    char *endp = (char*) 0;

    if (! str)
        return 0;
    num = strtol(str, &endp, 10);
    if (num == LONG_MAX || num == LONG_MIN)
        return 0;
    if (! endp || *endp != '\0')
        return 0;
    if (num < low || num > high)
        return 0;
    return 1;
}

int test_mask(const char *mask)
{
    return test_integer(mask, 0, 32);
}

int test_port(const char *port)
{
    return test_integer(port, 0, 65535);
}

int test_mtu(const char *mtu)
{
    return test_integer(mtu, 68, 1500);
}

int test_ipm(const char *ipmask)
{
    int ret = 0;
    char *dup = strdup(ipmask);

    if (dup) {
        char *mask = strchr(dup, '/');

        if (mask) {
            *mask = '\0';
            mask++;
        }
        if (test_ip(dup) && test_mask(mask))
            ret = 1;
        free(dup);
    }
    return ret;
}

int test_iprange(const char *range, uint32_t exceed)
{
    char *low, *high;
    in_addr_t inlow, inhigh;
    uint32_t u1, u2;
    
    low = high = (char*) 0;

#define RETURN_FALSE()                          \
    do {                                        \
        if (low)                                \
            free(low);                          \
        return 0;                               \
    }while (0)
    
    low = strdup(range);
    if (! low)
        RETURN_FALSE();
    high = strchr(low, '-');
    if (! high)
        RETURN_FALSE();
    *high = '\0'; high++;

    if (inet_pton(AF_INET, low, &inlow) != 1 ||
        inet_pton(AF_INET, high, &inhigh) != 1)
        RETURN_FALSE();

    u1 = ntohl(inlow);
    u2 = ntohl(inhigh);
    if (u2 <= u1 || u2 - u1 > exceed)
        RETURN_FALSE();

    free(low);
    return 1;
}

int test_portrange(const char *portrange)
{
    int ret = 0;
    
    char *low = strdup(portrange);
    

    if (low) {
        char *high = (char*) 0;
        high = strchr(low, ':');
	if (!high)
		high = strchr(low, '-');
        if (high) {
            *high = '\0'; high++;
            if (test_port(low) && test_port(high) &&
                atoi(low) < atoi(high))
                ret = 1;
        }else {
            ret = test_port(low);
        }
        free(low);
    }
    return ret;
}

int test_wday(const char *wday)
{
    int ret = 0;
    char *start, *stop;

    if (! (start = strdup(wday)))
        return 0;
    stop = strchr(start, '-');
    if (stop) {
        *stop = '\0';
        stop++;
    }

    if (test_integer(start, 1, 7)) {
        ret = 1;
        if (stop) {
            if (! test_integer(stop, 1, 7))
                ret = 0;
        }
    }
    free(start);
    return ret;
}

int test_wdayrange(const char *wdayrange)
{
    int ret = 0;
    
    char *fir = strdup(wdayrange);
    
    if (fir) {
        char *sec = (char*) 0;
        sec = strchr(fir, '-');
        if (sec) {
            *sec = '\0'; sec++;
            if (test_wday(fir) && test_wday(sec))
                ret = 1;
        }
        free(fir);
    }
    return ret;
}

int test_time(const char *stime)
{
    int ret = 0;
    char *fir = (char*) 0;

    fir = strdup(stime);
    if (fir) {
        char *sec = strchr(fir, ':');
        if (sec) {
            *sec = '\0'; sec++;
            if (test_integer(fir, 0, 23) &&
                test_integer(sec, 0, 59))
                ret = 1;
        }
        free(fir);
    }
    return ret;
}

int test_timerange(const char *timerange)
{
    int ret = 0;
    char *fir = (char*) 0;

    if ((fir = strdup(timerange))) {
        char *sec = strchr(fir, '-');

        if (sec) {
            *sec = '\0'; sec++;
            if (test_time(fir) && test_time(sec))
                ret = 1;
        }
        free(fir);
    }
    
    return ret;
}

static int ismac(int c)
{
    return ((c >= '0' && c <= '9') ||
             (c >= 'A' && c <= 'F') ||
            (c >= 'a' && c <= 'f')) ? 1 : 0;
}

int test_mac(const char *mac)
{
    const char *p = (const char*) mac;
    int i = 0;

    for (i = 0; i < 6; i++) {
        if (! *p || ! ismac(*p))
            return 0;
        p++;
        if (! *p || ! ismac(*p))
            return 0;
        p++;
        if (i == 5) {
            if (*p)
                return 0;
        }else {
            if (*p != ':')
                return 0;
            p++;
        }
    }
    return 1;
}

int test_icmptype(const char *icmptype)
{
    if (strcmp("any", icmptype) &&
        strcmp("ping", icmptype) &&
        strcmp("pong", icmptype) &&
        strcmp("source-quench", icmptype) &&
        strcmp("destination-unreachable", icmptype) &&
        strcmp("redirect", icmptype) &&
        strcmp("router-advertisement", icmptype) &&
        strcmp("router-solicitation", icmptype) &&
        strcmp("ttl-exceeded", icmptype) &&
        strcmp("parameter-problem", icmptype) &&
        strcmp("timestamp-request", icmptype) &&
        strcmp("timestamp-reply", icmptype) &&
        strcmp("address-mask-request", icmptype) &&
        strcmp("address-mask-reply", icmptype))
        return 0;
    return 1;
}

/* tcpflags = "!SYN,ACK,FIN/SYN" */
int test_tcpflags(const char *tcpflags)
{
    char *brkp;
    char *p;
    char *deli;
    char *flag;
    char *dup = strdup(tcpflags);

    if (! dup)
        return 0;
    
    if (! (deli = strchr(dup, '/'))) {
        free(dup);
        return 0;
    }
    *deli = '\0'; deli++;
    p = dup;
    if (*p == '!')
        p++;
    if (! *p || ! *deli) {
        free(dup);
        return 0;
    }
    
    flag = strtok_r(p, ",", &brkp);
    for (; flag;) {
        if (strcmp("SYN", flag) && strcmp("ACK", flag) &&
            strcmp("FIN", flag) && strcmp("RST", flag) &&
            strcmp("URG", flag) && strcmp("PSH", flag) &&
            strcmp("ALL", flag) && strcmp("NONE", flag)) {
            free(dup);
            return 0;
        }
        flag = strtok_r(NULL, ",", &brkp);
    }
    flag = strtok_r(deli, ",", &brkp);
    for (; flag;) {
        if (strcmp("SYN", flag) && strcmp("ACK", flag) &&
            strcmp("FIN", flag) && strcmp("RST", flag) &&
            strcmp("URG", flag) && strcmp("PSH", flag) &&
            strcmp("ALL", flag) && strcmp("NONE", flag)) {
            free(dup);
            return 0;
        }
        flag = strtok_r(NULL, ",", &brkp);
    }
    free(dup);
    return 1;
}

int test_state(const char *state)
{
    char *dup = NULL;
    char *brkp = NULL;
    char *p = NULL;
    
    if (! (dup = strdup(state)))
        return 0;
    p = strtok_r(dup, ",", &brkp);
    for (; p;) {
        if (strcmp("INVALID", p) && strcmp("ESTABLISHED", p) &&
            strcmp("NEW", p) && strcmp("RELATED", p)) {
            free(dup);
            return 0;
        }
        p = strtok_r(NULL, ",", &brkp);
    }
    free(dup);
    return 1;
}

int test_timezone(const char *timezone)
{
    if (! timezone)
        return 0;
    if (strcmp("Chongqing", timezone) && strcmp("Shanghai", timezone) &&
        strcmp("Singapore", timezone) && strcmp("Taipei", timezone) &&
        strcmp("HongKong", timezone))
        return 0;
    return 1;
}

int test_date(const char *spec)
{
    if (! spec || ! *spec)
        return 0;
    
    for (; *spec; spec++) {
        if (! ((*spec >= '0' && *spec <= '9') || *spec == '.'))
            return 0;
    }
    return 1;
}

int test_user(const char *user)
{
	if (! user || ! *user || strlen(user) < 3)
		return -1;
	
	for (; *user; ++user) {
		if (! isalnum(*user))
			return 0;
	}
	return 1;
}

int test_password(const char *pass)
{
	return test_user(pass);
}
