#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <pwd.h>

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sysexits.h>

#include <stdarg.h>

#include <sys/mman.h>

#include "peer.h"
#include "ifns.h"
#include "defs.h"

#define LOCAL_SOCK "/etc/rfwadm.sock"

int lis = -1;
int maxsock = -1;
int signalpipe[2] = {-1, -1};
fd_set allset;

static int listenUnixSocket(const char *sockpath)
{
	int fd = -1;

	if (sockpath) {
		struct sockaddr_un sunServer;

		fd = socket(AF_LOCAL, SOCK_STREAM, 0);
		if (fd < 0)
			return -1;
		unlink(sockpath);
		sunServer.sun_family = AF_LOCAL;
		strncpy(sunServer.sun_path, sockpath, sizeof(sunServer.sun_path) - 1);
		if (bind(fd, (struct sockaddr *) &sunServer, sizeof(struct sockaddr_un)) < 0) {
			close(fd);
			return -1;
		}
		if (listen(fd, 5) < 0) {
			close(fd);
			return -1;
		}
	}
	if (chmod(LOCAL_SOCK, 0777) < 0) {
		close(fd);
		return -1;
	}
	return fd;
}

void signalFlag(int c)
{
	int ch = c;
	if (write(signalpipe[1], &ch, 1) != 1) {
		_exit(EX_OSERR);
	}
}

void handleSIGCHLD(int unused)
{
	signalFlag('C');
}

void signalInit()
{
	struct sigaction sa;

	/* zombie process cleaner */
	sa.sa_handler = handleSIGCHLD;
	sa.sa_flags = 0;
	if (sigaction(SIGCHLD, &sa, (struct sigaction *)0) < 0) {
		exit(EX_OSERR);
	}

	/* ignore SIGPIPE */
	sa.sa_handler = SIG_IGN;
	sa.sa_flags = 0;
	if (sigaction(SIGPIPE, &sa, (struct sigaction *)0) < 0) {
		exit(EX_OSERR);
	}
}

void pipeInit()
{
	if (pipe(signalpipe) != 0) {
		exit(EX_OSERR);
	}

	FD_SET(signalpipe[0], &allset);
}

void childwait()
{
	pid_t pid = 1;
	int status;

	for (; pid > 0;)
		pid = waitpid(-1, &status, WNOHANG);
}

void daemonInit()
{
	pid_t pid;

	if ((pid = fork()) < 0) {
		_exit(EX_OSERR);
	}

	if (pid != 0)
		_exit(EX_OK);    /* parent byebye. */

	if (setsid() < 0) {
		_exit(EX_OSERR);
	}
}

void config()
{
	maxsock = (signalpipe[0] > signalpipe[1]) ?
		signalpipe[0] : signalpipe[1];

	if (lis > 0) {
		if (FD_ISSET(lis, &allset))
			FD_CLR(lis, &allset);
		close(lis);
		lis = -1;
	}

	if ((lis = listenUnixSocket(LOCAL_SOCK)) < 0) {
		exit(EX_CONFIG);
	}

	if (lis > maxsock)
		maxsock = lis;

	FD_SET(lis, &allset);
}

static int closeSTD()
{
	int nfd = -1;

	if ((nfd = open("/dev/null", O_WRONLY)) < 0)
		return -1;
	if (dup2(nfd, STDOUT_FILENO) < 0 || dup2(nfd, STDERR_FILENO) < 0) {
		close(nfd);
		return -1;
	}
	close(nfd);
	return 0;
}

int main(int argc, char **argv)
{
	signalInit();

#ifndef DEBUG
	daemonInit();
#endif

	pipeInit();

	config();

	ifnsInit();

	umask(022);

	for (;;) {
		int n = -1;
		fd_set readable = allset;

		n = select(maxsock + 1, &readable, (fd_set *)0, (fd_set *)0, (struct timeval *)0);

		if (n <= 0) {
			if (n < 0 && EINTR != errno)
				break;
			continue;
		}
		if (FD_ISSET(signalpipe[0], &readable)) {
			int nsig;

			if (ioctl(signalpipe[0], FIONREAD, &nsig) != 0) {
				exit(EX_OSERR);
			}
			while (--nsig >= 0) {
				char c;
				if (read(signalpipe[0], &c, 1) != 1) {
					exit(EX_OSERR);
				}
				switch(c) {
					case 'C':
						childwait();
						break;
				} /* switch */
			}
		}
		if (FD_ISSET(lis, &readable)) {
			int ctrl = -1;
			pid_t pid = 0;

			ctrl = accept(lis, (struct sockaddr *)0, (socklen_t *)0);
			if (ctrl < 0) {
				continue;
			}

#ifndef DEBUG
			if ((pid = fork()) < 0) {
				close(ctrl);
				continue;
			}
#endif

			if (pid == 0) {
				/* this child process */
				struct sigaction sahup;

				closeSTD();     /* رձ׼׼ */

				sahup.sa_handler = SIG_DFL;
				sahup.sa_flags = 0;
				if (sigaction(SIGALRM, &sahup, (struct sigaction *)0) < 0) {
					exit(EX_OSERR);
				}

				sahup.sa_handler = SIG_IGN;
				sahup.sa_flags = 0;
				if (sigaction(SIGHUP, &sahup, (struct sigaction *)0) < 0) {
					exit(EX_OSERR);
				}

				sahup.sa_handler = SIG_DFL;
				sahup.sa_flags = 0;
				if (sigaction(SIGCHLD, &sahup, (struct sigaction *)0) < 0) {
					exit(EX_OSERR);
				}
				close(lis);
				close(signalpipe[0]);
				close(signalpipe[1]);

				peerDispose(ctrl);
				close(ctrl);

				exit(EX_OK);
			}
			close(ctrl);
		}
	} /* for (;;) */

	/* avoid compile warning message */
	return 0;
}
