#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/param.h>

#include <unistd.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>

#include <stdarg.h>

#include "defs.h"
#include "ifns.h"
#include "vlist.h"
#include "fgetline.h"

//#define PATH_IFNS "/etc/ifns.conf"

vlist _ifnList;

static IfnNode* makeIfnNode(const char *ifn, const char *alias)
{
	IfnNode *pnode = (IfnNode*) 0;

	if (ifn && alias &&
			strlen(ifn) < MAX_IFN_LENGTH &&
			strlen(alias) < MAX_IFN_LENGTH) {
		pnode = (IfnNode*) malloc(sizeof(IfnNode));
		if (pnode) {
			strcpy(pnode->alias, alias);
			strcpy(pnode->ifn, ifn);
		}
	}
	return pnode;
}

void ifnsInit()
{
	FILE *fp = (FILE*) 0;
	char buffer[MAX_IFN_LINE + 1];
	char *ifn, *alias, *more, *brkp;
	const char *sep = " \t";
	IfnNode *pnode = (IfnNode*) 0;
	int ret = 0;

	ifn = alias = more = brkp = (char*)0;

	vlistInit(&_ifnList);

	fp = fopen(PATH_IFNS, "r");
	if (! fp)
                return;


	for (; (ret = fgetline(buffer, MAX_IFN_LINE, fp)) != -1; ) {
		if (! ret)
			continue;
		ifn = strtok_r(buffer, sep, &brkp);
		if (! ifn)
			continue;
		alias = strtok_r(NULL, sep, &brkp);
		if (! alias)
			continue;
		more = strtok_r(NULL, sep, &brkp);
		if (more)
			continue;
		if ((pnode = makeIfnNode(ifn, alias))) {
			if (! vlistInsertLast(&_ifnList, pnode))
				free(pnode);
		}
	}

	fclose(fp);
}

const char *ifnByAlias(const char *alias)
{
	const char *ifn = (const char*) 0;
	vlistEntry *iter = (vlistEntry*) 0;
	IfnNode *pnode = (IfnNode*) 0;

	if (alias) {
		iter = vlistFirstEntry(&_ifnList);
		for (; iter && ! ifn; iter = vlistNextEntry(iter)) {
			pnode = iter->elem;
			if (! strcmp(pnode->alias, alias))
				ifn = pnode->ifn;
		}
	}
	return ifn;
}

char *ifaddrByAlias(const char *alias)
{
	const char *ifn = ifnByAlias(alias);
	struct ifreq ifr; 
	int sockfd = -1;

	if (! ifn || (sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
		return (char*) 0;

	strncpy(ifr.ifr_name, ifn, IF_NAMESIZE);
	if (ioctl(sockfd, SIOCGIFADDR, &ifr) < 0) {
		close(sockfd);
		return (char*) 0;
	}
	close(sockfd);
	return inet_ntoa( ((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr );
}

char *ifmaskByAlias(const char *alias)
{
	const char *ifn = ifnByAlias(alias);
	struct ifreq ifr; 
	int sockfd = -1;

	if (! ifn || (sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
		return (char*) 0;

	strncpy(ifr.ifr_name, ifn, IF_NAMESIZE);
	if (ioctl(sockfd, SIOCGIFNETMASK, &ifr) < 0) {
		close(sockfd);
		return (char*) 0;
	}
	close(sockfd);
	return inet_ntoa( ((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr );
}

static void inaddr_and(in_addr_t *op1, const in_addr_t *op2)
{
	int i = 0;
	unsigned char *u1 = (unsigned char*) op1;
	const unsigned char *u2 = (const unsigned char*) op2;

	for (i = 0; i < 4; i++)
		(*u1++) &= (*u2++);
}

char *ifinetByAlias(const char *alias)
{
	const char *ifn = ifnByAlias(alias);
	int sock = -1;
	struct ifreq if1;
	struct ifreq if2;

	strncpy(if1.ifr_name, ifn, IF_NAMESIZE);
	strncpy(if2.ifr_name, ifn, IF_NAMESIZE);

	sock = socket(AF_INET, SOCK_DGRAM, 0);
	if (ioctl(sock, SIOCGIFADDR, &if1) < 0) {
		close(sock);
		return (char*) 0;
	}
	if (ioctl(sock, SIOCGIFNETMASK, &if2) < 0) {
		close(sock);
		return (char*) 0;
	}
	close(sock);
	inaddr_and(&((struct sockaddr_in*)&if1.ifr_addr)->sin_addr.s_addr,
			&((struct sockaddr_in*)&if2.ifr_addr)->sin_addr.s_addr);
	return inet_ntoa( ((struct sockaddr_in *)&if1.ifr_addr)->sin_addr );
}

char *ifstatByAlias(const char *alias)
{
	static char status[5];

	const char *ifn = ifnByAlias(alias);
	struct ifreq ifr; 
	int sockfd = -1;

	strcpy(status, "ERR");
	if (! ifn || (sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
		return status;

	strncpy(ifr.ifr_name, ifn, IF_NAMESIZE);
	if (ioctl(sockfd, SIOCGIFFLAGS, &ifr) < 0) {
		close(sockfd);
		return status;
	}
	close(sockfd);
	if (ifr.ifr_flags & IFF_UP)
		strcpy(status, "UP");
	else
		strcpy(status, "DOWN");
	return status;
}

int test_interfaces(const char *ifs)
{
	char buffer[MAXPATHLEN + 1];
	char *ifn = NULL;
	char *p = NULL;
	vlistEntry *iter = NULL;
	IfnNode *pn = NULL;
	vlist recentList;

	vlistInit(&recentList);

	strcpy(buffer, ifs);
	p = ifn = buffer;
	do {
		if ((p = strchr(ifn, ',')))
			*p = '\0';

		if (! *ifn) {
			vlistDestroy(&recentList, NULL);
			return 0;
		}

		/* ǷѾָ */
		iter = vlistFirstEntry(&recentList);
		for (; iter; iter = vlistNextEntry(iter)) {
			if (! strcmp((char*) iter->elem, ifn)) {
				vlistDestroy(&recentList, NULL);
				return 0;
			}
		}
		vlistInsertLast(&recentList, ifn);

		/* ǲǺϷĽӿڱ */
		iter = vlistFirstEntry(&_ifnList);
		for (; iter; iter = vlistNextEntry(iter)) {
			pn = iter->elem;
			if (! strcmp(pn->alias, ifn))
				break;
		}
		if (! iter) {
			vlistDestroy(&recentList, NULL);
			return 0;
		}
		if (p) {
			ifn = p + 1;
		}
	} while(p);

	vlistDestroy(&recentList, NULL);
	return 1;
}
