/* vi: set sw=4 ts=4 sts=4: */
/*
 *	utils.c -- System Utilities 
 *
 *	Copyright (c) Ralink Technology Corporation All Rights Reserved.
 *
 *	$Id: utils.c,v 1.99.2.1 2009-04-08 08:52:59 chhung Exp $
 */
#include	<time.h>
#include	<signal.h>
#include	<sys/ioctl.h>
#include	<sys/time.h>
#include	<sys/sysinfo.h>
#include	<sys/socket.h>
#include	<netinet/in.h>
#include	<arpa/inet.h>
#include	<dirent.h>

#include	"sdk_version.h"
#include	"nvram.h"
#include	"ralink_gpio.h"
#include	"linux/autoconf.h"  //kernel config
#include	"config/autoconf.h" //user config
#include	"../../sdk_version.h"
#include	"../../../config/autoconf.h"

#include	"webs.h"
#include	"internet.h"
#include	"nvram.h"
#include	"utils.h"
#include	"linux/autoconf.h"
#include	"wireless.h"

#if defined CONFIG_USB_STORAGE && defined CONFIG_USER_STORAGE
extern void setFirmwarePath(void);
#endif

#include <stdio.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <errno.h>

#define MULTI_LANG_REQUEST	9988
#define MULTI_LANG_FEEDBACK	9977
#define webPath "/etc_ro/web"
#define tagLen 32
#define transStringLen 256
	
struct transLang {
	char tag[tagLen];
	char transString[transStringLen];
}; 

struct transLang *currentLang;
int langCounter=0;
char multiLangInit='0';

static int  getCfgGeneral(int eid, webs_t wp, int argc, char_t **argv);
static int  getCfgNthGeneral(int eid, webs_t wp, int argc, char_t **argv);
static int  getCfgZero(int eid, webs_t wp, int argc, char_t **argv);
static int  getCfgNthZero(int eid, webs_t wp, int argc, char_t **argv);
static int  getCfg2General(int eid, webs_t wp, int argc, char_t **argv);
static int  getCfg2NthGeneral(int eid, webs_t wp, int argc, char_t **argv);
static int  getCfg2Zero(int eid, webs_t wp, int argc, char_t **argv);
static int  getCfg2NthZero(int eid, webs_t wp, int argc, char_t **argv);
static int  getCfg3General(int eid, webs_t wp, int argc, char_t **argv);
static int  getCfg3Zero(int eid, webs_t wp, int argc, char_t **argv);
static int  getDpbSta(int eid, webs_t wp, int argc, char_t **argv);
static int  getLangBuilt(int eid, webs_t wp, int argc, char_t **argv);
static int  getMiiInicBuilt(int eid, webs_t wp, int argc, char_t **argv);
static int  getPlatform(int eid, webs_t wp, int argc, char_t **argv);
static int  getStationBuilt(int eid, webs_t wp, int argc, char_t **argv);
static int  getSysBuildTime(int eid, webs_t wp, int argc, char_t **argv);
static int  getSdkVersion(int eid, webs_t wp, int argc, char_t **argv);
static int  getProjectFirmwareVersion(int eid, webs_t wp, int argc, char_t **argv);
static int  getProjectName(int eid, webs_t wp, int argc, char_t **argv);
static int  getSysTime(int eid, webs_t wp, int argc, char_t **argv);
static int  getSysUptime(int eid, webs_t wp, int argc, char_t **argv);
static int  getPortStatus(int eid, webs_t wp, int argc, char_t **argv);
static int  isOnePortOnly(int eid, webs_t wp, int argc, char_t **argv);
static void forceMemUpgrade(webs_t wp, char_t *path, char_t *query);
static void setOpMode(webs_t wp, char_t *path, char_t *query);
static void loadDefault(webs_t wp, char_t *path, char_t *query);
#if defined CONFIG_USB_STORAGE && defined CONFIG_USER_STORAGE
static void ScanUSBFirmware(webs_t wp, char_t *path, char_t *query);
#endif

/*********************************************************************
 * System Utilities
 */

//Jacky.Yang 21-Oct-2008, start - debug printf
void debug_printf(char_t *fmt, ...)
{
	if (openDebug == 1) {
		va_list		vargs;
		char_t		*cmd = NULL;
	
		va_start(vargs, fmt);
		if (fmtValloc(&cmd, WEBS_BUFSIZE, fmt, vargs) >= WEBS_BUFSIZE) {
			trace(0, T("doSystem: lost data, buffer overflow\n"));
		}
		va_end(vargs);

		if (cmd) {
			//trace(0, T("%s\n"), cmd);
			//rc = system(cmd);
			printf(cmd);
			
			DIR *dir;
			struct dirent *ptr;
			FILE *fp;
			char path[32];
			
			dir = opendir("/dev/pts");
			while((ptr = readdir(dir)) != NULL)
			{
				//printf("ptr->d_name:%s\n", ptr->d_name);
				if (strcmp(ptr->d_name, ".") && strcmp(ptr->d_name, ".."))
				{
					memset(path, '\0', sizeof(path));
					snprintf(path, sizeof(path), "/dev/pts/%s", ptr->d_name);
					//printf("path:%s\n", path);
					fp = fopen(path, "w");
					fputs(cmd, fp);
					fclose(fp);
				}
			}
			closedir(dir);

			bfree(B_L, cmd);
		}
	}
}
//Jacky.Yang 21-Oct-2008, end - debug printf

void arplookup(char *ip, char *arp)
{
    char buf[256];
    FILE *fp = fopen("/proc/net/arp", "r");
    if(!fp){
        trace(0, T("no proc fs mounted!\n"));
        return;
    }
    strcpy(arp, "00:00:00:00:00:00");
    while(fgets(buf, 256, fp)){
        char ip_entry[32], hw_type[8],flags[8], hw_address[32];
        sscanf(buf, "%s %s %s %s", ip_entry, hw_type, flags, hw_address);
        if(!strcmp(ip, ip_entry)){
            strcpy(arp, hw_address);
            break;
        }
    }

    fclose(fp);
    return;
}

//Add for status page - U-Media Ricky Cao on Mar. 18 2008
//description: get the process id of pName by ps command
int getPsPid(char *pName)
{
	FILE *fptr;
	int pid, vsz;
	char uid[8], stat[8], command[64], temp[256]; 

	doSystem("ps > /var/psInfo 2>/dev/null");
	fptr = fopen("/var/psInfo", "r");
	if(!fptr){
		printf("open \"/var/psInfo\" fail\n");
		return;
	}
	
	if(fgets(temp, 256, fptr)!=NULL){
		if(!strstr(temp, "PID")){
			rewind(temp);
		}
	}
	memset(temp, 0, 256);
	
	while(!feof(fptr)){
		if(fgets(temp, 256, fptr)==NULL){
			memset(temp, 0, 256);
			continue;
		}
		
		sscanf(temp, "%d %s %d %s %s", &pid, uid, &vsz, stat, command);
		if(strstr(command, pName)){
			//printf("GOT IT!!!!\n");
			goto end;
		}
		
		memset(temp, 0, 256);
		//printf("pid=%d, uid=%s, vsz=%d, stat=%s, command=%s\n", pid, uid, vsz, stat, command);
	}

	pid=0;
end:
	fclose(fptr);
	doSystem("rm /var/psInfo");

	return pid;
}

/*
 * description: get process whose pid was recorded in file
 *              (va is supported)
 */
int getPid(char_t *fmt, ...)
{
	va_list		vargs;
	char_t		*pid_fname = NULL;
	struct stat	st;
	FILE *fptr;
	char pidStr[8];
	int pidNum = 0;

	va_start(vargs, fmt);
	if (fmtValloc(&pid_fname, WEBS_BUFSIZE, fmt, vargs) >= WEBS_BUFSIZE) {
		trace(0, T("doKillPid: lost data, buffer overflow\n"));
	}
	va_end(vargs);

	if (pid_fname) {
		if (-1 == stat(pid_fname, &st)) //check the file existence
			return 0;
		fptr = fopen(pid_fname, "r");
		fgets(pidStr, 8, fptr);
		pidNum = atoi(pidStr);
		fclose(fptr);
		bfree(B_L, pid_fname);
	}
	return pidNum;
}
//U-Media Ricky Cao on Mar. 18 2008

/*
 * description: kill process whose pid was recorded in file
 *              (va is supported)
 */
int doKillPid(char_t *fmt, ...)
{
	va_list		vargs;
	char_t		*pid_fname = NULL;
	struct stat	st;

	va_start(vargs, fmt);
	if (fmtValloc(&pid_fname, WEBS_BUFSIZE, fmt, vargs) >= WEBS_BUFSIZE) {
		trace(0, T("doKillPid: lost data, buffer overflow\n"));
	}
	va_end(vargs);

	if (pid_fname) {
		if (-1 == stat(pid_fname, &st)) //check the file existence
			return 0;
		doSystem("kill `cat %s`", pid_fname);
		doSystem("rm -f %s", pid_fname);
		bfree(B_L, pid_fname);
	}
	return 0;
}

/*
 * description: parse va and do system
 */
int doSystem(char_t *fmt, ...)
{
	va_list		vargs;
	char_t		*cmd = NULL;
	int			rc = 0;
	
	va_start(vargs, fmt);
	if (fmtValloc(&cmd, WEBS_BUFSIZE, fmt, vargs) >= WEBS_BUFSIZE) {
		trace(0, T("doSystem: lost data, buffer overflow\n"));
	}
	va_end(vargs);

	if (cmd) {
		trace(0, T("%s\n"), cmd);
		rc = system(cmd);
		bfree(B_L, cmd);
	}
	return rc;
}

/*
 * arguments: modname - module name
 * description: test the insertion of module through /proc/modules
 * return: -1 = fopen error, 1 = module inserted, 0 = not inserted yet
 */
int getModIns(char *modname)
{
	FILE *fp;
	char buf[128];
	int i;

	if (NULL == (fp = fopen("/proc/modules", "r"))) {
		error(E_L, E_LOG, T("getModIns: open /proc/modules error"));
		return -1;
	}

	while (NULL != fgets(buf, 128, fp)) {
		i = 0;
		while (!isspace(buf[i++]))
			;
		buf[i - 1] = '\0';
		if (!strcmp(buf, modname)) {
			fclose(fp);
			return 1;
		}
	}
	fclose(fp);
	//error(E_L, E_LOG, T("getModIns: module %s not inserted"), modname);
	return 0;
}

/*
 * arguments: index - index of the Nth value
 *            values - un-parsed values
 * description: parse values delimited by semicolon, and return the value
 *              according to given index (starts from 0)
 * WARNING: the original values will be destroyed by strtok
 */
char *getNthValue(int index, char *values)
{
	int i;
	static char *tok;

	if (NULL == values)
		return NULL;
	for (i = 0, tok = strtok(values, ";");
			(i < index) && tok;
			i++, tok = strtok(NULL, ";")) {
		;
	}
	if (NULL == tok)
		return "";
	return tok;
}

/*
 * arguments: index - index of the Nth value (starts from 0)
 *            old_values - un-parsed values
 *            new_value - new value to be replaced
 * description: parse values delimited by semicolon,
 *              replace the Nth value with new_value,
 *              and return the result
 * WARNING: return the internal static string -> use it carefully
 */
char *setNthValue(int index, char *old_values, char *new_value)
{
	int i;
	char *p, *q;
	static char ret[2048];
	char buf[8][256];

	memset(ret, 0, 2048);
	for (i = 0; i < 8; i++)
		memset(buf[i], 0, 256);

	//copy original values
	for ( i = 0, p = old_values, q = strchr(p, ';')  ;
	      i < 8 && q != NULL                         ;
	      i++, p = q + 1, q = strchr(p, ';')         )
	{
		strncpy(buf[i], p, q - p);
	}
	strcpy(buf[i], p); //the last one

	//replace buf[index] with new_value
	strncpy(buf[index], new_value, 256);

	//calculate maximum index
	index = (i > index)? i : index;

	//concatenate into a single string delimited by semicolons
	strcat(ret, buf[0]);
	for (i = 1; i <= index; i++) {
		strncat(ret, ";", 2);
		strncat(ret, buf[i], 256);
	}

	p = ret;
	return p;
}

/*
 * arguments: values - values delimited by semicolon
 * description: parse values delimited by semicolon, and return the number of
 *              values
 */
int getValueCount(char *values)
{
	int cnt = 0;

	if (NULL == values)
		return 0;
	while (*values++ != '\0') {
		if (*values == ';')
			++cnt;
	}
	return (cnt + 1);
}

/*
 * check the existence of semicolon in str
 */
int checkSemicolon(char *str)
{
	char *c = strchr(str, ';');
	if (c)
		return 1;
	return 0;
}

/*
 * argument: ip address
 * return: 1 = the given ip address is within LAN's scope
 *         0 = otherwise
 */
int isInLan(char *radius_ip_addr)
{
    char lan_if_addr[16];
    char lan_if_netmask[16];

    struct in_addr lan_ip;
    struct in_addr lan_netmask;
    struct in_addr radius_ip;

    if ((getIfIp(getLanIfName(), lan_if_addr)) == -1) {
        printf("getLanIP error\n");
        return 0;
    }
    if ((getIfNetmask(getLanIfName(), lan_if_netmask)) == -1) {
        printf("getLanNetmask error\n");
        return 0;
    }

    inet_aton(lan_if_addr, &lan_ip);
    inet_aton(lan_if_netmask, &lan_netmask);
    inet_aton(radius_ip_addr, &radius_ip);

    printf("lan_ip:%08x\n", lan_ip.s_addr);
    printf("lan_netmask:%08x\n", lan_netmask.s_addr);
    printf("radius_ip:%08x\n", radius_ip.s_addr);

    if ((lan_ip.s_addr & lan_netmask.s_addr) == (radius_ip.s_addr & lan_netmask.s_addr) ){
        printf("in Lan\n");
        return 1;
    } else {
        printf("not in lan\n");
        return 0;
    }
}

/*
 * substitution of getNthValue which dosen't destroy the original value
 */
int getNthValueSafe(int index, char *value, char delimit, char *result, int len)
{
    int i=0, result_len=0;
    char *begin, *end;

    if(!value || !result || !len)
        return -1;

    begin = value;
    end = strchr(begin, delimit);

    while(i<index && end){
        begin = end+1;
        end = strchr(begin, delimit);
        i++;
    }

    //no delimit
    if(!end){
		if(i == index){
			end = begin + strlen(begin);
			result_len = (len-1) < (end-begin) ? (len-1) : (end-begin);
		}else
			return -1;
	}else
		result_len = (len-1) < (end-begin)? (len-1) : (end-begin);

	memcpy(result, begin, result_len );
	*(result+ result_len ) = '\0';

	return 0;
}

/*
 *  argument:  [IN]     index -- the index array of deleted items(begin from 0)
 *             [IN]     count -- deleted itmes count.
 *             [IN/OUT] value -- original string/return string
 *             [IN]     delimit -- delimitor
 */
int deleteNthValueMulti(int index[], int count, char *value, char delimit)
{
	char *begin, *end;
	int i=0,j=0;
	int need_check_flag=0;
	char *buf = strdup(value);

	begin = buf;

	end = strchr(begin, delimit);
	while(end){
		if(i == index[j]){
			memset(begin, 0, end - begin );
			if(index[j] == 0)
				need_check_flag = 1;
			j++;
			if(j >=count)
				break;
		}
		begin = end;

		end = strchr(begin+1, delimit);
		i++;
	}

	if(!end && index[j] == i)
		memset(begin, 0, strlen(begin));

	if(need_check_flag){
		for(i=0; i<strlen(value); i++){
			if(buf[i] == '\0')
				continue;
			if(buf[i] == ';')
				buf[i] = '\0';
			break;
		}
	}

	for(i=0, j=0; i<strlen(value); i++){
		if(buf[i] != '\0'){
			value[j++] = buf[i];
		}
	}
	value[j] = '\0';

	free(buf);
	return 0;
}



/*
 * nanosleep(2) don't depend on signal SIGALRM and could cooperate with
 * other SIGALRM-related functions(ex. setitimer(2))
 */
unsigned int Sleep(unsigned int secs)
{
	int rc;
	struct timespec ts, remain;
	ts.tv_sec  = secs;
	ts.tv_nsec = 0;

sleep_again:
	rc = nanosleep(&ts, &remain);
	if(rc == -1 && errno == EINTR){
		ts.tv_sec = remain.tv_sec;
		ts.tv_nsec = remain.tv_nsec;
		goto sleep_again;
	}	
	return 0;
}

/*
 * The setitimer() is Linux-specified.
 */
int setTimer(int microsec, void ((*sigroutine)(int)))
{
	struct itimerval value, ovalue;
   
	signal(SIGALRM, sigroutine);
	value.it_value.tv_sec = 0;
	value.it_value.tv_usec = microsec;
	value.it_interval.tv_sec = 0;
	value.it_interval.tv_usec = microsec;
	return setitimer(ITIMER_REAL, &value, &ovalue);
}

void stopTimer(void)
{
	struct itimerval value, ovalue;
   
	value.it_value.tv_sec = 0;
	value.it_value.tv_usec = 0;
	value.it_interval.tv_sec = 0;
	value.it_interval.tv_usec = 0;
	setitimer(ITIMER_REAL, &value, &ovalue);
}

/*
 * configure LED blinking with proper frequency (privatly use only)
 *   on: number of ticks that LED is on
 *   off: number of ticks that LED is off
 *   blinks: number of on/off cycles that LED blinks
 *   rests: number of on/off cycles that LED resting
 *   times: stop blinking after <times> times of blinking
 * where 1 tick == 100 ms
 */
static int gpioLedSet(int gpio, unsigned int on, unsigned int off,
		unsigned int blinks, unsigned int rests, unsigned int times)
{
	int fd;
	ralink_gpio_led_info led;

	//parameters range check
	if (gpio < 0 || gpio >= RALINK_GPIO_NUMBER ||
			on > RALINK_GPIO_LED_INFINITY ||
			off > RALINK_GPIO_LED_INFINITY ||
			blinks > RALINK_GPIO_LED_INFINITY ||
			rests > RALINK_GPIO_LED_INFINITY ||
			times > RALINK_GPIO_LED_INFINITY) {
		return -1;
	}
	led.gpio = gpio;
	led.on = on;
	led.off = off;
	led.blinks = blinks;
	led.rests = rests;
	led.times = times;

	fd = open("/dev/gpio", O_RDONLY);
	if (fd < 0) {
		perror("/dev/gpio");
		return -1;
	}
	if (ioctl(fd, RALINK_GPIO_LED_SET, &led) < 0) {
		perror("ioctl");
		close(fd);
		return -1;
	}
	close(fd);
	return 0;
}

int ledAlways(int gpio, int on)
{
	if (on)
		return gpioLedSet(gpio, RALINK_GPIO_LED_INFINITY, 0, 1, 1, RALINK_GPIO_LED_INFINITY);
	else
		return gpioLedSet(gpio, 0, RALINK_GPIO_LED_INFINITY, 1, 1, RALINK_GPIO_LED_INFINITY);
}

int ledWps(int gpio, int mode)
{
	switch (mode) {
		case WPS_LED_RESET:
			return gpioLedSet(gpio, 0, RALINK_GPIO_LED_INFINITY, 1, 1, RALINK_GPIO_LED_INFINITY);
			break;
		case WPS_LED_PROGRESS:
			return gpioLedSet(gpio, 2, 1, RALINK_GPIO_LED_INFINITY, 1, 1);
			break;
		case WPS_LED_ERROR:
			//return gpioLedSet(gpio, 1, 1, 800, 1, 1); //Jacky.Yang 12-Mar-2008, working 120s
			//return gpioLedSet(gpio, 1, 1, RALINK_GPIO_LED_INFINITY, 1, 1); //Jacky.Yang 12-Mar-2008, working 120s
			return gpioLedSet(gpio, 1, 1, 3000, 1, 1); //Jacky.Yang 12-Mar-2008, working 120s
			break;
		case WPS_LED_SESSION_OVERLAP:
			//return gpioLedSet(gpio, 1, 1, 10, 5, 80); //Jacky.Yang 12-Mar-2008, working 120s
			//return gpioLedSet(gpio, 1, 1, RALINK_GPIO_LED_INFINITY, 5, 80); //Jacky.Yang 12-Mar-2008, working 120s
			return gpioLedSet(gpio, 1, 1, 100, 5, 80); //Jacky.Yang 12-Mar-2008, working 120s
			break;
		case WPS_LED_SUCCESS:
			//gpioLedSet(gpio, 1200, 1, 1, 1, 1); //Jacky.Yang 12-Mar-2008, working 120s
			gpioLedSet(gpio, 3000, 1, 1, 1, 1); //Jacky.Yang 12-Mar-2008, working 120s
			break;
	}
	return 0;
}

/*
 * concatenate a string with an integer
 * ex: racat("SSID", 1) will return "SSID1"
 */
char *racat(char *s, int i)
{
	static char str[32];
	snprintf(str, 32, "%s%1d", s, i);
	return str;
}

void websLongWrite(webs_t wp, char *longstr)
{
    char tmp[513] = {0};
    int len = strlen(longstr);
    char *end = longstr + len;

    while(longstr < end){
        strncpy(tmp, longstr, 512);
        websWrite(wp, T("%s"), tmp);
        longstr += 512;
    }
    return;
}

/* 
 * arguments: type - 0 = return the configuration of 'field' (default)
 *                   1 = write the configuration of 'field' 
 *            field - parameter name in nvram
 * description: read general configurations from nvram
 *              if value is NULL -> ""
 */
static int getCfgGeneral(int eid, webs_t wp, int argc, char_t **argv)
{
	int type, flag=0, loopCount, rc;
	char_t *field;
	char *value;
	char checkSSIDName[8];
	unsigned char *transPt;
	unsigned char transToHex[128];
	//Tom.Hung 24-Nov-2008, trans WdsKey to Hex character.
	char checkWdsKey[8];

	if (ejArgs(argc, argv, T("%d %s"), &type, &field) < 2) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	value = nvram_bufget(RT2860_NVRAM, field);
	if (1 == type) {
		if (NULL == value)
			return websWrite(wp, T(""));
		//Jacky.Yang 24-Apr-2008, check SSID and use escape to parse sepcial character.
		for (loopCount=1; loopCount<=8; loopCount++)
		{
			snprintf(checkSSIDName, sizeof(checkSSIDName), "SSID%d", loopCount);
			//printf("jacky - value=%s, checkSSIDName=%s\n", field, checkSSIDName);
			if (!strcmp(field, checkSSIDName)) {
				flag = 1;
				loopCount = 9;
			}
		}

		//Tom.Hung 24-Nov-2008, trans WdsKey to Hex character.
		snprintf(checkWdsKey, sizeof(checkWdsKey), "WdsKey");
		if (!strcmp(field, checkWdsKey)) {
			memset(transToHex, '\0', sizeof(transToHex));
			rc=0;
			transPt = transToHex;
			for (loopCount=0; loopCount<strlen(value); loopCount++)
			{
				rc = snprintf(transPt, sizeof(transToHex), "%x", value[loopCount]);
				transPt = transPt + rc;
			}
			return websWrite(wp, T("%s"), transToHex);
		}
		//Tom.Hung 24-Nov-2008

		if (flag == 0)
			return websWrite(wp, T("%s"), value);
		else
		{
			memset(transToHex, '\0', sizeof(transToHex));
			rc=0;
			transPt = transToHex;
			for (loopCount=0; loopCount<strlen(value); loopCount++)
			{
				//transToHex[loopCount] = toascii(value[loopCount]);
				rc = snprintf(transPt, sizeof(transToHex), "%x", value[loopCount]);
				transPt = transPt + rc;
				//printf("transToHex[%d]=%x\n", loopCount, transToHex[loopCount]);
				/*if (value[loopCount] == '"') {
					printf("find - value[%d]=%c\n", loopCount, value[loopCount]);
				}
				else {
					printf("value[%d]=%c\n", loopCount, value[loopCount]);
				}*/
			}
			//return websWrite(wp, T("%s"), value);
			return websWrite(wp, T("%s"), transToHex);
		}
	}
	if (NULL == value)
		ejSetResult(eid, "");
	ejSetResult(eid, value);
	return 0;
}

/* 
 * arguments: type - 0 = return the configuration of 'field' (default)
 *                   1 = write the configuration of 'field' 
 *            field - parameter name in nvram
 *            idx - index of nth
 * description: read general configurations from nvram (if value is NULL -> "")
 *              parse it and return the Nth value delimited by semicolon
 */
static int getCfgNthGeneral(int eid, webs_t wp, int argc, char_t **argv)
{
	int type, idx;
	char_t *field;
	char *value;
	char *nth;

	if (ejArgs(argc, argv, T("%d %s %d"), &type, &field, &idx) < 3) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	value = nvram_bufget(RT2860_NVRAM, field);
	if (1 == type) {
		if (NULL == value)
			return websWrite(wp, T(""));
		nth = getNthValue(idx, value);
		if (NULL == nth)
			return websWrite(wp, T(""));
		return websWrite(wp, T("%s"), nth);
	}
	if (NULL == value)
		ejSetResult(eid, "");
	nth = getNthValue(idx, value);
	if (NULL == nth)
		ejSetResult(eid, "");
	ejSetResult(eid, value);
	return 0;
}

/*
 * arguments: type - 0 = return the configuration of 'field' (default)
 *                   1 = write the configuration of 'field' 
 *            field - parameter name in nvram
 * description: read general configurations from nvram
 *              if value is NULL -> "0"
 */
static int getCfgZero(int eid, webs_t wp, int argc, char_t **argv)
{
	int type;
	char_t *field;
	char *value;

	if (ejArgs(argc, argv, T("%d %s"), &type, &field) < 2) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	value = nvram_bufget(RT2860_NVRAM, field);
	if (1 == type) {
		if (!strcmp(value, ""))
			return websWrite(wp, T("0"));
		return websWrite(wp, T("%s"), value);
	}
	if (!strcmp(value, ""))
		ejSetResult(eid, "0");
	ejSetResult(eid, value);
	return 0;
}

/* 
 * arguments: type - 0 = return the configuration of 'field' (default)
 *                   1 = write the configuration of 'field' 
 *            field - parameter name in nvram
 *            idx - index of nth
 * description: read general configurations from nvram (if value is NULL -> "0")
 *              parse it and return the Nth value delimited by semicolon
 */
static int getCfgNthZero(int eid, webs_t wp, int argc, char_t **argv)
{
	int type, idx;
	char_t *field;
	char *value;
	char *nth;

	if (ejArgs(argc, argv, T("%d %s %d"), &type, &field, &idx) < 3) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	value = nvram_bufget(RT2860_NVRAM, field);
	if (1 == type) {
		if (!strcmp(value, ""))
			return websWrite(wp, T("0"));
		nth = getNthValue(idx, value);
		if (NULL == nth)
			return websWrite(wp, T("0"));
		return websWrite(wp, T("%s"), nth);
	}
	if (!strcmp(value, ""))
		ejSetResult(eid, "0");
	nth = getNthValue(idx, value);
	if (NULL == nth)
		ejSetResult(eid, "0");
	ejSetResult(eid, value);
	return 0;
}

/* 
 * arguments: type - 0 = return the configuration of 'field' (default)
 *                   1 = write the configuration of 'field' 
 *            field - parameter name in nvram
 * description: read general configurations from nvram
 *              if value is NULL -> ""
 */
static int getCfg2General(int eid, webs_t wp, int argc, char_t **argv)
{
	int type;
	char_t *field;
	char *value;

	if (ejArgs(argc, argv, T("%d %s"), &type, &field) < 2) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	value = nvram_bufget(RTINIC_NVRAM, field);
	if (1 == type) {
		if (NULL == value)
			return websWrite(wp, T(""));
		return websWrite(wp, T("%s"), value);
	}
	if (NULL == value)
		ejSetResult(eid, "");
	ejSetResult(eid, value);
	return 0;
}

/* 
 * arguments: type - 0 = return the configuration of 'field' (default)
 *                   1 = write the configuration of 'field' 
 *            field - parameter name in nvram
 *            idx - index of nth
 * description: read general configurations from nvram (if value is NULL -> "")
 *              parse it and return the Nth value delimited by semicolon
 */
static int getCfg2NthGeneral(int eid, webs_t wp, int argc, char_t **argv)
{
	int type, idx;
	char_t *field;
	char *value;
	char *nth;

	if (ejArgs(argc, argv, T("%d %s %d"), &type, &field, &idx) < 3) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	value = nvram_bufget(RTINIC_NVRAM, field);
	if (1 == type) {
		if (NULL == value)
			return websWrite(wp, T(""));
		nth = getNthValue(idx, value);
		if (NULL == nth)
			return websWrite(wp, T(""));
		return websWrite(wp, T("%s"), nth);
	}
	if (NULL == value)
		ejSetResult(eid, "");
	nth = getNthValue(idx, value);
	if (NULL == nth)
		ejSetResult(eid, "");
	ejSetResult(eid, value);
	return 0;
}

/*
 * arguments: type - 0 = return the configuration of 'field' (default)
 *                   1 = write the configuration of 'field' 
 *            field - parameter name in nvram
 * description: read general configurations from nvram
 *              if value is NULL -> "0"
 */
static int getCfg2Zero(int eid, webs_t wp, int argc, char_t **argv)
{
	int type;
	char_t *field;
	char *value;

	if (ejArgs(argc, argv, T("%d %s"), &type, &field) < 2) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	value = nvram_bufget(RTINIC_NVRAM, field);
	if (1 == type) {
		if (!strcmp(value, ""))
			return websWrite(wp, T("0"));
		return websWrite(wp, T("%s"), value);
	}
	if (!strcmp(value, ""))
		ejSetResult(eid, "0");
	ejSetResult(eid, value);
	return 0;
}

/* 
 * arguments: type - 0 = return the configuration of 'field' (default)
 *                   1 = write the configuration of 'field' 
 *            field - parameter name in nvram
 *            idx - index of nth
 * description: read general configurations from nvram (if value is NULL -> "0")
 *              parse it and return the Nth value delimited by semicolon
 */
static int getCfg2NthZero(int eid, webs_t wp, int argc, char_t **argv)
{
	int type, idx;
	char_t *field;
	char *value;
	char *nth;

	if (ejArgs(argc, argv, T("%d %s %d"), &type, &field, &idx) < 3) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	value = nvram_bufget(RTINIC_NVRAM, field);
	if (1 == type) {
		if (!strcmp(value, ""))
			return websWrite(wp, T("0"));
		nth = getNthValue(idx, value);
		if (NULL == nth)
			return websWrite(wp, T("0"));
		return websWrite(wp, T("%s"), nth);
	}
	if (!strcmp(value, ""))
		ejSetResult(eid, "0");
	nth = getNthValue(idx, value);
	if (NULL == nth)
		ejSetResult(eid, "0");
	ejSetResult(eid, value);
	return 0;
}

/* 
 * arguments: type - 0 = return the configuration of 'field' (default)
 *                   1 = write the configuration of 'field' 
 *            field - parameter name in nvram
 * description: read general configurations from nvram
 *              if value is NULL -> ""
 */
static int getCfg3General(int eid, webs_t wp, int argc, char_t **argv)
{
	int type;
	char_t *field;
	char *value;
	
	if (ejArgs(argc, argv, T("%d %s"), &type, &field) < 2) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	value = nvram_bufget(RT2561_NVRAM, field);
	if (1 == type) {
		if (NULL == value)
			return websWrite(wp, T(""));
		return websWrite(wp, T("%s"), value);
	}
	if (NULL == value)
		ejSetResult(eid, "");
	ejSetResult(eid, value);
	return 0;
}

/*
 * arguments: type - 0 = return the configuration of 'field' (default)
 *                   1 = write the configuration of 'field' 
 *            field - parameter name in nvram
 * description: read general configurations from nvram
 *              if value is NULL -> "0"
 */
static int getCfg3Zero(int eid, webs_t wp, int argc, char_t **argv)
{
	int type;
	char_t *field;
	char *value;

	if (ejArgs(argc, argv, T("%d %s"), &type, &field) < 2) {
		return websWrite(wp, T("Insufficient args"));
	}
	value = nvram_bufget(RT2561_NVRAM, field);
	if (1 == type) {
		if (!strcmp(value, ""))
			return websWrite(wp, T("0"));
		return websWrite(wp, T("%s"), value);
	}
	if (!strcmp(value, ""))
		ejSetResult(eid, "0");
	ejSetResult(eid, value);
	return 0;
}

static int getDpbSta(int eid, webs_t wp, int argc, char_t **argv)
{
#ifdef CONFIG_RT2860V2_STA_DPB
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

static int getLangBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
	char_t *lang;

	if (ejArgs(argc, argv, T("%s"), &lang) < 1) {
		return websWrite(wp, T("0"));
	}
	if (!strncmp(lang, "en", 3))
#ifdef CONFIG_USER_GOAHEAD_LANG_EN
		return websWrite(wp, T("1"));
#else
		return websWrite(wp, T("0"));
#endif
	else if (!strncmp(lang, "zhtw", 5))
#ifdef CONFIG_USER_GOAHEAD_LANG_ZHTW
		return websWrite(wp, T("1"));
#else
		return websWrite(wp, T("0"));
#endif
	else if (!strncmp(lang, "zhcn", 5))
#ifdef CONFIG_USER_GOAHEAD_LANG_ZHCN
		return websWrite(wp, T("1"));
#else
		return websWrite(wp, T("0"));
#endif

	return websWrite(wp, T("0"));
}

static int getMiiInicBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined (CONFIG_INIC_MII)
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

/*
 * description: write the current system platform accordingly
 */
static int getPlatform(int eid, webs_t wp, int argc, char_t **argv)
{
#ifdef CONFIG_RAETH_ROUTER
	return websWrite(wp, T("RT2880 with IC+ MACPHY"));
#endif
#ifdef CONFIG_ICPLUS_PHY
    return websWrite(wp, T("RT2880 with IC+ PHY"));
#endif
#ifdef CONFIG_RT_MARVELL
	return websWrite(wp, T("RT2880 with MARVELL"));
#endif
#ifdef CONFIG_MAC_TO_MAC_MODE
	return websWrite(wp, T("RT2880 with Vitesse"));
#endif
#ifdef CONFIG_VSC7385XYV
	return websWrite(wp, T("RT2880 with VSC7385XYV"));
#endif
#ifdef CONFIG_RTL8366SR
	return websWrite(wp, T("RT2880 with RTL8366SR"));
#endif
#ifdef CONFIG_RT_3052_ESW
	return websWrite(wp, T("RT3052 embedded switch"));
#endif
    
	return 0;
}

static int getStationBuilt(int eid, webs_t wp, int argc, char_t **argv)
{
#if defined CONFIG_RT2860V2_STA || defined CONFIG_RT2860V2_STA_MODULE
	return websWrite(wp, T("1"));
#else
	return websWrite(wp, T("0"));
#endif
}

/*
 * description: write System build time
 */
static int getSysBuildTime(int eid, webs_t wp, int argc, char_t **argv)
{
	return websWrite(wp, T("%s"), __DATE__);
}

/*
 * description: write RT288x SDK version
 */
static int getSdkVersion(int eid, webs_t wp, int argc, char_t **argv)
{
	return websWrite(wp, T("%s"), RT288X_SDK_VERSION);
}

/*
 * Jacky.Yang 06-Dec-2007
 * Display Project firmware version and date.
 * Define the value in xxx_config.verdor-2.4x
 */
static int getProjectFirmwareVersion(int eid, webs_t wp, int argc, char_t **argv)
{
	return websWrite(wp, T("%s"), ProjectFirmwareVersionDate);
}

static int getProjectName(int eid, webs_t wp, int argc, char_t **argv)
{
	return websWrite(wp, T("%s"), TitleName);
}

//Jacky.Yang Begin 12-Sep-2007
static int getSysTime(int eid, webs_t wp, int argc, char_t **argv)
{
	time_t timep;
	struct tm *p;
	time(&timep);
	//printf("jacky - UTC Second:%d\n", timep);
	//p = localtime(&timep);
	//timep = mktime(p);
	//printf("jacky - UTC Second:%d\n", timep);
	return websWrite(wp, T("%d"), timep);
	/*time_t sysTime;
	time(&sysTime);
	
	return websWrite(wp, T("%s"), ctime(&sysTime));*/

}
//Jacky.Yang End 12-Sep-2007

//Jacky.Yang Begin 17-Sep-2007
static int getSysUptime(int eid, webs_t wp, int argc, char_t **argv)
{
	int updays, uphours, upminutes;
	struct sysinfo info;
	struct tm *current_time;
	time_t current_secs;

	time(&current_secs);
	current_time = localtime(&current_secs);

	sysinfo(&info);
	updays = (int) info.uptime / (60*60*24);
	return websWrite(wp, T("%d"), info.uptime);
}
//Jacky.Yang End 17-Sep-2007

static int getPortStatus(int eid, webs_t wp, int argc, char_t **argv)
{
//Tim Wang, for RTL8366SR driver
//#if defined CONFIG_RAETH_ROUTER && defined CONFIG_USER_ETHTOOL
#if (defined CONFIG_RAETH_ROUTER || defined CONFIG_RT_3052_ESW || defined CONFIG_RTL8366SR) && defined CONFIG_USER_ETHTOOL
	int port, rc=-1;
	FILE *fp;
	char buf[1024];

	for(port=0; port<5; port++){
		char *pos;
		char link = '0';
		int speed = 100;
		char duplex = 'F';
		FILE *proc_file = fopen("/proc/rt2880/gmac", "w");
		if(!proc_file){
			websWrite(wp, T("-1"));		// indicate error
			return 0;
		}
		fprintf(proc_file, "%d", port);
		fclose(proc_file);

		if((fp = popen("ethtool eth2", "r")) == NULL){
			websWrite(wp, T("-1"));		// indicate error
			return 0;
		}
		rc = fread(buf, 1, 1024, fp);
		pclose(fp);
		if(rc == -1){
			websWrite(wp, T("-1"));		// indicate error
			return 0;
		}else{
			// get Link status
			if((pos = strstr(buf, "Link detected: ")) != NULL){
				pos += strlen("Link detected: ");
				if(*pos == 'y')
					link = '1';
			}
			// get speed
			if((pos = strstr(buf, "Speed: ")) != NULL){
				pos += strlen("Speed: ");
				if(*pos == '1' && *(pos+1) == '0' && *(pos+2) == 'M')
					speed = 10;
				if(*pos == '1' && *(pos+1) == '0' && *(pos+2) == '0' && *(pos+3) == '0' && *(pos+4) == 'M')
					speed = 1000;
			}
			// get duplex
			if((pos = strstr(buf, "Duplex: ")) != NULL){
				pos += strlen("Duplex: ");
				if(*pos == 'H')
					duplex = 'H';
			}

			websWrite(wp, T("%c,%d,%c,"), link, speed, duplex);
		}
	}
	return 0;
#else
	websWrite(wp, T("-1"));
	return 0;
#endif

}

inline int getOnePortOnly(void)
{
//Tim Wang, for RTL8366SR driver
//#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW
#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW || defined CONFIG_RTL8366SR
	return 0;
#elif defined CONFIG_ICPLUS_PHY
	return 1;
#else
	return 0;
#endif
	return 0;
}

static int isOnePortOnly(int eid, webs_t wp, int argc, char_t **argv)
{
	if( getOnePortOnly() == 1)
		websWrite(wp, T("true"));
	else
		websWrite(wp, T("false"));		 
	return 0;
}

void redirect_wholepage(webs_t wp, const char *url)
{
	websWrite(wp, T("HTTP/1.1 200 OK\nContent-type: text/html\nPragma: no-cache\nCache-Control: no-cache\n\n"));
	websWrite(wp, T("<html><head><script language=\"JavaScript\">"));
	websWrite(wp, T("parent.location.replace(\"%s\");"), url);
	websWrite(wp, T("</script></head></html>"));
}

int netmask_aton(const char *ip)
{
	int i, a[4], result = 0;
	sscanf(ip, "%d.%d.%d.%d", &a[0], &a[1], &a[2], &a[3]);
	for(i=0; i<4; i++){	//this is dirty
		if(a[i] == 255){
			result += 8;
			continue;
		}
		if(a[i] == 254)
			result += 7;
		if(a[i] == 252)
			result += 6;
		if(a[i] == 248)
			result += 5;
		if(a[i] == 240)
			result += 4;
		if(a[i] == 224)
			result += 3;
		if(a[i] == 192)
			result += 2;
		if(a[i] == 128)
			result += 1;
		//if(a[i] == 0)
		//	result += 0;
		break;
	}
	return result;
}
static void forceMemUpgrade(webs_t wp, char_t *path, char_t *query)
{
	char_t *mode  = websGetVar(wp, T("ForceMemUpgradeSelect"), T("0"));
	if(!mode)
		return;
	if(!strcmp(mode, "1"))
		nvram_bufset(RT2860_NVRAM, "Force_mem_upgrade", "1");
	else
		nvram_bufset(RT2860_NVRAM, "Force_mem_upgrade", "0");
	nvram_commit(RT2860_NVRAM);
    websHeader(wp);
    websWrite(wp, T("<h2>force mem upgrade</h2>\n"));
    websWrite(wp, T("mode: %s<br>\n"), mode);
    websFooter(wp);
    websDone(wp, 200);	
}

#if defined CONFIG_USB_STORAGE && defined CONFIG_USER_STORAGE
static void ScanUSBFirmware(webs_t wp, char_t *path, char_t *query)
{
	setFirmwarePath();
	printf("%s enter\n", __FUNCTION__);

	websRedirect(wp, "adm/upload_firmware.asp");
}
#endif

/* goform/setOpMode */
static void setOpMode(webs_t wp, char_t *path, char_t *query)
{
	char_t	*mode, *nat_en;
	char	*old_mode = nvram_bufget(RT2860_NVRAM, "OperationMode");
	char	*old_nat = nvram_bufget(RT2860_NVRAM, "natEnabled");
	int		need_commit = 0;
//Tim Wang, for RTL8366SR driver
//#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW || defined CONFIG_ICPLUS_PHY
#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW || defined CONFIG_ICPLUS_PHY  || defined CONFIG_RTL8366SR
#else
	char	*wan_ip, *lan_ip;
#endif
#ifdef CONFIG_RT2860V2_STA_DPB
	char_t	*econv = "";
#endif
#if defined INIC_SUPPORT || defined INICv2_SUPPORT
	char_t	*mii;
#endif

	mode = websGetVar(wp, T("opMode"), T("0")); 
	nat_en = websGetVar(wp, T("natEnbl"), T("0"));

	if (!strncmp(old_mode, "0", 2)) {
		if (!strncmp(mode, "1", 2)) {
			nvram_bufset(RT2860_NVRAM, "wanConnectionMode", "DHCP");
			need_commit = 1;
		}
	}
	else if (!strncmp(old_mode, "1", 2) || !strncmp(old_mode, "3", 2)) {
		if (!strncmp(mode, "0", 2)) {
			nvram_bufset(RT2860_NVRAM, "wanConnectionMode", "STATIC");
			need_commit = 1;
//Tim Wang, for RTL8366SR driver
//#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW || defined CONFIG_ICPLUS_PHY
#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW || defined CONFIG_ICPLUS_PHY  || defined CONFIG_RTL8366SR
#else
			/*
			 * mode: gateway (or ap-client) -> bridge
			 * config: wan_ip(wired) overwrites lan_ip(bridge)
			 */
			wan_ip = nvram_bufget(RT2860_NVRAM, "wan_ipaddr");
			nvram_bufset(RT2860_NVRAM, "lan_ipaddr", wan_ip);
			need_commit = 1;
#endif
		}
		if (!strncmp(mode, "2", 2)) {
//Tim Wang, for RTL8366SR driver
//#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW || defined CONFIG_ICPLUS_PHY
#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW || defined CONFIG_ICPLUS_PHY  || defined CONFIG_RTL8366SR
#else
			/*
			 * mode: gateway (or ap-client) -> ethernet-converter
			 * config: wan_ip(wired) overwrites lan_ip(wired) 
			 *         lan_ip(wireless) overwrites wan_ip(wireless)
			 */
			wan_ip = nvram_bufget(RT2860_NVRAM, "wan_ipaddr");
			lan_ip = nvram_bufget(RT2860_NVRAM, "lan_ipaddr");
			nvram_bufset(RT2860_NVRAM, "lan_ipaddr", wan_ip);
			nvram_bufset(RT2860_NVRAM, "wan_ipaddr", lan_ip);
			need_commit = 1;
#endif
		}
	}
	else if (!strncmp(old_mode, "2", 2)) {
		if (!strncmp(mode, "0", 2)) {
			/*
			 * mode: wireless-isp -> bridge
			 * config: lan_ip(wired) overwrites lan_ip(bridge) -> the same
			 */
		}
		else if (!strncmp(mode, "1", 2) || !strncmp(mode, "3", 2)) {
//Tim Wang, for RTL8366SR driver
//#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW || defined CONFIG_ICPLUS_PHY
#if defined CONFIG_RAETH_ROUTER || defined CONFIG_MAC_TO_MAC_MODE || defined CONFIG_RT_3052_ESW || defined CONFIG_ICPLUS_PHY  || defined CONFIG_RTL8366SR
#else
			/*
			 * mode: ethernet-converter -> gateway (or ap-client)
			 * config: lan_ip(wired) overwrites wan_ip(wired) 
			 *         wan_ip(wireless) overwrites lan_ip(wireless)
			 */
			wan_ip = nvram_bufget(RT2860_NVRAM, "wan_ipaddr");
			lan_ip = nvram_bufget(RT2860_NVRAM, "lan_ipaddr");
			nvram_bufset(RT2860_NVRAM, "lan_ipaddr", wan_ip);
			nvram_bufset(RT2860_NVRAM, "wan_ipaddr", lan_ip);
			need_commit = 1;
#endif
		}
	}

#ifdef CONFIG_RT2860V2_STA_DPB
	if (!strncmp(mode, "0", 2)) {
		char *old;

		econv = websGetVar(wp, T("ethConv"), T("0"));
		old = nvram_bufget(RT2860_NVRAM, "ethConvert");
		if (strncmp(old, econv, 2)) {
			nvram_bufset(RT2860_NVRAM, "ethConvert", econv);
			need_commit = 1;
		}
		if (!strncmp(econv, "1", 2)) {
			//disable dhcp server in this mode
			old = nvram_bufget(RT2860_NVRAM, "dhcpEnabled");
			if (!strncmp(old, "1", 2)) {
				nvram_bufset(RT2860_NVRAM, "dhcpEnabled", "0");
				need_commit = 1;
			}
		}
	}
#endif

	//new OperationMode
	if (strncmp(mode, old_mode, 2)) {
		nvram_bufset(RT2860_NVRAM, "OperationMode", mode);

		//from or to ap client mode
		if (!strncmp(mode, "3", 2))
			nvram_bufset(RT2860_NVRAM, "ApCliEnable", "1");
		else if (!strncmp(old_mode, "3", 2))
			nvram_bufset(RT2860_NVRAM, "ApCliEnable", "0");
		need_commit = 1;
	}

	if (strncmp(nat_en, old_nat, 2)) {
		nvram_bufset(RT2860_NVRAM, "natEnabled", nat_en);
		need_commit = 1;
	}

	// For 100PHY  ( Ethernet Convertor with one port only)
	// If this is one port only board(IC+ PHY) then redirect
	// the user browser to our alias ip address.
	if( getOnePortOnly() ){
		//     old mode is Gateway, and new mode is BRIDGE/WirelessISP/Apcli
		if (    (!strcmp(old_mode, "1") && !strcmp(mode, "0"))  ||
				(!strcmp(old_mode, "1") && !strcmp(mode, "2"))  ||
				(!strcmp(old_mode, "1") && !strcmp(mode, "3"))  ){
			char redirect_url[512];
			char *lan_ip = nvram_bufget(RT2860_NVRAM, "lan_ipaddr");

			if(! strlen(lan_ip))
				lan_ip = "10.10.10.254";
			snprintf(redirect_url, 512, "http://%s", lan_ip);
			redirect_wholepage(wp, redirect_url);
			goto final;
        }

		//     old mode is BRIDGE/WirelessISP/Apcli, and new mode is Gateway
		if (    (!strcmp(old_mode, "0") && !strcmp(mode, "1"))  ||
				(!strcmp(old_mode, "2") && !strcmp(mode, "1"))  ||
				(!strcmp(old_mode, "3") && !strcmp(mode, "1"))  ){
			redirect_wholepage(wp, "http://172.32.1.254");
			goto final;
		}
	}
    
#if defined INIC_SUPPORT || defined INICv2_SUPPORT
	mii = websGetVar(wp, T("miiMode"), T("0"));
	if (!strncmp(mode, "0", 2)) {
		char *old_mii = nvram_bufget(RTINIC_NVRAM, "InicMiiEnable");

		if (strncmp(mii, old_mii, 2)) {
			nvram_set(RTINIC_NVRAM, "InicMiiEnable", mii);
			need_commit = 1; //force to run initInternet
		}
	}
	else {
		nvram_set(RTINIC_NVRAM, "InicMiiEnable", "0");
		need_commit = 1; //force to run initInternet
	}
#endif

final:
	sleep(1);	// wait for websDone() to finish tcp http session(close socket)

	//restart internet if any changes
	if (need_commit) {
		nvram_commit(RT2860_NVRAM);
		updateFlash8021x(RT2860_NVRAM);
		//initInternet("");
	}
	//Tom.Hung 2009-3-18, reboot after change mode
	//doSystem("sleep 3 && reboot &");
	//websRedirect(wp, "/wait_page.asp");
	//Tom.Hung 2009-3-18
	websRedirect(wp, "/applied2.asp?url=/adm/opmode.asp");
}

/* goform/loadDefault */
static void loadDefault(webs_t wp, char_t *path, char_t *query)
{
    char defaultString1[256], defaultString2[256];
    
    strcpy(defaultString1, "ralink_init renew 2860 /etc_ro/Wireless/RT2860AP/");
	strcpy(defaultString2, "ralink_init renew inic /etc_ro/Wireless/RT2860AP/");
	strcat(defaultString1, LoadToDefaultFileName_2860);
	strcat(defaultString2, LoadToDefaultFileName_inic);
	
	doSystem("ralink_init clear 2860");
	doSystem("ralink_init clear inic");
	doSystem(defaultString1);
	doSystem(defaultString2);
	doSystem("sleep 3&& reboot &");
}

int checkDelete(char *nvramTagName, char *delRuleName)
{
	int loopCount, flag;
	char rec[256], getValue[32];
	//char *crontab = nvram_bufget(RT2860_NVRAM, "crontab");
	char *rules = nvram_bufget(RT2860_NVRAM, nvramTagName);

	flag = 0;
	loopCount=0;
	
	if (!strcmp(nvramTagName, "crontab"))
	{
		//get schedule
		while(getNthValueSafe(loopCount++, rules, ';', rec, sizeof(rec)) != -1 ){
			if((getNthValueSafe(2, rec, ',', getValue, sizeof(getValue)) == -1)){}
			if (!strcmp(delRuleName, getValue)) {
				flag = 1;
			}
		}
	}
	else if (!strcmp(nvramTagName, "SinglePortForwardRules") || !strcmp(nvramTagName, "PortRangeForwardRules"))
	{
		//get inbound filter
		while(getNthValueSafe(loopCount++, rules, ';', rec, sizeof(rec)) != -1 ){
			if((getNthValueSafe(3, rec, ',', getValue, sizeof(getValue)) == -1)){}
			if (!strcmp(delRuleName, getValue)) {
				flag = 1;
			}
		}
	}
	
	if (flag == 0)
		return 1;
	else
		return 0;
}

void lang_free()
{
	if (multiLangInit != '0')
	{
		printf("lang_free: start to release multi-lang mapping.\n");
		langCounter = 0;
		free(currentLang);
		multiLangInit='0';
		printf("lang_free: end to release multi-lang mapping.\n");
	}
	else
		printf("lang_free: Don't need release multi-lang mapping again.\n");
}

int lang_get(char *tag, char **getString)
{
	int loopCount=0;
	
	//printf("lang_get: check tag:%s(len=%d)\n", tag, strlen(tag));
	for (loopCount=0; loopCount < langCounter; loopCount++)
	{
		if (!strcmp(currentLang[loopCount].tag, tag))
		{
			//printf("lang_get: tag:%s(len=%d)\n", currentLang[loopCount].tag, strlen(currentLang[loopCount].tag));
			//printf("lang_get: transString:%s(len=%d)\n", currentLang[loopCount].transString, strlen(currentLang[loopCount].transString));
			*getString = currentLang[loopCount].transString;
			return 1;
		}
	}
	return 0;
}

int lang_init()
{
	lang_free();
	FILE *fp;
	char fileName[64], *language, readBuf[1024], tagTmp[32];
	struct stat filestat;
	char *fileContent, *p, *begin, *end;
	
	int rowCounter=0, loopCount=0, resultLen=0, checkCount=0, saveCount=0;
	//MT_Get_Param("Language", language, sizeof(language));
	language = nvram_bufget(RT2860_NVRAM, "Language");
	
	memset(fileName, '\0', sizeof(fileName));
	snprintf(fileName, sizeof(fileName), "%s/lang/STRINGS_%s.txt", webPath,language);
	
	if ((fp = fopen(fileName, "r")) != NULL)
	{
			//printf("original str:\n%s\n\n\n\n", fileContent);

			while (fgets(readBuf, sizeof(readBuf), fp ) != NULL)
			{
				//printf("lang_init: str:%s\n", p);
				rowCounter++;
			}
			langCounter = rowCounter;
			fclose(fp);
			//printf("lang_init: rowCounter=%d\n", rowCounter);
			if (multiLangInit == '0')
				currentLang = (struct transLang *)malloc(rowCounter * sizeof(struct transLang));  //We don't want free this space!! please call lang_free() function.

			if (currentLang != NULL)
			{
				fp = fopen(fileName, "r");
				memset(readBuf, '\0', sizeof(readBuf));
				while (fgets(readBuf, sizeof(readBuf), fp ) != NULL)
				{
					//printf("lang_init2: str:%s\n", p);
					//begin = p;
					begin = readBuf;
					end = strchr(readBuf, ',');
					resultLen = end - begin;
					//printf("lang_init2: str:%s\n", p);
					if (end)
					{
						memset(currentLang[loopCount].tag, '\0', tagLen);
						strncpy(currentLang[loopCount].tag, readBuf, resultLen);
						
						memset(currentLang[loopCount].transString, '\0', transStringLen);
						//strcpy(currentLang[loopCount].transString, begin);
						strncpy(currentLang[loopCount].transString, end+1, strlen(end+1));
						if ((p = strchr(currentLang[loopCount].transString, '\n')) != 0)
							*p = '\0';
						
						loopCount++;
					}
					else
						printf("lang_init: division fail!\n");
					memset(readBuf, '\0', sizeof(readBuf));
				}
				fclose(fp);
				if (loopCount > 0)
					multiLangInit = '1';
				//printf("lang_init: loopCount=%d\n", loopCount);
				
				if (loopCount != rowCounter)
				{
					printf("Please check mem content and original string file, that seems lose some row content!\n");
					printf("The file row counter is %d and mem content counter is %d\n", rowCounter, loopCount);
				}
			}
			else
			{
				printf("lang_init: Error, allocate memory to multi-lang fail!\n");
				return 0;
			}

		printf("lang_init: Initial multi-language success! Current language is %s.\n", language);
		
		//For check array content.
		/*for (loopCount=0; loopCount < langCounter; loopCount++)
		{
			printf("lang_init: tag[%d]:%s(len=%d)\n", loopCount, currentLang[loopCount].tag, strlen(currentLang[loopCount].tag));
			printf("lang_init: transString[%d]:%s(len=%d)\n", loopCount, currentLang[loopCount].transString, strlen(currentLang[loopCount].transString));
		}*/
		return 1;
	}
	else
	{
		printf("translateString: %s file open fail!\n", fileName);
		
		//clear all tag and value, we don't want use free to release it.
		if (multiLangInit == '1')
		{
			for (loopCount=0; loopCount < langCounter; loopCount++)
			{
				memset(currentLang[loopCount].tag, '\0', strlen(currentLang[loopCount].tag));
				memset(currentLang[loopCount].transString, '\0', strlen(currentLang[loopCount].transString));
			}
		}
		//websWrite(wp, T("%s"), strBuf);
		return 0;
	}
}

//dynamicHTML
//We don't need parse multi-lang tag, because we must know current tag name and input them.
/*void dynamicHTML(webs_t wp, char *multiLangTag, char *outputContent)
{
	char *transString;
	if (lang_get(multiLangTag, &transString))
		websWrite(wp, T("%s"), transString);
	else
		websWrite(wp, T("%s"), outputContent);
	//websWrite(wp, T(%s), "Connected");
}*/
void dynamicHTML(webs_t wp, char *outputContent)
{
	//websWrite(wp, T("%s"), outputContent);
	char_t *nextp, *multiLangParser;
	char_t	*last, *lastMultiLangParser;
	char startTagStart[8], startTagEnd[8], endTag[18];
	int rc=0;
	
	last = outputContent;
	nextp = gstrstr(last, T("<!--#tr"));
	
	strcpy(startTagStart, "<!--#tr");
	strcpy(startTagEnd, "-->");
	strcpy(endTag, "<!--#endtr-->");
	
	startMultiLangParser(wp, &nextp, &last, &rc, startTagStart, startTagEnd, endTag, "dynamicHTML");
}

static int getDynamicHTMLRequest(int eid, webs_t wp, int argc, char_t **argv)
{
	//int type;
	char *field, *type;
	char_t *nextp, *multiLangParser;
	char_t	*last, *lastMultiLangParser, *transString;
	char startTagStart[8], startTagEnd[8], endTag[18];
	int rc=0;
	
	if (ejArgs(argc, argv, T("%s %s"), &type, &field) < 2) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	else
	{
		//printf("getDynamicHTMLRequest: type:%s, field:%s\n", type, field);
		//lang_get(type, &transString);
		if (lang_get(type, &transString))
		{
			*(transString + strlen(transString) - 1) = '\0'; //sometime we will get \n in the gui that return from there.
			//printf("getDynamicHTMLRequest: transString:%s\n", transString);
			return websWrite(wp, T("\"%s\""), transString);
		}
		else
			return websWrite(wp, T("\"%s\""), field);
		//return websWrite(wp, T("\"JJJJJ\""));
	}
}

static int getMultiLang(int eid, webs_t wp, int argc, char_t **argv)
{
	//int type;
	char *outputContent;
	char_t *nextp, *multiLangParser;
	char_t	*last, *lastMultiLangParser, *transString;
	char startTagStart[8], startTagEnd[8], endTag[18];
	int rc=0;
	
	if (ejArgs(argc, argv, T("%s"), &outputContent) < 1) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	else
	{
		websWrite(wp, T("\""));
		last = outputContent;
		rc = 0;
		//Jacky.Yang 5-Feb-2009, for multi-lang
		int optionMode=0, stopLoop=0;
		char startTagStart[8], startTagEnd[8], endTagStart[8], endTag[18];
		//printf("websAspRequest: last:%s\n", last);
		//while (rc == 0 && *last && ((nextp = gstrstr(last, T("<%"))) != NULL)) {
		while (rc == 0 && *last && !stopLoop) {
			lastMultiLangParser = last;
			multiLangParser = NULL;
			//printf("websAspRequest: lastOrigParser=%x, lastMultiLangParser=%x\n", lastOrigParser, lastMultiLangParser);
			multiLangParser = gstrstr(lastMultiLangParser, T("<!--#tr"));
			//printf("websAspRequest: lastOrigParser=%x, lastMultiLangParser=%x\n", lastOrigParser, lastMultiLangParser);
			if (multiLangParser != NULL) {
				//printf("websAspRequest: origParser=%x, multiLangParser=%x\n", origParser, multiLangParser);
				if (multiLangParser != 0)
				{
					//printf("websAspRequest: 2\n");
					nextp = multiLangParser;
					last = lastMultiLangParser;
					strcpy(startTagStart, "<!--#tr");
					strcpy(startTagEnd, "-->");
					strcpy(endTag, "<!--#endtr-->");
				}
				websWriteBlock(wp, last, (nextp - last));
				//printf("websAspRequest: lpath=%x, ejid=%d, nextp=%x, last=%x, rc=%d\n", lpath, ejid, nextp, last, rc);
				//printf("websAspRequest: 5\n");
				startMultiLangParser(wp, &nextp, &last, &rc, startTagStart, startTagEnd, endTag, "fromGUI");
			}
			else
				stopLoop  = 1;
		}
		/*
 		 *	Output any trailing HTML page text
		 */
		if (last && *last && rc == 0) {
			websWriteBlock(wp, last, gstrlen(last));
		}
	}
	websWrite(wp, T("\";"));
	return 1;
}

void dynamicString(webs_t wp, char *outputContent)
{
	char_t			*last, *nextp, *lastMultiLangParser, *multiLangParser;
	int				rc;
	
	last = outputContent;
	rc = 0;
	//Jacky.Yang 5-Feb-2009, for multi-lang
	int optionMode=0, stopLoop=0;
	char startTagStart[8], startTagEnd[8], endTagStart[8], endTag[18];
	//printf("websAspRequest: last:%s\n", last);
	//while (rc == 0 && *last && ((nextp = gstrstr(last, T("<%"))) != NULL)) {
	while (rc == 0 && *last && !stopLoop) {
		lastMultiLangParser = last;
		multiLangParser = NULL;
		//printf("websAspRequest: lastOrigParser=%x, lastMultiLangParser=%x\n", lastOrigParser, lastMultiLangParser);
		multiLangParser = gstrstr(lastMultiLangParser, T("<!--#tr"));
		//printf("websAspRequest: lastOrigParser=%x, lastMultiLangParser=%x\n", lastOrigParser, lastMultiLangParser);
		if (multiLangParser != NULL) {
			//printf("websAspRequest: origParser=%x, multiLangParser=%x\n", origParser, multiLangParser);
			if (multiLangParser != 0)
			{
				//printf("websAspRequest: 2\n");
				nextp = multiLangParser;
				last = lastMultiLangParser;
				strcpy(startTagStart, "<!--#tr");
				strcpy(startTagEnd, "-->");
				strcpy(endTag, "<!--#endtr-->");
			}
			websWriteBlock(wp, last, (nextp - last));
			//printf("websAspRequest: lpath=%x, ejid=%d, nextp=%x, last=%x, rc=%d\n", lpath, ejid, nextp, last, rc);
			//printf("websAspRequest: 5\n");
			startMultiLangParser(wp, &nextp, &last, &rc, startTagStart, startTagEnd, endTag, "fullHTML");
		}
		else
			stopLoop  = 1;
	}
/*
 *	Output any trailing HTML page text
 */
	if (last && *last && rc == 0) {
		websWriteBlock(wp, last, gstrlen(last));
	}
}

/*********************************************************************
 * Web Related Utilities
 */

void formDefineUtilities(void)
{
	websAspDefine(T("getCfgGeneral"), getCfgGeneral);
	websAspDefine(T("getCfgNthGeneral"), getCfgNthGeneral);
	websAspDefine(T("getCfgZero"), getCfgZero);
	websAspDefine(T("getCfgNthZero"), getCfgNthZero);
	websAspDefine(T("getCfg2General"), getCfg2General);
	websAspDefine(T("getCfg2NthGeneral"), getCfg2NthGeneral);
	websAspDefine(T("getCfg2Zero"), getCfg2Zero);
	websAspDefine(T("getCfg2NthZero"), getCfg2NthZero);
	websAspDefine(T("getCfg3General"), getCfg3General);
	websAspDefine(T("getCfg3Zero"), getCfg3Zero);
	websAspDefine(T("getDpbSta"), getDpbSta);
	websAspDefine(T("getLangBuilt"), getLangBuilt);
	websAspDefine(T("getMiiInicBuilt"), getMiiInicBuilt);
	websAspDefine(T("getPlatform"), getPlatform);
	websAspDefine(T("getStationBuilt"), getStationBuilt);
	websAspDefine(T("getSysBuildTime"), getSysBuildTime);
	websAspDefine(T("getSdkVersion"), getSdkVersion);
	websAspDefine(T("getProjectFirmwareVersion"), getProjectFirmwareVersion);
	websAspDefine(T("getProjectName"), getProjectName);
	websAspDefine(T("getSysTime"), getSysTime);
	websAspDefine(T("getSysUptime"), getSysUptime);
	websAspDefine(T("getPortStatus"), getPortStatus);
	websAspDefine(T("isOnePortOnly"), isOnePortOnly);
	websFormDefine(T("forceMemUpgrade"), forceMemUpgrade);
	websFormDefine(T("setOpMode"), setOpMode);
	websFormDefine(T("loadDefault"), loadDefault);
	websAspDefine(T("getDynamicHTMLRequest"), getDynamicHTMLRequest);
	websAspDefine(T("getMultiLang"), getMultiLang);
}


//Tom.Hung 2009-12-30, check device need reboot or not.
static int checkRebootStatus(int eid, webs_t wp, int argc, char_t **argv)
{
	char nbStatus[8] = {0};
	FILE *nbStatusFile;

	nbStatusFile = fopen("/var/needReboot", "r");
	if(nbStatusFile){
		fgets(nbStatus, 8, nbStatusFile);
		fclose(nbStatusFile);
		if(atol(nbStatus)==0)
			return websWrite(wp, T("0"));
		else
			return websWrite(wp, T("1"));
	}else{
		//printf("Open \"/var/needReboot\" fail");
	}
	return websWrite(wp, T(""));
}
//Tom.Hung 2009-12-30

