/*
 *	firewall.c -- Firewall Settings 
 *
 *	Copyright (c) Ralink Technology Corporation All Rights Reserved.
 *
 *	$Id: firewall.c,v 1.11.2.4 2008-01-21 06:59:08 yy Exp $
 */

/*
 *	if  WAN or LAN ip changed, we must restart firewall.
 */
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <ctype.h>
#include "nvram.h"
#include "utils.h"
#include "webs.h"
#include "firewall.h"
#include "internet.h"

#define DD printf("---> %d\n", __LINE__);
#define ruleLength	4096

int getGoAHeadServerPort(void);
void PortTriggerRunBySchedule(char *scheduleAction, char *ruleName); //Add for run port trigger with scheduling - U-Media Ricky Cao on Apr. 11 2008
void checkWebServicePort(char *port);
void AccessControlRun(char *iptablesRuleName, char *ruleName, char *ruleEnable);
void PortForwardAllClear(char *tagName, char *iptablesRuleTag);
void iptablesSinglePortForwardRun(char *iptablesRuleName, char *ruleName, char *ruleEnable);
void iptablesPortRangeForwardRun(char *iptablesRuleName, char *ruleName, char *ruleEnable);
void iptablesInboundFilterRun(char *iptablesRuleName, char *ruleName, char *protocol, char *portRange);

int isMacValid(char *str)
{
	int i, len = strlen(str);
	if(len != 17)
		return 0;

	for(i=0; i<5; i++){
		if( (!isxdigit( str[i*3])) || (!isxdigit( str[i*3+1])) || (str[i*3+2] != ':') )
			return 0;
	}
	return (isxdigit(str[15]) && isxdigit(str[16])) ? 1: 0;
}

static int isIpValid(char *str)
{
	struct in_addr addr;	// for examination
	if( (! strcmp(T("any"), str)) || (! strcmp(T("any/0"), str)))
		return 1;

	if(! (inet_aton(str, &addr))){
		//printf("isIpValid(): %s is not a valid IP address.\n", str);
		return 0;
	}
	return 1;
}

static int isNumOnly(char *str){
	int i, len = strlen(str);
	for(i=0; i<len; i++){
		if((str[i] >= '0' && str[i] <= '9'))
			continue;
		return 0;
	}
	return 1;
}

static int isAllNumAndSlash(char *str){
	int i, len = strlen(str);
	printf("str = %s\n", str);
	for(i=0; i<len; i++){
		if( (str[i] >= '0' && str[i] <= '9') || str[i] == '.' || str[i] == '/' )
			continue;
		return 0;
	}
	return 1;
}

static int isOnlyOneSlash(char *str)
{
	int i, count=0;
	int len = strlen(str);
	for(i=0; i<len; i++)
		if( str[i] == '/')
			count++;
	return count <= 1 ? 1 : 0;
}

int isIpNetmaskValid(char *s)
{
	char str[32];
	char *slash;
	struct in_addr addr;    // for examination

	if(!s || !strlen(s)){
		return 0;
	}

	strncpy(str, s, sizeof(str));

    if( (!strcmp("any", str)) || (!strcmp("any/0", str)))
        return 1;

	if (!isAllNumAndSlash(str)){
		return 0;
	}

	if(!isOnlyOneSlash(str)){
		return 0;
	}

	slash = strchr(str, '/');
	if(slash){
		int mask;

		*slash = '\0';
		slash++;
		if(!strlen(slash)){
			return 0;
		}

		if(!isNumOnly(slash)){
			return 0;
		}

		mask = atoi(slash);
		if(mask < 0 || mask > 32){
			return 0;
		}
	}

	if(! (inet_aton(str, &addr))){
        printf("isIpNetmaskValid(): %s is not a valid IP address.\n", str);
        return 0;
    }
    return 1;
}

static int getDMZEnableASP(int eid, webs_t wp, int argc, char_t **argv)
{
	int type, value;
	char *dmze = nvram_bufget(RT2860_NVRAM, "DMZEnable");
	if(dmze)
		value = atoi(dmze);
	else
		value = 0;

	if( ejArgs(argc, argv, T("%d"), &type) == 1){
		if(type == value)
			websWrite(wp, T("selected"));
		else
			websWrite(wp, T(" "));
		return 0;
	}
	return -1;        
}

static int  getPortForwardEnableASP(int eid, webs_t wp, int argc, char_t **argv)
{
	int type, value;
	char *pfe = nvram_bufget(RT2860_NVRAM, "PortForwardEnable");

	if(pfe)
		value = atoi(pfe);
	else
		value = 0;

	if( ejArgs(argc, argv, T("%d"), &type) == 1){
		if(type == value)
			websWrite(wp, T("selected"));
		else
			websWrite(wp, T(" "));
		return 0;
	}
	return -1;
}

static int  getIPPortFilterEnableASP(int eid, webs_t wp, int argc, char_t **argv)
{
	int type, value;
	char *pfe = nvram_bufget(RT2860_NVRAM, "IPPortFilterEnable");

	if(pfe)
		value = atoi(pfe);
	else
		value = 0;

	if( ejArgs(argc, argv, T("%d"), &type) == 1){
		if(type == value)
			websWrite(wp, T("selected"));
		else
			websWrite(wp, T(" "));
		return 0;
	}
	return -1;
}

/*
 * hide the possible "error/warn" message when deleting a non-exist chain.
 */
static void iptablesForwardFilterClear(void)
{
	doSystem("iptables -F -t filter 1>/dev/null 2>&1");
}

/*
static void iptablesForwardFilterFlush(void)
{
	doSystem("iptables -t filter -F FORWARD  1>/dev/null 2>&1");
}
*/

static void iptablesIPPortFilterFlush(void){
	doSystem("iptables -F %s  1>/dev/null 2>&1", IPPORT_FILTER_CHAIN);
}

static void iptablesIPPortFilterClear(void){
	doSystem("iptables -D FORWARD -j %s  1>/dev/null 2>&1", IPPORT_FILTER_CHAIN);
	doSystem("iptables -F %s  1>/dev/null 2>&1", IPPORT_FILTER_CHAIN);
}

static void iptablesDMZFlush(void){
    doSystem("iptables -t nat -F %s", "PREROUTING_DMZ");
    doSystem("iptables -t nat -F %s", "POSTROUTING_DMZ");
}

static void iptablesFilterFlush(char *chainName, char *iptablesRuleName){
	doSystem("iptables -F %s_%s", chainName, iptablesRuleName);
}

static void iptablesFlush(char *iptablesParameter, char *chainName, char *ruleName){
    doSystem("iptables %s -F %s_%s", iptablesParameter, chainName, ruleName);
}

static void iptablesDMZClear(void){
	doSystem("iptables -t nat -F %s","PREROUTING_DMZ");
	doSystem("iptables -t nat -F %s","POSTROUTING_DMZ");
}

static void iptablesAllFilterClear(void)
{
	//Jacky.Yang 22-Apr-2008, clear all filter rule.
	/*doSystem("iptables -F INPUT");
	doSystem("iptables -F OUTPUT");
	doSystem("iptables -F FORWARD");*/
	//doSystem("iptables -X");
	
	iptablesForwardFilterClear();
	iptablesIPPortFilterClear();
	
	doSystem("iptables -P INPUT ACCEPT");
	doSystem("iptables -P OUTPUT ACCEPT");
	doSystem("iptables -P FORWARD ACCEPT");
}

static void iptablesAllNATClear(void)
{
	PortForwardAllClear("SinglePortForwardRules", "SPort");
	PortForwardAllClear("PortRangeForwardRules", "MPort");
	PortForwardAllClear("AccessControlPolicy", "AC");
	 
	//doSystem("iptables -t nat -F");
	
	iptablesDMZClear();
	//2010.02.01 Joan.Huang Modify ,for keep NAT rule ("iptables -t nat -F" will clear all train)
	doSystem("iptables -t nat -F POSTROUTING");
	doSystem("iptables -t nat -F PREROUTING");
	doSystem("iptables -t nat -F PREROUTING_ICMP");
}

/*int deleteNthValueMulti(int index[], int count, char *value, char delimit)
{
	char *begin, *end;
	int i=0,j=0;
	int need_check_flag=0;
	char *buf = strdup(value);

	begin = buf;

	end = strchr(begin, delimit);
	while(end){
		if(i == index[j]){
			memset(begin, 0, end - begin );
			if(index[j] == 0)
				need_check_flag = 1;
			j++;
			if(j >=count)
				break;
		}
		begin = end;

		end = strchr(begin+1, delimit);
		i++;
	}

	if(!end && index[j] == i)
		memset(begin, 0, strlen(begin));

	if(need_check_flag){
		for(i=0; i<strlen(value); i++){
			if(buf[i] == '\0')
				continue;
			if(buf[i] == ';')
				buf[i] = '\0';
			break;
		}
	}

	for(i=0, j=0; i<strlen(value); i++){
		if(buf[i] != '\0'){
			value[j++] = buf[i];
		}
	}
	value[j] = '\0';

	free(buf);
	return 0;
}*/

static int getNums(char *value, char delimit)
{
	char *pos = value;
    int count=1;
    if(!pos)
    	return 0;
	while( (pos = strchr(pos, delimit))){
		pos = pos+1;
		count++;
	}
	return count;
}

/*
 *
 */
static void makeIPPortFilterRule(char *buf, int len, char *mac_address,
char *sip_1, char *sip_2, int sprf_int, int sprt_int, 
char *dip_1, char *dip_2, int dprf_int, int dprt_int, int proto, int action)
{
		int rc = 0;
		char *pos = buf;

		switch(action){
		case ACTION_DROP:
			rc = snprintf(pos, len-rc, 
				"iptables -A %s -m state --state NEW,INVALID ", IPPORT_FILTER_CHAIN);
			break;
		case ACTION_ACCEPT:
			rc = snprintf(pos, len-rc, 
				"iptables -A %s ", IPPORT_FILTER_CHAIN);
			break;
		}
		pos = pos + rc;

		// write mac address
		if(mac_address && strlen(mac_address)){
			rc = snprintf(pos, len-rc, "-m mac --mac-source %s ", mac_address);
			pos = pos+rc;
		}

		// write source ip
		rc = snprintf(pos, len-rc, "-s %s ", sip_1);
		pos = pos+rc;
		
		// write dest ip
		rc = snprintf(pos, len-rc, "-d %s ", dip_1);
		pos = pos+rc;

		// write protocol type
		if(proto == PROTO_NONE){
			rc = snprintf(pos, len-rc, " ");
			pos = pos + rc;
		}else if(proto == PROTO_ICMP){
			rc = snprintf(pos, len-rc, "-p icmp ");
			pos = pos + rc;
		}else{
			if(proto == PROTO_TCP)
				rc = snprintf(pos, len-rc, "-p tcp ");
			else if (proto == PROTO_UDP)
				rc = snprintf(pos, len-rc, "-p udp ");
			pos = pos + rc;

			// write source port
			if(sprf_int){
				if(sprt_int)
					rc = snprintf(pos, len-rc, "--sport %d:%d ", sprf_int, sprt_int);
				else
					rc = snprintf(pos, len-rc, "--sport %d ", sprf_int);
				pos = pos+rc;
			}

			// write dest port
			if(dprf_int){
				if(dprt_int)
					rc = snprintf(pos, len-rc, "--dport %d:%d ", dprf_int, dprt_int);
				else
					rc = snprintf(pos, len-rc, "--dport %d ", dprf_int);
				pos = pos+rc;
			}
		}

		switch(action){
		case ACTION_DROP:			// 1 == ENABLE--DROP mode
			rc = snprintf(pos, len-rc, "-j DROP");
			break;
		case ACTION_ACCEPT:			// 2 == ENABLE--ACCEPT mode
			rc = snprintf(pos, len-rc, "-j ACCEPT");
			break;
		}
}

int makeSinglePortForwardRule(char *buf, int len, char *iptablesParameter, char *chainName, char *inboundFilterRuleName, char *iptablesRuleName, char *privateIP, int proto, int privatePort_int, int publicPort_int)
{
	int rc = 0;
	char *pos = buf;
	char wanIP[16]={0}, lanIP[16]={0}, protocol[16]={0};
	char iptablesInboundFilterRuleName[64]={0}, portRange[32]={0};
	char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");
	
	if(!strcmp(connType, "PPPOE") || !strcmp(connType, "L2TP")  || !strcmp(connType, "PPTP") || !strcmp(connType, "3G")){
		if (-1 == getIfIp("ppp0", wanIP)) {
			printf("Can't get ppp0 ip address!!\n");
			//exit(0);
			return -1;
		}		
	}else{
		if (-1 == getIfIp(getWanIfName(), wanIP)) {
			printf("Can't get wan ip address!!\n");
			//exit(0);
			return -1;
		}
	}

	if (-1 == getIfIp(getLanIfName(), lanIP)) {
		printf("Can't get lan ip address!!\n");
		//exit(0);
		return -1;
	}
	
	// write protocol type
	if(proto == PROTO_TCP)
		strcpy(protocol, "-p tcp ");
	else if (proto == PROTO_UDP)
		strcpy(protocol, "-p udp ");
	else if (proto == PROTO_TCP_UDP)
		strcpy(protocol, " ");

	if (!strcmp(chainName, "PREROUTING")) {		
		//inbound filter only need run once.
		//printf("jacky - programming - inbound filter:%s running it\n", inboundFilterRuleName);
		memset(portRange, '\0', sizeof(portRange));
		snprintf(portRange, sizeof(portRange), "%d", privatePort_int);
		if (strcmp(inboundFilterRuleName, "ACCEPT"))
			iptablesInboundFilterRun(iptablesRuleName, inboundFilterRuleName, protocol, portRange);
		
		rc = snprintf(pos, len-rc, "iptables %s -I %s_%s -d %s %s -j DNAT ", iptablesParameter, chainName, iptablesRuleName, wanIP, protocol);
		pos = pos + rc;
		
		if(publicPort_int != 0) {
			rc = snprintf(pos, len-rc, "--dport %d ", publicPort_int);
		}
		else {
			rc = snprintf(pos, len-rc, "--dport %d ", privatePort_int);
		}
		pos = pos + rc;

		rc = snprintf(pos, len-rc, "--to %s:%d ", privateIP, privatePort_int);
	}
	else if (!strcmp(chainName, "POSTROUTING")) {
		//rc = snprintf(pos, len-rc, "iptables %s -A %s_%s -d %s %s --dport %d ", iptablesParameter, chainName, iptablesRuleName, privateIP, protocol, privatePort_int);
		rc = snprintf(pos, len-rc, "iptables %s -I %s_%s -d %s %s --dport %d ", iptablesParameter, chainName, iptablesRuleName, privateIP, protocol, privatePort_int);
		pos = pos + rc;

		rc = snprintf(pos, len-rc, "-s %s/24 -j SNAT --to %s", lanIP, lanIP);
	}
	return 0;
}

static void makePortRangeForwardRule(char *buf, int len, char *iptablesParameter, char *chainName, char *inboundFilterRuleName, char *iptablesRuleName, char *privateIP, int proto, char *portRange)
{
	int rc = 0;
	char *pos = buf;
	//char wanIP[16], lanIP[16], protocol[16], iptablesInboundFilterRuleName[64];
	char wanIP[16]={0}, lanIP[16]={0}, protocol[16]={0};
	char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");
	
	if(!strcmp(connType, "PPPOE") || !strcmp(connType, "L2TP")  || !strcmp(connType, "PPTP") || !strcmp(connType, "3G")){
		if (-1 == getIfIp("ppp0", wanIP)) {
			printf("Can't get ppp0 ip address!!\n");
			//exit(0);
		}		
	}else{
		if (-1 == getIfIp(getWanIfName(), wanIP)) {
			printf("Can't get wan ip address!!\n");
			//exit(0);
		}
	}

	if (-1 == getIfIp(getLanIfName(), lanIP)) {
		printf("Can't get lan ip address!!\n");
		//exit(0);
	}
	
	// write protocol type
	if(proto == PROTO_TCP)
		strcpy(protocol, "-p tcp ");
	else if (proto == PROTO_UDP)
		strcpy(protocol, "-p udp ");
	else if (proto == PROTO_TCP_UDP)
		strcpy(protocol, " ");
	
	//Only need run once.
	/*printf("jacky - programming - inbound filter:%s running it\n", inboundFilterRuleName);
	iptablesInboundFilterRun(iptablesRuleName, inboundFilterRuleName, protocol, portRange);*/
	
	/*doSystem("iptables %s -N %s_%s", iptablesParameter, chainName, iptablesRuleName);
	doSystem("iptables %s -D %s -j %s_%s", iptablesParameter, chainName, chainName, iptablesRuleName);
	//Jacky.Yang 18-Apr-2008, use insert to add itpables rule for H.323 and outsourcing.
	//doSystem("iptables %s -A %s -j %s_%s", iptablesParameter, chainName, chainName,  iptablesRuleName);
	doSystem("iptables %s -I %s -j %s_%s", iptablesParameter, chainName, chainName,  iptablesRuleName);*/
	if (!strcmp(chainName, "PREROUTING")) {
		//inbound filter only need run once.
		//printf("jacky - programming - inbound filter:%s running it\n", inboundFilterRuleName);
		if (strcmp(inboundFilterRuleName, "ACCEPT"))
			iptablesInboundFilterRun(iptablesRuleName, inboundFilterRuleName, protocol, portRange);
		
		//rc = snprintf(pos, len-rc, "iptables %s -A %s_%s -d %s %s -j DNAT --dport %s ", iptablesParameter, chainName, iptablesRuleName, wanIP, protocol, portRange);
		rc = snprintf(pos, len-rc, "iptables %s -I %s_%s -d %s %s -j DNAT --dport %s ", iptablesParameter, chainName, iptablesRuleName, wanIP, protocol, portRange);
		pos = pos + rc;
		
		rc = snprintf(pos, len-rc, "--to %s ", privateIP);
	}
	else if (!strcmp(chainName, "POSTROUTING")) {
		//rc = snprintf(pos, len-rc, "iptables %s -A %s_%s -d %s %s --dport %s ", iptablesParameter, chainName, iptablesRuleName, privateIP, protocol, portRange);
		rc = snprintf(pos, len-rc, "iptables %s -I %s_%s -d %s %s --dport %s ", iptablesParameter, chainName, iptablesRuleName, privateIP, protocol, portRange);
		pos = pos + rc;

		// write remote ip
		rc = snprintf(pos, len-rc, "-s %s/24 -j SNAT --to %s", lanIP, lanIP);
	}
}

static void makeInboundFilterRule(char *buf, int len, char *iptablesParameter, char *chainName, char *wan_name, char *iptablesRuleName, char *ip_address, char *protocol, char *portRange, char *ruleAction)
{
	int rc = 0;
	char *pos = buf;
	
	//iptablesFilterFlush(chainName, iptablesRuleName);

	/*doSystem("iptables -N %s_%s", chainName, iptablesRuleName);
	doSystem("iptables %s -D %s -j %s_%s", iptablesParameter, chainName, chainName, iptablesRuleName);
	//doSystem("iptables %s -A %s -j %s_%s", iptablesParameter, chainName, chainName, iptablesRuleName);
	doSystem("iptables %s -I %s -j %s_%s", iptablesParameter, chainName, chainName, iptablesRuleName);*/
	
	rc = snprintf(pos, len-rc, "iptables -A %s_%s ", chainName, iptablesRuleName);
	pos = pos + rc;

	rc = snprintf(pos, len-rc, "%s -s %s --dport %s ", protocol, ip_address, portRange);
	pos = pos + rc;
		
	//ruleAction=1: Deny, ruleAction=2: Allow
	if (!strcmp(ruleAction, "on")) {
		rc = snprintf(pos, len-rc, "-j DROP");
	}
	else if (!strcmp(ruleAction, "0")) {
		rc = snprintf(pos, len-rc, "-j ACCEPT");
	}
}

//static void makeAccessControlRule(char *buf, int len, char *iptablesParameter, char *chainName, char *iptablesRuleName, char *scheduleRule, int proto, char *ipRange, char *Port)
static void makeAccessControlRule(char *buf, int len, char *iptablesParameter, char *chainName, char *iptablesRuleName, int proto, char *ipRange, char *Port)
{
	int rc = 0;
	char *pos = buf;
	/*doSystem("iptables %s -N %s_%s", iptablesParameter, chainName, iptablesRuleName);
	doSystem("iptables %s -D %s -j %s_%s", iptablesParameter, chainName, chainName, iptablesRuleName);
	//doSystem("iptables %s -A %s -j %s_%s", iptablesParameter, chainName, chainName, iptablesRuleName);
	doSystem("iptables %s -I %s -j %s_%s", iptablesParameter, chainName, chainName, iptablesRuleName);*/

	//rc = snprintf(pos, len-rc, "iptables %s -A %s_%s ", iptablesParameter, chainName, iptablesRuleName);
	rc = snprintf(pos, len-rc, "iptables %s -I %s_%s ", iptablesParameter, chainName, iptablesRuleName);
	pos = pos + rc;

	if(proto == PROTO_TCP) {
		//rc = snprintf(pos, len-rc, "-p tcp -m iprange --dst-range %s --dport %s ", ipRange, Port);
		rc = snprintf(pos, len-rc, "-p tcp -m iprange --src-range %s --dport %s -j DROP", ipRange, Port);
	}
	else if (proto == PROTO_UDP) {
		//rc = snprintf(pos, len-rc, "-p udp -m iprange --dst-range %s --dport %s ", ipRange, Port);
		rc = snprintf(pos, len-rc, "-p udp -m iprange --src-range %s --dport %s -j DROP", ipRange, Port);
	}
	else {
		printf("User doesn't select any protocol!\n");
	}
	pos = pos + rc;

	//ruleAction=1: Deny, ruleAction=2: Allow
	/*if (!strcmp(scheduleRule, "Always")) {
		rc = snprintf(pos, len-rc, "-j DROP");
	}
	else if (!strcmp(scheduleRule, "Never")) {
		rc = snprintf(pos, len-rc, "-j ACCEPT");
	}*/
}

void iptablesRemoteManagementRun(void)
{
	char wanIP[16]={0}, lanIP[16]={0}, ethWanIP[16]={0};
	char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");
	char *dmz_enable = nvram_bufget(RT2860_NVRAM, "DMZEnable");
	char *rmE = nvram_bufget(RT2860_NVRAM, "RemoteManagement");
	char *opmode = nvram_bufget(RT2860_NVRAM, "OperationMode");
	char *portNum = nvram_bufget(RT2860_NVRAM, "RemotePort");
	char *lan_ip = nvram_bufget(RT2860_NVRAM, "lan_ipaddr");

	if(!strcmp(connType, "PPPOE") || !strcmp(connType, "L2TP")  || !strcmp(connType, "PPTP") || !strcmp(connType, "3G")){
		if (-1 == getIfIp("ppp0", wanIP)) {
			printf("Can't get ppp0 ip address!!\n");
			//exit(0);
			return -1;
		}		
	}else{
		if (-1 == getIfIp(getWanIfName(), wanIP)) {
			printf("Can't get wan ip address!!\n");
			//exit(0);
			return -1;
		}
	}
	
	if (-1 == getIfIp(getLanIfName(), lanIP)) {
		printf("Can't get lan ip address!!\n");
		//exit(0);
		return -1;
	}
	
	// TODO: make a new chain instead of flushing the INPUT chain
	//doSystem("iptables -t nat -F remotePortForward");
	doSystem("iptables -t nat -F PREROUTING_Remote");
	doSystem("iptables -t nat -F POSTROUTING_Remote");
    
    //Jacky.Yang 24-Oct-2008, The web server port 80, if dmz has enabled, we can't drop port 80 in the firewall.
    if(!atoi(dmz_enable)) {
		//doSystem("iptables -t nat -I remotePortForward -j DROP -i %s -p tcp --dport %d", getWanIfName(), getGoAHeadServerPort());
		//doSystem("iptables -t nat -I PREROUTING_Remote -j DROP -p tcp -d %s --to %s:%d",  wanIP, lanIP, getGoAHeadServerPort());
		//doSystem("iptables -t nat -I POSTROUTING_Remote -j DROP -p tcp -d %s -s %s/24 --to %s:%d", lanIP, lanIP, lanIP, getGoAHeadServerPort());
	}
	if(atoi(rmE) == 1){
		//doSystem("iptables -t nat -I remotePortForward -j DNAT -i %s -p tcp --dport %s --to %s:%d", getWanIfName(), portNum, lan_ip, getGoAHeadServerPort());
		doSystem("iptables -t nat -I PREROUTING_Remote -j DNAT -p tcp -d %s --dport %s --to %s:%d",  wanIP, portNum, lanIP, getGoAHeadServerPort());
		doSystem("iptables -t nat -I POSTROUTING_Remote -j SNAT -p tcp -d %s --dport %s -s %s/24 --to %s:%d", lanIP, portNum, lanIP, lanIP, getGoAHeadServerPort());
	}

	//2009.11.13 Joan.Huang ,Fix remote management always enable when WAN mode is PPP(pppop,pptp,l2tp).
	if(!strcmp(connType, "PPPOE") || !strcmp(connType, "L2TP")  || !strcmp(connType, "PPTP")){
		//doSystem("iptables -t nat -I PREROUTING_Remote -j DROP -p tcp -d %s --dport 80",  ethWanIP);
		doSystem("iptables -t nat -I PREROUTING_Remote -j DROP -p tcp -i eth2.2 --dport 80");
	}
	
	if(!opmode)
		return;

	// "Gateway mode" only
	if(strcmp(opmode , "1"))
		return;

	if(rmE && atoi(rmE) == 1)
		return;

	/*doSystem("iptables -A INPUT -i %s -m state --state RELATED,ESTABLISHED -j ACCEPT", getWanIfName());
	doSystem("iptables -A INPUT -i %s -m state -p tcp --dport 80 --state NEW,INVALID -j DROP", getWanIfName());*/
	/*doSystem("iptables -N remoteManagement");
	doSystem("iptables -D INPUT -j remoteManagement");
	doSystem("iptables -I INPUT -j remoteManagement");
	
	doSystem("iptables -A remoteManagement -i %s -m state --state RELATED,ESTABLISHED -j ACCEPT", getWanIfName());
	//doSystem("iptables -A remoteManagement -i %s -m state -p tcp --dport 80 --state NEW,INVALID -j DROP", getWanIfName());
	doSystem("iptables -A remoteManagement -i %s -m state -p tcp --dport %d --state NEW,INVALID -j DROP", getWanIfName(), getGoAHeadServerPort());
	//doSystem("iptables -A remoteManagement -i %s -m state -p tcp --dport %s --state NEW,INVALID -j DROP", getWanIfName(), portNum);
	*/
	return;
}

static void iptablesDMZRun(void)
{
	char wanIP[16]={0}, lanIP[16]={0};
	char cmd[1024]={0}, *ip_address;
	char *dmz_enable = nvram_bufget(RT2860_NVRAM, "DMZEnable");
	char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");
	
	if(!dmz_enable){
		printf("Warning: can't find \"DMZEnable\" in flash\n");
		return;
	}
	if(!atoi(dmz_enable))
		return;

	ip_address = nvram_bufget(RT2860_NVRAM, "DMZIPAddress");
	if(!ip_address){
		printf("Warning: can't find \"DMZIPAddress\" in flash\n");
		return;
	}
	
	if(!strcmp(connType, "PPPOE") || !strcmp(connType, "L2TP")  || !strcmp(connType, "PPTP") || !strcmp(connType, "3G")){
		if (-1 == getIfIp("ppp0", wanIP)) {
			printf("Can't get ppp0 ip address!!\n");
			//exit(0);
			return -1;
		}		
	}else{
		if (-1 == getIfIp(getWanIfName(), wanIP)) {
			printf("Can't get wan ip address!!\n");
			//exit(0);
			return -1;
		}
	}
	
	if (-1 == getIfIp(getLanIfName(), lanIP)) {
		printf("Can't get lan ip address!!\n");
		//exit(0);
		return -1;
	}

	snprintf(cmd, sizeof(cmd), "iptables -t nat -I %s -d %s -j DNAT --to %s", "PREROUTING_DMZ", wanIP, ip_address);
	doSystem(cmd);
	/** 2009.12.23 Joan.Huang modify for fix LAN PC can not access DMZ server and upnp can not work when DMZ enable **/
	//iptables -t nat -I POSTROUTING_DMZ -d 192.168.1.252 -s 192.168.1.1/24 -j SNAT --to 192.168.2.248
	snprintf(cmd, sizeof(cmd), "iptables -t nat -I %s -d %s -s %s/24 -j SNAT --to %s", "POSTROUTING_DMZ", ip_address, lanIP, wanIP);
	doSystem(cmd);
	/** end by Joan.Huang **/
	/*
	//snprintf(cmd, sizeof(cmd), "iptables -t nat -A %s -j DNAT -i %s -p udp --dport ! %d --to %s", "PREROUTING_DMZ", getWanIfName(), getGoAHeadServerPort(), ip_address);
	snprintf(cmd, sizeof(cmd), "iptables -t nat -I %s -d %s -p udp  -j DNAT --to %s", "PREROUTING_DMZ", wanIP, ip_address);
	doSystem(cmd);
	snprintf(cmd, sizeof(cmd), "iptables -t nat -I %s -d %s -p udp -s %s/24 -j SNAT --to %s", "POSTROUTING_DMZ", lanIP, lanIP, ip_address);
	doSystem(cmd);
	//snprintf(cmd, sizeof(cmd), "iptables -t nat -A %s -j DNAT -i %s -p tcp --dport ! %d --to %s", "PREROUTING_DMZ", getWanIfName(), getGoAHeadServerPort(), ip_address);
	
	snprintf(cmd, sizeof(cmd), "iptables -t nat -I %s -d %s -p tcp -j DNAT --to %s", "PREROUTING_DMZ", wanIP, ip_address);
	doSystem(cmd);
	snprintf(cmd, sizeof(cmd), "iptables -t nat -I %s  -d %s -p tcp -s %s/24 -j SNAT --to %s", "POSTROUTING_DMZ", lanIP, lanIP, ip_address);
	doSystem(cmd);
	
	snprintf(cmd, sizeof(cmd), "iptables -t nat -I %s -d %s -p icmp -j DNAT --to %s", "PREROUTING_DMZ", wanIP, ip_address);
	doSystem(cmd);
	snprintf(cmd, sizeof(cmd), "iptables -t nat -I %s -d %s -p icmp -s %s/24 -j SNAT --to %s", "POSTROUTING_DMZ", lanIP, lanIP, ip_address);
	doSystem(cmd);*/
	return;
}

static void iptablesIPPortFilterRun(void)
{
	int i=0;
	char rec[256]={0};
	char cmd[1024]={0};
	int sprf_int, sprt_int, proto, action;
	int dprf_int, dprt_int;
	char sprf[8]={0}, sprt[8]={0}, protocol[8]={0};
	char dprf[8]={0}, dprt[8]={0};
	char mac_address[32]={0};
	char sip_1[32]={0}, sip_2[32]={0}, action_str[4]={0};
	char dip_1[32]={0}, dip_2[32]={0};
    char *firewall_enable, *default_policy, *rule;
    int mode;

    firewall_enable = nvram_bufget(RT2860_NVRAM, "IPPortFilterEnable");
    if(!firewall_enable){
        printf("Warning: can't find \"IPPortFilterEnable\" in flash.\n");
        return;
    }
    mode = atoi(firewall_enable);
    if(!mode)
		return;

	rule = nvram_bufget(RT2860_NVRAM, "IPPortFilterRules");
	if(!rule){
		printf("Warning: can't find \"IPPortFilterRules\" in flash.\n");
		return;
	}

	default_policy = nvram_bufget(RT2860_NVRAM, "DefaultFirewallPolicy");
	// add the default policy to the end of FORWARD chain
	if(!default_policy)
		default_policy = "0";

	if(atoi(default_policy) == 1){
		//the default policy is drop
		doSystem("iptables -t filter -A %s -m state --state RELATED,ESTABLISHED -j ACCEPT", IPPORT_FILTER_CHAIN);
	}

	while( (getNthValueSafe(i++, rule, ';', rec, 256) != -1) ){
        // get sip 1
        if((getNthValueSafe(0, rec, ',', sip_1, 32) == -1)){
			continue;
		}
		if(!isIpNetmaskValid(sip_1)){
			continue;
		}

		// we dont support ip range yet.
        // get sip 2
        //if((getNthValueSafe(1, rec, ',', sip_2, 32) == -1))
        //	continue;
		//if(!isIpValid(sip_2))
		//	continue;

		// get source port range "from"
		if((getNthValueSafe(2, rec, ',', sprf, 8) == -1)){
			continue;
		}
		if( (sprf_int = atoi(sprf)) > 65535)
			continue;
		// get dest port range "to"
		if((getNthValueSafe(3, rec, ',', sprt, 8) == -1)){
			continue;
		}
		if( (sprt_int = atoi(sprt)) > 65535)
			continue;

		// Destination Part
        // get dip 1
		if((getNthValueSafe(4, rec, ',', dip_1, 32) == -1)){
			continue;
		}
		if(!isIpNetmaskValid(dip_1)){
			continue;
		}
		// we dont support ip range yet
        // get sip 2
        //if((getNthValueSafe(5, rec, ',', dip_2, 32) == -1))
        //    continue;
        //if(!isIpValid(dip_2))
        //    continue;

		// get source port range "from"
		if((getNthValueSafe(6, rec, ',', dprf, 8) == -1)){
			continue;
		}
		if( (dprf_int = atoi(dprf)) > 65535)
			continue;

		// get dest port range "to"
		if((getNthValueSafe(7, rec, ',', dprt, 8) == -1)){
			continue;
		}
		if( (dprt_int = atoi(dprt)) > 65535)
			continue;


		// get protocol
		if((getNthValueSafe(8, rec, ',', protocol, 8) == -1))
			continue;
		proto = atoi(protocol);

		// get action
        if((getNthValueSafe(9, rec, ',', action_str, 4) == -1)){
            continue;
        }
        action = atoi(action_str);

        // getNthValueSafe(10) is "comment".

        // get mac address
        if((getNthValueSafe(11, rec, ',', mac_address, 32) == -1))
            continue;
		if(strlen(mac_address)){
	        if(!isMacValid(mac_address))
	        	continue;
		}

        //TODO:
		// supposed to do validation here but  we didn't do it because code size.
/*
# iptables example
# iptables -t nat -A POSTROUTING -o eth0  -s 10.10.10.0/24 -j MASQUERADE
# iptables -A FORWARD -m physdev --physdev-in ra0 --physdev-out eth2 -m state --state ESTABLISHED,RELATED -j ACCEPT
# iptables -A FORWARD -m physdev --physdev-in eth0 --physdev-out eth2 -j DROP
# iptables -A FORWARD -i eth0 -o eth2 -j DROP
# iptables -A FORWARD -p tcp --dport 139 -j DROP
# iptables -A FORWARD -i eth0 -o eth2 -m state --state NEW,INVALID -p tcp --dport 80 -j DROP
*/
		makeIPPortFilterRule(cmd, sizeof(cmd), mac_address, sip_1, sip_2, sprf_int, sprt_int, dip_1, dip_2, dprf_int, dprt_int, proto, action);
		doSystem(cmd);
	}


	switch(atoi(default_policy)){
	case 0:
		doSystem("iptables -t filter -A %s -j ACCEPT", IPPORT_FILTER_CHAIN);
		break;
	case 1:
		doSystem("iptables -t filter -A %s -j DROP", IPPORT_FILTER_CHAIN);
		break;
	}

}

static void PortForwardAllClear(char *tagName, char *iptablesRuleTag)
{
	char iptablesRuleName[64]={0};

	strcpy(iptablesRuleName, iptablesRuleTag);
	iptablesFilterFlush("INPUT", iptablesRuleName);
	iptablesFilterFlush("FORWARD", iptablesRuleName);
	iptablesFlush("-t nat", "PREROUTING", iptablesRuleName);
	iptablesFlush("-t nat", "POSTROUTING", iptablesRuleName);
}

void PortForwardAllRun(char *tagName, char *iptablesRuleTag)
{
	int i=0;
	char ruleEnable[4]={0}, ruleName[32]={0};
	char rec[128]={0}, iptablesRuleName[64]={0}, schedule[32]={0};
	//char *rules = nvram_bufget(RT2860_NVRAM, "SinglePortForwardRules");
	char *ruleEnableOutput;
	char *rules = nvram_bufget(RT2860_NVRAM, tagName);
	
	if (!strcmp(tagName, "AccessControlPolicy"))
		ruleEnableOutput = nvram_bufget(RT2860_NVRAM, "AccessControlEnable");

	if(!rules)
		return 0;
	if(!strlen(rules))
		return 0;

	while(getNthValueSafe(i++, rules, ';', rec, 2048) != -1 ){
		if((getNthValueSafe(0, rec, ',', ruleName, sizeof(ruleName)) == -1)){
			continue;
		}
		
		if((getNthValueSafe(1, rec, ',', ruleEnable, sizeof(ruleEnable)) == -1)){
			continue;
		}
		getNthValueSafe(2, rec, ',', schedule, sizeof(schedule));
		
		//strcpy(iptablesRuleName, "SPort_");
		strcpy(iptablesRuleName, iptablesRuleTag);
		//strcat(iptablesRuleName, ruleName);
		//iptablesFilterFlush("INPUT", iptablesRuleName);
		//iptablesFilterFlush("FORWARD", iptablesRuleName);
		//iptablesFlush("-t nat", "PREROUTING", iptablesRuleName);
		//iptablesFlush("-t nat", "POSTROUTING", iptablesRuleName);
		
		/*if (!strcmp(tagName, "InboundFilterRules"))
			iptablesInboundFilterRun(iptablesRuleName, ruleName, ruleEnable);*/
		
		if (!strcmp(ruleEnable, "on")) {
			if (!strcmp(tagName, "SinglePortForwardRules")) {
				if (!strcmp(schedule, "Always"))
					iptablesSinglePortForwardRun(iptablesRuleName, ruleName, ruleEnable);
				/*else
					addScheduleToCron("SinglePortForwardRules", ruleName, "SPort_", schedule);*/
			}
			else if (!strcmp(tagName, "PortRangeForwardRules")) {
				if (!strcmp(schedule, "Always"))
					iptablesPortRangeForwardRun(iptablesRuleName, ruleName, ruleEnable);
				/*else
					addScheduleToCron("PortRangeForwardRules", ruleName, "MPort_", schedule);*/
			}
			else if (!strcmp(tagName, "AccessControlPolicy") && (atoi(ruleEnableOutput) == 1)) {
				if (!strcmp(schedule, "Always"))
					AccessControlRun(iptablesRuleName, ruleName, ruleEnable);
				/*else
					addScheduleToCron("AccessControlPolicy", ruleName, "AC_", schedule);*/
			}
		}
	}
}

static void iptablesSinglePortForwardRun(char *iptablesRuleName, char *ruleName, char *ruleEnable)
{
	char rec[256]={0}, cmd[1024]={0}, wan_name[16]={0}, temp[64]={0};

	int privatePort_int, publicPort_int, proto;
	char ruleEnableCheck[4]={0}, ruleNameCheck[32]={0}, ipaddress[32]={0}, privatePort[8]={0}, publicPort[8]={0}, protocol[2]={0};
	char schedule[32]={0}, inboundFilter[32]={0};
    	char *rules, *pt;

    if(!ruleEnable){
        printf("Warning: can't find \"PortForwardEnable\" in flash\n");
        return;
    }

    if(!strcmp(ruleEnable, "on")){
        rules = nvram_bufget(RT2860_NVRAM, "SinglePortForwardRules");
        //printf("jacky - rules:%s\n", rules);
        if(!rules){
            printf("Warning: can't find \"SinglePortForwardRules\" in flash\n");
            return ;
        }

	//search ruleName start point
	sprintf(temp,";%s,",ruleName);   //for some short name ,example ruleName=1
	pt = strstr(rules, temp);
	if(pt != NULL)	{
		memset(temp, '\0', sizeof(temp));	
		sprintf(temp,"%s,",ruleName);
		pt = strstr(pt, temp);
	}else{
		memset(temp, '\0', sizeof(temp));	
		sprintf(temp,"%s,",ruleName);
		pt = strstr(rules, temp);
	}
        //printf("jacky - check pt:%s\n", pt);
        if (getNthValueSafe(0, pt, ';', rec, sizeof(rec)) != -1)
        {
        	//printf("jacky - check rec:%s\n", rec);
        	// Don't need to get Rule Enable
			// get Rule Name
			if((getNthValueSafe(0, rec, ',', ruleNameCheck, sizeof(ruleNameCheck)) == -1)){
				//continue;
			}
			//printf("jacky - check ruleNameCheck(%d):%s\n", sizeof(ruleNameCheck), ruleNameCheck);
			
			if((getNthValueSafe(1, rec, ',', ruleEnableCheck, sizeof(ruleEnableCheck)) == -1)){
				//continue;
			}
			//printf("jacky - check ruleEnableCheck(%d):%s\n", sizeof(ruleEnableCheck), ruleEnableCheck);
			
			if((getNthValueSafe(2, rec, ',', schedule, sizeof(schedule)) == -1)){
				//continue;
			}
			//printf("jacky - check schedule(%d):%s\n", sizeof(schedule), schedule);
			
			if((getNthValueSafe(3, rec, ',', inboundFilter, sizeof(inboundFilter)) == -1)){
				//continue;
			}
			//printf("jacky - check inboundFilter(%d):%s\n", sizeof(inboundFilter), inboundFilter);
			
			// get ip address
			if((getNthValueSafe(4, rec, ',', ipaddress, sizeof(ipaddress)) == -1)){
				//continue;
			}
			//printf("jacky - check ipaddress(%d):%s\n", sizeof(ipaddress), ipaddress);
			/*if(!isIpValid(ipaddress)){
				//continue;
			}*/
                	
			// get private port
			if((getNthValueSafe(5, rec, ',', privatePort, sizeof(privatePort)) == -1)){
				//continue;
			}
			//printf("jacky - check privatePort(%d):%s\n", sizeof(privatePort), privatePort);
			
			if( (privatePort_int = atoi(privatePort)) == 0 || privatePort_int > 65535){
				//continue;
			}
                	
			// get public port
			if((getNthValueSafe(6, rec, ',', publicPort, sizeof(publicPort)) == -1)){
				//continue;
			}
			//printf("jacky - check publicPort(%d):%s\n", sizeof(publicPort), publicPort);
			if( (publicPort_int = atoi(publicPort)) > 65535){
				//continue;
			}
                	
			// get protocol
			if((getNthValueSafe(7, rec, ',', protocol, sizeof(protocol)) == -1)){
				//continue;
			}
			//printf("jacky - check protocol(%d):%s\n", sizeof(protocol), protocol);
			proto = atoi(protocol);
        }
    }else
		return;
	
	//printf("jacky - check ipaddress:%s\n", ipaddress);
	//strncpy(wan_name, getWanIfName(), 16-1);
	switch(proto){
			case PROTO_TCP:
				checkWebServicePort(publicPort);
				//break;
			case PROTO_UDP:
				if (0 == makeSinglePortForwardRule(cmd, sizeof(cmd), "-t nat", "PREROUTING", inboundFilter, iptablesRuleName, ipaddress, proto, privatePort_int, publicPort_int))
					doSystem(cmd);
				if (0 == makeSinglePortForwardRule(cmd, sizeof(cmd), "-t nat", "POSTROUTING", inboundFilter, iptablesRuleName, ipaddress, proto, privatePort_int, publicPort_int))
					doSystem(cmd);
				break;
			case PROTO_TCP_UDP:
				checkWebServicePort(publicPort);
				if (0 == makeSinglePortForwardRule(cmd, sizeof(cmd), "-t nat", "PREROUTING", inboundFilter, iptablesRuleName, ipaddress, PROTO_TCP, privatePort_int, publicPort_int))
					doSystem(cmd);
				if (0 == makeSinglePortForwardRule(cmd, sizeof(cmd), "-t nat", "POSTROUTING", inboundFilter, iptablesRuleName, ipaddress, PROTO_TCP, privatePort_int, publicPort_int))
					doSystem(cmd);
				
				if (0 == makeSinglePortForwardRule(cmd, sizeof(cmd), "-t nat", "PREROUTING", inboundFilter, iptablesRuleName, ipaddress, PROTO_UDP, privatePort_int, publicPort_int))
					doSystem(cmd);
				if (0 == makeSinglePortForwardRule(cmd, sizeof(cmd), "-t nat", "POSTROUTING", inboundFilter, iptablesRuleName, ipaddress, PROTO_UDP, privatePort_int, publicPort_int))
					doSystem(cmd);
				break;
			default:
				//continue;
				break;
	}
}

static void parsePortRange(int portCount, char *portArray, char *iptablesParameter, char *chainName, char *inboundFilterRuleName, char *iptablesRuleName, char *ipaddress, int protocol, char *APIName, char *scheduleRule)
{
	int loopCount;
	char wan_name[16]={0}, portRange[32]={0}, cmd[1024]={0};
	
	if (portCount > 0) {
		for (loopCount=0; loopCount<portCount; loopCount++) {
			memset(portRange, '\0', sizeof(portRange));
			if((getNthValueSafe(loopCount, portArray, ',', portRange, sizeof(portRange)) == 0)){
				if (strchr(portRange, '-'))
				{
					*strchr(portRange, '-') = ':';
					//printf("jacky - check replace string:%s\n", portRange);
				}
				if (!strcmp(APIName, "PortRangeForward"))
					makePortRangeForwardRule(cmd, sizeof(cmd), iptablesParameter, chainName, inboundFilterRuleName, iptablesRuleName, ipaddress, protocol, portRange);
				else if (!strcmp(APIName, "AccessControl")) {
					//makeAccessControlRule(cmd, sizeof(cmd), "-t nat", "PREROUTING", iptablesRuleName, scheduleRule, protocol, ipaddress, portRange);
					makeAccessControlRule(cmd, sizeof(cmd), "-t nat", "PREROUTING", iptablesRuleName, protocol, ipaddress, portRange);
				}
				
				doSystem(cmd);
			}
			else {
				printf("jacky - error, can't get value!\n");
			}
		}
	}
}

static void iptablesPortRangeForwardRun(char *iptablesRuleName, char *ruleName, char *ruleEnable)
{
	char rec[256]={0};
	char *begin, *end;
	char temp[64]={0}, ruleEnableCheck[4]={0}, ruleNameCheck[32]={0}, ipaddress[32]={0}, TCPPorts[128]={0}, UDPPorts[128]={0};
	char schedule[32]={0}, inboundFilter[32]={0};
	char *rules, *pt;

    if(!ruleEnable){
        printf("Warning: can't find \"PortForwardEnable\" in flash\n");
        return;
    }

    if(!strcmp(ruleEnable, "on")){
        rules = nvram_bufget(RT2860_NVRAM, "PortRangeForwardRules");
        
        if(!rules){
            printf("Warning: can't find \"PortRangeForwardRules\" in flash\n");
            return ;
        }
        
        //strcpy(temp, "on,");
        strcpy(temp, ruleName);
        strcat(temp, ",");
        pt = strstr(rules, temp);
        //printf("jacky - check pt addr:%x\n", pt);
        //printf("jacky - check rules addr:%x\n", rules);
        if (pt != rules) {
        	strcpy(temp, ";");
        	strcat(temp, ruleName);
        	strcat(temp, ",");
        	pt = strstr(rules, temp);
        	pt = pt+sizeof(char);
        }
        //printf("jacky - check pt:%s\n", pt);
        if (getNthValueSafe(0, pt, ';', rec, 256) != -1)
        {
			if((getNthValueSafe(0, rec, ',', ruleNameCheck, sizeof(ruleNameCheck)) == -1)){
				//continue;
			}
			if((getNthValueSafe(1, rec, ',', ruleEnableCheck, sizeof(ruleEnableCheck)) == -1)){
				//continue;
			}
			if((getNthValueSafe(2, rec, ',', schedule, sizeof(schedule)) == -1)){
				//continue;
			}
			if((getNthValueSafe(3, rec, ',', inboundFilter, sizeof(inboundFilter)) == -1)){
				//continue;
			}
			if((getNthValueSafe(4, rec, ',', ipaddress, sizeof(ipaddress)) == -1)){
				//continue;
			}
			if(!isIpValid(ipaddress)){
				//continue;
			}
            
            //TCPPorts
            memset(TCPPorts, '\0', sizeof(TCPPorts));
            begin = strchr(rec, '\'');
            end = strchr(begin+1, '\'');
            memcpy(TCPPorts, begin+1, ((end-1) - (begin+1) + 1));
            
            //UDPPorts
            memset(UDPPorts, '\0', sizeof(UDPPorts));
            begin = strchr((end+1), '\'');
            end = strchr((begin + 1), '\'');
            memcpy(UDPPorts, begin+1, ((end-1) - (begin+1) + 1));
        }
    }else
		return;

	parsePortRange(getNums(TCPPorts, ','), TCPPorts, "-t nat", "PREROUTING", inboundFilter, iptablesRuleName, ipaddress, PROTO_TCP, "PortRangeForward", "");
	parsePortRange(getNums(TCPPorts, ','), TCPPorts, "-t nat", "POSTROUTING", inboundFilter, iptablesRuleName, ipaddress, PROTO_TCP, "PortRangeForward", "");
	parsePortRange(getNums(UDPPorts, ','), UDPPorts, "-t nat", "PREROUTING", inboundFilter, iptablesRuleName, ipaddress, PROTO_UDP, "PortRangeForward", "");
	parsePortRange(getNums(UDPPorts, ','), UDPPorts, "-t nat", "POSTROUTING", inboundFilter, iptablesRuleName, ipaddress, PROTO_UDP, "PortRangeForward", "");
}

//static void iptablesInboundFilterRun(char *iptablesRuleName, char *ruleName, char *ruleEnable)
static void iptablesInboundFilterRun(char *iptablesRuleName, char *ruleName, char *protocol, char *portRange)
{
	char temp[64]={0}, rec[256]={0}, cmd[1024]={0}, wan_name[16]={0};
	char ruleEnableCheck[4]={0}, ruleNameCheck[32]={0}, ipaddress[16]={0};
	char *rules, *pt;
    
    rules = nvram_bufget(RT2860_NVRAM, "InboundFilterRules");
	
	if(!rules){
	    printf("Warning: can't find \"InboundFilterRules\" in flash\n");
	    return ;
	}
	
	//strcpy(temp, "on,");
    strcpy(temp, ruleName);
    strcat(temp, ",");
	pt = strstr(rules, temp);
	//printf("jacky - check pt addr:%x\n", pt);
    //printf("jacky - check rules addr:%x\n", rules);
    if (pt != rules) {
       	strcpy(temp, ";");
       	strcat(temp, ruleName);
      	strcat(temp, ",");
      	pt = strstr(rules, temp);
      	pt = pt+sizeof(char);
    }
    //printf("jacky - check pt:%s\n", pt);
	if (getNthValueSafe(0, pt, ';', rec, 256) != -1)
	{
		if((getNthValueSafe(0, rec, ',', ruleNameCheck, sizeof(ruleNameCheck)) == -1)){
			//continue;
		}
		if((getNthValueSafe(1, rec, ',', ruleEnableCheck, sizeof(ruleEnableCheck)) == -1)){
			//continue;
		}
		
		if((getNthValueSafe(2, rec, ',', ipaddress, sizeof(ipaddress)) == -1)){
			//continue;
		}
		
		if(!isIpValid(ipaddress)){
			//continue;
		}
	}

    //if (!strcmp(ruleEnableCheck, "on")) {
    strncpy(wan_name, getWanIfName(), 16-1);
    makeInboundFilterRule(cmd, sizeof(cmd), "", "INPUT", wan_name, iptablesRuleName, ipaddress, protocol, portRange, ruleEnableCheck);
    doSystem(cmd);
    makeInboundFilterRule(cmd, sizeof(cmd), "",  "FORWARD", wan_name, iptablesRuleName, ipaddress, protocol, portRange, ruleEnableCheck);
    doSystem(cmd);
    //}

}

static void parsePortNum(char *iptablesRuleName, char *scheduleRule, char *ipRange, char *protocol, char *portRange)
{
	char cmd[1024]={0}, wan_name[16]={0};
	int portNumStart, portNumEnd;

	strncpy(wan_name, getWanIfName(), 16-1);
   	//makeAccessControlRule(cmd, sizeof(cmd), "-t nat", "PREROUTING", iptablesRuleName, scheduleRule, protocol, ipRange, portRange);
   	makeAccessControlRule(cmd, sizeof(cmd), "-t nat", "PREROUTING", iptablesRuleName, protocol, ipRange, portRange);
   	doSystem(cmd);
}

static void AccessControlRun(char *iptablesRuleName, char *ruleName, char *ruleEnable)
{
	char temp[64], rec[256];
	char ruleEnableOutput[4], ruleNameCheck[32], ipRange[64], scheduleRule[16], ruleDefine[2], tempArray[16];
	char TCPPorts[32], UDPPorts[23];
    char *rules, *pt, *begin, *end;
    int loopCount, specialServiceCount=8;

    if(!ruleEnable){
        printf("Warning: Access Control status is disable .\n");
        return;
    }
    //ruleEnable = nvram_bufget(RT2860_NVRAM, "AccessControlEnable");
    rules = nvram_bufget(RT2860_NVRAM, "AccessControlPolicy");

	if(!rules){
	    printf("Warning: can't find \"AccessControlPolicy\" in flash\n");
	    return ;
	}
	
	//special service of format: ruleName, ipaddress_start-ipaddress_end, scheduleRule, atoi(ruleDefine), wwwRule, smtpRule, pop3Rule, ftpRule, telnetRule, dnsRule, tcpRule, udpRule
	//user define of format: ruleName, ipaddress_start-ipaddress_end, scheduleRule, atoi(ruleDefine), atoi(protocol), TCPPorts, UDPPorts
	//strcpy(temp, "on,");
    strcpy(temp, ruleName);
    strcat(temp, ",");
	pt = strstr(rules, temp);
	//printf("jacky - check pt addr:%x\n", pt);
	//printf("jacky - check rules addr:%x\n", rules);
	if (pt != rules) {
		strcpy(temp, ";");
		strcat(temp, ruleName);
		strcat(temp, ",");
		pt = strstr(rules, temp);
		pt = pt+sizeof(char);
	}
	//printf("jacky - check pt:%s\n", pt);
	//if (atoi(ruleEnable) == 1)
	if (!strcmp(ruleEnable, "on"))
	{
		if (getNthValueSafe(0, pt, ';', rec, 256) != -1)
		{
			if((getNthValueSafe(0, rec, ',', ruleNameCheck, sizeof(ruleNameCheck)) == -1)){
				//continue;
			}
			
			if((getNthValueSafe(1, rec, ',', ruleEnableOutput, sizeof(ruleEnableOutput)) == -1)){
				//continue;
			}
			if((getNthValueSafe(2, rec, ',', scheduleRule, sizeof(scheduleRule)) == -1)){
				//continue;
			}
			if((getNthValueSafe(3, rec, ',', ipRange, sizeof(ipRange)) == -1)){
				//continue;
			}
			
			if((getNthValueSafe(4, rec, ',', ruleDefine, sizeof(ruleDefine)) == -1)){
				//continue;
			}
			//ruleDefine=0: special service, ruleDefine=1: user define.
    		
			if (atoi(ruleDefine) == 0) {
				for (loopCount=0; loopCount<specialServiceCount; loopCount++) {
					if((getNthValueSafe((5+loopCount), rec, ',', tempArray, 16) == -1)){ return; } //wwwRule
					if (!strcmp(tempArray, "on")) {
						switch(loopCount) {
							case 0: //wwwRule
								parsePortNum(iptablesRuleName, scheduleRule, ipRange, PROTO_TCP, "80");
								parsePortNum(iptablesRuleName, scheduleRule, ipRange, PROTO_TCP, "8080");
								break;
							case 1: //smtpRule
								parsePortNum(iptablesRuleName, scheduleRule, ipRange, PROTO_TCP, "25");
								break;
							case 2: //pop3Rule
								parsePortNum(iptablesRuleName, scheduleRule, ipRange, PROTO_TCP, "110");
								break;
							case 3: //ftpRule
								parsePortNum(iptablesRuleName, scheduleRule, ipRange, PROTO_TCP, "21");
								break;
							case 4: //telnetRule
								parsePortNum(iptablesRuleName, scheduleRule, ipRange, PROTO_TCP, "23");
								break;
							case 5: //dnsRule
								parsePortNum(iptablesRuleName, scheduleRule, ipRange, PROTO_UDP, "53");
								break;
							case 6: //tcpRule
								parsePortNum(iptablesRuleName, scheduleRule, ipRange, PROTO_TCP, "1:65535");
								break;
							case 7: //udpRule
								parsePortNum(iptablesRuleName, scheduleRule, ipRange, PROTO_UDP, "1:65535");
								break;
						}
					}
				}
			}
			else if (atoi(ruleDefine) == 1) {
				//TCPPorts
            	memset(TCPPorts, '\0', sizeof(TCPPorts));
            	begin = strchr(rec, '\'');
            	end = strchr(begin+1, '\'');
            	memcpy(TCPPorts, begin+1, ((end-1) - (begin+1) + 1));
            	
	            //UDPPorts
    	        memset(UDPPorts, '\0', sizeof(UDPPorts));
        	    begin = strchr((end+1), '\'');
            	end = strchr((begin + 1), '\'');
            	memcpy(UDPPorts, begin+1, ((end-1) - (begin+1) + 1));
				
				parsePortRange(getNums(TCPPorts, ','), TCPPorts, "-t nat", "PREROUTING", "", iptablesRuleName, ipRange, PROTO_TCP, "AccessControl", scheduleRule);
				parsePortRange(getNums(UDPPorts, ','), UDPPorts, "-t nat", "PREROUTING", "", iptablesRuleName, ipRange, PROTO_UDP, "AccessControl", scheduleRule);
			}
		}
	}
}

static void iptablesAllFilterRun(void)
{
	//iptablesIPPortFilterRun();

	/* system filter */
	iptablesRemoteManagementRun();

}

void iptablesRuleFlush(char *iptablesRuleName)
{
	iptablesFilterFlush("INPUT", iptablesRuleName);
	iptablesFilterFlush("FORWARD", iptablesRuleName);
	iptablesFlush("-t nat", "PREROUTING", iptablesRuleName);
	iptablesFlush("-t nat", "POSTROUTING", iptablesRuleName);
}

void iptablesAllNATRun(void)
{
	//Must have serial
	iptablesDMZRun();
	//iptablesAllFilterRun(); the same with iptablesRemoteManagementRun
	iptablesRemoteManagementRun();
	
	//Jacky.Yang 19-May-2008, all run file run in addScheduleToCron();
	//Tom.Hung 2010-1-21, remove addScheduleToCron() because when reboot, it will do twice and cause UI no response.
	//addScheduleToCron();
	
}

inline int getRuleNums(char *rules){
	return getNums(rules, ';');
}

static int getDefaultFirewallPolicyASP(int eid, webs_t wp, int argc, char_t **argv)
{
	int value;
	char *p = nvram_bufget(RT2860_NVRAM, "DefaultFirewallPolicy");
	int default_policy;
	if(!p)
		default_policy = 0;
	else
		default_policy = atoi(p);

	if( ejArgs(argc, argv, T("%d"), &value) != 1){
		return -1;
	}

	if(default_policy == value )
		websWrite(wp, T(" selected "));
	return 0;	
}

static int checkIfUnderBridgeModeASP(int eid, webs_t wp, int argc, char_t **argv)
{
	char *mode = nvram_bufget(RT2860_NVRAM, "OperationMode");
	if(!mode)
		return -1;	// fatal error, make ASP engine complained.
	if(atoi(mode) == 0)	// bridge mode
		websWrite(wp, T(HTML_NO_FIREWALL_UNDER_BRIDGE_MODE));
	return 0;
}

/*
 * ASP function
 */
static int getPortForwardRuleNumsASP(int eid, webs_t wp, int argc, char_t **argv)
{
    char *rules = nvram_bufget(RT2860_NVRAM, "PortForwardRules");
	if(!rules || !strlen(rules) ){
		websWrite(wp, T("0"));
		return 0;
	}
	websWrite(wp, T("%d"), getRuleNums(rules));
	return 0;
}

/*
 * ASP function
 */
static int getIPPortRuleNumsASP(int eid, webs_t wp, int argc, char_t **argv)
{
    char *rules = nvram_bufget(RT2860_NVRAM, "IPPortFilterRules");
	if(!rules || !strlen(rules) ){
		websWrite(wp, T("0"));
		return 0;
	}

	websWrite(wp, T("%d"), getRuleNums(rules));
	return 0;
}

static void getRulesPacketCount(webs_t wp, char_t *path, char_t *query)
{
	FILE *fp;
	int i, step_in_chains=0;
	char buf[1024]={0}, *default_policy;
	int default_drop_flag;
	int index=0, pkt_count;
	int *result;

	// check if the default policy is "drop" 
	default_policy = nvram_bufget(RT2860_NVRAM, "DefaultFirewallPolicy");
	if(!default_policy)
		default_policy = "0";
	default_drop_flag = atoi(default_policy);

	websWrite(wp, T("HTTP/1.1 200 OK\nContent-type: text/plain\nPragma: no-cache\n\n"));	

	result = (int *)malloc(sizeof(int) * 128);
	if(!result)
		goto error;

	fp = popen("iptables -t filter -L -v", "r");
	if(!fp){
		free(result);
		goto error;
	}

	while(fgets(buf, sizeof(buf), fp) && index < 128){
		if(step_in_chains){
			if(buf[0] == '\n')
				break;
			if(buf[0] == ' ' && buf[1] == 'p' && buf[2] == 'k' && buf[3] == 't' )
				continue;
			// Skip the first one rule if default policy is drop.
			if(default_drop_flag){
				default_drop_flag = 0;
				continue;
			}
			sscanf(buf, "%d ", &pkt_count);
			result[index++] = pkt_count;
		}

		if(strstr(buf, "Chain " IPPORT_FILTER_CHAIN))
			step_in_chains = 1;
	}
	pclose(fp);

	if(index > 0)
		websWrite(wp, "%d", result[0]);
	for(i=1; i<index; i++)
		websWrite(wp, " %d", result[i]);

	free(result);
error:
	websDone(wp, 200);
	return;
}


/*
 * ASP function
 */
static int showIPPortFilterRulesASP(int eid, webs_t wp, int argc, char_t **argv)
{
	int i;
	int sprf_int, sprt_int, proto;
	char mac_address[32]={0};
	char sip_1[32]={0}, sip_2[32]={0}, sprf[8]={0}, sprt[8]={0}, comment[16]={0}, protocol[8]={0}, action[4]={0};
	char dip_1[32]={0}, dip_2[32]={0}, dprf[8]={0}, dprt[8]={0};
	int dprf_int, dprt_int;
	char rec[256]={0};
	char *default_policy;
	char *rules = nvram_bufget(RT2860_NVRAM, "IPPortFilterRules");
	if(!rules)
		return 0;

    default_policy = nvram_bufget(RT2860_NVRAM, "DefaultFirewallPolicy");
    // add the default policy to the end of FORWARD chain
    if(!default_policy)
		return 0;
	if(!strlen(default_policy))
		return 0;

	i=0;
	while(getNthValueSafe(i, rules, ';', rec, 256) != -1 && strlen(rec)){
		printf("i=%d, rec=%s, strlen(rec)=%d\n", i, rec, strlen(rec));
		// get ip 1
        if((getNthValueSafe(0, rec, ',', sip_1, 32) == -1)){
			i++;
			continue;
		}
        if(!isIpNetmaskValid(sip_1)){
			i++;
			continue;
		}
		// translate "any/0" to "any" for readable reason
		if( !strcmp(sip_1, "any/0"))
			strcpy(sip_1, "-");

		// get ip 2
        // get ip address
        if((getNthValueSafe(1, rec, ',', sip_2, 32) == -1)){
			i++;
			continue;
		}
		// dont verify cause we dont have ip range support
		//if(!isIpValid(sip_2))
        //    continue;

		// get port range "from"
		if((getNthValueSafe(2, rec, ',', sprf, 8) == -1)){
			i++;
			continue;
		}
		if( (sprf_int = atoi(sprf)) > 65535){
			i++;
			continue;
		}

		// get port range "to"
		if((getNthValueSafe(3, rec, ',', sprt, 8) == -1)){
			i++;
			continue;
		}
		if( (sprt_int = atoi(sprt)) > 65535){
			i++;
			continue;
		}

		// get ip 1
        if((getNthValueSafe(4, rec, ',', dip_1, 32) == -1)){
			i++;
            continue;
		}
        if(!isIpNetmaskValid(dip_1)){
			i++;
            continue;
		}
		// translate "any/0" to "any" for readable reason
		if( !strcmp(dip_1, "any/0"))
			strcpy(dip_1, "-");
		
		// get ip 2
        if((getNthValueSafe(5, rec, ',', dip_2, 32) == -1)){
			i++;
            continue;
		}
		// dont verify cause we dont have ip range support
		//if(!isIpValid(dip_2))
        //    continue;

		// get protocol
		if((getNthValueSafe(8, rec, ',', protocol, 8) == -1)){
			i++;
			continue;
		}
		proto = atoi(protocol);
		switch(proto){
			case PROTO_TCP:
			case PROTO_UDP:
			case PROTO_NONE:
			case PROTO_ICMP:
				break;
			default:
				continue;
		}

		// get port range "from"
		if((getNthValueSafe(6, rec, ',', dprf, 8) == -1)){
			i++;
			continue;
		}
		if( (dprf_int = atoi(dprf)) > 65535){
			i++;
			continue;
		}

		// get port range "to"
		if((getNthValueSafe(7, rec, ',', dprt, 8) == -1)){
			i++;
			continue;
		}
		if( (dprt_int = atoi(dprt)) > 65535){
			i++;
			continue;
		}

		// get action
		if((getNthValueSafe(9, rec, ',', action, 4) == -1)){
			i++;
			continue;
		}

		// get comment
		if((getNthValueSafe(10, rec, ',', comment, 16) == -1)){
			i++;
			continue;
		}

		// get mac address
		if((getNthValueSafe(11, rec, ',', mac_address, 32) == -1)){
			i++;
			continue;
		}
		if(!strlen(mac_address))
			gstrcpy(mac_address, T("-"));

		websWrite(wp, T("<tr>\n"));
		// output No.
		websWrite(wp, T("<td> %d&nbsp; <input type=\"checkbox\" name=\"delRule%d\"> </td>"), i+1, i );

		// output Mac address
		websWrite(wp, T("<td align=center> %s </td>"), mac_address);

		// output DIP
		websWrite(wp, T("<td align=center> %s </td>"), dip_1);
		// we dont support ip range 
		// websWrite(wp, T("<td align=center> %s-%s </td>"), ip_1, ip_2);

		// output SIP
		websWrite(wp, T("<td align=center> %s </td>"), sip_1);
		// we dont support ip range 
		// websWrite(wp, T("<td align=center> %s-%s </td>"), ip_1, ip_2);

		// output Protocol
        switch(proto){
            case PROTO_TCP:
				websWrite(wp, T("<td align=center> TCP </td>"));
				break;
            case PROTO_UDP:
				websWrite(wp, T("<td align=center> UDP </td>"));
				break;
            case PROTO_ICMP:
				websWrite(wp, T("<td align=center> ICMP </td>"));
				break;
            case PROTO_NONE:
				websWrite(wp, T("<td align=center> - </td>"));
				break;
		}

		// output dest Port Range
		if(dprt_int)
			websWrite(wp, T("<td align=center> %d - %d </td>"), dprf_int, dprt_int);
		else{
			// we re-descript the port number here because 
			// "any" word is more meanful than "0"
			if(!dprf_int){
				websWrite(wp, T("<td align=center> - </td>"), dprf_int);
			}else{
				websWrite(wp, T("<td align=center> %d </td>"), dprf_int);
			}
		}

		// output Source Port Range
		if(sprt_int)
			websWrite(wp, T("<td align=center> %d - %d </td>"), sprf_int, sprt_int);
		else{
			// we re-descript the port number here because 
			// "any" word is more meanful than "0"
			if(!sprf_int){
				websWrite(wp, T("<td align=center> - </td>"), sprf_int);
			}else{
				websWrite(wp, T("<td align=center> %d </td>"), sprf_int);
			}
		}


		// output action
        switch(atoi(action)){
            case ACTION_DROP:
				websWrite(wp, T("<td align=center id=portFilterActionDrop%d> Drop </td>"), i);
				break;
            case ACTION_ACCEPT:
				websWrite(wp, T("<td align=center id=portFilterActionAccept%d> Accept </td>"), i);
				break;
		}

		// output Comment
		if(strlen(comment))
			websWrite(wp, T("<td align=center> %s</td>"), comment);
		else
			websWrite(wp, T("<td align=center> &nbsp; </td>"));

		// output the id of "packet count"
		websWrite(wp, T("<td align=center id=pktCnt%d>-</td>"), i);

		websWrite(wp, T("</tr>\n"));

		i++;
	}	  

	switch(atoi(default_policy)){
		case 0:
			websWrite(wp, T("<tr><td align=center colspan=9 id=portCurrentFilterDefaultAccept> Others would be accepted.</td><td align=center id=pktCnt%d>-</td></tr>"), i);
			break;
		case 1:
			websWrite(wp, T("<tr><td align=center colspan=9 id=portCurrentFilterDefaultDrop> Others would be dropped.</td><td align=center id=pktCnt%d>-</td></tr>"), i);
			break;
	}

	return 0;	
}

static int showDMZIPAddressASP(int eid, webs_t wp, int argc, char_t **argv)
{
	char *DMZIPAddress = nvram_bufget(RT2860_NVRAM, "DMZIPAddress");
	if(!DMZIPAddress)
		return 0;
	if(!strlen(DMZIPAddress))
		return 0;

	websWrite(wp, T("%s"), DMZIPAddress);
	return 0;	
}

static void ipportFilterDelete(webs_t wp, char_t *path, char_t *query)
{
	int i, j, rule_count;
	char_t name_buf[16]={0};
	char_t *value;
	int *deleArray;
//	char *firewall_enable;

    char *rules = nvram_bufget(RT2860_NVRAM, "IPPortFilterRules");
    if(!rules || !strlen(rules) )
        return;

	rule_count = getRuleNums(rules);
	if(!rule_count)
		return;

	deleArray = (int *)malloc(rule_count * sizeof(int));

	for(i=0, j=0; i< rule_count; i++){
		snprintf(name_buf, 16, "delRule%d", i);
		value = websGetVar(wp, name_buf, NULL);
		if(value){
			deleArray[j++] = i;
		}
	}
	if(!j){
	    websHeader(wp);
	    websWrite(wp, T("You didn't select any rules to delete.<br>\n"));
	    websFooter(wp);
	    websDone(wp, 200);		
		return;
	}

	deleteNthValueMulti(deleArray, rule_count, rules, ';');
	free(deleArray);

	nvram_set(RT2860_NVRAM, "IPPortFilterRules", rules);
	nvram_commit(RT2860_NVRAM);

	iptablesIPPortFilterFlush();
	iptablesIPPortFilterRun();

    websHeader(wp);
    websWrite(wp, T("s<br>\n") );
    websWrite(wp, T("fromPort: <br>\n"));
    websWrite(wp, T("toPort: <br>\n"));
    websWrite(wp, T("protocol: <br>\n"));
    websWrite(wp, T("comment: <br>\n"));
    websFooter(wp);
    websDone(wp, 200);

	return;
}

static void ipportFilter(webs_t wp, char_t *path, char_t *query)
{
	char rule[ruleLength];
	char *mac_address;
	char *sip_1, *sip_2, *sprf, *sprt, *protocol, *action_str, *comment;
	char *dip_1, *dip_2, *dprf, *dprt;
	char *IPPortFilterRules;
	
	int sprf_int, sprt_int, dprf_int, dprt_int, proto, action;

	mac_address = websGetVar(wp, T("mac_address"), T(""));

	sip_1 = websGetVar(wp, T("sip_address"), T("any"));
	sip_2 = websGetVar(wp, T("sip_address2"), T(""));
	sprf = websGetVar(wp, T("sFromPort"), T("0"));
	sprt = websGetVar(wp, T("sToPort"), T(""));

	dip_1 = websGetVar(wp, T("dip_address"), T("any"));
	dip_2 = websGetVar(wp, T("dip_address2"), T(""));
	dprf = websGetVar(wp, T("dFromPort"), T("0"));
	dprt = websGetVar(wp, T("dToPort"), T(""));

	protocol = websGetVar(wp, T("protocol"), T(""));
	action_str = websGetVar(wp, T("action"), T(""));
	comment = websGetVar(wp, T("comment"), T(""));

	if(!mac_address || !sip_1 || !dip_1 || !sprf || !dprf)
		return;

	if(!strlen(mac_address) && !strlen(sip_1) && !strlen(dip_1) && !strlen(sprf) && !strlen(dprf))
		return;

	// we dont trust user input.....
	if(strlen(mac_address)){
		if(!isMacValid(mac_address))
			return;
	}

	if(strlen(sip_1)){
		if(!isIpNetmaskValid(sip_1))
			return;
	}else
		sip_1 = T("any/0");

	if(strlen(dip_1)){
		if(!isIpNetmaskValid(sip_1))
			return;
	}else
    	dip_1 = T("any/0");

	sip_2 = dip_2 = "0";

	if(! strcmp(protocol, T("TCP")))
		proto = PROTO_TCP;
	else if( !strcmp(protocol, T("UDP")))
		proto = PROTO_UDP;
	else if( !strcmp(protocol, T("None")))
		proto = PROTO_NONE;
	else if( !strcmp(protocol, T("ICMP")))
		proto = PROTO_ICMP;
	else
		return;

	if(!strlen(sprf) || proto == PROTO_NONE || proto == PROTO_ICMP){
		sprf_int = 0;
	}else{
		sprf_int = atoi(sprf);
		if(sprf_int == 0 || sprf_int > 65535)
			return;
	}

	if(!strlen(sprt) || proto == PROTO_NONE || proto == PROTO_ICMP){
		sprt_int = 0;
	}else{
		sprt_int = atoi(sprt);
		if(sprt_int ==0 || sprt_int > 65535)
			return;
	}

	if(!strlen(dprf) || proto == PROTO_NONE || proto == PROTO_ICMP){
		dprf_int = 0;
	}else{
		dprf_int = atoi(dprf);
		if(dprf_int ==0 || dprf_int > 65535)
			return;
	}

	if(!strlen(dprt) || proto == PROTO_NONE || proto == PROTO_ICMP){
		dprt_int = 0;
	}else{
		dprt_int = atoi(dprt);
		if(dprt_int ==0 || dprt_int > 65535)
			return;
	}

	if(! (strcmp(action_str, T("Accept"))))
		action = ACTION_ACCEPT;
	else if(! (strcmp(action_str, T("Drop"))))
		action = ACTION_DROP;
	else
		return;

	if(strlen(comment) > 32)
		return;
	// i know you will try to break our box... ;) 
	if(strchr(comment, ';') || strchr(comment, ','))
		return;

	if(   ( IPPortFilterRules = nvram_bufget(RT2860_NVRAM, "IPPortFilterRules")) && strlen(IPPortFilterRules)){
		snprintf(rule, sizeof(rule), "%s;%s,%s,%d,%d,%s,%s,%d,%d,%d,%d,%s,%s", IPPortFilterRules, sip_1, sip_2, sprf_int, sprt_int, dip_1, dip_2, dprf_int, dprt_int, proto, action, comment, mac_address);
	}else{
		snprintf(rule, sizeof(rule), "%s,%s,%d,%d,%s,%s,%d,%d,%d,%d,%s,%s", sip_1, sip_2, sprf_int, sprt_int, dip_1, dip_2, dprf_int, dprt_int, proto, action, comment, mac_address);
	}

	nvram_set(RT2860_NVRAM, "IPPortFilterRules", rule);
	nvram_commit(RT2860_NVRAM);

	iptablesIPPortFilterFlush();
	iptablesIPPortFilterRun();

	websHeader(wp);
	websWrite(wp, T("mac: %s<br>\n"), mac_address);	
	websWrite(wp, T("sip1: %s<br>\n"), sip_1);	
	websWrite(wp, T("sip2: %s<br>\n"), sip_2);	
	websWrite(wp, T("sFromPort: %s<br>\n"), sprf);
	websWrite(wp, T("sToPort: %s<br>\n"), sprt);
	websWrite(wp, T("dip1: %s<br>\n"), dip_1);	
	websWrite(wp, T("dip2: %s<br>\n"), dip_2);	
	websWrite(wp, T("dFromPort: %s<br>\n"), dprf);
	websWrite(wp, T("dToPort: %s<br>\n"), dprt);
	websWrite(wp, T("protocol: %s<br>\n"), protocol);
	websWrite(wp, T("action: %s<br>\n"), action_str);
	websWrite(wp, T("comment: %s<br>\n"), comment);

    websFooter(wp);
    websDone(wp, 200);        
    return;
	
}

static void iptablesBasicSetting(void)
{
	char *firewall_enable;

	firewall_enable = nvram_bufget(RT2860_NVRAM, "IPPortFilterEnable");

	// flush  ipport   filter   chain
    iptablesIPPortFilterFlush();

	if(!firewall_enable || !atoi(firewall_enable))
		return;
    iptablesIPPortFilterRun();

	return;
}

static void BasicSettings(webs_t wp, char_t *path, char_t *query)
{
	char *default_policy, *firewall_enable;

	firewall_enable = websGetVar(wp, T("portFilterEnabled"), T(""));
	default_policy = websGetVar(wp, T("defaultFirewallPolicy"), T("0"));

	switch(atoi(firewall_enable)){
	case 0:
		nvram_set(RT2860_NVRAM, "IPPortFilterEnable", "0");
		break;
	case 1:
		nvram_set(RT2860_NVRAM, "IPPortFilterEnable", "1");
		break;
	}

	switch(atoi(default_policy)){
	case 1:
		nvram_set(RT2860_NVRAM, "DefaultFirewallPolicy", "1");
		break;
	case 0:
	default:
		nvram_set(RT2860_NVRAM, "DefaultFirewallPolicy", "0");
		break;
	}
	nvram_commit(RT2860_NVRAM);

	iptablesBasicSetting();

	websHeader(wp);
	websWrite(wp, T("default_policy: %s<br>\n"), default_policy);
    websFooter(wp);
    websDone(wp, 200);        
}

void checkWebServicePort(char *port)
{
	char rangeStart[6], rangeEnd[6];
	char *pt;
	//printf("check port: %s\n", port);
	memset(rangeStart, '\0', sizeof(rangeStart));
	memset(rangeEnd, '\0', sizeof(rangeEnd));
	if (strchr(port, "-") || strchr(port, ",")) //port range
	{
		//Jacky.Yang 26-Oct-2008, we doesn't support drop web serivce port for port range forwarding.
		/*pt = strchr(port, "-");
		snprintf(rangeStart, sizeof(rangeStart), "%s", pt);
		snprintf(rangeEnd, sizeof(rangeEnd), "%s", (port -pt));
		printf("rangeStart:%s, rangeEnd:%s\n", rangeStart, rangeEnd);*/
	}
	else
	{
		//printf("getGoAHeadServerPort:%d, atoi(port):%d\n", getGoAHeadServerPort(), atoi(port));
		if (getGoAHeadServerPort() == atoi(port)) {
			//doSystem("iptables -t nat -F remotePortForward");
			//doSystem("iptables -t nat -F PREROUTING_Remote");
			//doSystem("iptables -t nat -F POSTROUTING_Remote");
		}
	}
}

static void DMZ(webs_t wp, char_t *path, char_t *query)
{
	char *dmzE, *ip_address;

	dmzE = websGetVar(wp, T("DMZEnabled"), T(""));
	ip_address = websGetVar(wp, T("DMZIPAddress"), T(""));

	// someone use malform page.....
	if(!dmzE && !strlen(dmzE))
		return;

	// we dont trust user input, check all things before doing changes
	if(atoi(dmzE) && !isIpValid(ip_address))	// enable && invalid mac address
		return;

	iptablesDMZFlush();
	if(atoi(dmzE) == 0){		// disable
		nvram_set(RT2860_NVRAM, "DMZEnable", "0");
	}else{					// enable
		nvram_set(RT2860_NVRAM, "DMZEnable", "1");
		if(strlen(ip_address)){
			nvram_set(RT2860_NVRAM, "DMZIPAddress", ip_address);
		}
	}

	nvram_commit(RT2860_NVRAM);
	//iptablesDMZRun();
	firewall_init();
	
	//websRedirect(wp, "/advanced/dmz.asp");
	websRedirect(wp, "/applied.asp?url=/advanced/dmz.asp");
}

static void remoteManagement(webs_t wp, char_t *path, char_t *query)
{
	char *rmE = websGetVar(wp, T("remoteManagementEnabled"), T(""));
	char *portNum = websGetVar(wp, T("portNumber"), T("8080"));

	// someone use malform page.....
	if(!rmE && !strlen(rmE))
		return;
	
	if(atoi(rmE) == 0){		// disable
		nvram_set(RT2860_NVRAM, "RemoteManagement", "0");
	}else{					// enable
		nvram_set(RT2860_NVRAM, "RemoteManagement", "1");
	}
	
	nvram_set(RT2860_NVRAM, "RemotePort", portNum);
	nvram_commit(RT2860_NVRAM);

	iptablesRemoteManagementRun();

	//websRedirect(wp, "/adm/management.asp");
	websRedirect(wp, "/applied.asp?url=/adm/management.asp");
}

static void singlePortForwardAdd(webs_t wp, char_t *path, char_t *query)
{
	char rule[ruleLength]={0}, rec[256]={0};
	char *ruleNameID, *ruleName, *ruleEnable, *ip, *protocol, *privatePort, *publicPort, *action, *inboundFilter, *schedule;
	char iptablesRuleName[64]={0},oldRuleName[32]={0}, oldruleEnable[4]={0}, oldschedule[32]={0}, oldinboundFilter[32]={0}, oldipaddress[32]={0}, oldprivatePort[8]={0}, oldpublicPort[8]={0}, oldprotocol[8]={0};
	char orgRuleName[32]={0};  //original ruleName
	int proto, oldproto;
	int rule_count;
	char *rules = nvram_bufget(RT2860_NVRAM, "SinglePortForwardRules");
	
	action = websGetVar(wp, T("action"), T(""));
	ruleNameID = websGetVar(wp, T("ruleNameID"), T("0"));
	ruleEnable = websGetVar(wp, T("ruleEnable"), T("0"));
	ruleName = websGetVar(wp, T("ruleName"), T(""));
	ip = websGetVar(wp, T("ipaddress"), T(""));
	protocol = websGetVar(wp, T("protocol"), T(""));
	privatePort = websGetVar(wp, T("privatePort"), T(""));
	publicPort = websGetVar(wp, T("publicPort"), T(""));
	inboundFilter = websGetVar(wp, T("inboundFilter"), T("ACCEPT"));
	schedule = websGetVar(wp, T("schedule"), T("Always"));
	
	checkWebServicePort(publicPort);
	
	//printf("jacky - action:%s, ruleNameID:%s, ruleEnable:%s, ruleName:%s, ip:%s, protocol:%s, privatePort:%s, publicPort:%s, inboundFilter:%s, schedule:%s\n", action, ruleNameID, ruleEnable, ruleName, ip, protocol, privatePort, publicPort, inboundFilter, schedule);
	printf("salim - ruleName:%s,schedule:%s\n", ruleName,schedule);

	if (!strcmp(action, "edit"))
	{
		int *deleArray;
		int i=0, j=0;
		rule_count = getRuleNums(rules);
		if(!rule_count)
			return;
		deleArray = (int *)malloc(rule_count * sizeof(int));
		if (ruleNameID)
		{
			deleArray[j++] = atoi(ruleNameID);
		}
		
		//get original rule name
		if (getNthValueSafe(atoi(ruleNameID), rules, ';', rec, sizeof(rec)) != -1 )
		{
			//printf("jacky - ruleNameID=%d rec:%s\n", atoi(ruleNameID), rec);
			if((getNthValueSafe(0, rec, ',', orgRuleName, sizeof(orgRuleName)) == -1)){
			}
		}
		
		deleteNthValueMulti(deleArray, rule_count, rules, ';');
		free(deleArray);
		
		delScheduleToNvram("SinglePortForwardRules", orgRuleName, ruleName, "true", "");
	}
	
	if(! strcmp(protocol, "TCP"))
		proto = PROTO_TCP;
	else if( !strcmp(protocol, "UDP"))
		proto = PROTO_UDP;
	else if( !strcmp(protocol, "TCP&UDP"))
		proto = PROTO_TCP_UDP;
	else
		return;


	//Tom.Hung 3-Dec-2008, Check if rule duplication.
	if(rules && strlen(rules)) {
		int i=0;
		while(getNthValueSafe(i++, rules, ';', rec, sizeof(rec)) != -1 ){
			if((getNthValueSafe(0, rec, ',', oldRuleName, sizeof(oldRuleName)) == -1)){
				continue;
			}
			if((getNthValueSafe(1, rec, ',', oldruleEnable, sizeof(oldruleEnable)) == -1)){
				continue;
			}
			if((getNthValueSafe(2, rec, ',', oldschedule, sizeof(oldschedule)) == -1)){
				continue;
			}
			if((getNthValueSafe(3, rec, ',', oldinboundFilter, sizeof(oldinboundFilter)) == -1)){
				continue;
			}
			if((getNthValueSafe(4, rec, ',', oldipaddress, sizeof(oldipaddress)) == -1)){
				continue;
			}
			if((getNthValueSafe(5, rec, ',', oldprivatePort, sizeof(oldprivatePort)) == -1)){
				continue;
			}
			if((getNthValueSafe(6, rec, ',', oldpublicPort, sizeof(oldpublicPort)) == -1)){
				continue;
			}
			if((getNthValueSafe(7, rec, ',', oldprotocol, sizeof(oldprotocol)) == -1)){
				continue;
			}
			oldproto = atoi(oldprotocol);
			switch(oldproto){
				case PROTO_TCP:
				case PROTO_UDP:
				case PROTO_TCP_UDP:
					break;
				default:
					continue;
			}

			if(!strcmp(oldipaddress, ip) && oldproto==proto && !strcmp(oldprivatePort, privatePort) && !strcmp(oldpublicPort, publicPort)) {
				websWrite(wp, T("<script language=\"JavaScript\" type=\"text/javascript\">"));
				websWrite(wp, T("alert(\"This IP protocol is already set in other rule.\");"));
				websWrite(wp, T("location.href = \"/advanced/single_port.asp\";"));
				websWrite(wp, T("</script>"));
				websDone(wp, 200);
				return 0;
			}
		}
	}
	//Tom.Hung 3-Dec-2008



	if(rules && strlen( rules) ) {
		snprintf(rule, sizeof(rule), "%s;%s,%s,%s,%s,%s,%s,%s,%d",  rules, ruleName, ruleEnable, schedule, inboundFilter, ip, privatePort, publicPort, proto);
	}
	else {
		snprintf(rule, sizeof(rule), "%s,%s,%s,%s,%s,%s,%s,%d", ruleName, ruleEnable, schedule, inboundFilter, ip, privatePort, publicPort, proto);
	}
	
	nvram_bufset(RT2860_NVRAM, "SinglePortForwardRules", rule);
	if (!strcmp(ruleEnable, "on"))
		addScheduleToNvram("SinglePortForwardRules", orgRuleName, ruleName, schedule);
	nvram_commit(RT2860_NVRAM);
	
	if (!strcmp(schedule, "Always")) {
		//iptablesSinglePortForwardRun(iptablesRuleName, ruleName, ruleEnable);
		iptablesRuleFlush("SPort");
		PortForwardAllRun("SinglePortForwardRules", "SPort");
	}

	addScheduleToCron();
	
	//websRedirect(wp, "/advanced/single_port.asp");
	websRedirect(wp, "/applied.asp?url=/advanced/single_port.asp");
}

static void PortRangeForwardAdd(webs_t wp, char_t *path, char_t *query)
{
	char rule[ruleLength], rec[256]={0};
	char *ruleNameID, *ruleName, *ruleEnable, *ip, *protocol, *TCPPort, *UDPPort, *action, *inboundFilter, *schedule;
	char iptablesRuleName[64]={0}, oldRuleName[32]={0};
	//char *PortForwardRules;
	//int proto;
	int rule_count;
	char *rules = nvram_bufget(RT2860_NVRAM, "PortRangeForwardRules");
	
	
	action = websGetVar(wp, T("action"), T(""));
	ruleNameID = websGetVar(wp, T("ruleNameID"), T("0"));
	ruleEnable = websGetVar(wp, T("ruleEnable"), T("0"));
	ruleName = websGetVar(wp, T("ruleName"), T(""));
	ip = websGetVar(wp, T("ipaddress"), T(""));
	TCPPort = websGetVar(wp, T("TCPPorts"), T(""));
	UDPPort = websGetVar(wp, T("UDPPorts"), T(""));
	inboundFilter = websGetVar(wp, T("inboundFilter"), T("ACCEPT"));
	schedule = websGetVar(wp, T("schedule"), T("Always"));
	
	checkWebServicePort(TCPPort);
	if (!strcmp(action, "edit"))
	{
		int *deleArray;
		int i=0, j=0;
		rule_count = getRuleNums(rules);
		if(!rule_count)
			return;
		deleArray = (int *)malloc(rule_count * sizeof(int));
		if (ruleNameID)
		{
			deleArray[j++] = atoi(ruleNameID);
		}
		
		//get old rule name
		if (getNthValueSafe(atoi(ruleNameID), rules, ';', rec, sizeof(rec)) != -1 )
		{
			//printf("jacky - ruleNameID=%d rec:%s\n", atoi(ruleNameID), rec);
			if((getNthValueSafe(0, rec, ',', oldRuleName, sizeof(oldRuleName)) == -1)){
			}
		}
		//printf("jacky - oldRuleName=%s, newRuleName=%s\n", oldRuleName, ruleName);
		
		deleteNthValueMulti(deleArray, rule_count, rules, ';');
		free(deleArray);
		
		delScheduleToNvram("PortRangeForwardRules", oldRuleName, ruleName, "true", "");
	}
	
	if(rules && strlen( rules) ) {
		snprintf(rule, sizeof(rule), "%s;%s,%s,%s,%s,%s,'%s','%s'",  rules, ruleName, ruleEnable, schedule, inboundFilter, ip, TCPPort, UDPPort);
	}
	else {
		snprintf(rule, sizeof(rule), "%s,%s,%s,%s,%s,'%s','%s'", ruleName, ruleEnable, schedule, inboundFilter, ip, TCPPort, UDPPort);
	}
	
	nvram_bufset(RT2860_NVRAM, "PortRangeForwardRules", rule);
	if (!strcmp(ruleEnable, "on"))
		addScheduleToNvram("PortRangeForwardRules", oldRuleName, ruleName, schedule);
	nvram_commit(RT2860_NVRAM);
	
	if (!strcmp(schedule, "Always")) {
		//iptablesPortRangeForwardRun(iptablesRuleName, ruleName, ruleEnable);
		iptablesRuleFlush("MPort");
		PortForwardAllRun("PortRangeForwardRules", "MPort");
	}

	addScheduleToCron();
	
	//websRedirect(wp, "/advanced/port_range.asp");
	websRedirect(wp, "/applied.asp?url=/advanced/port_range.asp");
}

static void inboundFilterAdd(webs_t wp, char_t *path, char_t *query)
{
	char rule[ruleLength], rec[256]={0};
	char *ruleNameID, *ruleName, *ruleEnable, *ip, *action;
	char iptablesRuleName[64]={0}, ruleEnableInput[4]={0}, oldRuleName[32]={0};
	int rule_count;
	char *rules = nvram_bufget(RT2860_NVRAM, "InboundFilterRules");
	
	action = websGetVar(wp, T("action"), T(""));
	ruleNameID = websGetVar(wp, T("ruleNameID"), T("0"));
	ruleEnable = websGetVar(wp, T("ruleEnable"), T("0"));
	ruleName = websGetVar(wp, T("ruleName"), T(""));
	ip = websGetVar(wp, T("ipaddress"), T(""));
	
	strcpy(iptablesRuleName, "IF_");
	strcat(iptablesRuleName, ruleName);
	
	if (!strcmp(action, "edit"))
	{
		int *deleArray;
		int i=0, j=0;
		rule_count = getRuleNums(rules);
		if(!rule_count)
			return;
		deleArray = (int *)malloc(rule_count * sizeof(int));
		if (ruleNameID)
		{
			deleArray[j++] = atoi(ruleNameID);
		}
		
		//get old rule name
		if (getNthValueSafe(atoi(ruleNameID), rules, ';', rec, sizeof(rec)) != -1 )
		{
			//printf("jacky - ruleNameID=%d rec:%s\n", atoi(ruleNameID), rec);
			if((getNthValueSafe(0, rec, ',', oldRuleName, sizeof(oldRuleName)) == -1)){
			}
		}
		strcpy(iptablesRuleName, "IF_");
		strcat(iptablesRuleName, oldRuleName);
		//iptablesFilterFlush("INPUT", iptablesRuleName);
		//iptablesFilterFlush("FORWARD", iptablesRuleName);
		
		deleteNthValueMulti(deleArray, rule_count, rules, ';');
		free(deleArray);
	}
	
	if (!strcmp(ruleEnable, "1"))
		strcpy(ruleEnableInput, "on");
	else
		strcpy(ruleEnableInput, "0");
	
	if(rules && strlen( rules) ) {
		snprintf(rule, sizeof(rule), "%s;%s,%s,%s",  rules, ruleName, ruleEnableInput, ip);
	}
	else {
		snprintf(rule, sizeof(rule), "%s,%s,%s", ruleName, ruleEnableInput, ip);
	}
	
	nvram_bufset(RT2860_NVRAM, "InboundFilterRules", rule);
	nvram_commit(RT2860_NVRAM);

	//iptablesFlush(iptablesRuleName);
	// call iptables
	//iptablesInboundFilterRun(iptablesRuleName, ruleName, ruleEnableInput);
	if (!strcmp(action, "edit"))
		firewall_init();

	//websRedirect(wp, "/advanced/inbound_filter.asp");
	websRedirect(wp, "/applied.asp?url=/advanced/inbound_filter.asp");
}

static void AccessControlAdd(webs_t wp, char_t *path, char_t *query)
{
	char rule[ruleLength], strTemp[32], ruleEnableInput[4], rec[256];
	char *action, *AccessControlEnable, *ruleNameID, *ruleName, *ruleEnable, *schedule, *ipaddress_start, *ipaddress_end, *ruleDefine;
	char *TCPPorts, *UDPPorts;
	char *portRangeStart, *portRangeEnd, *pos;
	char iptablesRuleName[64], oldRuleName[32];
	int rule_count, loopCount;
	int rc=0;
	
	char *rules = nvram_bufget(RT2860_NVRAM, "AccessControlPolicy");
	action = websGetVar(wp, T("action"), T(""));
	ruleNameID = websGetVar(wp, T("ruleNameID"), T("0"));
	AccessControlEnable = websGetVar(wp, T("AccessControlEnable"), T("0"));
	ruleEnable = websGetVar(wp, T("ruleEnable"), T("0"));
	ruleName = websGetVar(wp, T("ruleName"), T(""));
	schedule = websGetVar(wp, T("schedule"), T("Always"));
	ipaddress_start = websGetVar(wp, T("ipaddress_start"), T("0"));
	ipaddress_end = websGetVar(wp, T("ipaddress_end"), T("0"));
	ruleDefine = websGetVar(wp, T("ruleDefine"), T("0")); //ruleDefine=0: Special Service, ruleDefine=1: User Define.
		
	if (!strcmp(action, "edit"))
	{
		printf("jacky - AccessControlAdd in the edit.\n");
		//If action=edit, delete this rule, then add modify rule again.
		int *deleArray;
		int i=0, j=0;
		rule_count = getRuleNums(rules);
		if(!rule_count)
			return;
		deleArray = (int *)malloc(rule_count * sizeof(int));
		if (ruleNameID)
		{
			deleArray[j++] = atoi(ruleNameID);
		}
		
		//get old rule name
		if (getNthValueSafe(atoi(ruleNameID), rules, ';', rec, sizeof(rec)) != -1 )
		{
			if((getNthValueSafe(0, rec, ',', oldRuleName, sizeof(oldRuleName)) == -1)){
			}
		}
		/*strcpy(iptablesRuleName, "AC");
		//strcat(iptablesRuleName, oldRuleName);
		iptablesFlush("-t nat", "PREROUTING", iptablesRuleName);
		iptablesFlush("-t nat", "POSTROUTING", iptablesRuleName);*/
		
		deleteNthValueMulti(deleArray, rule_count, rules, ';');
		free(deleArray);
		
		delScheduleToNvram("AccessControlPolicy", oldRuleName, ruleName, "true", "");
	}
	else if (!strcmp(action, "saveStatus")) {
		printf("jacky - AccessControlAdd in the saveStatus.\n");
		nvram_set(RT2860_NVRAM, "AccessControlEnable", AccessControlEnable); //AccessControlEnable=0: disable, AccessControlEnable=1: enable
		//nvram_commit(RT2860_NVRAM);
		iptablesRuleFlush("AC");
		if (atoi(AccessControlEnable) == 1)
			PortForwardAllRun("AccessControlPolicy", "AC");
		//websRedirect(wp, "/advanced/access_control.asp");
		websRedirect(wp, "/applied.asp?url=/advanced/access_control.asp");
		return;
	}
	printf("jacky - AccessControlAdd not in the saveStatus.\n");
	
	if (!isIpValid(ipaddress_end))
		ipaddress_end = ipaddress_start;
	
	memset(iptablesRuleName, '\0', sizeof(iptablesRuleName));
	strcpy(iptablesRuleName, "AC");
	//strcat(iptablesRuleName, ruleName);
	
	//ruleDefine=0: Special Service, ruleDefine=1: User Define.
	pos = rule;
	//special service of format: ruleName, ipaddress_start-ipaddress_end, schedule, atoi(ruleDefine), wwwRule, smtpRule, pop3Rule, ftpRule, telnetRule, dnsRule, tcpRule, udpRule
	//user define of format: ruleName, ipaddress_start-ipaddress_end, schedule, atoi(ruleDefine), TCPPorts, UDPPorts
	if (atoi(ruleDefine) == 0) {
		char *wwwRule, *smtpRule, *pop3Rule, *ftpRule, *telnetRule, *dnsRule, *tcpRule, *udpRule;
		wwwRule = websGetVar(wp, T("wwwRule"), T("0"));
		smtpRule = websGetVar(wp, T("smtpRule"), T("0"));
		pop3Rule = websGetVar(wp, T("pop3Rule"), T("0"));
		ftpRule = websGetVar(wp, T("ftpRule"), T("0"));
		telnetRule = websGetVar(wp, T("telnetRule"), T("0"));
		dnsRule = websGetVar(wp, T("dnsRule"), T("0"));
		tcpRule = websGetVar(wp, T("tcpRule"), T("0"));
		udpRule = websGetVar(wp, T("udpRule"), T("0"));
		
		if(rules && strlen( rules) ) {
			rc = snprintf(pos, sizeof(rule), "%s;%s,%s,%s,%s-%s,%d", rules, ruleName, ruleEnable, schedule, ipaddress_start, ipaddress_end, atoi(ruleDefine));
			pos = pos + rc;
		}
		else {
			rc = snprintf(pos, sizeof(rule), "%s,%s,%s,%s-%s,%d", ruleName, ruleEnable, schedule, ipaddress_start, ipaddress_end, atoi(ruleDefine));
			pos = pos + rc;
		}
		snprintf(pos, sizeof(rule)-rc, ",%s,%s,%s,%s,%s,%s,%s,%s", wwwRule, smtpRule, pop3Rule, ftpRule, telnetRule, dnsRule, tcpRule, udpRule);
	}
	else if (atoi(ruleDefine) == 1) {
		TCPPorts = websGetVar(wp, T("TCPPorts"), T(""));
		UDPPorts = websGetVar(wp, T("UDPPorts"), T(""));
		if(rules && strlen( rules) ) {
			rc = snprintf(pos, sizeof(rule), "%s;%s,%s,%s,%s-%s,%d,'%s','%s'", rules, ruleName, ruleEnable, schedule, ipaddress_start, ipaddress_end, atoi(ruleDefine), TCPPorts, UDPPorts);
			pos = pos + rc;
		}
		else {
			rc = snprintf(pos, sizeof(rule), "%s,%s,%s,%s-%s,%d,'%s','%s'", ruleName, ruleEnable, schedule, ipaddress_start, ipaddress_end, atoi(ruleDefine), TCPPorts, UDPPorts);
			pos = pos + rc;
		}
	}
	
	nvram_bufset(RT2860_NVRAM, "AccessControlPolicy", rule);
	nvram_bufset(RT2860_NVRAM, "AccessControlEnable", AccessControlEnable); //AccessControlEnable=0: disable, AccessControlEnable=1: enable
	if (!strcmp(ruleEnable, "on"))
		addScheduleToNvram("AccessControlPolicy", oldRuleName, ruleName, schedule);
	nvram_commit(RT2860_NVRAM);
	
	if (!strcmp(schedule, "Always")) {
		//AccessControlRun(iptablesRuleName, ruleName, ruleEnable);
		iptablesRuleFlush("AC");
		PortForwardAllRun("AccessControlPolicy", "AC");
	}
	/*else
		addScheduleToCron("AccessControlPolicy", ruleName, "AC_", schedule);*/
	addScheduleToCron();
	
	//websRedirect(wp, "/advanced/access_control.asp");
	websRedirect(wp, "/applied.asp?url=/advanced/access_control.asp");
}

static void singlePortForwardDelete(webs_t wp, char_t *path, char_t *query)
{
	int i, j, rule_count;
	char_t name_buf[16]={0};
	char_t *value, *ruleName, *action;
	int *deleArray;
	char iptablesRuleName[64]={0};

	action = websGetVar(wp, T("action"), T(""));

	if (!strcmp(action, "delete"))
	{
    	char *rules = nvram_bufget(RT2860_NVRAM, "SinglePortForwardRules");
    	if(!rules || !strlen(rules) )
        	return;
    
		rule_count = getRuleNums(rules);
		if(!rule_count)
			return;
	
		deleArray = (int *)malloc(rule_count * sizeof(int));
	
		value = websGetVar(wp, T("ruleNameID"), T(""));	

		i=j=0;
		if(value){
			snprintf(name_buf, sizeof(name_buf), "ruleName%d", atoi(value));
			ruleName = websGetVar(wp, T(name_buf), T(""));
		
			deleArray[j++] = atoi(value);
			//iptablesRuleFlush("SPort");
			//PortForwardAllRun("SinglePortForwardRules", "SPort");
		}

		deleteNthValueMulti(deleArray, rule_count, rules, ';');
		free(deleArray);
		
		nvram_set(RT2860_NVRAM, "SinglePortForwardRules", rules);
		delScheduleToNvram("SinglePortForwardRules", ruleName, ruleName, "true", "");
		//nvram_commit(RT2860_NVRAM);
		/*iptablesRuleFlush("SPort");
		PortForwardAllRun("SinglePortForwardRules", "SPort");*/
		firewall_init();
	}
    //websRedirect(wp, "/advanced/single_port.asp");
    websRedirect(wp, "/applied.asp?url=/advanced/single_port.asp");
}

static void PortRangeForwardDelete(webs_t wp, char_t *path, char_t *query)
{
	int i, j, rule_count;
	char_t name_buf[16]={0};
	char_t *value, *ruleName, *action;
	int *deleArray;
	char iptablesRuleName[64]={0};

	action = websGetVar(wp, T("action"), T(""));

	if (!strcmp(action, "delete"))
	{
    	char *rules = nvram_bufget(RT2860_NVRAM, "PortRangeForwardRules");
    	if(!rules || !strlen(rules) )
        	return;
    
		rule_count = getRuleNums(rules);
		if(!rule_count)
			return;
	
		deleArray = (int *)malloc(rule_count * sizeof(int));
	
		value = websGetVar(wp, T("ruleNameID"), T(""));	

		i=j=0;
		if(value){
			snprintf(name_buf, 16, "ruleName%d", atoi(value));
			ruleName = websGetVar(wp, T(name_buf), T(""));
		
			deleArray[j++] = atoi(value);
			//iptablesRuleFlush("MPort");
			//PortForwardAllRun("PortRangeForwardRules", "MPort");
		}

		deleteNthValueMulti(deleArray, rule_count, rules, ';');
		free(deleArray);

		nvram_set(RT2860_NVRAM, "PortRangeForwardRules", rules);
		delScheduleToNvram("PortRangeForwardRules", ruleName, ruleName, "true", "");
		//nvram_commit(RT2860_NVRAM);
		iptablesRuleFlush("MPort");
		PortForwardAllRun("PortRangeForwardRules", "MPort");
	}
    //websRedirect(wp, "/advanced/port_range.asp");
    websRedirect(wp, "/applied.asp?url=/advanced/port_range.asp");
}

static void inboundFilterDelete(webs_t wp, char_t *path, char_t *query)
{
	int i, j, rule_count;
	char_t name_buf[16]={0};
	char_t *value, *ruleName, *action;
	int *deleArray;
	char iptablesRuleName[64]={0};

	action = websGetVar(wp, T("action"), T(""));
	
	if (!strcmp(action, "delete"))
	{
	    char *rules = nvram_bufget(RT2860_NVRAM, "InboundFilterRules");
    	if(!rules || !strlen(rules) )
        	return;
    
		rule_count = getRuleNums(rules);
		if(!rule_count)
			return;
	
		deleArray = (int *)malloc(rule_count * sizeof(int));
	
		value = websGetVar(wp, T("ruleNameID"), T(""));	
		i=j=0;
		if(value){
			snprintf(name_buf, 16, "ruleName%d", atoi(value));
			ruleName = websGetVar(wp, T(name_buf), T(""));
		
			deleArray[j++] = atoi(value);
		}

		deleteNthValueMulti(deleArray, rule_count, rules, ';');
		free(deleArray);
		nvram_bufset(RT2860_NVRAM, "InboundFilterRules", rules);
		nvram_commit(RT2860_NVRAM);
	}
    //websRedirect(wp, "/advanced/inbound_filter.asp");
    websRedirect(wp, "/applied.asp?url=/advanced/inbound_filter.asp");
}

static void AccessControlDelete(webs_t wp, char_t *path, char_t *query)
{
	int i, j, rule_count;
	char_t name_buf[16]={0};
	char_t *value, *ruleName, *action, *AccessControlEnable;
	int *deleArray;
	char iptablesRuleName[64]={0};

	action = websGetVar(wp, T("action"), T(""));
	AccessControlEnable = websGetVar(wp, T("hiddenACEnable"), T("0"));
	
	if (!strcmp(action, "delete"))
	{
	    char *rules = nvram_bufget(RT2860_NVRAM, "AccessControlPolicy");
    	if(!rules || !strlen(rules) )
        	return;
    
		rule_count = getRuleNums(rules);
		if(!rule_count)
			return;
	
		deleArray = (int *)malloc(rule_count * sizeof(int));
	
		value = websGetVar(wp, T("ruleNameID"), T(""));	
		i=j=0;
		if(value){
			snprintf(name_buf, 16, "ruleName%d", atoi(value));
			ruleName = websGetVar(wp, T(name_buf), T(""));
		
			deleArray[j++] = atoi(value);
			//iptablesRuleFlush("AC");
			//PortForwardAllRun("AccessControlPolicy", "AC");
		}

		deleteNthValueMulti(deleArray, rule_count, rules, ';');
		free(deleArray);

		nvram_bufset(RT2860_NVRAM, "AccessControlPolicy", rules);
		//nvram_bufset(RT2860_NVRAM, "AccessControlEnable", AccessControlEnable);
		nvram_commit(RT2860_NVRAM);
		delScheduleToNvram("AccessControlPolicy", ruleName, ruleName, "true", "");
		iptablesRuleFlush("AC");
		PortForwardAllRun("AccessControlPolicy", "AC");
	}
    //websRedirect(wp, "/advanced/access_control.asp");
    websRedirect(wp, "/applied.asp?url=/advanced/access_control.asp");
}

static int showSinglePortForward(int eid, webs_t wp, int argc, char_t **argv)
{
	int i=0;
	char rec[256]={0}, scheduleInfo[64]={0};
	int privatePort_int, publicPort_int, proto;
	char ruleEnable[4]={0}, ruleName[32]={0}, ipaddress[32]={0}, privatePort[8]={0}, publicPort[8]={0}, protocol[8]={0};
	char schedule[32]={0}, inboundFilter[32]={0};
	char protocolOutputStr[8]={0};
	char *rules = nvram_bufget(RT2860_NVRAM, "SinglePortForwardRules");
	int listnumber=0;

	if(!rules)
		return 0;
	if(!strlen(rules))
		return 0;

	while(getNthValueSafe(i++, rules, ';', rec, sizeof(rec)) != -1 ){
		// get Rule Name
		if((getNthValueSafe(0, rec, ',', ruleName, sizeof(ruleName)) == -1)){
			continue;
		}
		
		if((getNthValueSafe(1, rec, ',', ruleEnable, sizeof(ruleEnable)) == -1)){
			continue;
		}
		if((getNthValueSafe(2, rec, ',', schedule, sizeof(schedule)) == -1)){
			continue;
		}
		if((getNthValueSafe(3, rec, ',', inboundFilter, sizeof(inboundFilter)) == -1)){
			continue;
		}
		if((getNthValueSafe(4, rec, ',', ipaddress, sizeof(ipaddress)) == -1)){
			continue;
		}
		if(!isIpValid(ipaddress)){
			continue;
		}

		// get port range "from"
		if((getNthValueSafe(5, rec, ',', privatePort, sizeof(privatePort)) == -1)){
			continue;
		}
		if( (privatePort_int = atoi(privatePort)) == 0 || privatePort_int > 65535){
			continue;
		}

		// get port range "to"
		if((getNthValueSafe(6, rec, ',', publicPort, sizeof(publicPort)) == -1)){
			continue;
		}
		if( (publicPort_int = atoi(publicPort)) > 65535){
			continue;
		}

		// get protocol
		if((getNthValueSafe(7, rec, ',', protocol, sizeof(protocol)) == -1)){
			continue;
		}
		
		proto = atoi(protocol);
		switch(proto){
			case PROTO_TCP:
			case PROTO_UDP:
			case PROTO_TCP_UDP:
				break;
			default:
				continue;
		}

		memset(scheduleInfo, '\0', sizeof(scheduleInfo));
		getScheduleInfo(schedule, scheduleInfo, sizeof(scheduleInfo));

		websWrite(wp, T("<tr id=\"drawColor%d\">\n"), i-1);
		// output No.
		if (!strcmp(ruleEnable, "on"))
			websWrite(wp, T("<td align=left style=\"word-break : break-all;\"> %d&nbsp; <input type=\"checkbox\" id=\"delRule%d\" name=\"delRule%d\" disabled=\"disabled\" checked=\"checked\"> </td>"), i, i-1, i-1);
		else
			websWrite(wp, T("<td align=left style=\"word-break : break-all;\"> %d&nbsp; <input type=\"checkbox\" id=\"delRule%d\" name=\"delRule%d\" disabled=\"disabled\"> </td>"), i, i-1, i-1);
		
		// Rule Name
		websWrite(wp, T("<input type=\"hidden\" id=\"scheduleInfo%d\" name=\"scheduleInfo%d\" value=\"%s\">"), i-1, i-1, scheduleInfo);
		websWrite(wp, T("<input type=\"hidden\" id=\"ruleName%d\" name=\"ruleName%d\" value=\"%s\">"), i-1, i-1, ruleName);
		if(strlen(ruleName))
			websWrite(wp, T("<td align=center style=\"word-break : break-all;\">%s</td>"), ruleName);
		else
			websWrite(wp, T("<td align=center style=\"word-break : break-all;\">&nbsp;</td>"));
		
		// output IP address
		websWrite(wp, T("<td id=\"ipaddress%d\" align=center style=\"word-break : break-all;\">%s</td>"), i-1, ipaddress);
		
		// output Protocol
        switch(proto){
            case PROTO_TCP:
            	websWrite(wp, T("<input type=\"hidden\" id=\"protocol%d\" value=\"TCP\">"), i-1);
				//websWrite(wp, T("<td align=center style=\"word-break : break-all;\">TCP</td>"), i-1);
				strcpy(protocolOutputStr, "TCP");
				break;
            case PROTO_UDP:
            	websWrite(wp, T("<input type=\"hidden\" id=\"protocol%d\" value=\"UDP\">"), i-1);
				//websWrite(wp, T("<td align=center style=\"word-break : break-all;\">UDP</td>"), i-1);
				strcpy(protocolOutputStr, "UDP");
				break;
            case PROTO_TCP_UDP:
            	websWrite(wp, T("<input type=\"hidden\" id=\"protocol%d\" value=\"TCP+UDP\">"), i-1);
				//websWrite(wp, T("<td align=center style=\"word-break : break-all;\">TCP+UDP</td>"), i-1);
				strcpy(protocolOutputStr, "TCP+UDP");
				break;
		}
		websWrite(wp, T("<input type=\"hidden\" id=\"publicPort%d\" value=\"%d\">"), i-1, publicPort_int);
		//websWrite(wp, T("<td align=center style=\"word-break : break-all;\">%d</td>"), i-1, publicPort_int);
		websWrite(wp, T("<input type=\"hidden\" id=\"privatePort%d\" value=\"%d\">"), i-1, privatePort_int);
		websWrite(wp, T("<td id=\"proto%d\" align=center style=\"word-break : break-all;\">%s,%d/%d</td>"), i-1, protocolOutputStr, publicPort_int, privatePort_int);
		websWrite(wp, T("<input type=\"hidden\" id=\"inboundFilter%d\" value=\"%s\">"), i-1, inboundFilter);
		if (!strcmp(inboundFilter, "ACCEPT"))
			websWrite(wp, T("<td align=center style=\"word-break : break-all;\">%s</td>"), "Allow All");
		else if (!strcmp(inboundFilter, "DROP"))
			websWrite(wp, T("<td align=center style=\"word-break : break-all;\">%s</td>"), "Deny All");
		else
			websWrite(wp, T("<td align=center style=\"word-break : break-all;\">%s</td>"), inboundFilter);
		websWrite(wp, T("<td id=\"schedule%d\" align=center style=\"word-break : break-all;\">%s</td>"), i-1, schedule);
		websWrite(wp, T("<td id=\"edit%d\" align=center><a href=\"javascript:submit_apply('edit', %d);\"><img src=\"/Images/img_edit.gif\"></a></td>"), i-1, i-1); //rulenameID
		websWrite(wp, T("<td id=\"del%d\" align=center><a href=\"javascript:submit_apply('delete', %d);\"><img src=\"/Images/img_delete.gif\"></a></td>"), i-1, i-1); //rulenameID
		websWrite(wp, T("</tr>\n"));
		listnumber++;
	}	  
	websWrite(wp, T("<input type=\"hidden\" id=\"rulenum\" name=\"rulenum\" value=\"%d\">\n"), listnumber);
	return 0;	
}

static int showPortRangeForward(int eid, webs_t wp, int argc, char_t **argv)
{
	int i=0;
	int privatePort_int, publicPort_int, proto;
	char *begin, *end;
	char ruleEnable[4]={0}, ruleName[32]={0}, ipaddress[32]={0}, TCPPort[128]={0}, UDPPort[128]={0};
	char rec[128]={0}, schedule[32]={0}, scheduleInfo[64]={0}, inboundFilter[32]={0};
	char *rules = nvram_bufget(RT2860_NVRAM, "PortRangeForwardRules");
	int listnumber=0;

	if(!rules)
		return 0;
	if(!strlen(rules))
		return 0;

	/* format is :
	 * [ip],[port_from],[port_to],[protocol],[comment],;
	 */
	while(getNthValueSafe(i++, rules, ';', rec, 256) != -1 ){
		if((getNthValueSafe(0, rec, ',', ruleName, sizeof(ruleName)) == -1)){
			continue;
		}
		
		if((getNthValueSafe(1, rec, ',', ruleEnable, sizeof(ruleEnable)) == -1)){
			continue;
		}
		if((getNthValueSafe(2, rec, ',', schedule, sizeof(schedule)) == -1)){
			continue;
		}
		if((getNthValueSafe(3, rec, ',', inboundFilter, sizeof(inboundFilter)) == -1)){
			continue;
		}
		if((getNthValueSafe(4, rec, ',', ipaddress, sizeof(ipaddress)) == -1)){
			continue;
		}
		if(!isIpValid(ipaddress)){
			continue;
		}
		memset(TCPPort, '\0', sizeof(TCPPort));
		memset(UDPPort, '\0', sizeof(UDPPort));
		begin = strchr(rec, '\'');
		end = strchr(begin+1, '\'');
		memcpy(TCPPort, (begin+1), ((end-1) - (begin+1) + 1));
		begin = strchr(end+1, '\'');
		end = strchr(begin+1, '\'');
		memcpy(UDPPort, (begin+1), ((end-1) - (begin+1) + 1));
		
		memset(scheduleInfo, '\0', sizeof(scheduleInfo));
		getScheduleInfo(schedule, scheduleInfo, sizeof(scheduleInfo));

		websWrite(wp, T("<tr id=\"drawColor%d\">\n"), i-1);
		// output No.
		if (!strcmp(ruleEnable, "on"))
			websWrite(wp, T("<td align=left> %d&nbsp; <input type=\"checkbox\" id=\"delRule%d\" name=\"delRule%d\" disabled=\"disabled\" checked=\"checked\"> </td>"), i, i-1, i-1);
		else
			websWrite(wp, T("<td align=left> %d&nbsp; <input type=\"checkbox\" id=\"delRule%d\" name=\"delRule%d\" disabled=\"disabled\"> </td>"), i, i-1, i-1);
		
		// Rule Name
		//websWrite(wp, T("<input type=\"hidden\" name=\"ruleNameID\" value=\"%d\">"), i-1);
		websWrite(wp, T("<input type=\"hidden\" id=\"scheduleInfo%d\" name=\"scheduleInfo%d\" value=\"%s\">"),i-1, i-1, scheduleInfo);
		websWrite(wp, T("<input type=\"hidden\" id=\"ruleName%d\" name=\"ruleName%d\" value=\"%s\">"),i-1, i-1, ruleName);
		if(strlen(ruleName))
			websWrite(wp, T("<td align=center style=\"word-break : break-all;\">%s</td>"), ruleName);
		else
			websWrite(wp, T("<td align=center>&nbsp;</td>"));
		
		// output IP address
		websWrite(wp, T("<td id=\"ipaddress%d\" align=center style=\"word-break : break-all;\">%s</td>"), i-1, ipaddress);
		websWrite(wp, T("<input type=\"hidden\" id=\"TCPPorts%d\" name=\"TCPPorts%d\" value=\"%s\">"), i-1, i-1, TCPPort);
		websWrite(wp, T("<input type=\"hidden\" id=\"UDPPorts%d\" name=\"UDPPorts%d\"value=\"%s\">"), i-1, i-1, UDPPort);
		websWrite(wp, T("<td align=center style=\"word-break : break-all;\">%s/%s</td>"), TCPPort, UDPPort);
		//websWrite(wp, T("<td id=\"UDPPorts%d\" align=center style=\"word-break : break-all;\">%s</td>"), i-1, UDPPort);
		websWrite(wp, T("<input type=\"hidden\" id=\"inboundFilter%d\" name=\"inboundFilter%d\" value=\"%s\">"), i-1, i-1, inboundFilter);
		if (!strcmp(inboundFilter, "ACCEPT"))
			websWrite(wp, T("<td align=center style=\"word-break : break-all;\">%s</td>"), "Allow All");
		else if (!strcmp(inboundFilter, "DROP"))
			websWrite(wp, T("<td align=center style=\"word-break : break-all;\">%s</td>"), "Deny All");
		else
			websWrite(wp, T("<td align=center style=\"word-break : break-all;\">%s</td>"), inboundFilter);
		//websWrite(wp, T("<td id=\"inboundShow%d\" align=center style=\"word-break : break-all;\">%s</td>"), i-1, "Allow All");
		websWrite(wp, T("<td id=\"schedule%d\" align=center style=\"word-break : break-all;\">%s</td>"), i-1, schedule);
		websWrite(wp, T("<td align=center><a href=\"javascript:submit_apply('edit', %d);\"><img src=\"/Images/img_edit.gif\"></a></td>"), i-1); //rulenameID
		websWrite(wp, T("<td align=center><a href=\"javascript:submit_apply('delete', %d);\"><img src=\"/Images/img_delete.gif\"></a></td>"), i-1); //rulenameID
		//websWrite(wp, T("<input type=\"hidden\" name=\"hiddenButton\" value=\"\">"));
		websWrite(wp, T("</tr>\n"));
		listnumber++;
	}	  
	websWrite(wp, T("<input type=\"hidden\" id=\"rulenum\" name=\"rulenum\" value=\"%d\">\n"), listnumber);
	return 0;	
}

static int showInboundFilter(int eid, webs_t wp, int argc, char_t **argv)
{
	int i=0;
	int privatePort_int, publicPort_int, proto;
	char ruleEnable[4]={0}, ruleName[32]={0}, ipaddress[32]={0}, TCPPort[32]={0}, UDPPort[32]={0};
	char rec[128]={0};
	char *rules = nvram_bufget(RT2860_NVRAM, "InboundFilterRules");
	int listnumber=0;

	if(!rules)
		return 0;
	if(!strlen(rules))
		return 0;

	while(getNthValueSafe(i++, rules, ';', rec, 256) != -1 ){
		if((getNthValueSafe(0, rec, ',', ruleName, sizeof(ruleName)) == -1)){
			continue;
		}
		if((getNthValueSafe(1, rec, ',', ruleEnable, 4) == -1)){
			continue;
		}
		
		// get ip address
		if((getNthValueSafe(2, rec, ',', ipaddress, 32) == -1)){
			continue;
		}
		if(!isIpValid(ipaddress)){
			continue;
		}

		websWrite(wp, T("<tr>\n"));
		// output No.
		/*if (!strcmp(ruleEnable, "on"))
			websWrite(wp, T("<td align=left> %d&nbsp; <input type=\"checkbox\" name=\"delRule%d\" disabled=\"disabled\" checked=\"checked\"> </td>"), i, i-1 );
		else
			websWrite(wp, T("<td align=left> %d&nbsp; <input type=\"checkbox\" name=\"delRule%d\" disabled=\"disabled\"> </td>"), i, i-1 );
		*/
		// Rule Name
		//websWrite(wp, T("<input type=\"hidden\" name=\"ruleNameID\" value=\"%d\">"), i-1);
		websWrite(wp, T("<input type=\"hidden\" id=\"ruleName%d\" name=\"ruleName%d\" value=\"%s\">"), i-1, i-1, ruleName);
		if ((checkDelete("SinglePortForwardRules", ruleName) == 1) && (checkDelete("PortRangeForwardRules", ruleName) == 1))
			websWrite(wp, T("<input type=\"hidden\" id=\"doDelete%d\" name=\"doDelete%d\" value=\"%s\">"), i-1, i-1, "true");
		else
			websWrite(wp, T("<input type=\"hidden\" id=\"doDelete%d\" name=\"doDelete%d\" value=\"%s\">"), i-1, i-1, "false");
		
		if(strlen(ruleName))
			websWrite(wp, T("<td align=center>%s</td>"), ruleName);
		else
			websWrite(wp, T("<td align=center>&nbsp;</td>"));
		
		if (!strcmp(ruleEnable, "on"))
			websWrite(wp, T("<td id=\"ruleEnable%d\" align=center>%s</td>"), i-1, "Deny");
		else if (!strcmp(ruleEnable, "0"))
			websWrite(wp, T("<td id=\"ruleEnable%d\" align=center>%s</td>"), i-1, "Allow");
		
		// output IP address
		websWrite(wp, T("<td id=\"ipaddress%d\" align=center>%s</td>"), i-1, ipaddress);
		websWrite(wp, T("<td align=center><a href=\"javascript:submit_apply('edit', %d);\"><img src=\"/Images/img_edit.gif\"></a></td>"), i-1); //rulenameID
		websWrite(wp, T("<td align=center><a href=\"javascript:submit_apply('delete', %d);\"><img src=\"/Images/img_delete.gif\"></a></td>"), i-1); //rulenameID
		websWrite(wp, T("</tr>\n"));
		listnumber++;
	}	  
	websWrite(wp, T("<input type=\"hidden\" id=\"rulenum\" name=\"rulenum\" value=\"%d\">\n"), listnumber);
	return 0;	
}

static int showAccessControl(int eid, webs_t wp, int argc, char_t **argv)
{
	int i=0, loopCount;
	int privatePort_int, publicPort_int, proto;
	char ruleEnable[4]={0}, ruleName[32]={0}, ipaddress[64]={0}, schedule[23]={0}, ruleDefine[4]={0}, tempArray[16]={0};
	char TCPPorts[32]={0}, UDPPorts[32]={0};
	char *begin, *end;
	char rec[128]={0};
	//char *AccessControlEnable = nvram_bufget(RT2860_NVRAM, "AccessControlEnable");
	char *rules = nvram_bufget(RT2860_NVRAM, "AccessControlPolicy");
	int listnumber=0;
	
	if(!rules)
		return 0;
	if(!strlen(rules))
		return 0;
	//special service of format: ruleName, ipaddress_start-ipaddress_end, schedule, atoi(ruleDefine), wwwRule, smtpRule, pop3Rule, ftpRule, telnetRule, dnsRule, tcpRule, udpRule
	//user define of format: ruleName, ipaddress_start-ipaddress_end, schedule, atoi(ruleDefine), atoi(protocol), portRangeStart1-portRangeEnd1, portRangeStart2-portRangeEnd2, ..., portRangeStart5-portRangeEnd5
	while(getNthValueSafe(i++, rules, ';', rec, sizeof(rec)) != -1 ){
		if((getNthValueSafe(0, rec, ',', ruleName, sizeof(ruleName)) == -1)){
			continue;
		}
		
		if((getNthValueSafe(1, rec, ',', ruleEnable, sizeof(ruleEnable)) == -1)){
			continue;
		}
		
		if((getNthValueSafe(2, rec, ',', schedule, sizeof(schedule)) == -1)){
			continue;
		}
		
		if((getNthValueSafe(3, rec, ',', ipaddress, sizeof(ipaddress)) == -1)){
			continue;
		}
		
		// get ip ruleDefine
		if((getNthValueSafe(4, rec, ',', ruleDefine, sizeof(ruleDefine)) == -1)){
			continue;
		}

		websWrite(wp, T("<tr>\n"));
		// output No.
		if (!strcmp(ruleEnable, "on"))
			websWrite(wp, T("<td align=left style=\"word-break : break-all;\"> %d&nbsp; <input type=\"checkbox\" id=\"delRule%d\" name=\"delRule%d\" disabled=\"disabled\" checked=\"checked\"> </td>"), i, i-1, i-1 );
		else
			websWrite(wp, T("<td align=left style=\"word-break : break-all;\"> %d&nbsp; <input type=\"checkbox\" id=\"delRule%d\" name=\"delRule%d\" disabled=\"disabled\"> </td>"), i, i-1, i-1 );
			websWrite(wp, T("<input type=\"hidden\" id=\"ruleName%d\" name=\"ruleName%d\" value=\"%s\">"),i-1, i-1, ruleName);
		//policy name		
		if(strlen(ruleName))
			websWrite(wp, T("<td align=center>%s</td>"),ruleName);
		else
			websWrite(wp, T("<td align=center>&nbsp;</td>"));
		//schedule
		websWrite(wp, T("<td id=\"schedule%d\" align=center style=\"word-break : break-all;\">%s</td>"), i-1, schedule);
		// output IP address
		websWrite(wp, T("<td id=\"ipaddress%d\" align=center>%s</td>"), i-1, ipaddress);
		websWrite(wp, T("<td align=center><a href=\"javascript:submit_apply('edit', %d);\"><img src=\"/Images/img_edit.gif\"></a></td>"), i-1); //rulenameID
		websWrite(wp, T("<td align=center><a href=\"javascript:submit_apply('delete', %d);\"><img src=\"/Images/img_delete.gif\"></a></td>"), i-1); //rulenameID
		websWrite(wp, T("<input type=\"hidden\" id=\"ruleDefine%d\" name=\"ruleDefine%d\" value=\"%d\">"), i-1,i-1, atoi(ruleDefine));
		//ruleDefine=0: speical service, ruleDefine=1: user define.
		if (atoi(ruleDefine) == 0) {
			if((getNthValueSafe(5, rec, ',', tempArray, 16) == -1)){	continue; }
			websWrite(wp, T("<input type=\"hidden\" id=\"wwwRule%d\" name=\"wwwRule%d\" value=\"%s\">"), i-1, i-1, tempArray); //wwwRule
			if((getNthValueSafe(6, rec, ',', tempArray, 16) == -1)){	continue; }
			websWrite(wp, T("<input type=\"hidden\" id=\"smtpRule%d\" name=\"smtpRule%d\" value=\"%s\">"), i-1, i-1, tempArray); //smtpRule
			if((getNthValueSafe(7, rec, ',', tempArray, 16) == -1)){	continue; }
			websWrite(wp, T("<input type=\"hidden\" id=\"pop3Rule%d\" name=\"pop3Rule%d\" value=\"%s\">"), i-1, i-1, tempArray); //pop3Rule
			if((getNthValueSafe(8, rec, ',', tempArray, 16) == -1)){	continue; }
			websWrite(wp, T("<input type=\"hidden\" id=\"ftpRule%d\" name=\"ftpRule%d\" value=\"%s\">"), i-1, i-1, tempArray); //ftpRule
			if((getNthValueSafe(9, rec, ',', tempArray, 16) == -1)){	continue; }
			websWrite(wp, T("<input type=\"hidden\" id=\"telnetRule%d\" name=\"telnetRule%d\" value=\"%s\">"), i-1, i-1, tempArray); //telnetRule
			if((getNthValueSafe(10, rec, ',', tempArray, 16) == -1)){	continue; }
			websWrite(wp, T("<input type=\"hidden\" id=\"dnsRule%d\" name=\"dnsRule%d\" value=\"%s\">"), i-1, i-1, tempArray); //dnsRule
			if((getNthValueSafe(11, rec, ',', tempArray, 16) == -1)){ continue; }
			websWrite(wp, T("<input type=\"hidden\" id=\"tcpRule%d\" name=\"tcpRule%d\" value=\"%s\">"), i-1, i-1, tempArray); //tcpRule
			if((getNthValueSafe(12, rec, ',', tempArray, 16) == -1)){ continue; }
			websWrite(wp, T("<input type=\"hidden\" id=\"udpRule%d\" name=\"udpRule%d\" value=\"%s\">"), i-1, i-1, tempArray); //udpRule
		}
		else if (atoi(ruleDefine) == 1) {
			memset(TCPPorts, '\0', sizeof(TCPPorts));
			memset(UDPPorts, '\0', sizeof(UDPPorts));
			//if((getNthValueSafe(5, rec, ',', TCPPorts, sizeof(TCPPorts)) == -1)){	continue; }
			begin = strchr(rec, '\'');
			end = strchr(begin+1, '\'');
			memcpy(TCPPorts, (begin+1), ((end-1) - (begin+1) + 1));
			
			//if((getNthValueSafe(6, rec, ',', UDPPorts, sizeof(UDPPorts)) == -1)){	continue; }
			begin = strchr(end+1, '\'');
			end = strchr(begin+1, '\'');
			memcpy(UDPPorts, (begin+1), ((end-1) - (begin+1) + 1));
			websWrite(wp, T("<input type=\"hidden\" id=\"TCPPorts%d\" name=\"TCPPorts%d\" value=\"%s\">"), i-1, i-1, TCPPorts); //TCPPorts
			websWrite(wp, T("<input type=\"hidden\" id=\"UDPPorts%d\" name=\"UDPPorts%d\" value=\"%s\">"), i-1, i-1, UDPPorts); //TCPPorts
		}
		websWrite(wp, T("</tr>\n"));
		listnumber++;
	}	  
	websWrite(wp, T("<input type=\"hidden\" id=\"rulenum\" name=\"rulenum\" value=\"%d\">\n"), listnumber);
	return 0;	
}

static int getAccessControlEnable(int eid, webs_t wp, int argc, char_t **argv)
{
	char *ruleEnable = nvram_bufget(RT2860_NVRAM, "AccessControlEnable");
	return websWrite(wp, T("%d"), atoi(ruleEnable));
}

/* Jacky.Yang 28-Mar-2008, 
 * IPC command: send_event <start/end> <nvram tag name> <rule name> <firewall tag>
 * IPC command: send_event start SinglePortForwardRules jacky222 SPort_
 * IPC command: send_event end SinglePortForwardRules jacky222 SPort_
 */
void runBySchedule(char *fullCommand)
{
	int rule_count, loopCount;
	int *rules, *getRule, *pt;
	//char nvramTagName[32], ruleName[32], iptablesRuleTag[64];
	char *startTag, *nvramTagName, *ruleName, *iptablesRuleTag;
	char rec[256]={0}, iptablesRuleName[64]={0}, temp[64]={0}, ruleNameCheck[32]={0}, ruleEnable[4]={0};

	memset(rec, '\0', sizeof(rec));
	memset(iptablesRuleName, '\0', sizeof(iptablesRuleName));
	memset(temp, '\0', sizeof(temp));
	memset(ruleNameCheck, '\0', sizeof(ruleNameCheck));
	memset(ruleEnable, '\0', sizeof(ruleEnable));
	startTag = strtok(fullCommand, ",");
	nvramTagName = strtok(NULL, ",");
	ruleName = strtok(NULL, ",");
	iptablesRuleTag = strtok(NULL, ",");

	//Add for run port trigger with scheduling - U-Media Ricky Cao on Apr. 11 2008
	if(!strcmp(nvramTagName, "PortTriggerRuleList")){
		printf("Run Port Trigger on schedule ..\n");
		PortTriggerRunBySchedule(startTag, ruleName);
		return;
	}
	//U-Media Ricky Cao on Apr. 11 2008
	
	rules = nvram_bufget(RT2860_NVRAM, nvramTagName);
	
	//search ruleName start point
	sprintf(temp,";%s,",ruleName);   //for some short name ,example sheduleName=1
	pt = strstr(rules, temp);
	if(pt != NULL)	{
		memset(temp, '\0', sizeof(temp));	
		sprintf(temp,"%s,",ruleName);
		pt = strstr(pt, temp);
	}else{
		memset(temp, '\0', sizeof(temp));	
		sprintf(temp,"%s,",ruleName);
		pt = strstr(rules, temp);
	}

	//printf("jacky - check pt:%s\n", pt);
    if (getNthValueSafe(0, pt, ';', rec, 256) != -1)
	{
		if((getNthValueSafe(0, rec, ',', ruleNameCheck, sizeof(ruleNameCheck)) == -1)){
			//continue;
		}
		
		if((getNthValueSafe(1, rec, ',', ruleEnable, sizeof(ruleEnable)) == -1)){
			//continue;
		}
	}

	strcpy(iptablesRuleName, iptablesRuleTag);
	if (!strcmp(nvramTagName, "SinglePortForwardRules")) {
		if (!strcmp(startTag, "start")) {
			//doSystem("echo single_start");
			iptablesSinglePortForwardRun(iptablesRuleName, ruleName, ruleEnable);
		}
		else if (!strcmp(startTag, "stop")) {
			addScheduleToCron();
		}
	}
	else if (!strcmp(nvramTagName, "PortRangeForwardRules")) {
		if (!strcmp(startTag, "start")) {
			//doSystem("echo range_start");
			iptablesPortRangeForwardRun(iptablesRuleName, ruleName, ruleEnable);
		}
		else if (!strcmp(startTag, "stop")) {
			addScheduleToCron();
		}
	}
	else if (!strcmp(nvramTagName, "AccessControlPolicy")) {
		if (!strcmp(startTag, "start")) {
			AccessControlRun(iptablesRuleName, ruleName, ruleEnable);
		}
		else if (!strcmp(startTag, "stop")) {
			addScheduleToCron();
		}
	}

}

static int showInboundFilterMenu(int eid, webs_t wp, int argc, char_t **argv)
{
	int i=0;
	char rec[128];
	char ruleName[32];
	char *rules = nvram_bufget(RT2860_NVRAM, "InboundFilterRules");
	if(!rules)
		return 0;
	if(!strlen(rules))
		return 0;

	while(getNthValueSafe(i++, rules, ';', rec, sizeof(rec)) != -1 ){
		
		if((getNthValueSafe(0, rec, ',', ruleName, sizeof(ruleName)) == -1)){
			continue;
		}
		websWrite(wp, T("<option id=%s name=%s>%s</option>"), ruleName, ruleName, ruleName);
	}	  
	return 0;
}

//Add for GUI operation of Port Trigger - U-Media Ricky Cao on Apr. 03 2008
#define ONBOOT		1
#define NOTONBOOT	0

#define ADDRULE		1
#define DELRULE		0

int PortTriggerReplacePortDelimited(char *portList, char orgDelimited, char newDelimited)
{
	int cnt = 0;

	if (NULL == portList)
		return -1;
	while (*portList++ != '\0') {
		if (*portList == orgDelimited){
			*portList = newDelimited;
			++cnt;
		}
	}
	return (cnt + 1);
}

int PortTriggerGetPortSettingNum(char *portList)
{
	int cnt = 0;

	if (NULL == portList)
		return 0;
	while (*portList++ != '\0') {
		if (*portList == '&')
			++cnt;
	}
	return (cnt + 1);
}

void PortTriggerSetActive(int isActive)
{
	if(!isActive){
		doSystem("iptables -t nat -D PREROUTING -i eth2.2 -j TRIGGER --trigger-type dnat");
		doSystem("iptables -t nat -D PREROUTING -i ppp0 -j TRIGGER --trigger-type dnat");
	}else{
		doSystem("iptables -t nat -D PREROUTING -i eth2.2 -j TRIGGER --trigger-type dnat"); //prevent duplicate entry
		doSystem("iptables -t nat -I PREROUTING -i eth2.2  -j TRIGGER --trigger-type dnat");
		doSystem("iptables -t nat -D PREROUTING -i ppp0 -j TRIGGER --trigger-type dnat"); //prevent duplicate entry
		doSystem("iptables -t nat -I PREROUTING -i ppp0  -j TRIGGER --trigger-type dnat");
	}

	return;
}

//void PortTriggerRunIPTables(int isAdd, char *triggerRule)
void PortTriggerRunIPTables(int isAdd, char *isEnabled, char *matchProtocol, char *matchPort, char *triggerProtocol, char *triggerPort)
{
	char matchPort2[16]={0};
	char *ptt;
	char command[512]={0};

	strcpy(matchPort2, matchPort);
	ptt = (char *) 0;
	ptt = strchr(matchPort2, '-');
	if (ptt){
		*ptt = ':';
	}

	if(isAdd){
		if(strcmp(isEnabled, "on")){
			return;	
		}
		printf("Insert port trigger rule to iptables with:\n");
		/*
		 * prevent to add duplicate rule in iptables, so I always delete rule first before add the new rule
		 */
		if(!strcmp(matchProtocol, "all")){
			//sprintf(command, "/bin/iptables -I FORWARD -o eth2.2 --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s", matchPort2, triggerProtocol, matchPort, triggerPort);
			//sprintf(command, "/bin/iptables -I FORWARD -o eth2.2 -p tcp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s", matchPort2, triggerProtocol, matchPort, triggerPort);
			//sprintf(command, "/bin/iptables -I FORWARD -o eth2.2 -p udp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s", matchPort2, triggerProtocol, matchPort, triggerPort);
			//add rule for eth wan
			sprintf(command, "/bin/iptables -D FORWARD -o eth2.2 -p tcp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s;/bin/iptables -D FORWARD -o eth2.2 -p udp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchPort2, triggerProtocol, matchPort, triggerPort, matchPort2, triggerProtocol, matchPort, triggerPort);
			printf(command);
			system(command);
			sprintf(command, "/bin/iptables -I FORWARD -o eth2.2 -p tcp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s;/bin/iptables -I FORWARD -o eth2.2 -p udp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchPort2, triggerProtocol, matchPort, triggerPort, matchPort2, triggerProtocol, matchPort, triggerPort);
			printf(command);
			system(command);
			//add rule for ppp
			sprintf(command, "/bin/iptables -D FORWARD -o ppp0 -p tcp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s;/bin/iptables -D FORWARD -o ppp0 -p udp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchPort2, triggerProtocol, matchPort, triggerPort, matchPort2, triggerProtocol, matchPort, triggerPort);
			printf(command);
			system(command);
			sprintf(command, "/bin/iptables -I FORWARD -o ppp0 -p tcp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s;/bin/iptables -I FORWARD -o ppp0 -p udp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchPort2, triggerProtocol, matchPort, triggerPort, matchPort2, triggerProtocol, matchPort, triggerPort);
			printf(command);
			system(command);
		}else{
			//add rule for eth wan
			sprintf(command, "/bin/iptables -D FORWARD -o eth2.2 -p %s --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchProtocol, matchPort2, triggerProtocol, matchPort, triggerPort);
			printf(command);
			system(command);
			sprintf(command, "/bin/iptables -I FORWARD -o eth2.2 -p %s --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchProtocol, matchPort2, triggerProtocol, matchPort, triggerPort);
			printf(command);
			system(command);
			//add rule for ppp
			sprintf(command, "/bin/iptables -D FORWARD -o ppp0 -p %s --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchProtocol, matchPort2, triggerProtocol, matchPort, triggerPort);
			printf(command);
			system(command);
			sprintf(command, "/bin/iptables -I FORWARD -o ppp0 -p %s --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchProtocol, matchPort2, triggerProtocol, matchPort, triggerPort);
			printf(command);
			system(command);
		}
	}else{
		printf("Delete port trigger rule to iptables with:\n");
		if(!strcmp(matchProtocol, "all")){
			//sprintf(command, "/bin/iptables -D FORWARD -o eth2.2 --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s", matchPort2, triggerProtocol, matchPort, triggerPort);
			//sprintf(command, "/bin/iptables -D FORWARD -o eth2.2 -p tcp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s", matchPort2, triggerProtocol, matchPort, triggerPort);
			//sprintf(command, "/bin/iptables -D FORWARD -o eth2.2 -p udp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s", matchPort2, triggerProtocol, matchPort, triggerPort);			
			sprintf(command, "/bin/iptables -D FORWARD -o eth2.2 -p tcp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s;/bin/iptables -D FORWARD -o eth2.2 -p udp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchPort2, triggerProtocol, matchPort, triggerPort, matchPort2, triggerProtocol, matchPort, triggerPort);			
			printf(command);
			system(command);
			sprintf(command, "/bin/iptables -D FORWARD -o ppp0 -p tcp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s;/bin/iptables -D FORWARD -o eth2.2 -p udp --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchPort2, triggerProtocol, matchPort, triggerPort, matchPort2, triggerProtocol, matchPort, triggerPort);			
			printf(command);
			system(command);
		}else{
			sprintf(command, "/bin/iptables -D FORWARD -o eth2.2 -p %s --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchProtocol, matchPort2, triggerProtocol, matchPort, triggerPort);
			printf(command);
			system(command);
			sprintf(command, "/bin/iptables -D FORWARD -o ppp0 -p %s --dport %s -j TRIGGER --trigger-type out --trigger-proto %s --trigger-match %s --trigger-relate %s\n", matchProtocol, matchPort2, triggerProtocol, matchPort, triggerPort);
			printf(command);
			system(command);
		}
	}
}

void PortTriggerParsePortRunIptables(int isAdd, char *triggerRule)
{
	char ruleName[16]={0}, isEnabled[8]={0};
	char matchProtocol[4]={0}, matchPortList[128]={0}, triggerProtocol[4]={0}, triggerPortList[128]={0};
	char matchPort[12]={0}, triggerPort[12]={0};
	int matchPortNum=0, triggerPortNum=0;
	int matchPortCount=0, triggerPortCount=0;

	// I record the rule format as below
	// [rule name], [is enabled], [match protocol], [match port], [trigger protocol], [trigger port];......... 
	if(getNthValueSafe(0, triggerRule, ',', ruleName, 16) == -1){
		printf("Take \"rule name\" out from fail!!\n");
		return;
	}
	if(getNthValueSafe(1, triggerRule, ',', isEnabled, 8) == -1){
		printf("Take \"rule enabled\" out fail!!\n");
		return;
	}
	if(getNthValueSafe(2, triggerRule, ',', matchProtocol, 4) == -1){
		printf("Take \"match portocol\" out fail!!\n");
		return;
	}
	if(getNthValueSafe(3, triggerRule, ',', matchPortList, 128) == -1){
		printf("Take \"match port\" out fail!!\n");
		return;
	}
	if(getNthValueSafe(4, triggerRule, ',', triggerProtocol, 4) == -1){
		printf("Take \"trigger protocol\" out fail!!\n");
		return;
	}
	if(getNthValueSafe(5, triggerRule, ',', triggerPortList, 128) == -1){
		printf("Take \"trigger port\" out fail!!\n");
		return;
	}

	//If this rule doesn't be enabled on default, we can just skip it
	if(strcmp(isEnabled, "on")){
		return;
	}

	matchPortNum = PortTriggerGetPortSettingNum(matchPortList);
	triggerPortNum = PortTriggerGetPortSettingNum(triggerPortList);
	
	for(matchPortCount=0; matchPortCount<matchPortNum; matchPortCount++){
		if(getNthValueSafe(matchPortCount, matchPortList, '&', matchPort, 12) != -1){
			for(triggerPortCount=0; triggerPortCount<triggerPortNum; triggerPortCount++){
				if(getNthValueSafe(triggerPortCount, triggerPortList, '&', triggerPort, 12) != -1){
					PortTriggerRunIPTables(isAdd, isEnabled, matchProtocol, matchPort, triggerProtocol, triggerPort);
				}
			}
		}
	}

	return;
}

void PortTriggerRunBySchedule(char *scheduleAction, char *scheduleRuleName)
{
	char *PortTriggerRuleList;
	int ruleCount;
	char rule[256]={0};
	char ruleName[32]={0};
	
	PortTriggerRuleList = nvram_bufget(RT2860_NVRAM, "PortTriggerRuleList");
	if(!PortTriggerRuleList || !strlen(PortTriggerRuleList)){
		printf("Port Trigger rule list is NULL!!\n");
       	return;
	}

	ruleCount = getValueCount(PortTriggerRuleList);
	if(!ruleCount){
		printf("Get the number of Port Trigger rule fail\n");
		return;
	}

	for(;ruleCount>0;ruleCount--){
		if(getNthValueSafe((ruleCount-1), PortTriggerRuleList, ';' , rule, 256)!=-1){
			if(getNthValueSafe(0, rule, ',' , ruleName, 256)!=-1){
				if(!strcmp(ruleName, scheduleRuleName)){
					if(!strcmp(scheduleAction, "start")){
						PortTriggerParsePortRunIptables(DELRULE, rule); //avoid the duplicate rule
						PortTriggerParsePortRunIptables(ADDRULE, rule);
					}else{
						PortTriggerParsePortRunIptables(DELRULE, rule);
					}
				}
			}
		}
	}
	
}

int PortTriggerProcessAllRule(int isAdd, int onBoot)
{
	int ruleCount;	
	char *PortTriggerRuleList;
	char rule[256]={0};
	char scheduleName[16]={0};
	char ruleName[16]={0};

	PortTriggerRuleList = nvram_bufget(RT2860_NVRAM, "PortTriggerRuleList");
	
	if(!PortTriggerRuleList || !strlen(PortTriggerRuleList)){
       	return -1;
	}

	ruleCount = getValueCount(PortTriggerRuleList);
	if(!ruleCount){
		printf("Initiate port trigger rule fail\n");
		return -1;
	}

	for(;ruleCount>0;ruleCount--){
		if(getNthValueSafe((ruleCount-1), PortTriggerRuleList, ';' , rule, 256)!=-1){
			if(isAdd){
				//Because the Jacky's Schedule will add all rule on boot time except the rule of schedule is set to "Always",
				//so I just add the rule of schdule is set to "Always".
				if(onBoot){
					if(getNthValueSafe(6, rule, ',' , scheduleName, 16)!=-1){
						if(strcmp(scheduleName, "Always")){
							getNthValueSafe(0, rule, ',' , ruleName, 16);
							printf("Skip to initiate the Port Trigger Rule : %s\n", ruleName);
							continue;
						}
					}
				}
				PortTriggerParsePortRunIptables(ADDRULE, rule);
			}else{
				PortTriggerParsePortRunIptables(DELRULE, rule);
			}
		}
	}

	return 0;
}

int PortTriggerOnInitiate(void)
{
	char *PortTriggerOn;

	PortTriggerOn = nvram_bufget(RT2860_NVRAM, "PortTriggerOn");

	if(!strcmp(PortTriggerOn, "1")){
		PortTriggerSetActive(1);
	}

	return 0;
}

int PortTriggerRulesInitiate(void)
{
	char *PortTriggerOn;

	PortTriggerOn = nvram_bufget(RT2860_NVRAM, "PortTriggerOn");
	if(!strcmp(PortTriggerOn, "1")){
		PortTriggerProcessAllRule(ADDRULE, ONBOOT);
	}
	
	return 0;
}

int deleteNthPortTriggerRule(int delIndex, int count, char *value, char delimit)
{
	char *begin, *end;
	int i=0,j=0;
	int need_check_flag=0;
	char rule[256] = {0}; 
	char *buf = strdup(value);

	begin = buf;

	end = strchr(begin, delimit);
	while(end){
		if(i == delIndex){
			memset(rule, 0, 256);
			if(i==0){
				memcpy(rule, begin, end-begin);
			}else{
				memcpy(rule, begin+1, end-(begin+1));
			}
			PortTriggerParsePortRunIptables(DELRULE, rule);
			memset(begin, 0, end - begin );
			if(delIndex == 0)
				need_check_flag = 1;
		}
		begin = end;

		end = strchr(begin+1, delimit);
		i++;
	}

	if(!end && delIndex == i){
		memset(rule, 0, 256);
		if(i==0){
			memcpy(rule, begin, strlen(begin));
		}else{
			memcpy(rule, begin+1, strlen(begin+1));
		}
		PortTriggerParsePortRunIptables(DELRULE, rule);
		memset(begin, 0, strlen(begin));
	}

	if(need_check_flag){
		for(i=0; i<strlen(value); i++){
			if(buf[i] == '\0')
				continue;
			if(buf[i] == ';')
				buf[i] = '\0';
			break;
		}
	}

	for(i=0, j=0; i<strlen(value); i++){
		if(buf[i] != '\0'){
			value[j++] = buf[i];
		}
	}
	value[j] = '\0';

	free(buf);
	return 0;
}

//The rName variable is for Jacky's scheduling
static void PortTriggerRuleAdd(webs_t wp, char_t *path, char_t *query, char *rName)
{
	char_t *ruleName, *ruleEnabled, *schedule; 
	char_t *matchProtocol, *matchPortS/*, *matchPortE*/; 
	char_t *triggerProtocol, *triggerPortS/*, *triggerPortE*/;

	char matchPort[16]={0}, matchPort2[16]={0}, triggerPort[16]={0};

	char *PortTriggerRuleList;
	char portTriggerRuleListNew[1024] = {0}, triggerRule[512]={0};

	char oldRuleName[32]={0}; //for Jacky's scheduling

	ruleName = websGetVar(wp, T("triggerRuleName"), T("none"));
	ruleEnabled = websGetVar(wp, T("triggerRuleEnable"), T("off"));	
	matchProtocol = websGetVar(wp, T("triggerMatchProtocol"), T("none"));
	matchPortS = websGetVar(wp, T("triggerMatchPortStart"), T("none"));
	triggerProtocol = websGetVar(wp, T("triggerTriggerProtocol"), T("none"));
	triggerPortS = websGetVar(wp, T("triggerTriggerPortStart"), T("none"));
	schedule = websGetVar(wp, T("schedule"), T("Always"));	

	if(!strcmp(ruleName, "none") ||
		!strcmp(matchProtocol, "none") || !strcmp(matchPortS, "none") ||
		!strcmp(triggerProtocol, "none") || !strcmp(triggerPortS, "none")){

		printf("got invalid variable from GUI\n");
printf("salim:ruleName=%s\n",ruleName);
printf("salim:matchProtocol=%s\n",matchProtocol);
printf("salim:matchPortS=%s\n",matchPortS);
printf("salim:triggerProtocol=%s\n",triggerProtocol);
printf("salim:triggerPortS=%s\n",triggerPortS);
		websRedirect(wp, "/advanced/port_trigger.asp");
		
		return;
	}

	printf("before replace delimited: 1.matchPortS=%s 2.triggerPortS=%s\n", matchPortS, triggerPortS);
	PortTriggerReplacePortDelimited(matchPortS, ',', '&');
	PortTriggerReplacePortDelimited(triggerPortS, ',', '&');
	printf("after replace delimited: 1.matchPortS=%s 2.triggerPortS=%s\n", matchPortS, triggerPortS);
	
	// I record the rule format as below
	// [rule name], [is enabled], [match protocol], [match port], [trigger protocol], [trigger port];......... 
	if((PortTriggerRuleList = nvram_bufget(RT2860_NVRAM, "PortTriggerRuleList")) && strlen(PortTriggerRuleList)){
		snprintf(portTriggerRuleListNew, ruleLength, "%s;%s,%s,%s,%s,%s,%s,%s",  PortTriggerRuleList, ruleName, ruleEnabled, matchProtocol, matchPortS, triggerProtocol, triggerPortS, schedule);
	}else{
		snprintf(portTriggerRuleListNew, ruleLength, "%s,%s,%s,%s,%s,%s,%s", ruleName, ruleEnabled, matchProtocol, matchPortS, triggerProtocol, triggerPortS, schedule);
	}	

	//for Jacky's scheduling
	if(rName!=NULL){
		strcpy(oldRuleName, rName);
	}

	nvram_set(RT2860_NVRAM, "PortTriggerRuleList", portTriggerRuleListNew);
	if (!strcmp(ruleEnabled, "on"))
		addScheduleToNvram("PortTriggerRuleList", oldRuleName, ruleName, schedule); //for Jacky's scheduling
	nvram_commit(RT2860_NVRAM);

	if (!strcmp(schedule, "Always")){ //for Jacky's scheduling	
		snprintf(triggerRule, ruleLength, "%s,%s,%s,%s,%s,%s", ruleName, ruleEnabled, matchProtocol, matchPortS, triggerProtocol, triggerPortS);			
		PortTriggerParsePortRunIptables(ADDRULE, triggerRule);
	}

	addScheduleToCron(); //for Jacky's scheduling
}

static void PortTriggerRuleDelete(webs_t wp, char_t *path, char_t *query)
{
	int ruleCount;	
	char *ruleIndexStr;
	int ruleIndexNum;
	char rule[256]={0}, ruleName[32]={0}; //for delete schedule entry from Jacky's scheduling
	char *PortTriggerRuleList;
	PortTriggerRuleList = nvram_bufget(RT2860_NVRAM, "PortTriggerRuleList");
	
	if(!PortTriggerRuleList || !strlen(PortTriggerRuleList)){
       	return;
	}

	ruleCount = getNums(PortTriggerRuleList, ';');
	if(!ruleCount){
		return;
	}
	//printf("got the number of rules in list when delete rule: %d\n", ruleCount);

	ruleIndexStr = websGetVar(wp, T("ruleIndex"), T("none"));
	if(!strcmp("ruleIndex", "none")){
		printf("Can not got ruleIndex from GUI !!\n");
		websRedirect(wp, "/advanced/port_trigger.asp");	
		return;
	}
	ruleIndexNum = atoi(ruleIndexStr);
	if(ruleIndexNum+1>ruleCount){
		printf("illegal rule index !!\n");
		websRedirect(wp, "/advanced/port_trigger.asp");	
		return;
	}

	//take the rule name for delete schedule entry from Jacky's scheduling
	getNthValueSafe(ruleIndexNum, PortTriggerRuleList, ';', rule, 256);
	printf("the rule = %s\n", rule);
	getNthValueSafe(0, rule, ',', ruleName, 32);
	printf("the rule name = %s\n", ruleName);
	
	deleteNthPortTriggerRule(ruleIndexNum, ruleCount, PortTriggerRuleList, ';');
	nvram_set(RT2860_NVRAM, "PortTriggerRuleList", PortTriggerRuleList);
	nvram_commit(RT2860_NVRAM);
	
	delScheduleToNvram("PortTriggerRuleList", ruleName, ruleName, "true", "");//for delete schedule entry from Jacky's scheduling
}


//For edit port trigger rules list, we will delete older rule first, then add the newer rule from GUI
static void PortTriggerRuleEdit(webs_t wp, char_t *path, char_t *query)
{
	int ruleCount;	
	char *ruleIndexStr;
	int ruleIndexNum;
	char oldRule[256]={0}, oldRuleName[32]={0}; //for Jacky's scheduling
	char *PortTriggerRuleList;

	//1.  delete older rule from rules list
	PortTriggerRuleList = nvram_bufget(RT2860_NVRAM, "PortTriggerRuleList");
	
	if(!PortTriggerRuleList || !strlen(PortTriggerRuleList)){
       	return;
	}

	ruleCount = getNums(PortTriggerRuleList, ';');
	if(!ruleCount){
		return;
	}

	ruleIndexStr = websGetVar(wp, T("ruleIndex"), T("none"));
	if(!strcmp("ruleIndex", "none")){
		printf("Can not got ruleIndex from GUI !!\n");
		websRedirect(wp, "/advanced/port_trigger.asp");	
		return;
	}
	ruleIndexNum = atoi(ruleIndexStr);
	if(ruleIndexNum+1>ruleCount){
		printf("illegal rule index !!\n");
		websRedirect(wp, "/advanced/port_trigger.asp");	
		return;
	}

	//take the old rule name for Jacky's scheduling
	getNthValueSafe(ruleIndexNum, PortTriggerRuleList, ';', oldRule, 256);
	printf("the old rule = %s\n", oldRule);
	getNthValueSafe(0, oldRule, ',', oldRuleName, 32);
	printf("the old rule name = %s\n", oldRuleName);
	
	deleteNthPortTriggerRule(ruleIndexNum, ruleCount, PortTriggerRuleList, ';');
	nvram_set(RT2860_NVRAM, "PortTriggerRuleList", PortTriggerRuleList);
	nvram_commit(RT2860_NVRAM);
	
	delScheduleToNvram("PortTriggerRuleList", oldRuleName, oldRuleName, "true", "");//for delete schedule entry from Jacky's scheduling

	//2.  Add newer rule to rules list
	PortTriggerRuleAdd(wp, path, query, oldRuleName);

	return;
}

static void PortTriggerManageRule(webs_t wp, char_t *path, char_t *query)
{
	char *formAction;

	formAction = websGetVar(wp, T("formAction"), T("none"));
	if(!strcmp("formAction", "none")){
		printf("Can not got action from GUI !!\n");
		websRedirect(wp, "/advanced/port_trigger.asp");	
		return;
	}

	if(!strcmp(formAction, "add")){
		//printf("Got a add action for port trigger!!\n");
		PortTriggerRuleAdd(wp, path, query, NULL);
	}else if(!strcmp(formAction, "delete")){
		//printf("Got a delete action for port trigger!!\n");
		PortTriggerRuleDelete(wp, path, query);
	}else if(!strcmp(formAction, "edit")){
		//printf("Got a edit action for port trigger!!\n");	
		PortTriggerRuleEdit(wp, path, query);
	}

	//websRedirect(wp, "/advanced/port_trigger.asp");
	websRedirect(wp, "/applied.asp?url=/advanced/port_trigger.asp");
}

static void PortTriggerActive(webs_t wp, char_t *path, char_t *query)
{
	char *PortTriggerOn;

	PortTriggerOn = websGetVar(wp, T("triggerFunctionActiveSetting"), T("none"));
	if(!strcmp(PortTriggerOn, "none")){
		printf("Take Port Trigger Feature status from GUI fail!!\n");
		return;
	}

	nvram_set(RT2860_NVRAM, "PortTriggerOn", PortTriggerOn);
	nvram_commit(RT2860_NVRAM);

	if(!strcmp(PortTriggerOn, "0")){
		PortTriggerSetActive(0);
		PortTriggerProcessAllRule(DELRULE, NOTONBOOT);
	}else{
		PortTriggerSetActive(1);
		PortTriggerProcessAllRule(ADDRULE, NOTONBOOT);
	}

	//websRedirect(wp, "/advanced/port_trigger.asp");
	websRedirect(wp, "/applied.asp?url=/advanced/port_trigger.asp");
}

void exposeTriggerRule(webs_t wp, char *triggerRule, int ruleIndex)
{
	char ruleName[16]={0}, isEnabled[8]={0};
	char matchProtocol[4]={0}, matchPort[128]={0}, triggerProtocol[4]={0}, triggerPort[128]={0};
	char scheduleName[16]={0};

	// I record the rule format as below
	// [rule name], [is enabled], [match protocol], [match port], [trigger protocol], [trigger port];......... 
	if(getNthValueSafe(0, triggerRule, ',', ruleName, 16) == -1){
		printf("Take \"rule name\" out from %dth Port Trigger rule fail!!\n", ruleIndex);
		return;
	}
	if(getNthValueSafe(1, triggerRule, ',', isEnabled, 8) == -1){
		printf("Take \"rule enabled\" out from %dth Port Trigger rule fail!!\n", ruleIndex);
		return;
	}
	if(getNthValueSafe(2, triggerRule, ',', matchProtocol, 4) == -1){
		printf("Take \"match portocol\" out from %dth Port Trigger rule fail!!\n", ruleIndex);
		return;
	}
	if(getNthValueSafe(3, triggerRule, ',', matchPort, 128) == -1){
		printf("Take \"match port\" out from %dth Port Trigger rule fail!!\n", ruleIndex);
		return;
	}
	if(getNthValueSafe(4, triggerRule, ',', triggerProtocol, 4) == -1){
		printf("Take \"trigger protocol\" out from %dth Port Trigger rule fail!!\n", ruleIndex);
		return;
	}
	if(getNthValueSafe(5, triggerRule, ',', triggerPort, 128) == -1){
		printf("Take \"trigger port\" out from %dth Port Trigger rule fail!!\n", ruleIndex);
		return;
	}
	if(getNthValueSafe(6, triggerRule, ',', scheduleName, 16) == -1){
		printf("Take \"schedule name\" out from %dth Port Trigger rule fail!!\n", ruleIndex);
		return;
	}	

	PortTriggerReplacePortDelimited(matchPort, '&', ',');
	PortTriggerReplacePortDelimited(triggerPort, '&', ',');
	
	websWrite(wp, T("<tr>\n"));
	websWrite(wp, T("	<td class=\"form_list_content\"> %d&nbsp; <input id=\"rule%dEnabled\" type=\"checkbox\" %s disabled=\"disabled\" style=\"margin:0 0 0 0;\" > </td>"), ruleIndex,ruleIndex-1, !strcmp(isEnabled, "on")?"checked=\"checked\"":"");
	websWrite(wp, T("	<td id=\"rule%dName\" class=\"form_list_content\">%s</td>"), ruleIndex-1, ruleName);
	websWrite(wp, T("	<td id=\"rule%dMatchPort\" class=\"form_list_content\" style=\"word-break:break-all\">%s/%s</td>"), ruleIndex-1, matchProtocol, matchPort);
	websWrite(wp, T("	<td id=\"rule%dTriggerPort\" class=\"form_list_content\" style=\"word-break:break-all\">%s/%s</td>"), ruleIndex-1, triggerProtocol, triggerPort);
	websWrite(wp, T("	<td id=\"rule%dScheduleName\" class=\"form_list_content\">%s</td>"), ruleIndex-1, scheduleName);
	websWrite(wp, T("	<td class=\"form_list_content\" ><a href=\"javascript:manageRule('edit', %d);\"><img src=\"/Images/img_edit.gif\" alt=\"Edit Rule\"></a></td>"), ruleIndex-1);
	websWrite(wp, T("	<td class=\"form_list_content\" ><a href=\"javascript:manageRule('delete', %d);\"><img src=\"/Images/img_delete.gif\" alt=\"Delete Rule\"></a></td>"), ruleIndex-1);
	websWrite(wp, T("</tr>\n"));

	return;
}

static int PortTriggerRuleListShowASP(int eid, webs_t wp, int argc, char_t **argv)
{
	char *PortTriggerRuleList;
	int ruleCount, count=0;
	char PortTriggerRule[256]={0};
	int listnumber=0;

	PortTriggerRuleList = nvram_bufget(RT2860_NVRAM, "PortTriggerRuleList");
	if(!PortTriggerRuleList || !strlen(PortTriggerRuleList)){
		printf("PortTriggerRuleList is NULL\n");
		return 0;
	}	
	ruleCount = getValueCount(PortTriggerRuleList);
	printf("PortTriggerRuleList = %s, ruleCount=%d\n", PortTriggerRuleList, ruleCount);
	
	if(ruleCount>0){
		while(getNthValueSafe(count++, PortTriggerRuleList, ';', PortTriggerRule, 256) != -1){
			exposeTriggerRule(wp, PortTriggerRule, count);
			listnumber++;
		}
		websWrite(wp, T("<input type=\"hidden\" id=\"rulenum\" name=\"rulenum\" value=\"%d\">\n"), listnumber);
	}

	return 0;
}
//U-Media Ricky Cao on Apr. 03 2008

/*
 * arguments: type - 0 = return the configuration of 'field' (default)
 *                   1 = write the configuration of 'field' 
 *            field - parameter name in nvram
 * description: read general configurations from nvram
 *              if value is NULL -> "0"
 */
static int getRuleCount(int eid, webs_t wp, int argc, char_t **argv)
{
	int type, rule_count;
	char_t *field;
	char *rules;

	if (ejArgs(argc, argv, T("%d %s"), &type, &field) < 2) {
		return websWrite(wp, T("Insufficient args\n"));
	}
	rules = nvram_bufget(RT2860_NVRAM, field);
	rule_count = getRuleNums(rules);
	if (type == 1) {
		if (!strcmp(rules, ""))
			return websWrite(wp, T("0"));
		return websWrite(wp, T("%d"), rule_count);
	}

	return 0;
}

void formDefineFirewall(void)
{
	websAspDefine(T("getDefaultFirewallPolicyASP"), getDefaultFirewallPolicyASP);
	websFormDefine(T("BasicSettings"), BasicSettings);

	websAspDefine(T("getIPPortFilterEnableASP"), getIPPortFilterEnableASP);
	websAspDefine(T("showIPPortFilterRulesASP"), showIPPortFilterRulesASP);
	websAspDefine(T("getIPPortRuleNumsASP"), getIPPortRuleNumsASP);
	websFormDefine(T("ipportFilter"), ipportFilter);
	websFormDefine(T("ipportFilterDelete"), ipportFilterDelete);
	websFormDefine(T("getRulesPacketCount"), getRulesPacketCount);

	websFormDefine(T("DMZ"), DMZ);
	websAspDefine(T("getDMZEnableASP"), getDMZEnableASP);
	websAspDefine(T("showDMZIPAddressASP"), showDMZIPAddressASP);

	websAspDefine(T("getPortForwardEnableASP"), getPortForwardEnableASP);

	websAspDefine(T("getPortForwardRuleNumsASP"), getPortForwardRuleNumsASP);
	
	//Jacky.Yang 03-Mar-2008,
	websFormDefine(T("singlePortForwardAdd"), singlePortForwardAdd);
	websFormDefine(T("singlePortForwardDelete"), singlePortForwardDelete);
	websAspDefine(T("showSinglePortForward"), showSinglePortForward);
	websFormDefine(T("PortRangeForwardAdd"), PortRangeForwardAdd);
	websFormDefine(T("PortRangeForwardDelete"), PortRangeForwardDelete);
	websAspDefine(T("showPortRangeForward"), showPortRangeForward);
	websFormDefine(T("inboundFilterAdd"), inboundFilterAdd);
	websAspDefine(T("showInboundFilter"), showInboundFilter);
	websFormDefine(T("inboundFilterDelete"), inboundFilterDelete);
	websFormDefine(T("AccessControlAdd"), AccessControlAdd);
	websAspDefine(T("showAccessControl"), showAccessControl);
	websFormDefine(T("AccessControlDelete"), AccessControlDelete);
	websAspDefine(T("getAccessControlEnable"), getAccessControlEnable);
	websAspDefine(T("showInboundFilterMenu"), showInboundFilterMenu);
	websAspDefine(T("getRuleCount"), getRuleCount);
	
	websFormDefine(T("remoteManagement"), remoteManagement);
	websAspDefine(T("checkIfUnderBridgeModeASP"), checkIfUnderBridgeModeASP);

	//Add for GUI operation of Port Trigger - U-Media Ricky Cao on Apr. 03 2008
	websFormDefine(T("PortTriggerManageRule"), PortTriggerManageRule);
	websFormDefine(T("PortTriggerActive"), PortTriggerActive);		
	websAspDefine(T("PortTriggerRuleListShowASP"), PortTriggerRuleListShowASP);
	//U-Media Ricky Cao on Apr. 03 2008

}

void dropPort0Port1()
{
	doSystem("iptables -A INPUT -j DROP -p tcp --sport 0");
	doSystem("iptables -A INPUT -j DROP -p udp --sport 0");
	doSystem("iptables -A INPUT -j DROP -p tcp --dport 0");
	doSystem("iptables -A INPUT -j DROP -p udp --dport 0 ");
	doSystem("iptables -A INPUT -j DROP -p tcp --sport 1");
	doSystem("iptables -A INPUT -j DROP -p udp --sport 1");
	doSystem("iptables -A INPUT -j DROP -p tcp --dport 1");
	doSystem("iptables -A INPUT -j DROP -p udp --dport 1");
}

void icmpEnabled()
{
	char *icmpEnabled = nvram_get(RT2860_NVRAM, "icmpEnabled");
	
	doSystem("iptables -t nat -F PREROUTING_ICMP");
	if (atoi(icmpEnabled) == 1)
		doSystem("iptables -I PREROUTING_ICMP -t nat -i %s -s 0.0.0.0/0 -p icmp --icmp-type echo-request -j DROP", getWanIfName());
}

void firewall_init(void)
{
	//Add for configure MSS to 1452 bytes when PPP link is up - U-Media Ricky Cao on May 16 2008
	char *connType = nvram_get(RT2860_NVRAM, "wanConnectionMode");
	char *lan_ipaddr = nvram_get(RT2860_NVRAM, "lan_ipaddr");
	char *lan_netmask = nvram_get(RT2860_NVRAM, "lan_netmask");

	char *cfgValue = nvram_get(RT2860_NVRAM, "wan_use_default_mtu");
	int useDefaultWanMtu = atoi(cfgValue);
	int wanMtu=1500;
	if(useDefaultWanMtu){
		if(!strcmp(connType, "PPPOE"))
			wanMtu=1492;
		else if(!strcmp(connType, "L2TP"))
			wanMtu=1452;
		else if( !strcmp(connType, "PPTP"))
			wanMtu=1452;
	}else{
		cfgValue = nvram_get(RT2860_NVRAM, "wan_mtu");
		wanMtu = atoi(cfgValue);
	}
	// Joan.huang 2010.02.03 set ip_forward is Block
	doSystem("echo 0 > /proc/sys/net/ipv4/ip_forward");
	
	iptablesAllFilterClear();
	iptablesAllNATClear();
	
	// init filter
	//iptablesAllFilterClear();
	// make a new chain
	//doSystem("iptables -t filter -N %s 1>/dev/null 2>&1", IPPORT_FILTER_CHAIN);
	//doSystem("iptables -t filter -A FORWARD -j %s 1>/dev/null 2>&1", IPPORT_FILTER_CHAIN);
	//pGA MSN @LksuAΪ̬OYǺ OK YǺ OKAiO MTU DA
    //AiHNUo@浹LѨӱҰ MTU d
	//doSystem("iptables -A FORWARD -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --clamp-mss-to-pmtu 1>/dev/null 2>&1");
	//iptablesAllFilterRun();

	/*********** nat table PREROUTING rule ********/
	doSystem("iptables -I PREROUTING -t nat -i %s -s 0.0.0.0/0 -p udp -m state --state NEW -j DROP", getWanIfName());
	doSystem("iptables -I PREROUTING -t nat -i %s -p tcp --syn -j DROP", getWanIfName());
	doSystem("iptables -I PREROUTING -t nat -i %s -s 0.0.0.0/0 -p udp --sport 520 -j ACCEPT", getWanIfName());
	doSystem("iptables -I PREROUTING -t nat -i %s -s 0.0.0.0/0 -p tcp -m state --state RELATED,ESTABLISHED -j ACCEPT", getWanIfName());
	
	//2009.11.19 Joan.huang , Port trigger initiate
	PortTriggerOnInitiate();

	doSystem("iptables -t nat -N POSTROUTING_NAT");
	doSystem("iptables -t nat -D POSTROUTING -j POSTROUTING_NAT");
	doSystem("iptables -t nat -I POSTROUTING -j POSTROUTING_NAT");
	
	doSystem("iptables -t nat -N PREROUTING_ICMP");
	doSystem("iptables -t nat -D PREROUTING -j PREROUTING_ICMP");
	doSystem("iptables -t nat -I PREROUTING -j PREROUTING_ICMP");
	icmpEnabled();
	doSystem("iptables -t nat -N PREROUTING_DMZ");
	doSystem("iptables -t nat -D PREROUTING -j PREROUTING_DMZ");
	doSystem("iptables -t nat -I PREROUTING -j PREROUTING_DMZ");
	doSystem("iptables -t nat -N POSTROUTING_DMZ");
	doSystem("iptables -t nat -D POSTROUTING -j POSTROUTING_DMZ");
	doSystem("iptables -t nat -I POSTROUTING -j POSTROUTING_DMZ");
	
	//Jacky.Yang 30-Dec-2008, The upnp control from RT288x_SDK\source\user\linux-igd\pmlist.c, globals.h, etc\upnpd.conf
	// We need remove to upnp initial before, becasue that will casue upnp can't work properly
	//doSystem("iptables -F FORWARD_UPnP");
	//doSystem("iptables -N FORWARD_UPnP");
	doSystem("iptables -D FORWARD -j FORWARD_UPnP");
	doSystem("iptables -I FORWARD -j FORWARD_UPnP");
	//doSystem("iptables -t nat -N PREROUTING_UPnP");
	doSystem("iptables -t nat -D PREROUTING -j PREROUTING_UPnP");
	doSystem("iptables -t nat -I PREROUTING -j PREROUTING_UPnP");
	
	doSystem("iptables -N INPUT_SPort");
	doSystem("iptables -D INPUT -j INPUT_SPort");
	doSystem("iptables -I INPUT -j INPUT_SPort");
	doSystem("iptables -N FORWARD_SPort");
	doSystem("iptables -D FORWARD -j FORWARD_SPort");
	doSystem("iptables -I FORWARD -j FORWARD_SPort");
	
	doSystem("iptables -N INPUT_MPort");
	doSystem("iptables -D INPUT -j INPUT_MPort");
	doSystem("iptables -I INPUT -j INPUT_MPort");
	doSystem("iptables -N FORWARD_MPort");
	doSystem("iptables -D FORWARD -j FORWARD_MPort");
	doSystem("iptables -I FORWARD -j FORWARD_MPort");
	
	doSystem("iptables -t nat -N PREROUTING_SPort");
	doSystem("iptables -t nat -D PREROUTING -j PREROUTING_SPort");
	doSystem("iptables -t nat -I PREROUTING -j PREROUTING_SPort");
	doSystem("iptables -t nat -N POSTROUTING_SPort");
	doSystem("iptables -t nat -D POSTROUTING -j POSTROUTING_SPort");
	doSystem("iptables -t nat -I POSTROUTING -j POSTROUTING_SPort");
	
	doSystem("iptables -t nat -N PREROUTING_MPort");
	doSystem("iptables -t nat -D PREROUTING -j PREROUTING_MPort");
	doSystem("iptables -t nat -I PREROUTING -j PREROUTING_MPort");
	doSystem("iptables -t nat -N POSTROUTING_MPort");
	doSystem("iptables -t nat -D POSTROUTING -j POSTROUTING_MPort");
	doSystem("iptables -t nat -I POSTROUTING -j POSTROUTING_MPort");
	
	doSystem("iptables -t nat -N PREROUTING_AC");
	doSystem("iptables -t nat -D PREROUTING -j PREROUTING_AC");
	doSystem("iptables -t nat -I PREROUTING -j PREROUTING_AC");
	
	doSystem("iptables -t nat -N PREROUTING_Remote");
	doSystem("iptables -t nat -D PREROUTING -j PREROUTING_Remote");
	doSystem("iptables -t nat -I PREROUTING -j PREROUTING_Remote");
	doSystem("iptables -t nat -N POSTROUTING_Remote");
	doSystem("iptables -t nat -D POSTROUTING -j POSTROUTING_Remote");
	doSystem("iptables -t nat -I POSTROUTING -j POSTROUTING_Remote");
	
	iptablesAllNATRun();

	// Jacky.Yang 26-Jul-2009, from /tmp/rfw/rule.cts for support VPN pass-through
	/*doSystem("iptables -D FORWARD -p 6 -j ACCEPT");
	doSystem("iptables -D FORWARD -p 17 -j ACCEPT");
	doSystem("iptables -D FORWARD -p 1 -j ACCEPT");
	doSystem("iptables -D FORWARD -p 47 -j ACCEPT");
	doSystem("iptables -D FORWARD -p 50 -j ACCEPT");
	doSystem("iptables -D FORWARD -p 51 -j ACCEPT");*/
	//doSystem("iptables -P INPUT DROP");

	doSystem("iptables -A FORWARD -p 1 -j ACCEPT");		//tcpmux(icmp)
	doSystem("iptables -A FORWARD -p 47 -j ACCEPT");  //GRE
	doSystem("iptables -A FORWARD -p 50 -j ACCEPT");	//IPSec ESP
	doSystem("iptables -A FORWARD -p 51 -j ACCEPT");  //IPSec AH
	doSystem("iptables -A FORWARD -p 6 -j ACCEPT");   
	doSystem("iptables -A FORWARD -p 17 -j ACCEPT");

	//Add for insert the port trigger rules to iptables - U-Media Ricky Cao on Apr. 23 2008
	PortTriggerRulesInitiate();

	//U-Media Ricky Cao on May 16 2008
	dropPort0Port1();
	
	doSystem("iptables -A INPUT -p tcp -j ACCEPT");
	doSystem("iptables -A INPUT -p udp -j ACCEPT");

	//Add for configure MSS to 1452 bytes when PPP link is up - U-Media Ricky Cao on May 16 2008
	/* 2010.01.07 Joan.Huang modify ,even though ppp0 no found the rule need add (no use wan IP) */
	if(!strcmp(connType, "L2TP") || !strcmp(connType, "PPTP")){
			doSystem("iptables -t nat -I PREROUTING -i eth2.2 -p udp --sport 53 -s 0.0.0.0/0 -d %s/%s -j ACCEPT", lan_ipaddr, lan_netmask);
		
			doSystem("iptables -t nat -I PREROUTING -i eth2.2 -s 0.0.0.0/0 -d %s/%s -j DROP", lan_ipaddr, lan_netmask);	//drop eth2.2 from WAN to All(Lan or wan) packet    
			doSystem("iptables -t nat -I POSTROUTING -s %s/%s  -o eth2.2 -j DROP", lan_ipaddr, lan_netmask);	//drop eth2.2 from LAN to WAN packet 
			//doSystem("iptables -t nat -A POSTROUTING -s %s/%s -o eth2.2 -j MASQUERADE", lan_ipaddr, lan_netmask); //interface eth2.2 NAT for ppp connection use ,if ppp-ip-up then delete rule					       
			doSystem("iptables -I FORWARD -p tcp -m tcp --tcp-flags SYN,RST SYN -m tcpmss --mss 1460:2500 -j TCPMSS --set-mss %d", wanMtu-40);
			doSystem("iptables -I FORWARD -p tcp --tcp-flags SYN,RST SYN -o ppp0 -j TCPMSS --clamp-mss-to-pmtu");
		}else if(!strcmp(connType, "PPPOE")){
			doSystem("ifconfig eth2.2 0.0.0.0"); //pppoe mode no need eth2.2
			doSystem("iptables -I FORWARD -p tcp -m tcp --tcp-flags SYN,RST SYN -m tcpmss --mss 1460:2500 -j TCPMSS --set-mss %d", wanMtu-40);
			doSystem("iptables -I FORWARD -p tcp --tcp-flags SYN,RST SYN -o ppp0 -j TCPMSS --clamp-mss-to-pmtu");
		}else if(!strcmp(connType, "RussiaPPTP") || !strcmp(connType, "RussiaL2TP")){
			//doSystem("iptables -t nat -A POSTROUTING -o eth2.2 -j MASQUERADE", lan_ipaddr, lan_netmask); //interface eth2.2 NAT for ppp connection use ,if ppp-ip-up then delete rule
			doSystem("iptables -t nat -A POSTROUTING -s %s/%s -o br0 -j ACCEPT", lan_ipaddr, lan_netmask);			
			//doSystem("iptables -t nat -A POSTROUTING -s %s/%s -o ! ppp0 -j MASQUERADE", lan_ipaddr, lan_netmask);
		          
			doSystem("iptables -I FORWARD -p tcp -m tcp --tcp-flags SYN,RST SYN -m tcpmss --mss 1460:2500 -j TCPMSS --set-mss %d", wanMtu-40);
			doSystem("iptables -I FORWARD -p tcp --tcp-flags SYN,RST SYN -o ppp0 -j TCPMSS --clamp-mss-to-pmtu");
		}

	//Tom.Hung 2009-11-9, re-initial NTP client.
	doSystem("ntp.sh");
	//Tom.Hung 2009-11-9
	addScheduleToCron();	//Joan.huang 2009-12-09 about schedule rule all move to addScheduleToCron().
	// Joan.huang 2010.02.03 set ip_forward is Forward and clear connection track table  
	doSystem("echo 1 > /proc/sys/net/ipv4/ip_forward");
	doSystem("echo 1 > /proc/sys/net/nf_conntrack_flush");

}

void firewall_fini(void)
{
	iptablesAllFilterClear();
	iptablesAllNATClear();
}
