/*
 * RTSP extension for IP connection tracking
 * (C) 2003 by Tom Marshall <tmarshall@real.com>
 * based on ip_conntrack_irc.c
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 * 6 Jan 2008: Arthur Tang <tangjingbiao@gmail.com>
 *     - modified for pbrang(client-port xxx-xxx)
 * 
 * Module load syntax:
 *   insmod ip_conntrack_rtsp.o ports=port1,port2,...port<MAX_PORTS>
 *                              max_outstanding=n setup_timeout=secs
 *
 * If no ports are specified, the default will be port 554.
 *
 * With max_outstanding you can define the maximum number of not yet
 * answered SETUP requests per RTSP session (default 8).
 * With setup_timeout you can specify how long the system waits for
 * an expected data channel (default 300 seconds).
 *
 *
 * 2008-01-19   correct for pbrange, Modified by Arthur
 * 
 * 2008-08-31   migrate for linux 2.6.21.x, modified by Arthur
 *
 *   -  Arthur Tang ( tangjingbiao@gmail.com )
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/netfilter.h>
#include <linux/ip.h>
#include <net/checksum.h>
#include <net/tcp.h>

#include <net/netfilter/nf_conntrack.h>
#include <net/netfilter/nf_conntrack_expect.h>
#include <net/netfilter/nf_conntrack_ecache.h>
#include <net/netfilter/nf_conntrack_helper.h>
#include <linux/netfilter/nf_conntrack_rtsp.h>

#include <linux/ctype.h>
#define NF_NEED_STRNCASECMP
#define NF_NEED_STRTOU16
#define NF_NEED_STRTOU32
#define NF_NEED_NEXTLINE
#include <linux/netfilter_helpers.h>
#define NF_NEED_MIME_NEXTLINE
#include <linux/netfilter_mime.h>

#define MAX_SIMUL_SETUP 8 /* XXX: use max_outstanding */

#if 0
#define INFOP(fmt, args...) printk(KERN_WARNING "%s: %s: " fmt, __FILE__, __FUNCTION__ , ## args)
#define DEBUGP(fmt, args...) printk(KERN_WARNING "%s: %s: " fmt, __FILE__, __FUNCTION__ , ## args)
#else
#define DEBUGP(fmt, args...)
#define INFOP(fmt, args...)
#endif

#define MAX_PORTS 8
static int ports[MAX_PORTS];
static int num_ports = 0;
static int max_outstanding = 8;
static unsigned int setup_timeout = 300;

MODULE_AUTHOR("Tom Marshall <tmarshall@real.com>");
MODULE_DESCRIPTION("RTSP connection tracking module");
MODULE_LICENSE("GPL");
MODULE_ALIAS("ip_conntrack_rtsp");
#ifdef MODULE_PARM
module_param_array(ports, int, &num_ports, 0600);
MODULE_PARM_DESC(ports, "port numbers of RTSP servers");
module_param(max_outstanding, int, 0600);
MODULE_PARM_DESC(max_outstanding, "max number of outstanding SETUP requests per RTSP session");
module_param(setup_timeout, uint, 0600);
MODULE_PARM_DESC(setup_timeout, "timeout on for unestablished data channels");
#endif

static char *rtsp_buffer;

DEFINE_RWLOCK(nf_rtsp_lock);
// struct module* ip_conntrack_rtsp = THIS_MODULE;

/*
 * Max mappings we will allow for one RTSP connection (for RTP, the number
 * of allocated ports is twice this value).  Note that SMIL burns a lot of
 * ports so keep this reasonably high.  If this is too low, you will see a
 * lot of "no free client map entries" messages.
 */
#define MAX_PORT_MAPS 16

/*** default port list was here in the masq code: 554, 3030, 4040 ***/

#define SKIP_WSPACE(ptr,len,off) while(off < len && isspace(*(ptr+off))) { off++; }

unsigned int (*nf_nat_rtsp_hook)(struct sk_buff **pskb,
				 enum ip_conntrack_info ctinfo,
				 unsigned int matchoff,
				 unsigned int matchlen,
				 struct nf_conntrack_expect *exp,
				 struct nf_conntrack_expect *exp2,
				 struct nf_ct_rtsp_expect *expinfo,
				 unsigned int destioff,
				 unsigned int destilen);

EXPORT_SYMBOL_GPL(nf_nat_rtsp_hook);

/*
 * Parse an RTSP packet.
 *
 * Returns zero if parsing failed.
 *
 * Parameters:
 *  IN      ptcp        tcp data pointer
 *  IN      tcplen      tcp data len
 *  IN/OUT  ptcpoff     points to current tcp offset
 *  OUT     phdrsoff    set to offset of rtsp headers
 *  OUT     phdrslen    set to length of rtsp headers
 *  OUT     pcseqoff    set to offset of CSeq header
 *  OUT     pcseqlen    set to length of CSeq header
 */
static int
rtsp_parse_message(char* ptcp, uint tcplen, uint* ptcpoff,
                   uint* phdrsoff, uint* phdrslen,
                   uint* pcseqoff, uint* pcseqlen)
{
    uint    entitylen = 0;
    uint    lineoff;
    uint    linelen;

    if (!nf_nextline(ptcp, tcplen, ptcpoff, &lineoff, &linelen))
    {
        return 0;
    }

    *phdrsoff = *ptcpoff;
    while (nf_mime_nextline(ptcp, tcplen, ptcpoff, &lineoff, &linelen))
    {
        if (linelen == 0)
        {
            if (entitylen > 0)
            {
                *ptcpoff += min(entitylen, tcplen - *ptcpoff);
            }
            break;
        }
        if (lineoff+linelen > tcplen)
        {
            INFOP("!! overrun !!\n");
            break;
        }

        if (nf_strncasecmp(ptcp+lineoff, "CSeq:", 5) == 0)
        {
            *pcseqoff = lineoff;
            *pcseqlen = linelen;
        }
        if (nf_strncasecmp(ptcp+lineoff, "Content-Length:", 15) == 0)
        {
            uint off = lineoff+15;
            SKIP_WSPACE(ptcp+lineoff, linelen, off);
            nf_strtou32(ptcp+off, &entitylen);
        }
    }
    *phdrslen = (*ptcpoff) - (*phdrsoff);

    return 1;
}

/*
 * Find lo/hi client ports (if any) in transport header
 * In:
 *   ptcp, tcplen = packet
 *   tranoff, tranlen = buffer to search
 *
 * Out:
 *   pport_lo, pport_hi = lo/hi ports (host endian)
 *
 * Returns nonzero if any client ports found
 *
 * Note: it is valid (and expected) for the client to request multiple
 * transports, so we need to parse the entire line.
 */
static int
rtsp_parse_transport(char* ptran, uint tranlen, uint* matchoff, uint* matchlen,
                     struct nf_ct_rtsp_expect* prtspexp, uint *destioff, uint *destilen)
{
    int     rc = 0;
    uint    off = 0;

    if (tranlen < 10 || !iseol(ptran[tranlen-1]) ||
        nf_strncasecmp(ptran, "Transport:", 10) != 0)
    {
        INFOP("sanity check failed\n");
        return 0;
    }
    DEBUGP("tran='%.*s'\n", (int)tranlen, ptran);
    off += 10;
    SKIP_WSPACE(ptran, tranlen, off);

    /* Transport: tran;field;field=val,tran;field;field=val,... */
    while (off < tranlen)
    {
        const char* pparamend;
        uint        nextparamoff;

        pparamend = memchr(ptran+off, ',', tranlen-off);
        pparamend = (pparamend == NULL) ? ptran+tranlen : pparamend+1;
        nextparamoff = pparamend-ptran;

        while (off < nextparamoff)
        {
            const char* pfieldend;
            uint        nextfieldoff;

            pfieldend = memchr(ptran+off, ';', nextparamoff-off);
            nextfieldoff = (pfieldend == NULL) ? nextparamoff : pfieldend-ptran+1;

	    if (strncmp(ptran+off, "destination=", 12) == 0)
	    {
		    uint poff;
		    uint plen=0;

		    off += 12;
		    poff = *destioff = off;
		    DEBUGP("found destination at off=%u\n", *destioff);
		    while ((ptran[poff] != ';')  &&
		           (ptran[poff] != 0x0d) &&
			   (ptran[poff] != 0x0a))
		    {
			    poff++;
			    plen++;
		    }
		    *destilen = plen;
		    DEBUGP("found destination len=%u, poff=%u, ending=%c%c\n", 
				    *destilen, poff, ptran[poff-1],ptran[poff]);
	    }


            if (strncmp(ptran+off, "client_port=", 12) == 0)
            {
                u_int16_t   port;
                uint        numlen;
		uint	    toff;

                off += 12;
                numlen = nf_strtou16(ptran+off, &port);
		toff = off;
                off += numlen;
                if (prtspexp->loport != 0 && prtspexp->loport != port)
                {
                    DEBUGP("multiple ports found, port %hu ignored\n", port);
                }
                else
                {
                    prtspexp->loport = prtspexp->hiport = port;
		    *matchoff = toff;
		    *matchlen = numlen;
                    if (ptran[off] == '-')
                    {
                        off++;
                        numlen = nf_strtou16(ptran+off, &port);
                        off += numlen;
                        prtspexp->pbtype = pb_range;
                        prtspexp->hiport = port;
		    	*matchlen += numlen + 1;

                        // If we have a range, assume rtp:
                        // loport must be even, hiport must be loport+1
                        if ((prtspexp->loport & 0x0001) != 0 ||
                            prtspexp->hiport != prtspexp->loport+1)
                        {
                            DEBUGP("incorrect range: %hu-%hu, correcting\n",
                                   prtspexp->loport, prtspexp->hiport);
                            prtspexp->loport &= 0xfffe;
                            prtspexp->hiport = prtspexp->loport+1;
                        }
                    }
                    else if (ptran[off] == '/')
                    {
                        off++;
                        numlen = nf_strtou16(ptran+off, &port);
                        off += numlen;
                        prtspexp->pbtype = pb_discon;
                        prtspexp->hiport = port;
		    	*matchlen += numlen + 1;
                    }
                    rc = 1;
                }
            }

            /*
             * Note we don't look for the destination parameter here.
             * If we are using NAT, the NAT module will handle it.  If not,
             * and the client is sending packets elsewhere, the expectation
             * will quietly time out.
             */

            off = nextfieldoff;
        }

        off = nextparamoff;
    }

    return rc;
}

/*** conntrack functions ***/

/* outbound packet: client->server */
static int
help_out(struct sk_buff **pskb, unsigned char *rtsp_ptr, unsigned int  datalen,
                struct nf_conn *ct, enum ip_conntrack_info ctinfo)
{
    int dir = CTINFO2DIR(ctinfo);   /* = IP_CT_DIR_ORIGINAL */
    char*   pdata = rtsp_ptr;
    uint    dataoff = 0;
    struct nf_ct_rtsp_expect rtsp_expinfo, *expinfo;
    struct nf_conntrack_expect *exp = NULL;
    struct nf_conntrack_expect *exp2 = NULL;
    __be16 loport, hiport;
    typeof(nf_nat_rtsp_hook) nf_nat_rtsp;

    expinfo = &rtsp_expinfo;

    memset(expinfo, 0, sizeof(struct nf_ct_rtsp_expect));

    while (dataoff < datalen)
    {
        uint    cmdoff = dataoff;
        uint    hdrsoff  = 0;
        uint    hdrslen  = 0;
        uint    cseqoff  = 0;
        uint    cseqlen  = 0;
        uint    lineoff  = 0;
        uint    linelen  = 0;
        uint    off;
	uint	matchoff = 0;
	uint	matchlen = 0;
	uint	destioff = 0;
	uint	destilen = 0;

        int     rc = 0;

        if (!rtsp_parse_message(pdata, datalen, &dataoff,
                                &hdrsoff, &hdrslen,
                                &cseqoff, &cseqlen))
        {
            break;      /* not a valid message */
        }

        if (strncmp(pdata+cmdoff, "SETUP ", 6) != 0)
        {
            continue;   /* not a SETUP message */
        }
        DEBUGP("found a setup message\n");

	exp = nf_conntrack_expect_alloc(ct);
	if (exp == NULL)
		return NF_DROP;

        off = 0;
        while (nf_mime_nextline(pdata+hdrsoff, hdrslen, &off,
                                &lineoff, &linelen))
        {
            if (linelen == 0)
            {
                break;
            }
            if (off > hdrsoff+hdrslen)
            {
                INFOP("!! overrun !!");
                break;
            }

            if (nf_strncasecmp(pdata+hdrsoff+lineoff, "Transport:", 10) == 0)
            {
                if (rtsp_parse_transport(pdata+hdrsoff+lineoff, linelen,
                                     &matchoff, &matchlen, expinfo, &destioff, &destilen))
		{
			matchoff += hdrsoff + lineoff;
			destioff += hdrsoff + lineoff;
			break;
		}
            }
        }

        if (expinfo->loport == 0)
        {
            DEBUGP("no udp transports found\n");
            continue;   /* no udp transports found */
        }

        DEBUGP("udp transport found, expinfo = %p,  ports=(%d,%hu,%hu)\n", expinfo,
              (int)expinfo->pbtype,
              expinfo->loport,
              expinfo->hiport);

	loport = htons(expinfo->loport);
	nf_conntrack_expect_init(exp, ct->tuplehash[!dir].tuple.src.l3num,
				 &ct->tuplehash[!dir].tuple.src.u3,
				 &ct->tuplehash[!dir].tuple.dst.u3,
				 IPPROTO_UDP, NULL, &loport); // Arthur

        DEBUGP("expect_related " NIPQUAD_FMT ":%u - " NIPQUAD_FMT ":%u   exp %p\n",
                NIPQUAD(exp->tuple.src.u3.ip),
                ntohs(exp->tuple.src.u.all),
                NIPQUAD(exp->tuple.dst.u3.ip),
                ntohs(exp->tuple.dst.u.all), exp);

	if (expinfo->pbtype == pb_range) {
	
		exp2 = nf_conntrack_expect_alloc(ct);
		if (exp2 == NULL)
			return NF_DROP;

		hiport = htons(expinfo->loport+1);
		nf_conntrack_expect_init(exp2, ct->tuplehash[!dir].tuple.src.l3num,
					 &ct->tuplehash[!dir].tuple.src.u3,
					 &ct->tuplehash[!dir].tuple.dst.u3,
					 IPPROTO_UDP, NULL, &hiport);

        	DEBUGP("expect_related " NIPQUAD_FMT ":%u - " NIPQUAD_FMT ":%u    exp2 %p\n",
                	NIPQUAD(exp2->tuple.src.u3.ip),
	                ntohs(exp2->tuple.src.u.all),
        	        NIPQUAD(exp2->tuple.dst.u3.ip),
                	ntohs(exp2->tuple.dst.u.all), exp2);

	}

	nf_nat_rtsp = rcu_dereference(nf_nat_rtsp_hook);
	if (nf_nat_rtsp && ct->status & IPS_NAT_MASK)
		nf_nat_rtsp(pskb, ctinfo, matchoff, matchlen,
				exp, exp2, expinfo, destioff, destilen);
	else {
	        /* pass the request off to the nat helper */
		DEBUGP("nf_conntrack_rtsp: Not define NAT Needed??? --Arthur\n");
        	rc = nf_conntrack_expect_related(exp);
		if (exp2 != NULL)
        		rc += nf_conntrack_expect_related(exp2);
	}

        if (rc == 0)
        {
            DEBUGP("rtsp: nf_conntrack_expect_related succeeded\n");
        }
        else
        {
            DEBUGP("rtsp: nf_conntrack_expect_related failed (%d)\n", rc);
        }

    	if (exp != NULL) nf_conntrack_expect_put(exp);
    	if (exp2 != NULL) nf_conntrack_expect_put(exp2);

    }

    return NF_ACCEPT;
}

/* inbound packet: server->client */
static int
help_in(struct sk_buff **pskb, unsigned char *rtsp_ptr, unsigned int  datalen,
                struct nf_conn *ct, enum ip_conntrack_info ctinfo)
{
    return NF_ACCEPT;
}

static int
help(struct sk_buff **pskb, unsigned int protoff,
                struct nf_conn *ct, enum ip_conntrack_info ctinfo)
{
    /* tcplen not negative guarenteed by ip_conntrack_tcp.c */
    struct tcphdr _tcph, *tcph;
    unsigned int dataoff, datalen;
    char *rtsp_ptr;

    /* Until there's been traffic both ways, don't look in packets. */
    if (ctinfo != IP_CT_ESTABLISHED && ctinfo != IP_CT_ESTABLISHED + IP_CT_IS_REPLY)
    {
        DEBUGP("conntrackinfo = %u\n", ctinfo);
        return NF_ACCEPT;
    }

    /* Not whole TCP header? */
    tcph = skb_header_pointer(*pskb, protoff, sizeof(_tcph), &_tcph);
    if (tcph == NULL)
	    return NF_ACCEPT;

    /* No data? */
    dataoff = protoff + tcph->doff * 4;
    datalen = (*pskb)->len - dataoff;
    
    if (dataoff >= (*pskb)->len)
    {
        return NF_ACCEPT;
    }

    spin_lock_bh(&nf_rtsp_lock);
    rtsp_ptr = skb_header_pointer(*pskb, dataoff, (*pskb)->len - dataoff, rtsp_buffer);
    BUG_ON(rtsp_ptr == NULL); 

    switch (CTINFO2DIR(ctinfo))
    {
    case IP_CT_DIR_ORIGINAL:
        help_out(pskb, rtsp_ptr, datalen, ct, ctinfo);
        break;
    case IP_CT_DIR_REPLY:
        help_in(pskb, rtsp_ptr, datalen, ct, ctinfo);
        break;
    }

    spin_unlock_bh(&nf_rtsp_lock);

    return NF_ACCEPT;
}

static struct nf_conntrack_helper rtsp_helpers[MAX_PORTS] __read_mostly;
static char rtsp_names[MAX_PORTS][10] __read_mostly;

/* This function is intentionally _NOT_ defined as __exit */
static void nf_conntrack_rtsp_fini(void)
{
    int i;
    for (i = 0; i < num_ports; i++)
    {
        DEBUGP("unregistering port %d\n", ports[i]);
        nf_conntrack_helper_unregister(&rtsp_helpers[i]);
    }
    kfree(rtsp_buffer);
}

static int __init nf_conntrack_rtsp_init(void)
{
    int i, ret;
    struct nf_conntrack_helper *hlpr;
    char *tmpname;

    printk("nf_conntrack_rtsp v" IP_NF_RTSP_VERSION " loading\n");

    if (max_outstanding < 1)
    {
        printk("nf_conntrack_rtsp: max_outstanding must be a positive integer\n");
        return -EBUSY;
    }
    if (setup_timeout < 0)
    {
        printk("nf_conntrack_rtsp: setup_timeout must be a positive integer\n");
        return -EBUSY;
    }

    rtsp_buffer = kmalloc(65536, GFP_KERNEL);
    if (rtsp_buffer == NULL)
	    return -ENOMEM;

    /* If no port given, default to standard rtsp port */
    if (ports[0] == 0)
    {
        ports[0] = RTSP_PORT;
    }

    for (i = 0; (i < MAX_PORTS) && ports[i]; i++)
    {
        hlpr = &rtsp_helpers[i];
        memset(hlpr, 0, sizeof(struct nf_conntrack_helper));

	hlpr->tuple.src.l3num = AF_INET;
        hlpr->tuple.src.u.tcp.port = htons(ports[i]);
        hlpr->tuple.dst.protonum = IPPROTO_TCP;
	hlpr->mask.src.l3num = 0xFFFF;
        hlpr->mask.src.u.tcp.port = 0xFFFF;
        hlpr->mask.dst.protonum = 0xFF;
        hlpr->max_expected = max_outstanding;
        hlpr->timeout = setup_timeout;
        hlpr->me = THIS_MODULE;
        hlpr->help = help;

        tmpname = &rtsp_names[i][0];
        if (ports[i] == RTSP_PORT)
        {
            sprintf(tmpname, "rtsp");
        }
        else
        {
            sprintf(tmpname, "rtsp-%d", i);
        }
        hlpr->name = tmpname;

        DEBUGP("port #%d: %d\n", i, ports[i]);

        ret = nf_conntrack_helper_register(hlpr);

        if (ret)
        {
            printk("nf_conntrack_rtsp: ERROR registering port %d\n", ports[i]);
            nf_conntrack_rtsp_fini();
            return -EBUSY;
        }
        num_ports++;
    }
    return 0;
}

#ifdef CONFIG_IP_NF_NAT_NEEDED
EXPORT_SYMBOL_GPL(nf_rtsp_lock);
#endif

module_init(nf_conntrack_rtsp_init);
module_exit(nf_conntrack_rtsp_fini);
