/*
 *      utility to merge all binary into one image
 *
 *      Authors: David Hsu	<davidhsu@realtek.com.tw>
 *
 *      $Id: mgbin.c,v 1.1.1.1 2005/05/18 09:58:35 CAMEO+brian Exp $
 *
 */
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#ifdef WIN32
	#include <io.h>
#endif
#include "apmib.h"

/////////////////////////////////////////////////////////////////////////////
/* Input file type */
typedef enum { BOOT_CODE=0, CONFIG, WEB_PAGES, SYS, ROOT, INVALID_FILE=-1 } TYPE_T;

typedef struct _sector {
	TYPE_T type;
	char filename[80];
	unsigned long offset;
	unsigned long size;
} SECTOR_T;

/////////////////////////////////////////////////////////////////////////////
static char *copyright="Copyright (c) Realtek Semiconductor Corp., 2003. All Rights Reserved.";
static char *version="1.2";
static SECTOR_T sector_noboot[]={
	{BOOT_CODE,	"\x0", (unsigned long)-1,0},
	{CONFIG,	"\x0", (unsigned long)-1,0},
	{WEB_PAGES,	"\x0", 0x10000,		0x20000},
	{SYS,		"\x0", 0x30000,		0},
	{ROOT,		"\x0", (unsigned long)-1,0}
};

static SECTOR_T sector_normal[]={
	{BOOT_CODE,	"\x0", 0,		0x6000},
	{CONFIG,	"\x0", 0x6000,		0xA000},
	{WEB_PAGES,	"\x0", 0x10000,		0x20000},
	{SYS,		"\x0", 0x30000,		0},
	{ROOT,		"\x0", (unsigned long)-1,0}
};

static SECTOR_T sector_compack[]={
	{BOOT_CODE,	"\x0", 0,		0x6000},
	{CONFIG,	"\x0", 0x6000,		0x2000},
	{WEB_PAGES,	"\x0", (unsigned long)-1,0},
	{SYS,		"\x0", 0x8000,		0},
	{ROOT,		"\x0", (unsigned long)-1,0}
};

//for  gateway 8M
static SECTOR_T sector_gw_8m[]={
	{BOOT_CODE,	"\x0", 0,		0x6000},
	{CONFIG,	"\x0", 0x6000,		0xA000},
	{WEB_PAGES,	"\x0", 0x10000,		0x10000},
	{SYS,		"\x0", 0x20000,		0xE0000},
	{ROOT,		"\x0", 0x100000,	0}
};
static SECTOR_T *sector=sector_normal;

#define BYTE_SWAP(word) (((word >> 8) &0xff) | ((((unsigned char)word)<<8)&0xff00) )

/////////////////////////////////////////////////////////////////////////////
static TYPE_T checkInputFile(char *filename)
{
	int fh;
	char signature[4];

	fh = open(filename, O_RDONLY);
	if ( fh == -1 )
		return INVALID_FILE;

	lseek(fh, 0, SEEK_SET);

	if ( read(fh, signature, sizeof(signature)) != sizeof(signature)) {
		close(fh);
		return INVALID_FILE;
	}
	close(fh);

	//if ( !memcmp(signature, "\x0\x0\x40\x21", 4))
	//return BOOT_CODE;

	if ( !memcmp(signature, "BOOT", 4))
	return BOOT_CODE;

	if ( !memcmp(signature, "HS", 2) )
		return CONFIG;

	if ( !memcmp(signature, "HTTP", 4) )
		return WEB_PAGES;

	if ( !memcmp(signature, "CSYS", 4) )
		return SYS;

	if ( !memcmp(signature, "ROOT", 4) )
		return ROOT;
	return INVALID_FILE;
}

////////////////////////////////////////////////////////////////////////////////
static void showHelp(void)
{
	printf("\nRTL8186 utility to merge binary.\n");
	printf("%s Ver %s.\n\n", copyright, version);
	printf("usage: mgbin [-s] [-c] [-d] -o outputfile bootcode config webpages linux root\n");
	printf("	-s : do byte swap\n");
	printf("	-c : compack layout. When this option is specified, webpages file\n");
	printf("	     will be skipped\n\n");
	printf("	-d : downsize gateway version(gw-8M)\n");
}


////////////////////////////////////////////////////////////////////////////////
static void do_byteswap(unsigned char *buf, int len)
{
	unsigned short wVal, *ptr;
	int i=0;

	while (len > 0) {
		len -= 2;
		ptr = (unsigned short *)&buf[2*i];
		wVal = *ptr;
		*ptr = BYTE_SWAP( wVal );
		i++;
	}
}


////////////////////////////////////////////////////////////////////////////////
int main(int argc, char** argv)
{
	int argNum=1;
	char outFile[80]={0};
	TYPE_T type;
	int mask=0, sect_num, cont_sect_num, fh_out, fh_in, len, i, total=0;
	unsigned char *buf;
	struct stat sbuf;
	int byteswap=0;
	int offset = 0 ;

	while (argNum < argc) {
		if ( !strcmp(argv[argNum], "-o") ) {
			if (++argNum >= argc)
				break;
			sscanf(argv[argNum], "%s", outFile);
		}
		else if ( !strcmp(argv[argNum], "-s") ) {
			byteswap = 1;
		}
		else if ( !strcmp(argv[argNum], "-n") ) {
			sector = sector_noboot;
		}
		else if ( !strcmp(argv[argNum], "-c") ) {
			sector = sector_compack;
		}
		else if ( !strcmp(argv[argNum], "-d") ) {
			sector = sector_gw_8m;
		}

		else {
			type=checkInputFile(argv[argNum]);
			if (type == INVALID_FILE) {
				printf("\nInvalid input file %s!!\n", argv[argNum]);
				showHelp();
				exit(1);
			}
			strcpy(sector[type].filename, argv[argNum]);
			mask |= (1 << type);
		}
		argNum++;
	}

	if (!outFile[0]) {
		printf("No output file specified!\n");
		showHelp();
		exit(1);
	}

	if (mask == 0) {
		printf("No valid input image found!\n");
		exit(1);
	}

#if 0
	// search image start and end section
	for (sect_num=0, cont_sect_num=0, i=BOOT_CODE; i<=SYS; i++) {
		if ( (mask >> i) & 1 ) {
			sect_num++;
			if ( i != SYS) {
				if ( (mask >> (i+1) ) & 1 )  // continous section
					cont_sect_num++;
			}
		}
	}

	if (sect_num != (cont_sect_num+1)) {
		printf("Invalid input file sequence!\n");
		exit(1);
	}
#endif

	// Create output file
#ifdef WIN32
	_chmod(outFile, S_IREAD|S_IWRITE);
	fh_out = open(outFile, O_RDWR|O_CREAT|O_TRUNC|O_BINARY);
#else
	chmod(outFile, S_IREAD|S_IWRITE);
	fh_out = open(outFile, O_RDWR|O_CREAT|O_TRUNC);
#endif
	if (fh_out == -1) {
		printf("Create output file %s error!\n", outFile);
		exit(1);
	}

	printf("\nMerge ");

	for (i=BOOT_CODE; i<=ROOT ; i++) {
		if (((int)sector[i].offset) == -1)
			continue;

		if (sector[i].filename[0]) {
			if ( stat(sector[i].filename, &sbuf) != 0 ) {
				printf("Stat file %s error!\n", sector[i].filename);
				exit(1);
			}
			buf = malloc(sbuf.st_size+1);
			if (buf == NULL) {
				printf("allocate buffer failed %d!\n", sbuf.st_size);
				exit(1);
			}
#ifdef WIN32
			fh_in = open(sector[i].filename, O_RDONLY|O_BINARY);
#else
			fh_in = open(sector[i].filename, O_RDONLY);
#endif
			if (fh_in < 0) {
				printf("Open file %s error!\n", sector[i].filename);
				close(fh_out);
				exit(1);
			}

			switch(sector[i].type) {
			case BOOT_CODE:
				printf("BOOT-CODE ");
				break;
			case CONFIG:
				printf("CONFIG-DATA ");
				break;
			case WEB_PAGES:
				printf("WEB-PAGES ");
				break;
			case SYS:
				printf("LINUX ");
				break;

			case ROOT:
				printf("ROOT ");
				break;
			}

			if (read(fh_in, buf, sbuf.st_size) != sbuf.st_size) {
				printf("Read file %s error!\n", sector[i].filename);
				close(fh_in);
				close(fh_out);
				free(buf);
				exit(1);
			}

			if (byteswap) {
				if (sbuf.st_size % 2) {
					buf[sbuf.st_size] = '\0';
					sbuf.st_size++;
				}
				do_byteswap(buf, sbuf.st_size);
			}
			//sc_yang
			if ( (sector[i].type == ROOT) || (sector[i].type == BOOT_CODE)){
				offset = sizeof(IMG_HEADER_T);
				sbuf.st_size -= sizeof(IMG_HEADER_T);
			}
			else
				offset = 0 ;
			if ( write(fh_out, buf + offset , sbuf.st_size) != sbuf.st_size) {
				printf("Write output file %s error!\n", outFile);
				close(fh_in);
				close(fh_out);
				free(buf);
				exit(1);
			}
			free(buf);
			close(fh_in);

			if (sbuf.st_size < (int)sector[i].size) {
				len = sector[i].size - sbuf.st_size;
				buf = calloc(len, 1);
				if (buf == NULL) {
					printf("allocate buffer failed %d!\n", len);
					exit(1);
				}
				write(fh_out, buf, len);	// pad 0
				free(buf);
			}
			total += sbuf.st_size;
		}
	}

	close(fh_out);

#ifdef WIN32
	_chmod(outFile, S_IREAD);
#else
	chmod(outFile, DEFFILEMODE);
#endif

	printf("=> %s ok, size=%d.\n", outFile, total);
	exit(0);
}

