/*
 *      utility to merge all binary into one image
 *
 *      Authors: David Hsu	<davidhsu@realtek.com.tw>
 *
 *      $Id: mgbin.c,v 1.7 2006/03/06 03:38:03 davidhsu Exp $
 *
 */
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#ifdef WIN32
	#include <io.h>
#endif
#include "apmib.h"

//#define DEBUG_PRINT	printf
#define DEBUG_PRINT

/////////////////////////////////////////////////////////////////////////////
/* Input file type */
typedef enum { BOOT_CODE=0, CONFIG, WEB_PAGES, SYS, ROOT, INVALID_FILE=-1 } TYPE_T;

typedef struct _sector {
	TYPE_T type;
	char filename[80];
	unsigned long offset;
	unsigned long size;
	int with_header;
} SECTOR_T;

/////////////////////////////////////////////////////////////////////////////
static char *copyright="Copyright (c) Realtek Semiconductor Corp., 2003~2005. All Rights Reserved.";
static char *version="1.4";

static SECTOR_T sector[ROOT+1];

#define BYTE_SWAP(word) (((word >> 8) &0xff) | ((((unsigned char)word)<<8)&0xff00) )

/////////////////////////////////////////////////////////////////////////////
static TYPE_T checkInputFile(char *filename, int *pWith_header)
{
	int fh;
	char signature[4];

#ifdef WIN32
	fh = open(filename, O_RDONLY|O_BINARY);
#else
	fh = open(filename, O_RDONLY,S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH);
#endif
	if ( fh == -1 )
		return INVALID_FILE;

	lseek(fh, 0, SEEK_SET);

	if ( read(fh, signature, sizeof(signature)) != sizeof(signature)) {
		close(fh);
		return INVALID_FILE;
	}
	close(fh);

	*pWith_header = 1;
	if ( !memcmp(signature, "BOOT", 4))
		return BOOT_CODE;
	else if ( !memcmp(signature, "WEBP", 4) )
		return WEB_PAGES;

	else if ( !memcmp(signature, "CSYS", 4) )
		return SYS;

	else if ( !memcmp(signature, "ROOT", 4) )
		return ROOT;
	
	else if ( !memcmp(signature, "HS", 2) || !memcmp(signature, "DS", 2) || !memcmp(signature, "CS", 2) )
		return CONFIG;
	
	else if ( !memcmp(signature, "\x0\x0\x40\x21\x40\x88\x60\x0", 8)) {
		*pWith_header = 0;
		return BOOT_CODE;
	}
	else if ( !memcmp(signature+8, "\x0\x0\x40\x21\x40\x88\x60\x0", 8)) {
		*pWith_header = 0;		
		return BOOT_CODE;
	}
	return INVALID_FILE;
}

////////////////////////////////////////////////////////////////////////////////
static void showHelp(void)
{
	printf("\nRTL8186 utility to merge binary.\n");
	printf("%s Ver %s.\n\n", copyright, version);
	printf("usage: mgbin [-s] [-c] -o outputfile bootcode config webpages linux root\n");
	printf("	-s : do byte swap\n");
	printf("	-c : cascade. May use this option to merge image for web upload\n");
}


////////////////////////////////////////////////////////////////////////////////
static void do_byteswap(unsigned char *buf, int len)
{
	unsigned short wVal, *ptr;
	int i=0;

	while (len > 0) {
		len -= 2;
		ptr = (unsigned short *)&buf[2*i];
		wVal = *ptr;
		*ptr = BYTE_SWAP( wVal );
		i++;
	}
}


////////////////////////////////////////////////////////////////////////////////
int main(int argc, char** argv)
{
	int argNum=1;
	char outFile[80]={0};
	TYPE_T type;
	int mask=0, fh_out, fh_in, len, i, total=0;
	unsigned char *buf, *buf1;
	struct stat sbuf;
	int byteswap=0, cascade=0, last_idx=-1;
	int offset=0, with_header;
	IMG_HEADER_Tp pHeader;
	unsigned long burnAddr;
		
	memset(&sector, 0, sizeof(sector));

	while (argNum < argc) {
		if ( !strcmp(argv[argNum], "-o") ) {
			if (++argNum >= argc)
				break;
			sscanf(argv[argNum], "%s", outFile);
		}
		else if ( !strcmp(argv[argNum], "-s") ) {
			byteswap = 1;
		}
		else if ( !strcmp(argv[argNum], "-c") ) {
			cascade = 1;
		}
		else {
			type=checkInputFile(argv[argNum], &with_header);
			DEBUG_PRINT("filename=%s, type=%d\n", argv[argNum], type);			
			if (type == INVALID_FILE) {
				printf("\nInvalid input file %s!!\n", argv[argNum]);
				showHelp();
				exit(1);
			}
			strcpy(sector[type].filename, argv[argNum]);
			sector[type].with_header = with_header;
			sector[type].type = type;
			mask |= (1 << type);
		}
		argNum++;
	}

	if (!outFile[0]) {
		printf("No output file specified!\n");
		showHelp();
		exit(1);
	}

	if (mask == 0) {
		printf("No valid input image found!\n");
		exit(1);
	}

	// Create output file
#ifdef WIN32
	_chmod(outFile, S_IREAD|S_IWRITE);
	fh_out = open(outFile, O_RDWR|O_CREAT|O_TRUNC|O_BINARY);
#else
	chmod(outFile, S_IREAD|S_IWRITE);
	fh_out = open(outFile, O_RDWR|O_CREAT|O_TRUNC);
#endif
	if (fh_out == -1) {
		printf("Create output file %s error!\n", outFile);
		exit(1);
	}

	printf("\nMerge ");

	for (i=BOOT_CODE; i<=ROOT ; i++) {
		if (sector[i].filename[0]) {
			if ( stat(sector[i].filename, &sbuf) != 0 ) {
				printf("Stat file %s error!\n", sector[i].filename);
				exit(1);
			}
			buf = malloc(sbuf.st_size+1);
			if (buf == NULL) {
				printf("allocate buffer failed %d!\n", sbuf.st_size);
				exit(1);
			}
#ifdef WIN32
			fh_in = open(sector[i].filename, O_RDONLY|O_BINARY);
#else
			fh_in = open(sector[i].filename, O_RDONLY,S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH);
#endif
			if (fh_in < 0) {
				printf("Open file %s error!\n", sector[i].filename);
				close(fh_out);
				exit(1);
			}

			switch(sector[i].type) {
			case BOOT_CODE:
				printf("BOOT-CODE ");
				break;
			case CONFIG:
				printf("CONFIG-DATA ");
				break;
			case WEB_PAGES:
				printf("WEB-PAGES ");
				break;
			case SYS:
				printf("LINUX ");
				break;
			case ROOT:
				printf("ROOT ");
				break;
			}

			if (read(fh_in, buf, sbuf.st_size) != sbuf.st_size) {
				printf("Read file %s error!\n", sector[i].filename);
				close(fh_in);
				close(fh_out);
				free(buf);
				exit(1);
			}

			if (sector[i].with_header) {
				if (sector[i].type == CONFIG)
					burnAddr = CURRENT_SETTING_OFFSET;
				else {
					pHeader = (IMG_HEADER_Tp)buf;
					burnAddr = DWORD_SWAP(pHeader->burnAddr);
				}
			}
			else 
				burnAddr = 0;			

			if (byteswap) {
				if (sbuf.st_size % 2) {
					buf[sbuf.st_size] = '\0';
					sbuf.st_size++;
				}
				do_byteswap(buf, sbuf.st_size);
			}
			
			// try to append 0 if necessary
			if (!cascade && last_idx!=-1 && sector[i].with_header) {
				if ((sector[last_idx].offset+sector[last_idx].size) < burnAddr) {
					len = burnAddr - (sector[i-1].offset+sector[i-1].size);
					buf1 = calloc(len, 1);
					if (buf1 == NULL) {
						printf("allocate buffer failed %d!\n", len);
						exit(1);
					}
					write(fh_out, buf1, len);	// pad 0
					free(buf1);
					total += len;
				}				
			}			

			// skip header if necessary
			if (!cascade && sector[i].with_header &&
					((sector[i].type == ROOT) || (sector[i].type == BOOT_CODE))) {
				offset = sizeof(IMG_HEADER_T);
				sbuf.st_size -= sizeof(IMG_HEADER_T);
			}
			else
				offset = 0;
			
			if ( write(fh_out, buf + offset , sbuf.st_size) != sbuf.st_size) {
				printf("Write output file %s error!\n", outFile);
				close(fh_in);
				close(fh_out);
				free(buf);
				exit(1);
			}
			close(fh_in);
			sector[i].offset = burnAddr;
			sector[i].size = sbuf.st_size;
			total += sbuf.st_size;
			free(buf);
			last_idx = i;
		}
	}

	close(fh_out);

#ifdef WIN32
	_chmod(outFile, S_IREAD);
#else
	chmod(outFile, DEFFILEMODE);
#endif

	printf("=> %s ok, size=%d.\n", outFile, total);
	exit(0);
}

